/*
 * dirread.c dated 08/22/95
 *
 * Author:
 * Thomas Binder
 * (binder@rbg.informatik.th-darmstadt.de)
 *
 * Purpose:
 * Example program for correct use of alternative
 * file systems, independent of the operating
 * system currently used (it should be TOS-
 * compatible, of course). DirRead reads the
 * directory given in the commandline and displays
 * the filenames to stdout, one per line (without
 * attributes, as the file would get too long,
 * then).
 *
 * History:
 * (Omitted in the English translation)
 */

/*
 * If your compiler does not provide its own
 * mintbind.h for the new GEMDOS functions, you
 * have to build this file yourself, e.g. containing
 * #defines for the new calls
 */
#include <mintbind.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/*
 * Some mintbind.h (not the one that comes with
 * the MiNTLib) already define the error codes.
 * In that case, you'll have to remove the
 * following #include. If you don't have such a
 * file at all, build it yourself using the
 * constants found e.g. in the compendium (or in
 * table 2 of the article ;)
 */
#include <atarierr.h>
/*
 * portab.h is a file that comes with Pure C. It
 * defines WORD, LONG, UWORD, ULONG, etc. to be
 * independent of the sizeof(int) a compiler
 * uses.
 */
#include <portab.h>

/*
 * The following two constants determine how
 * long a filename (DFLT_MAXFLEN) and how long
 * a path (DFLT_MAXPLEN) may be at max if
 * Dpathconf() is not available or returns
 " "unlimited".
 */
#define DFLT_MAXFLEN    64
#define DFLT_MAXPLEN    512

#define VERSION         "V1.05 dated 08/22/95"

/* Prototypes */
WORD rel2abs(char *path, char **destpath);
WORD d_getcwd(char *buf, WORD drive, WORD len);
void usage(void);

WORD main(WORD argc, char *argv[])
{
    LONG    maxflen,
            dirh;
    WORD    rootdrive = -1,
            count = 0,
            err;
    char    *unixmode,
            *argument,
            *setpath,
            *namebuf,
            newdrive,
            *r;
#ifdef __MINT__
    _DTA
#else
    DTA
#endif
            my_dta,
            *olddta;

/* Display usage if number of parameters if wrong */
    if (argc != 2)
    {
        usage();
        return(2);
    }
/* Switch to MiNT-domain */
    Pdomain(1);
/*
 * Take the given path and transform it into an
 * absolute one, i.e. of the form X:\...\
 */
    if ((err = rel2abs(argv[1], &argument)) != 0)
    {
	puts("DirRead: Error calling rel2abs!");
        return(err);
    }
/*
 * Now, evaluate $UNIXMODE. If it contains rX, X
 * will be the default drive (if it's known to
 * GEMDOS)
 */
    if ((unixmode = getenv("UNIXMODE")) != NULL)
    {
        if ((r = strchr(unixmode, 'r')) != NULL)
        {
            newdrive = r[1];
            if ((newdrive >= 'A') &&
                (newdrive <= 'Z'))
            {
                newdrive -= 'A';
                if (Dsetdrv(Dgetdrv()) &
                    (1L << (LONG)newdrive))
                {
                    rootdrive = (WORD)newdrive;
                    Dsetdrv(rootdrive);
                }
            }
        }
    }
/*
 * Start reading the directory. First, try it with
 * Dopendir(). If that's available, Dreaddir() and
 * Dclosdir() must be available, too, as it wouldn't
 * make any sense if not. If Dopendir() is not
 * supported, though, Fsfirst()/next() have to be
 * used instead.
 */
    if ((dirh = Dopendir(".", 0)) == EINVFN)
    {
        olddta = Fgetdta();
        Fsetdta(&my_dta);
        strcat(argument, "*.*");
        err = Fsfirst(argument, 0x17);
        while (!err)
        {
            count++;
#ifdef __MINT__
            puts(my_dta.dta_name);
#else
            puts(my_dta.d_fname);
#endif
            err = Fsnext();
        }
        if (err == EPTHNF)
        {
            printf("DirRead: Directory %s is not "
                "reachable!\n", argument);
            free(argument);
            return(err);
        }
        Fsetdta(olddta);
    }
    else
    {
        if ((dirh & 0xff000000L) != 0xff000000L)
            Dclosedir(dirh);
/*
 * In case Dopendir() & Co. are available, first
 * try to determine the maximum filename length
 * in the directory. Therefore, change to this
 * directory and call Dpathconf() on ".". If
 * there was a root drive supplied by $UNIXMODE,
 * change the path: X:\foo gets \X\foo, so that
 * the actual drive is accessed as a subdirectory
 * of the root drive. Of course, the transformation
 * must not take place if X is already the root
 * drive ...
 */
        if ((rootdrive >= 0) && (argument[0] !=
            ((char)rootdrive + 65)))
        {
            argument[1] = argument[0] | 32;
            argument[0] = '\\';
            setpath = argument;
        }
        else
        {
            Dsetdrv((WORD)argument[0] - 65);
            setpath = &argument[2];
        }
        if ((err = Dsetpath(setpath)) < 0)
        {
            printf("DirRead: Directory %s is not "
                "reachable!\n", argument);
            free(argument);
            return(err);
        }
        if (Dpathconf(".", -1) >= 3L)
        {
            maxflen = Dpathconf(".", 3);
            if (maxflen == 0x7fffffffL)
                maxflen = DFLT_MAXFLEN;
        }
/*
 * Increase maxflen by 5 due to the file index
 * and \0
 */
        maxflen += 5;
/* Allocate memory for the filename buffer */
        if ((namebuf = malloc(maxflen)) == NULL)
        {
            puts("DirRead: Not enough memory!");
            free(argument);
            return(ENSMEM);
        }
/* Open the directory */
        dirh = Dopendir(argument, 0);
        if ((dirh & 0xff000000L) == 0xff000000L)
        {
            printf("DirRead: Directory %s is not "
                "reachable!\n", argument);
            free(namebuf);
            free(argument);
            return((WORD)dirh);
        }
/*
 * After successfully opening the directory, read
 * filenames with Dreaddir() until it returns an
 * error. If the error is ERANGE, the filename
 * buffer is too small for the current name (this
 * should only happen if we had to use
 * DFLT_MAXFLEN) and we can safely continue reading.
 * In all other error situations, stop reading.
 */
        for (;;)
        {
            if ((err = (WORD)Dreaddir((WORD)
                maxflen, dirh, namebuf)) != 0L)
            {
                if (err == ERANGE)
                {
                    puts("*** Filename too long ***");
                    continue;
                }
                else
                    break;
            }
            count++;
            puts(&namebuf[4]);
        }
/* Important: Don't forget Dclosedir()! */
        Dclosedir(dirh);
        free(namebuf);
    }
    if (count)
        printf("%u files\n", count);
    else
        puts("No files!");
    free(argument);
    return(0);
}

/*
 * rel2abs
 *
 * Transforms a possibly relative path in to an
 * absoute one (X:\...\), without paying attention
 * if the given path exists or is really a
 * directory. Memory for the absolute path is
 * automatically allocated, the caller is then
 * responsible to free it. rel2abs tries to
 * remove as many path errors (like multiple
 * backslashes) as possible, but of course it's
 * helpless on totally senseless input.
 *
 * Input:
 * path: Pointer to the original path, which may
 *       be relative and may contain "." and ".."
 *       components
 * destpath: Pointer to a pointer for the new
 *           path. On successful transformation,
 *           the pointer to the malloc()ed
 *           memory block is stored here. In
 *           case an error occured, destpath will
 *           be set to NULL.
 *
 * Return:
 * 0: Everything OK, destpath contains pointer to
 *    the transformed path
 * otherwise: GEMDOS error code, destpath points
 *            to NULL
 */
WORD rel2abs(char *path, char **destpath)
{
    char    *act_path,
            *temp,
            *pos,
            *next,
            rootdir[4] = "x:\\";
    WORD    drive,
            err;
    LONG    maxplen;

/*
 * If path contains a drive letter, extract it,
 * otherwise determine the current drive
 */
    if (path[1] == ':')
    {
        drive = (path[0] & ~32) - 65;
        pos = &path[2];
    }
    else
    {
        if ((drive = Dgetdrv()) < 0)
            return(drive);
        pos = path;
    }
/*
 * Try to get the maximum path length for
 * the root directory of the affected drive.
 */
    rootdir[0] = (drive + 65);
    if (Dpathconf(rootdir, -1) >= 2)
    {
        maxplen = Dpathconf(rootdir, 2);
        if (maxplen == 0x7fffffffL)
            maxplen = DFLT_MAXPLEN;
    }
    else
        maxplen = DFLT_MAXPLEN;
/*
 * Add two bytes for the trailing backslash
 * and \0 for EOS
 */
    maxplen += 2;
/*
 * Allocated memory for two path buffers, this
 * also yields the address which is stored in
 * destpath
 */
    if ((*destpath = temp = malloc(maxplen * 2L))
        == NULL)
    {
        return(ENSMEM);
    }
/* Get the current path of the drive */
    act_path = (char *)((LONG)temp + maxplen);
    if ((err = d_getcwd(act_path, drive + 1,
        (WORD)maxplen)) < 0)
    {
        free(temp);
        *destpath = NULL;
        return(err);
    }
/* Remove traling backslash, if necessary */
    if (*act_path && (act_path[strlen(act_path) - 1] ==
        '\\'))
    {
        act_path[strlen(act_path) - 1] = 0;
    }
/*
 * Create beginning of the absolute path. If path
 * wasn't absolute, use the current path.
 * Otherwise, skip additional leading backslashes.
 */
    sprintf(temp, "%c:", drive + 65);
    if (*pos != '\\')
        strcat(temp, act_path);
    else
        for (; *pos && (*pos == '\\'); pos++);
/*
 * The rest of the path is now split into
 * components (from the current position up
 * to the next backslash). Is the current
 * component ".", it'll be ignored as it
 * stands for the same directory. ".." will
 * remove the last added component (only
 * if there are components to be removed
 * at all). Again, multiple backslashes will
 * be skipped.
 */
    next = pos;
    while ((next != NULL) && *pos)
    {
        if ((next = strchr(pos, '\\')) != NULL)
            for(; *next && (*next == '\\'); *next++ = 0);
        if (!strcmp(pos, ".."))
        {
            if (strlen(temp) > 2)
                *strrchr(temp, '\\') = 0;
        }
        else
        {
            if (strcmp(pos, "."))
            {
                strcat(temp, "\\");
                strcat(temp, pos);
            }
        }
        if (next != NULL)
            pos = next;
    }
/*
 * After the transformation, terminate the new
 * path with a backslash, to signal it's a
 * directory. There's no check if that's
 * really true, i.e. the created path may
 * be a file or may even not exist at all.
 * If you want to add the backslash only if
 * the new path is an existing directory,
 * use Fxattr() (or Fsfirst(), if the other
 * function is not available) and check the
 * return code and the attributes.
 */
    strcat(temp, "\\");
    return(0);
}

/*
 * d_getcwd
 *
 * Just like Dgetcwd(), but uses Dgetpath()
 * automatically when Dgetcwd() isn't available
 * (of course, in this case you can only pray
 * that buf is large enough as len will be
 * ignored). Dgetpath() is also used if
 * Dgetcwd() returned EDRIVE. This is due to a
 * problem in connection with MetaDOS, which
 * does not offer Dgetcwd(), so that this call
 * is directly passed to the underlying GEMDOS
 * which of course does not know about this
 * drive at all.
 *
 * Input:
 * buf: Pointer to buffer to which the current
 *      path should be copied
 * drive: The drive for which the current path
 *        should be determined (0 = current,
 *        (1 = A, 2 = B, etc.)
 * len: Maximum number of characters which can
 *      be copied into buf
 *
 * Return:
 * Return value of Dgetcwd() or Dgetpath()
 */
WORD d_getcwd(char *buf, WORD drive, WORD len)
{
    LONG    err;

	err = Dgetcwd(buf, drive, len);
	if ((err != EINVFN) && (err != EDRIVE))
    {
        return((WORD)err);
    }
    return(Dgetpath(buf, drive));
}

/*
 * usage
 *
 * Displays usage of DirRead.
 */
void usage(void)
{
    puts("DirRead "VERSION);
    puts("Geschrieben in Pure C von "
        "Thomas Binder");
    puts("\nAufruf: dirread Verzeichnis");
}

/* end */
