/********************************************************************/
/* getpure.c - read pages from an indexed Pure-C help source        */
/*                                                                  */
/* Copyright (c) 1993 by Hildo Biersma - Evil Eye Software          */
/*                            e-mail: boender@dutiws.twi.tudelft.nl */
/*        Evil Eye Software - ``Software with a Purpose''           */
/*                                                                  */
/* Freeware - do with this what you like, but leave my name.        */
/********************************************************************/
/*                                                                  */
/* Usage: getpure man-page [....]                                   */
/*        getpure prefix+                                           */
/*                                                                  */
/********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "pureind.h"

/* Local function prototypes */
void read_index(void);
void display_pages(const char *page_name);
void display_page(const struct page_record *page_ptr);
int  compare_name(const void *, const void *);
int  compare_prefix(const void *, const void *);

/* Global variables */
struct file_record *files;
struct page_record *pages;
char   				     *program_name;
long 				       no_files, no_pages;
FILE        			 *current_file = NULL;
char        			 current_file_no = ILLEGAL_FILE;

/* Read the index file into memory */
void read_index(void)
{
  FILE *index;
  char buf[4];

  if ((index = fopen(INDEXFILE_NAME, "rb")) == NULL)
  {
    fprintf(stderr, "%s: could not open index file\n", program_name);
    exit(1);
  }

  fread(buf, 4, sizeof(char), index);
  if (strncmp(buf, INDEXFILE_HEADER, 4) != 0)
  {
    fprintf(stderr, "%s: index file is of wrong type\n", program_name);
    exit(1);
  }

  fread(&no_files, 1, sizeof(long), index);
  fread(&no_pages, 1, sizeof(long), index);
  files = malloc(no_files * sizeof(struct file_record));
  pages = malloc(no_pages * sizeof(struct page_record));
  if ((files == NULL) || (pages == NULL))
  {
    fprintf(stderr, "%s: memory allocation error\n", program_name);
    exit(1);
  }

  fread(files, no_files, sizeof(struct file_record), index);
  fread(pages, no_pages, sizeof(struct page_record), index);
  fclose(index);
} /* End of read_index() */

/* Display pages from pure-c help source files */
void display_pages(const char *page_name)
{
  char               search_string[PAGENAME_SIZE + 1];
  struct page_record *page_ptr;
  const char         *ptr;
  size_t             pos;
  int (*cmp_func)(const void *, const void *);

  if ((ptr = strchr(page_name, '+')) == NULL)
  {
    strncpy(search_string, page_name, PAGENAME_SIZE);
    search_string[PAGENAME_SIZE] = 0x00;
    strlwr(search_string);
    cmp_func = compare_name;
  }
  else
  {
    pos = ptr - page_name;
    strncpy(search_string, page_name, pos);
    search_string[pos] = 0x00;
    strlwr(search_string);
    cmp_func = compare_prefix;
  }

  if ((page_ptr = bsearch(&search_string, pages, no_pages,
  												sizeof(struct page_record), cmp_func)) == NULL)
  {
    fprintf(stderr, "%s: page %s not found\n", program_name, 
    				search_string);
    return;
  }

  while ((page_ptr > pages) &&
  			 (cmp_func(page_ptr - 1, search_string) == 0))
    page_ptr--;

  while ((page_ptr < pages + no_pages) &&
         (cmp_func(page_ptr, search_string) == 0))
    display_page(page_ptr++);
} /* End of display_pages() */

/* Write one pure-c help source page to stdout */
void display_page(const struct page_record *page_ptr)
{
	int  page_done = 0;
	char line[BUFSIZ], buf[BUFSIZ];
	
  if (page_ptr->fileno != current_file_no)
  {
    if (current_file != NULL)
      fclose(current_file);
    current_file_no = page_ptr->fileno;
    if ((current_file = fopen(files[current_file_no].name,
    													"rt")) == NULL)
    {
      current_file_no = ILLEGAL_FILE;
      fprintf(stderr, "%s: could not open source file %s\n",
              program_name, files[current_file_no].name);
      return;
    }
  }

  fseek(current_file, page_ptr->pos, SEEK_SET);
  while ((page_done == 0) && !feof(current_file))
  {
  	char *ptr1 = line, *ptr2 = buf;
  	
  	fgets(line, BUFSIZ, current_file);
  	while (*ptr1 != 0x00)
	  {
	    if (*ptr1 != '\\')
	      *ptr2++ = *ptr1++;
	    else
	    {
	      if (ptr1[1] == '\\')  /* \\ means \ */
	      {
	        *ptr2++ = '\\';
	        ptr1 += 2;
	      }
	      else if (ptr1[1] == '#') /* Skip over \#...\# */
	      {
	        ptr1 += 2;
	        while ((*ptr1 != '\\') && (ptr1[1] != '#'))
	          *ptr2++ = *ptr1++;
	        ptr1 += 2;
	      }
	      else if ((ptr1[1] == 'l') && /* Skip over \link("...")...\# */
	               (ptr1[2] == 'i') &&
	               (ptr1[3] == 'n') &&
	               (ptr1[4] == 'k'))
	      {
	        ptr1 += 7; /* Skip over \link(" */
	        while ((*ptr1 != '"') && (ptr1[1] != ')'))
	          ptr1++;
	        ptr1 += 2;
	        while ((*ptr1 != '\\') && (ptr1[1] != '#'))
	          *ptr2++ = *ptr1++;
	        ptr1 += 2;
	      }
	      else if ((ptr1[1] == 'e') && /* End page on \end */
	               (ptr1[2] == 'n') &&
	               (ptr1[3] == 'd'))
	      {
	        *ptr2++ = '\n';
	        *ptr2++ = '\n';
	        *ptr1 = 0x00;
	        page_done = 1;
	      }
	      else
	      {
	        fprintf(stderr, "%s: choked on line: %s\n",
	                program_name, line);
	        exit(1);
	      }
	    }
	  } /* End of while (line not copied) */
	  *ptr2 = 0x00;
	  fputs(buf, stdout);
	} /* End of while (not end of page or file) */
} /* End of display_page() */

/* Compare page name and search string (full length) */
int compare_name(const void *page, const void *name)
{	
	struct page_record *page_ptr ;
	const char				 *name_ptr;
	
	page_ptr = (struct page_record *)page;
	name_ptr = (char *)name;

  return(strncmp(page_ptr->name, name_ptr, PAGENAME_SIZE));
} /* End of compare_name() */

/* Compare page name and search string (prefix) */
int compare_prefix(const void *page, const void *prefix)
{
	struct page_record *page_ptr ;
	const char				 *prefix_ptr;
	
	page_ptr = (struct page_record *)page;
	prefix_ptr = (char *)prefix;
	
  return(strncmp(page_ptr->name, prefix_ptr, strlen(prefix_ptr)));
} /* End of compare_prefix() */

void main(int argc, char *argv[])
{
  int counter = 1;

  program_name = argv[0];
  read_index();
  while (counter < argc)
    display_pages(argv[counter++]);
  exit(0);
} /* End of main() */
