/********************************************************************/
/* pureind.c - build index for a group of pure-c help source files  */
/*                                                                  */
/* Copyright (c) 1993 by Hildo Biersma - Evil Eye Software          */
/*                            e-mail: boender@dutiws.twi.tudelft.nl */
/*        Evil Eye Software - ``Software with a Purpose''           */
/*                                                                  */
/* Freeware - do with this what you like, but leave my name.        */
/********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "pureind.h"
#include "pure_lib.h"

/* Local function prototypes */
void index_file(const char *filename);
int  compare_pagenames(const void *page1, const void *page2);
void add_page_entry(const char *name, size_t pos);

/* Global variables */
struct file_record files[MAX_FILES];
struct page_record *pages = NULL;
long   no_files = 0, no_pages = 0, alloced_pages = 0;
char   *program_name;

/* Index a single pure-c help source file */
void index_file(const char *filename)
{
  char **header_names;
  
  if (open_source(filename) != 0)
    return;
  strncpy(files[no_files++].name, filename, FILENAME_SIZE);

  while ((header_names = get_next_header()) != NULL)
  {
    int counter = 0;
    size_t pos;

    while (header_names[counter] != NULL)
    {
    	pos = get_position();
      add_page_entry(header_names[counter++], pos);
     }
  }
  
  close_source();
} /* End of index_file() */

/* Compare two page names */
int compare_pagenames(const void *page1,
                      const void *page2)
{
  return(strncmp(((struct page_record *)page1)->name,
                 ((struct page_record *)page2)->name, PAGENAME_SIZE));
} /* End of compare_pagenames() */

/* Add a page entry */
void add_page_entry(const char *name, size_t position)
{
  char buf[PAGENAME_SIZE + 1];
  
  strncpy(buf, name, PAGENAME_SIZE);
  buf[PAGENAME_SIZE] = 0x00;
  strlwr(buf);
  
  if (alloced_pages == no_pages)
  {
    alloced_pages += 100;
    if ((pages = realloc(pages, sizeof(struct page_record) *
        alloced_pages)) == NULL)
    {
      fprintf(stderr, "%s: out of memory\n", program_name);
      exit(1);
    }
  }
  strncpy(pages[no_pages].name, buf, PAGENAME_SIZE);
  pages[no_pages].fileno = no_files - 1;
  pages[no_pages].pos = position;
  no_pages++;
} /* End of add_page_entry() */

void main(int argc, char *argv[])
{
  int  counter = 1;
  FILE *index;

  program_name = argv[0];
  if (argc > MAX_FILES + 1)
  {
    fprintf(stderr, "%s: too many files\n", program_name);
    exit(1);
  }
  
  while (counter < argc)
    index_file(argv[counter++]);

  /* Yes, this is a *lot* slower than sorting pointers to the records */
  qsort(pages, no_pages, sizeof(struct page_record), 
        compare_pagenames);

  if ((index = fopen(INDEXFILE_NAME, "wb")) == NULL)
  {
    fprintf(stderr, "%s: could not open file %s\n",
            program_name, INDEXFILE_NAME);
    exit(1);
  }

  fwrite(INDEXFILE_HEADER, sizeof(char), 4, index);
  fwrite(&no_files, 1, sizeof(long), index);
  fwrite(&no_pages, 1, sizeof(long), index);
  fwrite(files, no_files, sizeof(struct file_record), index);
  fwrite(pages, no_pages, sizeof(struct page_record), index);
  fclose(index);

  exit(0);
} /* End of main() */
