/********************************************************************/
/* pure_lib.c - library of support routines for programs processing */
/*              Pure-C help source files.                           */
/*                                                                  */
/* Copyright (c) 1993 by Hildo Biersma - Evil Eye Software          */
/*                            e-mail: boender@dutiws.twi.tudelft.nl */
/*        Evil Eye Software - ``Software with a Purpose''           */
/*                                                                  */
/* Freeware - do with this what you like, but leave my name.        */
/********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "pure_lib.h"

/* Local function prototypes */
static void dispose_names(void);
static void add_name(const char *page);
static void dispose_lines(void);
static void add_line(const char *line);

/* Locally defined constants */
#define STATE_SKIP    0
#define STATE_HEADER  1
#define STATE_BODY    2

/* Local variables */
static FILE *source = NULL;
static int  state = STATE_SKIP;
static int  names_alloced = 0;
static int  no_names = 0;
static char **names = NULL;
static int  lines_alloced = 0;
static int  no_lines = 0;
static char **lines = NULL;

/* Open a Pure-C help source file */
int open_source(const char *filename)
{
  if (source != NULL)
    fclose(source);
  if ((source = fopen(filename, "rt")) == NULL)
  {
    fprintf(stderr, "could not open file %s\n", filename);
    return(-1);
  }
  state = STATE_SKIP;
  return(0);
} /* End of open_source() */

/* Read in the current file; search for the next header and return */
/* the page names of the new page just found.                      */
char **get_next_header(void)
{
  char line[BUFSIZ];
  int  done = 0, paren_level = 0;

  if (source == NULL)
    return(NULL);
  
  dispose_names();
  
  fgets(line, BUFSIZ, source);
  while (!feof(source) && !done)
  {
    char *ptr1 = line, *ptr2;
    
    switch(state)
    {
      case STATE_SKIP:
        if (strstr(ptr1,  "screen") == NULL)
          break;
        ptr1 += strlen("screen");
        state = STATE_HEADER;
        /* FALL-THROUGH */
      case STATE_HEADER:
        while (*ptr1 != 0x00)
        {
          if (*ptr1 == '"')
          {
            if ((ptr2 = strchr(ptr1 + 1, '"')) == NULL)
            {
              fprintf(stderr, "error in source file\n");
              exit(1);
            }
            *ptr2 = 0x00;
            add_name(ptr1 + 1);
            ptr1 = ptr2 + 1;
          }
          else if (*ptr1 == '(')
          {
            paren_level += 1;
            ptr1 += 1;
          }
          else if (*ptr1 == ')')
          {
            paren_level -= 1;
            if (paren_level == 0)
            {
              state = STATE_BODY;
              done = 1;
              break;
            }
            ptr1 += 1;
          }
          else
            ptr1 += 1;
        }
        break;
      case STATE_BODY:
        if ((ptr1 = strstr(line, "\\end")) == NULL)
          break;
        if (ptr1 == line)
        {
          state = STATE_SKIP;
          break;
        }
        while ((ptr1 > line) && (*(ptr1 - 1) == '\\'))
          ptr1 -= 2;
        if ((ptr1 == line) || (*(ptr1 - 1) != '\\'))
          state = STATE_SKIP;
        break;
    } /* End of switch(state) */
    if (!done)
    	fgets(line, BUFSIZ, source);
  } /* End of while() */
  
  if (done)
    return(names);
  return(NULL);
} /* End of get_next_header() */

/* Get the position of the file pointer of the source file now open */
long get_position(void)
{
  if (source == NULL)
    return(0);
  else
    return(ftell(source));
} /* End of get_position() */

/* Return the body of the current page */
char **get_body(void)
{
  char line[BUFSIZ];
  int  done = 0;
  
  if ((source == NULL) || (state != STATE_BODY))
    return(NULL);

  dispose_lines();

  fgets(line, BUFSIZ, source);
  while (!feof(source) && !done)
  {
    char *ptr1 = line;
    
    if ((ptr1 = strstr(line, "\\end")) == NULL)
      add_line(line);
    else
    {
      if (ptr1 == line)
      {
        state = STATE_SKIP;
        add_line(line);
        done = 1;
      }
      else
      {
        char *ptr2 = ptr1;
        
        while ((ptr1 > line) && (*(ptr1 - 1) == '\\'))
          ptr1 -= 2;
        if ((ptr1 == line) || (*(ptr1 - 1) != '\\'))
        {
          state = STATE_SKIP;
          done = 1;
          *ptr2 = 0x00;
        }
        add_line(line);
      }
    } /* End of else (\\end in line) */
    if (!done)
    	fgets(line, BUFSIZ, source);
  } /* End of while() */
  
  if (done)
    return(lines);
  return(NULL);
} /* End of get_body() */

/* Close a Pure-C help source file and deallocate defined strings */
void close_source(void)
{
  if (source != NULL)
  {
    fclose(source);
    source = NULL;
    state = STATE_SKIP;
    dispose_names();
    dispose_lines();
  }
} /* End of close_source() */

/* Dispose of the contents of the names[] array (not the array itself) */
static void dispose_names(void)
{
  int counter = 0;
  
  while (counter < no_names)
  {
    free(names[counter]);
    names[counter++] = NULL;
  }
  no_names = 0; 
} /* End of dispose_names() */

/* Add a name to the names[] array */
static void add_name(const char *page)
{
  if (names_alloced - 1 <= no_names)
  {
    names_alloced += 20;
    if ((names = realloc(names, names_alloced * sizeof(char *))) == NULL)
    {
      fprintf(stderr, "out of memory\n");
      exit(1);
    }
  }
  
  names[no_names++] = strdup(page);
  names[no_names] = NULL;
} /* End of add_name() */

/* Dispose of the contents of the lines[] array (not the array itself) */
static void dispose_lines(void)
{
  int counter = 0;
  
  while (counter < no_lines)
  {
    free(lines[counter]);
    lines[counter++] = NULL;
  }
  no_lines = 0; 
} /* End of dispose_lines() */

/* Add a line to the lines[] array */
static void add_line(const char *line)
{
  if (lines_alloced - 1 <= no_lines)
  {
    lines_alloced += 20;
    if ((lines = realloc(lines, lines_alloced * sizeof(char *))) == NULL)
    {
      fprintf(stderr, "out of memory\n");
      exit(1);
    }
  }
  
  lines[no_lines++] = strdup(line);
  lines[no_lines] = NULL;
} /* End of add_line() */

