/*************************************************************************
 * Module contenant les fonctions de gestion sonore.                     *
 *************************************************************************/
#include <stdio.h>
#include <string.h>
#include <atari.h>
#include <tosbind.h>
#include <falcon30.h>
#include <stdio.h>
#include "audio.h"

/* Dfinition des adresses mmoire  utiliser pour le replay des samples */
#define	sndbase			((volatile short *)0xffff8900)
#define	sndbasehi		((volatile short *)0xffff8902)
#define	sndbasemid	((volatile short *)0xffff8904)
#define	sndbaselo		((volatile short *)0xffff8906)
#define	sndendhi		((volatile short *)0xffff890e)
#define	sndendmid		((volatile short *)0xffff8910)
#define	sndendlo		((volatile short *)0xffff8912)
#define	sndmode			((volatile short *)0xffff8920)
#define sndfreq			((volatile short *)0xffff8934)

/*************************************************************************/
void snd_wait(void);
void snd_kill(void);
void release_buffer(void);
void avr_file(SAMPLE *smp);
void snd_play(short *buf, long length, short rate, short mode);
void load_sample(SAMPLE *smp);

/* Routines  excuter avc Supexec */
long wait_sample(void);
long play_sound(void);

static short *sound_buf = (short *)NULL;
static avr_t avr_h;				/* global AVR header for sample loading/saving */

static SAMPLE *LstSample;														/* Liste des samples */
static SAMPLE *current_sound;												/* the current sound */

/*************************************************************************
 * Initialisation de la gestion des samples + vrouillage du circuit son *
 *************************************************************************/
int AudioInit(void)
{
	register int ret = SND_OK;
	
	if (Locksnd() < 0)
		ret = SND_LOCK;
	else
	 	LstSample = (SAMPLE *)NULL;

	return ret;
}

/*************************************************************************
 * On rend la mmoire au systme + dvrouillage du circuit son          *
 *************************************************************************/
void AudioExit(void)
{
	SAMPLE *TmpSample;

	while (LstSample	!= (SAMPLE *)NULL)
	{
		TmpSample = LstSample;
		LstSample = LstSample->suiv;
		free(TmpSample);
	}
	Unlocksnd();
}

/*************************************************************************
 * Ajout d'un nouveau sample dans la liste des samples.                  *
 *************************************************************************/
void AjouteSample(int type, char *nom, char *alias)
{
	SAMPLE *NewSample, *CurSample = LstSample;
	
	if (type != SMP_AVR)										/* Pour l'instant que les .Avr */
		return;

	NewSample = (SAMPLE *)malloc (sizeof(SAMPLE));
	if (NewSample)
	{
		strcpy (NewSample->name, nom);
		NewSample->suiv = (SAMPLE *)NULL;
		strcpy (NewSample->alias, alias);
		NewSample->alias[7] = '\0';															/* Au cas ou */
		avr_file(NewSample);

		if (CurSample == (SAMPLE *)NULL)
			LstSample = NewSample;
		else
		{
			while (CurSample->suiv != (SAMPLE *)NULL)
				CurSample = CurSample->suiv;
			CurSample->suiv = NewSample;
		}
	}
}

/*************************************************************************
 * Attend la fin du sample en cours...                                   *
 *************************************************************************/
void snd_wait(void)
{
	while((*sndbase) & 0xff);
}

/*************************************************************************
 * Arrt du sample                                                       *
 *************************************************************************/
void snd_kill(void)
{
	*sndbase = 0;
}

/*************************************************************************
 * release_buffer - check if a buffer is allocated and free it.          *
 *************************************************************************/
void release_buffer(void)
{
	if (sound_buf)
	{
		Mfree (sound_buf);
		sound_buf = NULL;
	}
}

/*************************************************************************
 * do_file - Traitemement d'un fichier son. Extrait la longueur et la    *
 * frquence dans la structure smp.                                      *
 *************************************************************************/
void avr_file(SAMPLE *smp)
{
	unsigned int mini = 50000, delta;
	int fd;
		
	fd = Fopen (smp->name, FO_READ);
	if (fd >= 0)
	{
/*		static unsigned short freqs[] = {0, 12517, 25033, 50066}; STE Mode*/
		static unsigned short freqs[][2] = {{8195,11}, {9834,9}, {12292,7}, {16390,5}, {19668,4}, {24585,3}, {32780,2}, {49170,1}};
		int i;

		smp->type = SMP_AVR;
		Fread (fd, sizeof(avr_h), &avr_h);
		smp->entete = (int)sizeof(avr_h);
		smp->length = avr_h.avr_length;
		/* place sentinel in buffer */
		smp->smp_name[sizeof(avr_h.avr_name) - 1] = '\0';
		strncpy(smp->smp_name, avr_h.avr_name, sizeof(avr_h.avr_name));
		/* did they hit the sentinel ? */
		if (smp->smp_name[sizeof(avr_h.avr_name) - 1] != '\0')
			strncpy(smp->smp_name + sizeof(avr_h.avr_name), avr_h.avr_xname, sizeof(avr_h.avr_xname));

		/*
		 * Approximation de la frquence stocke dans le Header
		 */
		for (i = 7; i >= 0; i--)
		{
			delta = (freqs[i][0] - (avr_h.avr_frequency & 0xffffff));
			delta = abs(delta);
			if (delta < mini)
			{
				mini = delta;
				smp->rate = freqs[i][1];
			}
		}
		smp->mode = (avr_h.avr_mode == AVR_MONO) ? 0x80 : 0x0;

		Fclose (fd);
	}
}

/*************************************************************************
 * Joue un sample...                                                     *
 *************************************************************************/
void snd_play(short *buf, long length, short rate, short mode)
{
	snd_kill();
	*sndbasehi  = (short)((unsigned long)buf>>16);
	*sndbasemid = (short)((unsigned long)buf>>8);
	*sndbaselo  = (short)buf;

	buf = (short *)((unsigned long)buf+length);
	*sndendhi   = (short)((unsigned long)buf>>16);
	*sndendmid  = (short)((unsigned long)buf>>8);
	*sndendlo   = (short)buf;
	
	*sndmode    = mode;	/* Setmode (MONO8/STEREO8/STEREO16) = Xbios(132) */
	*sndfreq		= rate;
	*sndbase    = 1;		/* start single play mode */
}

/*************************************************************************
 * load_sample - Charge un sample en mmoire.                            *
 *************************************************************************/
void load_sample(SAMPLE *smp)
{
	int fd = Fopen (smp->name, FO_READ);

	if (fd)
	{
		Fseek (smp->entete, fd, SEEK_SET);
		Fread (fd, smp->length, sound_buf); 
		Fclose (fd);
	}
}

/*************************************************************************
 * wait_sample - Attend l'arrt du Sampl en cours. Doit tre appel en   *
 * mode Superviseur avec Supexec().                                      *
 *************************************************************************/
long wait_sample(void)
{
	snd_wait();
	return 0L;
}

/*************************************************************************
 * play_sound - Joue le sample en cours. Elle doit tre excute avec    *
 * Supexec().                                                            *
 *************************************************************************/
long play_sound(void)
{
	snd_play (sound_buf, current_sound->length, current_sound->rate, current_sound->mode);
	return 0L;
}

/*************************************************************************
 * Rejout un chantillon sonore : fonction utilisateur.                  *
 *************************************************************************/
int PlaySample (char *alias)
{
	SAMPLE *smp = LstSample;
	int ret = TRUE;

	while (smp != (SAMPLE *)NULL && strcmp(smp->alias, alias) != 0)
		smp = smp->suiv;	

	if (smp != (SAMPLE *)NULL)
	{
		sound_buf = (short *)Malloc (smp->length);
		if ((long)sound_buf)
		{
			load_sample(smp);
			/*
			 * Play the sample, then wait for it to finish. The routines which
 			 * do this must be called at the Supervisor level as they play with
			 * the hardware.
		  */
			current_sound = smp;
    	
			Supexec (play_sound);
			Supexec (wait_sample);

			/*
			 * We must release the memory as quickly as possible so that there
			 * is no danger of it becoming lost as a result of a resolution
			 * change etc.
		  */
			release_buffer ();
		}
	}
	else
		ret = FALSE;
		
	return ret;
}

/*************************************************************************/
