\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth 
\ PROJECT     : CHForth, a Dutch public domain ANS Forth for PC's 
\ DESCRIPTION : Zilog Z80 assembler, disassembler and emulator 
\ CATEGORY    : Simulations 
\ AUTHOR      : Coos Haak 
\ LAST CHANGE : May 16, 1994, Coos Haak 
\ ----------------------------------------------------------------------



        NEEDS -assembler

        NEEDS -paragraphs

        NEEDS -stack

        MARKER -z80lib



warning off

hex

1000 segment trgseg

: ts:
        trgseg @ swap ;

: c@-t
        ts: c@x ;

: c!-t
        ts: c!x ;

: @-t
        dup c@-t swap 1+ c@-t join ;

: !-t
        >r split r@ 1+ c!-t r> c!-t ;

: dump-t
        bounds 0 local counter
        ?do     cr i h. space i 10 bounds
                do      i c@-t b. counter 1+ dup to counter 3 and 0=
                        if      space
                        then
                loop
                trgseg @ i 10 stypex stop? ?leave
        10 +loop
        ;

: b>s
        dup 80 and
        if      FF00 or
        then ;

0 value adres

vector .label   ' h. is .label
vector label?   ' 0= is label?

' and alias and'
' or alias or'
' xor alias xor'

vocabulary z80dis

z80dis definitions

0 value opcode
0 value offset

: nextb
        adres c@-t 1 +to adres ;

: shortlabel
        adres c@-t adres 1+ + label?
        if      adres c@-t adres 1+ + .label
        else    adres c@-t b>s adres 1+ + s>d
                push base hex
                <# # # # # #> type
                pop base
        then
        1 +to adres ;

: .byte
        nextb push base hex 0 <# # # #> type pop base ;

: nextw
        adres @-t 2 +to adres ;

: longlabel
        adres @-t label?
        if      adres @-t .label
        else    adres @-t push base hex 0 <# # # # # #> type pop base
        then
        2 +to adres ;

: ..r
        7 and'
        case
                0 of    ." B"   endof
                1 of    ." C"   endof
                2 of    ." D"   endof
                3 of    ." E"   endof
                4 of    ." H"   endof
                5 of    ." L"   endof
                6 of    ." (HL)"        endof
                7 of    ." A"   endof
        endcase ;

: ..r0
        opcode 7 and ..r ;

: ..r1
        opcode 3 rshift 7 and ..r ;

: ..x
        opcode 3 rshift 6 and
        case
                0 of    ." BC"  endof
                2 of    ." DE"  endof
                4 of    ." HL"  endof
                6 of    ." SP"  endof
        endcase ;

: ..p
        opcode 3 rshift 6 and
        case
                0 of    ." BC"  endof
                2 of    ." DE"  endof
                4 of    ." HL"  endof
                6 of    ." AF"  endof
        endcase ;

: self.l
        create
        does>   body> >head (.head) tuck type 8 swap - spaces ;

: .self
        create
        does>   body> >head .head ;

: comma
        ',' emit ;

\ 00

.self nop       self.l ld       self.l inc      self.l dec
.self rlca      self.l ex       self.l add      .self rrca

: .lxi
        ld ..x comma longlabel ;

: .stax
        ld '(' emit ..x ." ),A" ;

: .inx
        inc ..x ;

: .inr
        inc ..r1 ;

: .dcr
        dec ..r1 ;

: .mvi
        ld ..r1 comma nextb b. ;

: .exaf
        ex ." AF,A'F'" ;

: .dad
        add ." HL," ..x ;

: .ldax
        ld ." A,(" ..x ')' emit ;

: .dcx
        dec ..x ;

\ 10

self.l djnz     .self rla       self.l jr       .self rra

: .djnz
        djnz shortlabel ;

: .jr
        jr shortlabel ;

\ 20

.self daa       .self cpl

: .jrnz
        jr ." NZ," shortlabel ;

: .shld
        ld '(' emit longlabel ." ),HL" ;

: .jrz
        jr ." Z," shortlabel ;

: .lhld
        ld ." HL,(" longlabel ')' emit ;

\ 30

.self scf       .self ccf

: .jrnc
        jr ." NC," shortlabel ;

: .sta
        ld '(' emit longlabel ." ),A" ;

: .jrc
        jr ." C," shortlabel ;

: .lda
        ld ." A,(" longlabel ')' emit ;

\ 40

.self halt

: .mov
        ld ..r1 comma ..r0 ;

\ 80

self.l adc      self.l sub      self.l sbc
self.l and      self.l xor      self.l or       self.l cp

: .add
        add ." A," ..r0 ;

: .adc
        adc ." A," ..r0 ;

: .sub
        sub ..r0 ;

: .sbb
        sbc ." A," ..r0 ;

: .ana
        and ..r0 ;

: .xra
        xor ..r0 ;

: .ora
        or ..r0 ;

: .cmp
        cp ..r0 ;

self.l rlc      self.l rrc      self.l rl       self.l rr
self.l sla      self.l sra      self.l srl      .self ???       self.l !!!

: cb0s
        opcode 3 rshift 7 and'
        case
                0 of    rlc ..r0        endof
                1 of    rrc ..r0        endof
                2 of    rl ..r0 endof
                3 of    rr ..r0 endof
                4 of    sla ..r0        endof
                5 of    sra ..r0        endof
                6 of    ???     endof
                7 of    srl ..r0        endof
        endcase ;

: .xbyte
        ." (IX" offset b>s ?dup
        if      dup 0>
                if      '+' emit
                then
                push base decimal 0 .r pop base
        then
        ')' emit ;

: .ybyte
        ." (IY" offset b>s ?dup
        if      dup 0>
                if      '+' emit
                then
                push base decimal 0 .r pop base
        then
        ')' emit ;

: ddcb0s
        opcode 3 rshift 7 and'
        case
                0 of    rlc     endof
                1 of    rrc     endof
                2 of    rl      endof
                3 of    rr      endof
                4 of    sla     endof
                5 of    sra     endof
                6 of    !!!     endof
                7 of    srl     endof
        endcase
        .xbyte ;

: fdcb0s
        opcode 3 rshift 7 and'
        case
                0 of    rlc     endof
                1 of    rrc     endof
                2 of    rl      endof
                3 of    rr      endof
                4 of    sla     endof
                5 of    sra     endof
                6 of    !!!     endof
                7 of    srl     endof
        endcase
        .ybyte ;

self.l bit      self.l res      self.l set

: .bits
        opcode 3 rshift 7 and' '0' + emit comma ;

: .cbs
        nextb dup to opcode 6 rshift
        case
                0 of    cb0s    endof
                1 of    bit .bits ..r0  endof
                2 of    res .bits ..r0  endof
                3 of    set .bits ..r0  endof
        endcase ;

: ddcbs
        nextb to offset nextb to opcode
        opcode 6 rshift
        case
                0 of    ddcb0s  endof
                1 of    bit .bits .xbyte        endof
                2 of    res .bits .xbyte        endof
                3 of    set .bits .xbyte        endof
        endcase
        opcode 7 and' 6 <>
        if      ???
        then ;

: fdcbs
        nextb to offset nextb to opcode
        opcode 6 rshift
        case
                0 of    fdcb0s  endof
                1 of    bit .bits .ybyte        endof
                2 of    res .bits .ybyte        endof
                3 of    set .bits .ybyte        endof
        endcase
        opcode 7 and' 6 <>
        if      ???
        then ;

self.l ret      self.l pop      self.l jp       self.l call
self.l push     self.l rst

\ C0

: .rnz
        ret ." NZ" ;

: .pop
        pop ..p ;

: .jnz
        jp ." NZ," longlabel ;

: .jmp
        jp longlabel ;

: .cnz
        call ." NZ," longlabel ;

: .push
        push ..p ;

: .adi
        add ." A," .byte ;

: .rst
        rst opcode %00111000 and' . ;

: .rz
        ret 'Z' emit ;

: .jz
        jp ." Z," longlabel ;

: .cz
        call ." Z," longlabel ;

: .call
        call longlabel ;

: .aci
        adc ." A," .byte ;

: o.xbyte
        nextb to offset .xbyte ;

: .dds
        nextb
        case
                09 of   add ." IX,BC"   endof
                19 of   add ." IX,DE"   endof
                21 of   ld ." IX," longlabel    endof
                22 of   ld '(' emit longlabel ." ),IX"  endof
                23 of   inc ." IX"      endof
                29 of   add ." IX,IX"   endof
                2A of   ld ." IX,(" longlabel ')' emit  endof
                2B of   dec ." IX"      endof
                34 of   inc o.xbyte     endof
                35 of   dec o.xbyte     endof
                36 of   ld o.xbyte ',' emit .byte       endof
                39 of   add ." IX,SP"   endof
                46 of   ld ." B," o.xbyte       endof
                4E of   ld ." C," o.xbyte       endof
                56 of   ld ." D," o.xbyte       endof
                5E of   ld ." E," o.xbyte       endof
                66 of   ld ." H," o.xbyte       endof
                6E of   ld ." L," o.xbyte       endof
                70 of   ld o.xbyte ." ,B"       endof
                71 of   ld o.xbyte ." ,C"       endof
                72 of   ld o.xbyte ." ,D"       endof
                73 of   ld o.xbyte ." ,E"       endof
                74 of   ld o.xbyte ." ,H"       endof
                75 of   ld o.xbyte ." ,L"       endof
                77 of   ld o.xbyte ." ,A"       endof
                7E of   ld ." A," o.xbyte       endof
                86 of   add ." A," o.xbyte      endof
                8E of   adc ." A," o.xbyte      endof
                96 of   sub o.xbyte     endof
                9E of   sbc ." A," o.xbyte      endof
                A6 of   and o.xbyte     endof
                AE of   xor o.xbyte     endof
                B6 of   or o.xbyte      endof
                BE of   cp o.xbyte      endof
                CB of   ddcbs   endof
                E1 of   pop ." IX"      endof
                E3 of   ex ." (SP),IX"  endof
                E5 of   push ." IX"     endof
                E9 of   jp ." (IX)"     endof
                F9 of   ld ." SP,IX"    endof
                ???
        endcase ;

\ D0

self.l out      .self exx       self.l in

: .rnc
        ret ." NC" ;

: .jnc
        jp ." NC," longlabel ;

: .cnc
        call ." NC," longlabel ;

: .sui
        sub .byte ;

: .rc
        ret 'C' emit ;

: .jc
        jp ." C," longlabel ;

: .out
        out '(' emit .byte ." ),A" ;

: .in
        in ." A,(" .byte ')' emit ;

: .cc
        call ." C," longlabel ;

: .sbi
        sbc ." A," .byte ;

.self neg       .self retn      self.l im       .self reti
.self rrd       .self rld
.self ldi       .self cpi       .self ini       .self outi
.self ldd       .self cpd       .self ind       .self outd
.self ldir      .self cpir      .self inir      .self otir
.self lddr      .self cpdr      .self indr      .self otdr

: .eds
        nextb
        case
                40 of   in ." B,(C)"    endof
                41 of   out ." (C),B"   endof
                42 of   sbc ." HL,BC"   endof
                43 of   ld '(' emit longlabel ." ),BC"  endof
                44 of   neg     endof
                45 of   retn    endof
                46 of   im '0' emit     endof
                47 of   ld ." I,A"      endof
                48 of   in ." C,(C)"    endof
                49 of   out ." (C),C"   endof
                4A of   adc ." HL,BC"   endof
                4B of   ld ." BC,(" longlabel ')' emit  endof
                4D of   reti    endof
                4F of   ld ." R,A"      endof
                50 of   in ." D,(C)"    endof
                51 of   out ." (C),D"   endof
                52 of   sbc ." HL,DE"   endof
                53 of   ld '(' emit longlabel ." ),DE"  endof
                56 of   im '1' emit     endof
                57 of   ld ." A,I"      endof
                58 of   in ." E,(C)"    endof
                59 of   out ." (C),E"   endof
                5A of   adc ." HL,DE"   endof
                5B of   ld ." DE,(" longlabel ')' emit  endof
                5E of   im '2' emit     endof
                5F of   ld ." A,R"      endof
                60 of   in ." H,(C)"    endof
                61 of   out ." (C),H"   endof
                62 of   sbc ." HL,HL"   endof
                63 of   ld '(' emit longlabel ." ),HL"  endof
                67 of   rrd     endof
                68 of   in ." L,(C)"    endof
                69 of   out ." (C),L"   endof
                6A of   adc ." HL,HL"   endof
                6B of   ld ." HL,(" longlabel ')' emit  endof
                6F of   rld     endof
                70 of   in ." F,(C)"    endof
                72 of   sbc ." HL,SP"   endof
                73 of   ld '(' emit longlabel ." ),SP"  endof
                78 of   in ." A,(C)"    endof
                79 of   out ." (C),A"   endof
                7A of   adc ." HL,SP"   endof
                7B of   ld ." SP,(" longlabel ')' emit  endof
                A0 of   ldi     endof
                A1 of   cpi     endof
                A2 of   ini     endof
                A3 of   outi    endof
                A8 of   ldd     endof
                A9 of   cpd     endof
                AA of   ind     endof
                AB of   outd    endof
                B0 of   ldir    endof
                B1 of   cpir    endof
                B2 of   inir    endof
                B3 of   otir    endof
                B8 of   lddr    endof
                B9 of   cpdr    endof
                BA of   indr    endof
                BB of   otdr    endof
                ???
        endcase ;

\ E0

: .rpo
        ret ." PO" ;

: .jpo
        jp ." PO," longlabel ;

: .xthl
        ex ." (SP),HL" ;

: .pchl
        jp ." (HL)" ;

: .cpo
        call ." PO," longlabel ;

: .ani
        and .byte ;

: .rpe
        ret ." PE" ;

: .jpe
        jp ." PE," longlabel ;

: .xchg
        ex ." DE,HL" ;

: .cpe
        call ." PE," longlabel ;

: .xri
        xor .byte ;

: o.ybyte
        nextb to offset .ybyte ;

: .fds
        nextb
        case
                09 of   add ." IY,BC"   endof
                19 of   add ." IY,DE"   endof
                21 of   ld ." IY," longlabel    endof
                22 of   ld '(' emit longlabel ." ),IY"  endof
                23 of   inc ." IY"      endof
                29 of   add ." IY,IY"   endof
                2A of   ld ." IY,(" longlabel ')' emit  endof
                2B of   dec ." IY"      endof
                34 of   inc o.ybyte     endof
                35 of   dec o.ybyte     endof
                36 of   ld o.ybyte ',' emit .byte       endof
                39 of   add ." IY,SP"   endof
                46 of   ld ." B," o.ybyte       endof
                4E of   ld ." C," o.ybyte       endof
                56 of   ld ." D," o.ybyte       endof
                5E of   ld ." E," o.ybyte       endof
                66 of   ld ." H," o.ybyte       endof
                6E of   ld ." L," o.ybyte       endof
                70 of   ld o.ybyte ." ,B"       endof
                71 of   ld o.ybyte ." ,C"       endof
                72 of   ld o.ybyte ." ,D"       endof
                73 of   ld o.ybyte ." ,E"       endof
                74 of   ld o.ybyte ." ,H"       endof
                75 of   ld o.ybyte ." ,L"       endof
                77 of   ld o.ybyte ." ,A"       endof
                7E of   ld ." A," o.ybyte       endof
                86 of   add ." A," o.ybyte      endof
                8E of   adc ." A," o.ybyte      endof
                96 of   sub o.ybyte     endof
                9E of   sbc ." A," o.ybyte      endof
                A6 of   and o.ybyte     endof
                AE of   xor o.ybyte     endof
                B6 of   or o.ybyte      endof
                BE of   cp o.ybyte      endof
                CB of   fdcbs   endof
                E1 of   pop ." IY"      endof
                E3 of   ex ." (SP),IY"  endof
                E5 of   push ." IY"     endof
                E9 of   jp ." (IY)"     endof
                F9 of   ld ." SP,IY"    endof
                ???
        endcase ;

.self di        .self ei

\ F0

: .rp
        ret ." P" ;

: .jp
        jp ." P," longlabel ;

: .cp
        call ." P," longlabel ;

: .ori
        or .byte ;

: .rm
        ret ." M" ;

: .sphl
        ld ." SP,HL" ;

: .jm
        jp ." M," longlabel ;

: .cm
        call ." M," longlabel ;

: .cpi
        cp .byte ;

create opcode-table
( 00 )  ' nop ,         ' .lxi ,        ' .stax ,       ' .inx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' rlca ,
        ' .exaf ,       ' .dad ,        ' .ldax ,       ' .dcx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' rrca ,
( 10 )  ' .djnz ,       ' .lxi ,        ' .stax ,       ' .inx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' rla ,
        ' .jr ,         ' .dad ,        ' .ldax ,       ' .dcx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' rra ,
( 20 )  ' .jrnz ,       ' .lxi ,        ' .shld ,       ' .inx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' daa ,
        ' .jrz ,        ' .dad ,        ' .lhld ,       ' .dcx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' cpl ,
( 30 )  ' .jrnc ,       ' .lxi ,        ' .sta ,        ' .inx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' scf ,
        ' .jrc ,        ' .dad ,        ' .lda ,        ' .dcx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' ccf ,
( 40 )  ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
( 50 )  ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
( 60 )  ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
( 70 )  ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' halt ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
( 80 )  ' .add ,        ' .add ,        ' .add ,        ' .add ,
        ' .add ,        ' .add ,        ' .add ,        ' .add ,
        ' .adc ,        ' .adc ,        ' .adc ,        ' .adc ,
        ' .adc ,        ' .adc ,        ' .adc ,        ' .adc ,
( 90 )  ' .sub ,        ' .sub ,        ' .sub ,        ' .sub ,
        ' .sub ,        ' .sub ,        ' .sub ,        ' .sub ,
        ' .sbb ,        ' .sbb ,        ' .sbb ,        ' .sbb ,
        ' .sbb ,        ' .sbb ,        ' .sbb ,        ' .sbb ,
( A0 )  ' .ana ,        ' .ana ,        ' .ana ,        ' .ana ,
        ' .ana ,        ' .ana ,        ' .ana ,        ' .ana ,
        ' .xra ,        ' .xra ,        ' .xra ,        ' .xra ,
        ' .xra ,        ' .xra ,        ' .xra ,        ' .xra ,
( B0 )  ' .ora ,        ' .ora ,        ' .ora ,        ' .ora ,
        ' .ora ,        ' .ora ,        ' .ora ,        ' .ora ,
        ' .cmp ,        ' .cmp ,        ' .cmp ,        ' .cmp ,
        ' .cmp ,        ' .cmp ,        ' .cmp ,        ' .cmp ,
( C0 )  ' .rnz ,        ' .pop ,        ' .jnz ,        ' .jmp ,
        ' .cnz ,        ' .push ,       ' .adi ,        ' .rst ,
        ' .rz ,         ' ret ,         ' .jz ,         ' .cbs ,
        ' .cz ,         ' .call ,       ' .aci ,        ' .rst ,
( D0 )  ' .rnc ,        ' .pop ,        ' .jnc ,        ' .out ,
        ' .cnc ,        ' .push ,       ' .sui ,        ' .rst ,
        ' .rc ,         ' exx ,         ' .jc ,         ' .in ,
        ' .cc ,         ' .dds ,        ' .sbi ,        ' .rst ,
( E0 )  ' .rpo ,        ' .pop ,        ' .jpo ,        ' .xthl ,
        ' .cpo ,        ' .push ,       ' .ani ,        ' .rst ,
        ' .rpe ,        ' .pchl ,       ' .jpe ,        ' .xchg ,
        ' .cpe ,        ' .eds ,        ' .xri ,        ' .rst ,
( F0 )  ' .rp ,         ' .pop ,        ' .jp ,         ' di ,
        ' .cp ,         ' .push ,       ' .ori ,        ' .rst ,
        ' .rm ,         ' .sphl ,       ' .jm ,         ' ei ,
        ' .cm ,         ' .fds ,        ' .cpi ,        ' .rst ,

: all-opcodes
        nextb dup to opcode opcode-table []cell @ execute ;

forth definitions

: .inst
        [ z80dis ]
        cr adres h. space adres label?
        if      adres .label
        then
        #16 htab adres >r all-opcodes
        #40 htab r> space adres over - 2dup bounds
        do      i c@-t b.
        loop
        space trgseg @ -rot #64 htab stypex space
        eol ;    forth

: disasm
        to adres
        begin   .inst key ^[ =
        until ;

vocabulary z80sim

z80sim definitions

: byte
        create  random c,
        does>   c@ ;

methods byte

: to
        postpone literal postpone c!
        ;

: +to
        postpone literal postpone c+!
        ;

: clear
        postpone literal postpone c0!
        ;

end-methods

create flags    0 c,

0 value intflag

: carry@
        1 flags c@ and' ;

: carry!
        0<> 1 and' flags c@ %11111110 and' or' flags c! ;

code (flags)
                add     bl, # 0
                pushf
                pop     bx
                and     bx, # %11111110
                next    end-code

: flags!
        (flags) flags c@ 1 and' or' flags c! ;

: zero@
        %01000000 flags c@ and' ;

: half@
        %00010000 flags c@ and' ;

: sign@
        %10000000 flags c@ and' ;

: parity@
        %00000100 flags c@ and' ;

0 value %x

: >x
        %x >s ;

: x>
        s> to %x ;

0 value pc
0 value sp
byte opcode
byte _b
byte _c
byte _d
byte _e
byte _h
byte _l
byte _a
0 value _af'
0 value _bc'
0 value _de'
0 value _hl'
0 value _ix
0 value _iy

: _bc
        _c _b join ;

: _bc!
        split to _b to _c ;

: _de
        _e _d join ;

: _de!
        split to _d to _e ;

: _hl
        %x clear %x
        dup 1 =
        if      drop _ix exit
        then
        2 =
        if      _iy exit
        then
        _l _h join ;

: _hl!
        %x clear %x
        dup 1 =
        if      drop to _ix exit
        then
        2 =
        if      to _iy exit
        then
        split to _h to _l ;

: @inlw
        pc @-t 2 +to pc ;

: @inlb
        pc c@-t 1 +to pc ;

: _hlc@
        %x clear %x dup 1 =
        if      drop _ix @inlb b>s + c@-t exit
        then
        2 =
        if      _iy @inlb b>s + c@-t exit
        then
        _l _h join c@-t ;

: _hlc!
        %x clear %x dup 1 =
        if      drop _ix @inlb b>s + c!-t exit
        then
        2 =
        if      _iy @inlb b>s + c!-t exit
        then
        _l _h join c!-t ;

: get-arg
        7 and'
        case
                0 of    _b      endof
                1 of    _c      endof
                2 of    _d      endof
                3 of    _e      endof
                4 of    _h      endof
                5 of    _l      endof
                6 of    _hlc@   endof
                7 of    _a      endof
        endcase ;

: put-arg
        7 and'
        case
                0 of    to _b   endof
                1 of    to _c   endof
                2 of    to _d   endof
                3 of    to _e   endof
                4 of    to _h   endof
                5 of    to _l   endof
                6 of    _hlc!   endof
                7 of    to _a   endof
        endcase ;

: ???
        click ;

: nop ;

: ldp#
        @inlw swap 3 and'
        case
                0 of    _bc!    endof
                1 of    _de!    endof
                2 of    _hl!    endof
                3 of    to sp   endof
        endcase ;

: stap
        _a swap 1 and'
        case
                0 of    _bc     endof
                1 of    _de     endof
        endcase
        c!-t ;

: inc
        3 and'
        case
                0 of    _bc 1+ _bc!     endof
                1 of    _de 1+ _de!     endof
                2 of    >x _hl 1+ x> _hl!       endof
                3 of    1 +to sp        endof
        endcase ;

: inr
        7 and'
        case
                0 of    _b 1+ dup to _b endof
                1 of    _c 1+ dup to _c endof
                2 of    _d 1+ dup to _d endof
                3 of    _e 1+ dup to _e endof
                4 of    _h 1+ dup to _h endof
                5 of    _l 1+ dup to _l endof
                6 of    %x clear %x dup 1 =
                        if      drop _ix @inlb b>s +
                        else    2 =
                                if      _iy @inlb b>s +
                                else    _l _h join
                                then
                        then
                        dup c@-t 1+ dup rot c!-t        endof
                7 of    _a 1+ dup to _a endof
        endcase
        split carry! flags! ;

: der
        7 and'
        case
                0 of    _b 1- dup to _b endof
                1 of    _c 1- dup to _c endof
                2 of    _d 1- dup to _d endof
                3 of    _e 1- dup to _e endof
                4 of    _h 1- dup to _h endof
                5 of    _l 1- dup to _l endof
                6 of    %x clear %x dup 1 =
                        if      drop _ix @inlb b>s +
                        else    2 =
                                if      _iy @inlb b>s +
                                else    _l _h join
                                then
                        then
                        dup c@-t 1- dup rot c!-t        endof
                7 of    _a 1- dup to _a endof
        endcase
        split carry! flags! ;

: ld#
        7 and' dup 6 =
        if      drop %x clear %x dup 1 =
                if      drop _ix @inlb b>s +
                else    2 =
                        if      _iy @inlb b>s +
                        else    _l _h join
                        then
                then
                @inlb swap c!-t exit
        then
        @inlb swap put-arg ;

: rlc
        2* carry@ or' split carry! dup flags! ;

: rlca
        _a rlc to _a ;

: exaf
        flags c@ _a join _af' split to _a flags c! to _af' ;

: addp
        >x 3 and'
        case
                0 of    _bc     endof
                1 of    _de     endof
                2 of    _hl     endof
                3 of    sp      endof
        endcase
        x> >x _hl + x> _hl! ;

: ldap
        1 and'
        case
                0 of    _bc     endof
                1 of    _de     endof
        endcase
        c@-t to _a ;

: dec
        3 and'
        case
                0 of    _bc 1- _bc!     endof
                1 of    _de 1- _de!     endof
                2 of    >x _hl 1- x> _hl!       endof
                3 of    -1 +to sp       endof
        endcase ;

: rrc
        carry@ join 7 lshift split tuck flags! 0<> carry! ;

: rrca
        _a rrc to _a ;

: 00s
        $F and'
        case
                0 of    nop     endof
                1 of    0 ldp#  endof
                2 of    0 stap  endof
                3 of    0 inc   endof
                4 of    0 inr   endof
                5 of    0 der   endof
                6 of    0 ld#   endof
                7 of    rlca    endof
                8 of    exaf    endof
                9 of    0 addp  endof
                $A of   0 ldap  endof
                $B of   0 dec   endof
                $C of   1 inr   endof
                $D of   1 der   endof
                $E of   1 ld#   endof
                $F of   rrca    endof
        endcase ;

: jr
        @inlb b>s pc + to pc ;

: djnz
        _b 1- dup to _b 0=
        if      1 +to pc exit
        then
        jr ;

: rl
        dup join 7 rshift split carry! dup flags! ;

: rla
        _a rl to _a ;

: rr
        dup join 7 lshift split tuck flags! 0<> carry! ;

: rra
        _a rr to _a ;

: 10s
        $F and'
        case
                0 of    djnz    endof
                1 of    1 ldp#  endof
                2 of    1 stap  endof
                3 of    1 inc   endof
                4 of    2 inr   endof
                5 of    2 der   endof
                6 of    2 ld#   endof
                7 of    rla     endof
                8 of    jr      endof
                9 of    1 addp  endof
                $A of   1 ldap  endof
                $B of   1 dec   endof
                $C of   3 inr   endof
                $D of   3 der   endof
                $E of   3 ld#   endof
                $F of   rra     endof
        endcase ;

: jrnz
        zero@
        if      1 +to pc exit
        then
        jr ;

: sthl
        _hl @inlw !-t ;

: daa
        click ;

: jrz
        zero@ 0=
        if      1 +to pc exit
        then
        jr ;

: ldhl
        @inlw @-t _hl! ;

: cpl
        _a invert dup to _a flags! 0 carry! ;

: 20s
        $F and'
        case
                0 of    jrnz    endof
                1 of    2 ldp#  endof
                2 of    sthl    endof
                3 of    2 inc   endof
                4 of    4 inr   endof
                5 of    4 der   endof
                6 of    4 ld#   endof
                7 of    daa     endof
                8 of    jrz     endof
                9 of    2 addp  endof
                $A of   ldhl    endof
                $B of   2 dec   endof
                $C of   5 inr   endof
                $D of   5 der   endof
                $E of   5 ld#   endof
                $F of   cpl     endof
        endcase ;

: jrnc
        carry@
        if      1 +to pc exit
        then
        jr ;

: sta
        _a @inlw c!-t ;

: scf
        1 carry! ;

: jrc
        carry@ 0=
        if      1 +to pc exit
        then
        jr ;

: lda
        @inlw c@-t to _a ;

: ccf
        carry@ invert carry! ;

: 30s
        $F and'
        case
                0 of    jrnc    endof
                1 of    3 ldp#  endof
                2 of    sta     endof
                3 of    3 inc   endof
                4 of    6 inr   endof
                5 of    6 der   endof
                6 of    6 ld#   endof
                7 of    scf     endof
                8 of    jrc     endof
                9 of    3 addp  endof
                $A of   lda     endof
                $B of   3 dec   endof
                $C of   7 inr   endof
                $D of   7 der   endof
                $E of   7 ld#   endof
                $F of   ccf     endof
        endcase ;

: 00s-30s
        dup 4 rshift 3 &exec: 00s 10s 20s 30s ;

: 40s
        dup 76 =
        if      drop click exit
        then
        8 /mod 7 and' swap get-arg swap put-arg ;

: add
        get-arg _a + split carry! dup to _a flags! ;

: adc
        get-arg _a carry@ + + split carry! dup to _a flags! ;

: sub
        get-arg _a swap - split carry! dup to _a flags! ;

: sbc
        get-arg _a swap - carry@ - split carry! dup to _a flags! ;

: and
        get-arg _a and' dup to _a flags! ;

: xor
        get-arg _a xor' dup to _a flags! ;

: or
        get-arg _a or' dup to _a flags! ;

: cp
        get-arg _a swap - split carry! flags! ;

: 80s
        dup 3 rshift 7 &exec: add adc sub sbc and xor or cp ;

: sla
        2* split carry! dup flags! ;

: sra
        flip 2/ split tuck flags! carry! ;

: srl
        flip u2/ split tuck flags! carry! ;

: rotates
        case
                0 of    rlc     endof
                1 of    rrc     endof
                2 of    rl      endof
                3 of    rr      endof
                4 of    sla     endof
                5 of    sra     endof
                6 of    ???     endof
                7 of    sla     endof
        endcase ;

: bit   ( val n -- )
        1 swap lshift
        and' 0= %01000000 and' flags c@ %10111111 and' or' flags c! ;

: set   ( val1 n -- val2 )
        1 swap lshift or' ;

: res   ( val1 n -- val2 )
        1 swap lshift invert and' ;

: cbs
        %x clear %x dup 1 =
        if      drop @inlb b>s _ix + dup c@-t @inlb $40 /mod
                case
                        0 of 8 / rotates swap c!-t      endof
                        1 of 8 / bit drop       endof
                        2 of 8 / res swap c!-t  endof
                        3 of 8 / set swap c!-t  endof
                endcase
                exit
        then
        2 =
        if      @inlb b>s _iy + dup c@-t @inlb $40 /mod
                case
                        0 of 8 / rotates swap c!-t      endof
                        1 of 8 / bit drop       endof
                        2 of 8 / res swap c!-t  endof
                        3 of 8 / set swap c!-t  endof
                endcase
                exit
        then
        @inlb $40 /mod
        case
                0 of    8 /mod over get-arg swap rotates swap put-arg   endof
                1 of    8 /mod swap get-arg swap bit    endof
                2 of    8 /mod tuck get-arg swap res swap put-arg       endof
                3 of    8 /mod tuck get-arg swap set swap put-arg       endof
        endcase ;

: ret
        sp @-t to pc 2 +to sp ;

: retnz
        zero@ 0=
        if      ret
        then ;

: pop
        sp @-t 2 +to sp swap 3 and'
        case
                0 of    _bc!    endof
                1 of    _de!    endof
                2 of    _hl!    endof
                3 of    split to _a flags c!    endof
        endcase ;

: jp
        @inlw to pc ;

: jpnz
        zero@ 0=
        if      jp exit
        then
        2 +to pc ;

: call
        @inlw -2 +to sp pc sp !-t to pc ;

: callnz
        zero@ 0=
        if      call exit
        then
        2 +to pc ;

: push
        3 and'
        case
                0 of    _bc     endof
                1 of    _de     endof
                2 of    _hl     endof
                3 of    flags c@ _a join        endof
        endcase
        -2 +to sp sp !-t ;

: add#
        _a @inlb + split carry! dup to _a flags! ;

: rst
        -2 +to sp pc sp !-t %00111000 and' to pc ;

: retz
        zero@
        if      ret
        then ;

: jpz
        zero@
        if      jp exit
        then
        2 +to pc ;

: callz
        zero@
        if      call exit
        then
        2 +to pc ;

: adc#
        _a @inlb + carry@ + split carry! dup to _a flags! ;

: c0s
        $F and'
        case
                0 of    retnz   endof
                1 of    0 pop   endof
                2 of    jpnz    endof
                3 of    jp      endof
                4 of    callnz  endof
                5 of    0 push  endof
                6 of    add#    endof
                7 of    opcode rst      endof
                8 of    retz    endof
                9 of    ret     endof
                $A of   jpz     endof
                $B of   cbs     endof
                $C of   callz   endof
                $D of   call    endof
                $E of   adc#    endof
                $F of   opcode rst      endof
        endcase ;

: retnc
        carry@ 0=
        if      ret
        then ;

: jpnc
        carry@ 0=
        if      jp exit
        then
        2 +to pc ;

: out
        1 +to pc _a b. ;

: callnc
        carry@ 0=
        if      call exit
        then
        2 +to pc ;

: sub#
        _a @inlb - split carry! dup to _a flags! ;

: retc
        carry@
        if      ret
        then ;

: exx
        _bc _bc' _bc! to _bc'
        _de _de' _de! to _de'
        _hl _hl' _hl! to _hl' ;

: jpc
        carry@
        if      jp exit
        then
        2 +to pc ;

: in
        1 +to pc key to _a ;

: callc
        carry@
        if      call exit
        then
        2 +to pc ;

: sbc#
        _a @inlb - carry@ - split carry! dup to _a flags! ;

: d0s
        $F and'
        case
                0 of    retnc   endof
                1 of    1 pop   endof
                2 of    jpnc    endof
                3 of    out     endof
                4 of    callnc  endof
                5 of    1 push  endof
                6 of    sub#    endof
                7 of    opcode rst      endof
                8 of    retc    endof
                9 of    exx     endof
                $A of   jpc     endof
                $B of   in      endof
                $C of   callc   endof
                $D of   ???     endof
                $E of   sbc#    endof
                $F of   opcode rst      endof
        endcase ;

: retpo
        parity@ 0=
        if      ret
        then ;

: jppo
        parity@ 0=
        if      jp exit
        then
        2 +to pc ;

: exsp
        sp @-t >x _hl sp !-t x> _hl! ;

: callpo
        parity@ 0=
        if      call exit
        then
        2 +to pc ;

: and#
        _a @inlb and' dup to _a flags! 0 carry! ;

: retpe
        parity@
        if      ret
        then ;

: jphl
        _hl to pc ;

: jppe
        parity@
        if      jp exit
        then
        2 +to pc ;

: exde
        _de _hl _de! _hl! ;

: callpe
        parity@
        if      call exit
        then
        2 +to pc ;

: xor#
        _a @inlb xor' dup to _a flags! 0 carry! ;

: e0s
        $F and'
        case
                0 of    retpo   endof
                1 of    2 pop   endof
                2 of    jppo    endof
                3 of    exsp    endof
                4 of    callpo  endof
                5 of    2 push  endof
                6 of    and#    endof
                7 of    opcode rst      endof
                8 of    retpe   endof
                9 of    jphl    endof
                $A of   jppe    endof
                $B of   exde    endof
                $C of   callpe  endof
                $D of   ???     endof
                $E of   xor#    endof
                $F of   opcode rst      endof
        endcase ;

: retp
        sign@ 0=
        if      ret
        then ;

: jpp
        sign@ 0=
        if      jp exit
        then
        2 +to pc ;

: di
        clear intflag ;

: callp
        sign@ 0=
        if      call exit
        then
        2 +to pc ;

: or#
        _a @inlb or' dup to _a flags! 0 carry! ;

: retm
        sign@
        if      ret
        then ;

: ldsp
        _hl to sp ;

: jpm
        sign@
        if      jp exit
        then
        2 +to pc ;

: ei
        1 to intflag ;

: callm
        sign@
        if      call exit
        then
        2 +to pc ;

: cp#
        _a @inlb - split carry! flags! ;

: f0s
        $F and'
        case
                0 of    retp    endof
                1 of    3 pop   endof
                2 of    jpp     endof
                3 of    di      endof
                4 of    callp   endof
                5 of    3 push  endof
                6 of    or#     endof
                7 of    opcode rst      endof
                8 of    retm    endof
                9 of    ldsp    endof
                $A of   jpm     endof
                $B of   ei      endof
                $C of   callm   endof
                $D of   ???     endof
                $E of   cp#     endof
                $F of   opcode rst      endof
        endcase ;

: c0s-f0s
        dup 4 rshift 3 &exec: c0s d0s e0s f0s ;

: fetch
        pc c@-t dup to opcode 1 +to pc 0DD =
        if      1 to %x pc c@-t to opcode 1 +to pc exit
        then
        opcode 0FD =
        if      2 to %x pc c@-t to opcode 1 +to pc
        then ;

forth definitions

: reset
        [ z80sim ] clear pc clear intflag ;

: interrupt
        [ z80sim ]
        dup 66 =
        if      clear intflag -2 +to sp pc sp !-t to pc exit
        then
        intflag
        if      clear intflag rst
        else    drop
        then ;

: =row
        dup h. space
        dup 10 bounds
        do      i c@-t b.
        loop
        space 10 bounds
        do      i c@-t semit
        loop ;

: .regs
        [ z80sim ]
        cr ." BC " _bc =row
        cr ." DE " _de =row
        cr ." HL " _hl =row
        cr 3 spaces _ix 10 - =row
        cr ." IX " _ix =row
        cr 3 spaces _iy 10 - =row
        cr ." IY " _iy =row
        cr ." A  " _a b.
        cr ." SP " sp =row
        cr ." PC " pc =row
\       SZ-A-P-C
        cr ." Flags: " flags c@ $80 and' if ." MI " else ." PL " then
        flags c@ $40 and' if ." ZR " else ." NZ " then
        flags c@ 4 and' if ." PE " else ." PO " then
        flags c@ 1 and' if ." CY " else ." NC " then
        intflag if ." IE " else ." ID " then ;
        
: exec
        [ z80sim ]
        opcode dup $C0 and'
        case
                0 of    00s-30s endof
                $40 of  40s     endof
                $80 of  80s     endof
                $C0 of  c0s-f0s endof
        endcase
        .regs ;

: step
        [ z80sim ] hide-cursor home fetch exec
        pc to adres #10 0
        do      .inst
        loop
        show-cursor ;

: steps
        0
        ?do     step
        loop ;

: trace
        begin   step key >upc dup 'I' =
                if      opcode $FB <> intflag and'
                        if      $38 rst
                        then
                then
                ^[ =
        until ;

vocabulary z80asm

z80asm definitions

create aprior   4 cells allot
' drop aprior !
' drop aprior cell+ cell+ !

: a;!
        aprior cell+ cell+ 2! ;

: clr-a;
        0 ['] drop a;! ;

: a;
        aprior 2@ execute aprior cell+ cell+ 2@ aprior 2! clr-a; ;

2 cells constant /label
#100 constant #labels

create labels   #labels /label * allot
0 value lastlabel

: []label
        labels swap /label * + ;

:noname
        local x false lastlabel 0
        ?do     i []label @ x =
                if      invert leave
                then
        loop
        ;  is label?

:noname
        lastlabel 0
        ?do     dup i []label @ =
                if      i []label cell+ @ .head leave
                then
        loop
        drop
        ;  is .label

0 value here-t

: c,-t
        here-t c!-t 1 +to here-t ;

: ,-t
        here-t !-t 2 +to here-t ;

0 constant b    1 constant c    2 constant d    3 constant e
4 constant h    5 constant l    6 constant m    7 constant a
6 constant af   6 constant sp   6 constant af

0 value xy

: %x
        0DD to xy ;        %x

: %y
        0FD to xy ;

: x
        xy c,-t %x ;

: xl
        x l ;

: xh
        x h ;

: 8*
        3 lshift ;

: ?page
        dup 80 -80 within abort" Sprongafstand te ver" ;

: (m1)
        c@ c,-t ;

: m1
        create  c,
        does>   ['] (m1) a;! a; ;

00 m1 nop       76 m1 halt      F3 m1 di        FB m1 ei
07 m1 rlca      0F m1 rrca      17 m1 rla       1F m1 rra
E9 m1 jphl      F9 m1 ldsp      E3 m1 exsp      EB m1 exde
27 m1 daa       2F m1 cpl       37 m1 scf       3F m1 ccf

: (m2)
        c@ + c,-t ;

: m2
        create  c,
        does>   ['] (m2) a;! a; ;

80 m2 add       88 m2 adc       90 m2 sub       98 m2 sbc
A0 m2 and       A8 m2 xor       B0 m2 or        B8 m2 cp

: (m3)
        c@ swap 8* + c,-t ;

: m3
        create  c,
        does>   ['] (m3) a;! a; ;

09 m3 addp      C1 m3 pop       C5 m3 push      02 m3 stap
0A m3 ldap      04 m3 inr       05 m3 der       03 m3 inc
0B m3 dec       C7 m3 rst

: (m4)
        c@ c,-t c,-t ;

: m4
        create  c,
        does>   ['] (m4) a;! a; ;

D3 m4 out       DB m4 in
C6 m4 add#      CE m4 adc#      D6 m4 sub#      DE m4 sbc#
E6 m4 and#      EE m4 xor#      F6 m4 or#       FE m4 cp#

: (m5)
        c@ c,-t ,-t ;

: m5
        create  c,
        does>   ['] (m5) a;! a; ;

22 m5 sthl      2A m5 ldhl      32 m5 sta       3A m5 lda
0CD m5 call     C9 m1 ret       C3 m5 jp

: (m6)
        0CB c,-t c@ + c,-t ;

: m6
        create  c,
        does>   ['] (m6) a;! a; ;

00 m6 rlc       08 m6 rrc       10 m6 rl        18 m6 rr
20 m6 sla       28 m6 sra       38 m6 srl

: (m7)
        0CB c,-t c@ + swap 8* + c,-t ;

: m7
        create  c,
        does>   ['] (m7) a;! a; ;

40 m7 bit       80 m7 res       C0 m7 set

: (m8)
        @ ,-t ;

: m8
        create  ,
        does>   ['] (m8) a;! a; ;

B0ED m8 ldir    B8ED m8 lddr    44ED m8 neg     57ED m8 ldai
47ED m8 ldia    56ED m8 im1     5EED m8 im2     B1ED m8 cpir

: (m9)
        c@ c,-t here 1+ - ?page c,-t ;

: m9
        create  c,
        does>   ['] (m9) a;! a; ;

10 m9 djnz      18 m9 jr        20 m9 jrnz      28 m9 jrz
30 m9 jrnc      38 m9 jrc

: (ma)
        x c@ c,-t c,-t ;

: ma
        create  c,
        does>   ['] (ma) a;! a; ;

86 ma )add      8E ma )adc      96 ma )sub      9E ma )sbc
A6 ma )and      AE ma )xor      B6 ma )or       BE ma )cp
34 ma )inr      35 ma )der

: (mb)
        x 0CB c,-t c@ swap c,-t c,-t ;

: mb
        create  c,
        does>   ['] (mb) a;! a; ;

06 mb )rlc      0E mb )rrc      16 mb )rl       1E mb )rr
26 mb )sla      2E mb )sra      3E mb )srl

: (mc)
        x 0CB c,-t c@ -rot c,-t 8* + c,-t ;

: mc
        create  c,
        does>   ['] (mc) a;! a; ;

46 mc )bit      86 mc )res      C6 mc )set

: ()ld)
        drop x swap 8* 46 + c,-t c,-t ;

: )ld
        0 ['] ()ld) a;! a; ;

: ()st)
        drop x swap 70 + c,-t c,-t ;

: )st
        0 ['] ()st) a;! a; ;

: ()ld#)
        drop x 36 c,-t swap c,-t c,-t ;

: )ld#
        0 ['] ()ld#) a;! a; ;

: (ld)
        drop swap 8* 40 + + c,-t ;

: ld
        0 ['] (ld) a;! a; ;

: (ld#)
        drop swap 8* 6 + c,-t c,-t ;

: ld#
        0 ['] (ld#) a;! a; ;

: (ldp#)
        drop swap 8* 1+ c,-t ,-t ;

: ldp#
        0 ['] (ldp#) a;! a; ;

: (sbcp)
        drop 0ED c,-t swap 8* 42 + c,-t ;

: sbcp
        0 ['] (sbcp) a;! a; ;

: (adcp)
        drop 0ED c,-t swap 8* 4A + c,-t ;

: adcp
        0 ['] (adcp) a;! a; ;

: (stp)
        drop 0ED c,-t swap 8* 43 + c,-t ,-t ;

: stp
        0 ['] (stp) a;! a; ;

: (ldp)
        drop 0ED c,-t swap 8* 4B + c,-t ,-t ;

: ldp
        0 ['] (ldp) a;! a; ;

: (db)
        drop c,-t ;

: db
        0 ['] (db) a;! a; ;

: (dw)
        drop ,-t ;

: dw
        0 ['] (dw) a;! a; ;

: (ds)
        drop 0
        ?do     count c,-t
        loop    drop ;

: ds
        0 ['] (ds) a;! a; ;

C2 constant 0=' D2 constant CS' E2 constant PE' F2 constant 0<'
C3 constant never'

20 constant 0=  30 constant CS  18 constant never

: not
        8 xor' ;

: >mark'
        here-t >s 0 ,-t ;

: >resolve'
        here-t s> !-t ;

: <mark'
        here-t >s ;

: <resolve'
        s> ,-t ;

: >mark
        here-t >s 0 c,-t ;

: >resolve
        here-t 1- s> dup >s - ?page s> c!-t ;

: <mark
        here-t >s ;

: <resolve
        s> here-t 1+ - ?page c,-t ;

: cs-swap
        s> s> swap >s >s ;

: if'
        >r a; r> c,-t >mark' ;

: then'
        a; >resolve' ;

: ahead'
        never' if' ;

: else'
        ahead' cs-swap then' ;

: begin'
        a; <mark' ;

: until'
        >r a; r> c,-t <resolve' ;

: again'
        never' until' ;

: while'
        if' cs-swap ;

: repeat'
        again' then' ;

: if
        >r a; r> c,-t >mark ;

: then
        a; >resolve ;

: ahead
        never if ;

: else
        ahead cs-swap then ;

: begin
        a; <mark ;

: until
        >r a; r> c,-t <resolve ;

: again
        never until ;

: while
        if cs-swap ;

: repeat
        again then ;

: loop
        10 until ;

forth definitions

: clearlabels
        [ z80asm ] clr-a; labels #labels /label * erase clear lastlabel ;

: asm:
        [ z80asm ] z80asm !csp clr-a; ;

: asm;
        [ z80asm ] a; forth ?csp ;

: label
        create  [ z80asm ] a; here-t lastlabel []label ! here body> >head
                lastlabel []label cell+ ! 1 +to lastlabel here-t , asm:
        does>   @ ;

: org
        [ z80asm ] >r a; r> to here-t ;

: ==
        [ z80asm ] org label ;

trgseg 2@ swap 0 fillp

forth

warning on
                            \ (* End of Source *) /
