\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth 
\ PROJECT     : Meta compiler for CHForth version 1.1.0 
\ FILENAME    : BLOCK.FRT 
\ DESCRIPTION : Handling of Block files 
\ AUTHOR      : Coos Haak, Utrecht 
\ CREATED     : May 11, 1994
\ ----------------------------------------------------------------------



ALIGN

INTERNAL:

CREATE BLOCK-FILE-NAME  40 ALLOT

0 VALUE PREV

0 VALUE USE

0 VALUE BUFFERS

0 VALUE BUFFER-0

\ Read a block buffer from disk.
VECTOR READ-BLOCK       ( addr u -- )           \ EXTRA

\ Write a block buffer to disk.
VECTOR WRITE-BLOCK      ( addr u -- )           \ EXTRA

\ Returns the address of the next block buffer. The flag is true
\ if this buffer differs from the value in PREV .
VECTOR +BUF     ( addr1 -- addr2 flag )         \ EXTRA "plus-buf"

DOC
: +BUF          ( addr1 -- addr2 )
        402 + BUFFER-0 402 BUFFERS * +          \ go to next buffer
        OF      BUFFER-0                        \ if at end, go to first one
        THEN
        DUP PREV <>                             \ flag if different
        ;
ENDDOC

FORTH:

\G a-addr is the address of a cell containing zero or the number of
\G the mass-storage block being interpreted. If BLK contains zero,
\G the input source is not a block and can be identified by 
\G SOURCE-ID . A program may not directly alter the contents of BLK
VARIABLE BLK    ( -- a-addr )                   \ FORTH "b-l-k"
ANS

\G Mark the current block buffer as modified.
\G
\G UPDATE does not immediate cause I/O.
\G See also: BLOCK BUFFER FLUSH SAVE-BUFFERS
VECTOR UPDATE   ( -- )                          \ FORTH
\ : UPDATE      ( -- )
\       PREV @ 8000 OR PREV !                           \ set highest bit
\       ;
        ANS

\G a-addr is the address of the first character of the block
\G buffer assigned to u. The contents of the block are
\G unspecified. Exceptions -34 or -35 will occur if u is not an
\G available block number.
\G
\G If block u is already in a block buffer: a-addr is the address of
\G that block buffer. 
\G
\G If block u is not already in memory and there is an unassigned
\G block buffer. a-addr is the address of that block buffer.
\G
\G If block u is not already in memory and there are no unassigned
\G block buffers: unassign a block buffer. If the block in that
\G buffer has been UPDATEd, transfer the block to mass-storage.
\G a-addr is the address of that block buffer.
\G
\G At the conclusion of the operation the block buffer pointed to by
\G a-addr is the current block buffer and is assigned to u.
\G See also: BLOCK
: BUFFER        ( u -- a-addr )                         \ FORTH
        USE DUP>R
        BEGIN   +BUF
        UNTIL
        TO USE R@ @ 0<                                  \ if updated
        IF      R@ CELL+ R@ @ 7FFF AND WRITE-BLOCK      \ write block
        THEN
        R@ ! R@ TO PREV R> CELL+
        ;  ANS

\G a-addr is the address of the first character of the block buffer
\G assigned to mass-storage block u. Exceptions -33, -34 or -35
\G will occur if u is not an available block number.
\G
\G If block u is already in a block buffer: a-addr is the address of
\G that block buffer. 
\G
\G If block u is not already in memory and there is an unassigned
\G block buffer: transfer block u from mass-storage to an
\G usassigned block buffer. a-addr is the address of that block
\G buffer.
\G
\G If block u is not already in memory and there are no
\G unassigned block buffers: unassign a block buffer. If the
\G block in that buffer has been UPDATEd, transfer the block to
\G mass-storage and transfer block u from mass storage into the
\G buffer. a-addr is the address of that block buffer.
\G
\G At the conclusion of the operation the block buffer pointed to
\G by a-addr is the current block buffer and is assigned to u.
: BLOCK         ( u -- a-addr )                         \ FORTH
        >R PREV DUP @ R@ - 2*
        IF      BEGIN   +BUF 0=                         \ is PREV block
                        IF      DROP R@ BUFFER          \ make room
                                DUP R@ READ-BLOCK CELL- \ read block
                        THEN
                        DUP @ R@ - 2* 0=
                UNTIL   DUP TO PREV
        THEN    R>DROP CELL+
        ;  ANS

\G Transfer the contents of each UPDATEd block buffer to mass
\G storage. Mark all buffers as unmodified.
: SAVE-BUFFERS  ( -- )                          \ FORTH
        BUFFERS 1+ 0
        DO      7FFF BUFFER DROP                \ write any UPDATEd buffer
        LOOP
        ;  ANS

\G Unassign all block buffers. Do not transfer the contents of any
\G UPDATEd block buffer to mass storage.
\G See also: BLOCK
: EMPTY-BUFFERS ( -- )                          \ FORTH
        BUFFER-0 402 BUFFERS * ERASE
        ;  ANS

\G Perform the function of SAVE-BUFFERS and unassign all block
\G buffers.
: FLUSH         ( -- )                          \ FORTH
        SAVE-BUFFERS
        EMPTY-BUFFERS
        ;  ANS

\G Save the current input source specification. Store u in BLK ,
\G thus making block u the input source and setting the input buffer
\G to encompass its contents, set >IN to zero, and interpret. When
\G the parse area is exhausted, restore the prior input source
\G specification. Other stack effects are due to the words LOADed.
\G
\G Exceptions -33, -34 or -35 will occur if u is zero, or is not
\G valid block number.
: LOAD          ( i*x u -- j*x )                        \ FORTH
        SAVE-INPUT                                      \ Save input specs
        >R 2>R 2>R 2>R                                  \ Secret information
        DUP BLK ! BLOCK 400 SET-SOURCE                  \ set input stream
        INTERPRET                                       \ interpret block
        2R> 2R> 2R> R>                                  \ Secret information
        RESTORE-INPUT THROW                             \ restore input specs
        ;  ANS

\G LOAD the mass storage blocks numbered u1 through u2 in sequence.
\G Other stack effects are due to the words LOADed.
: THRU          ( i*x u1 u2 -- j*x )                    \ FORTH
        1+ SWAP
        ?DO     I LOAD
        LOOP
        ;  ANS

\G a-addr is the address of a cell containing the block number of
\G the block most recently LISTed.
VARIABLE SCR    ( -- a-addr )                           \ FORTH "s-c-r"
ANS

EXTRA:

\G Give the address c-addr and length u2 of the line n of the block
\G u1.
: (LINE)        ( n u1 -- c-addr u2 )                   \ EXTRA "paren-line"
        BLOCK SWAP 40 * + 40
        ;

\G Type line n of block u.
: .LINE         ( n u -- )                              \ EXTRA "dot-line"
        (LINE) -TRAILING TYPE
        ;

FORTH:

\G Display block u in an implementation-defined format. Store u in
\G SCR .
\G See also: BLOCK
: LIST          ( u -- )                        \ FORTH
        CR DUP SCR ! ." Screen " . ." of " BLOCK-FILE-NAME COUNT TYPE
        10 0
        DO      CR I 3 .R SPACE I SCR @ .LINE
        LOOP
        CR
        ;  ANS

                            \ (* End of Source *) /
