\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth 
\ PROJECT     : Meta compiler for CHForth version 1.1.0 
\ FILENAME    : INTERP.FRT 
\ DESCRIPTION : Interpreter and search routines 
\ AUTHOR      : Coos Haak, Utrecht 
\ CREATED     : May 11, 1994
\ ----------------------------------------------------------------------


\G Find the Forth word identified by the string c-addr u in the word
\G list identified by wid. If the word is not found, return zero. If
\G the word is found, return its execution token xt and 1 if the
\G word is immediate, -1 otherwise.
CODE SEARCH-WORDLIST    ( c-addr u wid -- 0 | xt 1 | xt -1 )    \ FORTH
                MOV     BX, 2 [BX]              \ last name
                POP     CX                      \ count
                POP     DX                      \ addr
        CNZ IF
                XCHG    SP, BP
                PUSH    ES                      \ save LSEG
                PUSH    SI                      \ save IP
                XCHG    SP, BP
                MOV     ES, hdrseg              \ dest for compare
                MOV     AX, CX                  \ copy count
        AHEAD                                   \ skip next instruction
        BEGIN
                MOV     BX, ES: 0 [BX]          \ next word
        ENTRY
                TEST    BX, BX                  \ end of chain
                JZ      1 $
                CMP     AL, ES: 4 CELLS [BX]    \ count bytes equal
0 CS-PICK
        0= UNTIL
                TEST    ES: 1 CELLS [BX], # =HIDDEN-T   \ hidden word
0 CS-PICK
        0= UNTIL
                MOV     SI, DX                  \ c-addr
                LEA     DI, 4 CELLS CHAR+ [BX]  \ name+1
                MOV     CX, AX                  \ restore count
                REPZ    CMPSB                   \ ds:si and es:di for cx bytes
        0= UNTIL
                PUSH    ES: 3 CELLS [BX]        \ execution token
                MOV     AX, ES: 1 CELLS [BX]    \ flags
                MOV     T' HEADFLAGS AX         \ save in value
                MOV     BX, # -1                \ normal = -1
                TEST    AX, # =IMMEDIATE-T
        0<> IF
                NEG     BX                      \ immediate = 1
        THEN
        1 $:    XCHG    SP, BP
                POP     SI                      \ restore IP
                POP     ES                      \ restore LSEG
                XCHG    SP, BP
                NEXT
        THEN
                MOV     BX, CX                  \ null string
                NEXT
END-CODE  ANS

\G Execute the definition specified by xt. Other stack effects are
\G due to the word EXECUTEd.
\G See also: ' [']
CODE EXECUTE        ( i*x xt -- j*x )                   \ FORTH
                MOV     AX, BX                          \ copy xt
                POP     BX                              \ pop TOS
                JMP     AX
END-CODE  ANS

EXTRA:

\G Perform a bitwise AND on the two numbers on the stack and use the
\G result as an index into the inline execution array and execute
\G the execution token stored there.
CODE &EXEC:             ( x1 x2 -- )            \ EXTRA "and-exec-colon"
                POP     AX                              \ pop
                AND     BX, AX                          \ and with TOS
END-CODE  COMPILE-ONLY                          \ fall through

\G Use x as an index into the inline execution array and execute
\G the execution token stored there.
CODE EXEC:              ( x -- )                \ EXTRA "exec-colon"
                SHL     BX, # 1                         \ cell to bytes
                ADD     SI, BX                          \ add to IP
                LODSW   ES:                             \ load found W
                XCHG    SP, BP
                POP     SI                              \ restore IP
                XCHG    SP, BP
                POP     BX                              \ pop TOS
                JMP     AX                              \ jump to W
END-CODE  COMPILE-ONLY

INTERNAL:

\ Convert the execution token on the stack to an address in the
\ hashtable in the header segment.
CODE CNHASH             ( xt -- h-addr )
                MOV     BL, BH
                AND     BX, # FE
                NEXT
END-CODE

\ Find the dictionary entry address dea that lies between h-addr1
\ and h-addr2 that belongs to the execution token xt. Return zero if
\ dea can not be found.
CODE CNSRCH     ( xt h-addr1 h-addr2 -- dea | 0 )
                MOV     DX, BX
                POP     DI
                POP     CX
                MOV     DS, hdrseg
        1 $:    LEA     BX, 3 CELLS [DI]        -- dea to code pointer
                CMP     DX, BX
                JBE     3 $
                MOV     AX, 0 [BX]              -- compare this xt
                CMP     AX, CX                  -- to the one given
                JNZ     2 $
                MOV     BX, DI                  -- this is the dea
                JMP     4 $
        2 $:    MOV     AX, 4 CELLS [DI]        -- dea to name
                AND     AX, # 1E
                ADD     DI, AX
                ADD     DI, # 5 CELLS           -- next dea
                JMP     1 $
        3 $:    XOR     BX, BX
        4 $:    PUSH    CS
                POP     DS
                NEXT
END-CODE

EXTRA:

\G dea is the dictionary entry address that is associated with
\G execution token xt. If this fails, dea is zero.
: >HEAD                 ( xt -- dea )           \ EXTRA "to-head"
        DUP CNHASH DUP H@ SWAP CELL+ H@ CNSRCH
        ;

\G xt is the the execution token that is associated with the
\G dictionary entry address dea.
CODE HEAD>              ( dea -- xt )           \ EXTRA "head-from"
                PUSH    DS
                MOV     DS, hdrseg
                MOV     BX, 3 CELLS [BX]
                POP     DS
                NEXT
END-CODE

\G h-addr is the forget field address of the dictionary entry dea.
CODE HEAD>FORGET        ( dea -- h-addr )       \ EXTRA "head-to-forget"
                ADD     BX, # 2 CELLS
                NEXT
END-CODE

\G h-addr is the flag field address of the dictionary entry dea.
CODE HEAD>FLAGS         ( dea -- h-addr )       \ EXTRA "head-to-flags"
                ADD     BX, # 1 CELLS
                NEXT
END-CODE

EXTRA:

\G h-addr is the name field address of the dictionary entry dea.
CODE HEAD>NAME          ( dea -- h-addr )       \ EXTRA "head-to-name
                ADD     BX, # 4 CELLS
                NEXT
END-CODE

FORTH:

\G a-addr is the data field address corresponding to execution token
\G xt. This is only valid for words defined via CREATE .
CODE >BODY              ( xt -- a-addr )        \ FORTH "to-body"
                ADD     BX, # 2 CELLS
                NEXT
END-CODE  ANS

EXTRA:

\G xt is execution token corresponding to the data field address
\G a-addr. This is only valid for a word defined via CREATE .
CODE BODY>              ( a-addr -- xt )        \ EXTRA "body-from"
                SUB     BX, # 2 CELLS
                NEXT
END-CODE

\G xt2 is the execution token of the DOES> part of the defining word
\G of an execution token xt1.
CODE >CALL              ( xt1 -- xt2 )          \ EXTRA "to-call"
                INC     BX
                ADD     BX, 0 [BX]
                INC     BX
                INC     BX
                NEXT
END-CODE

INTERNAL:

: GET-TOP-WORD          ( -- dea T | F )
        TEMPORARY @
        IF      TEMPORARY @ DUP H@ TEMPORARY ! TRUE EXIT
        THEN
        FALSE
        ;

: GET-VALID-WORD        ( -- dea T | F )
        STOP?
        IF      FALSE EXIT
        THEN
        BEGIN   GET-TOP-WORD
        WHILE   DUP HEAD>FLAGS H@ =HIDDEN AND 0=
                IF      TRUE EXIT
                THEN
                DROP
        REPEAT
        FALSE
        ;

EXTRA:

\G Return the next dea in the word list. Used in words as WORDS .
\G This word depends on the stored wid at TEMPORARY . When ANSI
\G does not contain zero, only words marked with ANS are
\G returned.
: ANOTHER               ( -- dea true | false )         \ EXTRA
        BEGIN   GET-VALID-WORD DUP
        WHILE   ANSI @
        WHILE   OVER HEAD>FLAGS H@ =ANSI AND 0=
        WHILE   2DROP
        REPEAT
        THEN
        THEN
        ;

LABEL 'WORD     20 ALLOT        \ Temporary buffer to hold name

\G Find the Forth word specified by the character string c-addr u in
\G all word lists in the search order, including LOCAL-WORDLIST when
\G STATE does not contain zero and there are local values. Return
\G the execution token and 1 if the word is IMMEDIATE and -1
\G otherwise. If name can not be found, return a false flag. The
\G name is internally converted to uppercase if the variable
\G CASESENSITIVE is false.
: SEARCH-CONTEXT        ( c-addr u -- 0 | xt 1 | xt -1 )        \ EXTRA
        1F UMIN (LIT) 'WORD PLACE CASESENSITIVE @ INVERT
        IF      (LIT) 'WORD COUNT UPPER
        THEN

        GET-ORDER LOCALS FROM STATE AND
        IF      LOCAL-WORDLIST SWAP 1+
        THEN

        FALSE SWAP 0
        ?DO     ?DUP
                IF      ROT DROP
                ELSE    (LIT) 'WORD COUNT ROT SEARCH-WORDLIST
                THEN
        LOOP
        ;

FORTH:

\G Find the Forth word named in the counted string at c-addr. If the
\G word is not found after searching all word list in the search
\G order, return c-addr and zero. If the definition is found, return
\G xt. If the definition is immediate, also return 1, otherwise
\G return -1.
\G See also: ' ['] POSTPONE
: FIND          ( c-addr -- c-addr 0 | xt 1 | xt -1 )   \ FORTH
        DUP COUNT SEARCH-CONTEXT DUP
        IF      HEADFLAGS =LOCAL AND -9 ?ERROR
                ROT DROP
        THEN
        ;  ANS

\G Skip leading space delimiters. Parse name delimited by a space.
\G Find name and return xt, the execution token for name. Exception
\G -13 occurs if name is not found.
\G
\G When interpreting ' name EXECUTE is equivalent to name.
\G See also: POSTPONE [']
: '             ( "name" -- xt )                \ FORTH "tick"
        BL WORD FIND 0= #-13 ?ERROR
        ;  ANS

HEADER FORTH ANS
ONLY:

\G Make the FORTH word list the first word list to be searched. Note
\G that this word list contains at startup only ANSI-standard words.
CODE FORTH              ( -- )                  \ FORTH
                JMP     (DOVOC)
                $EVEN
END-CODE  ANS

        HERE T' VOC-LINK !-T 0 ,
        TWORDLIST

\G Return wid, the identifier of the word list that includes all
\G standard words provided by the implementation. This word list is
\G initially the compilation word list and is part of the initial
\G search order.
CONSTANT FORTH-WORDLIST         ( -- wid )      \ ONLY
ANS

\G Make the EXTRA word list the first word list to be searched.
\G This word list contains all CHForth specific extensions to the
\G ANSI standard. Note that these words are non-standard.
CODE EXTRA              ( -- )                  \ ONLY
                JMP     (DOVOC)
                $EVEN
END-CODE

        HERE T' VOC-LINK DUP @-T , !-T
        TWORDLIST DROP

\G Make the EDITOR word list the first word list to be searched.
\G This word list contains CHForth specific extensions to the ANSI
\G standard for the line input editor and the block editor. Note
\G that these words are non-standard.
CODE EDITOR              ( -- )                  \ ONLY
                JMP     (DOVOC)
                $EVEN
END-CODE

        HERE T' VOC-LINK DUP @-T , !-T
        TWORDLIST DROP

\G Make the INTERNAL word list the first word list to be searched.
\G This word list contains CHForth specific extensions to the ANSI
\G standard that are not documented and can be changed by the author
\G by name or action without prior consent. Note that these words
\G are non-standard.
CODE INTERNAL              ( -- )                  \ ONLY
                JMP     (DOVOC)
                $EVEN
END-CODE

        HERE T' VOC-LINK DUP @-T , !-T
        TWORDLIST DROP

FORTH:

HEADER ONLY  ANS
ONLY:

\G Set the search order to the minimum search order. The minimum
\G search order includes the ability to interpret the words
\G FORTH-WORDLIST and SET-ORDER .
CODE ONLY               ( -- )                  \ ONLY
                JMP     DOONLY
                $EVEN
END-CODE  ANS

        HERE T' VOC-LINK DUP @-T , !-T
        TWORDLIST DROP

T' DOVOC 2 CELLS + @-T ,
( inherit GET method, ignore FORGET link )

TDO> DOONLY
        DUP 2 SET-ORDER
        ;

\G Return the wid of the LOCAL-WORDLIST .
VARIABLE LOCAL-WORDLIST         ( -- wid )      \ ONLY
        0 ,

\G Return wid, the identifier of the compilation word list.
CODE GET-CURRENT        ( -- wid )              \ ONLY
                PUSH    BX
                MOV     BX, current
                NEXT
END-CODE  ANS

\G Set the compilation word list to the word list identified by
\G wid.
CODE SET-CURRENT        ( wid -- )              \ ONLY
                MOV     current BX
                POP     BX
                NEXT
END-CODE  ANS

\G Return wid, the identifier of the first word list in the
\G search order.
CODE GET-CONTEXT        ( -- wid )              \ ONLY
                PUSH    BX
                MOV     BX, T' CONTEXT
                NEXT
END-CODE

\G Set the first searched word list in the search order to the
\G word list identified by wid.
CODE SET-CONTEXT        ( wid -- )              \ ONLY
                MOV     T' CONTEXT BX
                POP     BX
                NEXT
END-CODE

\G Returns the number of word lists n in the search order and the
\G word list identifiers wid1 .. widn identifying these word
\G lists. widn identifies the word list searched first, and wid1
\G the word list that is searched last. The search order is
\G unaffected.
CODE GET-ORDER          ( -- wid1 .. widn n )   \ ONLY
                PUSH    BX
                MOV     CX, T' VSP
                SHR     CX, # 1
                MOV     BX, CX
        CNZ IF
                MOV     DI, CX
                ADD     DI, DI
                ADD     DI, # T' CONTEXT
        DO
                DEC     DI
                DEC     DI
                PUSH    0 [DI]
        LOOP
        THEN
                NEXT
END-CODE  ANS

\G Set the search order to the word lists wid1 .. widn.
\G Subsequently, word list widn will be searched first, followed
\G by word list widn-1 and so on, with word list wid1 searched
\G last. If n is zero, empty the search order. If n is minus one,
\G set the search order to the minimum search order wid(ONLY)
\G wid(ONLY). When n is minus two, set the search order to
\G wid(ONLY) wid(EXTRA) wid(FORTH) wid(FORTH). The maximum of n
\G in this implementation is sixteen.
: SET-ORDER             ( wid1 .. widn n -- )   \ ONLY
        -1 OF   (GET) T[ T' ONLY ,-L T] DUP 2   THEN
        -2 OF   (GET) T[ T' ONLY ,-L T]
                (GET) T[ T' EXTRA ,-L T]
                FORTH-WORDLIST DUP 4    THEN
        DUP #VOCS > #-49 ?ERROR
        DUP 0< #-50 ?ERROR
        CELLS DUP TO VSP CONTEXT SWAP BOUNDS
        ?DO     I ! 1 CELLS
        +LOOP
        ;  ANS

\G Make the compilation word list the same as the first word list
\G in the search order. Specifies that the names of subsequent
\G definitions will be placed in the compilation word list.
\G Subsequent changes in the search order will not effect the
\G compilation word list.
: DEFINITIONS           ( -- )                  \ ONLY
        GET-CONTEXT SET-CURRENT
        ;  ANS

\G Transform the search order consisting of wid1 .. widn-1 widn
\G (where widn is searched first) into wid1 .. widn-1 widn widn.
\G An ambiguous condition exists if there are too many word lists
\G in the search order.
: ALSO                  ( -- )                  \ ONLY
        GET-ORDER OVER SWAP 1+ SET-ORDER
        ;  ANS

\G Transform the search order consisting of wid1 .. widn-1 widn
\G (where widn is searched first) into wid1 .. widn-1. An
\G ambiguous condition exists if the search order was empty
\G before PREVIOUS was executed.
: PREVIOUS              ( -- )                  \ ONLY
        GET-ORDER NIP 1- SET-ORDER
        ;  ANS

LABEL 'tib  #132 ALLOT      \ Terminal input buffer

:ORPHAN DFTCOLD
        RESTART? INVERT                     \ No restart and
        (LIT) cold C@ AND                   \ something on the command line
        IF      (LIT) cold C0!              \ No second time
                80 COUNT SET-SOURCE [
                ['] INTERPRET CATCH ?DUP
                IF      SHOW-ERROR
                THEN
                SIGNON @ LINESREAD @ AND
                IF      DIAGNOSE
                THEN
        THEN
        -2 SET-ORDER DEFINITIONS
        (LIT) 'tib C/L ERASE
        QUIT
        ;

INTERNAL:

\ Type a prompt consisting of name of first wid in the search
\ order, enclosed by brackets when STATE does not contain zero,
\ followed by a close angled bracket and a space. This is the
\ default action of PROMPT .
: (PROMPT)              ( -- )
        FROM STATE
        IF      '[' EMIT
        THEN
        GET-CONTEXT BODY> >HEAD DUP
        IF      (.HEAD) TYPE
        ELSE    DROP ." {NoVoc}"
        THEN
        FROM STATE
        IF      ']' EMIT
        THEN
        ." > "
        ;

\ Issue a warning when ANSI does not contains zero and the last
\ found word is not a standard word.
: ?ANSI         ( -- )
        HEADFLAGS =ANSI AND 0= ANSI @ AND
        IF      #-527 .MESS
        THEN
        ;

\ Issue a warning when ANSI does not contains zero and the last
\ found number does not have a standard format.
: ?PORTABLE     ( -- )
        ANSI @ (LIT) notport @ AND
        IF      #-529 .MESS
        THEN
        ;

EXTRA:

\G Try to find the name c-addr u in the search order and execute
\G it when found else convert the string to a number and place it
\G on the stack. Else abort with an exception message.
: $INTERPRET            ( c-addr u -- )         \ EXTRA "string-interpret"
        SEARCH-CONTEXT
        IF      ?ANSI HEADFLAGS =COMP AND #-14 ?ERROR EXECUTE EXIT
        THEN
        PARSED-WORD NUMBER? 0= #-13 ?ERROR ?PORTABLE
        ;

\G Try to find the name c-addr u in the search order and when
\G found execute it or compile it according to the flag returned
\G by FIND . Else try to convert the string to a number and
\G compile it.  Else issue a warning that the word can not be
\G found and compile a forward reference to it.
: $COMPILE              ( c-addr u -- )         \ EXTRA "string-compile"
        SEARCH-CONTEXT
        IF      ?ANSI HEADFLAGS =IMMEDIATE AND
                IF      EXECUTE EXIT
                THEN
                COMPILE, EXIT
        THEN
        PARSED-WORD NUMBER? ?DUP
        IF      ?PORTABLE LITERALS EXIT
        THEN
        #-13 ?CRASH
        ;

FORTH:

\G Enter interpretation state. [ is an immediate word.
\G See also: ]
: [             ( -- )                  \ FORTH "left-bracket"
        STATE OFF
        ;  IMMEDIATE  ANS

\G Enter compilation state.
\G See also: [
: ]             ( -- )                  \ FORTH "right-bracket"
        STATE ON
        ;  ANS

EXTRA:

\G Interpret the current input stream.
: INTERPRET     ( -- )                          \ EXTRA
        BEGIN   BL WORD DUP C@
        WHILE   COUNT FROM STATE
                IF      'COMPILE
                ELSE    'INTERPRET
                THEN
                ?STACK
        REPEAT
        DROP
        ;

\G Reset the input and output to the terminal.
: TERMINAL      ( -- )                          \ EXTRA
        VIDEO SHOW-CURSOR
        ['] (KEY) IS KEY
        ['] (EMIT) IS EMIT
        (LIT) 'tib #TIB CELL+ ! CLEAR SOURCE-ID BLK OFF
        ;

\G Skip leading space delimiters. Parse name delimited by zero
\G and when the length is not zero, store it in a special
\G location and append the extension in FEXT$ to it. Return
\G c-addr u of that string. If the length of name is zero, return
\G the string that was stored in the location by a previous call
\G of GETNAME .
: GETNAME           ( "name" -- c-addr u )              \ EXTRA
        BL PARSE-WORD DUP
        IF      THEFILE PACK COUNT '.' SCAN NIP 0=
                IF      FEXT$ COUNT THEFILE APPEND
                THEN
        ELSE    2DROP
        THEN
        THEFILE COUNT
        ;

\G Skip leading space delimiters. Parse name delimited by a space
\G and load the file with that name. If the length of name is
\G zero, load the file that was previously load with IN .
: IN            ( "name" -- )                           \ EXTRA
        GETNAME INCLUDED
        ;

\G Skip leading delimiters. Parse name delimited by a space and
\G load the file with that name. The appropriate extension must
\G be included in name.
: INCLUDE       ( "name" -- )                           \ EXTRA
        BL WORD COUNT INCLUDED
        ;

FORTH:

LABEL 'lib  #132 ALLOT      \ Line input buffer

\G Attempt to fill the current input stream, returning a true
\G flag if successful. The action depends on the source of the
\G current input stream.
\G If the input-stream source is a string from EVALUATE , REFILL
\G returns false and performs no other action.
\G Otherwise, REFILL attempts to receive input into the
\G text-input buffer whose address is given by TIB , making the
\G result the current input stream and returning a true flag if
\G successful. Receipt of a line containing no characters is
\G considered successful. A false flag is returned only when
\G there is no input available from the current input-stream
\G source.
\G If the input source is a block, REFILL makes the next block
\G the current input source and input buffer, by adding one to
\G the value of BLK and setting >IN to zero. True is returned if
\G the new value of BLK is a valid block number, false otherwise.
\G If the input-stream source is a text file, REFILL attempts to
\G read the next line from the text-input file, making the result
\G the current input stream and returning true if the read
\G succeeded, and returning false otherwise.
: REFILL        ( -- flag )                     \ FORTH
        BLK @                                   \ Is loading a block
        IF      BLK INCR                        \ next block
                BLK @ BLOCK                     \ read it
                400 SET-SOURCE TRUE EXIT        \ return with ok flag
        THEN
        SOURCE-ID 0=                            \ from terminal
        IF      QUERY TRUE EXIT                 \ get a line from the user
        THEN
        SOURCE-ID 0<                            \ from string in EVALUATE
        IF      FALSE EXIT                      \ is not ok
        THEN
        (LIT) 'lib CHAR+ /LINE SOURCE-ID        \ during file loading
        READ-LINE THROW
        >R                                      \ save flag
        (LIT) 'lib TUCK C! COUNT                \ Input line in buffer
        SET-SOURCE
        R>                                      \ return -eof flag
        #LINES INCR                             \ Increment lines read
        ;  ANS

\G Empty the return stack, store zero in SOURCE-ID , make the
\G user input device the input source, and enter interpretation
\G state. Do not display a message. Repeat the following:
\G  - Accept a line forth the input source into the input buffer,
\G    set >IN to zero and interpret.
\G  - Display the implementation defined input prompt if in
\G    interpretation state, all processing has been completed,
\G    and no ambiguous condition exists.
: QUIT          ( -- )                          \ FORTH
        (RESET)                                 \ Set stacks and SRCSEG
        START-FORTH                             \ Initialize everything
        BEGIN   CR .STATUS PROMPT               \ Show some information
                QUERY                           \ Wait for user response
                ['] INTERPRET CATCH             \ Interpret the line
                ?DUP 0=                         \ When no error occured
        WHILE   FROM STATE INVERT               \  and STATE is zero
                IF      ."  ok"                 \  all was right!
                THEN
        REPEAT
        SHOW-ERROR                              \ An error occured
        QUIT                                    \ Restart
        ;  ANS

EXTRA:

\G Set the source to the string c-addr u and set >IN to zero.
: SET-SOURCE            ( c-addr u -- )         \ EXTRA
        #TIB 2! >IN OFF
        ;

FORTH:

\G c-addr is the address of the terminal input buffer.
: TIB                   ( -- c-addr )           \ FORTH "t-i-b"
        #TIB CELL+ @
        ;  ANS

\G c-addr is the address of, and u is the number of characters
\G in, the input buffer.
: SOURCE                ( -- c-addr u )         \ FORTH
        #TIB 2@
        ;  ANS

\G x1 through xn describe the current state of the input source
\G specification for later use by RESTORE-INPUT .
: SAVE-INPUT    ( -- x1 .. xn n )                       \ FORTH
        #LINES @ SOURCE-ID BLK @ SOURCE >IN @ 6
        ;  ANS

\G Attempt to restore the input source specification to the state
\G described by x1 through xn, flag is true if the input source
\G specification can not be so restored.
\G
\G An ambiguous condition exists if the input source represented
\G by the arguments is not the same as the current input source.
\G See also: SAVE-INPUT
: RESTORE-INPUT     ( x1 .. xn n -- flag )              \ FORTH
        6 =
        IF      >IN ! #TIB 2! BLK ! TO SOURCE-ID
                BLK @
                IF      DROP BLK @ BLOCK #TIB CELL+ ! FALSE EXIT
                THEN
                SOURCE-ID 0>
                IF      REPOSITION-LINE EXIT
                THEN
                DROP FALSE EXIT
        THEN
        #-36
        ;  ANS

\G Save the current input source specification. Store minus one
\G in SOURCE-ID . Make the string described by c-addr and u both
\G the input source and input buffer, set >IN to zero, and
\G interpret. When the parse area is empty, restore the prior
\G input source specification. Other stack effects are due to the
\G words EVALUATEd.
: EVALUATE      ( i*x c-addr u -- j*x )                 \ FORTH
        SAVE-INPUT >R 2>R 2>R 2>R
        -1 TO SOURCE-ID BLK OFF SET-SOURCE INTERPRET
        2R> 2R> 2R> R> RESTORE-INPUT THROW
        ;  ANS

                            \ (* End of Source *) /
