\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth 
\ PROJECT     : CHForth, a Dutch public domain ANS Forth for PC's 
\ DESCRIPTION : Pilot interpreter 
\ CATEGORY    : Standard Programs 
\ AUTHOR      : Coos Haak 
\ LAST CHANGE : May 16, 1994, Coos Haak 
\ ----------------------------------------------------------------------


        ?DEF -pilot [IF] -pilot [THEN]

        MARKER -pilot


DOC
(*
  This is my interpretation of a PILOT interpreter.
  As I don't have any specification, this interpreter is not standard.

  command     meaning
     A     ask 'Y' or 'N', sets the condition flag
     C     clears the screen
     E     end of program
     J nn  jumps to the label
     K     ask for a key
     P nn  perform the labeled subroutine
     R     return from the subroutine
     S     skips the following line
     T     type the following line
     #     can't remember
     ?     can't figure this out
     .     Needed on the last line
     :nn   A label

   A label is a colon followed with a number, this can be followed
   by any command or an end of line.
   Prefixing a line with N or Y conditionally executes the line.
   Every text not following T is regarded as comment.
   Spaces before a label or command are ignored.
*)

   LOAD <filename> Loads a PILOT program.
   RUN             Runs the PILOT program.
   NEW             Clears the program.
   LIST            Lists the program.
   SLIST           Lists the structure.
   TRACE           Is a control variable.
ENDDOC


privates

-- read a file

#1024 constant /inbuffer                        private \ size of inputbuffer
variable inptr                                  private \ pointer to char
variable inread                                 private \ read so far
0 value infile                                  private \ file handle
create inbuffer                                 private \ the buffer
        inbuffer /inbuffer dup allot erase

: getch         ( -- ch | -1 )                          \ read a character
        inptr @ inread @ =                                      \ end of buffer
        if      inbuffer /inbuffer infile read-file throw
                inread ! inptr off
        then
        inread @ 0=                                             \ end of file
        if      true exit
        then
        inbuffer inptr @ + c@                                   \ get char
        inptr incr                                              \ increment ptr
        ;

: openr         ( c-addr u -- )                         \ open the file
        r/o bin open-file throw to infile       \ bin reserves no buffer
        inptr off inread off                            \ reset pointers
        ;

: closer        ( -- )                                  \ close the file
        infile close-file throw
        ;

deprive

-- circular user stack

privates

#10 constant /stack                             private \ maximal depth
variable sp                                     private \ stack pointer
create stack                                    private \ circular stack
        here /stack cells dup allot erase

: >t            ( x -- )                        \ push one number
        sp incr                                         \ pre-increment
        sp @ /stack circular                            \ circular index
        stack []cell !                                  \ store data
        ;

: t>            ( -- x )                        \ pop one number
        sp @ /stack circular                            \ circular index
        stack []cell @                                  \ fetch data
        sp decr                                         \ post-decrement
        ;

deprive

-- program space

privates

#1024 constant /program                         private \ max size of program
create program                                  private \ the program space
        program /program dup allot erase

variable dp dp off                              private \ dictionary pointer
variable sp                                     private \ string pointer

: >real         ( x1 -- x2 )                    \ convert to real address
        /program circular program +
        ;  private

: phere         ( -- addr )
        dp @ /program circular                          \ circular addressing
        ;  private

: pallot        ( x -- )
        dp +!
        ;  private

: _@            ( addr -- x )
        >real @
        ;  private

: _c@           ( addr -- c )
        >real c@
        ;  private

: _!            ( x addr -- )
        >real !
        ;  private

: _c!           ( c addr -- )
        >real c!
        ;  private

: _,            ( x -- )
        phere _! 1 cells pallot
        ;  private

: _c,           ( c -- )
        phere _c! 1 pallot
        ;  private

: _s",          ( c-addr u -- )                 \ compile a string
        dup 1+ negate sp +!                             \ downward from sp
        tuck sp @ 1+ >real swap cmove                   \ move string
        sp @ _c!                                        \ store count
        ;  private

6 constant /command                     private \ byte count per command

: prt           ( x -- )                        \ print when not zero
        ?dup
        if      5 .r space
        else    6 spaces
        then
        ;  private

-- some error messages

102 constant p:notfound private
103 constant p:noend    private
104 constant p:noinst   private
105 constant p:nopgm    private

p:notfound      mess" Can't find label"
p:noend         mess" Unexpected error"
p:noinst        mess" Not a PILOT instruction"
p:nopgm         mess" No PILOT program loaded"

create sbuf     private here $100 dup allot erase

: >b
        sbuf count + c! 1 sbuf c+! ;    private

: skipbl
        false
        begin   drop getch dup bl >
        until
        ;  private

: getnum
        sbuf off skipbl
        begin   >b getch dup bl > invert
        until
        drop source 2>r >in @ >r sbuf count set-source
        bl parse-word number? 1 <> #-13 ?error
        r> >in ! 2r> #tib 2!
        ;  private

: getstr
        sbuf off
        begin   getch dup bl < invert
        while   >b
        repeat
        drop
        sbuf count _s",
        ;  private

variable cond   private

: cond,
        cond @ _c, cond off
        ;  private

variable thelabel       private

: lab,
        thelabel @ _, thelabel off
        ;  private

variable opc    private

: opc,
        opc @ _c, opc off
        ;  private

: inst,
        lab, cond, opc, 0 _,
        ;  private

: comment
        begin   getch bl <
        until
        ;  private

: new
        dp off /program sp !
        cr ." Memory free " sp @ dp @ - . ." bytes."
        ;

: load
        new cond off thelabel off s" .pil" bl word append
        here count openr
        timer-reset 0 local regels
        begin   getch >upc dup opc !
                case
                        -1 of   closer p:noend throw    endof
                        bl of   endof
                        ^M of   endof
                        ^J of   1 +to regels endof
                        ^Z of   closer p:noend throw    endof
                        '#' of  lab, cond, opc, getnum _,       endof
                        ''' of  comment endof
                        ':' of  getnum thelabel !       endof
                        '.' of  closer
                                cr regels . ." lines compiled in " .ms
                                cr ." Memory left " sp @ dp @ - . ." bytes."
                                exit endof
                        '?' of  inst,   endof
                        'A' of  inst,   endof
                        'C' of  inst,   endof
                        'E' of  inst,   endof
                        'J' of  lab, cond, opc, getnum _,       endof
                        'K' of  inst,   endof
                        'N' of  'N' cond !      endof
                        'P' of  lab, cond, opc, getnum _,       endof
                        'R' of  inst,   endof
                        'S' of  inst, comment   endof
                        'T' of  lab, cond, opc, getstr sp @ _, endof
                        'Y' of  'Y' cond !      endof
                        closer p:noend throw
                endcase
        again
        ;

: seek
        phere 0
        ?do     dup i _@ =
                if      drop i unloop exit
                then
        /command +loop
        p:notfound throw
        ;  private

variable pc     private
variable thekey private

: skip_line
        /command pc +!
        ;  private

: y/n
        false local vlag false local antwoord
        begin   ?at key >upc dup 'Y' = over 'N' = or to vlag
                'Y' = to antwoord vlag invert
        while   at-xy
        repeat
        2drop space antwoord
        ;  private

: step
        pc @ cell+ _c@ ?dup
        if      cond @ <>
                if      skip_line exit
                then
        then
        pc @ cell+ char+ _c@
        case
                '#' of  pc @ cell+ cell+ _@ skip_line   endof
                '?' of  . skip_line ?stack      endof
                'A' of  y/n
                        if      'Y'
                        else    'N'
                        then
                        cond ! skip_line        endof
                'C' of  page skip_line  endof
                'E' of  r>drop  endof
                'J' of  pc @ cell+ cell+ _@ seek pc !   endof
                'K' of  key thekey ! skip_line  endof
                'P' of  pc @ /command + >t pc @ cell+ cell+ _@ seek pc !
                        endof
                'R' of  t> pc ! endof
                'S' of  skip_line skip_line     endof
                'T' of  cr pc @ cell+ cell+ _@ >real count type
                        skip_line       endof
                p:noinst throw
        endcase
        ;  private

variable trace  trace off

: list
        dp @ 0= p:nopgm ?error
        phere 0
        ?do     cr i /command / 2 .r space
                i _@ prt i cell+ _c@ emit i cell+ char+ _c@ dup emit space
                i cell+ cell+ _@ swap 'T' =
                if      >real count type
                else    prt
                then
                key ^[ =
                if      leave
                then
        /command +loop
        ;

: slist
        dp @ 0= p:nopgm ?error
        phere 0
        ?do     i cell+ char+ _c@ 'T' <> i _c@ or
                if      cr i /command / 2 .r space
                        i _@ prt i cell+ _c@ emit
                        i cell+ char+ _c@ dup emit space
                        i cell+ cell+ _@ swap 'T' =
                        if      drop
                        else    prt
                        then
                        key ^[ =
                        if      leave
                        then
                then
        /command +loop
        ;

: run
        dp @ 0= p:nopgm ?error
        pc off cond off
        begin   trace @
                if      '[' emit pc @ /command / 0 .r ']' emit
                then
                step
        again
        ;

deprive

                            \ (* End of Source *) /
