\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth 
\ PROJECT     : CHForth, a Dutch public domain ANS Forth for PC's 
\ DESCRIPTION : Generate helpfiles, like GLOSSARY of Lennart Benschop 
\ CATEGORY    : Tools 
\ AUTHOR      : Coos Haak 
\ LAST CHANGE : October 5, 1994, Coos Haak, COMPARE-UPPERCASE
\ LAST CHANGE : September 8, 1994, Coos Haak
\ LAST CHANGE : September 4, 1994, Willem Ouwerkerk
\ ----------------------------------------------------------------------

        MARKER -makehelp

\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth 
\ PROJECT     : CHForth, a Dutch public domain ANS Forth for PC's 
\ DESCRIPTION : Forth Documentation; Het Vijgeblad 44, (1994-1) pp 25-30 
\ CATEGORY    : Tools
\ AUTHOR      : Lennart Benschop
\ LAST CHANGE : September 06, 1994, Willem O, real search for gloss data
\ LAST CHANGE : June 01, 1994, Coos Haak 
\ LAST CHANGE : April 22, 1994, Marcel Hendrix, WORDLIST and pronunciation
\ LAST CHANGE : April 21, 1994, Marcel Hendrix, fixed COMPARE internal bug 
\ LAST CHANGE : April 20, 1994, Marcel Hendrix 
\ ----------------------------------------------------------------------

        PRIVATES

?UNDEF \G [IF]
: \G    POSTPONE \ ; IMMEDIATE
\G \G is an alias for \, so it is a comment till end-of-line, but 
\G has a special meaning for the Glossary Generator.

[THEN]

?UNDEF PARSE-WORD [IF]

: PARSE-WORD            \ Needed in generating kernel.hlp (CHForth)
        WORD COUNT      \  because then this program is run in iForth (alas)
        ;

: COMPARE-UPPERCASE     ( c-addr1 u1 c-addr2 u2 -- -1 | 0 | 1 )
        LOCALS| u2 c2 u1 c1 |
        u1 u2 MIN 0
        DO      c1 I + C@ >UPC
                c2 I + C@ >UPC -
                ?DUP
                IF      0<
                        IF      -1
                        ELSE    1
                        THEN
                        UNLOOP EXIT
                THEN
        LOOP
        u1 u2 -
        DUP
        IF      0<
                IF      -1
                ELSE    1
                THEN
        THEN
    ;

[THEN]

\G \G comments should appear immediately above or below the definition of
\G the word it belongs to. The definition line should contain no more
\G than the definition name, a stack comment and a \ comment after which
\G the wordset and pronunciation are placed.
\G An isolated block of \G comments is placed at the beginning of the 
\G glossary file.

VARIABLE gloslist       PRIVATE

0 VALUE current-comment PRIVATE
0 VALUE fixline         PRIVATE


DOC Data layout
(*
 The Glossary entries in memory have the following format.
   address of next entry        [1 cell]
   address of comment field     [1 cell]
   name                         [counted string]
   stack picture                [counted string]
   wordset                      [counted string]
   pronunciation                [counted string]
*)
ENDDOC

\G This command starts a fresh glossary.
: NEW-GLOSS   ( -- )    \ FORTH  "new-gloss"
        C" GSTART" S" FIND" EVALUATE NIP  ( lekker anders he )
        IF   S" FORGET GSTART" EVALUATE
        THEN
        S" CREATE GSTART" EVALUATE
        gloslist OFF 
        ;


CREATE oldline  PRIVATE  #256 CHARS ALLOT
0 VALUE charptr PRIVATE


-- Insert the header into the alphabetically sorted list.
: INSERT-HEADER ( addr -- )
        TO charptr
        gloslist
        BEGIN
            DUP @ 
            IF   DUP @   2 CELLS + COUNT     \ ^name
                 charptr 2 CELLS + COUNT     \ ^name
                 COMPARE-UPPERCASE 0> INVERT
            ELSE FALSE
            THEN
        WHILE  
            @
        REPEAT
        DUP @  charptr !   charptr SWAP ! 
        ;  PRIVATE

: BLorTAB?                                      \ <> --- <bool>
        charptr C@ BL = 
        charptr C@ ^I = OR 
        ;  PRIVATE

-- Scan a word on oldline through pointer charptr
: SCAN-WORD ( -- addr len )
        BEGIN
            charptr  oldline -   oldline C@ > INVERT
            BLorTAB? AND
        WHILE
            1 CHARS +TO charptr
        REPEAT
        charptr 0
        BEGIN
            charptr oldline -   oldline C@ > INVERT
            BLorTAB? 0= AND
        WHILE 
            1 CHARS +TO charptr  1+
        REPEAT 
        ;  PRIVATE

-- The first words on the line are a colon, followed by a name
-- OR it is a bunch of words followed by VALUE VARIABLE =: CONSTANT ...
-- The possibilities are endless. 
-- We can't just warn the user to fix it up himself as SEARCH-NAME is
-- used for other things too ... ????? Not any more WO !!!!

: FIND-DEF-WORD     ( -- )
        FALSE LOCAL flag
        1 +TO charptr   ( Skip count byte )
        SCAN-WORD
        BEGIN
            2DUP S" :"          COMPARE-UPPERCASE 0= flag OR TO flag
            2DUP S" CODE"       COMPARE-UPPERCASE 0= flag OR TO flag
            2DUP S" CREATE"     COMPARE-UPPERCASE 0= flag OR TO flag
            2DUP S" PREFIX"     COMPARE-UPPERCASE 0= flag OR TO flag
            2DUP S" DOER:"      COMPARE-UPPERCASE 0= flag OR TO flag
            2DUP S" DOERCODE"   COMPARE-UPPERCASE 0= flag OR TO flag
            2DUP S" VOCABULARY" COMPARE-UPPERCASE 0= flag OR TO flag
            2DUP S" VARIABLE"   COMPARE-UPPERCASE 0= flag OR TO flag
            2DUP S" VALUE"      COMPARE-UPPERCASE 0= flag OR TO flag
            2DUP S" VECTOR"     COMPARE-UPPERCASE 0= flag OR TO flag
            2DUP S" 2VARIABLE"  COMPARE-UPPERCASE 0= flag OR TO flag
            2DUP S" ALIAS"      COMPARE-UPPERCASE 0= flag OR TO flag
            2DUP S" HEADER"     COMPARE-UPPERCASE 0= flag OR TO flag  \ In Metaforth
            2DUP S" CONSTANT"   COMPARE-UPPERCASE 0= flag OR 0=
        WHILE
            2DROP  SCAN-WORD  DUP 0=            \ Yes, next word is name
        UNTIL
        THEN
        2DROP
        ;

: SEARCH-NAME
        FIND-DEF-WORD                           \ Yes, which
        SCAN-WORD                               \ Get word name
        HERE OVER 1+ CHARS ALLOT  PACK DROP 
        ;  PRIVATE

-- A stack comment is OPTIONAL !!!
: SEARCH-STACK
        0 C,

\       SCAN-WORD TUCK S" (" COMPARE-UPPERCASE IF NEGATE +TO charptr EXIT ENDIF

        charptr                                 \ Find stack comment
        BEGIN
            SCAN-WORD DUP 0=                    \ Input line empty ?
            IF   2DROP  TO charptr  EXIT  THEN  \ Yes, restore input pointer
             S" (" COMPARE-UPPERCASE 0=                   \ No, Stack comment found ?
        UNTIL

        DROP  HERE 1 CHARS - 
        BEGIN   charptr oldline -  oldline C@ > INVERT
                charptr C@ ')' <> AND
        WHILE   charptr C@ C,
                1 OVER C+!
                1 CHARS +TO charptr
        REPEAT DROP  1 CHARS +TO charptr 
        ;  PRIVATE

-- The WORDLIST name is OPTIONAL, separated by a backslash from the comment
: SEARCH-SETS

\       SCAN-WORD 0=  SWAP C@ '\' <> OR IF 0 C, EXIT ENDIF  ( skip \ if there)

        BEGIN                       ( skip \ if there and any other string )
            SCAN-WORD  DUP 0= >R
            S" \" COMPARE-UPPERCASE  R@ OR 
\           IF   0 C,  EXIT         ( No backslash found ! )
            IF   0 C, R> DROP EXIT  ( No backslash found ! )    ( CH )
            THEN
            R> 0=                   ( Backslash found ? )
        UNTIL

        SCAN-WORD OVER C@ &" = 
        IF   2DROP  0 C,  EXIT      ( No word list found ! )
        THEN
        HERE OVER 1+ CHARS ALLOT  PACK DROP  ( Word list found )
        ;  PRIVATE

-- The pronunciation is OPTIONAL, but should be within quotes.
: SEARCH-PRON
        SCAN-WORD OVER C@ &" <> 
        IF   2DROP  0 C,  EXIT 
        THEN
        HERE OVER 1+ CHARS ALLOT  PACK DROP 
        ;  PRIVATE


-- Process the header information stored in oldline.
: PROCESS-HEADER
        HERE 0 ,  current-comment ,
        oldline TO charptr
        SEARCH-NAME
        SEARCH-STACK
        SEARCH-SETS
        SEARCH-PRON
        INSERT-HEADER 
        ;  PRIVATE

-- Determine if line at HERE is a glossary comment. If so, allot it, else
-- store into oldline .
: GLOSS-COMMENT? ( -- flag )
        HERE C@ 1 >  HERE CHAR+ 2  S" \G" COMPARE-UPPERCASE 0= AND
        IF   HERE C@ 1+ CHARS ALLOT  TRUE     \ incorporate current line
        ELSE fixline 0=
             IF   HERE oldline HERE C@ 1+ CHARS CMOVE
             THEN FALSE
        THEN
        ;  PRIVATE

-- Read lines from the file fid until \G line encountered.
-- Collect all adjacent \G lines and find header line.
-- Then insert entry into list. Flag is false if no entry found.
: MAKE-GLOSSENTRY ( fid -- fid flag )
        >R
        HERE TO current-comment
        CLEAR fixline  0 oldline C!
        BEGIN
            HERE CHAR+ #255 R@ READ-LINE THROW  0=
            IF   DROP  R> FALSE  EXIT       \ end of file
            THEN
            HERE C!                         \ store length
            GLOSS-COMMENT?
        UNTIL
        oldline COUNT -TRAILING NIP 
        IF   1 TO fixline 
        THEN
        BEGIN
            HERE CHAR+ #255 R@  READ-LINE THROW
            IF   HERE C!  GLOSS-COMMENT?
            ELSE DROP FALSE
            THEN
            0=
        UNTIL
        R> TRUE   0 C, ALIGN  PROCESS-HEADER 
        ;  PRIVATE

\G This word reads a source file and builds the glossary information
\G for it in memory.
: MAKE-GLOSS  ( "name" -- )     \ FORTH  "make-glossary"
        CR ." Processing: " 
        BL PARSE-WORD  2DUP TYPE SPACE
        R/O OPEN-FILE THROW
        BEGIN  MAKE-GLOSSENTRY 0= 
        UNTIL
        CLOSE-FILE THROW 
        ;

-- Build header line for glossary entry
: BUILD-HLINE ( addr -- )
        #79 oldline C!                          \ line is 79 characters long
        oldline CHAR+ #79 BL FILL
        2 CELLS +       
        COUNT 2DUP oldline CHAR+  SWAP CMOVE            \ place name
        CHARS +
\       COUNT 2DUP oldline #20 CHARS + SWAP CMOVE       \ stack diagram
\       CHARS +
        DUP COUNT CHARS +
        COUNT 2DUP oldline #60 CHARS + SWAP CMOVE       \ wordsets field
        CHARS +
        COUNT oldline #20 CHARS + SWAP CMOVE            \ pronunciation field
        ;  PRIVATE

-- Write the glossary entry at address addr to file fid.
: WRITE-GLOSSENTRY ( addr fid -- )
        >R 
        DUP 2 CELLS + C@
        IF   DUP BUILD-HLINE           \ write header-line
             oldline CHAR+  oldline C@
             -TRAILING #20 MAX
             R@  WRITE-LINE THROW
             #79 oldline C!
             oldline CHAR+ #79 BL FILL
             '(' oldline 5 + C!
             COUNT TUCK oldline 6 + SWAP CMOVE
             ')' SWAP oldline + 6 + C!
             oldline COUNT -TRAILING DUP 6 <>
            IF      R@ WRITE-LINE THROW
            ELSE    2DROP                       \ No line with () only
            THEN
        THEN
        CELL+ @
        BEGIN
            DUP C@ 1 >
        WHILE                               \ write all comment lines without \G
            S"     " R@ WRITE-FILE THROW    \ 4 spaces in continuation lines
            DUP 4 CHARS +  OVER C@ 3 - 0 MAX  R@ WRITE-LINE THROW
            COUNT CHARS +
        REPEAT  
        DROP
        HERE 0 R> WRITE-LINE THROW          \ final empty line
        ;  PRIVATE

\G This word writes the glossary info from memory into a file.
\G The information may be collected from several source files.
: WRITE-GLOSS ( "name" -- )     \ FORTH  "write-glossary"
        CR ." Generating: "
        BL PARSE-WORD 2DUP TYPE SPACE
        W/O CREATE-FILE THROW
        gloslist
        BEGIN
            @ DUP
        WHILE
            2DUP SWAP WRITE-GLOSSENTRY
        REPEAT 
        DROP
        CLOSE-FILE THROW 
        ;

\G Make a glossary with name2 out of the origin file name1 .
: GLOSS       ( "fname1" "fname2" -- )  \ FORTH "glossary"           
        NEW-GLOSS  MAKE-GLOSS  WRITE-GLOSS 
        ;

\G A typical glossary session may look like:
\G NEW-GLOSS   MAKE-GLOSS fname1   MAKE-GLOSS fname2   WRITE-GLOSS fname3

        DEPRIVE

                            \ (* End of Source *) /
