/*
    MSGLIB - a message handling library
    Copyright (C) 1995,1997-98  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/* $Id: MSG2.H 1.5 1998/10/07 04:42:53 ska Rel ska $
   $Locker: ska $	$Name:  $	$State: Rel $

	Special shared declarations for message method #2.

*/

#ifndef __MSG2_H
#define __MSG2_H

#include <io.h>
#include <fcntl.h>
#ifndef O_BINARY
#define O_BINARY 0
#endif /*#	!defined(O_BINARY) */

#include "msgconf2.h"

#include "msg2_cfg.h"


/*********************
		Macros
				****************/
/* Strength IDs (if bit0 == 1 --> this definition is from current file */
#define STR_UNUSED 0
#define STR_FIRST 2		/* first entry with non-matching charset */
#define STR_CHARSET 4	/* first entry with matching charset */
#define STR_ALIAS 6		/* alias match */
#define STR_EXACT 8		/* match of language and charset */

/*********************
		Used types
				****************/


typedef struct MsgStr_ {	/* list of all currently locked messages */
	struct MsgStr_ *nxt;
	MSGID2 id;				/* msgID of cached message */
	int links;				/* how often this message was locked */
	msgTExit exitcode;		/* associated exit code */
	msgTClass msgclass;		/* message class */
	char text[1];			/* dynamically extended message text */
} MsgStr;

typedef struct MsgCatFile_ {	/* raw information about one catalogue */
	dword pos, limit;	/* start & end of useable area of the file */
	char *fnam;		/* name of message catalogue file, if fd != MSGFILE */
	MSGFILE fd;			/* opened catalogue file */
	struct MsgCatFile_ *nxt;
} MsgCatFile;

typedef struct {	/* list of all catalogue files */
	MsgCatFile *head, *last;	/* first & last entry within the chain of
									message catalogue files */
} msgFiles;

typedef struct {	/* information about the access path to one message group */
	dword pos;			/* start of MsgStru3's within file */
	MSGFILE file;		/* read-only opened file */
	int entries;		/* number of messages with this group */
	int strength;		/* match strength of this group with curr. env. */
} msgGrpDef;

typedef struct {	/* information about the access paths of all groups */
	msgGrpDef groups[MSGGROUPMAX];
	MsgStr *strings;	/* list of all cached message texts */
	MSGID2 lastRetrieve;	/* ID of latest msgRetrieve()'ed msg */
} msgCat;

typedef struct {	/* information of one language definition */
	MsgStru1 msg1;		/* non-variant part of stru1 item */
	msgTLngcode *aliases;	/* language aliases of this item */
	msgTCharset *charsets;	/* supported charactersets */
	MsgStru2 *msg2;		/* all stru2's of this language definition,
							the ms2_ms3 member is an absolute position */
} msgLng;

/*********************
		Global variables
				****************/

#ifdef MSG_MULTIPLE_CATALOGUES
	/* the catalogue is dynamically generated */
extern msgCat *msg_glbCat;
#else /*#	!(defined(MSG_MULTIPLE_CATALOGUES)) */
	/* the catalogue is statically generated */
extern msgCat msg_glbCat;
#endif /*#	defined(MSG_MULTIPLE_CATALOGUES) */

#ifndef MSG_SINGLE_CATALOGUE_FILE
extern msgFiles msg_catalogues;		/* list of all catalogue files */
#endif /*#	!defined(MSG_SINGLE_CATALOGUE_FILE) */

extern msgTCharset msgCurCharset;			/* Character set to use */
extern msgTLngcode msgCurLngcode;			/* language code to use */


/*********************
		Some definitions to let dynamic/static global catalogue the same
				****************/
#ifdef MSG_MULTIPLE_CATALOGUES
	/* Declare the global catalogue at first argument of functions
		and let this local parameter superceed the global variable */
#define DCL_GCAT	msgCat *msg_glbCat,
	/* Pass the global catalogue as argument to the function */
#define PAS_GCAT	msg_glbCat,
	/* Pass a local argument forth to a sub-function */
#define PAS_ARG		msg_glbCat,
	/* How to access a member */
#define GCAT msg_glbCat->
	/* pointer to global message catalogue */
#define IF_GCAT if(msg_glbCat)
#define IFN_GCAT if(!msg_glbCat)
#else /*#	!(defined(MSG_MULTIPLE_CATALOGUES)) */

	/* Leaving both empty results in functions that uses the global
		variable within themselves */
#define DCL_GCAT
#define PAS_GCAT
#define PAS_ARG
#define GCAT msg_glbCat.
#define IF_GCAT
#define IFN_GCAT if(0)
#endif /*#	defined(MSG_MULTIPLE_CATALOGUES) */



/*********************
		Low-level catalogue functions
				****************/

#define msg2OpenCat(fnam) open((fnam), O_RDONLY | O_BINARY)
/*
 *	Open a catalogue file
 */

#define msg2CloseCat(fd)	close((fd))
/*
 *	Close an opened catalogue file
 */

#define msg2DupCat(fd)	dup((fd))
/*
 *	Duplicate an opened catalogue file
 */

/*********************
		Helper functions
				****************/

int msg2IdentStru0(const MsgStru0 * const msg0);
/*
 *	Identify if msg0 contains valid values, this includes:
 *	+ msglib ID string
 *	+ msglib version
 *	+ magic number
 *
 *	Return:
 *		0: OK
 *		1: version number mismatch (unsupperted version)
 *		-1: no valid catalogue file
 */

int msg2ScanMsg1(MSGFILE f, msgLng *lng);
/*
 *	Fill 'lng' with the constant and variant fields of a language definition
 *	from file 'f'.
 *
 *	Transforms the relative ms2_ms3 members into absolute positions.
 *	The file 'f' is positioned at the first message of the first message
 *	group on success; otherwise at any arbitary place (usually not
 *	re-useable).
 *
 *	Return:	0 on success
 */

int msg2ScanLngDefs(DCL_GCAT MSGFILE f, dword iM(*)pos, dword iM(*)limit);
/*
 *	Join the catalogue with 'f' starting at pos into the passed
 *	catalogue (see DCL_GCAT above).
 *
 *	Return:	as msgDatabase()
 */

int msg_dupfd(DCL_GCAT MSGFILE f);
/*
 *	Duplicate the file descriptor
 */

 void msg_closefd(DCL_GCAT MSGFILE f);
/*
 *	Close the file descriptor if the catalogue does no longer use it
 */

int msgValidMagicNumber(msgTMagic mnum);
/*
 *	Check if 'mnum' is a valid magic number
 */

MsgStr *msg2Find(MsgStr *strings, MSGID2 id, char *text, MsgStr **prev);
/*
 *	Search the list of cached messages for the message with the id or
 *	which text member is 'text'.
 *
 *	strings, text, and prev may be NULL.
 *	id may be 0.
 *
 *	If neither id nor text is 0, the first item is returned, that matches
 *	id, text, or both.
 *
 *	*prev is updated only, if a match was found.
 *
 *	Return:
 *		NULL: no match found
 *		else: pointer to the match
 */

MsgStr *msg2MsgLock(MSGID2 id);
/*
 *	As msg2Lock(), but returns the pointer to the MsgBuf structure
 *	rather than the text member
 *
 *	Return:
 *		always != NULL
 */

char *msg2Rtrv_(DCL_GCAT MSGID2 id);
/* Unlock the previous "msgRetrieve_"ed message, then lock id.

	This is a combination of msgUnlock() and msgLock().

	Return:
		as of msg2Lock(id)
*/


msgCat *msg_mkCatBuf(void);
/* Create a dynamical buffer for a catalogue and initialize it.

	Return:
		NULL: on failure
		else: pointer to the allocated buffer
*/

void msg_initCatBuf(msgCat *cat);
/* Initialize the catalogue buffer to hold absolutely nothing
 */

/*********************
		Environment accessing functions
				****************/

#define msg2StdCats()		\
	msgCatalogue(G_ARGV[0])
/*
 *	Process the standard catalogues
 */


/*********************
		Catalogue file functions
				****************/


MsgStr *msg2Get(DCL_GCAT MSGID2 id);
/*
 *	Load the message with the ID into memory and chains it into the
 *	cached message list of the catalogue.
 *	This process "locks" the message.
 *
 *	Retrun:
 *		NULL: message not found, memory allocation rror
 *		else: pointer to the cached message
 */

void msg2Free(DCL_GCAT MSGID2 id, char *text);
/*
 *	"Unlock" the first message that matches id, text, or both.
 */


void msg2InitX(void);
/* Initializes just the retreiver portion, but does not read the
	standard catalogues */

void msg2StrClear(MsgStr **str);
/* Purge all cached messages */

#endif /*#	!defined(__MSG2_H) */
