;       Floating point routines for Modula-2
;       Using 4 and 8 byte reals in 8087 format.

; (C) Copyright 1987-1992 Fitted Software Tools. All rights reserved.


IFNDEF M2O
data            segment public 'data'
                public  DATA_M2REALS
DATA_M2REALS    equ     $
data            ends
ENDIF


code            segment public 'code'
                assume  cs:code
IFNDEF M2O
                public  M2REALS_INIT
                public  M2REALS_FPADD
                public  M2REALS_FPSUB
                public  M2REALS_FPMUL
                public  M2REALS_FPDIV
                public  M2REALS_FPNEG
                public  M2REALS_FPABS
                public  M2REALS_FPCMP
                public  M2REALS_FPFLOAT
                public  M2REALS_FPTRUNC

                public  M2REALS_FPDADD
                public  M2REALS_FPDSUB
                public  M2REALS_FPDMUL
                public  M2REALS_FPDDIV
                public  M2REALS_FPDNEG
                public  M2REALS_FPDABS
                public  M2REALS_FPDCMP
                public  M2REALS_FPDFLOAT
                public  M2REALS_FPDTRUNC
                public  M2REALS_FPSHORT
                public  M2REALS_FPLONG
ENDIF


IFDEF M2O
                dw      20                  ; # of procs
                dw      offset M2REALS_INIT     ; 0
                dw      offset M2REALS_FPADD    ; 1
                dw      offset M2REALS_FPSUB    ; 2
                dw      offset M2REALS_FPMUL    ; 3
                dw      offset M2REALS_FPDIV    ; 4
                dw      offset M2REALS_FPNEG    ; 5
                dw      offset M2REALS_FPABS    ; 6
                dw      offset M2REALS_FPCMP    ; 7
                dw      offset M2REALS_FPFLOAT  ; 8
                dw      offset M2REALS_FPTRUNC  ; 9

                dw      offset M2REALS_FPDADD   ;10
                dw      offset M2REALS_FPDSUB   ;11
                dw      offset M2REALS_FPDMUL   ;12
                dw      offset M2REALS_FPDDIV   ;13
                dw      offset M2REALS_FPDNEG   ;14
                dw      offset M2REALS_FPDABS   ;15
                dw      offset M2REALS_FPDCMP   ;16
                dw      offset M2REALS_FPDFLOAT ;17
                dw      offset M2REALS_FPDTRUNC ;18
                dw      offset M2REALS_FPSHORT  ;19
                dw      offset M2REALS_FPLONG   ;20
ENDIF


InitFlag        db      0
Has8087         db      0
RTError         dd      300h,0          ; Vector C0
SourceLineNo    dw      0               ; save line number for error reporting

RealOverflow    equ     3
RealUnderflow   equ     3
TruncOverflow   equ     3
TruncNegative   equ     3

begin           macro
                push    bp
                mov     bp, sp
                endm

return          macro   parsize
                mov     sp, bp
                pop     bp
                ret     parsize
                endm

;====================================================================
;       short STUFF
;====================================================================

ExpBias         equ     127

op1             equ     word ptr [bp+10]
op2             equ     word ptr [bp+6]
op              equ     word ptr [bp+6]


M2REALS_FPADD   proc    far
                begin
                mov     SourceLineNo, cx
fpaddsub:       ; add / sub common code
                test    Has8087, 1
                jz      fpaddsub1
                jmp     near ptr fp87add

fpaddsub1:      mov     si, 8000h       ; prepare to save signs
                mov     di, si

                mov     dx, op2+2       ; dx = hi(op2)
                mov     bx, dx          ; op2 = 0 ?
                and     bx, 7FFFh       ; ignore sign !
                or      bx, op2
                jnz     add0
                mov     ax, op1
                mov     dx, op1+2
                jmp     add11

add0:           and     di, dx          ; di = sign(op2)
                shl     dx, 1           ; dh = exp(op2)
                shr     dl, 1
                or      dl, 80h         ; dl = upper(op2)

                mov     ax, op1+2       ; ax = hi(op1)
                mov     bx, ax          ; op1 = 0 ?
                and     bx, 7FFFh       ; ignore sign !
                or      bx, op1
                jnz     add1
                mov     ax, op2
                mov     dx, op2+2
                jmp     add11

add1:           and     si, ax          ; si = sign(op1)
                shl     ax, 1           ; ah = exp(op1)
                shr     al, 1
                or      al, 80h         ; al = upper(op1)

                cmp     ah, dh          ; exp(op1) >= exp(op2) ?
                jae     add2
                xchg    si, di          ;  no, swap them
                xchg    ax, dx
                mov     bx, op1
                xchg    bx, op2
                mov     op1, bx

add2:           ; exp(op1) >= exp(op2)
                xor     di, si          ; set add/sub flag
                push    si              ; PUSH sign of result
                push    ax              ; PUSH exp(result) * 256
                push    di              ; PUSH add/sub flag

                mov     di, ax
                and     di, 0FFh        ; di = upper(op1)

                sub     ah, dh
                mov     al, ah
                xor     ah, ah
                mov     si, ax          ; si = shift count

                mov     cx, op2         ; cx = lower(op2)
                xor     dh, dh          ; dh = 0
                xor     bx, bx
                xor     ax, ax          ; bh..al cleared

add3:           cmp     si, 8
                jb      add4
                mov     al, ah
                mov     ah, bl
                mov     bl, bh
                mov     bh, cl
                mov     cl, ch
                mov     ch, dl
                mov     dl, dh
                sub     si, 8
                jmp     add3

add4:           cmp     si, 0
                je      add5
                shr     dl, 1
                rcr     cx, 1
                rcr     bx, 1
                rcr     ax, 1
                dec     si
                jmp     add4

add5:           ; add / sub
                pop     si
                test    si, si
                jnz     sub5
                mov     si, op1
                add     cx, si
                adc     dx, di
                jmp     add10
sub5:           mov     si, 0
                sub     si, ax
                mov     ax, si
                mov     si, 0
                sbb     si, bx
                mov     bx, si
                mov     si, op1
                sbb     si, cx
                sbb     di, dx
                mov     cx, si
                mov     dx, di

add10:          call    norm
add11:          return  8
M2REALS_FPADD   endp


norm            proc    near
sign            equ     byte ptr [bp+7]
exp             equ     word ptr [bp+4]
                begin
                ; adjust exp
                push    ax
                mov     ax, exp
                mov     al, ah
                xor     ah, ah
                mov     exp, ax
                pop     ax
                jmp     normmd1
norm            endp


normmd          proc    near
                begin
normmd1:        ; zero ?
                mov     si, ax
                or      si, bx
                or      si, cx
                or      si, dx
                jnz     norm0
                jmp     normzero

norm0:          cmp     dh, 0
                je      norm1
                cmp     dh, 0FFh        ; overflow or underflow ?
                jne     normover1
                not     ax              ; underflow
                not     bx
                not     cx
                not     dx
                add     ax, 1
                adc     bx, 0
                adc     cx, 0
                adc     dx, 0
                xor     sign, 80h
                jmp     norm1

normover1:      shr     dx, 1
                rcr     cx, 1
                rcr     bx, 1
                rcr     ax, 1
                mov     si, exp
                inc     si
                jmp     norm10

norm1:          mov     si, exp
norm2:          test    dl, 80h
                jnz     norm10
                shl     ax, 1
                rcl     bx, 1
                rcl     cx, 1
                rcl     dl, 1
                dec     si
                jmp     norm2

norm10:         ; round
                test    bh, 80h
                jz      norm12
                and     bh, 7Fh
                or      ax, bx
                jnz     norm11
                ; .5, round to nearest even
                test    cx, 1
                jz      norm12
norm11:         ; round up
                add     cx, 1
                adc     dl, 0
                jnc     norm12
                rcr     dl, 1
                rcr     cx, 1
                inc     si
norm12:         ; fix it up
                cmp     si, 0FFh
                jge     overflow
                cmp     si, 0
                jle     underflow
                mov     bx, si
                mov     dh, bl
                mov     ax, cx
                and     dl, 7Fh
                mov     cl, 0
                shr     dh, 1
                rcr     cl, 1
                or      dl, cl
                or      dh, sign
                return  4

normzero:       xor     ax, ax
                mov     dx, ax
                return  4

overflow:       mov     sp, bp
                pop     bp
                mov     sp, bp
                pop     bp
                mov     al, RealOverflow
                mov     cx, SourceLineNo
                les     di, RTError
                jmp     dword ptr es:[di]
underflow:      mov     sp, bp
                pop     bp
                mov     sp, bp
                pop     bp
                mov     al, RealUnderflow
                mov     cx, SourceLineNo
                les     di, RTError
                jmp     dword ptr es:[di]
normmd          endp


M2REALS_FPSUB   proc    far
                begin
                mov     SourceLineNo, cx
                xor     op2+2, 8000h
                jmp     fpaddsub
M2REALS_FPSUB   endp


M2REALS_FPMUL   proc    far
                begin
                mov     SourceLineNo, cx
                test    Has8087, 1
                jz      fpmul1
                jmp     near ptr fp87mul

fpmul1:         mov     ax, op1+2       ; ax = hi(op1)
                mov     dx, ax          ; op1 = 0 ?
                and     dx, 7FFFh
                or      dx, op1
                jz      mulzero

                mov     bx, op2+2       ; bx = hi(op2)
                mov     dx, bx          ; op2 = 0 ?
                and     dx, 7FFFh
                or      dx, op2
                jz      mulzero

                mov     si, ax
                xor     si, bx
                and     si, 8000h
                push    si                  ; PUSH sign
                shl     ax, 1
                shr     al, 1
                or      al, 80h
                shl     bx, 1
                shr     bl, 1
                or      bl, 80h

                ; take care of exp
                xchg    ah, al
                xchg    bh, bl
                mov     si, ax
                and     si, 0FFh
                mov     di, bx
                and     di, 0FFh
                sub     si, ExpBias-1
                add     si, di
                push    si                  ; PUSH exp
                xchg    ah, al
                xchg    bh, bl

                xor     ah, ah
                xor     bh, bh
                mov     si, ax
                mul     bl
                xchg    si, ax              ; si = hi * hi
                mul     op2
                mov     cx, ax
                add     si, dx              ; si:cx += hi * low
                mov     ax, bx
                mul     op1
                add     cx, ax
                adc     si, dx              ; si:cx += low * hi
                mov     ax, op1
                mul     op2
                add     cx, dx
                adc     si, 0               ; si:cx:ax = result
                ; move it
                mov     bl, ah
                mov     ah, al
                mov     al, 0
                mov     bh, cl
                mov     cl, ch
                mov     dx, si
                mov     ch, dl
                mov     dl, dh
                xor     dh, dh
                call    normmd
                return  8
mulzero:        mov     ax, 0
                mov     dx, ax
                return  8
M2REALS_FPMUL   endp


M2REALS_FPDIV   proc    far
shiftcnt        equ     byte ptr [bp-2]
q               equ     word ptr [bp-6]
                begin
                mov     SourceLineNo, cx
                test    Has8087, 1
                jz      fpdiv1
                jmp     near ptr fp87div

fpdiv1:         sub     sp, 6                   ; reserve space
                mov     ax, op1+2               ; ax = hi(op1)
                mov     dx, ax                  ; op1 = 0 ?
                and     dx, 7FFFh
                or      dx, op1
                jnz     div0
                mov     ax, 0
                mov     dx, ax
                return  8

div0:           mov     bx, op2+2               ; bx = hi(op2)
                mov     dx, bx
                and     dx, 7FFFh
                or      dx, op2
                jnz     div0a
                jmp     divbyzero
div0a:          mov     si, ax
                xor     si, bx
                and     si, 8000h
                push    si                      ; PUSH sign
                shl     ax, 1
                shr     al, 1
                or      al, 80h
                shl     bx, 1
                shr     bl, 1
                or      bl, 80h

                ; take care of exp
                xchg    ah, al
                xchg    bh, bl
                mov     si, ax
                and     si, 0FFh
                mov     di, bx
                and     di, 0FFh
                sub     si, di
                add     si, ExpBias
                push    si                      ; PUSH exp
                xchg    ah, al
                xchg    bh, bl

                xor     ah, ah
                xor     bh, bh

                mov     dx, op1
                xchg    ax, dx
                mov     cx, op2
                xchg    bx, cx

                ; dx:ax:si = op1, cx:bx:di = op2
                mov     dh, dl
                mov     dl, ah
                mov     ah, al
                mov     al, 0
                xor     si, si
                mov     ch, cl
                mov     cl, bh
                mov     bh, bl
                mov     bl, 0
                xor     di, di

                ; zero q
                mov     q, 0
                mov     q+2, 0

                mov     shiftcnt, 24
                cmp     dx, cx
                jb      div1
                ja      div2
                cmp     ax, bx
                jae     div2
div1:           inc     shiftcnt                ; op1 < op2, shift one more.
                dec     word ptr [bp-10]        ; adjust exp
div2:           ; ready to div dx:ax:si by cx:bx:di
divloop:        shl     q, 1
                rcl     q+2, 1
                ; op1 >= op2 ?
                cmp     dx, cx
                ja      divsub
                jb      divnosub
                cmp     ax, bx
                ja      divsub
                jb      divnosub
                cmp     si, di
                jb      divnosub
divsub:         sub     si, di
                sbb     ax, bx
                sbb     dx, cx
                inc     q
divnosub:       shr     cx, 1
                rcr     bx, 1
                rcr     di, 1
                dec     shiftcnt
                jnz     divloop

                ; generate rounding bits
                cmp     dx, cx
                ja      divroundup
                jb      divtrunc
                cmp     ax, bx
                ja      divroundup
                jb      divtrunc
                cmp     si, di
                ja      divroundup
                jb      divtrunc
                ; may go up/down
                mov     bx, 8000h
                jmp     divnorm
divroundup:     mov     bx, 0C000h
                jmp     divnorm
divtrunc:       xor     bx, bx

divnorm:        xor     ax, ax
                ; move result
                mov     cx, q
                mov     dx, q+2

                call    normmd
                return  8

divbyzero:      mov     al, RealOverflow
                mov     sp, bp
                pop     bp
                mov     cx, SourceLineNo
                les     di, RTError
                jmp     dword ptr es:[di]
M2REALS_FPDIV   endp


M2REALS_FPNEG   proc    far
                begin
                mov     SourceLineNo, cx
                mov     ax, op
                mov     dx, op+2
                mov     bx, ax
                or      bx, dx
                jz      neg10
                xor     dh, 80h
neg10:          return  4
M2REALS_FPNEG   endp


M2REALS_FPABS   proc    far
                begin
                mov     SourceLineNo, cx
                mov     ax, op
                mov     dx, op+2
                and     dh, 7Fh
                return  4
M2REALS_FPABS   endp


M2REALS_FPCMP   proc    far
                begin
                mov     SourceLineNo, cx
                mov     ax, 0
                mov     dx, 1
                mov     bx, op1+2
                mov     cx, op2+2
                mov     si, bx
                xor     si, cx
                test    si, 8000h
                jz      cmp0
                ; different signs
                test    bh, 80h
                jnz     cmp10
                jmp     short cmp11
cmp0:           test    bh, 80h         ; same sign
                jz      cmp1
                xchg    ax, dx
cmp1:           cmp     bx, cx          ; adjusted for positive cmp
                je      cmp2
                jb      cmp10
                jmp     short cmp11
cmp2:           mov     bx, op1
                cmp     bx, op2
                jb      cmp10
                ja      cmp11
                mov     dx, ax
                jmp     short cmp10
cmp11:          xchg    ax, dx
cmp10:          return  8
M2REALS_FPCMP   endp


M2REALS_FPFLOAT proc    far
                begin
                mov     SourceLineNo, cx
fpfloat1:       mov     cx, op          ; cx = op
                cmp     cx, 0           ; op = 0 ?
                je      floatzero

                xor     dx, dx
                push    dx              ; PUSH sign
                mov     ax, (ExpBias+23) * 256
                push    ax              ; PUSH exp
                xor     bx, bx
                xor     ax, ax
                call    norm
                return  2
floatzero:      mov     ax, 0
                mov     dx, ax
                return  2
M2REALS_FPFLOAT endp


M2REALS_FPTRUNC proc    far
                begin
                mov     SourceLineNo, cx
fptrunc1:       mov     ax, op          ; ax = low(op)
                mov     dx, op+2        ; dx = hi(op)
                mov     bx, dx          ; op = 0 ?
                and     bx, 7FFFh
                or      bx, ax
                jz      trunczero

                test    dh, 80H
                jnz     truncneg

                shl     dx, 1
                shr     dl, 1
                or      dl, 80H
                mov     cl, dh
                xor     ch, ch
                sub     cx, 23
                sub     cx, ExpBias
trunc1:         cmp     cx, 0
                jg      truncover
                je      trunc10
                shr     dl, 1
                rcr     ax, 1
                inc     cx
                jmp     trunc1

truncover:      mov     al, TruncOverflow
                mov     sp, bp
                pop     bp
                mov     cx, SourceLineNo
                les     di, RTError
                jmp     dword ptr es:[di]

truncneg:       mov     al, TruncNegative
                mov     sp, bp
                pop     bp
                mov     cx, SourceLineNo
                les     di, RTError
                jmp     dword ptr es:[di]

trunczero:      mov     ax, 0
                return  4
trunc10:        cmp     dl, 0
                jnz     truncover
                return  4
M2REALS_FPTRUNC endp

;===========================================================================
;               8087 code follows...
;===========================================================================
.8087

dop1            equ     dword ptr [bp+10]
dop2            equ     dword ptr [bp+6]
dop             equ     dword ptr [bp+6]


fp87add         proc    far
                fld     dop1
                fadd    dop2
                fstp    dop1
                wait
                mov     ax, op1
                mov     dx, op1+2
fp87add1:       return  8
fp87add         endp


fp87mul         proc    far
                fld     dop1
                fmul    dop2
                fstp    dop1
                wait
                mov     ax, op1
                mov     dx, op1+2
fp87mul1:       return  8
fp87mul         endp


fp87div         proc    far
                fld     dop1
                fdiv    dop2
                fstp    dop1
                wait
                mov     ax, op1
                mov     dx, op1+2
fp87div1:       return  8
fp87div         endp



;====================================================================
;       long STUFF
;====================================================================

;       LONGREAL format:
;               sign bit:       63
;               exponent:       52..62
;               mantissa:       0..51 (and 52 == 1)

;       high word looks like:
;               |....;....|....;....|
;                ^_ sign bit  ^ ^
;                             ^ ^______ 4 high bits of mantissa
;                             ^ ___ exponent starts here.
;                                   this bit assumed 1 in mantissa

LongExpBias     equ     1023

lqop1           equ     qword ptr [bp+16]
lqop2           equ     qword ptr [bp+8]
lqop            equ     qword ptr [bp+8]
resptr          equ     word ptr [bp+6]

lop1            equ     word ptr [bp+16]
lop2            equ     word ptr [bp+8]
lop12           equ     word ptr [bp+8]         ; REAL/LONG to LONGREAL
lop21           equ     word ptr [bp+6]         ; LONGREAL to REAL/LONG

lqop            equ     qword ptr [bp+8]        ; LONGREAL op and result
lop             equ     word ptr [bp+8]         ; same, word type

ldop            equ     dword ptr [bp+8]        ; REAL op, LONGREAL result
lqop0           equ     qword ptr [bp+6]        ; LONGREAL, non LONREAL result

dwordres        equ     dword ptr [bp+6]        ; temp work area
wordres1        equ     word ptr [bp+6]
wordres2        equ     word ptr [bp+8]

dLocSize        equ     30
dDivLocSiz      equ     54
daddacc         equ     word ptr [bp-16]
; another 8 bytes reserved for move in dadd3
daddexp         equ     word ptr [bp-26]
daddsign        equ     word ptr [bp-28]
daddflag        equ     word ptr [bp-30]
daddacc2        equ     word ptr [bp-46]
daddacc3        equ     word ptr [bp-54]


M2REALS_FPDADD  proc    far
                begin
                mov     SourceLineNo, cx
fpdaddsub:      ; add / sub common code
                test    Has8087, 1
                jz      fpdaddsub1
                jmp     near ptr fp87dadd
fpdaddsub1:
                sub     sp, dLocSize
                push    ds                      ; save DS
                mov     ax, ss
                mov     es, ax
                mov     ds, ax                  ; DS = ES := SS
                lea     di, daddacc
                mov     cx, 8
                sub     ax, ax
                rep stosw

                mov     ax, lop2+6              ; DX = high word of op2
                mov     dx, ax
                and     ax, 7FFFh               ; test op2 for zero
                or      ax, lop2
                or      ax, lop2+2
                or      ax, lop2+4
                jnz     dadd1
                jmp     dadd11
dadd1:
                mov     ax, lop1+6              ; BX = high word of op1
                mov     bx, ax
                and     ax, 7FFFh
                or      ax, lop1
                or      ax, lop1+2
                or      ax, lop1+4
                jnz     dadd2
                jmp     dadd12
dadd2:  ; isolate exponents
                and     bx, 7FFFh               ; mask sign bit off
                mov     cl, 4
                shr     bx, cl                  ; BX = exp of op1

                and     dx, 7FFFh               ; mask sign bit off
                mov     cl, 4
                shr     dx, cl                  ; DX = exp of op2

        ; which way to operate ?
                lea     di, lop1                ; DI = ^op1 (result)
                lea     si, lop2                ; SI = ^op2
                cmp     bx, dx                  ; exp(op1) >= exp(op2) ?
                jae     dadd3
                xchg    si, di
                xchg    bx, dx

dadd3:  ; save result exp, sign, set flag
                mov     daddexp, bx
                mov     ax, [di+6]
                and     ax, 8000h
                mov     daddsign, ax
                xor     ax, [si+6]
                mov     daddflag, ax
        ; save ^op1
                push    di
        ; load op2 into accumulator
                and     word ptr ss:[si+6], 0Fh ; set high word of mantissa
                or      word ptr ss:[si+6], 10h
                sub     bx, dx                  ; BX = shift count
                cmp     bx, 128
                jae     dadd5                   ; if out of range...
                lea     di, daddacc+8
                mov     ax, bx
                mov     cl, 3
                shr     ax, cl                  ; SI = bytes to shift
                sub     di, ax
        ; move 4 words
                mov     cx, 4
                rep movsw
        ; now shift remaining bits
                mov     cx, bx
                and     cx, 7
                lea     di, daddacc
                cmp     cx, 0
                jz      dadd5
dadd4:  ; shift loop
                clc
                mov     ax, [di+14]
                rcr     ax, 1
                mov     [di+14], ax
                mov     ax, [di+12]
                rcr     ax, 1
                mov     [di+12], ax
                mov     ax, [di+10]
                rcr     ax, 1
                mov     [di+10], ax
                mov     ax, [di+8]
                rcr     ax, 1
                mov     [di+8], ax
                mov     ax, [di+6]
                rcr     ax, 1
                mov     [di+6], ax
                mov     ax, [di+4]
                rcr     ax, 1
                mov     [di+4], ax
                mov     ax, [di+2]
                rcr     ax, 1
                mov     [di+2], ax
                mov     ax, [di]
                rcr     ax, 1
                mov     [di], ax
                loop    dadd4

dadd5:  ; SI = ^op1
                pop     si
        ; add / subtract
                lea     di, daddacc
                rcl     daddflag, 1
                jnc     daddadd
        ; subtract
                sub     ax, ax
                sub     ax, [di]
                mov     [di], ax
                mov     ax, 0
                sbb     ax, [di+2]
                mov     [di+2], ax
                mov     ax, 0
                sbb     ax, [di+4]
                mov     [di+4], ax
                mov     ax, 0
                sbb     ax, [di+6]
                mov     [di+6], ax
                mov     ax, [si]
                sbb     ax, [di+8]
                mov     [di+8], ax
                mov     ax, [si+2]
                sbb     ax, [di+10]
                mov     [di+10], ax
                mov     ax, [si+4]
                sbb     ax, [di+12]
                mov     [di+12], ax
                mov     ax, [si+6]
                pushf
                and     ax, 0Fh
                or      ax, 10h
                popf
                sbb     ax, [di+14]
                mov     [di+14], ax
                jmp     dadd6
daddadd:
                mov     ax, [si]
                add     ax, [di+8]
                mov     [di+8], ax
                mov     ax, [si+2]
                adc     ax, [di+10]
                mov     [di+10], ax
                mov     ax, [si+4]
                adc     ax, [di+12]
                mov     [di+12], ax
                mov     ax, [si+6]
                pushf
                and     ax, 0Fh
                or      ax, 10h
                popf
                adc     ax, [di+14]
                mov     [di+14], ax

dadd6:  ; result in daddacc
                call    dnorm
                jmp     daddret

dadd11: ; result is op1
                mov     ax, lop1
                mov     bx, lop1+2
                mov     cx, lop1+4
                mov     dx, lop1+6
                jmp     daddret
dadd12: ; result is op2
                mov     ax, lop2
                mov     bx, lop2+2
                mov     cx, lop2+4
                mov     dx, lop2+6
                ;jmp    daddret
daddret:        mov     di, resptr
                mov     [di], ax
                mov     [di+2], bx
                mov     [di+4], cx
                mov     [di+6], dx
                pop     ds
                return  18
M2REALS_FPDADD  endp


dnorm           proc    near
        ; on entry
        ;       mantissa in daddacc
        ;       exp in daddexp
        ;       sign in daddsign
        ;       return result in DX..AX

                mov     ax, daddacc+8           ; result in DX..AX
                mov     bx, daddacc+10
                mov     cx, daddacc+12
                mov     dx, daddacc+14
                mov     si, daddexp             ; exp in SI

                mov     di, ax                  ; zero ?
                or      di, bx
                or      di, cx
                or      di, dx
                or      di, daddacc
                or      di, daddacc+2
                or      di, daddacc+4
                or      di, daddacc+6
                jnz     dnorm0
                jmp     dnormzero

dnorm0:         ; over or underflow ?
                mov     di, dx
                and     di, 0FFE0h
                jz      dnorm1
                cmp     di, 0FFE0h
                jne     dnormover
        ; underflow
                not     daddacc
                not     daddacc+2
                not     daddacc+4
                not     daddacc+6
                not     ax
                not     bx
                not     cx
                not     dx
                add     daddacc, 1
                adc     daddacc+2, 0
                adc     daddacc+4, 0
                adc     daddacc+6, 0
                adc     ax, 0
                adc     bx, 0
                adc     cx, 0
                adc     dx, 0
                xor     daddsign, 8000h
                jmp     dnorm1
dnormover:
                shr     dx, 1
                rcr     cx, 1
                rcr     bx, 1
                rcr     ax, 1
                rcr     daddacc+6, 1
                rcr     daddacc+4, 1
                rcr     daddacc+2, 1
                rcr     daddacc, 1
                inc     si
dnorm1:
                test    dl, 10h
                jnz     dnorm10
                shl     daddacc, 1
                rcl     daddacc+2, 1
                rcl     daddacc+4, 1
                rcl     daddacc+6, 1
                rcl     ax, 1
                rcl     bx, 1
                rcl     cx, 1
                rcl     dx, 1
                dec     si
                jmp     dnorm1
dnorm10:        ; round
                mov     di, daddacc+6
                test    di, 8000h
                jz      dnorm12
                and     di, 7FFFh
                or      di, daddacc
                or      di, daddacc+2
                or      di, daddacc+4
                jnz     dnorm11
                ; 0.5 round to nearest even
                test    ax, 1
                jz      dnorm12
dnorm11:        ; round up
                add     ax, 1
                adc     bx, 0
                adc     cx, 0
                adc     dx, 0
                test    dl, 20h
                jz      dnorm12
                shr     dx, 1
                rcr     cx, 1
                rcr     bx, 1
                rcr     ax, 1
                inc     si
dnorm12:        ; mantissa done, check exp
                cmp     si, 07FFh
                jl      dnorm12a
                begin   ; trash for overflow routine
                jmp     overflow
dnorm12a:       cmp     si, 0
                jg      dnorm12b
                begin   ; trash for underflow routine
                jmp     underflow
dnorm12b:       shl     si, 1
                shl     si, 1
                shl     si, 1
                shl     si, 1
                and     dx, 0Fh
                or      dx, si
                or      dx, daddsign
                ret
dnormzero:      sub     ax, ax
                mov     bx, ax
                mov     cx, ax
                mov     dx, ax
                ret
dnorm           endp


M2REALS_FPDSUB  proc    far
                begin
                mov     SourceLineNo, cx
                xor     lop2+6, 8000h
                jmp     fpdaddsub
M2REALS_FPDSUB  endp

dmulzero:                               ; placed here to be in range of jz
                xor     ax, ax
                mov     bx, ax
                mov     cx, ax
                mov     dx, ax
                jmp     daddret


M2REALS_FPDMUL  proc    far
                begin
                mov     SourceLineNo, cx
                test    Has8087, 1
                jz      fpdmul1
                jmp     near ptr fp87dmul
fpdmul1:
                sub     sp, dLocSize
                push    ds
                mov     ax, ss          ; set DS = ES := SS
                mov     ds, ax
                mov     es, ax
                lea     di, daddacc     ; clear acc
                mov     cx, 8
                sub     ax, ax
                rep stosw

                mov     ax, lop1+6
                mov     dx, ax          ; DX = high(op1)
                and     ax, 07FFFh
                or      ax, lop1+4
                or      ax, lop1+2
                or      ax, lop1
                jz      dmulzero

                mov     ax, lop2+6
                mov     bx, ax          ; BX = high(op2)
                and     ax, 07FFFh
                or      ax, lop2+4
                or      ax, lop2+2
                or      ax, lop2
                jz      dmulzero

                mov     ax, dx          ; get sign of result
                xor     ax, bx
                and     ax, 8000h
                mov     daddsign, ax

                mov     cl, 4           ; calculate exp of result
                shr     dx, cl
                and     dx, 07FFh
                mov     cl, 4
                shr     bx, cl
                and     bx, 07FFh
                sub     dx, LongExpBias-12
                add     dx, bx
                mov     daddexp, dx

                and     lop1+6, 0Fh     ; fix high word for mul
                or      lop1+6, 10h
                and     lop2+6, 0Fh
                or      lop2+6, 10h

                ; set up for multiply
                lea     di, daddacc     ; load pointers
                lea     si, lop1
                lea     bx, lop2
                mov     cx, 4
dmulloop:
                mov     ax, [si]
                mul     word ptr [bx]
                add     [di], ax
                adc     [di+2], dx
                jnc     dml1
                call    dmlcarry
dml1:
                inc     di
                inc     di
                mov     ax, [si]
                mul     word ptr [bx+2]
                add     [di], ax
                adc     [di+2], dx
                jnc     dml2
                call    dmlcarry
dml2:
                inc     di
                inc     di
                mov     ax, [si]
                mul     word ptr [bx+4]
                add     [di], ax
                adc     [di+2], dx
                jnc     dml3
                call    dmlcarry
dml3:
                inc     di
                inc     di
                mov     ax, [si]
                mul     word ptr [bx+6]
                add     [di], ax
                adc     [di+2], dx
                jnc     dml4
                call    dmlcarry
dml4:
                sub     di, 4
                inc     si
                inc     si
                loop    dmulloop

                call    dnorm
                jmp     daddret
M2REALS_FPDMUL  endp


dmlcarry        proc    near            ; propagate carry
                push    di
                pushf
                add     di, 4
                popf
dmlcloop:       adc     word ptr [di], 0
                pushf
                inc     di
                inc     di
                popf
                jc      dmlcloop
                pop     di
                ret
dmlcarry        endp


M2REALS_FPDDIV  proc    far
                begin
                mov     SourceLineNo, cx
                test    Has8087, 1
                jz      fpddiv1
                jmp     near ptr fp87ddiv
fpddiv1:
                sub     sp, dDivLocSiz
                push    ds
                mov     ax, ss          ; DS := ES := SS
                mov     ds, ax
                mov     es, ax
                lea     di, daddacc     ; clear acc
                mov     cx, 8
                sub     ax, ax
                rep stosw

                mov     ax, lop1+6
                mov     dx, ax          ; DX = high(op1)
                and     ax, 07FFFh
                or      ax, lop1+4
                or      ax, lop1+2
                or      ax, lop1
                jnz     ddiv0
                jmp     ddivzero
ddiv0:
                mov     ax, lop2+6
                mov     bx, ax          ; BX = high(op2)
                and     ax, 07FFFh
                or      ax, lop2+4
                or      ax, lop2+2
                or      ax, lop2
                jnz     ddiv0a
                jmp     ddivover
ddiv0a:
                mov     ax, dx          ; daddsign := sign of result
                xor     ax, bx
                and     ax, 8000h
                mov     daddsign, ax

                mov     cl, 4           ; daddexp := exp of result
                shr     dx, cl
                and     dx, 07FFh
                mov     cl, 4
                shr     bx, cl
                and     bx, 07FFh
                add     dx, LongExpBias
                sub     dx, bx
                mov     daddexp, dx

        ; load op1 in daddacc2
                mov     ax, lop1
                mov     bx, lop1+2
                mov     si, lop1+4
                mov     dx, lop1+6
                and     dx, 0Fh
                or      dl, 10h
                mov     cx, 11
ddiv1:          shl     ax, 1
                rcl     bx, 1
                rcl     si, 1
                rcl     dx, 1
                loop    ddiv1
                mov     daddacc2, 0
                mov     daddacc2+2, 0
                mov     daddacc2+4, 0
                mov     daddacc2+6, 0
                mov     daddacc2+8, ax
                mov     daddacc2+10, bx
                mov     daddacc2+12, si
                mov     daddacc2+14, dx

        ; load op2 in acc3+6:acc+4:acc3+2:DI:SI:DX:BX
                mov     ax, lop2
                mov     bx, lop2+2
                mov     si, lop2+4
                mov     dx, lop2+6
                and     dx, 0Fh
                or      dl, 10h
                mov     cx, 11
ddiv2:          shl     ax, 1
                rcl     bx, 1
                rcl     si, 1
                rcl     dx, 1
                loop    ddiv2
                mov     daddacc3+6, dx
                mov     daddacc3+4, si
                mov     daddacc3+2, bx
                mov     di, ax
                xor     si, si
                xor     dx, dx
                xor     bx, bx
        ; prepare loop count
                mov     cx, 53
                mov     ax, daddacc2+14
                cmp     ax, daddacc3+6
                ja      ddiv3
                jb      ddiv4
                mov     ax, daddacc2+12
                cmp     ax, daddacc3+4
                ja      ddiv3
                jb      ddiv4
                mov     ax, daddacc2+10
                cmp     ax, daddacc3+2
                ja      ddiv3
                jb      ddiv4
                cmp     daddacc2+8, di
                jae     ddiv3
ddiv4:          inc     cx              ; op1 < op2, need 1 more loop
                dec     daddexp

ddiv3:  ; daddacc = Q, daddacc2 = op1, daddacc3 = op2
ddivloop:       shl     daddacc+8, 1            ; shift Q left
                rcl     daddacc+10, 1
                rcl     daddacc+12, 1
                rcl     daddacc+14, 1

                ; op1 >= op2 ?
                mov     ax, daddacc2+14
                cmp     ax, daddacc3+6
                ja      ddivsub
                jb      ddivnosub
                mov     ax, daddacc2+12
                cmp     ax, daddacc3+4
                ja      ddivsub
                jb      ddivnosub
                mov     ax, daddacc2+10
                cmp     ax, daddacc3+2
                ja      ddivsub
                jb      ddivnosub
                cmp     daddacc2+8, di
                ja      ddivsub
                jb      ddivnosub
                cmp     daddacc2+6, si
                ja      ddivsub
                jb      ddivnosub
                cmp     daddacc2+4, dx
                ja      ddivsub
                jb      ddivnosub
                cmp     daddacc2+2, bx
                jb      ddivnosub

ddivsub:        sub     daddacc2+2, bx
                sbb     daddacc2+4, dx
                sbb     daddacc2+6, si
                sbb     daddacc2+8, di
                mov     ax, daddacc3+2
                sbb     daddacc2+10, ax
                mov     ax, daddacc3+4
                sbb     daddacc2+12, ax
                mov     ax, daddacc3+6
                sbb     daddacc2+14, ax
                inc     daddacc+8

ddivnosub:      shr     daddacc3+6, 1
                rcr     daddacc3+4, 1
                rcr     daddacc3+2, 1
                rcr     di, 1
                rcr     si, 1
                rcr     dx, 1
                rcr     bx, 1
                loop    ddivloop

        ; generate rounding bits
                mov     ax, daddacc2+14
                cmp     ax, daddacc3+6
                ja      ddivroundup
                jb      ddivtrunc
                mov     ax, daddacc2+12
                cmp     ax, daddacc3+4
                ja      ddivroundup
                jb      ddivtrunc
                mov     ax, daddacc2+10
                cmp     ax, daddacc3+2
                ja      ddivroundup
                jb      ddivtrunc
                cmp     daddacc2+8, di
                ja      ddivroundup
                jb      ddivtrunc
                cmp     daddacc2+6, si
                ja      ddivroundup
                jb      ddivtrunc
                cmp     daddacc2+4, dx
                ja      ddivroundup
                jb      ddivtrunc
                cmp     daddacc2+2, bx
                ja      ddivroundup
                jb      ddivtrunc
        ; may go either way
                mov     daddacc+6, 8000h
                jmp     ddivnorm
ddivroundup:    mov     daddacc+6, 0C000h
                jmp     ddivnorm
ddivtrunc:      mov     daddacc+6, 0
ddivnorm:       call    dnorm
                jmp     daddret
ddivzero:       xor     ax, ax
                xor     bx, bx
                xor     cx, cx
                xor     dx, dx
                jmp     daddret
ddivover:
                begin   ; trash for fperr rtn
                jmp     overflow
M2REALS_FPDDIV  endp


M2REALS_FPDNEG  proc    far
                begin
                mov     SourceLineNo, cx
                mov     ax, lop
                or      ax, lop+2
                or      ax, lop+4
                or      ax, lop+6
                jz      fpdneg10
                mov     ax, lop+6
                xor     ah, 80h
                mov     lop+6, ax
fpdneg10:       mov     bx, resptr
                mov     ax, lop
                mov     ss:[bx], ax
                mov     ax, lop+2
                mov     ss:[bx+2], ax
                mov     ax, lop+4
                mov     ss:[bx+4], ax
                mov     ax, lop+6
                mov     ss:[bx+6], ax
                return  10
M2REALS_FPDNEG  endp


M2REALS_FPDABS  proc    far
                begin
                mov     SourceLineNo, cx
                mov     bx, resptr
                mov     ax, lop
                mov     ss:[bx], ax
                mov     ax, lop+2
                mov     ss:[bx+2], ax
                mov     ax, lop+4
                mov     ss:[bx+4], ax
                mov     ax, lop+6
                and     ah, 07Fh
                mov     ss:[bx+6], ax
                return  10
M2REALS_FPDABS  endp


M2REALS_FPDCMP  proc    far
lcop1           equ     word ptr [bp+14]
lcop2           equ     word ptr [bp+6]
                begin
                mov     SourceLineNo, cx
                mov     ax, 0
                mov     dx, 1
                mov     bx, lcop1+6
                mov     cx, lcop2+6
                mov     si, bx
                xor     si, cx
                test    si, 8000h
                jz      dcmp0
                ; different signs
                test    bh, 80h
                jnz     dcmp10
                jmp     short dcmp11
dcmp0:  ; same sign
                test    bh, 80h
                jz      dcmp1
                xchg    ax, dx
dcmp1:  ; adjusted for positive compare
                cmp     bx, cx
                je      dcmp2
                jb      dcmp10
                jmp     short dcmp11
dcmp2:          mov     bx, lcop1+4
                cmp     bx, lcop2+4
                je      dcmp3
                jb      dcmp10
                jmp     short dcmp11
dcmp3:          mov     bx, lcop1+2
                cmp     bx, lcop2+2
                je      dcmp4
                jb      dcmp10
                jmp     short dcmp11
dcmp4:          mov     bx, lcop1
                cmp     bx, lcop2
                jb      dcmp10
                ja      dcmp11
                mov     dx, ax
                jmp     short dcmp10
dcmp11:         xchg    ax, dx
dcmp10:         return  16
M2REALS_FPDCMP  endp


M2REALS_FPDFLOAT proc    far
                begin
                mov     SourceLineNo, cx
fpdfloat1:
                sub     sp, dLocSize
                push    ds                      ; save DS
                mov     ax, ss
                mov     es, ax
                mov     ds, ax                  ; DS = ES := SS

                mov     ax, lop12
                mov     dx, lop12+2
                mov     bx, ax
                or      bx, dx
                jz      dfloatzero

                mov     daddexp, 0
                mov     daddsign, 0
                mov     daddacc, 0
                mov     daddacc+2, 0
                mov     daddacc+4, 0
                mov     daddacc+6, 0
                mov     daddacc+8, ax
                mov     daddacc+10, dx
                mov     daddacc+12, 0
                mov     daddacc+14, 0
                mov     daddexp, LongExpBias+52
                call    dnorm
dfloatret:      mov     di, resptr
                mov     [di], ax
                mov     [di+2], bx
                mov     [di+4], cx
                mov     [di+6], dx
                pop     ds
                return  6
dfloatzero:
                xor     ax, ax
                mov     bx, ax
                mov     cx, ax
                mov     dx, ax
                jmp     dfloatret
M2REALS_FPDFLOAT endp


M2REALS_FPDTRUNC proc    far
                begin
                mov     SourceLineNo, cx
fpdtrunc1:
                mov     ax, lop21
                mov     bx, lop21+2
                mov     cx, lop21+4
                mov     dx, lop21+6
                mov     si, dx
                and     si, 7FFFh
                or      si, cx
                or      si, bx
                or      si, ax
                jz      dtrunczero
                test    dh, 80h
                jnz     dtruncneg

                mov     si, dx
                mov     cl, 4
                shr     si, cl
                sub     si, 52
                sub     si, LongExpBias

                and     dl, 0Fh
                or      dl, 10h
                mov     cx, lop21+4     ; CL used above!

dtrunc1:        cmp     si, 0
                jg      dtruncover
                je      dtrunc10
                shr     dl, 1
                rcr     cx, 1
                rcr     bx, 1
                rcr     ax, 1
                inc     si
                jmp     dtrunc1

dtruncover:     mov     al, TruncOverflow
                mov     sp, bp
                pop     bp
                mov     cx, SourceLineNo
                les     di, RTError
                jmp     dword ptr es:[di]

dtruncneg:      mov     al, TruncNegative
                mov     sp, bp
                pop     bp
                mov     cx, SourceLineNo
                les     di, RTError
                jmp     dword ptr es:[di]

dtrunczero:     xor     ax, ax
                mov     dx, ax
                return  8
dtrunc10:       cmp     dl, 0
                jnz     dtruncover
                cmp     cx, 0
                jnz     dtruncover
                mov     dx, bx
                return  8
M2REALS_FPDTRUNC endp


M2REALS_FPSHORT proc    far
                begin
                mov     SourceLineNo, cx
                test    Has8087, 1
                jz      fpshort1
                jmp     near ptr fp87short
fpshort1:
                mov     ax, lop21
                mov     bx, lop21+2
                mov     cx, lop21+4
                mov     dx, lop21+6             ; op in DX:CX:BX:AX
                mov     si, ax
                or      si, bx
                or      si, cx
                or      si, dx
                jz      fpshortzero
                mov     di, dx
                and     di, 8000h
                push    di          ; sign
                mov     si, dx
                and     si, 7FF0h
                mov     di, cx      ; save cx
                mov     cl, 4
                shr     si, cl
                sub     si, LongExpBias-ExpBias
                jc      fpshortzero0    ; too small
                cmp     si, 0FFh
                jb      fpshort2
                begin   ; trash for fperr routine
                jmp     overflow
fpshort2:
                push    si          ; exp
                and     dx, 0Fh
                or      dx, 10h
                mov     cx, 3
fpshort3:       shl     ax, 1
                rcl     bx, 1
                rcl     di, 1
                rcl     dx, 1
                loop    fpshort3
                mov     cx, di      ; get it (cx) back
                call    normmd
fpshortzero:
                return  8
fpshortzero0:
                xor     ax, ax
                xor     dx, dx
                return  8
M2REALS_FPSHORT endp


M2REALS_FPLONG  proc    far
                begin
                mov     SourceLineNo, cx
                test    Has8087, 1
                jz      fplong1
                jmp     near ptr fp87long
fplong1:
                sub     sp, dLocSize
                push    ds                      ; save DS
                mov     ax, ss
                mov     es, ax
                mov     ds, ax                  ; DS = ES := SS
        ; zero accumulator
                lea     di, daddacc
                mov     cx, 8
                sub     ax, ax
                rep stosw

                mov     ax, lop12
                mov     dx, lop12+2
                mov     bx, ax
                or      bx, dx
                jz      fplongzero
                mov     daddacc+10, ax
                mov     ax, dx
                xor     ah, ah
                or      al, 80h
                mov     daddacc+12, ax
                mov     si, dx
                and     si, 8000h
                mov     daddsign, si
                shl     dx, 1
                mov     al, dh
                xor     ah, ah
                add     ax, LongExpBias-ExpBias+13
                mov     daddexp, ax
                call    dnorm
fplongret:      mov     di, resptr
                mov     [di], ax
                mov     [di+2], bx
                mov     [di+4], cx
                mov     [di+6], dx
                pop     ds
                return  6

fplongzero:
                xor     ax, ax
                mov     bx, ax
                mov     cx, ax
                mov     dx, ax
                jmp     fplongret
M2REALS_FPLONG  endp


fp87dadd        proc    far
                fld     lqop1
                fadd    lqop2
                mov     bx, resptr
                fstp    qword ptr ss:[bx]
                wait
                return  18
fp87dadd        endp


fp87dmul        proc    far
                fld     lqop1
                fmul    lqop2
                mov     bx, resptr
                fstp    qword ptr ss:[bx]
                wait
                return  18
fp87dmul        endp


fp87ddiv        proc    far
                fld     lqop1
                fdiv    lqop2
                mov     bx, resptr
                fstp    qword ptr ss:[bx]
                wait
                return  18
fp87ddiv        endp


fp87short       proc    far
                fld     lqop0
                fstp    dwordres
                wait
                mov     ax, wordres1
                mov     dx, wordres2
                return  8
fp87short       endp


fp87long        proc    far
                fld     ldop
                mov     bx, resptr
                fstp    qword ptr ss:[bx]
                wait
                return  6
fp87long        endp


M2REALS_INIT    proc    far
cw87            equ     word ptr [bp-2]
                test    InitFlag, 1
                jz      init
                ret
init:           inc     InitFlag
                begin
                sub     sp, 2
                FNINIT
                FNSTCW  cw87
                ;cmp     byte ptr cw87+1, 03H -- MASM 5.0 generates bad code...
                mov     ax, cw87
                cmp     ah, 03H
                jne     no87
                mov     cw87, 0320H
                FLDCW   cw87
                inc     Has8087
no87:           return
M2REALS_INIT    endp

code            ends

end
