\Th.XPL		23-Jul-2011		Loren Blaney	loren.blaney@gmail.com
\Displays thumbnail images for all the recognized files (BMP, LBM, IFF, BBM,
\ PCX, and GIF) in a directory.
\
\Compile with XPLPX (xpx.bat)
\
\This program is free software; you can redistribute it and/or modify it under
\ the terms of the GNU General Public License version 2 as published by the
\ Free Software Foundation.
\This program is distributed in the hope that it will be useful, but WITHOUT
\ ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
\ FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
\ details.
\
\This program uses 1024x768x24 graphics. The width of a thumbnail image is
\ determined by the width of its legend, for example:
\
\	FILENAME.GIF
\	1024x1024x24
\	10-Jul-2011
\	23:09:17
\
\The 12 characters are displayed with an 8-pixel wide font, giving a total of
\ 8*12 = 96 pixels. Ten thumbnails fit in the width of the screen, since 1024/96
\ = 10.67. The actual spacing is: 96*10 for the thumbnails + 6*9 for the spaces
\ between them, and + 5 + 5 for borders on the left and right sides.
\
\The height of the legend with an 8-pixel high font is 8*4 = 32. Combining this
\ with 96 pixels (which gives a square thumbnail area) fits 6 thumbnails in the
\ height of the screen: 768/(32+96) = 6.0.
\
\Thus 60 thumbnails fit on a screen in a 10x6 array.

string 0;		\all strings used here are terminated with 0 (ASCIIZ)
include	C:\CXPL\CODES;	\intrinsic routine code definitions
def	IntSize = 4,	\number of bytes in an integer
	Esc = $1B;	\ASCII control character

int	CpuReg,		\address of CPU register array (from GetReg)
	DataSeg,	\segment address where the .exe loader put our data
	PspSeg,		\Program Segment Prefix segment, holds command line
	Files,		\number of file images in directory
	FileHandle,	\image file handle (DOS needs this to read files)
	SwExt, SwDate, SwSize,	\command-line switches for sorting order
	Image,		\array: full-size graphic image promoted to 24-bit RGB
	Width, Height,	\full-size image dimensions (pixels)
	Depth,		\number of bits of color (4 bits gives 16 colors, etc.)
	ThumbX, ThumbY;	\upper-left corner of frame for thumbnail image (pixels)

char	PathName($100); \path name from command line [drive:][path]
def	WidthMax=5000;	\maximum width of an image that can be displayed
def	FilesMax=2000;	\maximum number of file names allowed
char	FileName(FilesMax, 14); \(8+1+3+1=13 characters long maximum)
def	ID = 13;	\the 14th byte is a file type identifier:
			\ 0=BMP, 1=LBM, 2=BBM, 3=IFF, 4=PCX, 5=GIF
int	FileDate(FilesMax), \date of file (compressed format)
	FileTime(FilesMax), \time of file (compressed format)
	FileSize(FilesMax); \size of file (bytes)



proc	Fatal(Str);		\Display error message and exit program
char	Str;
begin
SetVid($03);			\make sure screen is in normal text mode
Text(0, Str);
CrLf(0);
CrLf(0);
Text(0, "Press any key to exit program"); \ deal with windows that slam shut
OpenI(1);			\wait for new keystroke
if ChIn(1) then [];
exit;
end;	\Fatal



func	CallInt(Int, AX, BX, CX, DX, BP, DS, ES); \Call software interrupt
int	Int, AX, BX, CX, DX, BP, DS, ES; \(unused arguments need not be passed)
begin
CpuReg(0):= AX;
CpuReg(1):= BX;
CpuReg(2):= CX;
CpuReg(3):= DX;
CpuReg(6):= BP;
CpuReg(9):= DS;
CpuReg(11):= ES;
SoftInt(Int);
return CpuReg(0);		\return AX register
end;	\CallInt



func	RevBytes(N);		\Reverse the order of the bytes in a 32-bit word
int	N;			\ (Swap endians)
return N<<24 ! N>>24 ! N<<8 & $00FF0000 ! N>>8 & $0000FF00;



proc	StrNCopy(A, B, N);	\Copy string that is N bytes long: A --> B
char	A, B;	\strings: B must already have enough space "Reserved"
int	N;	\number of bytes to copy
int	I;
for I:= 0, N-1 do
	B(I):= A(I);



func	StrNCmp(A, B, N);	\Compare string A to string B up to N bytes long
\This returns:
\	>0 if A > B
\	=0 if A = B
\	<0 if A < B
\This provides a general string compare, for example:
\ if StrNCmp(A, B, N) >= 0 then...	(if A >= B then...)
char	A, B;	\strings to be compared (must be right justified)
int	N;	\number of bytes to compare
int	I;
begin
for I:= 0, N-1 do
	if A(I) # B(I) then
		return A(I) - B(I);
return 0;			\they're equal
end;	\StrNCmp



func	StrNLen(Str, N);	\Returns number of chars in a string
char	Str;
int	N;	\maximum length, in case string is not terminated with a zero
int	I;
begin
I:= 0;
while Str(I) & I<N do I:= I+1;
return I;
end;	\StrNLen



proc	StrCat(S0, S1, S2);	\Concatenate S0 and S1 and output to S2
char	S0, S1, S2;
int	I, J;
begin
I:= 0;
while S0(I) \#0\ do [S2(I):= S0(I);  I:= I+1];
J:= 0;
loop	begin
	S2(I):= S1(J);
	if S1(J) = 0 then quit;
	I:= I+1;
	J:= J+1;
	end;
end;	\StrCat



proc	Beep;			\A not-too-obnoxious beep
begin
Sound(false, 1, 1000);		\synchronize with system timer to make tone a
Sound(true, 1, 3000);		\ consistent duration and a consistent sound
end;	\Beep



func	LookKey;		\Returns next keystroke without reading it in
int	SC, Ch;			\ so it can be read in later (e.g: IntIn(0))
begin
repeat until KeyHit;
SC:= CallInt($16, $0100);	\BIOS function $01
Ch:= SC & $FF;
if Ch = 0 then Ch:= -(SC>>8);	\return non-ASCII chars as negative scan code
return Ch;
end;	\LookKey



proc	ShowRectangle(X, Y, W, H, C); \Display a rectangle
int	X, Y, W, H,	\upper-left corner coordinates, width, height (pixels)
	C;		\color
begin
Move(X, Y);
Line(X+W-1, Y, C);
Line(X+W-1, Y+H-1, C);
Line(X, Y+H-1, C);
Line(X, Y, C);
end;	\ShowRectangle



proc	TimeOut(Dev, Time);	\Display time e.g: 14:25:06
int	Dev, Time;	\time in DOS packed format
int	S, M, H;	\second, minute, hour
begin
S:= Time<<1 & $003E;		\0..58
M:= Time>>5 & $003F;		\0..59
H:= (Time>>11 & $001F);		\0..23

if H < 10 then ChOut(Dev, ^0);
IntOut(Dev, H);
ChOut(Dev, ^:);
if M < 10 then ChOut(Dev, ^0);
IntOut(Dev, M);
ChOut(Dev, ^:);
if S < 10 then ChOut(Dev, ^0);
IntOut(Dev, S);
end;	\TimeOut



proc	DateOut(Dev, Date);	\Display date e.g: 03-Jul-2011
int	Dev, Date;	\date in DOS packed format
int	D, M, Y,	\day, month, year
	I, J;
char	Str;
begin
D:= Date & $001F;		\1..31
M:= Date>>5 & $000F;		\1..12
Y:= (Date>>9 & $007F) + 1980;	\1980..2107

if D < 10 then ChOut(Dev, ^0);
IntOut(Dev, D);
ChOut(Dev, ^-);
Str:= "JanFebMarAprMayJunJulAugSepOctNovDec";
J:= 3*(M-1);
for I:= 0, 3-1 do ChOut(Dev, Str(I+J));
ChOut(Dev, ^-);
IntOut(Dev, Y);
end;	\DateOut

\-------------------------------------------------------------------------------

proc	ShowThumbnail;		\Display a thumbnail for Image
\Inputs: Image, Width, Height
int	XC, YC,	\upper-left corner of actual thumbnail image
	I, J,	\pixel coordinates in image file
	X, Y,	\pixel coordinates for thumbnail image
	Size,	\size of larger dimension: width or height (pixels)
	II, JJ;	\decision variables for when to plot a pixel
int	C, R, G, B, AR(96), AG(96), AB(96), AN(96);
begin
Size:= Width;				\set Size to larger dimension
if Height > Size then Size:= Height;

\Center thumbnail within frame
if Size >= 96 then
	begin
	XC:= ThumbX + (Size-Width)*96/Size/2;
	YC:= ThumbY + (Size-Height)*96/Size/2;
	end
else	begin		\center small images
	XC:= ThumbX + (96-Width)/2;
	YC:= ThumbY + (96-Height)/2;
	end;

for X:= 0, 96-1 do
	begin
	AN(X):= 0;		\initialize RGB area counter
	AR(X):= 0;		\ and accumulators
	AG(X):= 0;
	AB(X):= 0;
	end;
Y:= 0;
JJ:= 0;
for J:= 0, Height-1 do
	begin
	X:= 0;
	II:= 0;
	for I:= 0, Width-1 do
		begin
		C:= Image(I, J);		\get pixel's color
		R:= (C & $FF0000) >> 16;	\accumulate its RGB values
		G:= (C & $00FF00) >> 8;
		B:= (C & $0000FF) >> 0;
		AN(X):= AN(X) + 1;		\count number of RGB samples
		AR(X):= AR(X) + R;		\accumulate sample
		AG(X):= AG(X) + G;
		AB(X):= AB(X) + B;
		if II >= Size then
			begin
			II:= II - Size;
			if JJ >= Size then
				begin
				R:= AR(X)/AN(X);	\average the RGB values
				G:= AG(X)/AN(X);
				B:= AB(X)/AN(X);
				C:= R<<16 + G<<8 + B;	\combine into 24-bits
				Point(X+XC, Y+YC, C);	\display thumbnail pixel
				end;
			X:= X+1;
			end;
		II:= II + 96;
		end;
	if JJ >= Size then
		begin
		JJ:= JJ - Size;
		Y:= Y+1;
		for X:= 0, 96-1 do
			begin
			AN(X):= 0;		\reset RGB counter
			AR(X):= 0;		\ and accumulators
			AG(X):= 0;
			AB(X):= 0;
			end;
		end;
	JJ:= JJ + 96;
	end;

ShowRectangle(ThumbX, ThumbY, 96, 96, $808080);		\put a frame around it
end;	\ShowThumbnail



func	EnoughMemory;		\Check if enough memory to create Image array
begin
if Width*(Height+1)*IntSize < Free-WidthMax*IntSize-100 then return true;

\Let 'em know Windows XP is stingy with memory
Attrib($00FFFFFF);
Move((96-10*8)/2+ThumbX, 16*1+ThumbY);	Text(6, "Not enough");
Move((96- 9*8)/2+ThumbX, 16*2+ThumbY);	Text(6, "memory to");
Move((96-10*8)/2+ThumbX, 16*3+ThumbY);	Text(6, "show image");
Beep;
return false;
end;	\EnoughMemory

\###############################################################################

proc	ShowGIF;		\Display a GIF image file
\Based on CompuServe document 89a, and GifSlow.pas (1/19/88) by Jim Griebel
\"GIF" and "Graphics Interchange Format" are trademarks of CompuServe, Inc.
\Note that there are a maximum of 256 colors in a GIF image.
int	XC, YC,		\coordinates of current pixel to be plotted
	X1, Y1,		\offset to upper-left corner of image (usually 0,0)
	Pass,		\pass counter, used for interlace (0..3)
	HaveCTbl,	\flag: color table follows current block
	Interlaced,	\flag: scan lines are interlaced (usually false)
	ColorTblSize,	\number of bytes in color table
	ByteCtr,	\number of bytes of data remaining to be read from block
	SP,		\stack pointer
	J, T,		\index and temporary scratch

	Code,		\compressed code (up to 12 bits long)
	CodeBuf,	\3-byte input buffer
	ReadMask,	\Code AND mask for current CodeSize
	ShiftCtr,	\8-counter; counts number of bits shifted in CodeBuf

	ClearCode,	\clear code resets CodeSize and Inx
	EOFCode,	\end-of-image code
	OldCode,	\old Code
	NewCode,	\new Code
	Inx0,		\initial empty location in StrTbl and PixTbl
	Inx,		\table index: next empty location in table
	CodeSize,	\Code size read from header blk (LZW Minimum Code Size)
	CodeSize0,	\initial code size
	FinPix,		\final pixel color (not a code)
	BitMask;	\AND mask for pixel colors (has Depth number of 1 bits)

int	StrTbl(4096),	\table of indexes to substrings
	PixTbl(4096);	\pixel colors
def	StackSize = 4096; \maximum number of items in a repeated sequence
int	Stack(StackSize);
int	CLUT(256);	\Color Look-Up Table (R<<16 + G<<8 + B)



	proc	PlotPix(Color);	\Plot pixel in current raster position
	int	Color;
	int	Tbl1, Tbl2;
	begin
	if XC<Width & YC<Height then
		Image(XC, YC):= CLUT(Color);

	XC:= XC+1;		\move to next raster position
	if XC >= Width then
		begin
		XC:= 0;
		if Interlaced then
			begin
			Tbl1:= [8, 8, 4, 2];
			Tbl2:= [0, 4, 2, 1];
			YC:= YC + Tbl1(Pass);
			if YC >= Height then
				begin
				Pass:= Pass+1 & $03;
				YC:= Tbl2(Pass);
				end;
			end
		else	YC:= YC+1;
		end;
	end;	\PlotPix



	proc	Push(B);	\Push an item onto the Stack
	int	B;
	begin
	if SP >= StackSize then Fatal("GIF stack overflow");
	Stack(SP):= B;
	SP:= SP+1;
	end;	\Push



	func	Pop;		\Pop item off Stack
	begin
	SP:= SP-1;
	return Stack(SP);
	end;	\Pop



	func	GetByte;	\Read next byte of image data from block
	begin
	if ByteCtr = 0 then		\get the number of bytes of image data
		ByteCtr:= ChIn(3);	\ that will follow in the current block
		\if ByteCtr gets set to 0 then there's a terminator block and
		\ no more bytes should be read from device 3
	ByteCtr:= ByteCtr-1;
	return if ByteCtr < 0 then 0 else ChIn(3);
	end;	\GetByte



	func	GetCode;	\Read next image code
	int	C, I;
	begin
	C:= CodeBuf & ReadMask;
	for I:= 0, CodeSize-1 do
		begin
		CodeBuf:= CodeBuf >> 1;
		ShiftCtr:= ShiftCtr - 1;
		if ShiftCtr <= 0 then
			begin
			CodeBuf:= CodeBuf + GetByte<<16;
			ShiftCtr:= 8;
			end;
		end;
	return C;
	end;	\GetCode



	proc	OpenGetCode;	\Initialize GetCode procedure
	begin
	ByteCtr:= 0;		\initialize for GetByte
	CodeBuf:= GetByte + GetByte<<8 + GetByte<<16;
	ShiftCtr:= 8;
	end;	\OpenGetCode



	proc	ReadColorTbl;	\Read in the color table block
	int	I;
	begin
	for I:= 0, ColorTblSize-1 do
		CLUT(I):= ChIn(3)<<16 + ChIn(3)<<8 + ChIn(3);	\RGB
	end;	\ReadColorTbl



	func	ReadWord;		\Read in a 16-bit word
	return	ChIn(3) + ChIn(3)<<8;



	proc	ReadImage;	\Read in image data and display it
	int	T;
	begin
	\Read in Image Descriptor Block:
	X1:= ReadWord;				\image position
	Y1:= ReadWord;				\ (mostly used by animations)
	Width:= ReadWord;			\5
	Height:= ReadWord;			\7

	T:= ChIn(3);				\8, packed bits
	HaveCTbl:= T & $80;
	Interlaced:= T & $40;
	if HaveCTbl then
		begin				\read local color table
		Depth:= (T&7) + 1;
		ColorTblSize:= 1 << Depth;	\number of colors in palette
		ReadColorTbl;
		end;

	\Read in image data blocks and display them
	CodeSize:= ChIn(3);			\LZW minimum code size (bits)
	ClearCode:= 1<<CodeSize;
	EOFCode:= ClearCode + 1;
	Inx0:= ClearCode + 2;			\initial table index
	Inx:= Inx0;
	BitMask:= 1<<Depth - 1;

	\The GIF spec states that the code size used to compute the above values
	\is the code size given in the file, but it's actually the code size + 1
	CodeSize:= CodeSize + 1;
	CodeSize0:= CodeSize;
	ReadMask:= 1<<CodeSize - 1;

	XC:= 0;   YC:= 0;			\start plotting at upper-left
	Pass:= 0;
	SP:= 0;					\empty stack
	OpenGetCode;

	\Decompress image data
	loop	begin
		Code:= GetCode;
		if Code = EOFCode then quit;

		\The Clear code sets everything back to its initial value then
		\ reads the next code as uncompressed data
		if Code = ClearCode then
			begin
			CodeSize:= CodeSize0;
			ReadMask:= 1<<CodeSize - 1;
			Inx:= Inx0;

			Code:= GetCode;
			OldCode:= Code;
			FinPix:= Code & BitMask;
			PlotPix(FinPix);
			end
		else	begin			\not a clear code, must be data
			NewCode:= Code;
			if Code >= Inx then
				begin		\not in table yet
				Push(FinPix);	\handle this exceptional case
				Code:= OldCode;
				end;
			while Code > BitMask do
				begin	       \follow chain of links thru table
				Push(PixTbl(Code)); \push associated output code
				Code:= StrTbl(Code);
				end;
			FinPix:= Code & BitMask;  \the last code is raw data
			PlotPix(FinPix);

			while SP > 0 do PlotPix(Pop);	\plot stacked pixels

			if Inx<0 ! Inx>=4096 then [Beep;  return];
			StrTbl(Inx):= OldCode;	 \rebuild table on the fly
			PixTbl(Inx):= FinPix;	 \(table is not stored with GIF)
			OldCode:= NewCode;

			\Point to next location in table. If the current
			\ maximum value is exceeded, increment code size unless
			\ it's already 12. If it is then do nothing. The next
			\ code decompressed better be a ClearCode
			Inx:= Inx+1;
			if Inx>ReadMask & CodeSize<12 then
				begin
				CodeSize:= CodeSize + 1;
				ReadMask:= 1<<CodeSize - 1;
				end;
			end;
		end;
	end;	\ReadImage



	proc	EatBlks;	\Skip data sub-blocks
	int	Size, I, T;
	begin
	repeat	Size:= ChIn(3);
		for I:= 0, Size-1 do T:= ChIn(3);
	until Size = 0;			\block terminator
	end;	\EatBlks



begin	\ShowGIF
\Read in Header Block and Logical Screen Descriptor:
if ChIn(3) # ^G then [Beep;   return];	\file must begin with "GIF"
if ChIn(3) # ^I then [Beep;   return];
if ChIn(3) # ^F then [Beep;   return];
for J:= 3, 5 do T:= ChIn(3);		\skip version info

Width:= ReadWord;			\get screen height and width
Height:= ReadWord;			\the image is <= to these dimensions

if not EnoughMemory then return;

Image:= Reserve(Width*IntSize);
for J:= 0, Width-1 do
	Image(J):= Reserve(Height*IntSize);

T:= ChIn(3);				\get packed bits
HaveCTbl:= T & $80;
Depth:= 1;				\default
if HaveCTbl then
	begin
	Depth:= (T&7) + 1;
	ColorTblSize:= 1 << Depth;	\number of colors in palette
	end;
T:= ChIn(3);				\background color (not used)
T:= ChIn(3);				\aspect ratio (not used)
if HaveCTbl then ReadColorTbl;		\global color table

loop	begin
	case ChIn(3) of
	  $21:	begin			\"!"
		case ChIn(3) of
		  $F9:	EatBlks;
		  $FF:	EatBlks;
		  $01:	EatBlks;	\Plain Text Ext [S ]...[S ] 00
		  $FE:	EatBlks		\Comment Ext [S ]...[S ] 00
		other	EatBlks;	\ignore any unknown $21 labels
		end;
	  $2C:	begin			\","
		ReadImage;
		ShowThumbnail;
		return;			\only show first frame of an animation
		end;
	  $3B:	quit			\";" trailer
	other	[Beep;  return];	\(probably out of sync)
	end;
end;	\ShowGIF

\###############################################################################

proc	ShowPCX;		\Display a .PCX file
\Reference: "Graphics File Formats" by Kay & Levine; and lots of hex dumps.
int	BitsPerPixel,	\number of bits per pixel per plane (1 or 8)
	Planes,		\number of planes (1..4)
	BytesPerLine;	\number of bytes per scan line per plane
int	I, J;		\indexes
int	CLUT(256);	\Color Look-Up Table (R<<16 + G<<8 + B)



proc	Show24;		\Display a 24-bit color PCX file
int	Byte,	\byte of (decompressed) image data
	Cnt,	\number of bytes to repeat (for compression)
	X, Y,	\coordinates (pixels)
	P,	\plane
	S;	\amount to shift color byte (16, 8, or 0)
int	LineBuf(WidthMax); \buffer for one scan line
begin
for Y:= 0, Height-1 do
	begin
	for X:= 0, Width-1 do		\initialize line buffer
		LineBuf(X):= 0;
	Cnt:= 0;			\initialize repeat count
	for P:= 0, 3-1 do		\for 3 planes...
		begin
		S:= (2-P)*8;		\amount to shift for R,G,B: 16, 8, 0
		for X:= 0, BytesPerLine-1 do
			begin
			if Cnt > 0 then
				Cnt:= Cnt-1
			else	begin
				Byte:= ChIn(3);
				if (Byte & $C0) = $C0 then
					begin		       \set repeat count
					Cnt:= (Byte & $3F) - 1;
					Byte:= ChIn(3);
					end;
				end;
			LineBuf(X):= LineBuf(X) ! Byte<<S;	\RGB
			end;
		end;
	for X:= 0, Width-1 do		\display the scan line
		Image(X, Y):= LineBuf(X);
	end;
end;	\Show24



proc	Show8;		\Display an 8-bit (256-color) PCX file
int	Byte,	\byte of (decompressed) image data
	Cnt,	\number of bytes to repeat
	X, Y,	\coordinates (pixels)
	I;	\index
begin
for Y:= 0, Height-1 do
    for X:= 0, BytesPerLine-1 do
	begin
	Byte:= ChIn(3);
	if (Byte & $C0) # $C0 then
		[if X < Width then Image(X, Y):= Byte]
	else	begin			\decompress multiple copies
		Cnt:= Byte & $3F;
		Byte:= ChIn(3);
		for I:= 0, Cnt-1 do
			if X+I < Width then Image(X+I, Y):= Byte;
		X:= X + Cnt - 1;
		end;
	end;
end;	\Show8



proc	ShowPlanar;	\Display a planar PCX file
int	Byte,	\byte of (decompressed) image data
	Cnt,	\number of bytes to repeat (to decompress)
	B,	\working copy of image byte
	X, Y,	\coordinates (pixels)
	XB,	\horizontal (scan line) byte
	P,	\plane
	I;	\index
int	LineBuf(WidthMax); \buffer for one scan line
begin
for Y:= 0, Height-1 do
	begin
	for X:= 0, Width-1 do LineBuf(X):= 0;	\initialize line buffer
	Cnt:= 0;

	for P:= 0, Planes-1 do
		begin
		X:= 0;
		for XB:= 0, BytesPerLine-1 do
			begin
			if Cnt > 0 then Cnt:= Cnt-1	\use current Byte
			else	begin
				Byte:= ChIn(3);
				if (Byte & $C0) = $C0 then
					begin		\decompress
					Cnt:= (Byte & $3F) - 1;
					Byte:= ChIn(3);
					end;
				end;
			B:= Byte;			\get working copy
			for I:= 0, 8-1 do		\for all of the bits...
				begin
				if B & $80 then
				    if X < WidthMax then
					LineBuf(X):= LineBuf(X) ! 1<<P;
				B:= B<<1;		\next bit
				X:= X+1;
				end;
			end;	\for XB
		end;	\for P

	for X:= 0, Width-1 do
		Image(X, Y):= CLUT(LineBuf(X)&$0F);
	end;
end;	\ShowPlanar



begin	\ShowPCX
\Read image file header information
if ChIn(3) # $0A then [Beep;   return];	\0 ignore file if invalid format
I:= ChIn(3);				\1 skip version
I:= ChIn(3);				\2 skip
BitsPerPixel:= ChIn(3);			\3
Width:= ChIn(3) + ChIn(3)<<8;		\4,5 XMin
Height:= ChIn(3) + ChIn(3)<<8;		\6,7 YMin
Width:= ChIn(3) + ChIn(3)<<8 \XMax\ - Width + 1;	\8,9
Height:= ChIn(3) + ChIn(3)<<8 \YMax\ - Height + 1;	\10,11
for I:= 12, 15 do J:= ChIn(3);		\12-15 skip
for I:= 0, 16-1 do			\16-63
	CLUT(I):= ChIn(3)<<16 + ChIn(3)<<8 + ChIn(3);
I:= ChIn(3);				\64 skip
Planes:= ChIn(3);			\65
BytesPerLine:= ChIn(3) + ChIn(3)<<8;	\66,67
for I:= 68, 127 do J:= ChIn(3);		\68-127 skip
Depth:= BitsPerPixel * Planes;

if not EnoughMemory then return;

Image:= Reserve(Width*IntSize);
for J:= 0, Width-1 do
	Image(J):= Reserve(Height*IntSize);

if BitsPerPixel = 8 then
	begin
	if Planes = 3 then Show24
	else	begin
		Show8;

		\Load Color Look Up Table with image's palette info
		if ChIn(3) # $0C then	\check for if valid palette info
			[Beep;   return];\don't change colors if invalid
		for I:= 0, 256-1 do
			CLUT(I):= ChIn(3)<<16 + ChIn(3)<<8 + ChIn(3);	\RGB

		\Convert indexed colors to 24-bit RGB colors
		for J:= 0, Height-1 do
		    for I:= 0, Width-1 do
			Image(I,J):= CLUT(Image(I,J));
		end;
	end
else	ShowPlanar;
ShowThumbnail;
end;	\ShowPCX

\###############################################################################

proc	ShowLBM;		\Display the opened .LBM (or .BBM or .IFF) file
\ Reference: "Amiga ROM Kernel Reference Manual", IFF, p I-28. "Supercharged
\ Bitmapped Graphics" and "Graphics Workshop" by Rimmer don't handle widths with
\ odd numbers of pixels correctly. Lots of hex dumps were used. Deluxe Paint II
\ is the standard. VPic 6.1e.6 also has problems similar to GWS 7.0d.
\ILBM (InterLeaved BitMap) is the most common file format. It uses separate
\ planes for each color bit. PBM (Deluxe Paint's Proprietary BitMap) uses a
\ byte (like BMP and PCX) to represent one of 256 colors in a palette defined
\ by CMAP. PBM doesn't seem to be used for 24-bit color.

def	FORM= $464F524D, ILBM= $494C424D, PBM = $50424D20,	\chunk headers
	BMHD= $424D4844, CMAP= $434D4150, BODY= $424F4459;

int	HaveBMHD,	\flag: bit map header chunk has been read in
	HavePBM,	\flag: image data are in PBM format, not ILBM
	Masking,	\masking type (0=none, 1=extra bit plane, 2=transparent)
	Compressed,	\compression algorithm (0= none, 1=ByteRun1)
	Transparent,	\color designated as transparent (black) when masking=2
	DW,		\double word (32-bit value)
	J;		\index
int	CLUT(256);	\Color Look-Up Table (R<<16 + G<<8 + B)



func	ReadWord;	\Return word from disk file (big endian style)
return ChIn(3)<<8 + ChIn(3);



func	ReadDWord;	\Return double word from disk file (big endian)
return ReadWord <<16 + ReadWord;

\-------------------------------------------------------------------------------

proc	EatChunk;	\Read in and dispose of a chunk
int	CSize, N, T;
begin
CSize:= ReadDWord;
for N:= 0, CSize-1 do
	T:= ChIn(3);
if CSize & 1 then		\make sure it's ending on an even-byte boundary
	T:= ChIn(3);
end;	\EatChunk

\-------------------------------------------------------------------------------

proc	DoPBM;		\Load and display image data in PBM format
int	CSize,	\chunk size (bytes) (ignored)
	X, Y,	\coordinates (pixels)
	P;	\padding byte (discarded)



proc	DoLine;		\Read in a compressed scan line in PBM format
int	N, B,	\bytes from input file
	J;	\index
begin
X:= 0;					\start at left side
repeat	begin
	N:= ChIn(3);			\get number N
	if N <= $7F then
		begin			\copy the next N+1 bytes literally
		for J:= 0, N do		\ (i.e. no compression)
			begin
			B:= ChIn(3);
			if X < Width then Image(X, Y):= CLUT(B);
			X:= X+1;
			end;
		end

	\else if N = $80 then []	\\incorrectly used by Adobe Photoshop

	else	begin			\N is in the range: $81..$FF
					\replicate next byte -extend(N)+1 times
		B:= ChIn(3);		\read next byte
		for J:= 0, -ext(N) do	\(byte run compression)
			begin
			if X < Width then Image(X, Y):= CLUT(B);
			X:= X+1;
			end;
		end;
	end;
until X >= ((Width+1) & $FFFE);		\an even number of PIXELS must be read
end;	\DoLine				 "transparent" pixels pad the width



begin	\DoPBM
Depth:= 8;				\always 8 bits per color (VPic bug)
CSize:= ReadDWord;			\get chunk size and ignore it
for Y:= 0, Height-1 do			\for all of the scan lines...
	begin
	if Compressed then DoLine
	else	begin
		for X:= 0, Width-1 do	\PBM (chunky) mode is straightforward
			Image(X, Y):= CLUT(ChIn(3));
		if Width & 1 then P:= ChIn(3);		\eat odd padding byte
		end;
	end;
end;	\DoPBM

\-------------------------------------------------------------------------------

proc	DoILBM;		\Load and display image data in ILBM format
int	CSize,	\chunk size (bytes) (ignored)
	WWidth,	\image width in words (rounded up)
	X, Y,	\coordinates (pixels)
	D,	\depth of bit plane (0..Depth-1)
	W,	\word
	XW,	\horizontal word
	I;	\index
int	LineBuf(WidthMax); \buffer for one scan line



func	BGR2RGB(BGR);	\Swap the red and blue bytes
int	BGR;
int	R, G, B;
begin
R:= BGR & $0000FF;
G:= BGR & $00FF00;
B:= BGR & $FF0000;
return R<<16 + G + B>>16;
end;	\BGR2RGB



proc	DoLine;		\Read in a compressed scan line for a single bit plane
int	N, B, B0,	\bytes from input file
	J;		\index
begin
X:= 0;					\start at left side
repeat	begin
	N:= ChIn(3);			\get number N
	if N <= $7F then
		begin			\copy the next N+1 bytes literally
		for J:= 0, N do		\ (i.e. no compression)
			begin
			B:= ChIn(3);
			for I:= 0, 8-1 do
				begin
				if B & $80 then
				    if X < WidthMax then
					LineBuf(X):= LineBuf(X) ! 1<<D;
				B:= B<<1;
				X:= X+1;
				end;
			end;
		end

	\else if N = $80 then []	\\incorrectly used by Adobe Photoshop

	else	begin			\N is in the range: $81..$FF
					\replicate next byte -extend(N)+1 times
		B0:= ChIn(3);		\read next byte
		for J:= 0, -ext(N) do	\(byte-run compression)
			begin
			B:= B0;		\working copy
			for I:= 0, 8-1 do
				begin	\distribute bits in B into 8 bytes
				if B & $80 then
				    if X < WidthMax then
					LineBuf(X):= LineBuf(X) ! 1<<D;
				B:= B<<1;
				X:= X+1;
				end;
			end;
		end;
	end;
until X >= Width;
end;	\DoLine



begin	\DoILBM
WWidth:= (Width+15)/16;			\width of scan line in words rounded up
CSize:= ReadDWord;			\get chunk size and ignore it
for Y:= 0, Height-1 do			\for all of the scan lines...
	begin
	for X:= 0, WidthMax-1 do LineBuf(X):= 0;	\initialize line buffer

	for D:= 0, Depth-1 do		\for all of the bit planes...
		begin
		if Compressed then DoLine
		else	begin
			X:= 0;				\start at left side
			for XW:= 0, WWidth-1 do		\for all of the words...
				begin
				W:= ReadWord;
				for I:= 0, 16-1 do	\for all of the bits...
					begin
					if W & $8000 then	\set bit
					    if X < WidthMax then
						LineBuf(X):= LineBuf(X) ! 1<<D;
					W:= W<<1;
					X:= X+1;
					end;
				end;
			end;
		end;	\for loop

	if Depth = 24 then			\display the scan line
		for X:= 0, Width-1 do
			Image(X, Y):= BGR2RGB(LineBuf(X))
	else	for X:= 0, Width-1 do
			Image(X, Y):= CLUT(LineBuf(X)&$FF);

	if Masking = 1 then			\read mask in and discard it
		if Compressed then DoLine
		else	for XW:= 0, WWidth-1 do	\for all of the words...
				W:= ReadWord;
	end;
end;	\DoILBM

\-------------------------------------------------------------------------------

proc	DoCMAP;		\Read in color map chunk and set up color lookup table
int	CSize,	\number of bytes in CMAP field
	ColRegs,\number of color registers to use
	N, T;
begin
CSize:= ReadDWord;
ColRegs:= CSize/3;
if ColRegs > 256 then ColRegs:= 256;	\limit maximum number of color registers
for N:= 0, ColRegs-1 do
	CLUT(N):= ChIn(3)<<16 ! ChIn(3)<<8 ! ChIn(3);	\R, G, B order

for N:= N*3, CSize-1 do		\eat the rest of the chunk, if any
	T:= ChIn(3);
if N & 1 then T:= ChIn(3);	\must end on an even-byte boundary
end;	\DoCMAP

\-------------------------------------------------------------------------------

proc	DoBMHD;		\Read in BitMap Header chunk
int	T;
begin
T:= ReadDWord;		\chunk size in bytes (not used, it's always 20)
Width:= ReadWord;	\raster width and height in pixels
Height:= ReadWord;
T:= ReadWord;		\offset (in pixels) for this image
T:= ReadWord;		\ (not used)
Depth:= ChIn(3);	\number of bit planes (unless PBM, which is always 8)
Masking:= ChIn(3);	\masking technique (1=mask stored in separate bit plane)
Compressed:= ChIn(3);	\compression algorithm (0=none, 1=byte run)
T:= ChIn(3);		\unused by IFF standard
Transparent:= ReadWord;	\transparent color (only used with masking = 2)
T:= ChIn(3);		\aspect ratio (1:1 is assumed here)
T:= ChIn(3);
T:= ReadWord;		\source page size (not used here)
T:= ReadWord;

HaveBMHD:= true;	\indicate that BMHD chunk has been read in
end;	\DoBMHD

\-------------------------------------------------------------------------------

begin	\ShowLBM
HaveBMHD:= false;
if ReadDWord # FORM then [Beep;   return];	\file did not start with "FORM"

DW:= ReadDWord;		\read in chunk size and ignore it
DW:= ReadDWord;		\read in 4-character chunk ID
if DW#ILBM & DW#PBM then [Beep;   return];	\"ILBM" or "PBM " missing
HavePBM:= DW = PBM;

for DW:= 0, 256-1 do	\set up default gray-scale palette, in case of no CMAP
	CLUT(DW):= DW<<16 ! DW<<8 ! DW;

loop	begin
	case ReadDWord of		\handle chunk
	  BMHD:	DoBMHD;
	  CMAP:	DoCMAP;
	  BODY:	begin
		if ~HaveBMHD then [Beep;   return];	\"BMHD" missing
		if ~EnoughMemory then return;

		Image:= Reserve(Width*IntSize);
		for J:= 0, Width-1 do
			Image(J):= Reserve(Height*IntSize);

		if Masking = 2 then CLUT(Transparent&$FF):= $FFFFFF; \br. white
		if HavePBM then DoPBM else DoILBM;
		quit;
		end
	other	EatChunk;		\ignore any unrecognized chunks
	if GetErr \#0\ then [Beep;   return];		\BODY missing
	end;

ShowThumbnail;
end;	\ShowLBM

\###############################################################################

proc	ShowBMP;		\Display the opened .BMP file as a thumbnail
\Reference: Microsoft MSDN Library; and "Graphics File Formats" by Kay & Levine
int	Offset,		\offset from start of file to image data (bytes)
	HeadSize,	\header size: 40=Windows format; 12=OS/2 (old) format
	R, G, B,	\red, green, blue (0..255)
	T,		\temporary scratch
	CTblSize,	\number of bytes in BMP's color table
	X, Y,		\pixel coordinates for thumbnail image
	I, J;		\pixel coordinates in image file
int	CLUT(256);	\Color Look-Up Table (R<<16 + G<<8 + B)

begin
if ChIn(3) # ^B then [Beep;   return];	   \file must begin with "BM" (Bit Map)
if ChIn(3) # ^M then [Beep;   return];
for I:= 2, 9 do X:= ChIn(3);	\2, skip unused header info
Offset:= ChIn(3) + ChIn(3)<<8;	\10, offset from start of file to image data
T:= ChIn(3);   T:= ChIn(3);	\12, skip unused header info (32767 max. is ok)

HeadSize:= ChIn(3) + ChIn(3)<<8;	   \14, size of header (40 or 12)
T:= ChIn(3);   T:= ChIn(3);		   \16, skip high bytes of size
Width:= ChIn(3) + ChIn(3)<<8;	   	   \18
if HeadSize = 12 then
	begin				   \OS/2 (old) format:
	Height:= ChIn(3) + ChIn(3)<<8;	   \20
	T:= ChIn(3);   T:= ChIn(3);	   \22, skip "number of image planes"
	Depth:= ChIn(3) + ChIn(3)<<8;	   \24, bits per pixel
	end
else	begin				   \Windows (normal) format:
	T:= ChIn(3);   T:= ChIn(3);	   \20, skip unused high bytes of Width
	Height:= ChIn(3) + ChIn(3)<<8;	   \22
	T:= ChIn(3);   T:= ChIn(3);	   \24, skip unused high bytes of Height
	T:= ChIn(3);   T:= ChIn(3);	   \26, skip "number of image planes"
	Depth:= ChIn(3) + ChIn(3)<<8;	   \28, bits per pixel
	for I:= 30, 53 do T:= ChIn(3);	   \30, skip rest of header
	end;

if not EnoughMemory then return;

Image:= Reserve(Width*IntSize);
for J:= 0, Width-1 do
	Image(J):= Reserve(Height*IntSize);

\Read in the color table up to a maximum of 256 (4-byte) entries
CTblSize:= Offset - 14 - HeadSize;	\number of bytes of color-table data
if CTblSize > 256*4 then CTblSize:= 256*4;
I:= 0;   J:= 0;
while J < CTblSize do
	begin
	B:= ChIn(3);			\(things tend to be backwards in BMPs)
	G:= ChIn(3);
	R:= ChIn(3);
	J:= J + 3;
	if HeadSize # 12 then		\if not OS/2 (old) format
		[T:= ChIn(3);   J:= J+1];
	if I < 256 then			\(for safety)
		[CLUT(I):= R<<16 + G<<8 + B;   I:= I+1];
	end;

for Y:= -(Height-1), 0 do		\they're upside down (!)
	begin		\note that lines are padded to end on a 4-byte boundary
	case Depth of
	  32:	begin
		for X:= 0, Width-1 do
			begin
			Image(X, -Y):= ChIn(3) + ChIn(3)<<8 + ChIn(3)<<16;
			T:= ChIn(3);	\discard unused high byte
			end;
		end;
	  24:	begin
		for X:= 0, Width-1 do
			Image(X, -Y):= ChIn(3) + ChIn(3)<<8 + ChIn(3)<<16;
		for I:= Width*3, ((Width*3+3)&$FFFC)-1 do
			T:= ChIn(3);			\4-byte boundary
		end;
	  16:	begin
		for X:= 0, Width-1 do
			begin
			T:= ChIn(3) + ChIn(3)<<8;	\5 bits of R, G, and B
			R:= (T&$7C00)<<9;		\repackage in 24-bit
			G:= (T&$03E0)<<6;		\ format; (5:6:5 format
			B:= (T&$001F)<<3;		\ is not an issue here:
			Image(X, -Y):= R+G+B;		\ hardware-independent)
			end;
		for I:= Width*2, ((Width*2+3)&$FFFC)-1 do
			T:= ChIn(3);			\4-byte boundary
		end;
	  8:	begin
		for X:= 0, Width-1 do
			Image(X, -Y):= CLUT(ChIn(3));
		for I:= Width, ((Width+3)&$FFFC)-1 do
			T:= ChIn(3);			\4-byte boundary
		end;
	  4:	for X:= 0, ((Width+7)&$FFF8)-1 do	\4-byte boundary =
			begin				\ 8-nibble boundary
			T:= ChIn(3);
			if X < Width then Image(X, -Y):= CLUT(T>>4);
			X:= X+1;
			if X < Width then Image(X, -Y):= CLUT(T&$0F);
			end;
	  1:	for X:= 0, ((Width+31)&$FFE0)-1 do	\4-byte boundary =
			begin				\ 32-bit boundary
			T:= ChIn(3);
			for I:= 0, 7 do
				begin
				if X < Width then
					Image(X, -Y):= CLUT(T>>7&1);
				X:= X+1;
				T:= T<<1;
				end;
			X:= X-1;
			end
	other	[Beep;   return];
	end;

ShowThumbnail;
end;	\ShowBMP

\###############################################################################

func	OpenImageFile(FN);	\Open (initialize) an image file for viewing
char	FN;			\string containing name of file to open (ASCIIZ)
char	FN2($100);
begin
StrCat(PathName, FN, FN2);	\prepend any path name onto the file name
FileHandle:= FOpen(FN2, 0);	\get input handle
if GetErr \#0\ then
	[Beep;   return false]	\just beep if file was not found, somehow
else	[FSet(FileHandle, ^I);   OpenI(3);   return true];	\else open file
end;	\OpenImageFile		 leave error trapping turned off



proc	CloseImageFile;		\Finish with an image file
begin
FClose(FileHandle);		\close out file handle so it can be reused
if GetErr \#0\ then Beep;	\notify if there was a read beyond EOF, etc.
end;	\CloseImageFile

\-------------------------------------------------------------------------------

proc	SortFiles;	\Sort file names according to command-line switches
int	I, Key(FilesMax,3);	\3*IntSize = 12 = 8+1+3 bytes



proc	RotFN(S);	\Rotate the file name in string until extension is first
char	S;
int	L, T, I;
\	012345678
\	ABCD.EXT0	length = 8
begin
\If string S is 12 characters long it will not be terminated with a zero
L:= StrNLen(S, 12);
repeat	T:= S(0);
	for I:= 0, L-2 do S(I):= S(I+1);
	S(L-1):= T;
until	S(0) = ^.;
\	.EXTABCD0
end;	\RotFN



proc	Sort(S);	\Sort Key and its associated arrays using Shell's method
int	S;		\size of strings in bytes
int	FN, J, JG, Gap, I, T;
begin
FN:= FileName;		\to access address pointers rather than bytes
Gap:= Files/2;
while Gap > 0 do
	begin
	for I:= Gap, Files-1 do
	    begin
	    J:= I - Gap;
	    loop begin
		if J < 0 then quit;
		JG:= J + Gap;
		if StrNCmp(Key(J), Key(JG), S) <= 0 then quit;
		\swap all entries at J and JG
		T:= Key(J);       Key(J):= Key(JG);            Key(JG):= T;
		T:= FN(J);        FN(J):= FN(JG);              FN(JG):= T;
		T:= FileDate(J);  FileDate(J):= FileDate(JG);  FileDate(JG):= T;
		T:= FileTime(J);  FileTime(J):= FileTime(JG);  FileTime(JG):= T;
		T:= FileSize(J);  FileSize(J):= FileSize(JG);  FileSize(JG):= T;
		J:= J - Gap;
		end;
	    end;
	    Gap:= Gap/2;
	end;
end;	\Sort



begin	\SortFiles
case of
SwExt:	begin				\sort by extension, then name
	for I:= 0, Files-1 do
		begin
		StrNCopy(addr FileName(I,0), Key(I), 12);
		RotFN(Key(I));
		end;
	Sort(1+3+8);			\dot + extension + name
	end;
SwDate:	begin				\sort by date, then time
	for I:= 0, Files-1 do
		begin			\put MSB in first byte of Key, etc.
		Key(I,0):= RevBytes(FileDate(I));
		Key(I,1):= RevBytes(FileTime(I));
		end;
	Sort(8);
	end;
SwSize:	begin				\sort by file size
	for I:= 0, Files-1 do
		Key(I,0):= RevBytes(FileSize(I));
	Sort(4);
	end
other	begin				\sort by file name
	for I:= 0, Files-1 do
		StrNCopy(addr FileName(I,0), Key(I), 12);
	Sort(8+1+3);			\name + dot + extension
	end;
end;	\SortFiles

\-------------------------------------------------------------------------------

proc	MakeFileList;
\Create a list of files from the current directory, store them into FileName,
\ FileDate (etc.) and return Files, a count of the number of items in the list.
int	ExtList,	\list of file extensions
	ExtID;		\extension ID (= index into ExtList)



proc	ReadFileNames(Ext);
\Read directory at PathName, and store the file names that match Ext into
\ the FileName (etc.) arrays. Returns the index of the last name stored (+1).
char	Ext;		\string containing extension, e.g: "*.BMP"
int	FNDOS,		\segment address of path and file name in low memory
	DTA		\seg addr of Disk Transfer Area, in low memory
	I;
char	FN($100);	\complete file name including path name and extension
begin	
StrCat(PathName, Ext, FN);	\prepend any path name onto the extension
FNDOS:= MAlloc($100/16);	\copy resultant file name where DOS can see it
Blit(DataSeg, FN, FNDOS, 0, $100);

DTA:= MAlloc(43/16+1);		\DOS function $4E needs a "Disk Transfer Area"
CallInt($21, $1A00, 0, 0, 0, 0, DTA);

if CallInt($21, $4E00, 0, $37, 0, 0, FNDOS) = 0 then	\look up 1st file name
   loop	begin			\copy file names into the FileName (etc.) arrays
	if Files >= FilesMax then quit;			\protect from overflows
	for I:= 0, 12 do FileName(Files,I):= Peek(DTA,30+I);\get filename.ext,
	FileName(Files,ID):= ExtID;			    \ file type,
	FileDate(Files):= Peek(DTA,24) + Peek(DTA,25)<<8;   \ date,
	FileTime(Files):= Peek(DTA,22) + Peek(DTA,23)<<8;   \ time,
	FileSize(Files):= Peek(DTA,26) + Peek(DTA,27)<<8 +  \ and size
		Peek(DTA,28)<<16 + Peek(DTA,29)<<24;
	Files:= Files+1;
	if CallInt($21, $4F00) \#0\ then quit;	\look up next file name, if one
	end;
if CpuReg(0) = 3 then Fatal("Path not found");

Release(DTA);
Release(FNDOS);
end;	\ReadFileNames



begin	\MakeFileList
\List of file extensions to read in:
\ Items in this array must correspond with ID numbers used in ShowAllThumbnails
ExtList:= ["*.BMP", "*.LBM", "*.BBM", "*.IFF", "*.PCX", "*.GIF"];
Files:= 0;
for ExtID:= 0, 6-1 do			\for each item in ExtList...
	ReadFileNames(ExtList(ExtID));	\read list of file names
SortFiles;
end;	\MakeFileList

\-------------------------------------------------------------------------------

proc	ShowImageInfo(N);	\Show image information
int	N;
int	X, Y;
begin
X:= ThumbX;
Y:= ThumbY + 96+1;
Attrib($00FFFFFF);			\bright white characters

Move(X, Y);
Text($106, addr FileName(N,0));		\show image file name using 8x8 font

Move(X, Y+8);				\image dimensions
IntOut($106, Width); ChOut($106, ^x);
IntOut($106, Height); ChOut($106, ^x);
IntOut($106, Depth);

Move(X, Y+16);				\date of creation or last modification
DateOut($106, FileDate(N));

Move(X, Y+24);				\time of creation or last modification
TimeOut($106, FileTime(N));
end;	\ShowImageInfo



proc	ShowAllThumbnails;	\Show a screenful of thumbnail images
\Inputs: FileName, Files, Width, Height, Depth
\Outputs: ThumbX, ThumbY
int X, Y, I;
begin
SetVid($118);				\set 1024x768x24 graphics
I:= 0;
loop	begin
	for Y:= 0, 6-1 do
	    for X:= 0, 10-1 do
		begin
		ThumbX:= X*(96+6) + 5;	\thumbnail's screen coordinates
		ThumbY:= Y*(96+4*8);
		Width:= 0;  Height:= 0;  Depth:= 0; \in case ShowXXX aborts

		case FileName(I,ID) of	\file type
		  0:	if OpenImageFile(addr FileName(I,0)) then
				[ShowBMP;   CloseImageFile];
		  1, 2, 3:	\LBM, BBM, IFF
			if OpenImageFile(addr FileName(I,0)) then
				[ShowLBM;   CloseImageFile];
		  4:	if OpenImageFile(addr FileName(I,0)) then
				[ShowPCX;   CloseImageFile];
		  5:	if OpenImageFile(addr FileName(I,0)) then
				[ShowGIF;   CloseImageFile]
		other	Beep;	\should be impossible, but don't abort

		ShowImageInfo(I);

		I:= I+1;
		if I >= Files then quit;
		if KeyHit then
			if LookKey = Esc then quit;
		end;
	if ChIn(1) = Esc then		\wait for keystroke to do next page, but
		[SetVid($03);  return];	\ handle Esc too
	Clear;
	end;	\loop
if ChIn(1) then [];			\wait for keystroke
SetVid($03);				\restore normal text mode
end;	\ShowAllThumbnails



func	GetPathName;		\Get any drive or path typed on the command line
\Also set up any switches. Return 'false' if error.
\Outputs: PathName, switches
int	I, J, Ch;
char	CmdTail($80);		\command tail from Program Segment Prefix (PSP)
begin
Blit(PspSeg, $80, DataSeg, CmdTail, $80); \get rest of command line

\Copy CmdTail into PathName. Remove any spaces or switches. Set up switches.
SwExt:= false;  SwDate:= false;  SwSize:= false;
J:= 0;
for I:= 1, CmdTail(0) do		\for rest of characters on command line
	begin
	Ch:= CmdTail(I);
	if Ch = ^/ then			\remove switch
		begin
		I:= I+1;		\skip the "/"
		if I > CmdTail(0) then return false;
		Ch:= CmdTail(I);
		case Ch of
		 ^E,^e:	SwExt:= true;
		 ^D,^d:	SwDate:= true;
		 ^S,^s:	SwSize:= true;
		 ^N,^n:	[]		\ignore: sort by name is the default
		other return false;	\illegal switch character
		end
	else if Ch # $20 \space\ then
		begin
		PathName(J):= Ch;	\output character to PathName
		J:= J+1;
		case Ch of
		  ^*,^?:return false;	\no wild cards allowed
		  ^::	if J # 2 then return false	\drive
		other	[];
		end;
	end;

\If PathName doesn't end with a slash, but a name was entered, then add a slash.
\	(nothing)
\	\
\	a:
\	a:\
\	path		add \
\	path\

if J > 0 then				\command line has a drive and/or path
	begin
	Ch:= PathName(J-1);
	if Ch#^: & Ch#^\ then
		[PathName(J):= ^\;  J:= J+1];
	end;
PathName(J):= 0;			\terminate string
return true;				\indicate success
end;	\GetPathName



proc	ShowHelp;
begin
Text(0, "
Displays THumbnails of image files in a directory.

Usage: TH [drive:][path] [/E] [/D] [/S] [/N]

  /E	Sort by Extension.
  /D	Sort by Date and time.
  /S	Sort by file Size.
  /N	Sort by file Name (default).


Thumbnail Viewer  Version 1.0  Copyright (C) 2011 Loren Blaney

This comes with ABSOLUTELY NO WARRANTY. It is free software.
You are welcome and encouraged to redistribute it under
certain conditions. For details see LICENSE.TXT.
");
end;	\ShowHelp

\-------------------------------------------------------------------------------

begin	\Main
CpuReg:= GetReg;
PspSeg:= CpuReg(11);	\get these to read in any command-line
DataSeg:= CpuReg(12);	\ parameters, such as the path name
TrapC(true);		\disable Ctrl+C, so normal text display must be restored
Trap(false);		\turn off all error trapping; if error, don't abort

if GetPathName then
	begin
	MakeFileList;
	if Files > 0 then ShowAllThumbnails
	else	Text(0, "No image files found");
	end
else	ShowHelp;
CrLf(0);
end;	\Main
