
/************************************************************************/
/*                                                                      */
/* FILE: env.c                                                          */
/* command.com Environment support                                      */
/*                                                                      */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* This file is part of G-COM.                                          */
/* G-COM is derived from DOS-C source (GPL).                            */
/*                                                                      */
/* (C) Copyright 1999-2000  Roberto Gordo Saez   (GCOM)                 */
/* (C) Copyright 1995-1998  Pasquale J. Villani  (DOSC)                 */
/*                                                                      */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2, or (at your option)  */
/* any later version.                                                   */
/*                                                                      */
/* This program is distributed in the hope that it will be useful, but  */
/* WITHOUT ANY WARRANTY; without even the implied warranty of           */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU     */
/* General Public License for more details.                             */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to the Free Software          */
/* Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.                  */
/*                                                                      */
/************************************************************************/


#include "globals.h"


/* -- Function prototypes --------------------------------------------- */

unsigned int EnvGetSize (void);
void EnvAlloc (unsigned int);
void EnvCopy (char FAR*, char FAR*);
char FAR *EnvLookup (char*);
BOOL EnvClearVar (char*);
void EnvSetVar (char*, char*);
void EnvDump (void);


/* -- Global variables ------------------------------------------------ */

static unsigned int EnvSize = 0; /* Startup environment size */


/* -------------------------------------------------------------------- */
/* EnvGetSize                                                           */
/* -------------------------------------------------------------------- */

unsigned int EnvGetSize(void)
{
    char FAR *lpszSrc = lpEnviron;
    unsigned int Size = 0;

    while(*lpszSrc != '\0')
    {
        while(*lpszSrc)
            lpszSrc++, ++Size;
        ++lpszSrc, ++Size;
    }
    /* Count the terminating entry */
    lpszSrc++, ++Size;

    /* Count the invocation part */
    ((UWORD FAR *)lpszSrc)++;
    Size += 2;
    while(*lpszSrc)
        lpszSrc++, ++Size;

    /* Count the terminating null ... */
    Size++;

    /* ... and return the count. */
    return Size;
}


/* -------------------------------------------------------------------- */
/* EnvAlloc                                                             */
/* -------------------------------------------------------------------- */

void EnvAlloc(unsigned int Size)
{
    int Klicks;
    DWORD Error;
    SEGMENT Seg;

    if(Size < EnvGetSize())
    {
        EnvSize = EnvGetSize();
        error_message(OUT_ENV_SPACE);
        return;
    }

    /* Do the allocation, then copy the environment */
    Klicks = (Size + PARASIZE - 1)/PARASIZE;
    Seg = DosAllocMem(Klicks, (DWORD FAR *)&Error);
    if(Error == DE_SUCCESS)
    {
        EnvCopy(MK_FP(Seg, 0), lpEnviron);
        if(DosFreeMem(FP_SEG(lpEnviron)) != DE_SUCCESS)
        {
            DosFreeMem(Seg);
            EnvSize = 0;
            return;
        }
        lpEnviron = MK_FP(Seg, 0);
        EnvSize = Size;
    }
    else
    {
        EnvSize = 0;
        error_message(INSUFF_MEM);
    }
}


/* -------------------------------------------------------------------- */
/* EnvCopy                                                              */
/* -------------------------------------------------------------------- */

void EnvCopy(char FAR *lpszDest, char FAR *lpszSrc)
{
    while(*lpszSrc != '\0')
    {
        while(*lpszSrc)
            *lpszDest++ = *lpszSrc++;
        ++lpszSrc;
        *lpszDest++ = '\0';
    }
    /* Copy the terminating entry */
    *lpszDest++ = *lpszSrc++;

    /* Copy the invocation part */
    *((UWORD FAR *)lpszDest)++ = *((UWORD FAR *)lpszSrc)++;
    while(*lpszSrc)
        *lpszDest++ = *lpszSrc++;

    /* and finally null terminate. */
    *lpszDest++ = '\0';
}


/* -------------------------------------------------------------------- */
/* EnvLookup                                                            */
/* -------------------------------------------------------------------- */

char FAR *EnvLookup(char *pszName)
{
    char FAR *lpszString;
    unsigned int nNameLen;

    nNameLen = strlen(pszName);

    lpszString = lpEnviron;
    while(*lpszString != '\0')
    {
        if(fstrnicmp(lpszString, (char FAR *)pszName, nNameLen))
        {
            lpszString += nNameLen;
            if(*lpszString == '=')
                return ++lpszString;
        }
        while(*lpszString)
            ++lpszString;
        ++lpszString;
    }
    return (char FAR *)"";
}


/* -------------------------------------------------------------------- */
/* EnvClearVar                                                          */
/* -------------------------------------------------------------------- */

BOOL EnvClearVar(char *pszName)
{
    char FAR *lpszSrc, FAR *lpszDest;

    lpszDest = EnvLookup(pszName);

    if(*lpszDest != '\0')
    {
        lpszSrc = lpszDest;
        while(*lpszDest != '\0')
            --lpszDest;
        ++lpszDest;
        while(*lpszSrc != '\0')
            lpszSrc++;
        lpszSrc++;
        EnvCopy(lpszDest, lpszSrc);
    }
    else
        return FALSE;

    return TRUE;
}


/* -------------------------------------------------------------------- */
/* EnvSetVar                                                            */
/* -------------------------------------------------------------------- */

void EnvSetVar(char *pszName, char *pszValue)
{
    unsigned int size;
    char FAR *lpszSrc, FAR *lpszDest;
    char FAR *pointer;

    pointer = EnvLookup(pszName);
    size = strlen(pszName)+strlen(pszValue)+2;
    if(*pointer != '\0')
        size = strlen(pszValue)-fstrlen(pointer);
    if(size >= EnvSize - EnvGetSize())
    {
        error_message(OUT_ENV_SPACE);
        return;
    }
    size = strlen(pszName)+strlen(pszValue)+2;

    /* Clear the variable if it's already defined */
    EnvClearVar(pszName);

    lpszSrc = lpEnviron;
    while(*lpszSrc != '\0')
    {
        while(*lpszSrc)
            lpszSrc++;
        ++lpszSrc;
    }
    pointer = ++lpszSrc;
    pointer += sizeof(WORD);

    while(*pointer != '\0')
        pointer++;

    /* Move the invocation part */
    lpszDest = pointer+size;
    while(pointer >= lpszSrc)
        *lpszDest-- = *pointer--;

    /* Insert the new variable */
    lpszDest = --lpszSrc;
    lpszSrc = pszName;
    while(*lpszSrc != '\0')
        *lpszDest++ = *lpszSrc++;
    *lpszDest++ = '=';
    lpszSrc = pszValue;
    while(*lpszSrc != '\0')
        *lpszDest++ = *lpszSrc++;

    *lpszDest++ = '\0';
    *lpszDest = '\0';
}


/* -------------------------------------------------------------------- */
/* EnvDump                                                              */
/* -------------------------------------------------------------------- */

void EnvDump(void)
{
    char FAR *lpszEnv = lpEnviron;
    unsigned int nCount;

    while(*lpszEnv != '\0')
    {
        nCount = fstrlen(lpszEnv);
        DosWrite(STDOUT, lpszEnv, nCount);
        if(break_flag) return;
        DosWrite(STDOUT, (char FAR *)newline, 2);
        if(break_flag) return;
        lpszEnv += nCount+1;
    }
}
