/*
// Program:  Format
// Version:  0.91d
// (0.90b/c - avoid saving info when inappropriate - Eric Auer 2003)
// (0.91d - handle non-standard FAT12/FAT16 formats, debugging: Eric Auer 2003)
// Written By:  Brian E. Reifsnyder
// Copyright:  2002 under the terms of the GNU GPL, Version 2
// Module Name:  savefs.c
// Module Description:  Save File System Function
*/


#include "format.h"
#include "floppy.h"
#include "driveio.h"
#include "btstrct.h"


/* Convert Huge number into 4 LMB integer values */
void Convert_Huge_To_Integers(unsigned long number)
{
  integer1=(unsigned)((number >> 0 ) & 0xff);
  integer2=(unsigned)((number >> 8 ) & 0xff);
  integer3=(unsigned)((number >> 16) & 0xff);
  integer4=(unsigned)((number >> 24) & 0xff);
}

/* Save the old file system for possible recovery with unformat */
/* */
/* *** WARNING: This only works if (...) the root directory follows    *** */
/* *** right after the 2nd FAT. This is NOT always true for FAT32!     *** */
/* Changed in 0.91d to handle != 2 FATs and != 1 reserved/boot sector...   */
/* Depends on the unformat data format whether this can be more flexible!  */
/* */
void Save_File_System(void)
{
  int bad_boot_sector = TRUE;
  int end_of_root_directory_flag;

  unsigned  loop = 512;

  unsigned char mirror_map[5120];

  unsigned pointer = 0;

  unsigned long offset_from_end;

  unsigned long destination_sector;
  unsigned long source_sector;

  unsigned long mirror_beginning;
  unsigned long mirror_map_beginning;
  unsigned long mirror_map_size;
  unsigned long mirror_size;
  unsigned long number_of_bytes_in_mirror_map;
  unsigned long number_of_logical_sectors_on_drive;
  unsigned long number_of_root_directory_entries;

  unsigned long beginning_of_fat;
  unsigned long beginning_of_root_directory;

  unsigned long sectors_per_fat;
  unsigned long reserved_sectors; /* usually 1 - boot sector [0e]w */
  unsigned long number_of_fats;   /* usually 2, but can be 1 [10]b */

  unsigned long end_of_root_directory;

  unsigned long temp_buffer_1 = 0;
  unsigned long temp_buffer_2 = 0;
  unsigned long temp_buffer_3 = 0;
  unsigned long temp_buffer_4 = 0;

  unsigned int sector_flag[] = {
  'A','M','S','E','S','L','I','F','V','A','S',
  'R','O','R','I','M','E','S','A','E','P'};

  unsigned int mirror_map_header[] = {
  ':','\\','M','I','R','R','O','R','.','F','I','L'};

  if (param.drive_type==HARD)
    {
    offset_from_end = 20;
    }
  else
    {
    offset_from_end = 5;
    }

  /* Get the boot sector, compute the FAT size, compute the root dir size,*/
  /* and get the end of the logical drive. */
  Drive_IO(READ,0,1);

  if ( (sector_buffer[510]==0x55) && (sector_buffer[511]==0xaa) )
    {
    bad_boot_sector = FALSE;

    if ( (sector_buffer[0x0b]!=0) || (sector_buffer[0x0c]!=2) )
      {
      printf("Sector size is not 512 bytes. Cannot save UNFORMAT data.\n");
      bad_boot_sector = TRUE;
      }

    number_of_fats = sector_buffer[0x10];
    if ( (number_of_fats < 1) || (number_of_fats > 2) )
      {
      printf("There <1 or >2 FAT copies. Cannot save UNFORMAT data.\n");
      bad_boot_sector = TRUE;
      }

    temp_buffer_1 = sector_buffer[0x0e];
    temp_buffer_2 = sector_buffer[0x0f];
    reserved_sectors = temp_buffer_1 | (temp_buffer_2<<8);
    if (reserved_sectors != 1)
      {
      printf("WARNING: Number of reserved / boot sectors is not 1.\n");
      if (reserved_sectors < 1)
        bad_boot_sector = TRUE;
      }

    temp_buffer_1 = sector_buffer[0x11];
    temp_buffer_2 = sector_buffer[0x12];
    number_of_root_directory_entries = temp_buffer_1 | (temp_buffer_2<<8);

    temp_buffer_1 = sector_buffer[0x13]; /* first, read the 16bit value */
    temp_buffer_2 = sector_buffer[0x14];
    number_of_logical_sectors_on_drive = temp_buffer_1 | (temp_buffer_2<<8);

    if (number_of_logical_sectors_on_drive==0)
      {
      temp_buffer_1 = sector_buffer[0x20]; /* if it was 0, use 32bit value. */
      temp_buffer_2 = sector_buffer[0x21];
      temp_buffer_3 = sector_buffer[0x22];
      temp_buffer_4 = sector_buffer[0x23];

      number_of_logical_sectors_on_drive=temp_buffer_1 | (temp_buffer_2<<8)
        | (temp_buffer_3<<16) | (temp_buffer_4<<24);
      }

    temp_buffer_1=sector_buffer[0x16];
    temp_buffer_2=sector_buffer[0x17];
    sectors_per_fat=temp_buffer_1 | (temp_buffer_2<<8);

    /* cluster size, geometry, media descriptor and hidden sectors are */
    /* not relevant for the UNFORMAT information */

    if ( ((number_of_root_directory_entries & 15) != 0) ||
         (number_of_root_directory_entries < 16) ||
         (sectors_per_fat < 1) ||
         (number_of_logical_sectors_on_drive < 200) )
      { /* not plausible root directory or FAT or drive size */
      bad_boot_sector = TRUE; /* -ea */
      } else {
      if ( ( 1 + sectors_per_fat + (number_of_root_directory_entries >> 4) +
        5 ) > ( number_of_logical_sectors_on_drive >> 1 ) )
        {
        printf(" UNFORMAT info would take more than half the space?\n");
        printf(" Not saving UNFORMAT info! Bad boot sector?\n");
        bad_boot_sector = TRUE;
        }
      } /* root directory an FAT and drive size was plausible */

    } /* boot sector had 0x55 0xaa magic */
    

  /* If the boot sector is not any good, don't save the file system. */
  if (bad_boot_sector==TRUE)
    {
    printf(" Drive appears unformatted, UNFORMAT information not saved.\n");
    return;
    }

  printf(" Saving UNFORMAT information\n");

  /* Compute the beginning sector of the mirror map and the size of */
  /* the mirror image.     */
  /* Always mirroring only ONE boot sector and ONE (the last) FAT.  */
  mirror_size = 1 + sectors_per_fat + (number_of_root_directory_entries/16);

  mirror_map_size = ((mirror_size+(88/8)+63)/64);
    /* added +(88/8)+63, removed +1 (for better rounding) in 0.91d */
    /* 84+4 bytes are header/trailer, plus 8 bytes per mirrored sector! */

  if (mirror_map_size > 10) /* mirror map buffer is only 5120 bytes */
    {
    printf("one FAT > 160k??? Not saving UNDELETE data.\n");
    /* I believe there is no UNDELETE for FAT32 anyway! */
    return;
    }

  mirror_beginning = (number_of_logical_sectors_on_drive - mirror_size)
    - offset_from_end;
  mirror_map_beginning = mirror_beginning - mirror_map_size;

  /* Compute the locations of the second FAT  and  the root directory */
  /* ***  We use the LAST FAT - root directory follows after it!  *** */
  beginning_of_fat = (sectors_per_fat * (number_of_fats-1))
    + reserved_sectors;

  beginning_of_root_directory = (sectors_per_fat * number_of_fats)
    + reserved_sectors;

  end_of_root_directory = beginning_of_root_directory +
    (number_of_root_directory_entries/16) - 1;
    /* (last sector which is still PART of the root directory) */

  /* Write the mirror map pointer to the last sectors of the logical drive. */
  Clear_Sector_Buffer();

  Convert_Huge_To_Integers(mirror_map_beginning);

  sector_buffer[0] = integer1;
  sector_buffer[1] = integer2;
  sector_buffer[2] = integer3;
  sector_buffer[3] = integer4;

  pointer = 4;

  do                                 /* Add pointer sector flag */
    {
      sector_buffer[pointer] = sector_flag[pointer-4];
      pointer++;
    } while (pointer <= 24);

  if (debug_prog==TRUE)
    {
      printf("[DEBUG]  Writing mirror map pointer to sector ->  %lu\n",
       (number_of_logical_sectors_on_drive - offset_from_end) );
      printf("[DEBUG]  Mirror map will start at sector ->  %lu\n",
        mirror_map_beginning);
    }

  Drive_IO(WRITE, (number_of_logical_sectors_on_drive - offset_from_end), 1);

  /* Create the mirror map and copy the file system to the mirror.  */
  Clear_Sector_Buffer();

  pointer = 0;

  do                                 /* Clear mirror_map buffer */
    {
      mirror_map[pointer] = 0;
      pointer++;
    } while (pointer <= 5119);

  mirror_map[0] = param.drive_letter[0];

  pointer = 1;

  do                                 /* Add mirror map header */
    {
      mirror_map[pointer] = mirror_map_header[pointer-1];
      pointer++;
    } while (pointer <= 12);

  /* Main mirror map creation and copying loop follows:  */

  pointer = 84;
  source_sector = 0;
  destination_sector = mirror_beginning;

  end_of_root_directory_flag = FALSE;
  number_of_bytes_in_mirror_map = 0;

  do
    {
      if ( (source_sector>0) && (source_sector<beginning_of_fat) )
        source_sector=beginning_of_fat;
        /* skip first FAT if 2 FATs present, skip additional res. sectors */

      if (debug_prog==TRUE)
        {
        if (source_sector <= beginning_of_fat) /* 0.91d */
          {
          if (source_sector < beginning_of_fat) /* (first) BOOT sector */
            {
            printf("[DEBUG]  Mirroring BOOT sector %lu at sector %lu\n",
              source_sector, destination_sector);
            }
          else /* beginning of (last) FAT */
            {
            printf("[DEBUG]  Mirroring FAT %lu sector %lu at sector %lu, ",
              number_of_fats, source_sector, destination_sector);
            }
          }
        else /* after FATs */
          {
          if (source_sector == beginning_of_root_directory)
            {
            printf("done.\n");
            printf("[DEBUG]  Mirroring ROOT DIR sector %lu at sector %lu, ",
              source_sector, destination_sector);
            }
          else
            {
            printf("%lu at %lu, ", source_sector, destination_sector);
            }
          }
        } /* debug_prog */

      /* Copy mirror image one sector at a time */
      Drive_IO(READ,source_sector,1);

      Drive_IO(WRITE,destination_sector,1);

      /* Enter mapping information into mirror map buffer */

      Convert_Huge_To_Integers(source_sector);

      mirror_map[pointer+0] = integer1;
      mirror_map[pointer+1] = integer2;
      mirror_map[pointer+2] = integer3;
      mirror_map[pointer+3] = integer4;

      Convert_Huge_To_Integers(destination_sector);

      mirror_map[pointer+4] = integer1;
      mirror_map[pointer+5] = integer2;
      mirror_map[pointer+6] = integer3;
      mirror_map[pointer+7] = integer4;

      source_sector++;
      destination_sector++;
      pointer = pointer + 8;
      number_of_bytes_in_mirror_map = pointer;

      if (source_sector > end_of_root_directory)
        end_of_root_directory_flag = TRUE;
        /* changed >= into > in 0.91d (save ALL root dir sectors) */

    } while (end_of_root_directory_flag==FALSE);
    /* End of main mirror map creation and copying loop!  */

  if (debug_prog==TRUE) printf("done.\n");

  /* Write trailer in mirror map */

  mirror_map[pointer+0] = 0;
  mirror_map[pointer+1] = 0;
  mirror_map[pointer+2] = 0;
  mirror_map[pointer+3] = 0;

  number_of_bytes_in_mirror_map += 4; /* add trailer - 0.91d */

  if ( ((number_of_bytes_in_mirror_map+511)>>9) > mirror_map_size)
    {
    printf("Found a BUG: Mirror map bigger than expected, please report.\n");
    printf("Mirror map will be truncated and mostly unusable.\n");
    }

  /* Write the mirror map onto the disk.   */

  pointer = 0;
  destination_sector = mirror_map_beginning;

  do /* mirror map writing loop */
    {
      loop=0;
      do                               /* Load the sector buffer */
	{
	sector_buffer[loop] = mirror_map[pointer + loop];

	loop++;
	} while (loop < 512);

      if (debug_prog==TRUE)
        printf("[DEBUG]  Writing mirror map data to sector %lu\n",
          destination_sector);

      Drive_IO(WRITE, destination_sector, 1);    /* Write the mirror map   */
						 /* sector.                */
      destination_sector++;
      pointer = pointer + 512;
    } while (pointer < number_of_bytes_in_mirror_map);
    /* end of mirror map writing loop */

  return;

}

