/*  FreeDOS HTML Help Viewer

    READFILE - opens and reads in a text file

    Copyright (c) Express Software 2002-3
    See doc\htmlhelp\copying for licensing terms
    Created by: Joe Cosentino/Rob Platt
*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <dir.h>
#include "help_gui.h"
#include "readfile.h"
#include "unz\unzip.h"


/* min macro isn't implemented by some compilers */
#ifndef min
#define min(a,b)    (((a) < (b)) ? (a) : (b))
#endif

char * readUncompressedFile (const char *filename);
char * readZipFile (unzFile *handle);
void readFileError (const char *filename);

char *
readUncompressedFile (const char *filename)
{
  FILE *f;
  long l;
  char *text_buf;

  f = fopen (filename, "rb");

  if (!f)
    {
      readFileError (filename);
      return 0;
    }

  /* determine the length of file using standard file i/o. -RP
   */
  fseek (f, 0L, SEEK_END);
  l = min (ftell (f), 65530L);
  fseek (f, 0L, SEEK_SET);

  /* Allocate memory for file text buffer */
  while ((text_buf = (char *) malloc (l + 1)) == NULL)
    if (pesListDeleteOldest ())
      {
	show_error ("Cannot allocate enough memory.");
	return 0;
      }
  /* Read file into text buffer */
  if (!fread (text_buf, 1, l, f))
    {
      readFileError (filename);
      free (text_buf);
      return 0;
    }

  /* Clean up and 0-terminate the text buffer */
  fclose (f);
  text_buf[l] = 0;

  return text_buf;
}


char *
readZipFile (unzFile* handle)
{
  long l;
  char *text_buf;
  unz_file_info info;


  /* determine the length of file */
  unzGetCurrentFileInfo(handle, &info, NULL, 0, NULL, 0, NULL, 0);
  l = min (info.uncompressed_size, 65530L);
  if (l == 0) return NULL;

  /* Allocate memory for file text buffer */
  while ((text_buf = (char *) malloc (l + 1)) == NULL)
    if (pesListDeleteOldest ())
      {
	show_error ("Cannot allocate enough memory.");
	return NULL;
      }
  /* Read file into text buffer */
  if (unzReadCurrentFile(handle, text_buf, l) < 0)
  {
    show_error("Could not read the compressed file.");
    free(text_buf);
    return NULL;
  }

  /* 0-terminate the text buffer */
  text_buf[l] = '\0';

  return text_buf;
}


void
readFileError (const char *filename)
{
  char *err_msg;
  int i;

  err_msg = (char *) malloc (strlen (filename) + 12);
  if (err_msg == 0)
    {
      show_error ("Cannot allocate memory\n");
      return;
    }

  sprintf (err_msg, "Can't open %s", filename);

  /* Beautify the err_msg by replacing / with \ */
  for (i = 15; i < strlen (err_msg); i++)
    if (err_msg[i] == '/')
      err_msg[i] = '\\';

  show_error (err_msg);
  free (err_msg);
  return;
}


char* readfile(const char *basedir, const char *filename)
{
   char filetoopen[MAXPATH];
   char fileinzip[MAXPATH];
   char dir[MAXDIR];
   char name[MAXFILE];
   char extension[MAXEXT];
   unzFile *handle;

   fnsplit(filename, NULL, dir, name, extension);

   /* If extension is ".zip" */
   if (strcmpi(extension, ".zip") == 0)
   {
     /* try to load this zip file. */
      strcpy(filetoopen, basedir);
      strcat(filetoopen, filename);
      strcpy(fileinzip, "index.htm");
   }
   else
   {
      char temp[MAXPATH];
      /* other/no extension:
      /* Look at the directory and try to open the associated zip file */
      strcpy(filetoopen, dir);
      strtok(filetoopen, "/\\");
      if (filetoopen[0] != '\0')
      {
         char *p;

         /* Get name of file within the zip: */
         fnmerge(fileinzip, NULL, dir+strlen(filetoopen)+1, name, extension);
         /* Unzip code expects forward slashes: */
         while((p = strchr(fileinzip, '\\')) != NULL)
            *p = '/';

         /* get the zip file name: */
         strcpy(temp, filetoopen);
         sprintf(filetoopen, "%s%s\\%s.zip", basedir, temp, temp);
      }
   }

   if (filetoopen[0] != '\0')
   {
      handle = unzOpen(filetoopen);
      if (handle)
      {
         if (unzLocateFile(handle, fileinzip, 0) != UNZ_OK)
         {
            /* Could not find this file in the zip file */
            /* Look for first html file: */
            if (unzGoToFirstFile(handle) != UNZ_OK)
            {
               show_error("Zip file empty or corrupt");
               return NULL;
            }
            unzGetCurrentFileInfo(handle, NULL, fileinzip, MAXPATH,
                                         NULL, 0, NULL, 0);
            fnsplit(fileinzip, NULL, NULL, NULL, extension);
            while(stricmp(extension, ".htm") != 0)
            {
               if (unzGoToNextFile(handle) != NULL)
               {
                  show_error("Could not find html file in zip file");
                  return NULL;
               }
               unzGetCurrentFileInfo(handle, NULL, fileinzip, MAXPATH,
                                            NULL, 0, NULL, 0);
               fnsplit(fileinzip, NULL, NULL, NULL, extension);
            }
         }

         if (unzOpenCurrentFile(handle) == UNZ_OK)
         {
            char *text_buf = readZipFile(handle);
            unzCloseCurrentFile(handle);
            unzClose(handle);
            return text_buf;
         }
         unzClose(handle);
      }
   }

   /* try as uncompressed .htm file */
	strcpy(filetoopen, basedir);
   strcat(filetoopen, filename);
	if (fclose (fopen (filetoopen, "r")) == -1)
	{
	   /* Could not open as is */
		/* Try with .htm on end */
		strcat(filetoopen, ".htm");
		if (fclose (fopen (filetoopen, "r")) == -1)
      {
         filetoopen[strlen(filetoopen)-4] = '\0'; /* removed added .htm */
         readFileError(filetoopen);
         return NULL;
      }
   }
   return readUncompressedFile(filetoopen);
}