/*
	emm386c.c
	
	Copyright (c) by tom ehlert 2001-2005 - all rights reserved

	Licensed under the Artistic License version
	
	please see LICENSE.TXT for details

	modified for >64M and VCPI support, Michael Devore

	Documented EMS 4.0 only supports up to 32M (0x800 pages), but some EMS
	drivers allow more than 32M.  The EMS 4.0 spec could be extended to 1G-32K
	(map value of 0xFFFF means unmap) without breaking the spec API,
    but unfortunately breaking unsophisticated programs which get confused.
	I have arbitrarily decided to allow up to 32M of EMS allocations,
	leaving the high bit of page allocation alone since specific drivers
	may make use of high bit values other than 0xFFFF for their own purposes,
	or may only check high bit for unmap -- as FreeDOS EMM386 does.

	Michael Devore's changes are not copyrighted and are released
	to the public domain.
	This does not affect copyright on the rest of the code.

	this code is based on ct0890s (copyright 1990 by harry albrecht, c't)
	
	this was taken, hacked, patched, tweaked, to make a real EMM386

	for current operational status see status.txt

*/

#define PROGRAM "JEMM386"

#include "useful.h"

#define VDS

/******************** globals in EMM386.ASM **************************************/
#define MAXK_EMS_ALLOWED (512L*1024L-32L)
#define	UMB_MAX_BLOCKS	8

/* compiler switches, requires switches in JEMM386.INC to be set as well */

#define SBSUPP
#define VMESUPP
#define LOADSUPP
#define A20SUPP
#define EMXSUPP
#define PGESUPP

/* equates, must match the ones in JEMM386.ASM */

#define V86F_SB       1    /* SB compat switch set */
#define V86F_NOCHECK  2    /* NOCHECK switch set */
#define V86F_EMX      4    /* EMX compat switch set */

#define CHECK00FFONLY 0    /* I=TEST checks pages for 0x00 and 0xFF values only */

extern ushort far FRAME;
extern ushort far MAXEMSPAGES;
extern ulong  far TOTAL_MEMORY;
extern ulong  far MAXMEM16K;
#ifdef A20SUPP
extern uchar far NoA20;
#endif
extern uchar far NoVCPI;
extern uchar far NoEMS;
extern uchar far NoFrame;
extern uchar far NoPool;
extern uchar far NoInvlPg;
extern uchar far bV86Flags;

/* these variables are in DGROUP */

extern ulong  MONITOR_ADDR;
extern ulong  EMM_MEMORY_END;
static ulong  POTENTIAL_EMSVCPI_MEMORY;
extern ushort XMS_CONTROL_HANDLE;
extern ushort DMABUFFSIZE;
#ifdef VMESUPP
extern uchar NoVME;
#endif
#ifdef PGESUPP
extern uchar NoPGE;
#endif
extern uchar NoVDS;
extern uchar AltBoot;

/* external ASM functions used */

ushort ISPROTECTEDMODE(void);
int emmcall(uchar);
int xmscall(uchar);
int xmscall32(uchar);
int XMSinit(void);
int VMwareDetect(void);
int EmmStatus(void);
int EmmUpdate(void);
int AddIfContiguousWithDosMem(int,int);
int InstallXMSHandler(int);
int TestForSystemRAM(void *, int, unsigned int *);

/* prototypes */

int MyFunnyMain(void);

/* structures */

/* UMB block 'array' */
struct {
	ushort segment;
	ushort size;
} UMBsegments[UMB_MAX_BLOCKS] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

struct streg32
{
	ulong eax;
	ulong ebx;
	ulong ecx;
	ulong edx;
};

struct streg16
{
	ushort ax;
	ushort bx;
	ushort cx;
	ushort dx;
};

/* public globals */

uchar startup_verbose = 0;
ulong  XMShandleTable = 0;

unsigned char emmfunction;
struct streg16 emmreg16;

void (far *XMSdriverAddress)(void);
struct streg16 reg16;
struct streg32 reg32;

unsigned char bLoad = 0;

/* local globals */

static ushort wFRAMEwanted = 0;		/* should be set from commandline */
static unsigned long dwKBforEMSwanted = 0L;	/* MIN= amount for EMS/VCPI */

static unsigned char MinRequest = 0;	/* MIN= has been set */

static ulong  XmsLinearAdress;			/* address for our allocated memory */
static ulong  XmsAllocatedBytes;		/* memory reserved for EMM          */
static ulong  XmsHighestMemoryByte;		/* simply end of ALL memory         */

static ushort xmsspec3 = 0;
static unsigned long xmslargest = 0;
static unsigned long xmstotal   = 0;
static unsigned long xmshighest = 0;

static uchar SystemMemory[256];	/* 256*4K pagesize = 1MB */
static char ExcludeTest = 0;
static char IncludeTest = 0;
static char szError[] = {"ERROR"};
static char szWarning[] = {"WARNING"};

/*****************************************************************************/

#if 0
static int xmscall(uchar function)
{
	asm mov dx,reg16.dx
    asm mov bx,reg16.bx
	asm mov ah,function

	XMSdriverAddress();

	asm mov reg16.dx,dx
	asm mov reg16.bx,bx
	asm mov reg16.ax,ax

	return reg16.ax;
}      
#endif

#if 0
/* various machinations required since TCC doesn't support 32-bit inline asm */
static int xmscall32(uchar function)
{
	asm db 0x66
	asm mov dx,reg32.edx_low
	asm db 0x66
	asm mov bx,reg32.ebx_low
	asm mov ah,function

	XMSdriverAddress();

	asm	db 0x66
	asm mov	reg32.edx_low,dx
	asm	db 0x66
	asm mov	reg32.ecx_low,cx
	asm	db 0x66
	asm mov	reg32.ebx_low,bx
	asm	db 0x66
	asm mov	reg32.eax_low,ax

	return reg32.eax_low;
}
#endif

#if 0
static int emmcall(uchar function)
{
	emmfunction = function;
	
	emmreg16.ax = (emmreg16.ax & 0xff ) | (function << 8);

/*	printf("%25s in  %04x %04x %04x %04x -", "",emmreg16.ax,emmreg16.bx,emmreg16.cx,emmreg16.dx);
*/
	asm mov dx,emmreg16.dx
	asm mov cx,emmreg16.cx
	asm mov bx,emmreg16.bx
	asm mov ax,emmreg16.ax

	asm int 0x67

	asm mov emmreg16.dx,dx
	asm mov emmreg16.cx,cx
	asm mov emmreg16.bx,bx
	asm mov emmreg16.ax,ax

/*	printf("%1s out %04x %04x %04x %04x \n", "",emmreg16.ax,emmreg16.bx,emmreg16.cx,emmreg16.dx);
*/
	return emmreg16.ax >>8;
	
}
#endif

#if 0
static int XMSinit(void)
{   
	{
   asm     mov ax, 4300h;
   asm     int 2fh;                 /*  XMS installation check */

   asm     cmp al, 80h;
   asm     jne not_detected;

   asm     mov ax, 4310h;           /*  XMS get driver address */
   asm     int 2fh;
        
   asm     mov word ptr XMSdriverAddress+0, bx;
   asm     mov word ptr XMSdriverAddress+2, es;

   asm     mov ax, 4309h;           /*  XMS get xms handle table */
   asm     int 2fh;

   asm     cmp al,43h;
   asm     jne no_table;

   asm     mov word ptr XMShandleTable+0, bx;
   asm     mov word ptr XMShandleTable+2, es;

	}
no_table:
   return 1;
not_detected:
   return 0;
}
#endif

/* try to detect VMware */

#if 0
static int VMwareDetect()
{
	asm	db 0x66			/* mov eax,564d5856h */
	asm	mov ax,0x5868
	asm	dw 0x564d
	asm	db 0x66			/* mov ecx,0ah */
	asm mov cx,0x0a
	asm	dw 0x0000
	asm db 0x66			/* mov ebx,ecx */
	asm	mov bx,cx
	asm	db 0x66			/* mov edx,5658h */
	asm	mov dx,0x5658
	asm	dw 0x0000
	asm	db 0x66			/* in eax,dx */
	asm	in ax,dx
	asm	db 0x66			/* cmp ebx,564d5868h */
	asm cmp bx,0x5868
	asm	dw 0x564d
	asm	jne failed
	return 1;

failed:
	return 0;
}
#endif

#if 0
void Pause()
{
	printf("Any key please");
	asm mov ax,0;
	asm int 0x16;
	
	printf("\n");
}	
#endif

/* end of assembler functions */


/* set memory type , but honour "EXCLUDE=" and "INCLUDE=" types */

static void pascal SetMemoryType(ushort addr, uchar type)
{
	uchar *mem = &SystemMemory[addr >>8];
	
	if (*mem == 'I' && type != 'X')
		;
	else
		*mem = type;
}	

/*
	'R' = RAM
    'E' = EPROM
    'S' = Shadow-RAM activated by UMBPCI
    'G' = GRAPHICS
	'V' = VMWARE allocated, but possibly re-usable via I= (0e800-0ebffh)
	
	'U' = possible UMB space, because nothing else found
	'P' = PAGEFRAME
	
	'I' = INCLUDE = forced from commandline
	'X' = EXCLUDE = forbidden from commandline
	
	
*/	

/* 
	search memory for ROMS, adapters, graphics,...
	
    builds SystemMemory map

    the "checks" which are done are:

    - memory range 0000-9FFF is regarded as "system memory" - dont touch
    - memory range A000-BFFF is regarded as "video memory" -dont touch
    - memory range C000-EFFF is scanned for ROMS,
      if one is found, it is checked if there are pages filled with
      0x00 or 0xFF.
    - memory range C000-EFFF is also checked for RAM. if found pages are
      regarded as "reserved" (must be explicitely included with I= or S=)
    - with option "X=TEST", memory range C000-EFFF is also tested if
      content is 0x00 or 0xFF, anything else excludes page

*/	

static void ScanSystemMemory(void)
{
	uint mem,i;
    uchar far * pmem;
    uchar uc;
	uchar reuse;

	
	for (mem = 0; mem < 0xa0; mem++) /* system memory - reserved */
		SetMemoryType(mem << 8,'R');
		
	for (mem = 0xf0; mem < 0x100; mem++)	/* system EPROM F000..FFFF */
		SetMemoryType(mem << 8,'E');


/* feel free to optimize graphics memory */

	for (mem = 0xa0; mem < 0xb0; mem++) 	/* VGA graphics */
		SetMemoryType(mem << 8,'G');

	for (mem = 0xB0; mem < 0xB8; mem++) 	/* MONO TEXT+graphic */
		SetMemoryType(mem << 8,'G');

	for (mem = 0xB8; mem < 0xC0; mem++) 	/* VGA TEXT */
		SetMemoryType(mem << 8,'G');

	if (VMwareDetect()) {
		/* exclude E800-EFFF for VMware, if present */
		/* E800-EBFF range may be recoverable via I=, so
		use 'V' instead of absolute 'X' */
		for (mem = 0xe8; mem < 0xec; mem++)	{
			SetMemoryType(mem << 8,'V');
		}
		for (mem = 0xec; mem <= 0xf0; mem++) {
			SetMemoryType(mem << 8,'X');
		}
	}

    /* scan for ROMS */

	for (mem = 0xc000; mem < 0xf000;) {
		uint romsize;
		pmem = (uchar far *)MK_FP(mem,0);
		
		if ( pmem[0] != 0x55u || pmem[1] != 0xaau) {
								/* no adapter EPROM */
			mem += 2048/16;		/* advance by 2K    */   
			continue;
		}

		/* ensure valid EPROM signature allocates minimum 2K for 0 size */
		/* keep romsize aligned to 2K boundary, rounding up */
        romsize = (((uint)pmem[2] * 2 + 3) / 4);

		if (startup_verbose)	
			printf(" EPROM at %X, size %u kB\n", mem, romsize);
								/* romsize given in 512 byte*/

		for ( i = 0; i < romsize; i+=2)	{
            if (SystemMemory[mem>>8] == 'X')
                ;
            else
			if (!IncludeTest || (mem & 0xff)) {
				SetMemoryType(mem,'E');
			} else {
				pmem = (uchar far *)MK_FP(mem, 0);
                reuse = 1;
#if CHECK00FFONLY
		/* it the whole 4K block filled with 00 or FF? */
				for (i = 0; i < 4096; i++) {
                    if (*(pmem+i) != 0 && *(pmem+i) != 0xff) {
#else
        /* it the whole 4K block filled with one value? */
        /* and do NOT check the last byte in the page! */
				for (i = 0, uc = *pmem; i < 4095; i++) {
                    if (*(pmem+i) != uc) {
#endif
						SetMemoryType(mem,'E');
						reuse = 0;
						break;
					}
				}
				if (reuse) {
					SetMemoryType(mem,'I');
					mem += 2048/16;	/* advanced by 2K to make 4K total */
				}
			}
			mem += 2048/16;		/* advance by 2K	*/
		}
	}

#if 1
    /* test if RAM is in the non-excluded regions. */
    i = 0;
    do {
        if (i = TestForSystemRAM(&SystemMemory, i, &mem)) {
            printf(" %s: system memory found at %X-%X\n", szWarning, i, i+mem-1);
            i = (i + mem) >> 8;
        }
    } while (i);
#endif

	/* if ExcludeTest is set, we need to scan all 'U' memory and ensure
     it is all 0 or FF values */

	if (ExcludeTest) {
		for (mem = 0xa0; mem < 0xf0; mem++) {
			if (SystemMemory[mem] == 'U') {
				/* this would be an upper memory block by default */
				/* don't check final byte as this can be garbage */
				pmem = (uchar far *)MK_FP(mem << 8, 0);
                for (i = 0; i < 4095; i++, pmem++) {
                    uc = *pmem;
					if ((uc != 0) && (uc != 0xff)) {
						/* found a nonzero, non-FF value in memory block */
						/* mark the whole block as excluded */
						SystemMemory[mem] = 'X';
						break;
					}
				}
			}
		}
	}

	/* for (i = 0xa0; i < 0xf8; i++)
			printf("%x00 : %c\n",i,SystemMemory[i]); */
}

/* 
	find a contiguous area of 64 KB 
	should handle commandline option like "FRAME=D000"
*/	
static ushort LocatePageFrame(void)
{   
    int base,i;
	ushort frame = 0;
    uchar bSearching = 0;
    uchar bWarning = 0;
    uchar bHardWanted = 0;

    if (wFRAMEwanted)
        bHardWanted = 1;
    else
        wFRAMEwanted = 0xE000;

	if (wFRAMEwanted) {
		base = wFRAMEwanted >> 8;

		for (i = 0; i < 16; i++) {
/* if a FRAME is explicitely set, ignore RAM/Video pages above A000 */
			if (bHardWanted && (base >= 0xA0) && ((SystemMemory[base+i] == 'R') || (SystemMemory[base+i] == 'G')))
                bWarning = 1;
            else
 			if (SystemMemory[base+i] != 'U')
				break;
		}
		if (i == 16) {
			frame = base;
			goto frameset;
        }
        if (bHardWanted)
            printf(" selected page frame %04x not available, searching automatically\n", wFRAMEwanted);
		bSearching = 1;
	}

	for (base = 0xa0; base <= 0xE8; base++)	{
		for (i = 0; i < 16; i++) {
			if (SystemMemory[base+i] != 'U')
				break;
		}
		if (i == 16) {
			frame = base;
		}
	}

	if (frame == 0)	{
/*		printf("no suitable page frame found, which we can't handle\n");	*/
        printf(" %s: no suitable page frame found, EMS functions limited.\n", szWarning);
		NoFrame = 1;
		return 0;
	}


frameset:
	if (startup_verbose || bSearching)
		printf(" using PAGEFRAME %02x00\n", frame);
	if (bWarning && (!bSearching))
        printf(" %s: PAGEFRAME %02x00 might not work reliably\n", szWarning, frame);

	fmemset(SystemMemory+frame,'P',16);
		
	return frame << 8;
}

/* old:check if there is 16 KB contiguos memory here to be used as UMB */ 
/* new:check if there is 4K KB contiguous memory here to be used as UMB */ 

static int isUMBMemory(ushort base)
{
	if ((SystemMemory[base] == 'U') || (SystemMemory[base] == 'I') ||
	( (NoEMS || NoFrame) && SystemMemory[base] == 'P') )
        return TRUE;
	else
		return FALSE;
}

/* test if page is (shadow) RAM */

static int isShadowRAM(ushort base)
{
    uchar far * pmem;
    int rc = FALSE;
    uchar c;

    if (SystemMemory[base] == 'S') {
        pmem = (uchar far *)MK_FP(base,0);
        c = *pmem;
        *pmem = 0x55;
        if (*pmem == 0x55) {
            *pmem = 0xAA;
            if (*pmem == 0xAA)
                rc = TRUE;
        }
        *pmem = c;
    }
	return rc;
}

/*
	return number of pages, we need to do UMB mapping ,
	return value is in 4K pages
	does not count mapping at FF00
*/
static int UMBpageswanted(void)
{
    int wanted = 0;
    int i;
	for (i = 0xa0; i < 0xf8; i++)
		if (isUMBMemory(i)) {
			wanted++;
		}

	return wanted;
}	

/* get total/free XMS memory */

static int GetXMSMemoryStatus(ushort usev3)
{
	int badstatus = 0;
	if (usev3) {
        if (xmscall32(0x88)) {  /* query free extended memory */
            xmslargest = reg32.eax;
            xmstotal   = reg32.edx;
            goto done;
        }
        badstatus = reg32.ebx & 0xff;
	}
	reg16.bx = 0;
    if (xmscall(8)) {  /* query free extended memory */
        xmslargest = reg16.ax;
        xmstotal   = reg16.dx;
        badstatus  = 0;
    } else
        badstatus = reg16.bx & 0xff;
done:
    if (badstatus)
        return 0;
    return 1;
}

/* if XMS manager doesnt export handle table, calc a reasonable
 amount of fixed memory to allocate
*/

static unsigned int GetReasonableWantedEMMAmount(void)
{
    unsigned int wWanted;

    if (!GetXMSMemoryStatus(xmsspec3))
        return 0;
    if (xmslargest >= (64 * 1024UL))
        wWanted = 32 * 1024;
    else
        wWanted = (unsigned int)(xmslargest / 2);
    return wWanted;
}

/* alloc XMS memory block for EMM
*/

ushort AllocXMSMemory(int usev3, ulong kbneeded)
{
    ushort xmshandle = 0;
    ulong kbtotal;

    for ( ; !xmshandle; dwKBforEMSwanted /= 2) {
        kbtotal = kbneeded + dwKBforEMSwanted;
        if (kbtotal <= 0xFFFFUL) {
            reg16.dx = kbtotal;
            if (xmscall(9)) {
                xmshandle = reg16.dx;
                break;
            }
            if (usev3) goto usev3ver;
            /* try largest free block based allocation */
            if ((xmslargest > kbneeded + dwKBforEMSwanted / 2) &&
                (xmslargest < kbtotal)) {
                reg16.dx = xmslargest;
                if (xmscall(9)) {
                    dwKBforEMSwanted = xmslargest - kbneeded;
                    xmshandle = reg16.dx;
                    break;
                }
            }
        } else {
        usev3ver:
            reg32.edx = kbtotal;
            if (xmscall32(0x89)) {
                xmshandle = reg32.edx & 0xFFFF;
                break;
            }
            /* try largest free block based allocation */
            if ((xmslargest <= kbneeded + dwKBforEMSwanted / 2) ||
                (xmslargest >= kbtotal)) {
                /* outside range, try next loop */
                break;
            }
            reg32.edx = xmslargest;
            if (xmscall32(0x89)) {
                dwKBforEMSwanted = xmslargest - kbneeded;
                xmshandle = reg32.edx & 0xFFFF;
                break;
            }
        }
        if (startup_verbose)
            printf(" allocated %lu kB (%lu + %lu) from XMS\n", kbtotal, kbneeded, dwKBforEMSwanted);

    }
    return xmshandle;
}

/*
	allocate memory from XMS
	find highest memory address
    determine monitor load address (must currently be <= 15MB
    dwKBforEMSwanted = kB explicitely wanted for EMS with MIN=xxxx
    + memory required for UMBs
*/

static int XMSallocAndInitMem(unsigned long kbneeded)
{   
    unsigned long ulcalc;
    unsigned u;
    unsigned long dwEMSoriginal = dwKBforEMSwanted;

    if (!GetXMSMemoryStatus(xmsspec3)) {
        printf(" %s: can't get XMS memory status\n", szError);
		return 0;
	}


	if (startup_verbose)	
		printf(" XMS largest block %lu kB, XMS total mem %lu kB\n",
				xmslargest, xmstotal);

/* reality check to throttle requests far beyond available XMS, later actual
	adjustments	are small and need not be compensated for here */
	if ((dwKBforEMSwanted + kbneeded > xmstotal) && (kbneeded < xmstotal)) {
		dwKBforEMSwanted = xmstotal - kbneeded;
    }

/* leave a little extended memory, if possible, for programs that want some XMS */
	if ((xmslargest > kbneeded + 384UL) && (xmslargest < kbneeded + dwKBforEMSwanted + 384UL)) {
        dwKBforEMSwanted = xmslargest - kbneeded - 384UL;
	}

/* kbwanted is memory in EMS pages, must be 16 kB aligned */

    dwKBforEMSwanted = (dwKBforEMSwanted + 15) & 0xfffffff0l;
    if (dwKBforEMSwanted > MAXK_EMS_ALLOWED) {
        dwKBforEMSwanted = MAXK_EMS_ALLOWED;
    }

    /* default is: all memory */
    POTENTIAL_EMSVCPI_MEMORY = xmstotal * 1024UL;

    if (NoPool)	{ /* Pool sharing off? */
        if (dwKBforEMSwanted < xmstotal)
            POTENTIAL_EMSVCPI_MEMORY = dwKBforEMSwanted * 1024UL;
	}

    if (startup_verbose)
        printf(" potential EMS/VCPI memory: %lu kB\n", POTENTIAL_EMSVCPI_MEMORY / 1024UL);

    /* if MIN= is set, increase MAXEMSPAGES and MAXMEM16 if needed */
    /* here dwKBforEMSwanted is always below 512 MB */
    if (dwKBforEMSwanted > MAXEMSPAGES * 16UL) {
        MAXEMSPAGES = dwKBforEMSwanted / 16;
        if (MAXEMSPAGES > MAXMEM16K)
            MAXMEM16K = MAXEMSPAGES;
/*      printf("MAXMEM16K=%lu, MAXEMSPAGES=%u\n", MAXMEM16K, MAXEMSPAGES); */
    } else
        if (NoEMS) {
            MAXEMSPAGES = dwKBforEMSwanted / 16;
  /* even with NOEMS alloc space for EMS memory management for 8192 kb */
  /* else some DOS extenders (among them DOS4G) won't work (wastes 2,5 kB) */
#if 1
            if (MAXEMSPAGES < 512)
                MAXEMSPAGES = 512;
#endif
        }

    /* MAXMEM16K may have been set by MAX=, and above the limit */
    if (MAXMEM16K > (POTENTIAL_EMSVCPI_MEMORY / (1024UL * 16)))
        MAXMEM16K = POTENTIAL_EMSVCPI_MEMORY / (1024UL * 16);

    /* MAXMEM16K may have been set by MAX=, and below 32 MB! */
    /* this is valid, but then adjust max EMS pages as well */
    if (MAXMEM16K < MAXEMSPAGES) {
        MAXEMSPAGES = MAXMEM16K;
    }

/*  the memory pooling need ((XMS total / 1.5M) + 1) * 64 bytes
    for pool allocation	table entries
    1.5M is pool allocation maximum memory control,
    64 is pool block size,
    if dynamic XMS allocation is on, 128 more items are needed,
    which represent the maximum number of XMS handles
*/

	ulcalc = MAXMEM16K * 16 / 1536 + 2; /* no of pool items */
	if (!NoPool)
		ulcalc = ulcalc + 128;
	ulcalc = ulcalc << 6;  /*  * 64 = pool mem to alloc */

	/* 4+1 bytes for each EMS page needed */
	/* 256*8 bytes for status table of EMS handles */
	ulcalc = ulcalc + 5 * MAXEMSPAGES + 256*8;
    /* 256*8 bytes for EMS names [8 bytes if NOEMS] */
#if 0 /* it is valid to alloc EMS even with NOEMS! */
	if (NoEMS)
		ulcalc = ulcalc + 8;
    else
#endif
		ulcalc = ulcalc + 256*8;
	ulcalc = (ulcalc + 1023) / 1024UL;	/* convert bytes back to K */
    ulcalc = (ulcalc + 3) & 0xfffffffc; /* 4k page align */
	if (startup_verbose)
		printf(" %lu kB needed for VCPI and EMS handling\n", ulcalc);
    kbneeded += ulcalc;
#if 1
    /*
     the DMA buffer must be 64kb aligned. Since EMS pages *must* be
     allocated from memory physically "behind" the DMA buffer, there may
     be some space wasted, max 60-16=44 kB
    */
    if (dwKBforEMSwanted && (DMABUFFSIZE > 4)) {
        u = DMABUFFSIZE >= 64 ? 32 : DMABUFFSIZE/2;
        kbneeded = kbneeded + u;
        if (startup_verbose)
            printf(" %u kB added to account for DMA buffer 64 kB alignment\n", u);
    }
#endif

	/* allocate memory from XMS */
    XMS_CONTROL_HANDLE = AllocXMSMemory(xmsspec3, kbneeded);

	if (!XMS_CONTROL_HANDLE) {
        printf(" %s: can't allocate enough XMS memory(%lu kB)\n", szError, kbneeded);
		return 0;
	}

	/* lock handle to make a linear adress */
	
	reg16.dx = XMS_CONTROL_HANDLE;
	if (!xmscall(0x0c))	{
        printf(" %s: can't lock XMS memory\n", szError);
        xmscall(0x0A);/* free the block */
		return 0;
    }
    if (dwKBforEMSwanted < dwEMSoriginal)
        printf(" %s: EMS memory has been reduced to %lu kB\n", szWarning, dwKBforEMSwanted);

	XmsAllocatedBytes =  ((ulong)(dwKBforEMSwanted + kbneeded) * 1024);
	
    XmsLinearAdress	     = 	((ulong)reg16.dx << 16) | reg16.bx;

	XmsHighestMemoryByte =  (ulong)xmstotal * 1024L + XmsLinearAdress;


/*	printf("xms locked memory at %lx, top of mem %lx(%luMB) alloc bytes %lx(%lu kB)\n",
				XmsLinearAdress,
				XmsHighestMemoryByte,XmsHighestMemoryByte/(1024*1024l),
				XmsAllocatedBytes,XmsAllocatedBytes/(1024));
*/

	return 1;	
	
}   

/* prepare UMB segment table */

static int PrepareUMBSegments(void)
{

	ushort mem,size,index;

				/* now prepare UMB segments to be used later */
	index = 0;
	for (mem = 0xa0; mem < 0xf8; )	/* allow umbs in f000-f7ff */
		if (((!isShadowRAM(mem)) && (!isUMBMemory(mem))))
			mem++;	/* allow 4K UMB's	*/
		else {
            for (size = 1; ; size++)	/* 4K UMB's */
				if (((!isShadowRAM(mem+size)) && (!isUMBMemory(mem+size))))
					break;

#if 0
            if (mem == 0xa0) {
                if (AddIfContiguousWithDosMem(mem << 8, size)) {
                    mem += size;
                    continue;
                }
            }
#endif

			UMBsegments[index].segment = mem  << 8;
			UMBsegments[index].size    = size << 8;			


			if (startup_verbose) 
				printf(" UMB %d at %x0-%xf (%u kB)\n",
					index,
					UMBsegments[index].segment,
					UMBsegments[index].segment + UMBsegments[index].size - 1,
					UMBsegments[index].size / (1024/16)
                      );

			index++;
			if (index >= UMB_MAX_BLOCKS)
				break;

			mem += size;				
		}

	if (UMBsegments[0].segment == 0) {
		printf(" no suitable UMB memory block found\n");
		return 0;
	}

    return 1;
}


void PrintStartup()
{
    printf( PROGRAM " v5.33 [" __DATE__ "]"
            " (c) tom ehlert 2001-2006 c't/H. Albrecht 1990\n");
    return;
}

/* conditional printf only if startup_verbose */

static void cprintf(char * pszText)
{
    if (startup_verbose)
        printf(pszText);
}

/* called on startup.
	handle commandline "I=B800-BFFF X=E000-EFFF" ...
	search for EPROMS+adapters (network cards)
	determine frame address
	...
	mode = 0 if called as driver
	mode = 1 if called as EXE
	
	return: 0         - everything fine
	        errorcode - exit code/abort driver
*/ 

int TheRealMain(int mode, char far *commandline)
{
    char far *found;
    int i,j;
    int bHelp = 0;
    int bOptionSet = 0;

    if (mode != EXECMODE_EXE) {
        PrintStartup();
        bLoad = 1;
    } else {
#ifdef LOADSUPP
        if (FindCommand(commandline, "LOAD", &found) )
            bLoad = 1;
        else {
            NoVCPI = 0xFF;
#ifdef A20SUPP
            NoA20  = 0xFF;
#endif
#ifdef VMESUPP
            NoVME  = 0xFF;
#endif
#ifdef PGESUPP
            NoPGE  = 0xFF;
#endif
        }
#endif
        if (FindCommand(commandline, "/?", &found) ||
            FindCommand(commandline, "-?", &found) ||
            FindCommand(commandline, "/H", &found) ||
            FindCommand(commandline, "-H", &found) ) {
            bHelp = 1;
        }
    }

/*    printf("'real' commandline is '%Fs'\n",commandline); */

	if (FindCommand(commandline, "NOVCPI", &found) ) {
		NoVCPI = TRUE;
        bOptionSet = 1;
	}
	if (FindCommand(commandline, "VCPI", &found) ) {
		NoVCPI = FALSE;
        bOptionSet = 1;
    }
#ifdef A20SUPP
    if (FindCommand(commandline, "NOA20", &found) ) {
        NoA20 = 1;
        bOptionSet = 1;
	}
    if (FindCommand(commandline, "A20", &found) ) {
        NoA20 = 0;
        bOptionSet = 1;
    }
#endif
#ifdef VMESUPP
    if (FindCommand(commandline, "NOVME", &found) ) {
        NoVME = 1;
        bOptionSet = 1;
	}
    if (FindCommand(commandline, "VME", &found) ) {
        NoVME = 0;
        bOptionSet = 1;
    }
#endif
#ifdef PGESUPP
    if (FindCommand(commandline, "NOPGE", &found) ) {
        NoPGE = 1;
        bOptionSet = 1;
	}
    if (FindCommand(commandline, "PGE", &found) ) {
        NoPGE = 0;
        bOptionSet = 1;
	}
#endif
    if (bLoad) {
	fmemset(SystemMemory,'U',sizeof(SystemMemory));

	/******* commandline handling **********/

	if (FindCommand(commandline, "/VERBOSE", &found) ||
	    FindCommand(commandline, "/V",       &found) ) {
		startup_verbose = 1;
	}

    if (FindCommand(commandline, "NODYN", &found)) {
		NoPool = 1; /* pool sharing off */
	}

    if (FindCommand(commandline, "NOINVLPG", &found)) {
		NoInvlPg = 1; /* dont use INVLPG opcode */
	}

    if (FindCommand(commandline, "MIN=", &found)) {
		dwKBforEMSwanted = GetValue(found,10,TRUE);
		if (startup_verbose)	
            printf(" wanted preallocated EMS memory: %lu kB\n", dwKBforEMSwanted);
		MinRequest = 1; /* limit EMS to this amount */
	}

	if (FindCommand(commandline, "NOEMS", &found) )	{
		cprintf(" NOEMS: EMS disabled (mostly :-)\n");
		NoEMS = TRUE;
	}

	if (FindCommand(commandline, "NOVDS", &found) )	{
		NoVDS = TRUE;
	}

	if (FindCommand(commandline, "VDS", &found) ) {
		NoVDS = FALSE;
	}

    if (FindCommand(commandline, "FRAME=NONE", &found)) {
        NoFrame = TRUE;
    } else {
        if (FindCommand(commandline, "FRAME=", &found) ||
            FindCommand(commandline, "/P",     &found) ) {
			wFRAMEwanted = GetValue(found,16,FALSE);
            if (startup_verbose)
                printf(" wanted FRAME=%X\n", wFRAMEwanted);
		}
	}

	if (FindCommand(commandline, "X=TEST", &found) ) {
		ExcludeTest = 1;
	}

	if (FindCommand(commandline, "I=TEST", &found) ) {
		IncludeTest = 1;
	}

#ifdef SBSUPP
	if (FindCommand(commandline, "SB", &found) ) {
		bV86Flags = bV86Flags | V86F_SB;
    }
#endif
#ifdef EMXSUPP
	if (FindCommand(commandline, "EMX", &found) ) {
		bV86Flags = bV86Flags | V86F_EMX;
	}
#endif
#if 0
    if (FindCommand(commandline, "MEMCHECK", &found) ) {
		MEMCHECK = 1;
	}
#endif

	if (FindCommand(commandline, "NOCHECK", &found) ) {
		bV86Flags = bV86Flags | V86F_NOCHECK;
	}

	if (FindCommand(commandline, "MAX=", &found) ) {
		MAXMEM16K = GetValue(found,10,TRUE);
		MAXMEM16K /= 16UL;
	}

/*
	if (FindCommand(commandline, "ENDALLOC", &found) )
		{ 
		ENDALLOC = 1;
		}
*/

	if (FindCommand(commandline, "ALTBOOT", &found) ) {
		AltBoot = 1;
	}

    if (FindCommand(commandline, "NOHI", &found) ) {
		/* NOHI is a no-op, but helps MS EMM386 switch compatibility */
	}

	if (FindCommand(commandline, "NOMOVEXBDA", &found) ) {
		/* NOMOVEXBDA is a no-op, but helps MS EMM386 switch compatibility */
	}

	if (FindCommand(commandline, "RAM", &found) ) {
		/* ignore bare RAM option */
	}


    if (FindCommand(commandline, "D=", &found)) {
        ushort dmasize =  GetValue(found,10,FALSE);
        if (dmasize <= 128)
            DMABUFFSIZE = (dmasize+3) & -4;
        else {
            DMABUFFSIZE = 128;
            printf(" %s: wanted DMA buffer size too large, set to 128 kB\n", szWarning);
        }
    }
					/* "I=a000-afff"  "X=d000-dfff" */
	for (;;) {
		ushort rangestart,rangestop;
		char memtype;
		
		memtype = 'I';
		
        if (!FindCommand(commandline, "I=", &found) ) {
            memtype = 'S';
            if (!FindCommand(commandline, "S=", &found) ) {
                memtype = 'X';
                if (!FindCommand(commandline, "X=", &found) ) {
                    break;
                }
            }
		}

		rangestart =  GetValue(found,16,FALSE);

		if (*found == '-') {
			fmemcpy( found, found+1, 
								fstrlen(found+1) +1); 

		
			rangestop  =  GetValue(found,16,FALSE);
		
            if (startup_verbose)
                printf(" %c=%x..%x\n",memtype, rangestart,rangestop);
			
			if (rangestart && rangestop && (rangestart <= rangestop))
				for ( ; rangestart < rangestop; rangestart++)
					SetMemoryType(rangestart,memtype);
				
		}
		
    }
    } /* endif bLoad */

	/******* commandline handling done, are there remainders **********/

	commandline = skipWhite(commandline);

    if (*commandline) {
        printf("* ignored commandline: '%Fs'\n", commandline);
        if (mode == EXECMODE_EXE)
            return 1;
    }


    /* shall the monitor be loaded? If no, do either:
		display usage info    or
		status report
	*/
	if (!bLoad) {
        printf("\n");

        if (bHelp) {
        PrintStartup();
        printf( "usage: add a line to CONFIG.SYS: DEVICE=" PROGRAM ".EXE [ options ]\n");

        printf("available options are:\n"
#ifdef A20SUPP
				"+A20/NOA20   - A20-disable emulation on/off (default on)\n"
#endif
				" ALTBOOT     - use alternate reboot hook (saves 4 kB, but is not 100%% safe)\n"
                " D=###       - set DMA buffer size in kB (default is 64, max is 128)\n"
#ifdef EMXSUPP
				" EMX         - increased EMX DOS extender compatibility\n"
#endif
				" FRAME=E000  - set wanted pageframe for EMS (FRAME=NONE disables frame)\n"
				" I=D000-D7FF - force a region to be used for UMBs. Without this option\n"
				"               range C000-EFFF is scanned for unused pages. May also be used\n"
				"               to add (parts of) regions A000-BFFF or F000-F7FF as UMBs. Don't\n"
				"               use this option if you don't know what you are doing!\n"
				" I=TEST      - scan ROMs for unused pages, include found regions as UMBs\n"
                " S=D000-D7FF - assume Shadow-RAM activated by UMBPCI, include it as UMB\n"
#ifdef LOADSUPP
                "*LOAD        - install JEMM386 from the command line\n"
#endif
				" MAX=#####   - limit for VCPI (and EMS if < 32M) memory in kB (default 120 MB)\n"
				" MIN=#####   - reserve up to ##### kB for EMS/VCPI memory on init (default 0)\n"
				" NOCHECK     - disallow access to address space without RAM (MMIO)\n"
				" NOEMS       - disable EMS handling\n"
				" NODYN       - no dynamic XMS memory allocation (use MIN= to set fix amount)\n"
				" NOINVLPG    - don't use INVLPG opcode\n"
#ifdef PGESUPP
                "+PGE/NOPGE   - Page Global Enable feature usage on/off (default off)\n"
#endif
#ifdef SBSUPP
				" SB          - SoundBlaster driver compatibility mode\n"
#endif
				"+VCPI/NOVCPI - VCPI Support on/off (default on)\n"
                " VDS/NOVDS   - Virtual DMA Services on/off (default on)\n"
#ifdef VMESUPP
                "+VME/NOVME   - V86-Mode Extensions on/off (default on)\n"
#endif
				" /VERBOSE    - display additional details during start\n"
				" X=D000-D7FF - exclude region from being touched or used by Jemm386\n"
				" X=TEST      - scan memory region C000-EFFF for UMB exclusion\n\n"
                " '+': option is also valid when loaded from the commandline\n"
                " '*': option is *only* valid when loaded from the commandline\n"
				);				
        } else {
            if (bOptionSet) {
                if (EmmUpdate())  /* update Jemm386 status */
                    printf("option(s) passed to " PROGRAM "\n");
#if 0
                else
                    printf("*** " PROGRAM " communication error ***\n");
#endif

            } else {
                PrintStartup();
                EmmStatus();  /* display Jemm386 status */
            }
        }

		return 1;
	}

    /******* options set, now process **********/

	if (ISPROTECTEDMODE()) {
		printf(" %s: already in protected mode, can't continue\n", szError);
		return 1;
	}

	if (!XMSinit())	{
		printf(" %s: no XMS handler found, required\n", szError);
		return 1;
    }
    if (xmscall(0) && (reg16.ax >= 0x300))
        xmsspec3 = 1;

    if (!xmscall(5)) {  	/* enable A20 local */
		printf(" %s: enable A20 failed\n", szError);
		return 1;
    }

/* system state ok for loading driver */

    if (NoVCPI)
        cprintf(" VCPI disabled\n");
    if (NoVDS)
		cprintf(" VDS disabled\n");
#if 0
    if (NoVCPI && NoEMS)  /* auto disable pooling is no longer useful */
        NoPool = 1;       /* if wanted it can be disabled with NODYN */
#endif
    /* if no int 2fh, function 4309h support, disable pool sharing */

    if ((!XMShandleTable) && (!NoPool)) {
        NoPool = 1;
        printf(" %s: XMS host doesn't provide handle array, dynamic memory allocation off!\n", szWarning);
	}
    if ((NoPool) && (!MinRequest)) {
        dwKBforEMSwanted = GetReasonableWantedEMMAmount();
    }

	ScanSystemMemory();			/* build up system memory map */

    if (NoEMS)
        NoFrame = 1;

	if (!NoFrame)
		FRAME = LocatePageFrame();	/* find a contiguos area of 64 KB */

	/*
			allocate from XMS the memory we need
    		this is memory for UMBs, including FF00 if ALTBOOT=0

			+ 16kB for the monitor code, GDT, IDT, stack
			+ 12kB for page tables
			+ 12kB for TSS + IO-Bitmap (includes 3 kB reserve for rounding)
			+ 64kB +-X for DMA buffering
			+ room for other control structures made inside function
								
			+ what the user wants for EMS
	*/

    i = (UMBpageswanted() + (AltBoot ? 0 : 1)) * 4;
    j = 16 + 24 + DMABUFFSIZE;

    if (startup_verbose)
        printf(" Needed: %u kB for UMBs, %u kB for DMA buffer, %u kB for monitor\n",
               i, DMABUFFSIZE, j - DMABUFFSIZE);
    dwKBforEMSwanted = dwKBforEMSwanted + i;

	if (!XMSallocAndInitMem(j)) {
        xmscall(6); /* local disable A20 */
		return 1;
	}

	MONITOR_ADDR   = XmsLinearAdress;
	EMM_MEMORY_END = XmsLinearAdress + XmsAllocatedBytes;
	TOTAL_MEMORY   = XmsHighestMemoryByte;  

	if (startup_verbose)	
        printf(" XMS memory block for monitor: %lx-%lx, XMS total=%lx\n",
			MONITOR_ADDR, EMM_MEMORY_END, TOTAL_MEMORY);

	return 0;				/* OK so far , continue */
}


/* called just before we go resident */
/* finishing_touches() will only be called if the monitor has installed successfully */

int finishing_touches(ulong ulFirstPage)
{ 

    if (startup_verbose) {
        printf(" physical start address of EMS pages: %lX\n", ulFirstPage);
        if (!emmcall(0x42))	{ /* kickstart dynamic count of available EMS pages */
            printf(" total/available EMS pages: %d/%d (= %lu(%lu) kB)\n",
               emmreg16.dx, emmreg16.bx, (ulong)emmreg16.dx*16, (ulong)emmreg16.bx*16);
        }
	}

	return MyFunnyMain();

}


static void emmerror(char *s)
{
	printf("EMM failed: %s\n",s);
	
	printf("func %02x, out %04x %04x %04x %04x\n",emmfunction, emmreg16.ax,emmreg16.bx,emmreg16.cx,emmreg16.dx);
}

/*  
	post processing:
	
	here some magic happens:
	we use (mostly) standard EMS functionality to map some memory
	at the UMB locations
*/

int MyFunnyMain()
{   
	int i;
	ushort pageswanted; 
	ushort emmhandle = 0;	/* allocate UMBs via system handle */
    ushort logicalpage;
    int    bInstallUMB = 1;

/* check install state */
/*	
	emmcall(0x46);
	printf("version %x\n",emmreg16.ax);

	emmcall(0x41);
	printf("page frame at %x\n",emmreg16.bx);
	
	emmcall(0x42);
	printf("total pages %x(%dMB), free %x(%dMB)\n",emmreg16.dx,emmreg16.dx/(1024/16),emmreg16.bx,emmreg16.bx/(1024/16));

	emmcall(0x4b);
	printf("emm handles %x\n",emmreg16.bx);
*/
/*
	for (mem =0xa0; mem < 0xf8; mem+=4)
		printf("mem %x %c - %d\n",mem,SystemMemory[mem], isUMBMemory(mem));
*/

    pageswanted = UMBpageswanted();

    /* if ALTBOOT not set, add 1 for ROM FF00h shadowing */
    /* then map page at FFxxx so we can trap jumps to FFFF:0 for rebooting */
    /* the 0xff physical page flags EMM386 to copy over ROM image */
    /* with ALTBOOT, a keyboard hook is installed instead */

    if (!AltBoot) {
        pageswanted++;
        SetMemoryType(0xff00, 'U');
    }

	if (!pageswanted) {
        bInstallUMB = 0;
		goto mapping_done;
	}

	/* allocate a handle + some memory */

	if (startup_verbose)
		printf(" allocating %u 4K pages = %u kB for UMBs\n",
									pageswanted,pageswanted*4);

/* UMB allocations done through system handle via reallocation, since
 system handle is already present with zero assigned pages */
	emmreg16.dx = emmhandle;
	emmreg16.bx = (pageswanted+3)/4;	/* round up */
/*	if (emmcall(0x43) != 0)	*/
	if (emmcall(0x51) != 0)	{
		printf(" allocating %d kB for UMBs:",pageswanted*4);
		emmerror("");
        bInstallUMB = 0;
		goto mapping_done;
	}

	/*  here comes the funny part:
		during initialization phase, calling EMM_MAP_PAGE (0x44)
		with pysical page (AL) > 3 is possible.
		meaning:
		
		AL = highest 8 bits of logical adress, AL=E4 --> address E4000

		initialization phase is terminated with AL=9F
		
	*/	

								/* map pages in to the UMB area */

	cprintf(" mapping UMBs (4K each) at:");

	logicalpage = 0;
	
	for (i = 0xa0; i < 0x100; i++)	/* allow 4K UMB's */
		if (isUMBMemory(i))	{
			if (startup_verbose)	
				printf("%x00 ",i);     
		
			emmreg16.ax = i;
			emmreg16.bx = logicalpage;
			emmreg16.dx = emmhandle;
		
            if (emmcall(0x44)) {
                emmerror("mapping UMB page !!");
                bInstallUMB = 0;
				goto mapping_done;
            }

			logicalpage++;
		}
			
	cprintf("\n");

mapping_done:

    if (bInstallUMB)
        bInstallUMB = PrepareUMBSegments();

    return bInstallUMB;

}
/* EOF */
