;
; UDMA.ASM    Written 20-Mar-2005 by Jack R. Ellis
;
; UDMA is free software.   You can redistribute and/or modify it, under
; the terms of the GNU General Public License (hereafter called GPL) as
; published by the Free Software Foundation, either version 2 of GPL or
; any later versions, at your option.	UDMA is distributed in the hope
; that it will be useful, but WITHOUT ANY WARRANTY and without even the
; implied warranties of MERCHANTABILITY nor of FITNESS FOR A PARTICULAR
; PURPOSE!  See the GPL for details.  You ought to have received a copy
; of the GPL with these UDMA files.  If not, write to the Free Software
; Foundation Inc., 59 Temple Place Ste. 330, Boston, MA 02111-1307 USA.
; http://www.gnu.org/licenses/
;
; Special thanks to Luchezar I. Georgiev for his INESTIMABLE advice and
; help in research, revising, enhancing, and test of the original UDMA!
;
; This is a DOS driver designed to handle 1 to 4 UltraDMA hard-disks on
; PC motherboards having a VIA 8235 or equivalent chipset.   The driver
; determines which of the IDE units are actually UltraDMA hard-disks at
; initialization and will run all such disks.	All UltraDMA disks from
; mode 0 ATA-16 thru mode 7 ATA-166 may be used.    An UltraDMA disk is
; assumed to handle full LBA mode (63 sectors, 255 heads and a designed
; cylinder count).   "LBA mode" I-O requests are supported for FreeDOS,
; MS-DOS V7.xx, and other systems that allow them.   LBA values over 28
; bits shall cause the driver to use "Read/Write Extended" DMA commands
; and need an ATA-6 or newer hard-disk.	  LBA values of 28 bits or less
; shall use regular DMA commands.   24-bit "CHS mode" is also supported
; for MS-DOS V6.xx and earlier.	  Data accessed using CHS calls must be
; located in the initial 8-GB of the disk.
;
; The driver intercepts BIOS INT13 read or write requests only.	  Other
; INT13 requests (including seeks) and read/write requests with invalid
; parameters will be "passed" back to the BIOS or some other driver for
; handling.   If a user I-O buffer is not DWORD aligned, crosses a 64K-
; boundary or fails a VDS "lock", the I-O request will use a 64K buffer
; in XMS memory with UltraDMA to or from the buffer, to avoid "passing"
; these requests to the BIOS for execution in slow PIO mode!   Although
; UltraDMA specifies word-aligned buffers, ERRATA in some chipsets does
; require DWORD alignment and avoiding a 64K DMA address boundary!
;
; Beginning with version 1.6 of this driver, the following return codes
; have been added to help in diagnosing "problem" systems and chipsets.
; On exit from successful I-O requests, the AH-register is zero and the
; carry flag is reset.	 If an error occurs, the carry flag is SET, and
; the AH-register contains one of the following codes:
;
;  Code 08h - DMA timed out
;	0Fh - DMA error
;	20h - Controller busy before I-O
;	21h - Controller busy after I-O
;	80h - First DRQ timed out
;	AAh - Disk not ready before I-O
;	ABh - Disk not ready after I-O
;	CCh - Disk FAULT before I-O
;	CDh - Disk FAULT after I-O
;	E0h - Hard error at I-O end
;	FEh - BIOS/driver read MISMATCH (init only)
;	FFh - XMS memory error
;
;
; Revision History:
; ----------------
;  V7.7  20-Mar-05   JE   Minor update "in step" with UDMA2/UDMA2A V2.2
;  V7.6  19-Feb-05   JE   Fixed error in setting LBA bits 32-47
;  V7.5   7-Jan-05   JE   Fixed no-read-test flag
;  V7.4  18-Dec-04   JE   Fixed DRQ timeout code
;  V7.3  11-Dec-04   JE   Fixed FOOLISH disk-select bug (My apologies!)
;  V7.2  10-Dec-04   JE   No EDD BIOS error abort, more code reductions
;  V7.1	  2-Dec-04   JE	  Total revision, derived from V1.6 UDMA2
;  V7.0	 14-Feb-04   JE	  Merged new init, V6.8 run-time, V6.9 CHS code
;  V6.9	  8-Feb-04   JE	  Any CHS "geometry" O.K., buffered I.D. input
;  V6.8	 28-Jan-04   JE	  If no EDD/DPTE, 4 disks at 80h-83h units only
;  V6.7	 16-Jan-04   JE	  Renumbered to replace UDMA, init code reduced
;  V2.2	 25-Dec-03   JE	  Corrected "read test" diagnostic messages
;  V2.1	 24-Dec-03   JE	  Use XMS for read tests, to reduce UDMA size
;  V2.0	 21-Dec-03   JE	  Controller-name displays, multi-sector tests
;  V1.9	  6-Dec-03   JE	  Fixed VDS init bug, buffer/diagnostic "swap"
;  V1.8	  3-Dec-03   JE	  Fixed "STI" bug, "DMA only" now 528 bytes
;  V1.7	 25-Nov-03   JE	  If no XMS driver, allow "DMA only" usage
;  V1.6	 22-Nov-03   JE	  Fixed init reads, added full error codes
;  V1.5	 15-Nov-03   JE	  Added all UDMA init functions but ctlr. name
;  V1.4	 14-Nov-03   JE	  Corrected DMA-status reset
;  V1.3	 13-Nov-03   JE	  "DoIO" does ALL I-O, "XMS error" now 0FFh
;  V1.2	 12-Nov-03   JE	  No "timeout error", other size reductions
;  V1.1	  7-Nov-03   JE	  Used 80386 test from V5.9 UDMA
;  V1.0	  6-Nov-03   JE	  Initial release (had been named UDMA-E)
;
;
; General Program Equations.
;
%define VER 'V7.7, 20-Mar-2005.'
RDYTO	equ	008h		;384-msec minimum I-O timeout.
RBYTES	equ	(2*1024*65536*12/14318180+1)*512 ;Read-test byte count.
	;( = microseconds per tick, but adjusted for binary megabytes).
	;Number of reads this length per tick = transfer rate in MB/s.
RS_SC	equ	RBYTES/512	;"Read speed" input sector count.
RC_SC	equ	58		;"Read compare" total sector count.
RC_CYL	equ	3		;"Read compare" starting disk address.
RC_HD	equ	15
RC_SEC	equ	5
BIOSTMR equ	0046Ch		;BIOS "tick" timer address.
HDISKS	equ	00475h		;BIOS hard-disk count address.
VDSFLAG equ	0047Bh		;BIOS "Virtual DMA" flag address.
CR	equ	00Dh		;ASCII carriage-return.
LF	equ	00Ah		;ASCII line-feed.
;
; Driver Return Codes.
;
DMATIMO equ	0E8h		;DMA timeout code, 008h at exit.
DMAERR	equ	0EFh		;DMA error   code, 00Fh at exit.
CTLRERR equ	000h		;Ctlr. busy  code, 020h/021h at exit.
DRQTIMO equ	080h		;DRQ timeout code.
DISKERR equ	08Ah		;Disk-busy   code, 0AAh/0ABh at exit.
WFLTERR equ	0ACh		;Write-fault code, 0CCh/0CDh at exit.
HARDERR equ	0BFh		;Hard-error  code, 0E0H at exit.
XMSERR	equ	0FFh		;XMS memory-error code.
;
; IDE Controller Register Definitions.
;
CDATA	equ	001F0h		;Data port.
CSUBCM	equ	CDATA+1		;Subcommand register.
CSECCT	equ	CDATA+2		;I-O sector count.
CDSEL	equ	CDATA+6		;Disk-select and upper LBA.
CCMD	equ	CDATA+7		;Command register.
CSTAT	equ	CDATA+7		;Primary status register.
CSTAT2	equ	CDATA+206h	;Alternate status register.
;
; Controller Status and Command Definitions.
;
BSY	equ	080h		;IDE controller is busy.
RDY	equ	040h		;IDE disk is "ready".
FLT	equ	020h		;IDE disk has a "fault".
DRQ	equ	008h		;IDE data request.
ERR	equ	001h		;IDE general error flag.
DMI	equ	004h		;DMA interrupt occured.
DME	equ	002h		;DMA error occurred.
DRCMD	equ	0C8h		;DMA read command (write is 0CAh,
				;    LBA48 commands are 025h/035h).
SETM	equ	003h		;Set Mode subcommand.
SETF	equ	0EFh		;Set Features command.
LBABITS equ	0E0h		;Fixed LBA command bits.
;
; LBA "Device Address Packet" Layout.
;
struc	DAP
DapPL	resb	1		;Packet length.
	resb	1		;(Reserved).
DapSC	resb	1		;I-O sector count.
	resb	1		;(Reserved).
DapBuf	resd	1		;I-O buffer address (segment:offset).
DapLBA	resw	1		;48-bit logical block address (LBA).
DapLBA1	resw	1
DapLBA2	resw	1
endstruc
;
; DOS "Request Packet" Layout.
;
struc	RP
	resb	2		;(Unused by us).
RPOp	resb	1		;Opcode.
RPStat	resw	1		;Status word.
	resb	9		;(Unused by us).
RPSize	resd	1		;Resident driver size.
endstruc
RPERR	equ	08003h		;Packet "error" flags.
RPDON	equ	00100h		;Packet "done" flag.
;
; DOS Driver Device Header.
;
@	dd	0FFFFFFFFh	;Link to next header block.
	dw	08000h		;Driver "device attributes".
	dw	Strat		;"Strategy" routine offset.
IDEAd	equ	$-1		;(Lower IDE status address, after Init).
	dw	DevInt		;"Device-Interrupt" routine offset.
DMAAd	equ	$-2		;(DMA command-reg. address, after Init).
	db	'UDMA$',0,0,0,0	;Driver name (NO "arrowhead" for V7.1+).
;
; Resident Driver "Data Page" Variables.
;
VLF	db	0		;VDS lock flag (1 = user buffer locked).
Units	dd	0000000FFh	;IDE "active units" table, set by Init:
	dd	0000000FFh	;  Byte 0:    BIOS unit (0FFh inactive).
	dd	0000000FFh	;  Byte 1:    CHS sectors per head.
	dd	0000000FFh	;  Byte 2-3:  CHS sectors/cylinder.
PRDAd	dd	IOAdr-@		;PRD 32-bit command addr. (Init set).
	db	0		;IDE "upper" sector count (always 0).
LBA2	db	0FAh,0F0h,08Ah	;IDE "upper" LBA bits 24-47.
SecCt	db	080h		;IDE "lower" sector count.
LBA	db	0,0,0		;IDE "lower" LBA bits 0-23.
DSCmd	db	0		;IDE disk-select and LBA commands.
IOCmd	db	0		;IDE command byte.
XMHdl	dw	0		;XMS buffer handle number (Init set).
XMOfs	dd	0		;XMS 32-bit buffer offset (Init set).
VDSLn	dd	0		;VDS and XMS buffer length.
VDSOf	dd	0		;VDS 32-bit offset.
VDSSg	dd	0		;VDS 16-bit segment (hi-order zero).
IOAdr	dd	0		;VDS and DMA 32-bit address.
IOLen	dd	080000000h	;DMA byte count and "end" flag.
XMSSH	equ	VDSOf		;(XMS parameters share the VDS block).
XMSSA	equ	VDSOf+2
XMSDH	equ	VDSSg+2
;
; Driver Entry Routine.  For CHS requests, the registers contain:
;
;   AH      Request code.  We handle 002h read and 003h write.
;   AL      I-O sector count.
;   CH      Lower 8 bits of starting cylinder.
;   CL      Starting sector and upper 2 bits of cylinder.
;   DH      Starting head.
;   DL      BIOS unit number.  We handle 080h and up (hard-disks).
;   ES:BX   I-O buffer address.
;
; For LBA requests, the registers contain:
;
;   AH      Request code.  We handle 042h read and 043h write.
;   DL      BIOS unit number.  We handle 080h and up (hard-disks).
;   DS:SI   Pointer to Device Address Packet ("DAP"), described above.
;
Entry	pusha			;Entry -- save all CPU registers.
	mov	bp,16		;Reset active-units table index.
@LastU	equ	$-2		;(Last-unit index, set by Init.  MUST
				;  be 16 while Init TESTS all disks!). 
NextU	sub	bp,byte 4	;Any more active units to check?
	js	QuickX		;No, request NOT for us -- exit quick!
	cmp	dl,[cs:bp+Units-@] ;Does request unit match our table?
	jne	NextU		;No, see if more table entries remain.
	sti			;Ensure CPU interrupts are enabled.
	cld			;Ensure FORWARD "string" commands!
	push	ds		;Save CPU segment registers.
	push	es
	mov	dl,0BEh		;Mask out LBA and write request bits.
	and	dl,ah
	cmp	dl,002h		;Is this a CHS or LBA read or write?
	jne	Pass		;No, let BIOS handle this request.
	shl	ah,1		;Is this an LBA read or write request?
	jns	ValSC		;No, go validate CHS sector count.
	mov	al,[si+DapSC]	;Get "DAP" I-O sector count.
	cmp	dword [si+DapBuf],byte -1 ;64-bit "DAP" buffer address?
	jne	ValSC		;No, go validate "DAP" sector count.
Pass	pop	es		;"Pass" request -- reload segment regs.
	pop	ds
QuickX	mov	bp,sp		;Reload CPU flags saved by Int 13h.
	push	word [bp+20]
	popf
	popa			;Reload all CPU registers.
	jmp	0000:0000	;Go to next routine in Int 13h chain.
@PrvI13 equ	$-4		;(Previous INT13 vector, set by Init).
ValSC	dec	al		;Is sector count zero or over 128?
	js	Pass		;Yes?  Let BIOS handle this "No-No!".
	inc	ax		;Restore sector count -- LBA request?
	js	GetDAP		;Yes, get remaining "DAP" parameters.
	xchg	ax,cx		;CHS -- save request code and sectors.
	mov	si,0003Fh	;Set SI-reg. to starting sector.
	and	si,ax
	dec	si
	mov	di,dx		;Set DI-reg. to starting head.
	shr	al,6		;Set AX-reg. to cylinder number.
	xchg	al,ah
	mul	word [cs:bp+Units+2-@]  ;Convert cylinder to sectors.
	xchg	ax,di		     ;Swap low-order and head number.
	mov	al,[cs:bp+Units+1-@] ;Convert head to sectors.
	mul	ah
	add	si,ax		;Add to starting sector.
	add	si,di		;Add in cylinder sectors.
	adc	dl,dh
	xchg	ax,bx		;Get buffer offset in AX-register.
	xchg	ax,cx		;Swap offset with request/sectors.
	xor	di,di		;Reset upper LBA address bits.
	jmp	short SetDS	;Go set our DS-register.
GetDAP	les	cx,[si+DapBuf]	;Get "DAP" I-O buffer address.
	mov	di,[si+DapLBA2]	;Get "DAP" logical block address.
	mov	dx,[si+DapLBA1]
	mov	si,[si+DapLBA]
SetDS	push	cs		;Set our DS-register.
	pop	ds
	xor	bx,bx		;Zero BX-reg. for relative commands.
	mov	[bx+LBA-@],si	;Save 48-bit logical block address.
	mov	[bx+LBA+2-@],dl
	mov	[bx+LBA2-@],dh
	mov	[bx+LBA2+1-@],di
	shr	dx,12		;Shift out LBA bits 16-27.
	or	di,dx		;Anything in LBA bits 28-47?
	jz	LBA28		;No, use LBA28 read/write command.
	shl	ah,3		;LBA48 -- get command as 020h/030h.
	jmp	short GetAdr	;Go get IDE and LBA address bytes.
LBA28	xchg	dh,[bx+LBA2-@]	;LBA28 -- reload & reset bits 24-27.
	or	ah,(DRCMD+1)	;Get LBA28 read/write command + 5.
GetAdr	mov	di,CDSEL-00100h	;Get primary device-address bytes.
@DMALo1	equ	$-1		;(Lower DMA command address, Init set).
	shr	bp,3		;Is this a primary-channel request?
	jz	DevAdr		;Yes, set IDE & PCI address bytes.
	mov	di,CDSEL+00680h	;Get secondary device-address bytes.
@DMALo2	equ	$-1		;(Lower DMA command address, Init set).
DevAdr	mov	[bx+IDEAd-@],di ;Set current IDE & PCI address bytes.
	mov	dl,(LBABITS/32)	;Get disk-select & LBA command bits.
	rcl	dl,5
	or	dl,dh		;"Or" in LBA28 bits 24-27 (if any).
	mov	dh,005h		;Get final IDE command byte.
	xor	dh,ah		;(LBA28 = C8h/CAh, LBA48 = 25h/35h).
	mov	[bx+DSCmd-@],dx	;Set LBA and IDE command bytes.
	mov	[bx+SecCt-@],al	;Set I-O sector count.
	mov	ah,0		;Set VDS/XMS and DMA buffer lengths.
	shl	ax,1
	mov	[bx+VDSLn+1-@],ax
	mov	[bx+IOLen+1-@],ax
	mov	[bx+VDSOf-@],cx	;Set VDS offset and segment.
	mov	[bx+VDSOf+2-@],bx
	mov	[bx+VDSSg-@],es
	or	dword [bx+IOAdr-@],byte -1 ;Invalidate VDS address.
	mov	ax,08103h		   ;VDS "lock" user buffer.
	mov	dx,0000Ch
	call	VDLock
	jc	GoToBuf			   ;Error -- do buffered I-O.
	cmp	dword [bx+IOAdr-@],byte -1 ;Got a valid VDS address?
	je	NoVDS		;No, set 20-bit user buffer address.
	inc	byte [bx+VLF-@]	;Set VDS "lock" flag.
	mov	ax,[bx+IOAdr-@]	;Get low-order VDS buffer address.
ChkOdd	test	al,003h		;Is user I-O buffer 32-bit aligned?
	jnz	NoLock		;No, use buffered I-O routines below.
	mov	cx,[bx+IOLen-@]	;Get lower ending DMA address.
	dec	cx		;(IOLen - 1 + IOAdr).
	add	ax,cx		;Would this I-O cross a 64K boundary?
	jc	NoLock		;Yes, use buffered I-O routines below.
	call	DoDMA		;Do direct DMA I-O with user's buffer.
Done	mov	bp,sp		;Done -- point to saved registers.
	mov	[bp+19],al	;Set error code in exiting AH-reg.
	rcr	byte [bp+24],1	;Set error flag in exiting carry
	rol	byte [bp+24],1	;  bit, in flags saved by Int 13h.
	call	VDUnlk		;If needed, "unlock" user I-O buffer.
	pop	es		;Reload all CPU registers and exit.
	pop	ds
	popa
	iret
NoVDS	mov	ax,16		;No VDS -- get 20-bit buffer segment.
	mul	word [bx+VDSSg-@]
	add	ax,[bx+VDSOf-@]	;Add in buffer offset value.
	adc	dx,bx
	mov	[bx+IOAdr-@],ax	;Set 20-bit user buffer address.
	mov	[bx+IOAdr+2-@],dx
	jmp	short ChkOdd	;Go check for "odd" buffer alignment.
NoLock	call	VDUnlk		;Buffered I-O -- "unlock" user buffer.
GoToBuf	jmp	BufIO		;Go to buffered I-O routines below.
	db	0		;(Unused alignment "filler").
;
; Subroutine to execute read and write commands.
;
BufDMA	mov	dword [bx+IOAdr-@],0  ;Buffered -- set XMS buffer addr.
@XBufAd	equ	$-4		;(XMS 32-bit buffer address, Init set).
DoDMA	mov	dx,[bx+DMAAd-@]	;Ensure any previous DMA is stopped!
	in	al,dx		;(On some older chipsets, if DMA is
	and	al,0FEh		;  running, reading an IDE register
	out	dx,al		;  causes the chipset to "HANG"!!).
	mov	al,[bx+DSCmd-@]	;Select our desired disk.
	and	al,0F0h
	mov	dl,[bx+IDEAd-@]
	mov	dh,001h
	out	dx,al
	mov	di,dx		;Save IDE drive-select address.
	mov	es,bx		;Point to low-memory BIOS timer.
	mov	si,BIOSTMR
	mov	cx,(RDYTO*256)+FLT   ;Get timeout count & "fault" mask.
	add	ch,[es:si]	     ;Set CH-reg. with timeout limit.
	call	ChkRdy		     ;Await controller- and disk-ready.
	jc	IOExit		     ;If error, exit immediately!
	test	byte [bx+IOCmd-@],012h  ;Is this a write request?
	jnz	SetDMA		;Yes, reset DMA command register.
	mov	al,008h		;Get "DMA read" command bit.
SetDMA	mov	dx,[bx+DMAAd-@]	;Reset DMA commands and set DMA mode.
	out	dx,al
	inc	dx		;Point to DMA status register.
	inc	dx
	in	al,dx		;Reset DMA status register.
	or	al,006h		;(Done this way so we do NOT alter
	out	dx,al		;  the "DMA capable" status flags!).
	push	si		;Save BIOS timer pointer.
	inc	dx		;Set PRD pointer to our DMA address.
	inc	dx
	mov	si,(PRDAd-@)
	outsd
	mov	bx,001F7h	;Set IDE parameter-output flags.
NxtPar	lea	dx,[di+CSECCT-CDSEL-1] ;Point to IDE sector count -1.
IDEPar	inc	dx		;Output all ten LBA48 parameter bytes.
	outsb			;(1st 4 overlayed by 2nd 4 if LBA28!).
	shr	bx,1		;More parameters to go in this group?
	jc	IDEPar		;Yes, loop back and output next one.
	jnz	NxtPar		;If first 4 done, go output last 6.
	pop	si		;Reload BIOS timer pointer.
	mov	dh,003h		;Get IDE alternate-status address.
	dec	dx		;(Primary-status address | 300h - 1).
ChkDRQ	cmp	ch,[es:si]	;Too long without 1st data-request?
	je	ErrDRQ		;Yes?  Return carry and DRQ error!
	in	al,dx		;Read IDE alternate status.
	and	al,DRQ		;Has 1st data-request arrived?
	jz	ChkDRQ		;No, loop back and check again.
	mov	dx,[bx+DMAAd-@]	;Set DMA Start/Stop bit (starts DMA).
	in	al,dx
	inc	ax
	out	dx,al
ChkDMA	inc	dx		;Point to DMA status register.
	inc	dx
	in	al,dx		;Read DMA controller status.
	dec	dx		;Point back to DMA command register.
	dec	dx
	and	al,DMI+DME	;DMA interrupt or DMA error?
	jnz	HltDMA		;Yes, halt DMA and check results.
	cmp	ch,[es:si]	;Has our DMA transfer timed out?
	jne	ChkDMA		;No, loop back and check again.
HltDMA	push	ax		;Save ending DMA status.
	in	al,dx		;Reset DMA Start/Stop bit.
	and	al,0FEh
	out	dx,al
	pop	ax		;Reload ending DMA status.
	cmp	al,DMI		;Did DMA end with only an interrupt?
	jne	ErrDMA		;No?  Go see what went wrong.
	inc	dx		;Reread DMA controller status.
	inc	dx
	in	al,dx
	test	al,DME		;Any "late" DMA error after DMA end?
	jnz	DMAEnd		;Yes?  Return carry and DMA error!
	inc	cx		;Check "fault" and hard-error at end.
ChkRdy	lea	dx,[di+CSTAT-CDSEL] ;Read IDE primary status.
	in	al,dx
	test	al,BSY+RDY	;Controller or disk still busy?
	jg	ChkErr		;No, check for "fault" or hard-error.
	cmp	ch,[es:si]	;Too long without becoming ready?
	jne	ChkRdy		;No, loop back and check again.
	test	al,BSY		;BAAAD News!  Did controller go ready?
	mov	ax,(256*CTLRERR)+DISKERR ;(Get not-ready error codes).
	jmp	short WhichE	;Go see which error code to return.
ErrDRQ	mov	al,DRQTIMO	;BAAAD News!  Get DRQ error code.
	stc			;Set carry flag (error) and exit!
IOExit	ret
ChkErr	and	al,cl		;Disk "fault" or hard-error?
	jz	IOExit		;No, all is well -- go exit above.
	test	al,FLT		;BAAAD News!  Is the disk "faulted"?
	mov	ax,(256*WFLTERR)+HARDERR ;(Get hardware error codes).
WhichE	jz	EndErr		;If "zero", use AL-reg. return code.
	mov	al,ah		;Use AH-reg. return code of this pair.
EndErr	add	al,cl		;Add 1 if error occurred at I-O end.
	stc			;Set carry flag to denote "error"!
	ret			;Exit.
ErrDMA	test	al,DME		;BAAAD News!  Did DMA end with error?
DMAEnd	mov	ax,(256*DMAERR)+DMATIMO	 ;(Get DMA error codes).
	jmp	short WhichE	;Go see which error code to return.
;
; Subroutine to do a VDS "lock" or "unlock".
;
VDUnlk	shr	byte [bx+VLF-@],1 ;Was input buffer "locked" by VDS?
	jnc	VDExit		;No, just exit below.
	mov	ax,08104h	;Get VDS "unlock" parameters.
	xor	dx,dx
VDLock	mov	di,(VDSLn-@)	;Point to VDS parameter block.
VDInit	push	cs
	pop	es
	push	bx		;Save BX-reg. and execute VDS request.
	int	04Bh
VDDone	pop	bx		;Reload our BX-register.
	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
VDExit	ret			;Exit.
NoXMEnd	equ	$		;End of resident "No XMS" driver.
;
; Buffered I-O routines, put here so they and the XMSMov subroutine
;   can be "dismissed" during driver-init if no XMS driver is found!
;
BufIO	shl	dword [bx+VDSOf-@],16  ;Convert to XMS handle & offset.
	test	byte [bx+IOCmd-@],012h ;Is this a write request?
	jnz	BufOut		;Yes, use output routine below.
	call	BufDMA		;Input all data to driver XMS buffer.
	jc	Done		;If error, post return code & exit!
	call	XMSMov		;Move XMS data to user input buffer.
	jmp	short DoneJ	;Done -- post any return code & exit.
BufOut	call	XMSMov		;Output -- move data to XMS buffer.
	jc	Done		;If error, post return code & exit!
	call	BufDMA		;Output all data from XMS buffer.
DoneJ	jmp	Done		;Done -- go post return code & exit.
;
; Subroutine to move data to or from the driver's XMS buffer.
;
XMSMov	push	cs		;Point ES-reg. to our data.
	pop	es
	mov	di,(XMSDH-@)	;Point to XMS destination field.
	jnz	XMSet		;If output, just set XMS destination!
	mov	si,(XMSSH-@)	;Point to XMS user-buffer address.
	movsw			;Move user-buffer address from
	movsw			;  XMS source to XMS destination.
	movsw
	mov	di,(XMSSH-@)	;Point to XMS source field.
XMSet	mov	si,(XMHdl-@)	;Set XMS handle and buffer offset as
	movsw			;  input source or output destination.
	movsw
	movsw
XMGo	mov	ah,00Bh		;Get XMS "move data" request code.
XMCall	push	bx		;Save BX-reg. and execute XMS request.
	call	0000:0000	;(SI-reg. points to IOLen after move).
@XEntry equ	$-4		;(XMS "entry" address, set by init).
	dec	ax		;Return 0 if success, -1 if error.
	sar	ax,1		;If error, set carry flag on.
	jmp	short VDDone	;Go reload BX-reg. and exit above.
	db	0		;(Unused alignment "filler").
ResEnd	equ	$		;End of resident driver.
;
; Initialization Messages.
;
UDMsg	db	CR,LF,'UDMA Disk Driver ',VER,CR,LF,'$'
DNMsg	db	'is '
DName	equ	$		;Disk names overlay the startup msgs.
DNEnd	equ	DName+40
PRMsg	db	'No 80386 CPU$'
PEMsg	db	'PCI BIOS below V2.0C$'
NEMsg	db	'No controller found$'
MEMsg	db	'Bus-Master setup BAD$'
NXMsg	db	'No XMS manager$'
XEMsg	db	'XMS init error$'
NBMsg	db	'; using only DMA I-O!',CR,LF,'$'
VEMsg	db	'VDS lock error$'
DspUnkn db	'Vendor ID '
DspVID	db	'0000h, device ID '
DspDID	db	'0000h$'
PCMsg	db	'UltraDMA controller found at I-O address '
DspAddr db	'0000h.',CR,LF,'    $'
PCMsg2	db	', PCI bus '
DspBus	db	'00h, device '
DspDev	db	'00h, function '
DspFnc	db	'0.',CR,LF,'$'
NDMsg	db	'No disks to use$'
EBMsg	db	'EDD error!  BIOS unit ignored$'
HOMsg	db	'Hardware-only disk scan:',CR,LF,'$'
PMMsg	db	'Primary-master disk $'
PSMsg	db	'Primary-slave disk $'
SMMsg	db	'Secondary-master disk $'
SSMsg	db	'Secondary-slave disk $'
MSMsg	db	'.',CR,LF,'    Set to UltraDMA mode '
CurMode db	'0, ATA-'
DMode	db	'16. $'
RTMsg	db	'   Read test = $'
RRate	db	'    0 MB/sec'
CRMsg	db	'.',CR,LF,'$'
AEMsg	db	'absent or non-ATA$'
DEMsg	db	'is not UltraDMA$'
NPMsg	db	'CHS-values ERROR$'
SEMsg	db	'Set-Mode error$'
BEMsg	db	'failed BIOS read!  $'
TEMsg	db	'FAILED read test!  $'
EMsg1	db	'DMA timed out$'
EMsg2	db	'DMA error$'
EMsg3	db	'Controller busy before I-O$'
EMsg4	db	'Controller busy after I-O$'
EMsg5	db	'First DRQ timed out$'
EMsg6	db	'Disk not ready before I-O$'
EMsg7	db	'Disk not ready after I-O$'
EMsg8	db	'Disk FAULT before I-O$'
EMsg9	db	'Disk FAULT after I-O$'
EMsg10	db	'Hard error at I-O end$'
EMsg11	db	'BIOS/driver read MISMATCH$'
EMsg12	db	'XMS memory error$'
RCMsg	db	'Return code '
RCode	db	'00h$'
Suffix	db	'; driver NOT loaded!',CR,LF,'$'
;
; Initialization Tables And Variables.
;
	align	4
HDNames dw	PMMsg		;Table of hard-disk "name" pointers.
	dw	PSMsg
	dw	SMMsg
	dw	SSMsg
Modes	db	'16. '		;Mode 0 = ATA-16  UltraDMA mode table.
	db	'25. '		;Mode 1 = ATA-25.
	db	'33. '		;Mode 2 = ATA-33.
	db	'44. '		;Mode 3 = ATA-44  (Rare but possible).
	db	'66. '		;Mode 4 = ATA-66.
	db	'100.'		;Mode 5 = ATA-100.
	db	'133.'		;Mode 6 = ATA-133.
	db	'166.'		;Mode 7 = ATA-166.
MsgTbl	db	008h		;Driver error-message codes/addresses.
	dw		EMsg1
	db	00Fh
	dw		EMsg2
	db	020h
	dw		EMsg3
	db	021h
	dw		EMsg4
	db	080h
	dw		EMsg5
	db	0AAh
	dw		EMsg6
	db	0ABh
	dw		EMsg7
	db	0CCh
	dw		EMsg8
	db	0CDh
	dw		EMsg9
	db	0E0h
	dw		EMsg10
	db	0FEh
	dw		EMsg11
MsgEnd	db	0FFh
	dw		EMsg12
IVDSLen	dd	ResEnd-@	;VDS length (driver size).
IVDSOfs	dd	0		;VDS 32-bit offset.
IVDSSeg	dd	0		;VDS 16-bit segment (hi-order zero).
IVDSAdr	dd	0		;VDS 32-bit driver address.
Packet	dd	0		;"Init" request packet address.
Bucket	dd	0		;Working 32-bit "bucket".
NoRdSw	db	0		;"No read tests" flag.
HDCount db	0		;Remaining hard-disk count.
EDDFlag db	0		;"EDD BIOS present" flag.
HDUnit	db	0		;Current BIOS unit number.
HDIndex db	0		;IDE "index" number.
RCSecNo	db	0		;"Read compare" sector address.
RCSects	db	0		;"Read compare" remaining sectors.
;
; Main driver-initialization routine, entered from the DOS "device
;   interrupt" logic below, after it does one-time-only functions.
;
I_RScn	mov	dx,NDMsg	;Point to "No disk to use" message.
	xor	ax,ax		;Load & reset EDD BIOS flag.
	xchg	al,[EDDFlag]
	or	ax,ax		;Were we scanning v.s. DPTE data?
	jz	near I_Err	;No?  Display "No disk" and exit!
I_Scan	mov	ax,00080h	;Reset hard-disk unit number & index.
	mov	[HDUnit],ax
	mov	byte [HDCount],0  ;Reset remaining hard-disk count.
@BIOSHD equ	$-1		;(BIOS hard-disk count, set above).
	cmp	ah,[EDDFlag]	;Will disk scan use the EDD BIOS?
	jne	I_Next		;Yes, go start with BIOS unit 80h.
	mov	dx,HOMsg	;Display "hardware-only" message.
	call	I_Dspl
I_Next	movzx	bx,[HDIndex]	;Get disk unit-number index.
	cmp	bh,[EDDFlag]	;Are we using DPTE data from BIOS?
	je	I_ChMS		;No, check disk at "fixed" addresses.
	mov	ah,048h		;Get next BIOS disk's EDD parameters.
	mov	dl,[HDUnit]
	mov	si,EDDBuff
	call	I_Int13
	jc	I_ErEDD		;Error?  Ignore this BIOS unit!
	cmp	dword [si+26],byte -1  ;Valid DPTE pointer?
	je	near I_More	;No, ignore unit & check for more.
	les	si,[si+26]	;Get this disk's DPTE pointer.
	mov	bx,15		;Calculate DPTE checksum.
	xor	cx,cx
I_CkSm	add	cl,[es:bx+si]
	dec	bx
	jns	I_CkSm
	jcxz	I_EDOK		;If checksum O.K., use parameters.
I_ErEDD	mov	dx,EBMsg	;Display "EDD error" and ignore unit!
	jmp	short I_ErrD
I_EDOK	mov	bx,00010h	;Initialize IDE unit number index.
	and	bl,[es:si+4]
	shr	bl,4
	mov	ax,[es:si]	;Get disk's IDE base address.
	cmp	ax,CDATA	;Is this a primary-channel disk?
	je	I_ChMS		;Yes, set disk channel and unit.
	cmp	ax,(CDATA-080h)	;Is this a secondary-channel disk?
	jne	I_More		;No, ignore unit & check for more.
	inc	bx		;Adjust for secondary channel.
	inc	bx
I_ChMS	mov	ax,bx		;Separate channel and master/slave.
	shr	al,1
	mov	ah,(LBABITS/32)	;Get drive-select "nibble".
	rcl	ah,5
	ror	al,1		;Set channel offset (2nd = 080h).
	mov	[@HDOffs],al
	mov	dx,CDSEL	;Get IDE disk-select address.
	xor	dl,al
	mov	al,ah		;Select master or slave disk.
	out	dx,al
	shl	bx,2		;Save disk's "Units" table index.
	push	bx
	shr	bx,1		;Get "channel name" message index.
	mov	dx,[bx+HDNames] ;Display disk's IDE channel name.
	call	I_Dspl		;("Primary master", etc.).
	mov	ah,008h		;Get BIOS CHS values for this disk.
	mov	dl,[HDUnit]
	call	I_Int13
	xchg	ax,dx		;Set AX-reg. with head-number value.
	pop	bx		;Reload disk's unit number index.
	mov	dx,NPMsg	;Point to "parameter ERROR" message.
	jc	I_ErrD		;Error -- display msg. & ignore disk.
	mov	al,cl		;Save sectors per head value.
	and	al,03Fh		;(Clear 2 high-order cylinder bits).
	mov	[bx+Units+1-@],al
	inc	ah		;Get heads/cylinder (BIOS value + 1).
	mul	ah		;Save sectors per cylinder value.
	mov	[bx+Units+2-@],ax
	mov	al,[HDUnit]	;Activate this disk in main driver.
	mov	[bx+Units-@],al
	push	bx		;Validate this UltraDMA disk.
	call	I_ValD
	pop	bx
	jnc	I_More		;If no errors, check for more disks.
	mov	byte [bx+Units-@],0FFh  ;DELETE disk in main driver!
I_ErrD	call	I_Dspl		;Display error for this disk.
	mov	dx,CRMsg	;Display error-message suffix.
	call	I_Dspl
I_More	add	word [HDUnit],00101h  ;Bump BIOS unit & disk index.
	cmp	word [EDDFlag],08400h ;No EDD and all 4 units tested?
	je	I_AnyD		;Yes, see if we found any disks.
	dec	byte [HDCount]	;More BIOS disks to check?
	jnz	I_Next		;Yes, loop back and do next one.
I_AnyD	mov	bx,16		;Set up to scan for last unit.
I_ChkU	sub	bx,byte 4	;Any more active units to check?
	js	near I_RScn	;No, see if we should do a re-scan.
	cmp	byte [bx+Units-@],0FFh  ;Is this unit active?
	je	I_ChkU		;No, loop back and check next unit.
	add	bx,byte 4	;Post last-unit index in main driver.
	mov	[@LastU],bx
	call	I_Hook		;"Hook" Int 13h calls to our driver.
	popad			;Reload all 32-bit registers.
	push	ax		;Save all 16-bit registers.
	push	bx
	push	cx
	push	dx
	push	si
	push	di
	mov	ax,[IVDSLen]	;Get driver size & "success" flag.
	mov	dx,RPDON
	jmp	short I_Exit	;Go post "init" packet results & exit.
I_VErr	mov	dx,VEMsg	;VDS "lock" error!  Point to message.
I_Err	push	dx		;Save error message pointer.
	shr	byte [IVDSOfs],1 ;Was driver "locked" by VDS?
	jnc	I_XUnl		;No, get rid of our XMS memory.
	mov	di,IVDSLen	;Point to initialization VDS block.
	mov	ax,08104h	;"Unlock" this driver from memory.
	xor	dx,dx
	call	VDInit
I_XUnl	mov	cx,[XMHdl]	;Did we reserve any XMS memory?
	jcxz	I_LdMP		;No, display desired error message.
	mov	ah,00Dh		;Unlock our XMS memory buffer.
	mov	dx,cx
	push	dx
	call	XMCall
	mov	ah,00Ah		;Free our XMS memory buffer.
	pop	dx
	call	XMCall
	mov	ah,006h		;Do local-disable of "A20 line".
	call	XMCall
I_LdMP	pop	dx		;Reload error message pointer.
I_EOut	call	I_Dspl		;Display desired error message.
	popad			;Reload all 32-bit registers.
	push	ax		;Save all 16-bit registers.
	push	bx
	push	cx
	push	dx
	push	si
	push	di
I_Quit	mov	dx,Suffix	;Display message suffix.
	call	I_Dspl
I_BadP	xor	ax,ax		;Get "null" length & error flags.
	mov	dx,RPDON+RPERR
I_Exit	les	bx,[Packet]	;Post results in "init" packet.
	mov	[es:bx+RPSize],ax
	mov	[es:bx+RPSize+2],cs
	mov	[es:bx+RPStat],dx
	pop	di		;Reload all CPU registers and exit.
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	pop	es
	pop	ds
	popf
	retf
;
; Subroutine to "validate" an UltraDMA hard-disk.
;
I_ValD	mov	al,0ECh		;Issue "Identify Device" command.
	call	I_Cmd
	jnc	I_PIO		;If no error, get "identify" data.
I_AErr	mov	dx,AEMsg	;Absent or non-ATA!   Point to msg.
	stc			;Set carry flag (error!) and exit.
	ret
I_PIO	add	dx,byte (CDATA-CCMD)  ;Point to PIO data register.
	in	ax,dx		;Read I.D. bytes 0 and 1.
	xchg	ax,si		;Save "ATA/ATAPI" flag word.
	mov	cx,26		;Skip I.D. bytes 2-53.
I_Skp0	in	ax,dx
	loop	I_Skp0
	cld			;Ensure FORWARD "string" commands!
	push	ds		;Point to disk-name message.
	pop	es
	mov	di,DName
	mov	cl,20		;Read & swap disk name into message.
I_RdNm	in	ax,dx		;(I.D. bytes 54-93).
	xchg	ah,al
	stosw
	loop	I_RdNm
	mov	cl,6		;Skip I.D. bytes 94-105.
I_Skp1	in	ax,dx
	loop	I_Skp1
	in	ax,dx		;Read I.D. bytes 106 and 107.
	mov	bh,al		;Save "DMA valid" flag byte.
	mov	cl,34		;Skip I.D. bytes 108-175.
I_Skp2	in	ax,dx
	loop	I_Skp2
	in	ax,dx		;Read I.D. bytes 176 and 177.
	mov	bl,ah		;Save "UltraDMA selected" flag byte.
	mov	cl,167		;Skip remaining I.D. data.
I_Skp3	in	ax,dx
	loop	I_Skp3
	shl	si,1		;Is this an "ATA" hard-disk?
	jc	I_AErr		;No?  Exit & display message!
	test	bh,004h		;Are UltraDMA flag bits valid?
	jz	I_DErr		;No?  Exit & display message!
	mov	di,Modes	;Point to UltraDMA mode table.
	mov	al,'0'		;Initialize "current mode" value.
	mov	cl,002h		;Set rotating mode-check bit.
	cmp	bl,001h		;Will disk do UltraDMA mode 0?
	jae	I_NxtM		;Yes, find its best UltraDMA mode.
I_DErr	mov	dx,DEMsg	;Not an UltraDMA disk!   Point to msg.
I_SErr	stc			;Set carry flag (error!) and exit.
	ret
I_NxtM	mov	[CurMode],al	;Update "current mode" value in message.
	cmp	al,0FFh		;Are we at the desired MAXIMUM "mode"?
@MaxMod	equ	$-1		;(Max. UltraDMA "mode", set by startup).
	jae	I_GotM		;Yes, use current mode.
	cmp	bl,cl		;Will disk do next UltraDMA mode?
	jb	I_GotM		;No, use current mode.
	inc	ax		;Set up for next UltraDMA mode.
	add	di,byte 4
	shl	cl,1		;More UltraDMA modes to check?
	jnz	I_NxtM		;Yes, loop back.
I_GotM	push	ax		;Save "current mode" value.
	mov	eax,[di]	;Set UltraDMA mode in set-mode message.
	mov	[DMode],eax
	inc	dx		;Set mode-select subcode.
	mov	al,SETM
	out	dx,al
	pop	ax		;Set desired UltraDMA mode.
	add	al,010h
	inc	dx
	out	dx,al
	mov	al,SETF		;Issue set-features command to disk.
	call	I_Cmd
	mov	dx,SEMsg	;Point to "Set-mode" error message.
	jc	I_SErr		;If error, set carry flag and exit.
	mov	di,DNEnd	;Point to end of disk name.
I_NxtN	cmp	di,DName	;Are we at the disk-name start?
	je	I_Unnm		;Yes, disk name is all spaces!
	cmp	byte [di-1],' '	;Is preceding byte a space?
	jne	I_TrmN		;No, terminate disk name message.
	dec	di		;Decrement disk name pointer.
	jmp	short I_NxtN	;Go see if next byte is a space.
I_Unnm	mov	dword [di],"unna"   ;Set "unnamed" as disk name.
	mov	dword [di+4],"med "
	add	di,byte 7
I_TrmN	mov	byte [di],'$'	;Set message terminator after name.
	sar	byte [NoRdSw],1	;Should we omit the "read tests"?
	jc	near I_DspNm	;Yes, just display disk name & mode.
	mov	byte [XMSMov],0C3h  ;Disable XMS moves with "ret" cmd.
	call	I_Hook		;"Hook" this driver into Int 13h.
	call	I_RdT		;Do initial read for synchronization.
	jnc	I_SetC		;If O.K., set up 4-pass read test.
I_RdEr	push	ax		;Read error!  Save driver return code.
	call	I_RVec		;Restore original INT 13h vector.
	pop	ax		;Reload driver return code.
I_Flunk	push	ax		;Display "FAILED read test" message.
	mov	dx,TEMsg
	call	I_Dspl
	pop	ax
	mov	di,(MsgTbl-3-@)	;Point to our error-message table.
I_EScn	add	di,byte 3	;Bump to next message code/address.
	mov	dx,[di+1]	;Set message address in DX-register.
	cmp	ah,[di]		;Driver return code = table code?
	je	I_RBye		;Yes, this is the message we want!
	cmp	di,(MsgEnd-@)	;Is driver return code unrecognized?
	jb	I_EScn		;No, loop back and check next entry.
I_RCMs	mov	cx,2		;Set return code in error message.
	mov	si,RCode
	call	I_Hex
	mov	dx,RCMsg	;Point to "Return code" message.
I_RBye	stc			;Set carry flag (error!) and exit.
	ret
I_SetC	xor	dx,dx		;Clear read counter.
	xor	si,si		;Point to low-memory BIOS timer.
	mov	es,si
	mov	si,BIOSTMR
	mov	cl,[es:si]	;Load current timer tick count LSB.
I_RDly	cmp	cl,[es:si]	;Has next timer "tick" occurred?
	je	I_RDly		;No, keep waiting.
	add	cl,1+4		;Yes, update & prepare for 4 passes.
I_RNxt	inc	dx		; Yes, count reads up
	push	es		;Save timer/counter registers.
	push	si
	push	dx
	push	cx
	call	I_RdT		;Read RBYTES into XMS buffer.
	pop	cx		;Reload timer/counter registers.
	pop	dx
	pop	si
	pop	es
	jc	I_RdEr		;Read error?  Display message & exit!
	cmp	cl,[es:si]	;Next timer interrupt arrived?
	jne	I_RNxt		;No, read once more.
	shr	dx,2		;Save average rate for 4 passes.
	mov	[Bucket],dx
	mov	al,RC_SC	;Set "read comparison" parameters.
	mov	[RCSects],al
	mov	cx,((RC_CYL*256)+RC_SEC)
	mov	[RCSecNo],cl
	mov	dh,RC_HD
	call	I_RdT1		;Input "read comparison" data to XMS.
	jc	I_RdEr		;Read error?  Display message & exit!
	call	I_RVec		;Restore original Int 13h vector.
	cld			;Set up to initialize XMS block.
	push	ds
	pop	es
	mov	di,(VDSLn-@)
	mov	eax,512		;Set 1-sector XMS block length.
	stosd
	mov	ax,[XMHdl]	;Reset XMS "source address".
	stosw
	mov	eax,[XMOfs]
	stosd
	xor	ax,ax		;Reset XMS "destination address".
	stosw
	mov	ax,(RBuff+3-@)
	stosw
	mov	ax,cs
	stosw
I_RCIn	mov	al,1		;Set BIOS input sector count of 1.
	mov	bx,(RBuff+3-@)	;Set BIOS input buffer address.
	mov	ch,RC_CYL	;Set BIOS input disk address.
	mov	cl,[RCSecNo]
	mov	dh,RC_HD
	call	I_RdT2		;Have BIOS read next data sector.
	jnc	I_RCkS		;Any error during BIOS read?
	push	ax		;Yes?  Save BIOS return code.
	mov	dx,BEMsg	;Display "failed BIOS read" message.
	call	I_Dspl
	pop	ax		;Reload BIOS return code.
	jmp	I_RCMs		;Go set up return-code msg. and exit.
I_RCkS	call	I_Roll		;"Roll" & save BIOS sector checksum.
	push	ax
	mov	si,(VDSLn-@)	;Get next data sector from XMS memory.
	call	XMGo
	pop	dx		;Reload BIOS sector checksum.
	or	ax,ax		;Any errors during XMS move?
I_RCEr	jnz	I_Flunk		;Yes?  Display DRIVER error and exit!
	call	I_Roll		;"Roll" checksum for XMS input sector.
	cmp	ax,dx		;BIOS and driver input "mismatch"?
	mov	ah,0FEh		;(Load "mismatch" return code if so).
	jne	I_RCEr		;Yes?  Display DRIVER error and exit!
	add	word [XMSSA+1],byte 2 ;Bump XMS source to next sector.
	inc	byte [RCSecNo]	;Update BIOS input sector number
	dec	byte [RCSects]	;More BIOS data to input?
	jnz	I_RCIn		;Yes, go read next sector.
I_DspNm	mov	dx,DNMsg	;Display disk "name" message.
	call	I_Dspl
	mov	dx,MSMsg	;Display "Set to mode" message.
	call	I_Dspl
	mov	dx,CRMsg+1	;Point to CR/LF message.
	sar	byte [NoRdSw],1	;Were the "read tests" omitted?
	jc	I_Dspl		;Yes, display CR/LF message and exit.
	mov	ax,[Bucket]	;Reload average read rate.
	mov	di,(RRate+4-@)	;Point to "read test" message digits.
	mov	byte [di],'0'	;Initialize "read test" digits to 0.
	or	ax,ax		;Did the disk read NOTHING?
	jz	I_Rate		;Yes, display "read test = 0".
	mov	cx,10		;Set divisor for ASCII conversion.
I_ItoA	xor	dx,dx		;Divide remaining read rate by 10.
	div	cx
	xchg	dx,ax		;Save quotient & get remainder in AX.
	add	al,'0'		;Convert next digit to ASCII.
	mov	[di],al		;Store digit in "read test" message. 
	dec	di
	xchg	dx,ax		;Reload quotient.
	or	ax,ax		;Is quotient zero?
	jnz	I_ItoA		;No, loop back & get next digit.
	inc	di		;Point to high-order rate digit.
I_Rate	push	di		;Save pointer to read-rate digits.
	mov	dx,RTMsg	;Display "Read Test" message.
	call	I_Dspl
	pop	dx		;Display read rate for this disk.
I_Dspl	mov	ah,009h
	call	I_Int21
	clc			;Clear carry (no errors!) and exit.
	ret
;
; Subroutine to do all "read test" inputs.
;
I_RdT	mov	al,RS_SC	;"Read speed" -- set sector count.
	mov	cx,1		;Set cylinder 0, sector 1.
	xor	dh,dh		;Set head 0.
I_RdT1	mov	bx,(RBuff+3-@)	;Use "odd" offset to avoid VDS use.
I_RdT2	push	ds		;Point ES-register to our data.
	pop	es
	mov	ah,2		;Set "CHS read" request code.
	mov	dl,[HDUnit]	;Set desired unit number.
I_Int13	int	013h		;Issue desired Int 13h request.
	push	cs		;Reload our DS-register.
	pop	ds
	ret			;Exit.
;
; Subroutine to issue disk initialization commands.
;
I_Cmd	mov	dx,CCMD		;Issue desired init command.
	xor	dl,0
@HDOffs	equ	$-1
	out	dx,al
	xor	si,si		;Point to low-memory BIOS timer.
	mov	es,si
	mov	si,BIOSTMR
	mov	cl,RDYTO	;Set I-O timeout limit in CL-reg.
	add	cl,[es:si]
I_CmdW	cmp	cl,[es:si]	;Has our command timed out?
	je	I_CmdE		;Yes, set CPU carry flag & exit.
	in	al,dx		;Get IDE controller status.
	test	al,BSY+RDY	;Controller or disk still busy?
	jle	I_CmdW		;Yes, loop back and check again.
	test	al,ERR		;Did command cause any errors?
	jz	I_CmdX		;No, leave carry flag off & exit.
I_CmdE	stc			;Error!  Set CPU carry flag.
I_CmdX	ret			;Exit.
;
; Subroutine to "hook" this driver into the Int 13h chain.
;
I_Hook	mov	ax,03513h	;Get current Int 13h vector.
	call	I_Int21
	push	es		;Save previous Int 13h vector.
	push	bx
	pop	dword [@PrvI13]
	mov	ax,02513h	;"Hook" this driver into Int 13h.
	mov	dx,(Entry-@)
I_Int21	int	021h
	push	cs		;Reload our DS-register.
	pop	ds
	ret			;Exit.
;
; Subroutine to restore the Int 13h vector after initialization reads.
;
I_RVec	mov	ax,02513h	;Set back the old Int 13h vector.
	lds	dx,[@PrvI13]
	call	I_Int21
	mov	al,00Eh		;Re-enable XMS moves with "push cs".
	mov	[XMSMov],al
	ret			;Exit.
;
; Subroutine to "roll" the checksum for data input to "RBuff".
;
I_Roll	xor	ax,ax		;Reset checksum "bucket".
	mov	cx,256		;"Roll" 256 words of data.
	mov	si,(RBuff+3-@)	;Point to read-test buffer.
I_RolA	add	ax,[si]		;Add next word to checksum.
	adc	ax,0		;If carryout, add it back.
	inc	si		;Point to next data word.
	inc	si
	loop	I_RolA		;If more words to go, loop back.
	ret			;Exit -- checksum is in AX-reg.
;
; Subroutine to convert hex digits to ASCII for output messages.
;   At entry the AX-reg. has the data, the CX-reg. has the digit
;   count, and the SI-reg. has the message pointer.
;
I_Hex4	mov	cx,4		;Set normal 4-digit count.
I_Hex	rol	ax,4		;Get next hex digit in low-order.
	push	ax		;Save remaining digits.
	and	al,00Fh		;Mask off next hex digit.
	cmp	al,009h		;Is digit 0-9?
	jbe	I_HexA		;Yes, convert to ASCII.
	add	al,007h		;Add A-F offset.
I_HexA	add	al,030h		;Convert digit to ASCII.
	mov	[si],al		;Set next ASCII digit in message.
	inc	si		;Bump message address.
	pop	ax		;Reload remaining digits.
	loop	I_Hex		;If more digits to go, loop back.
	ret			;Exit.
;
; Initialization Buffers.   The 516 bytes starting at "RBuff" must be
;   "disposable" and used once-only.   After, this space will be used
;   as our "read-test" input buffer.
;
	align	4
EDDBuff dd	30		;Start of 30-byte EDD input buffer.
RBuff	equ	$		;Start of "read-test" input buffer.
;
; "Strategy" routine -- At entry, ES:BX points to the DOS init request
;   packet, whose address is saved for processing below.
;
Strat	mov	[cs:Packet],bx	;Save DOS request-packet address.
	mov	[cs:Packet+2],es
	retf			;Exit & await DOS "Device Interrupt".
;
; "Device-Interrupt" routine -- This routine initializes the driver.
;
DevInt	pushf			;Entry -- save CPU flags.
	push	ds		;Save all 16-bit CPU registers.
	push	es
	push	ax
	push	bx
	push	cx
	push	dx
	push	si
	push	di
	push	cs		;Set our DS-register.
	pop	ds
	les	bx,[Packet]	;Point to DOS request packet.
	xor	ax,ax		;Get a zero for below.
	cmp	[es:bx+RPOp],al	;Is this an "Init" packet?
	je	I_TTL		;Yes, display our title message.
	jmp	I_BadP		;Go post errors and exit quick!
I_TTL	mov	dx,UDMsg	;Display driver "title" message.
	call	I_Dspl
	pushf			;80386 test -- save CPU flags.
	push	sp		;See if CPU is an 80286 or newer.
	pop	ax
	cmp	ax,sp		;(80286+ push SP, then decrement it)
	jne	I_Junk		;CPU is 8086/80186 -- cannot USE it!
	push	07000h		;80286 and up -- try to set NT|IOPL.
	popf
	pushf
	pop	ax
	test	ah,070h		;Did any NT|IOPL bits get set?
	jnz	I_386		;Yes, CPU is at least an 80386.
I_Junk	popf			;Reload starting CPU flags.
	mov	dx,PRMsg	;Display "No 80386 CPU" message.
	call	I_Dspl
	jmp	I_Quit		;Go display suffix and exit quick!
I_386	popf			;Reload starting CPU flags.
	pop	di		;Reload all 16-bit registers.
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	pushad			;Save all 32-bit registers.
	mov	ax,0B101h
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	cmp	edx,"PCI "	;Is PCI BIOS V2.0C or newer?
	mov	dx,PEMsg	;(Get error message pointer if not).
	jne	I_PCEr		;No?  Go display message and exit!
	mov	si,LBA2		;Point to interface byte table.
	cld			;Ensure FORWARD "string" commands!
I_GetD	mov	ecx,000010100h	;We want class 1 storage, subclass 1 IDE.
	lodsb			;Get next "valid" PCI interface byte.
	mov	cl,al
	push	si		;Search for our PCI class code.
	mov	ax,0B103h	;(Returns bus/device/function in BX).
	xor	si,si
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	pop	si
	jnc	I_GotD		;Found our boy!  Go process it.
	cmp	si,LBA		;More interface bytes to try?
	jb	I_GetD		;Yes, go try next one.
	mov	dx,NEMsg	;BAAAD News!  Point to error message.
	jmp	short I_PCEr	;Go display error message and exit.
I_GotD	push	bx		;Save PCI bus/device/function.
	mov	ax,0B108h	;Get low-order PCI command byte.
	mov	di,4
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	pop	bx		;Reload PCI bus/device/function.
	not	cl		;Mask Bus-Master and I-O Space bits.
	and	cl,005h		;Is this how our controller is set up?
	jz	I_BaseA		;Yes, get our PCI base address.
	mov	dx,MEMsg	;Cannot USE it -- point to error msg.
I_PCEr	jmp	I_EOut		;Go display "INVALID" message & exit!
I_BaseA push	bx		;Get PCI base address (register 4).
	mov	ax,0B109h
	mov	di,32
	int	01Ah
	pop	bx
	xchg	ax,cx		;Post run-time DMA command address.
	and	al,0FCh
	mov	[DMAAd],ax
	mov	[@DMALo1],al	;Set lower DMA command-address bytes.
	add	[@DMALo2],al
	mov	si,DspAddr	; Set hex address in display message
	call	I_Hex4
	push	bx		;Get Vendor and Device I.D.
	mov	ax,0B10Ah
	xor	di,di
	int	01Ah
	pop	bx
	xchg	eax,ecx		;Save Vendor and Device I.D.
	push	eax
	mov	si,DspVID	;Set Vendor I.D. in display message.
	call	I_Hex4
	shr	eax,16		;Set Device I.D. in display message.
	mov	si,DspDID
	call	I_Hex4
	mov	ah,bh		;Set PCI bus number in message.
	mov	cl,2
	mov	si,DspBus
	call	I_Hex
	mov	ah,bl		;Set PCI device number in message.
	shr	ah,3
	mov	cl,2
	mov	si,DspDev
	call	I_Hex
	and	bl,7		;Set PCI function number in message.
	or	bl,30h
	mov	[DspFnc],bl
	mov	dx,PCMsg	;Display initial controller data.
	call	I_Dspl
	cld			;Set up controller-name scan.
	push	cs
	pop	es
	pop	eax		;Reload Vendor and Device I.D.
	ror	eax,16		;"Swap" Vendor I.D. to high-order.
	mov	cx,(dt_end-devtab)/4 ;See if I.D.s are in our table.
	mov	di,devtab
	repne	scasd		;Do we know what our controller is?
	jnz	I_UnknC		;No, display Vendor/Device numbers.
	mov	dx,di		;Display the name of our controller.
	call	I_Dspl
	jmp	short I_TermC	;Go display controller bus data.
I_UnknC mov	dx,DspUnkn	;Display Vendor/Device I.D. numbers.
	call	I_Dspl
I_TermC mov	dx,PCMsg2	;Display all controller bus data.
	call	I_Dspl
	mov	ax,04300h	;Inquire about an XMS manager.
	int	02Fh
	push	cs		;Reload our DS-register.
	pop	ds
	mov	dx,NXMsg	;Point to "No XMS manager" message.
	cmp	al,080h		;Is an XMS manager installed?
	jne	I_XErr		;No, display message & disable XMS.
	mov	ax,04310h	;Get XMS manager "entry" address.
	int	02Fh
	push	cs		;Reload our DS-register.
	pop	ds
	push	es		;Save XMS manager "entry" address.
	push	bx
	pop	dword [@XEntry]
	mov	ah,009h		;Ask XMS manager for 128K of memory.
	mov	dx,128
	call	XMCall
	jc	I_XMErr		;If error, display msg. & disable XMS.
	mov	[XMHdl],dx	;Save XMS buffer handle number.
	mov	ah,00Ch		;"Lock" our XMS memory (direct call,
	call	far [@XEntry]	;  as our subroutine ZEROS BX-reg.!).
	push	cs		;Reload our DS-register.
	pop	ds
	dec	ax		;Did XMS memory get locked?
	jnz	I_RidX		;No?  Unuseable -- get RID of it!
	shl	edx,16		;Get unaligned XMS buffer address.
	mov	dx,bx
	mov	eax,edx		;Find 1st 64K boundary after start.
	add	eax,65536
	xor	ax,ax
	mov	[@XBufAd],eax	;Set final XMS 32-bit buffer address.
	sub	eax,edx		;Initialize "offset" into XMS memory.
	mov	[XMOfs],eax
	mov	ah,005h		;Local-enable "A20 line" FOREVER!
	call	XMCall		;("A20" CANNOT turn off during DMA!).
	jnc	I_Stop		;If no error, ensure DMA is stopped.
	mov	ah,00Dh		;Unuseable!  Unlock our XMS memory.
	mov	dx,[XMHdl]
	call	XMCall
I_RidX	xor	dx,dx		;Load & reset our XMS buffer handle.
	xchg	dx,[XMHdl]
	mov	ah,00Ah		;Free our XMS memory.
	call	XMCall
I_XMErr mov	dx,XEMsg	;Point to "XMS setup error" message.
I_XErr	call	I_Dspl		;Display desired XMS error message.
	mov	dx,NBMsg	;Display "no buffered I-O" message.
	call	I_Dspl
	mov	byte [NoRdSw],0FFh  ;Disable all "read tests".
	mov	ax,(Pass-GoToBuf-3) ;Reject buffered I-O using a
	mov	[GoToBuf+1],ax	    ;  "dirty-nasty" code change!
	mov	ax,(NoXMEnd-@)	;Reduce resident driver size.
	mov	[IVDSLen],ax
I_Stop	mov	dx,[DMAAd]	;Ensure any previous DMA is stopped.
	in	al,dx		;(See notes in "DoDMA" logic above).
	and	al,0FEh
	out	dx,al
	add	dx,byte 8	;Stop secondary-channel DMA, also!
	in	al,dx
	and	al,0FEh
	out	dx,al
	xor	eax,eax		;Zero EAX-reg. for 20-bit addressing.
	mov	es,ax		;Point ES-reg. to low memory.
	or	al,[es:HDISKS]	;Did BIOS find any hard-disks?
	jz	near I_RScn	;No?  Display "No disk" and exit!
	mov	[@BIOSHD],al	;Set BIOS hard-disk count below.
	mov	ax,cs		;Set our code segment in VDS block.
	mov	[IVDSSeg],ax
	shl	eax,4		;Get 20-bit driver virtual address.
	cli			;Avoid interrupts during VDS tests.
	test	byte [es:VDSFLAG],020h ;Are "VDS services" active?
	jz	I_SetA		;No, set 20-bit virtual addresses.
	mov	di,IVDSLen	;Point to initialization VDS block.
	mov	ax,08103h	;"Lock" this driver into memory.
	mov	dx,0000Ch
	call	VDInit
	jc	near I_VErr	;Error?  Display error msg. & exit!
	inc	byte [IVDSOfs]	;Set initialization VDS "lock" flag.
	mov	eax,[IVDSAdr]	;Get 32-bit starting driver address.
I_SetA	sti			;Re-enable CPU interrupts.
	add	[PRDAd],eax	;Set relocated 32-bit PRD address.
	mov	ah,041h		;See if we have an EDD BIOS.
	mov	bx,055AAh
	mov	dl,080h
	call	I_Int13
	jc	I_Srch		;No, search for disks without EDD.
	cmp	bx,0AA55h	;Did BIOS "reverse" our entry code?
	jne	I_Srch		;No, search for disks without EDD.
	xchg	ax,cx		;Set "EDD BIOS present" flag.
	and	al,004h
	mov	[EDDFlag],al
I_Srch	jmp	I_Scan		;Go search for UltraDMA disks to use.
;
; PCI Vendor/Device Table.   Format:  dd <id>, db <dword-aligned name>
;
	align	4
devtab:
%include "udma.pci"
dt_end	equ	$
