/*
 * CMCOMPRE.C
 *
 *  This file is part of DOSZIP
 *  Copyright (c) 1996 Hjort Nidudsson.
 */

#include <dzmain.h>
#include <errno.h>
#include <arch.h>
#include <string.h>
#include <io.h>
#include <progress.h>

#define MAXPACKERS	16
enum PackerConfig {
	P00_ADD,
	P01_ADD_PATH,
	P02_EXTRACT,
	P03_EXTRACT_PATH,
	P04_DELETE,
	P05_MAX,
	P06_MIN,
	P07_PASSWORD_PREFIX,
	P08_OUTPATH_PREFIX,
	P09_RESPONSE_PREFIX,
};

extern char *envtemp;
int recursive(char *, char *, char *);
int cpy_setselectedcnt(void);

const char INI_Packer  	[] = "Packer";
const char INI_00      	[] = "00";
const char cp_newline  	[] = "\r\n";
const char cp_ziplst   	[] = "ziplst";

#define MAXZCMD		64
#define MAXZSWITCH	32

typedef struct {
	char 	section[32];
	char	command[128];
	char	A00[MAXZCMD];	 /* Add a file to a compressed file        */
	char	A01[MAXZCMD];	 /* - and store the file's full pathname   */
	char	E02[MAXZCMD];    /* Extract a file from a compressed file  */
	char	E03[MAXZCMD];	 /* - and use the file's full pathname     */
	char	D04[MAXZCMD];	 /* Delete a file from a compressed file   */
	char	M05[MAXZSWITCH]; /* Create the smallest compressed file    */
	char	M06[MAXZSWITCH]; /* Select the best method supported       */
	char	P07[MAXZSWITCH]; /* Password switch                        */
	char	O08[MAXZSWITCH]; /* Char(s) that precede the target path   */
	char	R09[MAXZSWITCH]; /* Char(s) that precede the response file */
      } COMP;

static COMP *cpi;
static DOBJ *dialog;

enum {	ID1_OUTDIR,
	ID1_INCDIR,
	ID1_EDITCMD,
	ID1_COMPRESS,
	ID1_OPTIONS,
	ID1_METHOD,
	ID1_FILTER,
	ID1_CANCEL	};

enum {	ID2_OUTDIR,
	ID2_INCDIR,
	ID2_EDITCMD,
	ID2_COMPRESS,
	ID2_OPTIONS,
	ID2_METHOD,
	ID2_CANCEL	};


static int response_open(void);
static int response_make(void);
static int response_addfile(const char *);
static int response_dofile(FBLK *);
static int response_scan_dofile(const char *, wfblk *);

static int ini_ermsg(const char *__section, const char *__entry);
static int ini_packerinfo(void);
static int ini_initcommand(void);
static int method_print(void);
static int method_select(void);
static int event_compress(void);
static int event_filter(void);

/* Make Response File */

static long fcount;

int response_open(void)
{
	char b[WMAXPATH];

	strfcat(b, envtemp, cp_ziplst);
	arc_outhnd = osopen(b, 0, M_WRONLY, A_CREATE | A_TRUNC);
	if (arc_outhnd == -1)
		return stderror(b, EMERRNO, 0);
	cflag |= _C_DELZIPLST;
	return 1;
}

int response_addfile(const char *name)
{
	int size;

	fcount++;
	strcpy(__srcfile, name);
	strcat(__srcfile, cp_newline);
	size = strlen(__srcfile);
	return (oswrite(arc_outhnd, __srcfile, size) != size);
}

int response_dofile(FBLK *b)
{
	char fb[256];

	if (ISSUBDIR(b->flag)) {
		if (progress_set(NULL, b->name, 0))
			return _ER_USERABORT;
		if (zflag & _C_ZINCDIR) {
			strfcat(fb, __srcpath, b->name);
			return scansub(fb, cp_stdmask, 0);
		}
	} else {
		if (progress_set(NULL, b->name, 1))
			return _ER_USERABORT;
		if (test_filter(b->name, b->size, b->flag, b->date))
			return response_addfile(b->name);
	}
	return 0;
}

int response_scan_dofile(const char *p, wfblk *w)
{
	char *q;

	if (progress_set(NULL, w->name, 1))
		return _ER_USERABORT;
	if (!test_filter(w->name, w->size_low, w->attrib, w->date_modified))
		return 0;
	strfcat(__srcfile, p, w->name);
	q = __srcfile + strlen(__srcpath);
	if (*q == '\\')
		q++;
	return response_addfile(q);
}

int response_make(void)
{
	FBLK *b;
	int result;

	fcount = 0L;
	b = cpanel_findfirst();
	fp_fileblock = response_scan_dofile;
	fp_directory = scan_files;
	progress_open(cpi->section, NULL);
	if (ISSELECTED(b->flag)) {
		do {
			if ((result = response_dofile(b)) != 0)
				break;
			b->flag &= ~(_A_SELECTED);
		} while ((b = panel_ffselected(cpanel)) != NULL);
	} else {
		result = response_dofile(b);
	}
	progress_close();
	return result;
}

/* Read Config File --> [Packer] */

int ini_ermsg(const char *section, const char *entry)
{
	zflag &= ~_C_ZPACKERID;
	return ermsg(cp_badentry, cp_badentrymsg, section, entry);
}

int ini_packerinfo(void)
{
	char *p,*s;
	char entry[16];

	sprintf(entry, "%02u", zflag & _C_ZPACKERID);
	if ((p = inientry(INI_Packer, entry)) == NULL)
		return ini_ermsg(INI_Packer, entry);
	s = cpi->section;
	strncpy(s, p, 32);
	strcpy(entry, INI_00);
	if (*cpanel->flag & _W_LONGNAME)
		entry[0]++;
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->A00, p, MAXZCMD);
	entry[1]++; /* 01 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->A01, p, MAXZCMD);
	entry[1]++; /* 02 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->E02, p, MAXZCMD);
	entry[1]++; /* 03 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->E03, p, MAXZCMD);
	entry[1]++; /* 04 */
/*
 *	if ((p = inientry(s, entry)) != NULL)
 *		strncpy(cpi->D04, p, MAXZCMD);
 */
	entry[1]++; /* 05 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->M05, p, MAXZSWITCH);
	entry[1]++; /* 06 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->M06, p, MAXZSWITCH);
	entry[1]++; /* 07 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->P07, p, MAXZSWITCH);
	entry[1]++; /* 08 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->O08, p, MAXZSWITCH);
	entry[1]++; /* 09 */
	if ((p = inientry(s, entry)) != NULL)
		strncpy(cpi->R09, p, MAXZSWITCH);
	return 1;
}

/* Init command-string */

#define PWLEN	80

int ini_initcommand(void)
{
	int x;
	char *p;
	char *q;
	char password[PWLEN];

	q = cpi->command;
	if (zflag & _C_ZDECOMP) {
		if (zflag & _C_ZINCDIR)
			strcpy(q, cpi->E03);
		else
			strcpy(q, cpi->E02);
	} else {
		if (zflag & _C_ZINCDIR)
			strcpy(q, cpi->A01);
		else
			strcpy(q, cpi->A00);
	}
	if (q[0] == 0)
		return ini_ermsg(cpi->section, "00..03");
	strcat(q, " ");
	if ((zflag & _C_ZDECOMP) == 0) {
		p = NULL;
		x = (zflag & _C_ZBEST);
		if (x == _C_ZSMALLEST)
			p = cpi->M05;
		else if (x == _C_ZFASTEST)
			p = cpi->M06;
		if (p != NULL && *p) {
			strcat(q, p);
			strcat(q, " ");
		}
	}
	if (zflag & _C_ZPASSWORD) {
		if (cpi->P07[0] == 0)
			return ini_ermsg(cpi->section, cp_badpasswordsw);
		password[0] = 0;
		x = tgetline(cp_enterpassword, password, 32, PWLEN);
		if (x == 0 || password[0] == 0)
			return confirm_continue(cp_skippassword);
		strcat(q, cpi->P07);
		strcat(q, password);
		strcat(q, " ");
	}
	return 1;
}

/* Method Selection Dialog box */

int method_print(void)
{
	int x,y,z;

	z = (zflag & _C_ZDECOMP);
	y = dialog->rc.y + 2;
	x = dialog->rc.x + 14;
	if (z) {
		scputc(x, y, 0, 53, ' ');
		scputf(x, y, 0, 53, cp_compressusing,
			cpanel_findfirst()->name, cpi->section);
	} else {
		x += 4;
		scputc(x, y, 0, 32, ' ');
		scputf(x, y, 0, 32, cp_compressto, cpi->section);
	}
	return z;
}

int method_select(void)
{
	int x;
	DOBJ *d;
	TOBJ *o;
	RECT rc;
	char *p;
	char entry[32];

	memset(&cpi->section, 0, sizeof(COMP) - 10);
	if ((d = rsopen(IDD_Archive)) == NULL)
		return _C_ESCAPE;
	o = d->object;
	d->index = (zflag & _C_ZPACKERID);
	strcpy(entry, INI_00);
	for (x = 0; x < MAXPACKERS; x++) {
		if ((p = inientry(INI_Packer, entry)) == NULL)
			break;
		strcpy(o[x].data, ".");
		strcat(o[x].data, p);
		wcputs(rcbprc(d->wp + 2, o[x].rc, d->rc.col), 0, 13, o[x].data);
		o[x].flag &= ~_O_STATE;
		if (entry[1] == '9') {
			entry[1] = '0';
			entry[0]++;
		} else {
			entry[1]++;
		}
	}
	d->count = x;
	if (d->index >= x)
		d->index = 0;
	d->rc.row = x + 2;
	rc = d->rc;
	rc.x = 0;
	rc.y = 0;
	rcframe(rc, d->wp, 17, 0);
	if (x == 0 || (x = tdevent(d)) == 0) {
		tdclose(d);
		return _C_NORMAL;
	}
	tdclose(d);
	zflag &= ~_C_ZPACKERID;
	zflag |= (x - 1);
	if (ini_packerinfo() == 0)
		return _C_ESCAPE;
	method_print();
	while (mousep());
	return _C_NORMAL;
}

/* Compress Dialog box */

int event_compress(void)
{
	char *p,*q;
	char tmp[WMAXPATH];
	int result;

	if (envtemp == NULL)
		return stderror(cp_missingTEMP, EMUSER, _C_ESCAPE);
	if (ini_packerinfo() == 0)
		return _C_NORMAL;
	if (ini_initcommand() == 0)
		return _C_NORMAL;
	if (response_open() == 0)
		return _C_NORMAL;

	q = cpi->command;
	p = dialog->object[0].data;
	strcat(q, p);
	tmp[0] = cpi->R09[0];
	strfcat(&tmp[1], envtemp, cp_ziplst);
	strcat(q, " ");
	if (tmp[0])
		strcat(q, tmp);
	else
		strcat(q, tmp + 1);
	result = response_make();
	close(arc_outhnd);
	if (result)
		return _C_NORMAL;
	if (fcount == 0L)
		return ermsg(cp_compress, cp_nothingtodo);
	if (dialog->object[ID1_EDITCMD].flag & _O_FLAGB) {
		if (!tgetline(cp_compress, q, 40, 128))
			return _C_NORMAL;
	}
	return ex_command(q);
}

/* Operation Filters */

int event_filter(void)
{
	int x,y;

	cmfilter();
	x = dialog->rc.x + 44;
	y = dialog->rc.y + 7;
	if (filtp == NULL)
		scputc(x, y, 0, 1, ' ');
	else
		scputc(x, y, 0, 1, 7);
	return _C_NORMAL;
}

int cmcompress(void)
{
	int 	ifs;
	DOBJ *	d;
	TOBJ *	o;
	FBLK *	b;
	char *	p;
	panel *	t;
	COMP 	zip;

	if (!(b = cpanel_findfirst()))
		return 0;
	if ((p = cpanel_gettarget()) == NULL)
		return stderror(cp_needpanel, EMUSER, 0);
	t = &panela;
	if (cpanel == t)
		t = &panelb;
	if (*t->flag & _W_ARCHIVE || ISARCHIVE(b->flag))
		return stderror(cp_linkerror, EMUSER, 0);
	strcpy(__outpath, wlongpath(p, NULL));
	strcpy(__srcpath, wlongpath(cpanel->ide->path, NULL));
	if (stricmp(__srcpath, __outpath) != 0 &&
	    recursive(b->name, __srcpath, __outpath))
		return stderror(cp_recursive, EMUSER, 0);
	if ((d = rsopen(IDD_Compress)) == NULL)
		return 0;
	zflag &= ~_C_ZDECOMP;
	cpi = &zip;
	memset(cpi, 0, sizeof(COMP));
	filtp = NULL;
	dialog = d;
	o = d->object;
	o[ID1_OPTIONS].proc = cmcompression;
	o[ID1_METHOD].proc = method_select;
	o[ID1_FILTER].proc = event_filter;
	strfcat(o[ID1_OUTDIR].data, __outpath, "default");
	if (zflag & _C_ZINCDIR)
		o[ID1_INCDIR].flag |= _O_FLAGB;
	tdinit(d);
	tdshow(d);
	if (ini_packerinfo() == 0) {
		tdclose(d);
		return 0;
	}
	method_print();
	if (rsevent(IDD_Compress, d)) {
		zflag &= ~_C_ZINCDIR;
		if (o[ID1_INCDIR].flag & _O_FLAGB)
			zflag |= _C_ZINCDIR;
		ifs = _ifsmgr;
		if ((*cpanel->flag & _W_LONGNAME) == 0)
			_ifsmgr = 0;
		event_compress();
		_ifsmgr = ifs;
		tdclose(d);
		return 1;
	}
	tdclose(d);
	return 0;
}

/* Decompress Dialog box */

int cmdecompress(void)
{
	int 	x,y;
	DOBJ *	d;
	TOBJ *	o;
	FBLK *	b;
	COMP 	zip;
	char *	p;
	char *	q;
	char 	entry[32];
	char 	section[32];

	if (!(b = cpanel_findfirst()))
		return 0;
	if (ISSUBDIR(b->flag))
		return 0;
	if (ISARCHIVE(b->flag))
		return ermsg(NULL, cp_linkerror);
	if (ISSELECTED(b->flag))
		return notsup();

	if ((p = strrchr(b->name, '.')) != NULL) {
		strncpy(section, p + 1, 30);
		strupr(section);
		strcpy(entry, INI_00);
		for (x = 0; x < MAXPACKERS; x++) {
			if ((p = inientry(INI_Packer, entry)) == NULL)
				break;
			if (strnicmp(section, p, 12) == 0) {
				zflag &= ~_C_ZPACKERID;
				zflag |= x;
				break;
			}
			entry[1]++;
			if (entry[1] > '9') {
				entry[0]++;
				entry[1] = '0';
			}
		}
	}

	if ((p = cpanel_gettarget()) == NULL)
		return stderror(cp_needpanel, EMUSER, 0);
	zflag |= _C_ZDECOMP;
	cpi = &zip;
	memset(cpi, 0, sizeof(COMP));
	if ((d = rsopen(IDD_Decompress)) == NULL)
		return 0;
	dialog = d;
	o = d->object;
	strcpy(o[ID2_OUTDIR].data, p);
	if (zflag & _C_ZINCDIR)
		o[ID2_INCDIR].flag |= _O_FLAGB;
	o[ID2_OPTIONS].proc = cmcompression;
	o[ID2_METHOD].proc = method_select;
	tdinit(d);
	o[ID2_OUTDIR].flag |= _O_STATE;
	tdshow(d);
	if (ini_packerinfo() == 0) {
		tdclose(d);
		return 0;
	}
	method_print();
	x = rsevent(IDD_Decompress, d);
	y = (o[ID2_INCDIR].flag & _O_FLAGB);
	tdclose(d);
	if (x == 0)
		return 0;
	x = (o[ID2_EDITCMD].flag & _O_FLAGB);
	zflag &= ~_C_ZINCDIR;
	if (y)
		zflag |= _C_ZINCDIR;
	if (ini_packerinfo() == 0)
		return 0;
	if (ini_initcommand() == 0)
		return 0;
	q = cpi->command;
	strcat(q, wshortname(b->name));
	strcat(q, " ");
	if (cpi->O08[0])
		strcat(q, cpi->O08);
	strcat(q, p);
	if (x) {
		if (!tgetline(cp_decompress, q, 40, 128))
			return _C_NORMAL;
	}
	return ex_command(q);
}

/* Compression Setup Dialog box */

enum {  ID_ZAUTOSELECT,
	ID_ZSMALLEST,
	ID_ZFASTEST,
	ID_ZPASSWORD	};

int cmcompression(void)
{
	DOBJ *d;
	TOBJ *o;

	if ((d = rsopen(IDD_Compression)) == 0)
		return 0;
	o = d->object;
	if (zflag & _C_ZSMALLEST)
		o[ID_ZSMALLEST].flag |= _O_RADIO;
	else if (zflag & _C_ZFASTEST)
		o[ID_ZFASTEST].flag |= _O_RADIO;
	else
		o[ID_ZAUTOSELECT].flag |= _O_RADIO;
	if (zflag & _C_ZPASSWORD)
		o[ID_ZPASSWORD].flag |= _O_FLAGB;
	tdinit(d);
	if (rsevent(IDD_Compression, d)) {
		zflag &= ~(_C_ZPASSWORD|_C_ZSMALLEST|_C_ZFASTEST);
		if (o[ID_ZSMALLEST].flag & _O_RADIO)
			zflag |= _C_ZSMALLEST;
		else if (o[ID_ZFASTEST].flag & _O_RADIO)
			zflag |= _C_ZFASTEST;
		if (o[ID_ZPASSWORD].flag & _O_FLAGB)
			zflag |= _C_ZPASSWORD;
	}
	twclose(d);
	return 1;
}