/*
  pg.c

  (c) 1995-1998 Bill Weinman, wew@bearnet.com

  Generic Screen Pager for FreeDOS

  This program was coded particularly for use with my FreeDOS help
  system.  I am making it available as a module for general use because
  it's generally useful.  It is not intended to be the pager-to-beat-
  all-pagers because I don't need that.  If you do, you can either 
  suggest enhancements--and if I think they're really kewl and feel
  like coding them I will--or you can code them yourself and submit
  them to me; but please, if you want me to continue to support this
  code then please don't enhance the code without letting me know.
                      --BearHeart, Phoenix, AZ
                        wew@bearnet.com
  
  Programming Environment
  
  This program was written using Dave Dunfield's Micro-C (MCC).  MCC 
  is available as freeware from Dunfield Development Systems (DDS).  
  
  For a copy of MCC, contact Dunfield Development Systems at 
  P.O. Box 31044, Nepean, Ontario (Canada), K2B 8S8.  Or call
  Tel: 613-256-5820  Fax: 613-256-5821  BBS: 613-256-6289
  
  Legal Stuff
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <stdio.h>
#include "pg.h"

#define setattr(attr) (curatt = attr)

/* Globals */

struct longptr {
  int lsw;
  int msw;
} longbuf,                                      /* scratch buffer for longs */
 *longptr;                                      /* scratch pointer for longs */

unsigned int bookmark[5];                       /* BAHCL add 5 bookmarks */
unsigned int segments[MAXSEGMENTS]; /* array of segment addresses for page buffers */

unsigned int numlines;                          /* num of lines in the file */

int Color = false;                              /* color flag */

word VideoRegen;                                /* video regen pointer */
word RegenSize;                                 /* in words  */

byte DispMode;                                  /* display adapter mode */
byte DispRows;                                  /* display rows (from BIOS) */
word DispCols;                                  /* display cols (from BIOS) */

byte curatt;                                    /* current attribute */

char linebuf[MAXSIZELINE];                      /* buffer for reading lines */

int tabstops = TABSTOPS;                        /* should be run-time switch */

byte attr_status, attr_lo_status, attr_text;

FILE *infile;                                   /* input file pointer */
char *infilename;                               /* input file name */


/* Functions */

void InitVideo();                               /* call this first */
int readlines();                                /* read in the line addresses */
void mvaddnstr();                               /* print at only n char from str */
void prat();                                    /* print at */
register errorexit();                           /* exit with a message */
register pratf();                               /* print at with format */
void usage();                                   /* tell 'em how ta use it */




/* The program starts here */

int
main(int argc, char ** argv)
{
  register i;

  InitVideo();

  if (argc == 1)
    {
      usage();      /* BAHCL */
/* BAHCL
      infile = stdin;
      infilename = "<stdin>";
*/
    }
  else if (strcmp(argv[1], "/?") == 0)
    {
      usage();
    }
  else
    {
      infilename = argv[1];
      infile = fopen(infilename, "r");
      if (infile == NULL)
        {
          errorexit(1, "pg: cannot open file %s", infilename);
        }
    }

  readlines();              /* read in all the line addresses */
  pagefile();				/* page thru the file */

  setattr(attr_text);
  prat(DispRows - 1, 0, " ");

  for(i = 0; i < MAXSEGMENTS; i++)
    {
      if(segments[i])
        {
          free_seg(segments[i]);
        }
    }
  fclose(infile);       /* BAHCL */
  exit(0);
}

/* readlines() - Read the whole file and extract the address of the
   begining of each line.  These addresses are stored in far memory
   using alloc_seg().  NOTE: This function is MCC specific! */

readlines()
{
  register unsigned int linenum;
  int segnum, rc;
  unsigned memoffset;

  rc = 1;   /* BAHCL */
  for (segnum = 0; segnum < MAXSEGMENTS; segnum++) segments[segnum] = 0;
  cls();
  prat(0, 1, "Reading file . . . ");

  for (linenum = 1; rc; linenum++)
    {
      segnum = (linenum - 1) / LINESPERSEG;
      if(!segments[segnum])			/* alloc a new space */
        {
          if(!(segments[segnum] = (unsigned) alloc_seg(4096)))
            {
              errorexit(2, "pg: Cannot allocate far memory\n");
            }
        }

      ftell(infile, &longbuf.msw, &longbuf.lsw);
      if(!(linenum % 1024))
        {
          pratf(0, 20, "%05d", linenum);
        }
      memoffset = ((linenum % LINESPERSEG) - 1) * 4;
      pokew(segments[segnum], memoffset, longbuf.msw);
      pokew(segments[segnum], memoffset + 2, longbuf.lsw);
      rc = fgets(linebuf, MAXSIZELINE - 1, infile);
    } /* for */
  pratf(0, 20, "%05d", linenum);
  numlines = linenum - 1;

}

/* getline() -  Find a line of text from the line buffer. Returns
   pointer to linebuf or zero if cannot find */

int
getline(unsigned linenum)
{
  int segnum;
  unsigned memoffset;

  segnum = (linenum - 1) / LINESPERSEG;
  if(!segments[segnum])
    {
      return 0;
    }

  memoffset = ((linenum % LINESPERSEG) - 1) * 4;
  longbuf.msw = peekw(segments[segnum], memoffset);
  longbuf.lsw = peekw(segments[segnum], memoffset + 2);
  if(fseek(infile, longbuf.msw, longbuf.lsw, SEEK_SET))
    {
      return 0;
    }
  else
    {
      return fgets(linebuf, MAXSIZELINE - 1, infile);
    }
}

/* exptabs() -  Expand tabs in the line buffer.
   (Added 21 aug 98, wew) */

void
exptabs()
{
  register i;
  register o;
  char out[MAXSIZELINE];
  char *in;

  in = linebuf;
  memset(out, 0, MAXSIZELINE);

  for(i = o = 0; o < (MAXSIZELINE - 1) && in[i]; i++, o++)
    { 
      if(in[i] == TAB)
        {
          out[o] = SPACE;
          while((o+1) % tabstops && (o < (MAXSIZELINE - 1)))
            {
              out[++o] = SPACE;
            }
        }
      else
        {
          out[o] = in[i];
        }
    }

  memcpy(linebuf, out, MAXSIZELINE);
}


char *stat[] = 
{ 
  "     ", 
  "<top>", 
  "<end>" 
};


pagefile()
{
  register int i;
  int redraw;
  int lineno;                   /* current line number */
  int topline;                  /* top screen line */
  int botline;                  /* bottom screen line */
  int screenline;               /* current screenline */
  int pagesize;                 /* vert size of display area */
  int pageoffset;               /* offset into page */
  int kc;                       /* current keypress */
  int rc;                       /* return code */
  int pgstat;                   /* current status for display */
  int quit;			/* (jh) flag, for the exit condition */
  int pancolumn;    /* pan columns  max 50 */

  /* Initialize variables */
  pancolumn = 0;
  redraw = true;
  pagesize = DispRows - 2;
  topline = lineno = 1;
  botline = DispRows - 2;
  pageoffset = 0;
  rc = -1;
  pgstat = STATOK;
  quit = 0;					/* (jh) */

  while (!quit)
    {
      setattr(attr_status);
      clreol(0, 0);
      pratf(0, 1, "PG -- %-30s%c", infilename,  /* display filename */
      strlen(infilename) > 30 ? '+' : ' ');
      if (topline < 1)       /* alert top of file? */
        {
          topline = 1;
          pgstat = STATTOP;
        }
      else if (topline > (numlines - pagesize)) /* alert bot of file? */
        {
          topline = numlines - pagesize;
          pgstat = STATBOT;
        }

      pratf(0, DispCols - 6, "%5d", topline);   /* display top line number  */
      lineno = topline;

      while((lineno - topline) < botline)       /* display loop */
        {
          screenline = lineno - topline + 1;    /* where are we on the screen? */
          setattr(attr_text);
          rc = getline(lineno);                 /* read the next line from input */
          exptabs();                            /* expand the tabs (21 aug 98 wew) */
/* BAHCL  linebuf[DispCols] = 0;    */          /* keep it from wrapping */
                                                /* but lines > 80 column truncated */
          if(rc)
            {
              if(redraw)                        /* display it */
                {
/* BAHCL          linebuf[DispCols] = 0; */       /* keep it from wrapping */
                  mvaddnstr(screenline, pageoffset, DispCols, linebuf+pancolumn);

/* BAHCL          prat(screenline, pageoffset, linebuf); */
                  clreol(screenline, strlen(linebuf+pancolumn));
                  if(kc == KEY_UP)
                    redraw = false;
                }
              lineno++;
            }
          else   /* past end of file, blank rest of screen */
            {
              pgstat = STATBOT;
              setattr(attr_text);
              for( ; (screenline = lineno - topline) < botline; lineno ++)
              clreol(++screenline, 0);
            }
        } /* while */

      if (!redraw && rc && (kc == KEY_DN))	/* we scrolled up, */
        {                                   /* so need to disp last line */
          prat(botline, 0, linebuf+pancolumn);
          clreol(screenline, strlen(linebuf)-pancolumn);
        }
      if (!redraw && rc && (kc == RETURN))  /* scrolled up, */ /* BAHCL */
        {                                   /* so need to disp last line */
          prat(botline, 0, linebuf+pancolumn);
          clreol(screenline, strlen(linebuf)-pancolumn);
        }
      redraw = false;
      setattr(attr_status);
      prat(0, 50, stat[pgstat]);
      if(pgstat)
        {
          BELL;
        }

      pgstat = STATOK;
      switch(kc = kbget())
        {
          case PG_UP:
            topline -= (pagesize - 1);
            redraw = true;
            break;
          case SPACE:                 /* (jh) */
          case PG_DN:
          /* scroll down by one screen */
            if((topline + pagesize) < numlines)
              {
                topline = lineno - 1;
              }
            else
              {
                pgstat = STATBOT;
              }
            redraw = true;
            break;
          case KEY_HOME:
            topline = 1;
            redraw = true;
            break;
          case KEY_END:
            topline = numlines - pagesize;
            redraw = true;
            break;
          case KEY_UP:
            if(--topline)
              {
                scroll_down(1, pagesize - 1);
              }
            else
              {
                pgstat = STATTOP;
              }
            redraw = true;
            break;
          case RETURN:                    /* (jh) bug? */  /* Fixed by BAHCL */
          case KEY_DN:
          /* scroll down one line only */
            if((topline + pagesize) < numlines)
              {
                i = scroll_up(1, pagesize - 1);
                redraw = false;
                topline++;
              }
            else
              {
                pgstat = STATBOT;
              }
            break;
          case ESC:
          case 'Q':                   /* (jh) */
          case 'q':                   /* (jh) */
            quit = 1;                 /* (jh) */
            break;
          case '1':                   /* (BAHCL) */
            bookmark[0] = topline;    /* bookmark */
            break;
          case '2':
            bookmark[1] = topline;
            break;
          case '3':
            bookmark[2] = topline;
            break;
          case '4':
            bookmark[3] = topline;
            break;
          case '5':
            bookmark[4] = topline;
            break;
          case KEY_A1:
            topline = bookmark[0];
            redraw = true;
            break;
          case KEY_A2:
            topline = bookmark[1];
            redraw = true;
            break;
          case KEY_A3:
            topline = bookmark[2];
            redraw = true;
            break;
          case KEY_A4:
            topline = bookmark[3];
            redraw = true;
            break;
          case KEY_A5:
            topline = bookmark[4];
            redraw = true;
            break;
          case KEY_LT:
            pancolumn -= 10;
            if (pancolumn < 0) pancolumn = 0;
            redraw = true;
            break;
          case KEY_RT:
            pancolumn += 10;
            if (pancolumn > 50) pancolumn = 0;
            redraw = true;
            break;
          case BK_SPACE:
          default:
            BELL;
            break;
         } /* switch */
    }   /* !Quit */
}

void
InitVideo()
{
  asm {
    mov ax, BIOS_DATA_SEG
    mov es, ax
    mov al, [es:DISP_MODE]
    mov _DispMode, al
    mov ax, [es:DISP_COLS]
    mov _DispCols, ax
    mov al, [es:DISP_ROWS]         ; actually last row
    inc al
    mov _DispRows, al
  }

  RegenSize = DispRows * DispCols;     /* for use in asm functions */
  VideoRegen = (DispMode == MONO ? 0xb000 : 0xb800);
  Color = (DispMode == MONO ? false : true);

/* BAHCL moved from main() */
#ifdef USE_COLOR_MODE                   /* (jh) */
  if(Color)
    {
      attr_status    = CO_STATUS;
      attr_lo_status = CO_LO_STATUS;
      attr_text      = CO_TEXT;
    }
  else
    {
      attr_status    = MO_STATUS;
      attr_lo_status = MO_LO_STATUS;
      attr_text      = MO_TEXT;
    }
#else
  attr_status    = ATTR (BLACK, WHITE);			/* (jh) */
  attr_lo_status = ATTR (BLACK, WHITE);			/* (jh) */
  attr_text      = ATTR (WHITE, BLACK);			/* (jh) */
#endif /* USE_COLOR_MODE */

  setattr(attr_text);

}

cls()
{
  asm {
    push es
    mov ax, BIOS_DATA_SEG
    mov es, ax
    xor ax, ax
    mov [es:CURSOR], ax   ; home the cursor
    mov di, ax            ; zero origin for regen
    mov cx, _RegenSize
    mov ax, _VideoRegen
    mov es, ax
    mov ah, _curatt
    mov al, 20h           ; ' '
    rep stosw
    pop es
  }
}

scroll_down(top, bottom)
{
  asm {
    push ds
    std                   ; backwards for movsw

    ; number of words to move = (bottom - top + 1) * DispCols

    xor ax, ax
    mov al, 4[bp]         ; bottom
    mov bl, 6[bp]         ; top
    sub al, bl
    inc ax
    mov bx, _DispCols
    mul bl
    mov cx, ax            ; move size -> cx

    ; dest addr = (bottom * DispCols) + DispCols - 1) * 2 

    xor ah, ah
    mov al, 4[bp]
    inc ax
    mov bx, _DispCols
    push bx
    mul bl
    pop bx
    add ax, bx
    dec ax
    shl ax, 1
    mov di, ax

    ; source addr = source - (DispCols * 2)

    push ax
    mov ax, _DispCols
    shl ax, 1
    mov bx, ax
    pop ax
    sub ax, bx
    mov si, ax

    mov bx, _VideoRegen
    mov es, bx
    mov ds, bx
    rep movsw
    cld                   ; restore direction
    pop ds
  }
}  

scroll_up(top, bottom)
{
  asm {
    push ds
    cld

    ; number of words to move = (bottom - top + 1) * DispCols

    xor ax, ax
    mov al, 4[bp]         ; bottom
    mov bl, 6[bp]         ; top
    sub al, bl
    inc ax
    mov bx, _DispCols
    mul bl
    mov cx, ax            ; move size -> cx

    ; dest addr = top * 2 * DispCols 

    xor ah, ah
    mov al, 6[bp]
    shl ax, 1
    mov bx, _DispCols
    mul bl
    mov di, ax

    ; source addr = source + (DispCols * 2)

    push ax
    mov ax, _DispCols
    shl ax, 1
    mov bx, ax
    pop ax
    add ax, bx
    mov si, ax

    mov bx, _VideoRegen
    mov es, bx
    mov ds, bx
    rep movsw
    pop ds
  }
}

vidchar(vidrow, vidcol, vidchar, vidattr)
{
  asm {
    push es
    mov ax, BIOS_DATA_SEG
    mov es, ax
    mov al, 8[bp]        ; col
    mov ah, 10[bp]       ; row
    mov [es:CURSOR], ax
    push ax
    mov al, ah
    xor ah, ah
    mov bx, _DispCols    ; offset = vidrow * DispCols + vidrow
    mul bl
    pop bx
    xor bh, bh           ; add in the col
    add ax, bx
    shl ax, 1            ; * 2 for word based
    mov di, ax           ; zero origin for regen
    mov ax, _VideoRegen
    mov es, ax
    mov ah, 4[bp]        ; attr
    mov al, 6[bp]        ; char
    stosw
    pop es
  } 
}

clreol(row, col)
{
  asm {
    push es
    mov ax, BIOS_DATA_SEG
    mov es, ax
    cld

    ; num words = DispCols - col

    mov ax, _DispCols
    mov bx, 4[bp]        ; col
    cmp ax, bx           ; bail if DispCols <= col
    jle attahere
    push bx              ; use for di calc
    sub ax, bx
    mov cx, ax

    ; di = ((row * DispCols) + col) * 2

    mov ax, 6[bp]        ; row
    mov bx, _DispCols
    cmp ax, bx
    mul bl
    pop bx               ; col
    add ax, bx
    shl ax, 1
    mov di, ax
  
    mov ax, _VideoRegen
    mov es, ax
    mov ah, _curatt
    mov al, ' '
    rep stosw
  attahere:  
    pop es
  } 
}

/* BAHCL add : useful as the one in Linux ncurses */
void
mvaddnstr(int prow, int pcol, int n, char * s)
{
  register int i;
  i = 0;
  while (s[i] && (i < n))
    {
      vidchar(prow, pcol++, s[i++], curatt);
    }
}

void
prat(int prow, int pcol, char * s)
{
  register int i;

  for(i = 0; s[i]; i++)
    {
      vidchar(prow, pcol++, s[i], curatt);
      pcol %= DispCols;     /* wrap if necessary */
      if(pcol == 0)
        {
          prow++;
        }
    }
}

/* NOTE: I wrote the following routines before MCC's _format_ was
   documented and I haven't had the need to look at them since.  Be
   forwarned, they're probably weird. */

register
pratf(prow, pcol, unsigned args)
{
  unsigned *ptr;
  char buffer[256];
  int row, col;
  register int i;

  ptr = ((i = nargs()) - 2) * 2 + &args;
  _format_(ptr, buffer);

  col = * (int *) ptr;   /* ptr points at the next arg */
  ptr += 2;
  row = * (int *) ptr;
  prat(row, col, buffer);
}

register
errorexit(pcode, unsigned args)
{
  unsigned *ptr;
  int code;
  char buffer[256];

  ptr = (nargs() - 1) * 2 + &args;
  _format_(ptr, buffer);

  code = * (int *) ptr;  /* ptr points at the next arg */
  prat(24,0,buffer);
  exit(code);
}

void
usage()
{
  /* (jh) I moved all the usage text here so that the text wasn't
     separated from the usage() function */

  /* (jh) I moved the VERSION text here, since it is not used
     anywhere else. */

  /* (jh) Most DOS programs just show the name and usage, but you can
     build with the GNU 'copying' statement anyway by using
     -DSHOW_GNU_COPYING */
  int i;
  i = 0;
  cls();   /* BAHCL */
  prat(0,0,"PG, version 1.0 (c) 1995-1998 Bill Weinman, wew@bearnet.com");
  prat(1,0,"PG, version 1.02 2003 Maintainer BAHCL, freedos_pg@yahoo.com.hk");

#ifdef SHOW_GNU_COPYING
  prat(2,0,"Distributed under the terms of the GNU General Public License.");
  prat(3,0,"See the file COPYING for details.");
  i = 2;
#endif

  prat(3+i,0,"usage:  PG [/?] [filename]");
  prat(4+i,0,"  /?       - display this help screen.");
  prat(5+i,0,"  filename - page through filename on the PC screen");
  prat(6+i,0,"             if filename is omitted, shows this help screen");
  prat(7+i,0,"  Read pg.man for details.");

  exit(0);
}
