/*
  pg.c    1.05

  (c) 1995-1998 Bill Weinman, wew@bearnet.com

  Generic Screen Pager for FreeDOS

  This program was coded particularly for use with my FreeDOS help
  system.  I am making it available as a module for general use because
  it's generally useful.  It is not intended to be the pager-to-beat-
  all-pagers because I don't need that.  If you do, you can either 
  suggest enhancements--and if I think they're really kewl and feel
  like coding them I will--or you can code them yourself and submit
  them to me; but please, if you want me to continue to support this
  code then please don't enhance the code without letting me know.
                      --BearHeart, Phoenix, AZ
                        wew@bearnet.com

  Maintainer: BAHCL   freedos_pg@yahoo.com.hk


  Programming Environment:
    Please read pg.man section "INSTALL PG"


  Legal Stuff
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dir.h>
#include "pg.h"

#define setattr(attr) (curatt = attr)

/* Globals */

char *stat[] =
{ 
  "     ", 
  "<top>", 
  "<end>" 
};


char searchstr[20];                             /* search string */
char buffer[20];                                /* scratch buffer */
unsigned long bookmark[5];                      /* BAHCL add 5 bookmarks */
long searchline;                                /* search line number */
long toline;                                    /* copy to line */
long showline;
unsigned long offset;                           /* infile offset */
unsigned long numlines;                         /* num of lines in the file */
int Color = false;                              /* color flag */
unsigned int  VideoRegen;                       /* video regen pointer */
unsigned int  RegenSize;                        /* in words  */
unsigned char DispMode;                         /* display adapter mode */
unsigned char DispRows;                         /* display rows (from BIOS) */
unsigned int  DispCols;                         /* display cols (from BIOS) */
unsigned char curatt;                           /* current attribute */
unsigned char attr_status, attr_lo_status, attr_text;
char linebuf[MAXSIZELINE];                      /* buffer for reading lines */
int  tabstops = TABSTOPS;                       /* should be run-time switch */
int  filenum, maxfile;                          /* infile number, max. files */
char *infilename;                               /* input file name */
FILE *infile;                                   /* input file pointer */
FILE *pgsav=NULL;                               /* save file */
char *filelist;                                 /* filename list + open file line # */
char overrun;                                   /* signal to line # overrun */

/* Local Functions */
void InitVideo(void);                           /* call this first */
unsigned long readlines(FILE *);                /* read in the line addresses */
int  getline(long);                             /* get a line from infile */
void exptabs(void);                             /* expand tabstop */
int  pagefile(long);                            /* user interface */
void save_screen(long, long);                   /* save screen to disk */
void mvaddnstr(int,int,int,char *);             /* print at only n char from str */
void usage(void);                               /* tell 'em how ta use it */
void errorexit(int, char *);                    /* error!! bail out! */
void headline(long);                            /* program head line */
void clearline(int, char);                      /* clear a line */
char *strfind(char *, char *);                  /* find substring */
char *strrfind(char *, char *);                 /* find substr backward */
int accept(char *);                             /* accept files not blacklisted */
int display(long, int, int);                    /* display file content */

/* Global Assembler language functions */
extern scroll_down(int, int, int, int, int);    /* x1,y1,x2,y2,lines */
extern scroll_up(int, int, int, int, int);      /* x1,y1,x2,y2,lines */
extern mvaddch(int, int, int, int);             /* row,col,vidchar,attr */
extern getbiosinfo(void);                       /* get BIOS information */
extern getstr(char *);                          /* Get a string */
extern kbget(void);                             /* get keyboard scancode */
extern setup_help(void);                        /* setup help information */
extern on_line_help(void);                      /* On line help */


int
main(int argc, char **argv)
{
  int pathlen, filepos;
  char filepath[100];
  struct ffblk ffblk;

  fnsplit(argv[1], filepath, filepath+3, filepath+70, filepath+80);
  InitVideo();                                      /* Init & get screen info */
  if (argc == 1)                                    /* no file name */
    {
      usage();                                      /* show usage */
    }
  else if (strcmp(argv[1], "/?") == 0)              /* basic help */
    {
      usage();
    }
  else                                              /* get ready */
    {
      filelist = malloc(4096);                      /* allocate mem. for */
                                                    /* about 256 filenames */
      showline = 1;                                 /* each file starts line 1 */
      maxfile = 0;
      filenum = findfirst(argv[1],&ffblk,0);        /* find # of files */
      while(!filenum)
        {
          if (accept(ffblk.ff_name))                /* validate file ext. */
            {
              strncpy(filelist + maxfile * 16, ffblk.ff_name, 12);
              memcpy(filelist + 12 + maxfile * 16, &showline, sizeof(long));
              ++maxfile;
            }
          filenum = findnext(&ffblk);
        }
      free(filelist);
      if (maxfile == 0)                             /* No file */
        {
          errorexit(1, strcat("pg: cannot open file ", argv[1]));
        }
    }

  filelist = malloc((MAXLINES+50) * sizeof(long) + maxfile * 16);
                                            /* Max. # of lines buffer can hold */

  setup_help();                                     /* Setup on-line help */
  filenum = 1;
  strcat(filepath, filepath+3);
  pathlen = strlen(filepath);
  while (filenum)                                   /* process individual file */
    {
      filepos = (filenum - 1) * 16;
      strncpy(filepath+pathlen, filelist+filepos, 12);
      filepath[pathlen+12] = '\0';
      infile = fopen(filepath, "rb");
      infilename = filepath;

      numlines = readlines(infile);                 /* read all the line addresses */
      memcpy(&showline, filelist+filepos+12, sizeof(long));

      display(showline,0,0);
      filenum = pagefile(showline);                 /* page thru the file */
      memcpy(filelist+filepos+12, &showline, sizeof(long));

      if (filenum < 1)  filenum = 1;
      if (filenum > maxfile)  filenum = 0;
      scroll_up(0, 0, DispCols-1, DispRows, DispRows);  /* clear screen */
      setattr(attr_text);                               /* reset video attribute */

      fclose(infile);
    }
  free(filelist);

  return(0);
}

/*
   readlines() - Read the whole file and extract the address of the
   beginnng of each line.
   BAHCL : These addresses append to filelist
   Max. # of lines is defined in pg.h
*/
unsigned long
readlines(FILE *infile)
{
  unsigned long linenum, lineoffset;
  int rc;

  clrscr();
  mvaddnstr(0, 1, 30, "Reading file . . . ");
  lineoffset = maxfile * 16;
  rc = 1;
  for (linenum = 1; rc; linenum++)
    {
      offset = ftell(infile);
      memcpy(filelist+lineoffset,&offset,sizeof(long));
      lineoffset += 4;
      (char *) rc =  fgets(linebuf, MAXSIZELINE - 1, infile);
      if (linenum == MAXLINES)
        {
          rc = 0;
        }
    }
  return (linenum - 1);
}

/*
   getline() -  Find a line of text from the line buffer. Returns
   pointer to linebuf or zero if cannot find
*/
int
getline(long linenum)
{
  unsigned long tmpoffset;
  linebuf[0] = NULL;
  if (linenum >= numlines)
    {
      return 0;                                 /* error */
    }
  else
    {
      tmpoffset = (linenum - 1) * sizeof(long) + maxfile * 16;
      memcpy(&offset, filelist + tmpoffset, sizeof(long));
      fseek(infile, offset, SEEK_SET);
      fgets(linebuf, MAXSIZELINE - 1, infile);
      return 1;
    }
}

/*
   exptabs() -  Expand tabs in the line buffer.
   (Added 21 aug 98, wew)
   BAHCL : handles LF,CR as Turbo C 'gets' does not remove them;
           also SPACE => space
*/
void
exptabs(void)
{
  register i;
  register o;
  char out[MAXSIZELINE];
  char *in;
  char space;

  in = linebuf;
  memset(out, 0, MAXSIZELINE);
  space = '\x20';
  for(i = o = 0; o < (MAXSIZELINE - 1) && in[i]; i++, o++)
    { 
      switch(in[i])
        {
          case TAB:
            out[o] = space;
            while((o+1) % tabstops && (o < (MAXSIZELINE - 1)))
              {
                out[++o] = space;
              }
            break;
          case '\xD':                               /* replace linefeed */
            out[o] = space;
            break;
          case '\xA':                               /* replace return */
            out[o] = NULL;
            break;
          default:
            out[o] = in[i];
            break;
        }
    }
  memcpy(linebuf, out, MAXSIZELINE);
}

/* display lines to screen */
int display(long fromline, int pancolumn, int kc)
{

  long lineno;                  /* current line number */
  long topline;                 /* top screen line */

  int botline;                  /* bottom screen line */
  int screenline;               /* current screenline */
  int pagesize;                 /* vert size of display area */
  int pageoffset;               /* offset into page */
  int rc;                       /* return code */

  /* Initialize variables */
  topline    = fromline;
  botline    = DispRows - 2;
  pagesize   = DispRows - 2;
  pageoffset = 0;
  rc         = -1;

  if (numlines < (pagesize))
    {
      pagesize = numlines-1;
    }

  setattr(attr_status);
  if (topline < 1)          /* alert top of file? */
    {
      topline = 1;
    }
  else if (topline > (numlines - pagesize)) /* alert bot of file? */
    {
      topline = numlines - pagesize;
    }

  headline(topline);                        /* show file information */

  lineno = topline;
  while((lineno - topline) < botline)       /* display loop */
    {
      screenline = lineno - topline + 1;    /* where are we on the screen? */
      rc = getline(lineno++);               /* read the next line from input */
      exptabs();                            /* expand the tabs (21 aug 98 wew) */
      if(rc)
        {
          clearline(screenline, attr_text);
          setattr(attr_text);
          mvaddnstr(screenline, pageoffset, DispCols, linebuf+pancolumn);
        }
    } /* while */

  if (rc && (kc == KEY_DN))                 /* we scrolled up, */
    {                                       /* so need to disp last line */
      setattr(attr_text);
      mvaddnstr(botline, 0, DispCols, linebuf+pancolumn);
    }
  setattr(attr_status);
  return 0;
}

/* user interface here */
int
pagefile(long fromline)
{
  long topline;                 /* top screen line */
  int botline;                  /* bottom screen line */
  int screenline;               /* current screenline */
  int pagesize;                 /* vert size of display area */
  int kc;                       /* current keypress */
  int quit;                     /* (jh) flag, for the exit condition */
  int pancolumn;                /* pan columns */

  /* Initialize variables */
  pancolumn  = 0;
  searchline = 0;
  kc         = 0;
  topline    = fromline;
  botline    = DispRows - 2;
  pagesize   = DispRows - 2;
  quit       = 0;               /* (jh) */

  clearline(botline+1, attr_status);

  while (!quit)
    {
      switch(kc=kbget())
        {
          case PG_UP:
            topline -= (pagesize - 1);
            break;
          case SPACE:                   /* (jh) */
          case PG_DN:                   /* scroll down by one screen */
            if((topline + pagesize) < numlines)
              {
                topline = topline + pagesize - 1;
              }
            break;
          case KEY_HOME:
            topline = 1;
            break;
          case KEY_END:
            topline = numlines - pagesize;
            break;
          case KEY_UP:
            if(--topline)
              {
                scroll_down(0, 1, DispCols-1, pagesize, 1);
              }
            break;
          case RETURN:                          /* scroll down one line only */
          case KEY_DN:
            kc = KEY_DN;
            if((topline + pagesize) < numlines)
              {
                scroll_up(0, 1, DispCols-1, pagesize, 1);
                topline++;
              }
            break;
          case KEY_F9:                              /* F9 Prev file */
            quit = 1;
            --filenum;
            break;
          case KEY_F10:                             /* F10 Next file */
            quit = 1;
            ++filenum;
            break;
          case ESC:                                 /* Quit */
          case KEY_Q:
          case KEY_q:
            quit = 1;
            filenum = maxfile + 1;
            break;
          case SEARCH:                              /* Begin a search */
            clearline((int) DispRows-1, attr_status);
            mvaddnstr((int) DispRows-1, 0, 9, "Search:  ");
            searchstr[0]=19;
            searchline = topline;
            getstr(searchstr);
            if (searchstr[1])
            {
              if (strfind(linebuf, searchstr))
                {
                  ltoa(searchline-1, buffer, 10);
                  mvaddnstr((int) DispRows-1, DispCols-strlen(buffer)-1, 8, buffer);
                }
            }
            break;
          case KEY_JMP:                              /* ? Jump to line */
            clearline((int) DispRows-1, attr_status);
            mvaddnstr((int) DispRows-1, 0, 10, "Jump to:  ");
            buffer[0]=19;
            getstr(buffer);
            if (buffer[1])
              {
                topline = atol(buffer+2);
              }
            clearline((int) DispRows-1, attr_status);
            break;
          case KEY_1:                               /* set Bookmark #1 */
            bookmark[0] = topline;
            break;
          case KEY_2:
            bookmark[1] = topline;
            break;
          case KEY_3:
            bookmark[2] = topline;
            break;
          case KEY_4:
            bookmark[3] = topline;
            break;
          case KEY_5:
            bookmark[4] = topline;
            break;
          case KEY_A1:                              /* Back to bookmark #1 */
            topline = bookmark[0];
            break;
          case KEY_A2:
            topline = bookmark[1];
            break;
          case KEY_A3:
            topline = bookmark[2];
            break;
          case KEY_A4:
            topline = bookmark[3];
            break;
          case KEY_A5:
            topline = bookmark[4];
            break;
          case KEY_F1:                                /* On-line help */
            on_line_help();
            break;
          case KEY_F2:                                /* cont. search backward */
            mvaddnstr((int) DispRows-1, DispCols-9, 8, "        "); /* clear result area */
            if (searchstr[1])
            {
              if (strrfind(linebuf, searchstr))
                {
                  ltoa(searchline, buffer, 10);
                  mvaddnstr((int) DispRows-1, DispCols-strlen(buffer)-1, 8, buffer);
                }
            }
            break;
          case KEY_F3:                              /* cont. search forward */
            mvaddnstr((int) DispRows-1, DispCols-9, 8, "        ");
            if (searchstr[1])
            {
              if (strfind(linebuf, searchstr))
              {
                ltoa(searchline-1,buffer,10);
                mvaddnstr((int) DispRows-1, DispCols-strlen(buffer)-1, 8, buffer);
              }
            }
            break;
          case KEY_F4:                              /* save screen to file */
            save_screen(topline, topline + DispRows - 2);
            break;
          case KEY_LT:                              /* Left key */
            pancolumn -= 20;                        /* pan left */
            if (pancolumn < 0) pancolumn = 0;
            break;
          case KEY_RT:                              /* Right key */
            pancolumn += 20;                        /* pan right */
            if (pancolumn > PANLIMIT) pancolumn = 0;
            break;
          case BK_SPACE:
          default:
            break;
         } /* switch */
       if (topline < 1) topline = 1;
       display(topline, pancolumn, kc);
    }   /* !Quit */
    showline = topline;
    return filenum;
}

/* save screen to a save file */
void
save_screen(long saveline, long toline)
{
  long seekpos, pos1, pos2;
  char *savefile = "C:\\PG.SAV";

  clearline((int) DispRows-1, attr_status);
  mvaddnstr((int) DispRows-1, 0, 9, "Save to: ");
  if (pgsav == NULL)                            /* if not open yet */
    {
      searchstr[0]=2;                           /* get the drive letter only */
      getstr(searchstr);
      savefile[0] = searchstr[2];
    }

  if ((pgsav = fopen(savefile, "ab")) == NULL)   /* append data */
    {
      return;
    }
  strcpy(linebuf, "!!! Copy from ");            /* indentify copy info */
  strcat(linebuf, infilename);
  strcat(linebuf, " line ");
  strcat(linebuf, ltoa(saveline, buffer,10));
  strcat(linebuf, " !!! ");
  fwrite(linebuf, strlen(linebuf), 1, pgsav);

  if (toline >= numlines)
    toline = numlines;
  seekpos = maxfile * 16 + (saveline - 1) * sizeof(long);
  memcpy(&offset, filelist + seekpos, sizeof(long));
  fseek(infile, offset, SEEK_SET);
  pos1 = ftell(infile);
  while (--toline - (saveline-1))
    {
      fgets(linebuf, MAXSIZELINE - 1, infile);
      pos2 = ftell(infile);
      fwrite(linebuf, pos2 - pos1, 1, pgsav);
      pos1 = pos2;
    }
  fclose(pgsav);
  mvaddnstr((int) DispRows-1, 8, 10, savefile);
}

/* Get display adaptor info. and initialize */
void
InitVideo(void)
{
  getbiosinfo();
  RegenSize = DispRows * DispCols;     /* for use in asm functions */
  VideoRegen = (DispMode == MONO ? 0xb000 : 0xb800);
  Color = (DispMode == MONO ? false : true);

/* BAHCL moved from main() */
#ifdef USE_COLOR_MODE                   /* (jh) */
  if(Color)
    {
      attr_status    = CO_STATUS;
      attr_lo_status = CO_LO_STATUS;
      attr_text      = CO_TEXT;
    }
  else
    {
      attr_status    = MO_STATUS;
      attr_lo_status = MO_LO_STATUS;
      attr_text      = MO_TEXT;
    }
#else
  attr_status    = ATTR (BLACK, WHITE);			/* (jh) */
  attr_lo_status = ATTR (BLACK, WHITE);			/* (jh) */
  attr_text      = ATTR (WHITE, BLACK);			/* (jh) */
#endif /* USE_COLOR_MODE */

  setattr(attr_text);

}

/* BAHCL: useful as the one in Linux ncurses */
void
mvaddnstr(int prow, int pcol, int n, char *s)
{
  int i;
  i = 0;
  while (s[i] && (i < n))
    {
      mvaddch(prow, pcol++, s[i++], curatt);
    }
}

/* error exit */
void
errorexit(int pcode, char *args)
{
    mvaddnstr((int) DispRows-1,0,80,args);
    exit(pcode);
}

/* program usage */
void
usage(void)
{
  /* (jh) I moved all the usage text here so that the text wasn't
     separated from the usage() function */

  /* (jh) I moved the VERSION text here, since it is not used
     anywhere else. */

  /* (jh) Most DOS programs just show the name and usage, but you can
     build with the GNU 'copying' statement anyway by using
     -DSHOW_GNU_COPYING */
  int i;
  i = 0;
  clrscr();   /* BAHCL */
  mvaddnstr(0,0,80,"PG, version 1.0 (c) 1995-1998 Bill Weinman, wew@bearnet.com");
  mvaddnstr(1,0,80,"PG, version 1.05 2003 Maintainer BAHCL, freedos_pg@yahoo.com.hk");

#ifdef SHOW_GNU_COPYING
  mvaddnstr(2,0,80,"Distributed under the terms of the GNU General Public License.");
  mvaddnstr(3,0,80,"See the file COPYING for details.");
  i = 2;
#endif

  mvaddnstr(3+i,0,80,"usage:  PG [/?] [filename]");
  mvaddnstr(4+i,0,80,"  /?       - display this help screen.");
  mvaddnstr(5+i,0,80,"  filename - page through filename on the PC screen");
  mvaddnstr(6+i,0,80,"             filename maybe in wildcard format");
  mvaddnstr(7+i,0,80,"             if filename is omitted, shows this help screen");
  mvaddnstr(8+i,0,80,"  Read pg.man for details.");

  exit(0);
}

/* Just a blank line with status attribute */
void
clearline(int lin, char attr)
{
  int i;

  setattr(attr);
  for (i=0; i < DispCols; i++) mvaddnstr(lin, i, 1, " ");
}

/* PG -- file information headline */
void
headline(long line)
{
  int pgstat;

  clearline(0, attr_status);
  mvaddnstr(0, 1, 6, "PG -- ");
  mvaddnstr(0, 7, 40, infilename);
  pgstat = STATOK;

  if (line <= 1)
    {
      pgstat = STATTOP;
    }
  if ((line + DispRows - 2) >= numlines)
    {
      pgstat = STATBOT;
    }

  mvaddnstr(0, 50, 5, stat[pgstat]);
  ltoa(numlines-1, buffer, 10);
  mvaddnstr(0, DispCols-strlen(buffer)-10, 6, buffer);
  ltoa(line, buffer, 10);
  mvaddnstr(0, DispCols-strlen(buffer)-1, 6, buffer);
}

/* find sub-string in forward manner */
char *strfind(char *str1, char *str2)
{
  char *ptr;

  do
  {
    getline(searchline++);
    ptr = strstr(str1, str2+2);
  } while((searchline < numlines) && (ptr == NULL));

  return ptr;
}

/* find sub-string in backward manner */
char *strrfind(char *str1, char *str2)
{
  char *ptr;

  do
  {
    getline(--searchline);
    ptr = strstr(str1, str2+2);
  } while((searchline > 0) && (ptr == NULL));

  return ptr;
}

/* reject to page a file that's within a list of extensions */
int accept(char *src)
{
  char *blacklist="BINCOMDLLDRVEXEOBJSYSVXDZIP";     /* rejected extensions */
  char *ptr;
  int i, j, k;

  k = 1;
  ptr = strchr(src,'.');
  if (ptr)
  {
    for (i=0, j=0; j < (strlen(blacklist) / 3); j++)
      {
        if (strncmp(blacklist + i, ptr + 1, 3) == 0)
          {
            k = 0;                                  /* reject */
            break;
          }
        i += 3;
      }
  }
  return k;                                         /* accept */
}
