/*
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/***********************************************************************
 * This program saves a copy of its input to a file and prints a copy to
 * the standard output.
 *
 * Author: James Hall
 *
 * 2008-07-03: DOS-ified the code by using the INT21H functions.
 * 			   Only compiler I have is OpenWatcom, so am not sure
 * 			   if it's portable compiler-wise - AJM
 * 2008-07-01: Added append and ignore-interrupts functionality by
 *             porting GNU Coreutils code - AJM
 */

#include <dos.h>
#include <errno.h>
#include <fcntl.h>
#include <io.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>			/* strlen() */

#ifdef ENABLE_KITTEN
#include "kitten.h"
#else
#define kittengets(A, B, STR) STR
#define kittenclose(A)
#endif

#include "tee.h"


static int outtee(int, const char **);
static void usage(void);

char * progName = "TEE";
char * version = "v2.0.1";
char params;		/* User defined parameters */
char mode;		/* File's access permissions, either read or append */

#ifdef ENABLE_KITTEN
nl_catd kitten;
#endif

int
main (int argc, char **argv)
{
	int i; 

#ifdef ENABLE_KITTEN
	if ((kitten = kittenopen("TEE")) <= 0) {
		fprintf(stderr, "Error loading TEE kitten file\n");
	}
#endif

	/* Check the command line */
	for (i = 1; i < argc; ++i) {
		if (argv[i][0] =='/') {
			if (strlen(argv[i]) == 2) {
				switch (argv[i][1]) {
					case '?':
						usage();
						break;
					case 'A':
					case 'a':
						params |= APPEND;
						break;
					case 'I':
					case 'i':
						params |= INTERRUPT;
						break;
					default:
						PRINT_ERR_UNKNOWN_OPT(argv[i]);
						exit(1);
						break;
				}
			} else {
				PRINT_ERR_UNKNOWN_OPT(argv[i]);
				exit(1);
			}
		} else {
			/* All parameters after first non-argument are files */
			break;
		}
	}

	if (params & INTERRUPT)
		signal(SIGINT, SIG_IGN);

	mode = (params & APPEND) ? MODE_APPEND : MODE_WRITE;

	/* Tee the input */
	return outtee(argc - i, (const char **) &argv[i]);
}

int
outtee(int numFiles, const char **files)
{
	int i;
	char status;
	char buffer[BUFSIZ];
	int * handles;
	unsigned bytesRead;
	unsigned bytesWritten;

	status = OK;
	memset(buffer, '\0', sizeof buffer);

	if ((handles = (int *) calloc(numFiles + 1, sizeof(int))) == NULL) {
		PRINT_ERR_ALLOC;
		perror("");
		exit(1);		/* Unable to continue */
	}

	/* Move all files up one in array to make room for STDOUT */
	for (i = numFiles; i >= 1; --i)
		files[i] = files[i - 1];

	handles[0] = 1;						/* STDOUT */
	setvbuf(stdout, NULL, _IONBF, 0);	/* Don't buffer output */

	for (i = 1; i <= numFiles; ++i) {
		if (mode == MODE_WRITE) {
			/* Overwrite the file if one exists */
			if (_dos_creat(files[i], _A_NORMAL, &handles[i])) {
				PRINT_ERR_OPEN(files[i]);
				perror("");
				exit(1);
			}
		} else if (mode == MODE_APPEND) {
			/* Only create a new file if one does not exist */
			if (_dos_creatnew(files[i], _A_NORMAL, &handles[i])
					&& errno != EEXIST) {
				PRINT_ERR_OPEN(files[i]);
				perror("");
				exit(1);
			}
			/* If file already exists, open it */
			if (errno == EEXIST) {
				if (_dos_open(files[i], O_WRONLY, &handles[i])) {
					PRINT_ERR_OPEN(files[i]);
					perror("");
					exit(1);
				}
			}
		}

		if (mode == MODE_APPEND)
			lseek(handles[i], 0, SEEK_END);
	}

	while (TRUE) {
		if (_dos_read(0, buffer, BUFSIZ, &bytesRead))
			break;

		for (i = 0; i <= numFiles; ++i) {
			if (_dos_write(handles[i], buffer, bytesRead, &bytesWritten)
					|| bytesRead != bytesWritten) {
				PRINT_ERR_WRITE(files[i]);
				status = ERROR;
			}
		}

		if (eof(0))
			break;
	}

	for (i = 1; i <= numFiles; ++i) {
		if (handles[i] && _dos_close(handles[i])) {
			PRINT_ERR_CLOSE(files[i]);
			status = ERROR;
		}
	}

	free(handles);

	return status;
}

void 
usage (void)
{
	printf( "%s %s: %s\n"
			"%s: %s [/A] [/I] %s\n\n"
			"  /A    %s\n"
			"  /I    %s\n"
			"  /?    %s\n"
			"\n",
		progName, version,
		kittengets(MSG_SET, MSG_DESCRIPTION_NUM, MSG_DESCRIPTION),
		kittengets(MSG_SET, MSG_USAGE_STR_NUM, MSG_USAGE_STR),
		progName,
		kittengets(MSG_SET, MSG_FILE_STR_NUM, MSG_FILE_STR),
	 	kittengets(MSG_SET, MSG_A_OPT_NUM, MSG_A_OPT),
		kittengets(MSG_SET, MSG_I_OPT_NUM, MSG_I_OPT),
		kittengets(MSG_SET, MSG_HELP_OPT_NUM, MSG_HELP_OPT)
	);
	exit (1);
}
