/*
    JPC: A x86 PC Hardware Emulator for a pure Java Virtual Machine
    Release Version 2.0

    A project from the Physics Dept, The University of Oxford

    Copyright (C) 2007 Isis Innovation Limited

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2 as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 
    Details (including contact information) can be found at: 

    www.physics.ox.ac.uk/jpc
*/


package org.jpc.debugger;

import java.util.*;
import java.io.*;
import java.lang.reflect.*;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.undo.*;

import org.jpc.debugger.util.*;
import org.jpc.emulator.*;
import org.jpc.emulator.processor.*;
import org.jpc.emulator.motherboard.*;
import org.jpc.emulator.memory.*;
import org.jpc.emulator.memory.codeblock.*;

public class ExecutionTraceFrame extends UtilityFrame implements PCListener, ListSelectionListener
{
    private MicrocodeOverlayTable trace;
    private TraceModel model;
    
    private CodeBlockRecord codeBlocks;
    private long selectedBlock;

    public ExecutionTraceFrame()
    {
        super("Execution Trace Frame");
        codeBlocks = null;
        selectedBlock = -1;

        model = new TraceModel();
        trace = new MicrocodeOverlayTable(model, 1, false);
        model.setupColumnWidths(trace);
        trace.getSelectionModel().addListSelectionListener(this);
        trace.getColumnModel().removeColumn(trace.getColumnModel().getColumn(1));

        add("Center", new JScrollPane(trace));

        setPreferredSize(new Dimension(500, 530));
        JPC.getInstance().objects().addObject(this);
        JPC.getInstance().refresh();
    }

    public void valueChanged(ListSelectionEvent e) 
    {
        if (codeBlocks == null)
            return;

        int r = trace.getSelectedRow();
        if (r >= 0)
            selectedBlock = codeBlocks.getIndexNumberForRow(r);
    }

    public void PCCreated() {}

    public void frameClosed()
    {
        JPC.getInstance().objects().removeObject(this);
    }

    public void PCDisposed() 
    {
        codeBlocks = null;
        model.fireTableDataChanged();
    }
    
    public void executionStarted() {}

    public void executionStopped()
    {
        refreshDetails();
    }

    public void refreshDetails() 
    {
        codeBlocks = (CodeBlockRecord) JPC.getObject(CodeBlockRecord.class);
        if (codeBlocks == null)
            return;

        model.fireTableDataChanged();
        if (selectedBlock < 0)
            return;
        
        int r2 = codeBlocks.getRowForIndex(selectedBlock);
        try
        {
            trace.setRowSelectionInterval(r2, r2);
        }
        catch (Exception e) {}
    }

    class TraceModel extends BasicTableModel
    {
        TraceModel()
        {
            super(new String[]{"Index", "Code Block", "Address", "X86 Length", "X86 Count", "Decimal address"}, new int[]{100, 400, 150, 150, 150, 150});
        }

        public int getRowCount()
        {
            if (codeBlocks == null)
                return 0;
            return codeBlocks.getMaximumTrace();
        }

        public Object getValueAt(int row, int column)
        {
            if (row >= codeBlocks.getTraceLength())
                return null;

            CodeBlock block = codeBlocks.getTraceBlockAt(row);
            switch (column)
            {
            case 0:
                return new Long(codeBlocks.getIndexNumberForRow(row));
            case 1:
                return block;
            case 2:
                return Integer.toHexString(codeBlocks.getBlockAddress(row)).toUpperCase();
            case 3:
                return new Integer(block.getX86Length());
            case 4:
                return new Integer(block.getX86Count());
            case 5:
                return codeBlocks.getBlockAddress(row);
            default:
                return "";
            }
        }
    }
}

