// CMOS.cpp
// Utility to kill and or read from/write to CMOS memory/data file.
// Author: Sami ur Rahman Qazi <qsami(at)brain.net.pk>

#include <ctype.h>
#include <dos.h>
#include <fstream.h>
#include <iostream.h>

/*
Code for reading and writing to CMOS was riped off from code
written by Christophe GRENIER grenier@cgsecurity.org.

To access the CMOS data array write the index to port# 0x0070,
then read/write the data from/to the port# 0x0071. For example,
to read the 4th byte in the CMOS array:

	_outp(0x0070, 3);      // <-- 0-based index
	char c = _inp(0x0071);

The code was compiled with DigitalMars C++ compiler.

*/

const CMOS_SIZE   = 3*0x80;
const PORT_CMOS_0 = 0x70;
const PORT_CMOS_1 = 0x71;
const PORT_CMOS_2 = 0x72;
const PORT_CMOS_3 = 0x73;
const PORT_CMOS_4 = 0x74;
const PORT_CMOS_5 = 0x75;

// Data file to read from/write to.
const char *DATAFILE = "CMOS.DAT";

class CMOS{
private:
	unsigned char cmos[CMOS_SIZE];

	unsigned char Read(int cell);
	void Write(int cell, unsigned char value);

public:
	CMOS();
	~CMOS(){ /* do nothing */ }
	int Save();
	int Restore();
	void Kill();
};

// Constructor just save the CMOS data in case ...
CMOS::CMOS()
{
    int i;

    for(i=0; i < CMOS_SIZE; i++)
        cmos[i] = Read(i);
}

unsigned char CMOS::Read(int cell)
{
    if(cell<128)
    {
        _outp(PORT_CMOS_0,cell);
        return _inp(PORT_CMOS_1);
    }
    if(cell<2*128)
    {
        _outp(PORT_CMOS_2,cell);
        return _inp(PORT_CMOS_3);
    }
    if(cell<3*128)
    {
        _outp(PORT_CMOS_4,cell);
        return _inp(PORT_CMOS_5);
    }
    return 0;
}

void CMOS::Write(int cell, unsigned char value)
{
	if(cell<128)
	{
	    _outp(PORT_CMOS_0,cell);
        _outp(PORT_CMOS_1,value);
        return;
    }
    if(cell<2*128)
    {
        _outp(PORT_CMOS_2,cell);
        _outp(PORT_CMOS_3,value);
        return;
    }
    if(cell<3*128)
    {
        _outp(PORT_CMOS_4,cell);
        _outp(PORT_CMOS_5,value);
        return;
    }
}

int CMOS::Save()
{
	ofstream file(DATAFILE, ios::binary|ios::out);

	if (file.fail())
	{
		cerr << "ERROR: Cannot create CMOS data file (" << DATAFILE << ")." << endl;
		return 0;
	}

	for(int i=0; i< CMOS_SIZE; i++)
		cmos[i] = Read(i);		// Read the CMOS data into the buffer.

	file.write((const char *)cmos, CMOS_SIZE);
	file.close();

	return 1;
}

int CMOS::Restore()
{
	ifstream file(DATAFILE, ios::binary|ios::in);

	if (file.fail())
	{
		cerr << "ERROR: CMOS data file (" << DATAFILE << ") not yet created/present." << endl;
		return 0;
	}

	file.read((char *)cmos, CMOS_SIZE);
	file.close();

	for(int i=0; i< CMOS_SIZE; i++)
		Write(i, cmos[i]);	// Write the buffer's contents to CMOS memory.

	return 1;
}

void CMOS:: Kill()
{
	int i;

	for (i=0;i< CMOS_SIZE;i++)
		Write(i,0);
}

int main()
{
	char ch;
	int quit = 0;
	CMOS nvram;

	cout << "CMOS ver 1.0 Copyright (C) 2003 Blue Byte Computer Services." << endl;
	while (!quit)
	{
		cout << endl;
		cout << "  [1] Save CMOS data to file" << endl;
		cout << "  [2] Load CMOS data from file" << endl;
		cout << "  [3] Kill COMOS :)" << endl;
		cout << "  [4] Quit" << endl;
		cout << endl;
		cout << "Choose an option (1-4): ";
		cin >> ch;
		cout << endl;

		switch(ch) {

		case '1':
			if (nvram.Save())
				cout << "CMOS data saved to file (" << DATAFILE << ")." << endl;
			break;
		case '2':
			cout << "WARNING: this will replace your CMOS configuration with the" << endl
					 << "one saved in " << DATAFILE << ". Do you want to continue [y/n]? ";
			cin >> ch;
			if (toupper(ch) != 'Y') break;
			if (nvram.Restore())
				cout << "CMOS data restored from file." << endl;
			break;
		case '3':
			cout << "WARNING: this will erase your CMOS configuration." << endl
			     << "Do you want to continue [y/n]? ";
			cin >> ch;
			if (toupper(ch) != 'Y') break;
			nvram.Kill();
			cout << "Remember to reboot and set date and time." << endl;
			break;
		case '4':
			quit = 1;
			break;
		default:
			cerr << "Unsa ka, bogo?" << endl;
		}
	}

	return 0;
}
