/*
 * Package database manipulation routines.
 * This file is part of fdnpkg
 * 
 * Copyright (C) Mateusz Viste 2012,2013
 */


#include <stdio.h>  /* printf(), puts()... */
#include <stdlib.h> /* malloc(), free()... */
#include <string.h> /* strcmp() */
#include <time.h>   /* time() */
#include "getdelim.h" /* a getdelim() implementation - there is none availble on DJGPP */
#include "kprintf.h"
#include "pkgdb.h"


/* Free the memory occupied by the linked list */
void freedb(struct pkgdb **db) {
  struct pkgdb *nextdb, *curdb;
  struct pkgrepo *nextrepo, *currepo;
  curdb = *db;
  while (curdb != NULL) {
    nextdb = curdb->nextpkg;
    /* free all possible repositories */
    currepo = curdb->repolist;
    while (currepo != NULL) {
      nextrepo = currepo->nextrepo;
      free(currepo);
      currepo = nextrepo;
    }
    free(curdb);
    curdb = nextdb;
  }
}


/* Create the initial stucture for the db (first node) */
struct pkgdb *createdb() {
  struct pkgdb *newentry;
  newentry = malloc(sizeof(struct pkgdb));
  if (newentry == NULL) return(NULL);
  newentry->repolist = NULL;
  newentry->nextpkg = NULL;
  return(newentry);
}


/* adds a new entry to the db - returns a pointer to the new entry on success, NULL otherwise */
static struct pkgdb *addpkg(struct pkgdb *db, char *pkgname, char *desc) {
  struct pkgdb *oldnextdb, *newentry;
  newentry = createdb();
  if (newentry == NULL) return(NULL);
  strncpy(newentry->name, pkgname, 8);
  newentry->name[8] = 0; /* strncpy doesn't terminate strings if limit reached */
  strncpy(newentry->desc, desc, 63);
  newentry->desc[63] = 0; /* strncpy doesn't terminate strings if limit reached */
  if (strlen(desc) > 63) { /* if desc was longer than 63 bytes, terminate it nicely */
    newentry->desc[60] = '.';
    newentry->desc[61] = '.';
    newentry->desc[62] = '.';
  }
  oldnextdb = db->nextpkg;
  db->nextpkg = newentry;
  newentry->nextpkg = oldnextdb;
  return(newentry);
}


/* adds a new repo to entry *db - returns the new entry address on success, NULL otherwise */
static struct pkgrepo *addrepo(struct pkgdb *pkg, unsigned char repo, const char *version) {
  struct pkgrepo *oldnextrepo, *newrepo;
  newrepo = malloc(sizeof(struct pkgrepo));
  if (newrepo == NULL) return(NULL);
  newrepo->nextrepo = pkg->repolist;
  strncpy(newrepo->version, version, 15);
  newrepo->version[15] = 0; /* strncpy doesn't terminate strings if limit reached */
  newrepo->repo = repo;
  oldnextrepo = pkg->repolist;
  pkg->repolist = newrepo;
  newrepo->nextrepo = oldnextrepo;
  return(newrepo);
}


/* Searches after a pkg named 'pkgname' in the db. Returns a pointer to the package if found, or NULL otherwise.
   If no package found, then lastmatch is set to the last alphabetically inferior package */
struct pkgdb *findpkg(struct pkgdb *db, char *pkgname, struct pkgdb **lastmatch) {
  struct pkgdb *curpkg;
  int cmpres;
  *lastmatch = db;
  for (curpkg = db->nextpkg; curpkg != NULL; curpkg = curpkg->nextpkg) {
    cmpres = strcasecmp(curpkg->name, pkgname);
    if (cmpres < 0) {    /* if alphabetically inferior, remember lastmatch */
        *lastmatch = curpkg;
      } else if (cmpres == 0) { /* if equal, then we found what we came for! */
        return(curpkg);
      } else { /* if superior, then there's no point in loosing more time here (assuming the list is sorted) */
        return(NULL);
    }
  }
  return(NULL);
}


static void getpkgmetadata(char *lineptr, char **pkgmeta) {
  int x = 0, y = 1, lastnonemptychar = 0;
  pkgmeta[0] = lineptr;
  pkgmeta[1] = lineptr;
  pkgmeta[2] = lineptr;
  while (lineptr[x] != 0) {
    /* check out what character we have, and remember if it's whitespace or not */
    switch (lineptr[x]) {
      case '\t': /* delimiters */
      case '\r':
      case '\n':
        lineptr[lastnonemptychar + 1] = 0;
        if (y < 3) pkgmeta[y] = &lineptr[x + 1];
        lastnonemptychar = x + 1;
        y++;
        if (y > 3) return;
        break;
      case ' ': /* whitespace... */
        break;
      default: /* remember last non-whitespace char */
        lastnonemptychar = x;
    }
    x++;
  }
}


int loaddb_fromcache(struct pkgdb *db, char *datafile, unsigned long crc32val, int maxcachetime) {
  time_t curtime;
  unsigned long uint32val;
  char *pkgname, *pkgdesc;
  struct pkgdb *pkg, *lastmatch;
  int tmpchar, cursor;
  unsigned char repo;
  FILE *fd;
  fd = fopen(datafile, "rb");
  if (fd == NULL) return(-1);
  uint32val = fgetc(fd);
  uint32val <<= 8;
  uint32val |= fgetc(fd);
  uint32val <<= 8;
  uint32val |= fgetc(fd);
  uint32val <<= 8;
  uint32val |= fgetc(fd);
  if (uint32val != crc32val) return(-1);
  curtime = time(NULL);
  uint32val = fgetc(fd);
  uint32val <<= 8;
  uint32val |= fgetc(fd);
  uint32val <<= 8;
  uint32val |= fgetc(fd);
  uint32val <<= 8;
  uint32val |= fgetc(fd);
  if (curtime < uint32val) return(-1); /* cache is from the future, we don't like this */
  if (curtime > uint32val + maxcachetime) return(-1); /* cache is too old */
  if (fgetc(fd) != 0x0A) return(-1); /* we expect a LF here.. */
  /* allocated some memory for doing stuff */
  pkgname = malloc(64);
  if (pkgname == NULL) return(-2);
  pkgdesc = malloc(1024);
  if (pkgdesc == NULL) {
    free(pkgname);
    return(-2);
  }
  for (;;) {
    /* read the first byte, to be able to determine whether EOF is reached */
    tmpchar = fgetc(fd);
    if (tmpchar < 0) break; /* EOF -> get out! */
    /* Get pkgname */
    for (cursor = 0;;) {
      if (cursor > 0) tmpchar = fgetc(fd);
      if (tmpchar == '\t') break; /* end of record */
      if ((tmpchar < 32) || (cursor > 62)) { /* invalid char or unexpected enf of file or pkgname too long == corruption */
        free(pkgname);
        free(pkgdesc);
        return(-1);
      }
      pkgname[cursor++] = tmpchar;
    }
    pkgname[cursor] = 0; /* terminate the pkgname string */
    /* Get pkgdesc */
    for (cursor = 0;;) {
      tmpchar = fgetc(fd);
      if (tmpchar == '\t') break; /* end of record */
      if ((tmpchar < 32) || (cursor > 1022)) { /* invalid char or unexpected enf of file or pkgdesc too long == corruption */
        free(pkgname);
        free(pkgdesc);
        return(-1);
      }
      pkgdesc[cursor++] = tmpchar;
    }
    pkgdesc[cursor] = 0; /* terminate the pkgdesc string */
    /* add the package entry into the db - SORTED (important!) */
    /* look if package already exists in db... we can't skip this even if the package won't ever be present in cache twice, because we use this also to sort packages (and find them faster later) */
    pkg = findpkg(db, pkgname, &lastmatch);
    /* if not -> add the package to the db first */
    if (pkg == NULL) {
      pkg = addpkg(lastmatch, pkgname, pkgdesc);
      if (pkg == NULL) { /* Out of memory! */
        kitten_puts(6, 2, "Error: Out of memory while loading package database!");
        free(pkgname);
        free(pkgdesc);
        return(-1);
      }
    }
    /* now retrieve repositories */
    do {
      /* Get repo */
      tmpchar = fgetc(fd);
      if (tmpchar < 0) { /* unexpected enf == corruption */
        free(pkgname);
        free(pkgdesc);
        return(-1);
      }
      repo = tmpchar;
      /* Get version */
      for (cursor = 0;;) {
        tmpchar = fgetc(fd);
        if (tmpchar == '\t') break; /* end of record */
        if (tmpchar == '\n') break; /* end of record (and this is the last one) */
        if ((tmpchar < 32) || (cursor > 62)) { /* invalid char or unexpected enf of file or pkgdesc too long == corruption */
          free(pkgname);
          free(pkgdesc);
          return(-1);
        }
        pkgname[cursor++] = tmpchar;
      }
      pkgname[cursor] = 0; /* terminate the pkgname string */
      /* add the repo to the list of repos for the current package */
      if (addrepo(pkg, repo, pkgname) == NULL) { /* Out of memory! */
        kitten_puts(6, 2, "Error: Out of memory while loading package database!");
        free(pkgname);
        free(pkgdesc);
        return(-1);
      }
    } while (tmpchar == '\t');
  }
  free(pkgname);
  free(pkgdesc);
  return(0);
}


/* Load the db from file - returns 0 on success, non-zero on failure.
   crc32val is only used when trying to load from cache (repo == NULL) */
int loaddb(struct pkgdb *db, char *datafile, unsigned char repo) {
  FILE *fd;
  char *lineptr;
  char *pkgmeta[3];
  struct pkgdb *pkg, *lastmatch;
  int result = 0;
  size_t getdelimcount = 0;
  int getdelimlen;

  fd = fopen(datafile, "r");
  if (fd == NULL) {
    kitten_printf(6, 3, "Error: Could not open the datafile at '%s'", datafile);
    return(-1);
  }

  /* Check whether the index file has a valid header */
  lineptr = NULL;
  getdelimlen = getdelim(&lineptr, &getdelimcount, '\n', fd);
  if (getdelimlen < 0) {
      free(lineptr);
      kitten_puts(6, 0, "Error: invalid index file (bad hdr)! Repository skipped.");
      result = -6;
    } else {
      getpkgmetadata(lineptr, pkgmeta); /* pkgmeta[0] = "FD-REPOv1" */
      if (strcmp(pkgmeta[0], "FD-REPOv1") != 0) {
          free(lineptr);
          kitten_puts(6, 1, "Error: invalid index file (bad sig)! Repository skipped.");
          result = -7;
        } else {
          /* start reading the index content */
          for (;;) {
            /* read line from file */
            lineptr = NULL;
            getdelimlen = getdelim(&lineptr, &getdelimcount, '\n', fd);
            if (getdelimlen < 0) {
              free(lineptr);
              break;
            }
            /* printf("Got line: %s\r\n", lineptr); */
            if ((lineptr[0] == 0) || (lineptr[0] == '\r') || (lineptr[0] == '\n')) continue; /* skip empty lines */
            /* get pkgname and version */ /* PkgName <tab> version <tab> description */
            getpkgmetadata(lineptr, pkgmeta); /* pkgmeta[0] = PkgName ; pkgmeta[1] = version ; pkgmeta[2] = description */
            /* look if package already exists in db... */
            pkg = findpkg(db, pkgmeta[0], &lastmatch);
            /* if not -> add the package to the db first */
            if (pkg == NULL) {
              pkg = addpkg(lastmatch, pkgmeta[0], pkgmeta[2]);
              if (pkg == NULL) {
                kitten_puts(6, 2, "Error: Out of memory while loading package database!");
                free(lineptr);
                result = -1;
                break;
              }
            }
            /* now add the repository to the package */
            if (addrepo(pkg, repo, pkgmeta[1]) == NULL) { /* ooops, something went wrong */
              kitten_puts(6, 2, "Out of memory while loading package database!");
              result = -1;
              free(lineptr);
              break;
            }
            free(lineptr); /* free the memory occupied by the line */
          }
      }
  }
  fclose(fd);
  return(result);
}


/* dumps the entire content of the db into a cache file */
void dumpdb(struct pkgdb *db, char *datafile, unsigned long crc32val) {
  FILE *fd;
  time_t curtime;
  struct pkgdb *curnode;
  struct pkgrepo *currepo;
  fd = fopen(datafile, "wb");
  if (fd == NULL) {
    kitten_printf(6, 4, "Warning: Could not open db cache file at %s!", datafile);
    puts("");
    return;
  }
  curtime = time(NULL);
  /* write the CRC32 of the configfile */
  fprintf(fd, "%c%c%c%c", (int)(crc32val >> 24) & 0xFF, (int)(crc32val >> 16) & 0xFF, (int)(crc32val >> 8) & 0xFF, (int)(crc32val & 0xFF));
  /* write the current time, followed by a LF character */
  fprintf(fd, "%c%c%c%c%c", (int)(curtime >> 24) & 0xFF, (int)(curtime >> 16) & 0xFF, (int)(curtime >> 8) & 0xFF, (int)(curtime & 0xFF), 0x0A);
  /* start dumping records (PKGNAME, PKGDESC, REPO1, VER1, REPO2, VER2, ..., LF) */
  for (curnode = db->nextpkg; curnode != NULL; curnode = curnode->nextpkg) {
    fprintf(fd, "%s\t%s", curnode->name, curnode->desc);
    for (currepo = curnode->repolist; currepo != NULL; currepo = currepo->nextrepo) {
      fprintf(fd, "\t%c%s", currepo->repo, currepo->version);
    }
    fprintf(fd, "%c", 0x0A);
  }
  fclose(fd);
}
