/*
 * This file is part of fdnpkg
 * Copyright (C) Mateusz Viste 2012
 */

#include <stdio.h>
#include <string.h>    /* strlen() */
#include <stdlib.h>    /* free() */
#include <unistd.h>    /* rmdir(), unlink() */
#include "fileexst.h"
#include "getdelim.h"
#include "helpers.h"   /* slash2backslash() */
#include "kprintf.h"
#include "pkgrem.h"
#include "rtrim.h"


void pkgrem(char *pkgname, char *dosdir) {
  char fpath[512];
  char shellcmd[512];
  char *lineptr;
  FILE *flist;
  int getdelimlen;
  size_t getdelimcount = 0;
  #define DIRLIST_MAX 256
  char *dirlist[DIRLIST_MAX]; /* used to remember directories to remove */
  int x, dirlist_count;

  /* set all dirlist pointers to NULL */
  for (x = 0; x < DIRLIST_MAX; x++) dirlist[x] = NULL;

  /* Check if the file %DOSDIR%\packages\pkgname.lst exists (if not, the package is not installed) */
  sprintf(fpath, "%s\\packages\\%s.lst", dosdir, pkgname);
  if (fileexists(fpath) == 0) { /* file does not exist */
    kitten_printf(4, 0, "Package %s is not installed, so not removed.", pkgname);
    puts("");
    return;
  }

  /* open the file %DOSDIR%\packages\pkgname.lst */
  flist = fopen(fpath, "r");
  if (flist == NULL) {
    kitten_puts(4, 1, "Error opening lst file!");
    return;
  }

  /* remove all files/folders listed in pkgname.lst but NOT pkgname.lst */
  x = 0; /* used to count stored directories */
  for (;;) {
    /* read line from file */
    lineptr = NULL;
    getdelimlen = getdelim(&lineptr, &getdelimcount, '\n', flist);
    if (getdelimlen < 0) {
      free(lineptr);
      break;
    }
    rtrim(lineptr);  /* right-trim the filename */
    slash2backslash(lineptr); /* change all / to \ */
    if ((lineptr[0] == 0) || (lineptr[0] == '\r') || (lineptr[0] == '\n')) continue; /* skip empty lines */
    if (lineptr[strlen(lineptr) - 1] == '\\') { /* it's a directory */
        /* remember it for later - I must remove directory last */
        if (x >= DIRLIST_MAX) {
            kitten_printf(4, 2, "Dirlist limit reached. Directory %s won't be removed!", lineptr);
            puts("");
          } else {
            lineptr[strlen(lineptr) - 1] = 0; /* remove the ending \ */
            dirlist[x] = malloc(strlen(lineptr) + 1);
            if (dirlist[x] == NULL) { /* malloc() failed */
                kitten_printf(4, 3, "Out of memory! Could not store directory %s!", lineptr);
                puts("");
              } else {
                sprintf(dirlist[x], "%s", lineptr);
                x++;
            }
        }
      } else { /* it's a file */
        char pkglistfile[512];
        sprintf(pkglistfile, "packages\\%s.lst", pkgname);
        if (strcasecmp(pkglistfile, lineptr) != 0) { /* never delete pkgname.lst at this point - it will be deleted later */
          sprintf(shellcmd, "%s\\%s", dosdir, lineptr);
          kitten_printf(4, 4, "removing %s", shellcmd);
          puts("");
          unlink(shellcmd);
        }
    }
    free(lineptr); /* free the memory occupied by the line */
  }
  dirlist_count = x;

  /* close the file */
  fclose(flist);

  /* iterate through dirlist and remove directories if empty, from longest to shortest */
  while (dirlist_count > 0) {
    int longestdir = 0;
    size_t longestdirlen = strlen(dirlist[longestdir]);
    for (x = 0; x < DIRLIST_MAX; x++) {
      if (dirlist[x] != NULL) {
        if (strlen(dirlist[x]) > longestdirlen) {
          longestdir = x;
          longestdirlen = strlen(dirlist[longestdir]);
        }
      }
    }
    sprintf(shellcmd, "%s\\%s", dosdir, dirlist[longestdir]);
    rmdir(shellcmd);
    /* printf("Exec: %s\r\n", shellcmd); */
    free(dirlist[longestdir]);  /* free the allocated memory for this string */
    dirlist[longestdir] = NULL; /* set to NULL to avoid redoing it again */
    dirlist_count -= 1;
    /* system(shellcmd); */
  }

  /* remove %DOSDIR%\packages\pkgname.lst */
  unlink(fpath);
  kitten_printf(4, 5, "Package %s has been removed.", pkgname);
  puts("");
}
