(* ---------------------------------------------------------------
Title         Q&D Calendar
Author        PhG
Overview      see help
Notes
Bugs
Wish List     (nth day of week, etc.)
              add warning # days before or after event ?
              improve reminder display (colors, etc.)
              calendar by day (weird idea seen in yearcal.c by a Paul Sittler)
              other calendars ?
              keyboard pause (bah, redirection was not created for nothing)

              check against current date, show future events among months,
              show jj-mm (or mm-jj if US format)

--------------------------------------------------------------- *)

MODULE Cal;

IMPORT Lib;
IMPORT Str;
IMPORT FIO;
IMPORT MATHLIB;

FROM QD_ASCII IMPORT dash, slash, nullchar, tabchar, (* cr, lf, nl, bs, *)
space, dot, deg, doublequote, quote, colon, percent, vbar,
blank, equal, dquote, charnull, singlequote, antislash, dollar,
star, backslash, coma, question, underscore, tabul, hbar,
comma, semicolon, diese, pound, openbracket, closebracket, tilde, exclam,
stardotstar, dotdot, escCh, escSet, letters, digits,
lettersUpp, lettersLow, openbrace, closebrace;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits,
getAllLegalUnits, metaproc, getCli, argc, argv;

FROM QD_Text IMPORT
colortype, cursorshapetype, scrolltype,
ff, cr, lf, bs, tab, nl, mincolor, maxcolor,
BW40, CO40, BW80, CO80, CO80x43, CO80x50, MONO,
vesa80x60, vesa132x25, vesa132x43, vesa132x50, vesa132x60,
selectCursorEmulation,
setCursorShape,
handleVesa, setBrightPaper, setBlinkMode,
setFillChar, setFillInk, setFillPaper,
setFillInkPaper, setTxtInk, setTxtPaper, setTxtInkPaper, setWrapMode,
setUseBiosMode, setTabWidth, getScreenData, setWindow,
setMode, restoreMode,
gotoXY, xyToHtabVtab, home, setVisualPage, setActivePage,
scrollWindow, fillTextAttr, cls, writeStr, writeLn, getScreenWidth,
getScreenHeight, getScreenMinX, getScreenMinY, getScreenMaxX, getScreenMaxY,
getMinHtab, getMaxHtab, getMinVtab, getMaxVtab, getUseBiosMode,
getMinHtab, getMaxHtab, getMinVtab, getMaxVtab, getHtab, getVtab,
getWindowWidth, getWindowHeight,
initScreenConsole,
findInkPaperAtStartup, getInkAtStartup, getPaperAtStartup,
setFullScreenWindow, gotoFullScreenXY;

(* ------------------------------------------------------------ *)

CONST
    (* aliases for our own procedures *)
    WrStr  ::= writeStr;
    WrLn   ::= writeLn;
    WrChar ::= writeStr;

PROCEDURE colorhelp ();
BEGIN
    setTxtInk(getInkAtStartup());      (* was green *)
    setTxtPaper(getPaperAtStartup());  (* was black *)
    setFillInkPaper(getInkAtStartup(),getPaperAtStartup());
END colorhelp;

PROCEDURE color (ink,paper:colortype);
BEGIN
    setTxtInk   (ink);
    setTxtPaper (paper);
    setFillInkPaper(getInkAtStartup(),getPaperAtStartup());
END color;

(* ------------------------------------------------------------ *)

CONST
    extEXE        = ".EXE";
    extINI        = ".INI";
    cHilightleft  = ">";
    cHilightright = "<";
    cSepEvent     = ":";
    cTodayEvent   = "*";
    strMoreEvents = "...";
    strMaxEvent   = "400"; (* see maxevent infra *)
    (* same order as in QD_TEXT.DEF ! *)
    strcolorsdark  ="black,darkblue,darkgreen,darkcyan,darkred,darkmagenta,brown,gray,";
    strcolorsbright="darkgray,blue,green,cyan,red,magenta,yellow,white";
CONST
    progEXEname   = "CAL";
    progTitle     = "Q&D Calendar";
    progVersion   = "v1.1d";
    progCopyright = "by PhG";
    Banner        = progTitle+" "+progVersion+" "+progCopyright;
CONST
    errNone         = 0;
    errHelp         = 1;
    errUnknownOpt   = 2;
    errTooManyParms = 3;
    errNumber       = 4;
    errInk          = 5;
    errPaper        = 6;
    errYear         = 7;
    errMonth        = 8;
    errBadToday     = 9;
    errJulian       = 10;
    errEvent        = 11;
    errCmd          = 12;
    errNoEvent      = 13;

(* ------------------------------------------------------------ *)

PROCEDURE dbg ( ok,enquote:BOOLEAN; S1,S2,S3:ARRAY OF CHAR  );
BEGIN
    IF ok THEN
        WrStr("/// ");WrStr(S1);
        WrStr("::: ");WrStr(S2);
        WrStr(" : ");
        IF enquote THEN WrStr(dquote);END;
        WrStr(S3);
        IF enquote THEN WrStr(dquote);END;
        WrLn;
    END;
END dbg;

TYPE
    searchresulttype = (foundincurrentdir,foundinexedir,foundnowhere);

PROCEDURE whereIsIni (VAR ini:ARRAY OF CHAR   ):searchresulttype;
VAR
    p:CARDINAL;
    zeini:str128;
    rc:searchresulttype;
BEGIN
    Str.Concat(ini,progEXEname,extINI);
    IF FIO.Exists(ini) THEN
        rc:=foundincurrentdir;
    ELSE
        Lib.ParamStr(zeini,0);
        Str.Subst(zeini,extEXE,extINI); (* should always work but... *)
        p:=Str.RCharPos(zeini,dot);
        IF p # MAX(CARDINAL) THEN zeini[p]:=nullchar; END; (* brutal ! *)
        Str.Concat(ini,zeini,extINI);
        IF FIO.Exists(ini) THEN
            rc:=foundinexedir;
        ELSE
            rc:=foundnowhere;
        END;
    END;
    RETURN rc;
END whereIsIni;

(* ------------------------------------------------------------ *)

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
    helpmsg =
Banner+nl+
nl+
"Syntax 1 : "+progEXEname+" [option]..."+nl+
"Syntax 2 : "+progEXEname+" <year|*|+[+]|-[-]> [option]..."+nl+
"Syntax 3 : "+progEXEname+" <month> <year|*> [option]..."+nl+
nl+
"This program prints a monthly or yearly calendar."+nl+
nl+
"  -r      ignore existing "+progEXEname+extINI+" (forced by yearly calendar)"+nl+
"  -a      alternate events display (forced by -j[j] and -n)"+nl+
"  -![!]   create default "+progEXEname+extINI+" in current directory then terminate"+nl+
"  -e      list all events in "+progEXEname+extINI+" then terminate"+nl+
"  -y|-xx  alternate fix to year when parsing an event date"+nl+
"  -s      assume first day of week is Sunday (default is Monday)"+nl+
"  -u[k|s] English texts (default is French texts)"+nl+
"  -x|-p   special calendar"+nl+
"  -j[j]   show last four digits of Julian Day (-jj = show full JD in header)"+nl+
"  -n      show day in year (base is January 1st)"+nl+
"  -t:??   left and right markers to highlight today in monthly calendar"+nl+
'          (default is "'+cHilightleft+cHilightright+'")'+nl+
"  -q      do not highlight today"+nl+
"  -i?:#   ink [0..15] (im=main, it=title, id=days, ic=current day, ij=julian)"+nl+
"  -p?:#   paper [0..15] (pm=main, pt=title, pd=days, pc=current day, pj=julian)"+nl+
"  -b      monochrome BIOS output (no colors)"+nl+
nl+
"a) Month is [1..12], full month name or first three letters of month."+nl+
"b) Calendar is assumed julian for dates before October 15, 1582 :"+nl+
"   note this was not true in the whole Western world"+nl+
"   (a few countries did not adopt gregorian calendar until September, 1752 !)."+nl+
"c) Yearly calendars with any of -n or -j[j] options should be redirected."+nl+
"d) Julian Day begins at 12:00:00."+nl+
'e) "*" parameter stands for current year,'+nl+
'   "+[+]" for next month and "-[-]" for previous month.'+nl+
"f) Dark [0..7] : black, blue, green, cyan, red, magenta, brown and gray."+nl+
"   Bright [8..15] : gray, blue, green, cyan, red, magenta, yellow and white."+nl+
"g) Ink and paper may also be specified using the following tokens :"+nl+
"   "+strcolorsdark+nl+
"   "+strcolorsbright+"."+nl+
"h) "+progEXEname+extINI+" events file is searched for first in current directory,"+nl+
"   then in executable directory : it may contain up to "+strMaxEvent+" events."+nl+
"i) Default monthly calendar may not display all events (side by side view) :"+nl+
'   if final event line is "'+strMoreEvents+'", -a option is recommended.'+nl+
"j) -e option will list events using language format defined in "+progEXEname+extINI+"."+nl+
"k) When parsing an event date, program defaults to the following rule :"+nl+
"   it will add 2000 to year [0..79] and 1900 to year [80..99]."+nl+
"   Should -y option be specified, it will add 1900 to year [0..99]."+nl+
"l) -![!] and -e options ignore any other option or parameter."+nl+
"m) ~|~."+nl; (* yes, 3 chars marker ! *)
(*
nl+
"Examples : "+progEXEname+nl+
"           "+progEXEname+" *"+nl+
"           "+progEXEname+" 1987"+nl+
"           "+progEXEname+" fvrier 1963"+nl;
*)

VAR
    S : str256;
    (* used for errEvent *)
    p : CARDINAL;
    Z,ini : str128;
    hugestr:str4096;
BEGIN
    colorhelp;
    CASE e OF
    | errHelp :
        Z:=progEXEname+extINI;
        CASE whereIsIni(ini) OF
        | foundincurrentdir:
            Str.Append(Z," was found in current directory");
        | foundinexedir:
            Str.Append(Z," was found in executable directory");
        | foundnowhere:
            Str.Append(Z," was not found in either current or executable directory");
        END;
        Str.Copy(hugestr,helpmsg);
        Str.Subst(hugestr,tilde+vbar+tilde,Z);
        WrStr(hugestr);
    | errUnknownOpt :
        Str.Concat(S,"Unknown ",einfo); Str.Append(S," option !");
    | errTooManyParms :
        Str.Concat(S,"Unexpected ",einfo);Str.Append(S," parameter !");
    | errNumber:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," number !");
    | errInk:
        S := "Ink range is [0..15] !";
    | errPaper:
        S := "Paper range is [0..15] !";
    | errYear:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," year !");
    | errMonth:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," month !");
    | errBadToday:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," markers string !");
    | errJulian:
        S := "-j[j] and -n options are mutually exclusive !";
    | errEvent:
        S:="Problem at line | in | events file !";
        p:=Str.CharPos(einfo,vbar);
        Str.Slice(Z,einfo,0,p);
        Str.Subst(S,vbar,Z);
        Str.Copy(Z,einfo);
        Str.Delete(Z,0,p+1);
        Str.Subst(S,vbar,Z);
    | errCmd:
        S := "-![!] and -e options are mutually exclusive !";
    | errNoEvent:
        S := "No event to list, for "+progEXEname+extINI+" either does not exist or is empty !";
    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE  e OF
    | errNone, errHelp : ;
    ELSE
        WrStr(progEXEname+" : "); WrStr(S); WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

(* ripped/adapted from frozen QD_SKY *)

(* ---------------------------------------------------------------
   input  : value
   output : int(value) ie int(0.95) gives 0 (no rounding)

   TRUNC(n) gives only a CARDINAL without rounding either
--------------------------------------------------------------- *)

PROCEDURE int (v : LONGREAL) : LONGREAL;
VAR
    tmp : LONGINT; (* this should be enough! *)
BEGIN
    tmp := VAL(LONGINT,v); (* as intended, does NOT perform any rounding *)
    RETURN VAL(LONGREAL,tmp);
END int;

(* ---------------------------------------------------------------
   input  : value
   output : frac(value)

   frac keeps sign
--------------------------------------------------------------- *)

PROCEDURE frac (v : LONGREAL) : LONGREAL;
BEGIN
    RETURN (v - int (v));
END frac;

(* ---------------------------------------------------------------
   input  : value1, value2
   output : value1 mod value2

   mod keeps sign
--------------------------------------------------------------- *)

PROCEDURE mod (v1, v2 : LONGREAL) : LONGREAL;
BEGIN
    RETURN ( MATHLIB.Mod(v1, v2) );
END mod;

(* ---------------------------------------------------------------
   assume day with decimal, and whole date is ok
   valid only for JD >= 0 i.e. NOT before 1.5 Jan -4712 !
   input  : first gregorian date (yyyymmdd), day (with decimals), month, year
   output : JD
   thanks to Meeus !
--------------------------------------------------------------- *)

PROCEDURE dateToJD (Day,Month,Year:LONGREAL) : LONGREAL;
VAR
    YYYYMMDD  : LONGREAL;
    A,gregorian : LONGREAL; (* gregorian correction *)
    JD        : LONGREAL;
BEGIN
    YYYYMMDD := Year * 10000.0 + Month * 100.0 + Day;
    IF Month < 2.5 THEN
        Year  := Year - 1.0;
        Month := Month + 12.0;
    END;
    IF YYYYMMDD < LONGREAL(15821015.0) THEN (* before 15 october 1582, julian calendar *)
        gregorian := 0.0;
    ELSE
        A         := int (Year / 100.0);
        gregorian := 2.0 - A + int (A / 4.0);
    END;
    JD := int (365.25 * (Year + 4716.0)) + int (30.6001 * (Month + 1.0));
    JD := JD + Day + gregorian - 1524.5;
    RETURN JD;
END dateToJD;

(* ---------------------------------------------------------------
   assume JD >= 0
   input  : JD
   output : day (with decimals), month, year
--------------------------------------------------------------- *)

PROCEDURE JDtoDate (JD : LONGREAL; VAR Day,Month,Year : LONGREAL);
VAR
    Z,F,A,alf,B,C,D,E:LONGREAL;
BEGIN
    JD := JD + 0.5;
    Z := int(JD);
    F := frac(JD);
    IF Z < 2299161.0 THEN
        A := Z;
    ELSE
        alf := int( (Z - 1867216.25) / 36524.25);
        A := Z + 1.0 + alf - int( alf / 4.0 );
    END;
    B := A + 1524.0;
    C := int ( (B - 122.1) / 365.25);
    D := int ( 365.25 * C );
    E := int ( (B - D) / 30.6001 );
    Day := B - D - int( 30.6001 * E) + F;
    IF E < 13.5 THEN
        Month := E - 1.0;
    ELSE
        Month := E - 13.0;
    END;
    IF Month > 2.5 THEN
        Year := C - 4716.0;
    ELSE
        Year := C - 4715.0;
    END;
END JDtoDate;

(* ---------------------------------------------------------------
   find JD at 0h
   input  : JD
   output : JD for this day at 0H
--------------------------------------------------------------- *)

PROCEDURE JDtoJD0H (JD : LONGREAL) : LONGREAL;
VAR
    JD0H : LONGREAL;
BEGIN
    IF frac(JD) < 0.5 THEN
       JD0H := int(JD) - 0.5; (* was < 23H59 so take previous 0H *)
    ELSE
       JD0H := int(JD) + 0.5; (* was 0H or later so take current 0H *)
    END;
    RETURN JD0H;
END JDtoJD0H;

(* ---------------------------------------------------------------
   0=sunday, 1=monday... 6=saturday
   input  : JD
   output : day of week
--------------------------------------------------------------- *)

CONST
    firstdayofweek = 0; (* dimanche i.e. sunday *)
    lastdayofweek  = 6; (* samedi i.e. saturday *)

PROCEDURE JDtoWeekday (JD : LONGREAL) : CARDINAL;
VAR
    v : LONGREAL;
BEGIN
    v := mod ( (JDtoJD0H (JD) + 1.5), 7.0 );
    RETURN ( VAL(CARDINAL, v ) );
END JDtoWeekday;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

(* assume n is [1.. *)

PROCEDURE getElem (n:CARDINAL;delimiters,elements:ARRAY OF CHAR):str80;
VAR
    R:str80;
BEGIN
    Str.ItemS(R, elements, delimiters,n-1);
    RETURN R;
END getElem;

PROCEDURE inrange (n,mini,maxi:CARDINAL):BOOLEAN;
BEGIN
    IF n < mini THEN RETURN FALSE; END;
    IF n > maxi THEN RETURN FALSE; END;
    RETURN TRUE;
END inrange;

PROCEDURE fmtbignum (v:LONGCARD; base:CARDINAL;wi:INTEGER; pad:CHAR ):str16;
VAR
    S:str16;
    ok:BOOLEAN;
    i : CARDINAL;
BEGIN
    Str.CardToStr( v, S, base,ok);
    IF base=16 THEN Str.Lows(S);END;
    FOR i:=Str.Length(S)+1 TO ABS(wi) DO
         IF wi < 0 THEN
             Str.Append(S,pad);
         ELSE
             Str.Prepend(S,pad);
         END;
    END;
    RETURN S;
END fmtbignum;

PROCEDURE fmtnum ( v:CARDINAL; base:CARDINAL;wi:INTEGER; pad:CHAR ):str16;
BEGIN
    RETURN fmtbignum( LONGCARD(v),base,wi,pad);
END fmtnum;

PROCEDURE str2card (VAR n:CARDINAL;S:ARRAY OF CHAR):BOOLEAN;
VAR
    lc:LONGCARD;
    ok:BOOLEAN;
    base:CARDINAL;
BEGIN
    IF Str.Match(S,"$*") THEN
        base:=16;
        Str.Delete(S,0,1);
    ELSIF Str.Match(S,"0X*") THEN
        base:=16;
        Str.Delete(S,0,2);
    ELSE
        base:=10;
    END;
    lc:=Str.StrToCard(S,base,ok);
    IF NOT(ok) THEN RETURN FALSE; END;
    IF lc > MAX(CARDINAL) THEN RETURN FALSE; END;
    n:=CARDINAL(lc);
    RETURN TRUE;
END str2card;

(*
    gregorian calendar rule : leap 366 years if divisible by 4
    but centurial years NOT divisible by 4 are common
    Leap years are divisible by 400, or by 4 and not by 100
    1900 is common, 2000 is leap
*)

PROCEDURE getDaysInFebruary (annee:CARDINAL):CARDINAL;
CONST
    common = 28;
    leap   = 29;
BEGIN
    IF (annee MOD 400) = 0 THEN RETURN leap; END;
    IF (((annee MOD 4) = 0) AND ((annee MOD 100) # 0)) THEN RETURN leap; END;
    RETURN common;
END getDaysInFebruary;

(* assume month and year are correct *)

PROCEDURE getDaysInMonth (mois,annee:CARDINAL):CARDINAL;
TYPE
    daysinmonthtype = ARRAY [1..12] OF CARDINAL;
CONST
    (*                    JanFevMarAprMayJunJulAugSepOctNovDec *)
    (*                     1  2  3  4  5  6  7  8  9 10 11 12  *)
    dpm = daysinmonthtype(31, 0,31,30,31,30,31,31,30,31,30,31);
VAR
    m : CARDINAL;
BEGIN
    m:= dpm[mois];
    IF m=0 THEN m:=getDaysInFebruary(annee);END;
    RETURN m;
END getDaysInMonth;

(* ------------------------------------------------------------ *)

TYPE
    languagetype = (french,english);
    daytype      = (sunday,monday);
CONST
    abrevjours   = "Di Lu Ma Me Je Ve Sa";
    abrevjoursUS = "Su Mo Tu We Th Fr Sa";
    abrevmois    = "jan fv mar avr mai jui jul ao sep oct nov dc";
    abrevmoisUS  = "jan feb mar apr may jun jul aug sep oct nov dec";
    mois         = "Janvier Fvrier Mars Avril Mai Juin "+
                   "Juillet Aot Septembre Octobre Novembre Dcembre";
    moisUS       = "January February March April May June "+
                   "July August September October November December";
CONST
    firstmonth   = 1;  (* we may get 1-1 *)
    lastmonth    = 12;
    firstweekday = 1;
    lastweekday  = 7;

(* ------------------------------------------------------------ *)

(*
    when safety was 2, we lost time believing into a QD_TEXT bug
    while maxCol was just too short for /v /j display !
*)

CONST
    safety    = 8;                   (* was 2, paranoiac ! *)
    minRow    = 1;                   (* must be 1 *)
    maxRow    = (1+1+6+1)+safety;
    minCol    = 1;                   (* must be 1 *)
    maxCol    = 3*(1+7*3+1)+safety;  (* "## " 2+1 *)
    (* maxCol    = 1+7*10+1+safety;     (* "[####] ## " 6+1+3 *) *)
VAR
    hline     : ARRAY [1..3],[minRow..maxRow] OF str80; (* will do for year (1+7*3+1) * 3 *)
    inkattr   : ARRAY [1..3],[minRow..maxRow],[minCol..maxCol] OF colortype;
    paperattr : ARRAY [1..3],[minRow..maxRow],[minCol..maxCol] OF colortype;

PROCEDURE storeline (ndx,n:CARDINAL;ink,paper:colortype;S:ARRAY OF CHAR);
VAR
    i:CARDINAL;
BEGIN
    Str.Copy( hline[ndx,n],S);
    FOR i:=1 TO Str.Length(S) DO
        inkattr[ndx,n,i]:=ink;
        paperattr[ndx,n,i]:=paper;
    END;
END storeline;

PROCEDURE appendline (ndx,n:CARDINAL;ink,paper:colortype;S:ARRAY OF CHAR);
VAR
    i,len:CARDINAL;
BEGIN
    len:=Str.Length( hline[ndx,n] );
    Str.Append( hline[ndx,n],S);
    FOR i:=1 TO Str.Length(S) DO
        inkattr[ndx,n,len+i]:=ink;
        paperattr[ndx,n,len+i]:=paper;
    END;
END appendline;

PROCEDURE storeattr (p,wi,ndx,n:CARDINAL;ink,papercolor:colortype);
VAR
    i:CARDINAL;
BEGIN
    FOR i:= p TO (p+wi-1) DO
        inkattr[ndx,n,i]:=ink;
        paperattr[ndx,n,i]:=papercolor;
    END;
END storeattr;

PROCEDURE storechar (ndx,n,i:CARDINAL;ink,paper:colortype;ch:CHAR);
BEGIN
    hline[ndx,n,i]:=ch;
    INC(i); (* string is 0-based while attr arrays are 1-based *)
    inkattr[ndx,n,i]:=ink;
    paperattr[ndx,n,i]:=paper;
END storechar;

PROCEDURE retrievechar (ndx,n,i:CARDINAL):CHAR;
BEGIN
    RETURN hline[ndx,n,i];
END retrievechar;

PROCEDURE dmpline (ndx,n:CARDINAL);
VAR
    i:CARDINAL;
BEGIN
    FOR i:=1 TO Str.Length( hline[ndx,n] ) DO
        color ( inkattr[ndx,n,i], paperattr[ndx,n,i] );
        WrStr ( hline[ndx,n,i-1]); (* 0-based *)
    END;
    WrLn;
END dmpline;

PROCEDURE dmp(WHICH,currline:CARDINAL);
VAR
    i:CARDINAL;
BEGIN
    i:=minRow-1;
    LOOP
        INC(i);
        IF i > currline THEN EXIT; END;
        dmpline(WHICH,i);
    END;
    colorhelp; (* restore default just in case *)
END dmp;

PROCEDURE merge (ndst,nsrc,idst,isrc:CARDINAL;ink,paper:colortype):CARDINAL;
VAR
    i,last,nextpos,len:CARDINAL;
    S:str80;
BEGIN
    IF ndst < nsrc THEN
        last:=nsrc;
        FOR i:= ndst+1 TO last DO
            S:="";
            FOR len:=1 TO Str.Length(hline[idst,minRow]) DO Str.Append(S,space);END;
            storeline(idst,i,ink,paper,S);
        END;
    ELSIF nsrc < ndst THEN
        last:=ndst;
        FOR i:= nsrc+1 TO last DO
            S:="";
            FOR len:=1 TO Str.Length(hline[isrc,minRow]) DO Str.Append(S,space);END;
            storeline(isrc,i,ink,paper,S);
        END;
    ELSE
        last:=nsrc;
    END;

    (* now merge columns and attributes *)

    FOR i:=minRow TO last DO
        Str.Append(hline[idst,i],space);

        nextpos:=Str.Length(hline[idst,i]);

        FOR len:=1 TO Str.Length(space) DO
            inkattr  [idst,i,nextpos]:=ink;
            paperattr[idst,i,nextpos]:=paper;
            INC(nextpos);
        END;

        Str.Append(hline[idst,i], hline[isrc,i]);

        FOR len:=1 TO Str.Length(hline[isrc,i]) DO
            inkattr  [idst,i,nextpos]:=inkattr[isrc,i,len];
            paperattr[idst,i,nextpos]:=paperattr[isrc,i,len];
            INC(nextpos);
        END;
    END;
    RETURN last;
END merge;

PROCEDURE centerstr (wi:CARDINAL;pad:CHAR; VAR R:ARRAY OF CHAR);
VAR
    i,len,op:CARDINAL;
BEGIN
    len:=Str.Length(R);
    op:=0;
    FOR i:=len TO wi DO
        IF Str.Length(R) < wi THEN
            IF op=0 THEN
                Str.Append(R,pad);
            ELSE
                Str.Prepend(R,pad);
            END;
            INC(op);
            op:=op MOD 2;
        END;
    END;
END centerstr;

PROCEDURE getfmtparms (programmer:BOOLEAN;
                      VAR base,Ydigits,digits:CARDINAL;VAR padchar:CHAR);
BEGIN
    IF programmer THEN
        base:=16; Ydigits := 4; digits := 2; padchar:="0";
    ELSE
        base:=10; Ydigits := 1; digits := 2; padchar:=" ";
    END;
END getfmtparms;

PROCEDURE getlngparms (language:languagetype; VAR TM,TJ:ARRAY OF CHAR);
BEGIN
    CASE language OF
    | french:  Str.Copy(TM,mois);   Str.Copy(TJ,abrevjours);
    | english: Str.Copy(TM,moisUS); Str.Copy(TJ,abrevjoursUS);
    END;
END getlngparms;

(* too many IFs force an isl342 if we use small model *)

PROCEDURE buildmonth (programmer,julian,showfullJD,numbered:BOOLEAN;
                     WHICH,daynow,monthnow,yearnow, month,year:CARDINAL;
                     language:languagetype;weekstart:daytype;
                     hilightleft,hilightright:CHAR;
                     inkmain,papermain,inktitle,papertitle,
                     inkdays,paperdays,inknow,papernow,
                     inkjulian,paperjulian:colortype):CARDINAL;
CONST
    sepjulian      = " -- "; (* title *)

    wijulian       = 4;
    wipadjulian    = wijulian+2+1;   (* +2=open/close +1=trailing space *)
    padjulian      = " "+"    "+" "+space; (* "(####) " *)

    widiy          = 3;
    wipaddiy       = widiy+2+1;      (* +2=open/close +1=trailing space *)
    paddiy         = " "+"   " +" "+space; (* "(###) " *)

    wiquantieme    = 2;
    wipadquantieme = wiquantieme+1;
    padquantieme   = "  "+space;
    openjulian     = "[";
    closejulian    = "]";
    opennumbered   = "{";
    closenumbered  = "}";
VAR
    TM,TJ : str128; (* str80 would not be enough for months *)
    T,D,S:str80;
    currline,i,count,wical,baseJDofWeek,dow:CARDINAL;
    hereline,herepos: CARDINAL;
    JDYEAR,JD,jj,mm,yy:LONGREAL;
    state:(waiting,inmonth,done);
    base,Ydigits,digits:CARDINAL;
    padchar:CHAR;
    vcurr,currjdbase,currjd:LONGCARD;
    p,ii : CARDINAL;
BEGIN
    getfmtparms(programmer, base,Ydigits,digits,padchar);
    getlngparms(language, TM, TJ );

    (* create days header *)

    CASE weekstart OF
    | sunday: baseJDofWeek:=firstdayofweek;
    | monday: baseJDofWeek:=firstdayofweek+1;
    END;
    i:=baseJDofWeek; (* init first day *)
    Str.Copy(D,space);
    FOR count:=firstweekday TO lastweekday DO
        IF julian THEN
            Str.Append(D,padjulian);
        ELSIF numbered THEN
            Str.Append(D,paddiy);
        END;
        Str.Append(D,getElem(i+1,space,TJ));
        Str.Append(D,space);
        INC(i);
        IF i > lastdayofweek THEN i:=firstdayofweek;END;
    END;
    wical:=Str.Length(D);

    (* first day of month : JD stuff now *)

    JD:=dateToJD(LONGREAL( 1 ),LONGREAL(month),LONGREAL(year));
    currjd    := LONGCARD(JD); INC(currjd);

    (* create title bar *)

    Str.Concat(T,getElem(month,space,TM), " ");
    IF programmer THEN Str.Append(T,"$");END; (* reminder we're in hex mode *)
    Str.Append(T,fmtnum(year,base,Ydigits,padchar));
    IF showfullJD THEN
        Str.Append(T,sepjulian);
        Str.Append(T,fmtbignum( currjd,base,1," "));
    END;
    centerstr(wical,space,T);

    (* store title then days header *)

    currline:=minRow;
    storeline(WHICH,currline,inktitle,papertitle,T);
    INC(currline);
    storeline(WHICH,currline,inkdays,paperdays,D);
    INC(currline);

    (* don't be clever here : nowadays, what's the use for such an obsolete quality ? :-( *)

    JDYEAR:=dateToJD(LONGREAL( 1 ),LONGREAL(1),LONGREAL(year));
    currjdbase:= LONGCARD(JDYEAR); INC(currjdbase); (* poor man's floor() = +0.5 *)

    (* first day of month : JD computed earlier for title *)

    (*
    JD:=dateToJD(LONGREAL( 1 ),LONGREAL(month),LONGREAL(year));
    currjd    := LONGCARD(JD); INC(currjd);
    *)

    dow:=JDtoWeekday ( JD );

    S:=space; (* we'll be able to do len-1 without problem *)
    count:=firstweekday;

    i:=baseJDofWeek;
    state:=waiting;
    LOOP
        CASE state OF
        | waiting:
            IF i=dow THEN
                state:=inmonth;
            ELSE
                IF julian THEN
                    Str.Append(S,padjulian);
                ELSIF numbered THEN
                    Str.Append(S,paddiy);
                END;
                Str.Append(S,padquantieme);
                INC(i);
                IF i > lastdayofweek THEN i:=firstdayofweek;END;
                INC(count);
            END;
        | inmonth:
            JDtoDate(JD,jj,mm,yy);
            IF CARDINAL(mm) = month THEN
                IF CARDINAL(jj)=daynow THEN
                    hereline:=currline;
                    herepos:=Str.Length(S)-1; (* S always at least one space *)
                    IF julian THEN
                        INC(herepos,wipadjulian);
                    ELSIF numbered THEN
                        INC(herepos,wipaddiy);
                    END;
                END;
                IF julian THEN
                    vcurr:=(currjd MOD 10000); DEC(vcurr); (* fix *)
                    Str.Append(S,openjulian);
                    Str.Append(S,fmtbignum(vcurr,base,wijulian,"0"));
                    Str.Append(S,closejulian);
                    Str.Append(S,space);
                ELSIF numbered THEN
                    vcurr:=currjd-currjdbase+1; DEC(vcurr); (* fix *)
                    Str.Append(S,opennumbered);
                    Str.Append(S,fmtbignum(vcurr,base,widiy," "));
                    Str.Append(S,closenumbered);
                    Str.Append(S,space);
                END;
                Str.Append(S,fmtnum( CARDINAL(jj),base,digits,padchar));
                Str.Append(S,space);
                JD := JD + LONGREAL(1);
                INC(count);
            ELSE
                state:=done;
            END;
        | done:
            IF julian THEN
                Str.Append(S,padjulian);
            ELSIF numbered THEN
                Str.Append(S,paddiy);
            END;
            Str.Append(S,padquantieme);
            INC(count);
        END;
        IF count > lastweekday THEN
            storeline(WHICH,currline,inkmain,papermain,S);
            IF julian THEN
                 p:=minCol+1; (* right after first space *)
                 FOR ii:=firstweekday TO lastweekday DO
                     storeattr(p,wipadjulian-1, WHICH,currline,inkjulian,paperjulian);
                     INC(p,wipadjulian);
                     INC(p,wipadquantieme);
                 END;
            ELSIF numbered THEN
                 p:=minCol+1; (* right after first space *)
                 FOR ii:=firstweekday TO lastweekday DO
                     storeattr(p,wipaddiy-1, WHICH,currline,inkjulian,paperjulian);
                     INC(p,wipaddiy);
                     INC(p,wipadquantieme);
                 END;
            END;
            INC(currline);
            IF state=done THEN EXIT; END;
            JDtoDate(JD,jj,mm,yy);
            IF CARDINAL(mm) # month THEN EXIT; END; (* don't start a blank line *)
            S:=space;
            count:=firstweekday;
        END;
        IF (state # waiting) THEN INC(currjd);END;
    END;

    (* highlight current day if needed *)

    IF ( (month=monthnow) AND (year=yearnow) ) THEN
        p:=herepos;
        ii:=hereline;
        storechar(WHICH,ii,p  ,inknow,papernow,hilightleft);
        storechar(WHICH,ii,p+3,inknow,papernow,hilightright);
        storechar(WHICH,ii,p+1,inknow,papernow,retrievechar(WHICH,ii,p+1));
        storechar(WHICH,ii,p+2,inknow,papernow,retrievechar(WHICH,ii,p+2));
    END;

    DEC(currline); (* we were already at next line *)
    RETURN currline;
END buildmonth;

PROCEDURE dmpyear (programmer,julian,showfullJD,numbered:BOOLEAN;
                  daynow,monthnow,yearnow,year:CARDINAL;
                  language:languagetype;weekstart:daytype;
                  hilightleft,hilightright:CHAR;
                  inkmain,papermain,inktitle,papertitle,
                  inkdays,paperdays,inknow,papernow,
                  inkjulian,paperjulian:colortype);
VAR
    nlines:ARRAY [1..3] OF CARDINAL;
    month,i,len,last : CARDINAL;
BEGIN
    month:=firstmonth;
    IF (julian OR numbered) THEN
        last := 1; (* wide display *)
    ELSE
        last := 3;
    END;
    LOOP
        FOR i:=1 TO last DO
            nlines[i]:= buildmonth ( programmer,julian,showfullJD,numbered,
                                   i, daynow,monthnow,yearnow,month,year,
                                   language,weekstart,
                                   hilightleft,hilightright,
                                   inkmain,papermain,inktitle,papertitle,
                                   inkdays,paperdays,inknow,papernow,
                                   inkjulian,paperjulian);
            INC(month);
        END;
        IF last # 1 THEN
            nlines[2]:=merge(nlines[2],nlines[3], 2,3, inkmain,papermain);
            nlines[1]:=merge(nlines[1],nlines[2], 1,2, inkmain,papermain);
        END;
        dmp(1,nlines[1]);
        IF month > lastmonth THEN EXIT; END;

        color(inkmain,papermain);
        FOR i:=1 TO Str.Length(hline[1,minRow]) DO WrStr(space);END;
        WrLn;
    END;
END dmpyear;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    dontcare   = MIN(CARDINAL); (* firstday-1 *)
    pbnum      = MAX(CARDINAL);

PROCEDURE parsenum (S:ARRAY OF CHAR; mini,maxi:CARDINAL;ismonth:BOOLEAN):CARDINAL;
VAR
    i,v:CARDINAL;
    rc:BOOLEAN;
    R:str128; (* oversized *)
BEGIN
    IF same(S,"*") THEN RETURN dontcare;END;
    rc:=str2card (v, S);
    IF rc THEN
        IF ( (v<mini) OR (v>maxi) ) THEN v:=pbnum; END;
    ELSE
        IF ismonth THEN
            UpperCase(S); (* don't preserve accents *)
            i:=1;
            LOOP
                CASE i OF
                | 1 : R:=mois;
                | 2 : R:=abrevmois;
                | 3 : R:=moisUS;
                | 4 : R:=abrevmoisUS;
                END;
                UpperCase(R); (* don't preserve accents *)
                v:=getStrIndex(space, S, R);
                IF v # 0 THEN EXIT; END; (* jan=1, etc. *)
                INC(i);
                IF i > 4 THEN v:=pbnum;EXIT;END;
            END;
        ELSE
            v:=pbnum;
        END;
    END;
    RETURN v;
END parsenum;

(* ------------------------------------------------------------ *)

CONST
    ioBufferSize    = (8 * 512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = ioBufferSize;
TYPE
    ioBufferType    = ARRAY [firstBufferByte..lastBufferByte] OF BYTE;
VAR
    ioBufferIn,ioBufferOut : ioBufferType;

CONST
    defaultINI = nl+
"; first line is date format :"+nl+
"; F=FR=FRA=FRENCH=JMA (jour mois anne) or E=US=UK=ENGLISH=YMD (year month day)"+nl+
"; month is [1..12], full month name or first three letters of month"+nl+
"; *=end of data"+nl+
nl+
"; *"+nl+
nl+
"FR"+nl+
nl+
"; date then event description trimmed unless enclosed with quotes)"+nl+
"; description must be at most 80 characters long"+nl+
"; yearly events will use * as year"+nl+
"; monthly events will use * as month"+nl+
nl+
"1 2 1963   This is very unfortunate a day :-("+nl+
"1 2 *      At 19:45 U.T., this is my birthday :-("+nl+
"1 * *      This is the first day of current month"+nl+
nl;

PROCEDURE createini (ini:ARRAY OF CHAR);
VAR
    hout:FIO.File;
BEGIN
    hout:=FIO.Create(ini);
    FIO.AssignBuffer(hout,ioBufferOut);

    FIO.WrStr(hout,defaultINI);

    FIO.Close(hout);
END createini;

CONST
    firstevent = 1;
    maxevent   = 400; (* see strMaxEvent supra *)
TYPE
    eventtype = RECORD
       j,m,a:CARDINAL; (* yes, yes, we could pack these or use shortcards *)
       txt:str80;   (* oversized *)
    END;
VAR
    event:ARRAY[firstevent..maxevent] OF eventtype;

PROCEDURE fixcare ( VAR v:CARDINAL;ismonth:BOOLEAN  );

BEGIN
    IF ismonth THEN
        IF v=dontcare THEN v:=13; END;
    ELSE
        IF v=dontcare THEN v:=9999; END;
    END;
END fixcare;

PROCEDURE sortIsLess (i,j:CARDINAL):BOOLEAN;
CONST
    k1=10000;
    k2=1000000;
VAR
    ji,jj,mi,mj,ai,aj:CARDINAL;
    amji,amjj:LONGCARD;             (* jj mm aaaa *)
BEGIN
    ji:=event[i].j;
    mi:=event[i].m; fixcare(mi,TRUE );
    ai:=event[i].a; fixcare(ai,FALSE);
    jj:=event[j].j;
    mj:=event[j].m; fixcare(mj,TRUE );
    aj:=event[j].a; fixcare(aj,FALSE);

    amji:=LONGCARD(ai) + LONGCARD(mi)*k1 + LONGCARD(ji)*k2;
    amjj:=LONGCARD(aj) + LONGCARD(mj)*k1 + LONGCARD(jj)*k2;

    RETURN (amji < amjj);
END sortIsLess;

PROCEDURE sortIsLessYMD (i,j:CARDINAL):BOOLEAN;
CONST
    k1=10000;
    k2=1000000;
VAR
    ji,jj,mi,mj,ai,aj:CARDINAL;
    amji,amjj:LONGCARD;             (* jj mm aaaa *)
BEGIN
    ji:=event[i].j;
    mi:=event[i].m;
    ai:=event[i].a;
    jj:=event[j].j;
    mj:=event[j].m;
    aj:=event[j].a;

    amji:=LONGCARD(ji) + LONGCARD(mi)*k1 + LONGCARD(ai)*k2;
    amjj:=LONGCARD(jj) + LONGCARD(mj)*k1 + LONGCARD(aj)*k2;

    RETURN (amji < amjj);
END sortIsLessYMD;

PROCEDURE sortSwap (i,j:CARDINAL);
VAR
    tmp:eventtype;
BEGIN
    tmp:=event[i];
    event[i]:=event[j];
    event[j]:=tmp;
END sortSwap;

PROCEDURE fixCentury (VAR v:CARDINAL; doFix2000:BOOLEAN   );
CONST
    century1900 = 1900;
    century2000 = 2000;
BEGIN
    IF doFix2000 THEN
        IF v < 100 THEN
            IF v < 80 THEN
                INC(v,century2000);
            ELSE
                INC(v,century1900);
            END;
        END;
    ELSE
        IF v < 100 THEN INC(v,century1900); END;
    END;
END fixCentury;

PROCEDURE loadEvents ( VAR lastevent:CARDINAL; VAR frfmt:BOOLEAN;VAR errmsg:ARRAY OF CHAR;
                     doFix2000,ignoreini,DEBUG:BOOLEAN):BOOLEAN;
CONST
    me = "loadEvents()";
VAR
    hin:FIO.File;
    ini,S:str128;
    rc:BOOLEAN;
    p,currline:CARDINAL;
    P1,P2,P3,JJ,MM,AA:str16;
    state:(waiting,gotlanguage);
    language:(fr,us);
    e:eventtype;
    searchresult:searchresulttype;
BEGIN
    rc:=TRUE;
    lastevent:=firstevent-1; (* 1-1 *)
    IF ignoreini THEN RETURN rc;END; (* don't care about ini *)

    IF whereIsIni(ini)=foundnowhere THEN RETURN rc; END;
    Str.Concat(errmsg,vbar,ini);
    dbg(DEBUG,TRUE, me,"ini",ini);

    currline:=0;
    state:=waiting;
    hin:=FIO.OpenRead(ini);
    FIO.AssignBuffer(hin,ioBufferIn);
    FIO.EOF:=FALSE;
    LOOP
        IF FIO.EOF THEN EXIT; END;
        FIO.RdStr(hin,S);
        LtrimBlanks(S);
        RtrimBlanks(S);

        dbg(DEBUG,TRUE,me,"S",S);

        INC(currline);
        CASE S[0] OF
        | nullchar,semicolon,pound: (* ignore empty OR comment *)
            ;
        | star:
            EXIT; (* immediate exit *)
        ELSE
            CASE state OF
            | waiting:
                dbg(DEBUG,TRUE,me,"LANGUAGE",S);
                Str.Caps(S);
                p:=getStrIndex(delim,S,"F,FR,FRA,FRENCH,JMA,E,US,UK,ENGLISH,YMD");
                CASE p OF
                | 1..5:  language:=fr; frfmt:=TRUE;
                | 6..10: language:=us; frfmt:=FALSE;
                ELSE
                    rc:=FALSE;
                    EXIT;
                END;
                INC(state);
            | gotlanguage:
                dbg(DEBUG,TRUE,me,"EVENT",S);
                argv(P1,S,1,FALSE);
                argv(P2,S,2,FALSE);
                argv(P3,S,3,FALSE);
                CASE language OF
                | fr: JJ:=P1;   MM:=P2;   AA:=P3;
                | us: JJ:=P3;   MM:=P2;   AA:=P1;
                END;
                e.j:=parsenum( JJ,1,31,FALSE);
                e.m:=parsenum( MM,1,12,TRUE);
                e.a:=parsenum( AA,MIN(CARDINAL)+1,MAX(CARDINAL)-1,FALSE);

                IF e.j=pbnum THEN rc:=FALSE;EXIT;END;
                IF e.m=pbnum THEN rc:=FALSE;EXIT;END;
                IF e.a=pbnum THEN rc:=FALSE;EXIT;END;

                IF e.j=dontcare   THEN rc:=FALSE;EXIT;END;

                IF e.a # dontcare THEN fixCentury(e.a, doFix2000); END;

                (* grab text now : darn, this is really ugly *)

                Str.Subst(S,P1,"");
                Str.Subst(S,P2,"");
                Str.Subst(S,P3,"");
                LtrimBlanks(S);
                IF Str.Match(S,singlequote+star+singlequote) THEN
                    Str.Delete(S,0,1);
                    Str.Delete(S,Str.Length(S)-1,1);
                ELSIF Str.Match(S,doublequote+star+doublequote) THEN
                    Str.Delete(S,0,1);
                    Str.Delete(S,Str.Length(S)-1,1);
                END;
                Str.Copy(e.txt,S);

                INC(lastevent);
                IF lastevent > maxevent THEN rc:=FALSE;EXIT;END;
                event[lastevent]:=e;
            END;
        END;
        IF ( (S[0] = nullchar) AND FIO.EOF ) THEN EXIT; END;
    END;
    FIO.Close(hin);
    IF rc THEN
        Lib.QSort(lastevent,sortIsLess,sortSwap);
    ELSE
        Str.Prepend(errmsg,fmtnum(currline,10,1,""));
    END;
    IF DEBUG THEN
        FOR p:=firstevent TO lastevent DO
            S:='[~]  j=~  m=~  a=~  t="~"';
            Str.Subst(S,tilde,fmtnum(p,10,5," ") );
            Str.Subst(S,tilde,fmtnum(event[p].j,10,5," ") );
            Str.Subst(S,tilde,fmtnum(event[p].m,10,5," ") );
            Str.Subst(S,tilde,fmtnum(event[p].a,10,5," ") );
            Str.Subst(S,tilde,       event[p].txt );
            WrStr(S);WrLn;
        END;
    END;
    RETURN rc;
END loadEvents;

(* resort list by year/month/day *)

PROCEDURE listEvents (lastevent:CARDINAL;frfmt:BOOLEAN);
CONST
    wifr = 6; (* "## ###" *)
    wius = 7; (* "## ###," *)
VAR
    i,p,n,k,wi, j,m,a:CARDINAL;
    S,PAT,LISTE:str128;
    txt:str80;
    z:str16;
    ok:BOOLEAN;
BEGIN
    colorhelp;
    FOR i:=1 TO 3 DO
        IF frfmt THEN
            wi:=wifr;
            CASE i OF
            | 1: PAT:="| ~ |";    S:="Dates";
            | 2: PAT:="| ~";      S:="Evnements annuels";  DEC(wi,4);
            | 3: PAT:="|";        S:="Evnements mensuels"; DEC(wi,4);
            END;
        ELSE
            wi:=wius;
            CASE i OF
            | 1: PAT:="~ |, |";   S:="Date events";
            | 2: PAT:="~ |";      S:="Yearly events";       DEC(wi,4);
            | 3: PAT:="|";        S:="Monthly events";      DEC(wi,5);
            END;
        END;
        Str.Prepend(S,"::: ");
        IF i > 1 THEN WrLn;END;
        WrStr(S);WrLn;
        WrLn;
        CASE i OF
        | 1:  Lib.QSort(lastevent,sortIsLessYMD,sortSwap);
        | 2:  Lib.QSort(lastevent,sortIsLessYMD,sortSwap);
        | 3:  Lib.QSort(lastevent,sortIsLess   ,sortSwap);
        END;


        n:=0;
        FOR p:=firstevent TO lastevent DO
            j   :=event[p].j;
            m   :=event[p].m;
            a   :=event[p].a;
            txt :=event[p].txt;
            CASE i OF
            | 1: ok:=( (m # dontcare) AND (a # dontcare) );
            | 2: ok:=( (m # dontcare) AND (a = dontcare) );
            | 3: ok:=( (m = dontcare) AND (a = dontcare) );
            END;
            IF ok THEN
                S:=PAT;
                z:=fmtnum(j,10,2," ");
                (*
                IF frfmt THEN
                    IF j=1 THEN Str.Append(z,"er");END;
                END;
                *)
                Str.Subst(S,vbar,z);
                IF frfmt THEN
                    LISTE:=abrevmois;
                ELSE
                    LISTE:=abrevmoisUS;
                END;
                isoleItemS( z, LISTE,space,m-1);
                Str.Subst(S,tilde,z);
                z:=fmtnum(a,10,4,"0");
                Str.Subst(S,vbar,z );
                FOR k:=Str.Length(S)+1 TO ABS(wi) DO
                    IF wi < 0 THEN
                        Str.Prepend(S," ");
                    ELSE
                        Str.Append(S," ");
                    END;
                END;
                Str.Append(S," : ");
                Str.Append(S,txt);
                WrStr(S);WrLn;
                INC(n);
            END;
        END;
        IF n=0 THEN
            WrStr("None defined.");WrLn;
            IF i < 3 THEN WrLn;END;
        END;
    END;
END listEvents;

(* ------------------------------------------------------------ *)

(* could be more clever : find index then display all we can *)

PROCEDURE buildreminder (VAR ialt:CARDINAL;
                        altview,showtoday:BOOLEAN; currline,lastevent,
                        daynow,monthnow,yearnow, month,year:CARDINAL;
                        inkmain,papermain,inkdays,paperdays,inknow,papernow:colortype);
CONST
    ndx     = 1; (* monthly display *)
    ndxalt  = 2;
    separ   = "  ";
    pat     = separ+"~ "+cSepEvent+" ~";
    patmore = separ+"     "+strMoreEvents; (* "## : " *)
VAR
    i,j:CARDINAL;
    jj,mm,aa:CARDINAL;
    S:str80;
    ink,paper,inkhelp,paperhelp:colortype;
    okm,oka:BOOLEAN;
BEGIN
    inkhelp:=getInkAtStartup();
    paperhelp:=getPaperAtStartup();
    (* default to these because display would not look pretty with random length reminders *)
    ink:=inkhelp;
    paper:=paperhelp;

    (* event[firstevent..lastevent] dontcare=yearly OR monthly *)
    i:=minRow-1;
    ialt:=i;
    FOR j:=firstevent TO lastevent DO
        jj:=event[j].j;
        mm:=event[j].m;
        aa:=event[j].a;
        okm:=(mm=month) OR (mm=dontcare);
        oka:=(aa=year) OR (aa=dontcare);
        IF (okm AND oka) THEN
            S:=pat;
            IF showtoday THEN
                IF ((jj=daynow) AND okm) THEN Str.Subst(S,cSepEvent,cTodayEvent);END;
            END;
            Str.Subst(S,tilde,fmtnum(jj,10,2," "));       (* ## *)
            Str.Subst(S,tilde,event[j].txt);
            IF altview THEN
                INC(ialt);
                IF ialt < maxRow THEN
                    Str.Delete(S,0,Str.Length(separ));
                    storeline(ndxalt,ialt,ink,paper,S);
                END;
            ELSE
                INC(i);
                IF i <= currline THEN
                    IF i=currline THEN                 (* were at last displayed row ... *)
                        IF j # lastevent THEN S:=patmore;END; (* ... but not at last event *)
                    END;
                    appendline(ndx,i,ink,paper,S);
                END;
            END;
        END;
    END;
END buildreminder;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE parsecolor (VAR valcolor:colortype; VAR rc:CARDINAL;
                      isink:BOOLEAN;R:ARRAY OF CHAR) : BOOLEAN;
VAR
    v:LONGCARD;
    ok:BOOLEAN;
    p:CARDINAL;
    STR:str128;
BEGIN
    GetString(R,STR);
    Str.Lows(STR);
    p:=getStrIndex(delim,STR,strcolorsdark+strcolorsbright); (* 1-based, 0=notfound *)
    CASE p OF
    | mincolor+1..maxcolor+1: (* note +1 here ! *)
        rc:=errNone; (* useless *)
        ok:=TRUE;
        valcolor:=VAL(colortype,p-1); (* 0-based *)
        RETURN ok;
    END;
    IF GetLongCard(R,v) THEN
        IF ( (v < mincolor) OR (v > maxcolor) ) THEN
            IF isink THEN
                rc:=errInk;
            ELSE
                rc:=errPaper;
            END;
            ok:=FALSE;
        ELSE
            rc:=errNone; (* useless *)
            ok:=TRUE;
            valcolor:=VAL(colortype,CARDINAL(v));
        END;
    ELSE
        rc:=errNumber;
        ok:=FALSE;
    END;
    RETURN ok;
END parsecolor;

(* ------------------------------------------------------------ *)

CONST
    minparm = 1;
    maxparm = 2;
VAR
    parmcount,i,opt:CARDINAL;
    S,R:str128;
    parm : ARRAY [minparm..maxparm] OF str128;
    lastparm:CARDINAL;
    rc:CARDINAL;
    DEBUG:BOOLEAN;
VAR
    language:languagetype;
    weekstart:daytype;
    inkmain,papermain:colortype;
    inktitle,papertitle,inkdays,paperdays,inknow,papernow:colortype;
    inkjulian,paperjulian:colortype;
    inkspecial,paperspecial:colortype;
    hilightleft,hilightright:CHAR;
VAR
    yearnow,monthnow,daynow,year,month:CARDINAL;
    DoW:Lib.DayType;
    calmode : (monthly,yearly);
    altview,ignoreini:BOOLEAN;
    frfmt,programmer,julian,numbered,showtoday,showfullJD,doFix2000:BOOLEAN;
    gotoption,overwrite,ok:BOOLEAN;
    lasteventformonth,lastevent:CARDINAL;
    ini:str128;
    cmd:(cmddefault,cmdcal,cmdcreate,cmdlist);
BEGIN
    Lib.DisableBreakCheck();

    (* handleVesa; *) (* useless, because we won't change video mode *)
    setUseBiosMode ( IsRedirected() );
    findInkPaperAtStartup();

    WrLn; (* must be AFTER lib init ! *)

    lastparm  := minparm-1;
    DEBUG     := FALSE;
    language  := french;
    weekstart := monday;
    programmer:= FALSE;
    julian    := FALSE;
    showfullJD:= FALSE;
    numbered  := FALSE;
    showtoday := TRUE;
    cmd       := cmddefault;
    overwrite := FALSE;
    ignoreini := FALSE;
    altview   := FALSE;
    doFix2000 := TRUE;

    hilightleft    :=cHilightleft;
    hilightright   :=cHilightright;

    inkmain   := cyan;          papermain   := darkblue;
    inktitle  := white;         papertitle  := darkblue;
    inkdays   := black;         paperdays   := cyan;
    inknow    := yellow;        papernow    := darkred;
    inkspecial:= red;           paperspecial:= white; (* for future use *)
    inkjulian := yellow;        paperjulian := darkblue;

    parmcount := Lib.ParamCount();
    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i); cleantabs(S);
        Str.Copy(R,S); UpperCase(R); (* don't preserve accents *)

        IF ( same(R,"-") OR same(R,"--") ) THEN (* //V11B syntax *)
           gotoption:=FALSE;
        ELSE
           gotoption:=isOption(R);
        END;

        IF gotoption THEN
            opt := GetOptIndex (R, "?"+delim+"H"+delim+"HELP"+delim+
                                   "DEBUG"+delim+
                                   "B"+delim+"BIOS"+delim+
                                   "U"+delim+"US"+delim+"UK"+delim+"ENGLISH"+delim+
                                   "S"+delim+"SUNDAY"+delim+
                                   "IM:"+delim+
                                   "PM:"+delim+
                                   "IT:"+delim+
                                   "PT:"+delim+
                                   "ID:"+delim+
                                   "PD:"+delim+
                                   "IN:"+delim+"IC:"+delim+
                                   "PN:"+delim+"PC:"+delim+
                                   "IS:"+delim+
                                   "PS:"+delim+
                                   "T:"+delim+"TODAY:"+delim+
                                   "X"+delim+"PROGRAMMER"+delim+"P"+delim+
                                   "J"+delim+"JULIAN"+delim+
                                   "N"+delim+
                                   "IJ:"+delim+
                                   "PJ:"+delim+
                                   "Q"+delim+"NOHILIGHT"+delim+
                                   "JJ"+delim+
                                   "!"+delim+"CREATE"+delim+
                                   "!!"+delim+
                                   "R"+delim+"REMINDER"+delim+
                                   "A"+delim+"ALT"+delim+"ALTERNATE"+delim+
                                   "E"+delim+"EVENTS"+delim+
                                   "Y"+delim+"1900"+delim+"XX"+delim+"CENTURY"
                               );
            CASE opt OF
            | 1,2,3 :
                abort(errHelp,"");
            | 4:
                DEBUG:=TRUE;
            |5,6:
                setUseBiosMode ( TRUE );
            |7,8,9,10 :
                language:=english;
            |11,12:
                weekstart:=sunday;
            |13:
                IF parsecolor (inkmain   ,rc,  TRUE ,R)=FALSE THEN abort(rc,S);END;
            |14:
                IF parsecolor (papermain ,rc,  FALSE,R)=FALSE THEN abort(rc,S);END;
            |15:
                IF parsecolor (inktitle  ,rc,  TRUE ,R)=FALSE THEN abort(rc,S);END;
            |16:
                IF parsecolor (papertitle,rc,  FALSE,R)=FALSE THEN abort(rc,S);END;
            |17:
                IF parsecolor (inkdays   ,rc,  TRUE ,R)=FALSE THEN abort(rc,S);END;
            |18:
                IF parsecolor (paperdays ,rc,  FALSE,R)=FALSE THEN abort(rc,S);END;
            |19,20:
                IF parsecolor (inknow    ,rc,  TRUE ,R)=FALSE THEN abort(rc,S);END;
            |21,22:
                IF parsecolor (papernow  ,rc,  FALSE,R)=FALSE THEN abort(rc,S);END;
            |23:
                IF parsecolor (inkspecial,rc,  TRUE ,R)=FALSE THEN abort(rc,S);END;
            |24:
                IF parsecolor (paperspecial,rc,FALSE,R)=FALSE THEN abort(rc,S);END;
            |25,26:
                GetString(S,R);
                IF Str.Length(R) # 2 THEN abort(errBadToday,S);END;
                hilightleft:=R[0];
                hilightright:=R[1];
            |27,28,29:
                programmer:=TRUE;
            |30,31:
                julian := TRUE;
            |32:
                numbered:=TRUE;
            |33:
                IF parsecolor (inkjulian ,rc,  TRUE ,R)=FALSE THEN abort(rc,S);END;
            |34:
                IF parsecolor (paperjulian,rc, FALSE,R)=FALSE THEN abort(rc,S);END;
            |35,36:
                showtoday := FALSE;
            |37:
                julian     := TRUE;
                showfullJD := TRUE;
            |38,39:
                CASE cmd OF
                | cmddefault,cmdcreate:cmd:=cmdcreate;
                ELSE
                    abort(errCmd,"");
                END;
            |40:
                CASE cmd OF
                | cmddefault,cmdcreate:cmd:=cmdcreate;
                ELSE
                    abort(errCmd,"");
                END;
                overwrite:=TRUE;
            |41,42:
                ignoreini  := TRUE;
            |43,44,45:
                altview    := TRUE;
            |46,47:
                CASE cmd OF
                | cmddefault,cmdlist:cmd:=cmdlist;
                ELSE
                    abort(errCmd,"");
                END;
            | 48, 49, 50, 51:
                doFix2000 := FALSE;
            ELSE
                abort(errUnknownOpt,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            INC(lastparm);
            IF lastparm > maxparm THEN abort(errTooManyParms,S);END;
            Str.Copy(parm[lastparm],R);
        END;
    END;

    IF cmd=cmddefault THEN cmd:=cmdcal;END;

    (* ------------------------------------------------------------ *)

    CASE cmd OF
    | cmdcreate :
        ini:=progEXEname+extINI;
        IF FIO.Exists(ini) THEN
            IF overwrite THEN
                ok:=TRUE; S:="::: Overwrite existing ~";
            ELSE
                ok:=FALSE;S:="::: Skip existing ~";
            END;
        ELSE
                ok:=TRUE; S:="::: Create ~";
        END;
        Str.Subst(S,tilde,dquote+tilde+dquote);
        Str.Subst(S,tilde,ini);
        WrStr(S);WrLn;
        IF ok THEN createini(ini);END;
        abort(errNone,"");
    | cmdlist:
        IF loadEvents(lastevent,frfmt,R, doFix2000,FALSE,DEBUG)=FALSE THEN abort(errEvent,R);END;
        IF lastevent < firstevent THEN abort(errNoEvent,"");END;
        listEvents(lastevent,frfmt);
        abort(errNone,"");
    END;

    (* ------------------------------------------------------------ *)

    IF (julian AND numbered) THEN abort(errJulian,"");END;

    CASE lastparm OF
    | minparm:
        Str.Copy(S, parm[minparm]);
        i:=getStrIndex(coma,S,"+,++,-,--");
        IF i = 0 THEN ignoreini:=TRUE; END; (* yearly calendar display *)
    ELSE
        ;
    END;
    IF ignoreini=FALSE THEN
        IF (julian OR numbered) THEN altview:=TRUE; END;
    END;

    IF loadEvents(lastevent,frfmt,R, doFix2000,ignoreini,DEBUG)=FALSE THEN abort(errEvent,R);END;

    IF showtoday=FALSE THEN
        hilightleft  := space;
        hilightright := space;
        inknow       := inkmain;
        papernow     := papermain;
    END;

    (* let's really start working *)

    Lib.GetDate(yearnow,monthnow,daynow,DoW);

    CASE lastparm OF
    | minparm-1: (* current month and current year *)
        month:=monthnow;
        year :=yearnow;
        calmode:=monthly;
    | minparm:   (* year OR * OR + or - *)
        Str.Copy(S, parm[minparm]);

        i:=getStrIndex(coma,S, "+,++,-,--,*" );
        CASE i OF
        | 1,2 :
            year:=yearnow;
            month:=monthnow+1;
            IF month > lastmonth THEN month:=firstmonth;INC(year);END;
            calmode:=monthly;
        | 3,4 :
            year:=yearnow;
            month:=monthnow-1;
            IF month < firstmonth THEN month:=lastmonth;DEC(year);END;
            calmode:=monthly;
        | 5 :
            year:=yearnow;
            calmode:=yearly;
        ELSE
            IF str2card(year,S)=FALSE THEN abort(errYear,S); END;
            calmode:=yearly;
        END;
    | maxparm:   (* month year *)
        Str.Copy(S, parm[minparm]);
        IF str2card(month,S)=FALSE THEN
            i:=1;
            LOOP
                CASE i OF
                | 1 : R:=mois;
                | 2 : R:=abrevmois;
                | 3 : R:=moisUS;
                | 4 : R:=abrevmoisUS;
                END;
                UpperCase(R); (* don't preserve accents *)
                month:=getStrIndex(space, S, R);
                IF inrange(month,firstmonth,lastmonth) THEN EXIT; END;
                INC(i);
                IF i > 4 THEN abort(errMonth,S); END;
            END;
        END;
        IF inrange(month,firstmonth,lastmonth)=FALSE THEN abort(errMonth,S);END;
        Str.Copy(S, parm[minparm+1]);
        IF same(S,"*") THEN
            year:=yearnow;
        ELSE
            IF str2card(year,S)=FALSE THEN abort(errYear,S); END;
        END;
        calmode:=monthly;
    END;

    (*
    IF DEBUG THEN
        Str.Concat(S,"daynow    : ",fmtnum(daynow,10,8,space));WrStr(S);WrLn;
        Str.Concat(S,"monthnow  : ",fmtnum(monthnow,10,8,space));
        Str.Append(S," = ");Str.Append(S,getElem(monthnow,space,mois));
        Str.Append(S," = ");Str.Append(S,getElem(monthnow,space,moisUS));
        WrStr(S);WrLn;
        Str.Concat(S,"yearnow   : ",fmtnum(yearnow ,10,8,space));WrStr(S);WrLn;
        WrLn;
        Str.Concat(S,"month     : ",fmtnum(month,10,8,space));
        Str.Append(S," = ");Str.Append(S,getElem(month,space,mois));
        Str.Append(S," = ");Str.Append(S,getElem(month,space,moisUS));
        WrStr(S);WrLn;
        Str.Concat(S,"year      : ",fmtnum(year ,10,8,space));WrStr(S);WrLn;
    END;
    *)

    CASE calmode OF
    | monthly:        (* 1 is minRow *)
        i:=buildmonth ( programmer,julian,showfullJD,numbered,
                      1, daynow,monthnow,yearnow,month,year,
                      language,weekstart,
                      hilightleft,hilightright,
                      inkmain,papermain,inktitle,papertitle,
                      inkdays,paperdays,inknow,papernow,inkjulian,paperjulian);
        IF ignoreini=FALSE THEN
            buildreminder (lasteventformonth,
                          altview,showtoday,i,lastevent,
                          daynow,monthnow,yearnow,month,year,
                          inkmain,papermain,inkdays,paperdays,inknow,papernow);
        END;
        dmp( 1, i);
        IF ignoreini=FALSE THEN
            IF altview THEN WrLn; dmp( 2,lasteventformonth); END;
        END;
    | yearly:
        dmpyear       ( programmer,julian,showfullJD,numbered,
                      daynow,monthnow,yearnow,year,
                      language,weekstart,
                      hilightleft,hilightright,
                      inkmain,papermain,inktitle,papertitle,
                      inkdays,paperdays,inknow,papernow,inkjulian,paperjulian);
    END;

    abort(errNone,"");
END Cal.




