(* ---------------------------------------------------------------
Title         Q&D Filter
Overview      see help
Usage         see help
Notes         very, very, very quick & dirty... :-(
Bugs          replacement may occur in a sequence !
              lines are not checked for 128 chars limit
Wish List     allow user to specify verbatim or token-parsed mode ?

--------------------------------------------------------------- *)

MODULE Filter;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;

FROM QD_ASCII IMPORT dash, slash, nullchar, tabchar, cr, lf, nl,
space, dot, deg, doublequote, quote, colon, percent, vbar,
bs, blank, equal, dquote, charnull, singlequote, antislash, dollar,
star, backslash, coma, question, underscore, tabul, hbar,
comma, semicolon, diese, pound,
stardotstar, dotdot, escCh, escSet, letters, digits;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits,
getAllLegalUnits, metaproc, getCli, argc, argv;

FROM IO IMPORT WrStr, WrLn,WrCard,WrLngCard;

TYPE
    strtok = str128;

(* ------------------------------------------------------------ *)

CONST
    ProgEXEname   = "FILTER";
    ProgTitle     = "Q&D Filter";
    ProgVersion   = "v1.1p";
    ProgCopyright = "by PhG";
    Banner        = ProgTitle+" "+ProgVersion+" "+ProgCopyright;
CONST
    extEXE = ".EXE"; (* upper case *)
    extLST = ".LST";
CONST
    legalset = letters+digits+colon+backslash+dot+dash+underscore+"{"; (* eh eh ! *)
    netslash   = backslash+backslash;
    colonbackslash = colon+backslash;
    ANYCOLUMN = MAX(CARDINAL);
    MINCOL     = 1;
CONST
    errNone             = 0;
    errHelp             = 1;
    errOption           = 2;
    errTooMany          = 3;
    errWaiting          = 4;
    errBadSpec          = 5;
    errSame             = 6;
    errNotFound         = 7;
    errExists           = 8;
    errReadOnly         = 9;
    errNoExclude        = 10;
    errTooManyExclude   = 11;
    errQuoteMismatch    = 12;
    errTooLong          = 13;
    errNoToken          = 14;
    errTooManyTokens    = 15;
    errInvalidPair      = 16;
    errTooLongToken     = 17;
    errBadMeta          = 18;
    errCmd              = 19;
    errNoPair           = 20;
    errTooManyPairs     = 21;
    errBadCode          = 22;
    errRedefined        = 23;
    errNoStringPair     = 24;
    errTooManyStringPairs=25;
    errTooLongString    = 26;
    errNotAStringPair   = 27;
    errBadColumn        = 28;

(* ------------------------------------------------------------ *)

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);

    MODULE message;
    IMPORT Str;
    EXPORT msg3;

    PROCEDURE msg3 (VAR R:ARRAY OF CHAR;S1,S2,S3:ARRAY OF CHAR);
    BEGIN
        Str.Concat(R,S1,S2);Str.Append(R,S3);
    END msg3;

    END message;

CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)
    msgHelp=
Banner+nl+
nl+
"Syntax : "+ProgEXEname+" <source> <target> [list["+extLST+"]] [-t]|<-r|-f|-k|-d> [option]..."+nl+
nl+
"    -t   filter out matching lines (default)"+nl+
"         read each line from <source> and write it to <target>"+nl+
"         only if it does not contain any verbatim entry from list."+nl+
"    -r   replace tokens"+nl+
"         read each line from <source>, perform replacements"+nl+
"         using token pairs found in list, then write line to <target>."+nl+
"    -f   replace strings"+nl+
"         read each line from <source>, perform replacements"+nl+
"         using strings found in list, then write line to <target>."+nl+
"    -k   substitute string to code"+nl+
"         read each line from <source>, perform verbatim replacements"+nl+
"         using code-string pairs found in list, then write line to <target>."+nl+
nl+
"    -c   case-sensitive filter (-t, -r, -f and -d only)"+nl+
"    -i   inverse (-t|-d = inverse filter, -r = swap tokens, -f = swap strings)"+nl+
"    -u   do not filter out undefined codes (-k only)"+nl+
"    -c:# column (1 upwards, -r and -f only)"+nl+
"    -o   overwrite existing <target>"+nl+
"    -q   quiet (no eyecandy)"+nl+
"    -v   verbose (show final summary)"+nl+
nl+
"a) Up to 500 128-chars entries can be defined in list (default is "+ProgEXEname+extLST+")."+nl+
"   Search order for list is current directory first, then executable directory."+nl+
"b) -t and -r options support the following codes in their definitions :"+nl+
'   "' + escCh + 'n" = $0d0a, "' + escCh + 'p" = "%", "' + escCh+ 'q" = double quote, "' +
escCh + '[' + escSet + ']" = symbol,' + nl +
'   "' + escCh + "[$|x]#[" + escCh + ']" = hex/dec value ($00 value not allowed).'+nl+
"c) Replacement substrings must be checked as to avoid unwanted side effects."+nl+
"d) The following remarks apply to -f option only :"+nl+
"   list format is line pairs : one line for original, one line for replacement."+nl+
"   Empty lines are ignored unless they're enclosed with double quotes."+nl+
"e) The following remarks apply to -k option only : <source> may be binary ;"+nl+
"   list format is (hexa)decimal code in [$00..$ff] range or quoted character,"+nl+
"   then separator, then (quoted) string or hexadecimal string ;"+nl+
"   code will be filtered out if string is empty unless -u option is specified."+nl+
"f) Results are unpredictable if list contains lines longer than 128 chars."+nl+
"g) -d option (delete matching lines) is an alias for -t option."+nl;

VAR
    S : str256;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(msgHelp);
    | errOption :
        msg3(S,"Illegal ",einfo," option !");
    | errTooMany :
        msg3(S,"File ",einfo," is just one file too many !");
    | errWaiting :
        msg3(S,einfo," was not specified"," !");
    | errBadSpec :
        msg3(S,"Illegal ",einfo," file specification !");
    | errSame :
        msg3(S,einfo," are identical"," !");
    | errNotFound :
        msg3(S,"File ",einfo," not found !");
    | errExists :
        msg3(S,"File ",einfo," already exists !");
    | errReadOnly :
        msg3(S,"File ",einfo," is read-only !");
    | errNoExclude :
        S := "No filter is defined !";
    | errTooManyExclude :
        S := "Too many filters are defined !";
    | errQuoteMismatch :
        msg3(S,"Unmatched quote in <",einfo,"> line !");
    | errTooLong :
        msg3(S,"String <",einfo,"> too long !");
    | errNoToken :
        S := "No token pair is defined !";
    | errTooManyTokens :
        S := "Too many token pairs are defined !";
    | errInvalidPair:
        msg3(S,"Illegal ",einfo," token pair !");
    | errTooLongToken :
        msg3(S,"Token too long in <",einfo,"> entry !");
    | errBadMeta:
        msg3(S,"Illegal escape sequence in <",einfo,"> entry !");
    | errCmd:
        S := "-t, -r, -f, -k and -d options are mutually exclusive !";
    | errNoPair :
        S := "No code-string pair is defined !";
    | errTooManyPairs :
        S := "Too many code-string pairs are defined !";
    | errBadCode:
        msg3(S,"Illegal code in <",einfo,"> entry !");
    | errRedefined:
        msg3(S,"Redefined code in <",einfo,"> entry !");
    | errNoStringPair :
        S := "No string pair is defined !";
    | errTooManyStringPairs :
        S := "Too many string pairs are defined !";
    | errTooLongString :
        msg3(S,"String too long in <",einfo,"> entry !");
    | errNotAStringPair:
        S := "Missing last replacement string !";
    | errBadColumn:
        msg3(S,"Illegal ",einfo," column value !");
    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone, errHelp : ;
    ELSE
        WrStr(ProgEXEname+" : "); WrStr(S); WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE buildWithExeDir (VAR S:ARRAY OF CHAR);
VAR
    exepath,u,d,n,e:str128;
    uParm,dParm,nParm,eParm:str128;
BEGIN
    Lib.ParamStr(exepath,0);
    Lib.SplitAllPath(exepath,u,d,n,e);
    Lib.SplitAllPath(S,uParm,dParm,nParm,eParm);
    Str.Concat(S,u,d);
    Str.Append(S,nParm);
    Str.Append(S,eParm);
END buildWithExeDir;

PROCEDURE chkSpec (S : ARRAY OF CHAR ) : BOOLEAN;
(* minimalist check ! assume S is not "" and is already in uppercase *)
VAR
    i : CARDINAL;
BEGIN
    IF Str.CharPos(S,star) # MAX(CARDINAL) THEN RETURN FALSE; END;
    IF Str.CharPos(S,question) # MAX(CARDINAL) THEN RETURN FALSE; END;
    IF Str.Pos(S,dotdot) # MAX(CARDINAL) THEN RETURN FALSE; END;
    IF Str.Pos(S,netslash) # MAX(CARDINAL) THEN RETURN FALSE; END;
    (*
    FOR i := 0 TO (Str.Length(S)-1) DO
        IF Str.CharPos(legalset,S[i])=MAX(CARDINAL) THEN RETURN FALSE; END;
    END;
    *)
    RETURN TRUE;
END chkSpec;

PROCEDURE CanWrite (S : ARRAY OF CHAR) : BOOLEAN;
CONST
    spex = FIO.FileAttr{aR,aH,aS,aA};
VAR
    D  : FIO.DirEntry;
    rc : BOOLEAN;
BEGIN
    IF FIO.Exists(S)=FALSE THEN RETURN TRUE; END;
    rc := FIO.ReadFirstEntry(S,spex,D); (* assume file exists ! *)
    IF aR IN D.attr THEN RETURN FALSE; END;
    RETURN TRUE;
END CanWrite;

PROCEDURE wrq (S:ARRAY OF CHAR   );
BEGIN
    WrStr(doublequote);  WrStr(S);  WrStr(doublequote);
END wrq;

PROCEDURE showcount (showsummary:BOOLEAN; n:LONGCARD;F,what:ARRAY OF CHAR  );
BEGIN
    IF showsummary THEN
        WrLngCard(n,5);
        (* WrStr(" replacement"); IF replaced > 1 THEN WrStr("s");END; *)
        WrStr(" ");WrStr(what);
        WrStr(" made in "); WrStr(F); WrLn;
    END;
END showcount;

(* ------------------------------------------------------------ *)

CONST
    ioBufferSize    = (8 * 512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = ioBufferSize;
VAR
    inBuffer  : ARRAY [firstBufferByte..lastBufferByte] OF BYTE;
    outBuffer : ARRAY [firstBufferByte..lastBufferByte] OF BYTE;

(* ------------------------------------------------------------ *)

CONST
    msgLoading = "Loading ";
    msgFiltering="Filtering ";
    firstEntry = 1;
    maxEntry   = 500; (* was 64 *)
    firstcode  = MIN(SHORTCARD); (* complicated way to say 0 *)
    maxcode    = MAX(SHORTCARD); (* 255 *)
    maxcodecount=maxcode-firstcode+1;
VAR
    charcode : ARRAY [firstcode..maxcode] OF BOOLEAN;
    Token    : ARRAY [firstEntry..maxEntry] OF strtok;
    Token2   : ARRAY [firstEntry..maxEntry] OF strtok;
    lencode  : ARRAY [firstEntry..maxEntry] OF SHORTCARD;

(* ------------------------------------------------------------ *)

PROCEDURE ReadExcludeList (list : ARRAY OF CHAR;
                           keepcase,verbose,debug:BOOLEAN) : CARDINAL;
VAR
    S   : str128;
    hnd : FIO.File;
    ch  : CHAR;
    n,p,rc   : CARDINAL;
BEGIN
    IF debug THEN verbose:=FALSE;END; (* avoid artefacts when showing data *)
    rc := errNone;
    IF verbose THEN
        video(msgLoading,TRUE);
        video(list,TRUE);
        Work(cmdInit);
    END;

    n := 0;
    hnd:=FIO.OpenRead(list);
    FIO.AssignBuffer(hnd,inBuffer);
    LOOP
        IF verbose THEN Work(cmdShow);END;
        IF FIO.EOF THEN EXIT; END;
        FIO.RdStr(hnd,S);
        RtrimBlanks(S);
        LtrimBlanks(S);
        ch:=S[0];
        IF ((ch = doublequote) OR (ch=singlequote)) THEN
            p := Str.Length(S);
            IF (p<2) OR (S[p-1] <> ch) THEN
                rc := errQuoteMismatch;
                EXIT;
            END;
            S[p-1] := nullchar;
            Str.Delete(S,0,1);
        END;
        IF metaproc(S)=FALSE THEN
            rc:=errBadMeta;
            EXIT;
        END;
        IF same(S,"")=FALSE THEN
            IF n > maxEntry THEN EXIT; END;
            IF Str.Length(S) > SIZE(strtok) THEN
                rc := errTooLong;
                EXIT;
            END;
            Str.Copy(Token[firstEntry+n],S);
            IF keepcase=FALSE THEN UpperCase( Token[firstEntry+n] ); END;
            IF debug THEN WrStr( Token[firstEntry+n] );WrLn;END;
            INC(n);
        END;
    END;
    FIO.Close(hnd);
    IF verbose THEN
        Work(cmdStop);
        video(list,FALSE);
        video(msgLoading,FALSE);
    END;
    IF rc # errNone THEN abort(rc,S);END; (* //ugly *)
    RETURN n;
END ReadExcludeList;

PROCEDURE isExcluded (S : ARRAY OF CHAR;
                      keepcase:BOOLEAN; countExclude:CARDINAL ) : BOOLEAN;
VAR
    i : CARDINAL;
BEGIN
    IF keepcase=FALSE THEN UpperCase(S); END;
    i := 0;
    LOOP
        IF i >= countExclude THEN RETURN FALSE; END;
        IF Str.Pos(S,Token[firstEntry+i]) <> MAX(CARDINAL) THEN RETURN TRUE; END; (* could use Match to handle case *)
        INC(i);
    END;
END isExcluded;

(* ------------------------------------------------------------ *)

CONST
    steps = 10;
    animcmd=animShow;
VAR
    lastportion,portion,currportion:LONGCARD;

PROCEDURE doFilter (sourcefile,targetfile:ARRAY OF CHAR;
                    keepcase,inverse,verbose:BOOLEAN;countExclude:CARDINAL ):LONGCARD;
VAR
    hndin,hndout : FIO.File;
    S : str4096; (* oversized just in CASE *)
    processed:LONGCARD;
BEGIN
    processed:=0;
    IF verbose THEN
        video(msgFiltering,TRUE);
        video(sourcefile,TRUE);
        video(" ",TRUE);
        (* Work(cmdInit); *)
        (* completed(completedInit,getFileSize(sourcefile) ); *)
        animInit(steps, "[", "]", CHR(46), "", "\/" );
        portion:=getFileSize(sourcefile) DIV steps; INC(portion); (* avoid DIV 0 ! *)
        lastportion := steps+1;
    END;
    hndin := FIO.OpenRead(sourcefile);
    FIO.AssignBuffer(hndin,inBuffer);
    hndout:= FIO.Create(targetfile);
    FIO.AssignBuffer(hndout,outBuffer);
    FIO.EOF:=FALSE;
    LOOP
        IF verbose THEN
            (* Work(cmdShow); *)
            (* completed(completedSHOW,FIO.GetPos(hndin)); *)
            anim(animcmd);
            currportion:=FIO.GetPos(hndin) DIV portion;
            IF currportion # lastportion THEN
                anim(animAdvance);
                lastportion:=currportion;
            END;
        END;
        IF FIO.EOF THEN EXIT; END;
        FIO.RdStr(hndin,S);
        IF (S[0]=nullchar) AND FIO.EOF THEN EXIT; END; (* weird hack ! *)
        IF isExcluded(S,keepcase,countExclude)=inverse THEN (* hack ! FALSE=normal, TRUE=inverse *)
            FIO.WrStr(hndout,S);
            FIO.WrLn(hndout);
        ELSE
            INC(processed);
        END;
    END;
    FIO.Flush(hndout);
    FIO.Close(hndout);
    FIO.Close(hndin);
    IF verbose THEN
        (* Work(cmdStop); *)
        (* completed(completedEnd,0); *)
        anim(animEnd);anim(animClear);
        video(" ",FALSE);
        video(sourcefile,FALSE);
        video(msgFiltering,FALSE);
    END;
    RETURN processed;
END doFilter;

(* ------------------------------------------------------------ *)

(* compare two strings from token array *)

PROCEDURE tokLess (i : CARDINAL; j : CARDINAL) : BOOLEAN;
VAR
	isLess : BOOLEAN;
BEGIN
	IF Str.Compare( Token[i], Token[j])  < 0 THEN
		isLess := TRUE;
	ELSE
		isLess := FALSE;
	END;
	RETURN isLess;
END tokLess;

PROCEDURE tokSwap (i : CARDINAL; j : CARDINAL);
VAR
	tmp		: strtok;
BEGIN
	tmp 	 := Token[i];
	Token[i] := Token[j];
	Token[j] := tmp;
	tmp 	 := Token2[i];
	Token2[i]:= Token2[j];
	Token2[j]:= tmp;
END tokSwap;

(* s1 < s2, now check if s1 is included in s2 in sortedToken array *)

PROCEDURE tokMatching (i : CARDINAL; j : CARDINAL) : BOOLEAN;
VAR
    rc : BOOLEAN;

BEGIN
    IF Str.Pos( Token[j] , Token[i] )=0 THEN
        rc := TRUE;
    ELSE
        rc := FALSE;
    END;
    RETURN rc;
END tokMatching;

PROCEDURE specialsortTokenList (count:CARDINAL;debug:BOOLEAN );
VAR
    i : CARDINAL;
    done         : BOOLEAN;
BEGIN
	Lib.HSort(count, tokLess, tokSwap);

    (* taken from SORT4BOX *)

    LOOP
		done := TRUE;
		FOR i := firstEntry TO count-1 DO
 		   	IF Str.Compare(Token[i], Token[i+1])  < 0 THEN
				IF tokMatching(i,i+1) THEN
					tokSwap(i,i+1);
					done := FALSE;
				END;
           	END;			
		END;
		IF done THEN EXIT; END;
	END;
	IF debug THEN
	    FOR i:=firstEntry TO count DO
            wrq( Token [i] );  WrStr( " === ");  wrq( Token2[i] );  WrLn;
	    END;
	END;
END specialsortTokenList;

PROCEDURE unquote (VAR S : ARRAY OF CHAR);
VAR
    len : CARDINAL;
    ch:CHAR;
BEGIN
    len := Str.Length(S);
    IF len = 0 THEN RETURN; END; (* useless *)
    ch:=S[len-1];
    CASE ch OF
    | doublequote,singlequote:
        IF S[0] = ch THEN
            S[len-1]:=CHR(0); (* brutal 0C *)
            Str.Delete(S,0,1);
        END;
    ELSE
        RETURN;
    END;
END unquote;

PROCEDURE ReadTokenList (list : ARRAY OF CHAR;
                         keepcase,inverse,verbose,debug:BOOLEAN ) : CARDINAL;
VAR
    S   : str256;
    P1,P2 : str128;
    hnd : FIO.File;
    n,p,rc   : CARDINAL;
BEGIN
    IF debug THEN verbose:=FALSE;END; (* avoid artefacts when showing data *)
    rc := errNone;
    IF verbose THEN
        video(msgLoading,TRUE);
        video(list,TRUE);
        Work(cmdInit);
    END;
    n := 0;
    hnd:=FIO.OpenRead(list);
    FIO.AssignBuffer(hnd,inBuffer);
    LOOP
        IF verbose THEN Work(cmdShow);END;
        IF FIO.EOF THEN EXIT; END;
        FIO.RdStr(hnd,S);
        CASE argc(S,TRUE) OF
        | 0:
            ; (* empty line *)
        | 1,2:
            IF n > maxEntry THEN EXIT; END;
            argv (P1 , S , 1, TRUE); unquote(P1);
            argv (P2 , S , 2, TRUE); unquote(P2);
            IF metaproc(P1)=FALSE THEN
                rc:=errBadMeta;
                EXIT;
            END;
            IF metaproc(P2)=FALSE THEN
                rc:=errBadMeta;
                EXIT;
            END;
            IF Str.Length(P1) > SIZE(strtok) THEN
                rc:=errTooLongToken;
                EXIT;
            END;
            IF Str.Length(P2) > SIZE(strtok) THEN
                rc:=errTooLongToken;
                EXIT;
            END;
            IF same(P2,"") THEN
                Str.Copy(Token [firstEntry+n],P1);
                Str.Copy(Token2[firstEntry+n],P2);
            ELSE
                IF inverse THEN
                    Str.Copy(Token [firstEntry+n],P2);
                    Str.Copy(Token2[firstEntry+n],P1);
                ELSE
                    Str.Copy(Token [firstEntry+n],P1);
                    Str.Copy(Token2[firstEntry+n],P2);
                END;
            END;
            IF keepcase=FALSE THEN UpperCase( Token [firstEntry+n] );END;
            IF debug THEN
                wrq( Token [firstEntry+n] );  WrStr( " --- ");  wrq( Token2[firstEntry+n] ); WrLn;
            END;

            INC(n);
        ELSE
            rc:=errInvalidPair;
            EXIT;
        END;
    END;
    FIO.Close(hnd);
    IF verbose THEN
        Work(cmdStop);
        video(list,FALSE);
        video(msgLoading,FALSE);
    END;
    IF rc # errNone THEN abort(rc,S);END; (* //ugly *)
    RETURN n;
END ReadTokenList;

(* beware of side effects ! rebuild string *)

PROCEDURE fullreplace (VAR R:ARRAY OF CHAR;
                       column:CARDINAL;keepcase:BOOLEAN;
                       P1,P2:ARRAY OF CHAR):LONGCARD ;
VAR
    tmp,S : str4096; (* oversized the same way as R itself  *)
    n : LONGCARD;
    len1,len2,porg,p:CARDINAL;
BEGIN
    n := 0;
    IF Str.Length(R)=0 THEN RETURN n; END;
    Str.Copy(S, R);
    len1:=Str.Length(P1);
    len2:=Str.Length(P2);

    IF (column # ANYCOLUMN) THEN
        porg:=column-1; (* was 1-based *)
        Str.Copy(tmp,S);
        IF keepcase=FALSE THEN UpperCase(tmp);END;
        p:=Str.NextPos(tmp, P1,porg);
        IF p = porg THEN
            Str.Delete(S,p,len1);
            Str.Insert(S,P2,p);
            INC(n);
        END;
    ELSE
        porg:=0;
        LOOP
            Str.Copy(tmp,S);
            IF keepcase=FALSE THEN UpperCase(tmp);END;
            p := Str.NextPos(tmp, P1,porg);
            IF p = MAX(CARDINAL) THEN EXIT; END;
            Str.Delete(S,p,len1);
            Str.Insert(S,P2,p);
            porg := p+len2; (* tsss, I was incrementing porg with len2 ! *)
            INC(n);
        END;
    END;
    Str.Copy(R,S);
    RETURN n;
END fullreplace;

PROCEDURE doReplace (sourcefile,targetfile:ARRAY OF CHAR;
                     keepcase,verbose:BOOLEAN;countToken,column:CARDINAL ):LONGCARD;
VAR
    hndin,hndout : FIO.File;
    S : str4096; (* oversized just in case *)
    i : CARDINAL;
    replaced : LONGCARD ;
BEGIN
    replaced := 0;
    IF verbose THEN
        video(msgFiltering,TRUE);
        video(sourcefile,TRUE);
        video(" ",TRUE);
        (* Work(cmdInit); *)
        (* completed(completedInit,getFileSize(sourcefile) ); *)
        animInit(steps, "[", "]", CHR(46), "", "\/" );
        portion:=getFileSize(sourcefile) DIV steps; INC(portion); (* avoid DIV 0 ! *)
        lastportion := steps+1;
    END;
    hndin := FIO.OpenRead(sourcefile);
    FIO.AssignBuffer(hndin,inBuffer);
    hndout:= FIO.Create(targetfile);
    FIO.AssignBuffer(hndout,outBuffer);
    FIO.EOF:=FALSE;
    LOOP
        IF verbose THEN
            (* Work(cmdShow); *)
            (* completed(completedSHOW, FIO.GetPos(hndin) ); *)
            anim(animcmd);
            currportion:=FIO.GetPos(hndin) DIV portion;
            IF currportion # lastportion THEN
                anim(animAdvance);
                lastportion:=currportion;
            END;
        END;
        IF FIO.EOF THEN EXIT; END;
        FIO.RdStr(hndin,S);
        IF (S[0]=nullchar) AND FIO.EOF THEN EXIT; END; (* weird hack ! *)
        FOR i:=firstEntry TO countToken DO
            (* IF verbose THEN Work(cmdShow);END; *)
            INC(replaced, fullreplace(S, column,keepcase, Token [i], Token2[i] ) );
        END;
        FIO.WrStr(hndout,S);
        FIO.WrLn(hndout);
    END;
    FIO.Flush(hndout);
    FIO.Close(hndout);
    FIO.Close(hndin);
    IF verbose THEN
        (* Work(cmdStop); *)
        (* completed(completedEnd,0); *)
        anim(animEnd);anim(animClear);
        video(" ",FALSE);
        video(sourcefile,FALSE );
        video(msgFiltering,FALSE );
    END;
    RETURN replaced;
END doReplace;

(* ------------------------------------------------------------ *)

PROCEDURE printable (VAR R : ARRAY OF CHAR);
VAR
    len,i:CARDINAL;
BEGIN
    len:=Str.Length(R);
    i:=0;
    WHILE i < len DO
        IF ORD( R[i] ) < ORD(blank) THEN
            R[i] := ".";
        END;
        INC(i);
    END;
END printable;

PROCEDURE num2str (VAR S : ARRAY OF CHAR;
                  n,digits,base : CARDINAL; padchar : CHAR);
VAR
    ok   : BOOLEAN;
    v    : LONGCARD;
    len  : CARDINAL;
BEGIN
    v := LONGCARD(n);
    Str.CardToStr(v,S,base,ok);
    len := Str.Length(S);
    LOOP
        IF Str.Length(S) >= digits THEN EXIT; END;
        Str.Prepend(S,padchar);
    END;
END num2str;

PROCEDURE getCardCode (S : ARRAY OF CHAR; VAR value : CARDINAL) : BOOLEAN;
CONST
    hex     = "$";
    digit0  = "0";
    digit9  = "9";
VAR
    v  : LONGCARD;
    ok : BOOLEAN;
    base,len:CARDINAL;
BEGIN
    CASE S[0] OF
    | hex :
        Str.Delete(S,0,1); (* remove leading $ *)
        Str.Caps(S);       (* safety *)
        base:=16;
    | digit0..digit9 :
        base:=10;
    | doublequote,singlequote:
        len:=Str.Length(S);
        IF len # 3 THEN RETURN FALSE; END; (* legal : "?" '?' *)
        IF S[0] # S[len-1] THEN RETURN FALSE; END;
        S[len-1]:=CHR(0); (* brutal 0C *)
        Str.Delete(S,0,1);
        value := ORD(S[0]);
        RETURN TRUE;
    ELSE
        RETURN FALSE;
    END;
    IF Str.Length(S)=0 THEN RETURN FALSE; END;
    Str.Caps(S); (* this routine does not like $ff and the like ! *)
    v := Str.StrToCard(S,base,ok);
    IF ok = FALSE THEN RETURN FALSE; END;
    IF v > MAX(SHORTCARD) THEN RETURN FALSE; END;
    value := CARDINAL(v);
    RETURN TRUE;
END getCardCode;

PROCEDURE hexdig2val (ch:CHAR):CARDINAL;
VAR
    v:CARDINAL;
BEGIN
    v:=ORD(ch);
    CASE v OF
    | ORD("0")..ORD("9") : v:=v-ORD("0");
    | ORD("A")..ORD("F") : v:=v-ORD("A")+10;
    END;
    RETURN v;
END hexdig2val;

(* if string is "$??!!..." then parse it *)

PROCEDURE parsehex (VAR binlen:CARDINAL;VAR R:ARRAY OF CHAR);
CONST
    hexdigits = "0123456789ABCDEF";
VAR
    S:str128;
    i,p,len:CARDINAL;
    hi,lo:CARDINAL;
BEGIN
    binlen:=0;
    IF R[0] <> "$" THEN RETURN; END;
    Str.Copy(S,R);
    Str.Delete(S,0,1);
    len:=Str.Length(S);
    IF len=0 THEN RETURN; END;
    IF ODD ( len ) THEN RETURN; END;
    Str.Caps(S);
    IF verifyString(S,hexdigits) = FALSE THEN RETURN; END;
    (* len is at least 2 here *)
    p:=0;
    i:=0;
    LOOP
        IF i >= (len-1) THEN EXIT; END;
        hi:=hexdig2val(S[i]);
        lo:=hexdig2val(S[i+1]);
        R[p]:=CHR( hi << 4 + lo );
        INC(i,2);
        INC(p);
    END;
    R[p]:=0C;
    binlen:=p;
END parsehex;

PROCEDURE readPairs (list:ARRAY OF CHAR; keepundefined,verbose,debug:BOOLEAN):CARDINAL;
VAR
    S   : str256;
    P1,P2 : str128;
    hnd : FIO.File;
    n,p,rc,v,parmcount : CARDINAL;
    R:str16;
    charval,i,binlen:CARDINAL;
BEGIN
    IF debug THEN verbose:=FALSE;END; (* avoid artefacts when showing data *)
    rc := errNone;
    IF verbose THEN
        video(msgLoading,TRUE);
        video(list,TRUE);
        Work(cmdInit);
    END;

    FOR n:= firstcode TO maxcode DO
        charcode[n]:=FALSE;
    END;

    n := 0;
    hnd:=FIO.OpenRead(list);
    FIO.AssignBuffer(hnd,inBuffer);
    FIO.EOF:=FALSE;
    LOOP
        IF verbose THEN Work(cmdShow);END;
        IF FIO.EOF THEN EXIT; END;
        FIO.RdStr(hnd,S);
        parmcount:=argc(S,TRUE);
        CASE parmcount OF
        | 0:
            ; (* empty line *)
        ELSE
            argv (P1 , S , 1, TRUE);
            CASE P1[0] OF
            | ";" , "#" :
                parmcount:=0; (* remarks *)
            END;
        END;
        CASE parmcount OF
        | 0:
            ; (* empty line or comment *)
        | 1,2:
            IF n > maxcodecount THEN EXIT; END;
            argv (P2 , S , 2, TRUE);
            parsehex(binlen,P2);
            unquote(P2);

            IF ( same(P1,star) AND same(P2,"") ) THEN EXIT; END; (* comments marker *)

            IF getCardCode(P1,v)=FALSE THEN
                rc:=errBadCode;
                EXIT;
            END;
            IF Str.Length(P2) > SIZE(strtok) THEN
                rc:=errTooLongToken;
                EXIT;
            END;
            IF charcode[v] THEN
                rc:=errRedefined;
                EXIT;
            END;

            charcode[v]:= TRUE;
            lencode [v]:= SHORTCARD(binlen); (* flag : if # 0 then $hex string *)

            IF binlen=0 THEN
                Str.Copy(Token[v],P2);
            ELSE
                FOR i:=0 TO (binlen-1+1) DO (* include final 0C *)
                    Token[v][i]:=P2[i];
                END;
            END;

            IF debug THEN
                num2str (R, v,2,16,"0");
                Str.Lows(R);
                Str.Prepend(R,"$"); Str.Append(R," ("); WrStr(R);
                num2str (R, v,3,10," ");
                Str.Append(R,") ");  WrStr(R);
                Str.Copy(R,CHR(v));
                printable(R);
                wrq(R);  WrStr(" --- ");
                IF binlen=0 THEN
                    wrq( Token[v] );
                ELSE
                    WrStr(doublequote);
                    FOR i:=0 TO (binlen-1) DO
                        charval:=ORD(Token[v][i]);
                        CASE charval OF
                        | 0..31 :
                            WrStr(".");
                        ELSE
                            WrStr( CHR(charval) );
                        END;
                    END;
                    WrStr(doublequote);
                END;
                WrLn;
            END;

            INC(n);
        ELSE
            rc:=errInvalidPair;
            EXIT;
        END;
    END;
    FIO.Close(hnd);
    IF verbose THEN
        Work(cmdStop);
        video(list,FALSE);
        video(msgLoading,FALSE);
    END;

    IF rc # errNone THEN abort(rc,S);END; (* //ugly *)
    RETURN n;
END readPairs;

PROCEDURE doSubstitute (sourcefile,targetfile:ARRAY OF CHAR;
                       verbose,keepundefined:BOOLEAN):LONGCARD;
VAR
    hndin,hndout : FIO.File;
    i,got:CARDINAL;
    replaced : LONGCARD ;
    ch:CHAR;
BEGIN
    replaced := 0;
    IF verbose THEN
        video(msgFiltering,TRUE);
        video(sourcefile,TRUE);
        video(" ",TRUE);
        (* Work(cmdInit); *)
        (* completed(completedInit,getFileSize(sourcefile) ); *)
        animInit(steps, "[", "]", CHR(46), "", "\/" );
        portion:=getFileSize(sourcefile) DIV steps; INC(portion); (* avoid DIV 0 ! *)
        lastportion := steps+1;
    END;
    hndin := FIO.OpenRead(sourcefile);
    FIO.AssignBuffer(hndin,inBuffer);
    hndout:= FIO.Create(targetfile);
    FIO.AssignBuffer(hndout,outBuffer);
    FIO.EOF:=FALSE;
    LOOP
        IF verbose THEN
            (* Work(cmdShow); *)
            (* completed(completedSHOW, FIO.GetPos(hndin) ); *)
            anim(animcmd);
            currportion:=FIO.GetPos(hndin) DIV portion;
            IF currportion # lastportion THEN
                anim(animAdvance);
                lastportion:=currportion;
            END;
        END;
        got:=FIO.RdBin(hndin,ch,SIZE(ch));
        IF got # SIZE(ch) THEN EXIT; END;
        IF charcode[ ORD(ch) ] THEN
            IF lencode[ ORD(ch) ] = 0 THEN
                FIO.WrStr(hndout,Token[ ORD(ch) ]);
            ELSE
                FOR i:=1 TO CARDINAL(lencode[ ORD(ch) ]) DO
                    FIO.WrBin(hndout,Token[ ORD(ch) ][ i-1 ], 1);
                END;
            END;
            INC(replaced);
        ELSE
            IF keepundefined THEN FIO.WrBin(hndout,ch,SIZE(ch)); END;
        END;
    END;
    FIO.Flush(hndout);
    FIO.Close(hndout);
    FIO.Close(hndin);
    IF verbose THEN
        (* Work(cmdStop); *)
        (* completed(completedEnd,0); *)
        anim(animEnd);anim(animClear);
        video(" ",FALSE);
        video(sourcefile,FALSE );
        video(msgFiltering,FALSE );
    END;
    RETURN replaced;
END doSubstitute;

(* ------------------------------------------------------------ *)

PROCEDURE ReadStringPairs (list : ARRAY OF CHAR;
                          keepcase,inverse,verbose,debug:BOOLEAN ) : CARDINAL;
VAR
    S:str256;
    P1,P2:str128;
    n,rc:CARDINAL;
    what:(first,second);
    hnd:FIO.File;
BEGIN
    IF debug THEN verbose:=FALSE;END; (* avoid artefacts when showing data *)
    rc := errNone;
    IF verbose THEN
        video(msgLoading,TRUE);
        video(list,TRUE);
        Work(cmdInit);
    END;
    n := 0;
    what:=first;
    hnd:=FIO.OpenRead(list);
    FIO.AssignBuffer(hnd,inBuffer);
    LOOP
        IF verbose THEN Work(cmdShow);END;
        IF FIO.EOF THEN EXIT; END;
        FIO.RdStr(hnd,S);
        LtrimBlanks(S);
        RtrimBlanks(S);
        IF Str.Length(S) # 0 THEN
            unquote(S);
            IF Str.Length(S) > SIZE(strtok) THEN
                rc:=errTooLongString;
                EXIT;
            END;
            CASE what OF
            | first:
                IF n > maxEntry THEN EXIT; END;
                Str.Copy(P1,S);
                what:=second;
            | second:
                Str.Copy(P2,S);
                IF same(P2,"") THEN
                    Str.Copy(Token [firstEntry+n],P1);
                    Str.Copy(Token2[firstEntry+n],P2);
                ELSE
                    IF inverse THEN
                        Str.Copy(Token [firstEntry+n],P2);
                        Str.Copy(Token2[firstEntry+n],P1);
                    ELSE
                        Str.Copy(Token [firstEntry+n],P1);
                        Str.Copy(Token2[firstEntry+n],P2);
                    END;
                END;
                IF keepcase=FALSE THEN UpperCase( Token [firstEntry+n] );END;
                IF debug THEN
                    WrStr("Old : ");wrq( Token [firstEntry+n] ); WrLn;
                    WrStr("New : ");wrq( Token2[firstEntry+n] ); WrLn;
                END;
                what:=first;
                INC(n);
            END;
        END;
    END;
    FIO.Close(hnd);
    IF verbose THEN
        Work(cmdStop);
        video(list,FALSE);
        video(msgLoading,FALSE);
    END;
    IF rc = errNone THEN
        IF what # first THEN rc:=errNotAStringPair;END;
    END;
    IF rc # errNone THEN abort(rc,S);END; (* //ugly *)
    RETURN n;
END ReadStringPairs;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

VAR
    state      : (waiting,gotsource,gottarget,gotlist);
    sourcefile,targetfile,listfile : str128;
    overwrite, keepcase, inverse, keepundefined,verbose,showsummary:BOOLEAN;
    DEBUG : BOOLEAN;
    parmcount,i,opt  : CARDINAL;
    S,R        : str128;
    cmd        : (nocmd,filterout,replace,substitute,findreplace);
    count      : CARDINAL;
    column     : CARDINAL;
    lc,processed : LONGCARD ;
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE;
    WrLn;

    overwrite := FALSE;
    keepcase  := FALSE;
    inverse   := FALSE;
    keepundefined := FALSE;
    verbose   := TRUE;
    showsummary:=FALSE;
    column    := ANYCOLUMN;
    cmd       := nocmd;
    DEBUG     := FALSE;

    state := waiting;
    parmcount := Lib.ParamCount();
    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i); cleantabs(S);
        Str.Copy(R,S);
        UpperCase(R);
        IF isOption(R) THEN
            opt := GetOptIndex(R,"?"+delim+"H"+delim+"HELP"+delim+
                                 "O"+delim+"OVERWRITE"+delim+
                                 "C"+delim+"CASE"+delim+
                                 "I"+delim+"INVERSE"+delim+
                                 "R"+delim+"REPLACE"+delim+
                                 "Q"+delim+"QUIET"+delim+
                                 "K"+delim+"CONVERT"+delim+"SUBSTITUTE"+delim+
                                 "U"+delim+"UNDEFINED"+delim+
                                 "F"+delim+"FINDREPLACE"+delim+
                                 "T"+delim+"FILTER"+delim+
                                 "C:"+delim+"COLUMN:"+delim+
                                 "D"+delim+"DELETE"+delim+"DELMATCH"+delim+
                                 "V"+delim+"VERBOSER"+delim+
                                 "DEBUG");
            CASE opt OF
            | 1,2,3   : abort(errHelp,"");
            | 4,5     : overwrite := TRUE;
            | 6,7     : keepcase  := TRUE;
            | 8,9     : inverse   := TRUE;
            |10,11    : CASE cmd OF
                        | nocmd,replace:
                            cmd:= replace;
                        ELSE
                            abort(errCmd,"");
                        END;
            |12,13    : verbose   := FALSE ;
            |14,15,16 : CASE cmd OF
                        | nocmd,substitute:
                            cmd := substitute;
                        ELSE
                            abort(errCmd,"");
                        END;
            |17,18    : keepundefined := TRUE;
            |19,20    : CASE cmd OF
                        | nocmd,findreplace:
                            cmd := findreplace;
                        ELSE
                            abort(errCmd,"");
                        END;
            |21,22    : CASE cmd OF
                        | nocmd,filterout:
                            cmd := filterout;
                        ELSE
                            abort(errCmd,"");
                        END;
            |23,24    : IF GetLongCard(S,lc)=FALSE THEN abort(errBadColumn,S);END;
                        IF ( (lc < MINCOL) OR (lc > SIZE(str4096)) ) THEN
                            abort(errBadColumn,S);
                        ELSE
                            column:=CARDINAL(lc);
                        END;
            |25,26,27 : CASE cmd OF
                        | nocmd,filterout:
                            cmd := filterout;
                        ELSE
                            abort(errCmd,"");
                        END;
            |28,29    : showsummary:=TRUE;
            |30       : DEBUG     := TRUE;
            ELSE
                abort(errOption,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            CASE state OF
            | waiting :
                Str.Copy(sourcefile,R);
            | gotsource :
                Str.Copy(targetfile,R);
            | gottarget :
                Str.Copy(listfile,R);
            ELSE
                abort(errTooMany,S);
            END;
            INC(state);
        END;
    END;
    CASE state OF
    | waiting :
        abort(errHelp,"");
    | gotsource :
        abort(errWaiting,"<target>");
    | gottarget :
        Str.Concat(listfile,ProgEXEname,extLST);
    | gotlist :
        IF Str.CharPos(listfile,dot)=MAX(CARDINAL) THEN
            Str.Append(listfile,extLST);
        END;
    END;

    IF cmd = nocmd THEN cmd:=filterout;END;

    IF chkSpec(sourcefile)=FALSE THEN abort(errBadSpec,sourcefile); END;
    IF chkSpec(targetfile)=FALSE THEN abort(errBadSpec,targetfile); END;
    IF chkSpec(listfile)=FALSE THEN abort(errBadSpec,listfile); END;
    IF same(sourcefile,targetfile) THEN abort(errSame,"<source> and <target>"); END;
    IF same(targetfile,listfile) THEN abort(errSame,"<target> and <list>"); END;

    IF FIO.Exists(sourcefile)=FALSE THEN abort(errNotFound,sourcefile); END;
    IF FIO.Exists(targetfile)=TRUE THEN
        IF overwrite=FALSE THEN
            abort(errExists,targetfile);
        ELSE
            IF CanWrite(targetfile)=FALSE THEN
                abort(errReadOnly,targetfile);
            END;
        END;
    END;

    IF FIO.Exists(listfile)=FALSE THEN (* try given spec or current directory *)
        buildWithExeDir(listfile); (* then try exe directory *)
        IF FIO.Exists(listfile)=FALSE THEN abort(errNotFound,listfile);END;
    END;

    CASE cmd OF
    | filterout :
        count:=ReadExcludeList(listfile,keepcase,verbose,DEBUG);
        CASE count OF
        | 0 :  abort(errNoExclude,"");
        | firstEntry..maxEntry : ;
        ELSE
            abort(errTooManyExclude,"");
        END;
        processed:=doFilter(sourcefile,targetfile,keepcase,inverse,verbose,count);
        showcount(showsummary,processed,sourcefile,"deletion(s)");
    | replace:
        count:=ReadTokenList(listfile,keepcase,inverse,verbose,DEBUG);
        CASE count OF
        | 0 : abort(errNoToken,"");
        | firstEntry..maxEntry : ;
        ELSE
            abort(errTooManyTokens,"");
        END;
        specialsortTokenList(count,DEBUG);
        processed:=doReplace(sourcefile,targetfile,keepcase,verbose,count,column);
        showcount(showsummary,processed,sourcefile,"replacement(s)");
    | findreplace:
        count:=ReadStringPairs(listfile,keepcase,inverse,verbose,DEBUG);
        CASE count OF
        | 0 : abort(errNoStringPair,"");
        | firstEntry..maxEntry : ;
        ELSE
            abort(errTooManyStringPairs,"");
        END;
        processed:=doReplace(sourcefile,targetfile,keepcase,verbose,count,column);
        showcount(showsummary,processed,sourcefile,"replacement(s)");
    | substitute:
        count:=readPairs(listfile,keepundefined,verbose,DEBUG);
        CASE count OF
        | 0 : abort(errNoPair,"");
        | 1..maxcodecount : ;
        ELSE
            abort(errTooManyPairs,"");
        END;
        processed:=doSubstitute(sourcefile,targetfile,verbose,keepundefined);
        showcount(showsummary,processed,sourcefile,"replacement(s)");
    END;

    abort(errNone,"");
END Filter.

