(* ---------------------------------------------------------------
Title         Q&D Line Count
Overview      count lines in a text file
Usage         see help
Notes         very, very, very quick & dirty... :-(
              minimal error messages and checking, etc.
              assume legal (and rational !) comments for Modula-2 and Basic
              no handling of "(*" and "*)" in strings
              no handling of complex imbricated commments in M2 and C
              here are legal patterns :
              (* *)
              (*
              *)
              we don't care about M2 pragmas

Bugs
Wish List     accept string or val instead of char for -s option ?
              LFN support ? bah...

--------------------------------------------------------------- *)

MODULE LCount;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;
IMPORT SYSTEM;

FROM QD_ASCII IMPORT dash, slash, nullchar, tabchar, cr, lf, nl,
space, dot, deg, doublequote, quote, colon, percent, vbar,
bs, blank, equal, dquote, charnull, singlequote, antislash, dollar,
star, backslash, coma, question, underscore, tabul, hbar,
comma, semicolon,
stardotstar, dotdot, escCh, escSet, letters, digits;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits,
getAllLegalUnits;

FROM QD_File IMPORT pathtype, w9XnothingRequired,
fileOpenRead, fileOpen, fileExists, fileExistsAlt,
fileIsRO, fileSetRW, fileSetRO,
fileErase, fileCreate, fileRename, fileGetFileSize, fileGetFileStamp,
fileIsDirectorySpec, fileClose, fileSupportLFN;

(* shortened list *)

FROM QD_LFN IMPORT unicodeConversionFlagType, findDataRecordType,
w9XfindFirst, w9XfindClose, w9XfindNext, w9XshortToLong;

FROM IO IMPORT WrStr,WrLn,WrCard,WrLngCard;

FROM Storage IMPORT ALLOCATE, DEALLOCATE, Available,
HeapTotalAvail, MainHeap;

(* ------------------------------------------------------------ *)

CONST
    ProgEXEname     = "LCOUNT";
    ProgTitle       = "Q&D Line Counter";
    ProgVersion     = "v1.1c";
    ProgCopyright   = "by PhG";
    Banner          = ProgTitle+" "+ProgVersion+" "+ProgCopyright;
CONST
    extBAK        = ".BK!";
    basicrem0     = "REM";
    basicrem1     = "'";
    modularemopen = "(*";
    modularemclose= "*)";
    crem          = "//";
    cremopen      = "/*";
    cremclose     = "*/";
    asmcomment    = ";";
CONST
    extCOM      = ".COM";
    extEXE      = ".EXE";
    extDLL      = ".DLL";
    extOVR      = ".OVR";
    extOVL      = ".OVL";
    extDRV      = ".DRV";
    extVXD      = ".VXD";
    extBIN      = ".BIN";
    extZIP      = ".ZIP";
    extARJ      = ".ARJ";
    extLZH      = ".LZH";
    skippedextensions = extCOM+delim+extEXE+delim+
                        extDLL+delim+extOVR+delim+extOVL+delim+extDRV+delim+
                        extVXD+delim+extBIN+delim+
                        extZIP+delim+extARJ+delim+extLZH;
CONST
    sINFO         = "::: ";
    msgProcessing = sINFO+" Processing |";

CONST
    errNone         = 0;
    errHelp         = 1;
    errOption       = 2;
    errTooManyParms = 3;
    errMissingSpec  = 4;
    errTooMany      = 5;
    errNoMatch      = 6;
    errNonsense     = 7;
    errSilly        = 8;
    errComment      = 9;
    errRidiculous   = 10;
    errThereCanBeOnlyOne    = 11;
    errVerySilly            = 12;
    errBadRange             = 13;
    errThereCanBeOnlyOneHere= 14;
    errNonsenseWithSort     = 15;
    errBadSpec              = 16;
    errNonsenseFinding      = 17;
    errAborted              = 18;

(* ------------------------------------------------------------ *)

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)

errmsg =
Banner+nl+
nl+
"Syntax 1 : "+ProgEXEname+" <file(s)> [option]..."+nl+
"Syntax 2 : "+ProgEXEname+" <-f> [-v] <file(s)"+nl+
nl+
"This program counts lines in a text file (syntax 1)"+nl+
"or it finds the longest line (syntax 2), assuming at most 4096 characters."+nl+
nl+
'-m     ignore Modula-2 comments between "'+modularemopen+'" and "'+modularemclose+'"'+nl+
'-b     ignore BASIC comments beginning with "'+basicrem0+'" or "'+basicrem1+'"'+nl+
'-c     ignore C comments beginning with "'+crem+'" or between "'+cremopen+'" and "'+cremclose+'"'+nl+
'-a     ignore Assembler comments beginning with "'+asmcomment+'"'+nl+
"-k     display count of comments instead of count of lines"+nl+
"-e     count empty or blanks lines too"+nl+
"-s[:#] count sorted lines according to character at column specified with -c:#"+nl+
"       (-s:# = -s -c:#)"+nl+
"-c:#   character column (-s required, default is 1)"+nl+
"-g[g]  show bar graph (-s required, -bb = no semigraphics)"+nl+
"-w:#   bar graph width (default is 25 characters)"+nl+
"-f:#   number field width (default is 9)"+nl+
"-v     verbose processing"+nl+
"-d     dump lines or comments verbatim to screen"+nl+
"-f[f]  find the longest line (-ff = show rough lines average length too)"+nl+
"-x     disable LFN support even if available"+nl+
nl+
"a) Language comments support is minimal... and will remain so. ;-)"+nl+
"b) With -f[f] option, any option other than -v will be silently ignored."+nl+
"   Empty lines are not considered when computing rough lines average length."+nl+
"c) Program will not process files with any of these extensions :"+nl+
"   "+skippedextensions+" for safety."+nl;

VAR
    S : str1024; (* was str256 but in case we get a LFN... *)
BEGIN
    CASE e OF
    | errHelp :
        IF IsRedirected() THEN WrLn;END; (* silly but... *)
        WrStr(errmsg);
    | errOption:
        Str.Concat(S,"Unknown ",einfo);Str.Append(S," option !");
    | errTooManyParms:
        Str.Concat(S,einfo," parameter is one too many !");
    | errMissingSpec:
        S := "Missing file specification !";
    | errTooMany:
        Str.Concat(S,'Too many files match "',einfo);Str.Append(S,'" specification !');
    | errNoMatch:
        Str.Concat(S,'No file matches "',einfo);Str.Append(S,'" specification !');
    | errNonsense:
        S := "-m, -b, -c and -a options are mutually exclusive !";
    | errSilly:
        S := "-v, d- and -debug options are mutually exclusive !";
    | errComment:
        S := "-k option requires -m, -b, -c or -a option !";
    | errRidiculous:
        S := "-e option is a nonsense with any of -m, -b, -c or -a options !";
    | errThereCanBeOnlyOne:
        S := "-d option should be used for one file at a time !";
    | errVerySilly:
        S := "-d option is a nonsense without one of -m, -b, -c or -a options !";
    | errBadRange:
        Str.Concat(S,"Out of range ",einfo);Str.Append(S," value !");
    | errThereCanBeOnlyOneHere:
        S := "-s option should be used for one file at a time !";
    | errNonsenseWithSort:
        S := "-s option should be used alone !";
    | errBadSpec:
        Str.Concat(S,'Illegal "',einfo);Str.Append(S,'" specification !');
    | errNonsenseFinding:
        S := "-g[g] option is a nonsense with -f option !";
    | errAborted:
        S := "Aborted by user !";
    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp:
        ;
    ELSE
        WrStr(ProgEXEname+" : ");WrStr(S);WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE showmem(debug:BOOLEAN; S:ARRAY OF CHAR );
VAR
    heapsize    : CARDINAL; (* in PARAGRAPHS and not in bytes ! help is wrong ! *)
    n           : LONGCARD;
BEGIN
    IF debug THEN
        heapsize :=HeapTotalAvail(MainHeap);
        n := 16 * LONGCARD(heapsize);
        WrStr("::: ");
        WrLngCard(n,6);
        WrStr(" byte(s) free -- ");WrStr(S);WrLn;
    END;
END showmem;

(* ------------------------------------------------------------ *)

TYPE
    pFname = POINTER TO fnameType;
    fnameType = RECORD
        next      : pFname;
        slen      : CARDINAL; (* a SHORTCARD could do but who knows ? *)
        str       : CHAR;     (* variable length *)
    END;

PROCEDURE initList (VAR anchor : pFname );
BEGIN
    anchor := NIL;
END initList;

PROCEDURE freeList (anchor : pFname);
VAR
    needed : CARDINAL;
    p      : pFname;
BEGIN
    (* p:=anchor; *)
    WHILE anchor # NIL DO
        needed := SIZE(fnameType) - SIZE(anchor^.str) + anchor^.slen;
        p := anchor^.next;
        DEALLOCATE(anchor,needed);
        anchor:=p;
    END
END freeList;

PROCEDURE buildNewPtr (VAR anchor,p:pFname; len:CARDINAL):BOOLEAN;
VAR
    needed : CARDINAL;
BEGIN
    needed := SIZE(fnameType) - SIZE(p^.str) + len;
    IF Available(needed)=FALSE THEN RETURN FALSE; END;
    IF anchor = NIL THEN
        ALLOCATE(anchor,needed);
        p:=anchor;
    ELSE
        p:=anchor;
        WHILE p^.next # NIL DO
            p:=p^.next;
        END;
        ALLOCATE(p^.next,needed);
        p:=p^.next;
    END;
    p^.next := NIL;
    RETURN TRUE;
END buildNewPtr;

(* assume p is valid *)

PROCEDURE getStr (VAR S : pathtype; p:pFname);
VAR
    len:CARDINAL;
BEGIN
    len := p^.slen;
    Lib.FastMove( ADR(p^.str),ADR(S),len);
    S[len] := nullchar; (* REQUIRED safety ! *)
END getStr;

(* ------------------------------------------------------------ *)

(* Str.Match is not case sensitive *)

PROCEDURE isReservedPattern (S,skipthem:ARRAY OF CHAR):BOOLEAN;
VAR
    e3 : str16;
    n:CARDINAL;
    rc:BOOLEAN;
BEGIN
    rc:=FALSE;
    n:=0;
    LOOP
        isoleItemS(e3, skipthem,delim,n);
        IF same(e3,"") THEN EXIT; END;
        Str.Prepend(e3,"*");
        IF Str.Match(S,e3) THEN rc:=TRUE;EXIT; END;
        INC(n);
    END;
    RETURN rc;
END isReservedPattern;

PROCEDURE isReservedEntry (S:ARRAY OF CHAR) : BOOLEAN;
BEGIN
    IF same(S,dot) THEN RETURN TRUE; END;
    RETURN same(S,dotdot);
END isReservedEntry;

PROCEDURE buildFileList (VAR anchor:pFname;
                        useLFN,DEBUG :BOOLEAN;spec:pathtype):CARDINAL;
VAR
    count:CARDINAL; (* should do ! *)
    ok,found:BOOLEAN;
    unicodeconversion:unicodeConversionFlagType;
    w9Xentry : findDataRecordType;
    w9Xhandle,errcode:CARDINAL;
    entry : FIO.DirEntry;
    dosattr:FIO.FileAttr;
    entryname:pathtype;
    len : CARDINAL;
    pp:pFname;
    excludeme1,excludeme2:BOOLEAN;
BEGIN
    count:=0;
    IF useLFN THEN
        found := w9XfindFirst (spec,SHORTCARD(everything),SHORTCARD(w9XnothingRequired),
                              unicodeconversion,w9Xentry,w9Xhandle,errcode);
    ELSE
        found := FIO.ReadFirstEntry(spec,everything,entry);
    END;
    WHILE found DO
        IF useLFN THEN
            Str.Copy(entryname,w9Xentry.fullfilename);
        ELSE
            Str.Copy(entryname,entry.Name);
        END;
        excludeme1 := isReservedEntry   (entryname);  (* skip "." and ".." *)
        excludeme2 := isReservedPattern (entryname,skippedextensions );
        IF NOT(excludeme1 OR excludeme2) THEN
            IF useLFN THEN
                dosattr:=FIO.FileAttr(w9Xentry.attr AND 0FFH);
            ELSE
                dosattr:=entry.attr;
            END;
            IF NOT (aD IN dosattr) THEN
                (* if file has no extension, add it as a marker *)
                IF Str.RCharPos(entryname,".")=MAX(CARDINAL) THEN
                    Str.Append(entryname,".");
                END;
                IF DEBUG THEN WrStr("Included : ");WrStr(entryname);WrLn; END;
                len:=Str.Length(entryname);
                IF buildNewPtr(anchor,pp,len)=FALSE THEN
                    IF useLFN THEN ok:=w9XfindClose(w9Xhandle,errcode); END;
                    RETURN MAX(CARDINAL); (* errStorage *)
                END;
                INC(count);
                pp^.slen      := len;
                Lib.FastMove ( ADR(entryname),ADR(pp^.str),len );
            ELSE
                IF DEBUG THEN WrStr("Ignored  : ");WrStr(entryname);WrLn;END;
            END;
        ELSE
            IF DEBUG THEN WrStr("Excluded : ");WrStr(entryname);WrLn;END;
        END;
        IF useLFN THEN
            found :=w9XfindNext(w9Xhandle, unicodeconversion,w9Xentry,errcode);
        ELSE
            found :=FIO.ReadNextEntry(entry);
        END;
    END;
    IF useLFN THEN ok:=w9XfindClose(w9Xhandle,errcode); END;
    RETURN count;
END buildFileList;

PROCEDURE doGetCurrent (useLFN:BOOLEAN;drive:SHORTCARD;
                       VAR unit:str2; VAR current:pathtype);
VAR
    rc:CARDINAL;
    longform:pathtype;
BEGIN
    Str.Concat(unit, CHR( ORD("A")-1+ORD(drive) ),colon);

    FIO.GetDir(drive,current); (* \path without u: nor trailing \ except at root *)
    IF current[1] # colon THEN Str.Prepend(current,unit); END; (* safety *)
    IF useLFN THEN
        IF w9XshortToLong(current,rc,longform) THEN (* if error, keep DOS current *)
            Str.Copy(current,longform);
        END;
    END;
    (* LFN function seems to always return "u:\*" form except at root *)
    IF current[1] = colon THEN Str.Delete(current,0,2);END; (* safety *)
    fixDirectory(current);
END doGetCurrent;

PROCEDURE makebase (useLFN:BOOLEAN;spec:pathtype;VAR basepath:pathtype);
VAR
    u,d,n,e,current:pathtype;
    unit:str2;
    drive:SHORTCARD;
BEGIN
    Lib.SplitAllPath(spec,u,d,n,e);
    Str.Concat(basepath,u,d);
    IF same(basepath,"") THEN
        drive:=FIO.GetDrive(); (* yes we could use 0 as default drive *)
        doGetCurrent(useLFN,drive,  unit,current); (* "u:" and "\" or "\*\" *)
        Str.Concat(basepath, unit,current);
    END;
END makebase;

PROCEDURE WrQuoted (S:ARRAY OF CHAR);
BEGIN
    WrStr(dquote);  WrStr(S);  WrStr(dquote);
END WrQuoted;

PROCEDURE WrFname (useLFN:BOOLEAN;S:pathtype );
BEGIN
    IF useLFN THEN
        WrQuoted(S);
    ELSE
        WrStr(S);
    END;
END WrFname;

PROCEDURE chkUD (S:pathtype):BOOLEAN ;
VAR
    u,d,n,e:pathtype;
    pb:CARDINAL;
BEGIN
    Lib.SplitAllPath(S , u,d,n,e);
    pb:=0;
    IF chkJoker(u) THEN INC(pb);END;
    IF chkJoker(d) THEN INC(pb);END;
    RETURN (pb=0);
END chkUD;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE tally (substring,S:ARRAY OF CHAR) : CARDINAL;
VAR
    n,len,p:CARDINAL ;
BEGIN
    len := Str.Length(S);
    n:=0;
    p:=Str.Pos(S,substring);
    IF p # MAX(CARDINAL) THEN
        LOOP
            INC(n);
            INC(p);
            IF p >= len THEN EXIT;END;
            p := Str.NextPos(S,substring,p);
            IF p = MAX(CARDINAL) THEN EXIT;END;
        END;
    END;
    RETURN n;
END tally;

(* assume S is already fully trimed *)

PROCEDURE atstart (substring,S : ARRAY OF CHAR   ) : BOOLEAN ;
VAR
    S2 : str128;
BEGIN
    IF S[0] # substring[0] THEN RETURN FALSE; END; (* quick test *)
    IF same(S,substring) THEN RETURN TRUE; END;
    Str.Concat(S2,substring,blank);
    IF Str.Pos(S,S2)=0 THEN RETURN TRUE; END;
    Str.Concat(S2,substring,tabul);
    IF Str.Pos(S,S2)=0 THEN RETURN TRUE; END;
    RETURN FALSE;
END atstart;

PROCEDURE atend (substring,S : ARRAY OF CHAR   ) : BOOLEAN ;
VAR
    S2 : str128;
    len,lensub:CARDINAL;
BEGIN
    lensub:=Str.Length(substring);
    len:=Str.Length(S);
    IF len < lensub THEN RETURN FALSE; END;
    IF S[len-lensub] # substring[0] THEN RETURN FALSE; END; (* quick test *)
    IF same(S,substring) THEN RETURN TRUE; END;

    INC(lensub); (* we'll add a blank *)
    IF len < lensub THEN RETURN FALSE; END;
    Str.Concat(S2,blank,substring);
    IF Str.Pos(S,S2)=(len-lensub) THEN RETURN TRUE; END;
    Str.Concat(S2,tabul,substring);
    IF Str.Pos(S,S2)=(len-lensub) THEN RETURN TRUE; END;
    RETURN FALSE;
END atend;

PROCEDURE atstartandend (sub1,sub2,S:ARRAY OF CHAR   ):BOOLEAN ;
VAR
    len,sublen : CARDINAL;
    R : str128;
BEGIN
    (*
    IF atstart(sub1,S) THEN
        len:=Str.Length(S);
        sublen:=Str.Length(sub2);
        IF len >= sublen THEN
            Str.Slice(R, S, len-sublen, sublen);
            RETURN same(R,sub2);
        END;
    END;
    RETURN FALSE;
    *)
    RETURN ( atstart(sub1,S) AND atend(sub2,S) );
END atstartandend;

PROCEDURE herebutnotalone ( sub1,sub2,S:ARRAY OF CHAR  ):BOOLEAN ;
BEGIN
    IF Str.Pos(S,sub1)=MAX(CARDINAL) THEN RETURN FALSE; END;
    IF Str.Pos(S,sub2)=MAX(CARDINAL) THEN RETURN FALSE; END;
    (* ok, we've got both : now check if commenting code *)
    IF atstart(sub1,S) THEN
        IF atend(sub2,S) THEN
            RETURN FALSE;
        ELSE
            RETURN TRUE;
        END;
    ELSE
        IF atend(sub2,S) THEN
            RETURN TRUE;
        ELSE
            RETURN TRUE;
        END;
    END;
END herebutnotalone;

(* ------------------------------------------------------------ *)

TYPE
    languagetype = (none,modula,basic,c,assembler);

CONST
    ioBufferSize    = (8 * 512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = ioBufferSize;
TYPE
    ioBuffer  = ARRAY [firstBufferByte..lastBufferByte] OF BYTE;
VAR
    bufferIn  : ioBuffer;

(* ------------------------------------------------------------ *)

VAR
    SSS : ARRAY [0..24576-1] OF CHAR; (* //OUT moved out of getLongest() *)

PROCEDURE getLongest (VAR maxlen:CARDINAL;VAR whatline, totallines,totalchars:LONGCARD;
                     verbose,useLFN:BOOLEAN;  f : pathtype);
VAR
    msg : str256;
    hin : FIO.File;
    len:CARDINAL;
    currline:LONGCARD; (* take care of possible very long text file *)
BEGIN
    msg:=msgProcessing;
    IF useLFN THEN Str.Subst(msg,hbar,dquote+hbar+dquote);END;
    Str.Subst(msg,hbar,f);
    video(msg,TRUE);
    IF verbose THEN Work(cmdInit);END;

    whatline:=0;
    maxlen:=0;

    totallines:=0;
    totalchars:=0;

    currline:=0;

    hin := fileOpenRead(useLFN,f);
    FIO.AssignBuffer(hin,bufferIn);
    FIO.EOF := FALSE;
    LOOP
        FIO.RdStr(hin,SSS);
        IF ((SSS[0]=0C) AND FIO.EOF) THEN EXIT; END;
        IF verbose THEN Work(cmdShow); END;
        INC(currline);
        len:=Str.Length(SSS);
        IF len > maxlen THEN maxlen:=len; whatline:=currline; END;
        IF len # 0 THEN
            INC(totallines);
            INC(totalchars, LONGCARD(len));
        END;
    END;
    fileClose(useLFN,hin);
    IF verbose THEN Work(cmdStop); END;
    video(msg,FALSE);
END getLongest;

(* ------------------------------------------------------------ *)

PROCEDURE dbg (n:LONGCARD;S,S2:ARRAY OF CHAR   );
BEGIN
    WrLngCard(n,6);
    WrStr(" ");
    WrStr(S2);
    WrStr(" ");
    WrStr(S);WrLn; (* ###### should do ! *)
END dbg;

PROCEDURE countLines (debug,verbose,countcomments,countempty,dump,useLFN:BOOLEAN;
                     language:languagetype; f : pathtype) : LONGCARD;
VAR
    msg : str256;
    hin : FIO.File;
    count,comments,currline: LONGCARD;
    S,SORG : str4096;
    levels:CARDINAL;
    wasREMark:BOOLEAN;
BEGIN
    msg:=msgProcessing;
    IF useLFN THEN Str.Subst(msg,hbar,dquote+hbar+dquote);END;
    Str.Subst(msg,hbar,f);

IF dump=FALSE THEN
    IF debug THEN
        WrStr(msg);WrLn;
        WrLn;
    ELSE
        video(msg,TRUE);
    END;
    IF verbose THEN Work(cmdInit);END;
END;
    currline :=0;
    levels := 0;

    count := 0;
    comments := 0;

    hin := fileOpenRead(useLFN,f);
    FIO.AssignBuffer(hin,bufferIn);
    FIO.EOF := FALSE;
    LOOP
        IF FIO.EOF THEN EXIT; END;
        wasREMark:=TRUE; (* it's easier to set this variable to false ! *)
        FIO.RdStr(hin,S);
        IF ((S[0]=0C) AND FIO.EOF) THEN EXIT; END;
        INC(currline);
        IF dump THEN
            Str.Copy(SORG,S);
        ELSE
            IF verbose THEN Work(cmdShow); END;
        END;
        RtrimBlanks(S);
        LtrimBlanks(S);
        IF Str.Length(S) = 0 THEN
            IF countempty THEN INC(count);END;
            wasREMark:=FALSE;
        ELSE
            CASE language OF
            | none :
                INC(count);
            | modula :
                (* ugly, unable to parse complex cases but who cares ? *)
                IF same(S,modularemopen) THEN (* "( *" alone *)
                    INC(levels);
                    INC(comments);
                    IF debug THEN dbg(currline,S,"--");END;
                ELSIF same(S,modularemclose) THEN (* "* )" alone *)
                    IF levels > 0 THEN DEC(levels);END; (* yes, this test should not be necessary *)
                    INC(comments);
                    IF debug THEN dbg(currline,S,"--");END;
                ELSIF atstartandend(modularemopen,modularemclose,S) THEN (* "( *...* )" *)
                    INC(comments);
                    IF debug THEN dbg(currline,S,"--");END;

                ELSIF herebutnotalone(modularemopen,modularemclose,S) THEN
                    IF levels = 0 THEN
                        (* we don't incremend comments here : we process lines ! *)
                        INC(count);
                        wasREMark:=FALSE;
                        IF debug THEN dbg(currline,S,"++");END; (* show it but let know it's counted *)
                    ELSE
                        INC(comments);
                        IF debug THEN dbg(currline,S,"--");END;
                    END;
                ELSIF atstart(modularemopen,S) THEN (* "( *..." *)
                    INC(levels);
                    INC(comments);
                    IF debug THEN dbg(currline,S,"--");END;
                ELSIF atend(modularemclose,S) THEN (* "...* )" *)
                    IF levels > 0 THEN DEC(levels); END; (* yes, this test should not be necessary *)
                    INC(comments);
                    IF debug THEN dbg(currline,S,"--");END;

                ELSE
                    IF levels = 0 THEN
                        INC(count);
                        wasREMark:=FALSE;
                    ELSE
                        INC(comments);
                        IF debug THEN dbg(currline,S,"--");END;
                    END;
                END;
            | c :
                (* pretty... basic but anyway, I don't like C *)
                IF atstart(crem,S) THEN (* // *)
                    INC(comments);
                    IF debug THEN dbg(currline,S,"--");END;
                ELSE
                    IF same(S,cremopen) THEN
                        INC(levels);
                        INC(comments);
                        IF debug THEN dbg(currline,S,"--");END;
                    ELSIF same(S,cremclose) THEN
                        IF levels > 0 THEN DEC(levels);END;
                        INC(comments);
                        IF debug THEN dbg(currline,S,"--");END;
                    ELSIF atstartandend(cremopen,cremclose,S) THEN
                        INC(comments);
                        IF debug THEN dbg(currline,S,"--");END;
                    ELSE
                        IF levels = 0 THEN
                            INC(count);
                            wasREMark:=FALSE;
                        ELSE
                            INC(comments);
                            IF debug THEN dbg(currline,S,"--");END;
                        END;
                    END;
                END;
            | basic :
                IF S[0] # basicrem1 THEN (* first char is not a single quote *)
                    Str.Caps(S);
                    IF atstart(basicrem0,S) THEN (* REM *)
                        INC(comments);
                        IF debug THEN dbg(currline,S,"--");END;
                    ELSE
                        INC(count);
                        wasREMark:=FALSE;
                    END;
                ELSE
                    INC(comments);
                    IF debug THEN dbg(currline,S,"--");END;
                END;
            | assembler:
                IF S[0] # asmcomment THEN (* first char is not ";" *)
                    INC(count);
                    wasREMark:=FALSE;
                ELSE
                    INC(comments);
                    IF debug THEN dbg(currline,S,"--");END;
                END;
            END;
        END;
        IF dump THEN
            IF countcomments THEN
                IF wasREMark THEN WrStr(SORG);WrLn;END;
            ELSE
                IF wasREMark=FALSE THEN WrStr(SORG);WrLn;END;
            END;
        END;
    END;
    fileClose(useLFN,hin);
IF dump=FALSE THEN
    IF verbose THEN Work(cmdStop); END;
    IF debug THEN
        WrLn;
    ELSE
        video(msg,FALSE);
    END;
END;
    IF countcomments THEN
        RETURN comments;
    ELSE
        RETURN count;
    END;
END countLines;

(* ------------------------------------------------------------ *)

PROCEDURE fmtbar (percentused:LONGREAL;used,free:CHAR;field:CARDINAL):str80;
VAR
    R : str80;
    p,i : CARDINAL;
BEGIN
    percentused := (percentused / 100.0 ) * LONGREAL(field);
    p := CARDINAL(percentused + 0.5); (* round ! *)
    R := "";
    FOR i := 1 TO field DO
        IF i <= p THEN
           Str.Append(R,used);
        ELSE
           Str.Append(R,free);
        END;
    END;
    RETURN R;
END fmtbar;

PROCEDURE showbargraph (semigraphics:BOOLEAN;wi:CARDINAL;count,maxcount:LONGCARD  );
CONST
    chUsed = CHR(219);  chUsedTxt = "+";
    chFree = CHR(176);  chFreeTxt = "-";
VAR
    v      : LONGREAL;
    charUsed,charFree:CHAR;
BEGIN
    IF semigraphics THEN
        charUsed := chUsed ;     charFree := chFree;
    ELSE
        charUsed := chUsedTxt ;  charFree := chFreeTxt;
    END;
    v := ( LONGREAL(count) / LONGREAL(maxcount) ) * 100.0;
    WrStr(fmtbar(v,charUsed,charFree,wi));
END showbargraph;

(* ------------------------------------------------------------ *)

(* verbose,dump,debug: not used here *)

PROCEDURE showbychar (charcol,wi,barwi:CARDINAL;
                     verbose,dump,showbar,semigraphics,debug,useLFN:BOOLEAN;
                     f:pathtype);
CONST
    (*        "____'?'" *)
    sep     = "    ";
VAR
    hin : FIO.File;
    total,count : LONGCARD;
    S : str4096;
    oldch,newch:CHAR;
    pass,lastpass,len:CARDINAL;
    R:str16;
BEGIN
    DEC(charcol); (* 1.. to 0.. *)

    total := 0;

    IF showbar THEN
        lastpass:=2;
    ELSE
        lastpass:=1;
    END;

    FOR pass:=1 TO lastpass DO
        count := 0;
        oldch := '';

        hin := fileOpenRead(useLFN,f);
        FIO.AssignBuffer(hin,bufferIn);
        FIO.EOF := FALSE;
        LOOP
            IF FIO.EOF THEN EXIT; END;
            FIO.RdStr(hin,S);
            IF ((S[0]=0C) AND FIO.EOF) THEN EXIT; END;
            len:=Str.Length(S);
            IF len > 0 THEN
                DEC(len);
                IF charcol <= len THEN
                    newch:=S[charcol];
                    IF newch=oldch THEN
                        INC(count);
                    ELSE
                        IF count > 0 THEN
                            IF pass=1 THEN INC(total,count); END;
                            IF pass=lastpass THEN
                                WrLngCard(count,wi);
                                Str.Concat(R,sep+'"',oldch); Str.Append(R,'"');
                                WrStr(R);
                                IF pass=2 THEN
                                    WrStr("  ");
                                    showbargraph (semigraphics,barwi,count,total);
                                END;
                                WrLn;
                            END;
                        END;
                        count:=1;
                        oldch:=newch;
                    END;
                END;
            END;
        END;
        fileClose(useLFN,hin);
        (* handle pending *)
                        IF count > 0 THEN
                            IF pass=1 THEN INC(total,count); END;
                            IF pass=lastpass THEN
                                WrLngCard(count,wi);
                                Str.Concat(R,sep+'"',oldch); Str.Append(R,'"');
                                WrStr(R);
                                IF pass=2 THEN
                                    WrStr("  ");
                                    showbargraph (semigraphics,barwi,count,total);
                                END;
                                WrLn;
                            END;
                        END;
        IF pass=lastpass THEN
            FOR len:=1 TO wi DO WrStr("=");END;WrLn;
            WrLngCard(total,wi);WrLn;
        END;
    END;

END showbychar;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE getcard (VAR v:CARDINAL;lower,upper:CARDINAL;S:ARRAY OF CHAR):BOOLEAN;
VAR
    rc:BOOLEAN;
    lc:LONGCARD;
BEGIN
    rc:=GetLongCard(S,lc);
    IF rc THEN
        rc:=(lc <= MAX(CARDINAL) );
        IF rc THEN
            v:=CARDINAL(lc);
            rc:=( (v >= lower) AND (v <= upper) );
        END;
    END;
    RETURN rc;
END getcard;

(* yes, we know rounding is faulty but we don't care here *)

PROCEDURE fmtratio (totallines,totalchars:LONGCARD;wi:CARDINAL;fracsep:CHAR):str16;
VAR
    R,RR:str16;
    avg,frac:LONGCARD;
    ok:BOOLEAN;
    i:CARDINAL;
BEGIN
    IF totallines = 0 THEN totallines:=1; END;
    avg:= totalchars DIV totallines;
    Str.CardToStr(avg,R,10,ok);
    frac:=( totalchars - avg*totallines )*100;
    frac:=( frac DIV totallines );
    Str.CardToStr(frac,RR,10,ok);
    IF frac = 0 THEN
        Str.Concat(RR,fracsep,"00");
    ELSE
        FOR i:=Str.Length(RR)+1 TO 2 DO Str.Prepend(RR,"0");END;
        Str.Prepend(RR,fracsep);
    END;
    Str.Append(R,RR);
    FOR i:=Str.Length(R)+1 TO (wi+1+2) DO Str.Prepend(R," ");END;
    RETURN R;
END fmtratio;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    defaultBarWidth    = 25;
    minBarWidth        = 1;
    maxBarWidth        = 100;
    defaultFieldWidth  = 9;
CONST
    msgLines    = " line(s) in ";
    msgComments = " comment(s) in ";
    (*            "######### line(s) in ????????.???" *)
    msgDashes   = "==========================================";
    msgFiles    = " files";
VAR
    parmcount,i,opt : CARDINAL;
    S,R,filespec    : pathtype; (* was str128 *)
    NFO             : str128;
    LNG             : str16;
    state           : (waiting,gotparm1);
VAR
    countFile       : CARDINAL;
    file,basedir    : pathtype;
    ptr,anchor      : pFname;
VAR
    debug,useLFN,verbose,countcomments,countempty,dump,findaverage : BOOLEAN;
    language : languagetype;
    total,nlines,whatline,totallines,totalchars: LONGCARD;
    bychar,showbar ,semigraphics: BOOLEAN;
    charcol,wi,maxlen,barwi:CARDINAL;
    poll,currpoll:CARDINAL;
    redirected:BOOLEAN;
    cmd:(countlines,findlongestline);
BEGIN
    Lib.DisableBreakCheck();
    redirected:=IsRedirected();
    IF redirected=FALSE THEN WrLn; END;

    debug :=FALSE;

    useLFN  := TRUE;
    verbose := FALSE;
    countcomments :=FALSE;
    countempty := FALSE;
    language := none;
    dump     := FALSE;
    showbar  := FALSE;
    semigraphics:=TRUE;
    barwi    := defaultBarWidth;
    wi       := defaultFieldWidth;
    bychar   := FALSE; (* another mode *)
    charcol  := 1;
    findaverage:=FALSE;
    cmd      := countlines;

    state   := waiting;
    parmcount := Lib.ParamCount();
    IF parmcount=0 THEN abort(errHelp,"");END;
    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i);
        Str.Copy(R,S);
        UpperCase(R);
        cleantabs(R);
        IF isOption(R) THEN
            opt := GetOptIndex (R, "?"+delim+"H"+delim+"HELP"+delim+
                                   "V"+delim+"VERBOSE"+delim+
                                   "M"+delim+"MODULA"+delim+
                                   "B"+delim+"BASIC"+delim+
                                   "C"+delim+
                                   "DEBUG"+delim+
                                   "K"+delim+"COMMENTS"+delim+
                                   "E"+delim+"EMPTY"+delim+
                                   "A"+delim+"ASM"+delim+
                                   "D"+delim+"DUMP"+delim+
                                   "S"+delim+"BYCHAR"+delim+
                                   "C:"+delim+"COLUMN:"+delim+
                                   "F:"+delim+"FIELD:"+delim+
                                   "F"+delim+"FIND"+delim+"LONGEST"+delim+
                                   "X"+delim+"LFN"+delim+
                                   "S:"+delim+
                                   "G"+delim+"BARGRAPH"+delim+
                                   "GG"+delim+
                                   "W:"+delim+"WIDTH:"+delim+
                                   "FF"
                               );
            CASE opt OF
            | 1,2,3 : abort(errHelp,"");
            | 4,5   : verbose := TRUE;
            | 6,7   : CASE language OF
                      | none,modula: language := modula;
                      ELSE abort(errNonsense,"");
                      END;
            | 8,9   : CASE language OF
                      | none,basic : language := basic;
                      ELSE abort(errNonsense,"");
                      END;
            | 10    : CASE language OF
                      | none,c : language := c;
                      ELSE abort(errNonsense,"");
                      END;
            | 11    : debug := TRUE;
            | 12,13 : countcomments:=TRUE;
            | 14,15 : countempty := TRUE;
            | 16,17 : CASE language OF
                      | none,assembler : language := assembler;
                      ELSE abort(errNonsense,"");
                      END;
            | 18,19:  dump:=TRUE;
            | 20,21:  bychar:=TRUE;
            | 22,23:  IF getcard(charcol,1,127,R)=FALSE THEN abort(errBadRange,S);END;
            | 24,25:  IF getcard(wi,1,12,R)=FALSE THEN abort(errBadRange,S);END;
            | 26,27,28:cmd:=findlongestline;
            | 29,30:  useLFN:=FALSE;
            | 31:     bychar:=TRUE;
                      IF getcard(charcol,1,127,R)=FALSE THEN abort(errBadRange,S);END;
            | 32,33:  showbar        := TRUE;
            | 34:     showbar        := TRUE; semigraphics:=FALSE;
            | 35,36:  IF getcard(barwi,minBarWidth,maxBarWidth,R)=FALSE THEN abort(errBadRange,S);END;
            | 37:     cmd:=findlongestline;
                      findaverage:=TRUE;
            ELSE
                abort(errOption,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            CASE state OF
            | waiting : Str.Copy(filespec,S);
            | gotparm1: abort(errTooManyParms,S);
            END;
            INC(state);
        END;
    END;
    (* check nonsense *)
    IF state=waiting THEN abort(errMissingSpec,"");END;
    useLFN:=( useLFN AND fileSupportLFN() );

    CASE cmd OF
    | findlongestline:
        IF (debug AND verbose) THEN abort(errSilly,"");END;
        IF (debug AND dump) THEN abort(errSilly,"");END;
        IF (dump AND verbose) THEN abort(errSilly,"");END;
        IF (dump AND (language=none)) THEN abort(errVerySilly,"");END;

        IF ( countcomments AND (language=none) ) THEN abort(errComment,"");END;

        IF ( countempty AND (language # none) ) THEN abort(errRidiculous,"");END;

        IF ( bychar AND (language # none) ) THEN abort(errNonsenseWithSort,"");END;

        IF showbar THEN abort(errNonsenseFinding,"");END;

    END;

    (* a few q&d sanity checks *)

    IF chkUD(filespec)=FALSE THEN abort(errBadSpec,filespec);END;
    IF same(filespec,dot) THEN Str.Copy(filespec,stardotstar); END;
    IF same(filespec,star) THEN Str.Copy(filespec,stardotstar); END;
    IF Str.Match(filespec,"*"+antislash) THEN Str.Append(filespec,stardotstar); END;
    IF chkJoker(filespec)=FALSE THEN
        IF fileIsDirectorySpec ( useLFN,filespec) THEN
            fixDirectory(filespec);
            Str.Append(filespec,stardotstar);
        END;
    END;
    makebase(useLFN,filespec,basedir);

    showmem(debug,"Before buildFileList()");

    initList(anchor);
    countFile := buildFileList(anchor,useLFN,debug,filespec);

    showmem(debug,"After buildFileList()");
IF debug THEN
    ptr:=anchor;
    WHILE ptr # NIL DO
        getStr(filespec,ptr);
        Str.Concat(file,basedir,filespec);
        WrFname(useLFN,file);WrLn;
        ptr:=ptr^.next;
    END;
END;
    CASE countFile OF
    | 0 :            abort(errNoMatch,filespec);
    | 1 : ;
    | MAX(CARDINAL): abort(errTooMany,filespec);
    ELSE
        IF dump THEN abort(errThereCanBeOnlyOne,filespec); END;
    END;

    IF bychar THEN
        IF countFile # 1 THEN abort(errThereCanBeOnlyOneHere,filespec);END;
    END;

    CASE cmd OF
    | countlines:
        IF dump=FALSE THEN
            IF redirected=FALSE THEN
                WrStr(Banner);WrLn;
                WrLn;
            END;

            IF countcomments THEN
                Str.Copy(NFO,msgComments); (* " comments IN " *)
                CASE language OF
                | modula   : LNG:="Modula-2";
                | basic    : LNG:="BASIC";
                | c        : LNG:="C";
                | assembler: LNG:="Assembler";
                ELSE
                             LNG:="";
                END;
                IF same(LNG,"")=FALSE THEN
                    WrStr(sINFO+"Counting ");WrStr(LNG);WrStr(" comments."+nl+nl);
                END;
            ELSE
                Str.Copy(NFO,msgLines); (* " lines IN " *)
                CASE language OF
                | modula   : LNG:="Modula-2";
                | basic    : LNG:="BASIC";
                | c        : LNG:="C";
                | assembler: LNG:="Assembler";
                ELSE
                             LNG:="";
                END;
                IF same(LNG,"")=FALSE THEN
                    WrStr(sINFO);WrStr(LNG);WrStr(" comments will be ignored."+nl+nl);
                END;
            END;
            IF countempty THEN
                WrStr(sINFO+"Counting empty or blank lines too.");WrLn;WrLn;
            END;
        END;
    END;

    IF useLFN THEN
        poll := 50; (* vindoze does not like frequent calls TO chkEscape() *)
    ELSE
        poll := 5;
    END;
    currpoll:=0;

    total := 0;
    ptr:=anchor;
    WHILE ptr # NIL DO
        getStr(filespec,ptr);
        Str.Concat(file,basedir,filespec);

IF debug THEN WrFname(useLFN,file);WrLn;END;

        CASE cmd OF
        | findlongestline:
            getLongest(maxlen,whatline, totallines,totalchars, verbose,useLFN,file);
            WrStr("Maximum length = ");WrCard( maxlen,wi);
            WrStr("   "+" (line ");
            WrLngCard( whatline,wi); (* 9 digits should do ! *)
            WrStr(") in ");WrFname(useLFN,file);WrLn;
            IF findaverage THEN
                WrStr("Total chars    = ");WrLngCard(totalchars,wi);WrLn;
                WrStr("Total lines    = ");WrLngCard(totallines,wi);WrLn;
                WrStr("Average length = ");
                WrStr( fmtratio(totallines,totalchars,wi,coma) );WrLn;
                IF (ptr^.next # NIL) THEN WrLn; END;
            END;
        | countlines:
            IF bychar THEN
                showbychar (charcol,wi,barwi,
                           verbose,dump,showbar,semigraphics,debug,useLFN,file);
            ELSE
                nlines:=countLines (debug,verbose,countcomments,countempty,dump,useLFN,
                                   language,file);
                INC(total,nlines);
                IF dump=FALSE THEN
                    WrLngCard(nlines,wi);
                    WrStr(NFO);
                    WrFname(useLFN,file);WrLn;
                END;
            END;
        END;

        INC(currpoll);
        IF (currpoll MOD poll) = 0 THEN
            IF ChkEscape() THEN
                freeList(anchor);
                abort(errAborted,"");
            END;
            currpoll:=0;
        END;

        ptr:=ptr^.next;
    END;
    freeList(anchor);

    showmem(debug,"After freeList()");

    CASE cmd OF
    | countlines:
        IF dump=FALSE THEN
            IF countFile > 1 THEN
                WrStr(msgDashes);WrLn;
                WrLngCard(total,wi);
                WrStr(NFO);WrStr("all ");
                WrCard(countFile,0);
                WrStr(msgFiles);WrLn;
            END;
       END;
    END;

    abort(errNone,"");
END LCount.


