(* ---------------------------------------------------------------
Title         Q&D Trim
Overview      remove leading or trailing blanks and tabs
Usage         see help
Notes         very, very, very quick & dirty... :-(
              minimal error messages and checking, etc.
Bugs
Wish List     protect remarks ( ";*" "#*" ) ?

--------------------------------------------------------------- *)

MODULE Trim;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;
IMPORT SYSTEM;
IMPORT IO;

FROM QD_ASCII IMPORT dash, slash, nullchar, tabchar, cr, lf, nl, bs,
space, dot, deg, doublequote, quote, colon, percent, vbar,
blank, equal, dquote, charnull, singlequote, antislash, dollar,
star, backslash, coma, question, underscore, tabul, hbar,
comma, semicolon, diese, pound, openbracket, closebracket, tilde, exclam,
stardotstar, dotdot, escCh, escSet, letters, digits,
lettersUpp, lettersLow, openbrace, closebrace;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits,
getAllLegalUnits, metaproc, getCli, argc, argv;

FROM QD_File IMPORT pathtype, w9XnothingRequired,
fileOpenRead, fileOpen, fileExists, fileExistsAlt,
fileIsRO, fileSetRW, fileSetRO,
fileErase, fileCreate, fileRename, fileGetFileSize, fileGetFileStamp,
fileIsDirectorySpec, fileClose, fileFlush, fileSupportLFN;

FROM QD_LFN IMPORT path9X, huge9X, findDataRecordType,
unicodeConversionFlagType, w9XchangeDir,
w9XgetDOSversion, w9XgetTrueDOSversion, w9XisWindowsEnh, w9XisMSDOS7,
w9XfindFirst, w9XfindNext, w9XfindClose, w9XgetCurrentDirectory,
w9XlongToShort, w9XshortToLong, w9XtrueName, w9XchangeDir,
w9XmakeDir, w9XrmDir, w9Xrename, w9XopenFile, w9XcloseFile,
w9XsupportLFN;

FROM IO IMPORT WrStr, WrLn;

FROM Storage IMPORT DEALLOCATE,Available,ALLOCATE;

(* ------------------------------------------------------------ *)

CONST
    ProgEXEname     = "TRIM";
    ProgTitle       = "Q&D Trim";
    ProgVersion     = "v1.1";
    ProgCopyright   = "by PhG";
    Banner          = ProgTitle+" "+ProgVersion+" "+ProgCopyright;
CONST
    extBAK          = ".BK!";
    extCOM          = ".COM";
    extEXE          = ".EXE";
    extDLL          = ".DLL";
    extOVR          = ".OVR";
    extOVL          = ".OVL";
    extDRV          = ".DRV";
    extZIP          = ".ZIP";
    extARJ          = ".ARJ";
    extLZH          = ".LZH";
    skippedextensions=extBAK+delim+extCOM+delim+extEXE+delim+
                      extDLL+delim+extOVR+delim+extOVL+delim+extDRV+delim+
                      extZIP+delim+extARJ+delim+extLZH;
CONST
    errNone         = 0;
    errHelp         = 1;
    errOption       = 2;
    errTooManyParms = 3;
    errBadCard      = 4;
    errTabRange     = 5;
    errMissingSpec  = 6;
    errCmdNeeded    = 7;
    errTooManyFiles = 8;
    errNotFound     = 9;
    errFileNames    = 10;
    errQuotes       = 11;
    errRelaxMode    = 12;
    errGrabNonsense = 13;
    errGrabOnly     = 14;
    errBadSpec      = 15;
    (* errQuotesCmd    = 16; *)

(* ------------------------------------------------------------ *)

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
    placeholder = vbar;
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)

errmsg =
Banner+nl+
nl+
"Syntax 1 : "+ProgEXEname+" <file> [option]..."+nl+
"Syntax 2 : "+ProgEXEname+" <file> <-j> [-x]"+nl+
nl+
"This program removes trailing, leading or embedded spaces and tabs,"+nl+
"encloses each line with double quotes, removes enclosing double quotes,"+nl+
"filters out lines without any probable text string, or fixes files."+nl+
nl+
"-r      remove trailing spaces and tabs"+nl+
"-l      remove smallest possible number of leading spaces and tabs"+nl+
"-ll     remove all leading spaces and tabs"+nl+
"-b      remove both leading and trailing spaces and tabs (same as -r -ll)"+nl+
"-t:#    tabulation width ([1..256], default is 8)"+nl+
"-n      replace internal spaces and tabs with a dot for filenames (-b forced)"+nl+
"-nn     remove spaces and tabs preceding a dot for filenames (-b forced)"+nl+
"-e[e]   enclose each line with double quotes (-ee = -b then -q)"+nl+
"-q[q]   remove enclosing double quotes from each line (-qq = -b then -q)"+nl+
"-p      remove empty lines (forced by -s[s|a])"+nl+
"-j      remove leading or trailing double quote only if its count is odd"+nl+
"-k      ignore opening/closing match check when removing double quotes (-q[q])"+nl+
"-s[s|a] filter out lines without any probable text string (-ss = -sa = -s -a)"+nl+
"-a      show line number before any line containing a probable text string"+nl+
"-x      disable LFN support even if available"+nl+
nl+
'a) If no option is specified, program defaults to "-r -l".'+nl+
"b) With -l[l] option, leading tabs are expanded to spaces."+nl+
"c) Note -e and -q options will process lines only when necessary."+nl+
"d) A probable text string is delimited by single or double quotes."+nl+
"   Note false positive detection may happen."+nl+
"e) -s[s|a] option forces -p option for obvious reasons."+nl+
"f) -j option (syntax 2) ignores any other option than -x option :"+nl+
"   tabs, leading and/or trailing blanks, if any, are left unmodified."+nl+
"   This option was created in order to fix fortune cookies files."+nl+
"g) "+skippedextensions+" files will be ignored."+nl+
"h) As a safety, backups (with "+extBAK+" extension) are always created."+nl;

VAR
    S : str1024;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(errmsg);
    | errOption:
        S:='Unknown "'+placeholder+'" option !';
    | errTooManyParms:
        S:='"'+placeholder+'" parameter is one too many !';
    | errBadCard:
        S:='Illegal "'+placeholder+'" value !';
    | errTabRange:
        S := "Tabulation count must be in the [1..256] range !";
    | errMissingSpec:
        S := "Missing file specification !";
    | errCmdNeeded:
        S := "Missing command !";
    | errTooManyFiles:
        S:='Too many files match "'+placeholder+'" specification !';
    | errNotFound:  (* either FNF OR forbidden extension *)
        S:='No valid file matches "'+placeholder+'" specification !';
    | errFileNames:
        S := "-n and -nn options are mutually exclusive !";
    | errQuotes:
        S := "-e and -q options are mutually exclusive !";
    | errRelaxMode:
        S := "-k option requires -q[q] option !";
    | errGrabNonsense:
        S:='"'+placeholder+'" and -s options are mutually exclusive !';
    | errGrabOnly:
        S := "-a option is a nonsense without -s option !";
    | errBadSpec:
        S:='Illegal "'+placeholder+'" specification !';
    (*
    | errQuotesCmd:
        S := "-e and -q options cannot be mixed with any other option !";
    *)
    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp:
        ;
    ELSE
        Str.Subst(S,placeholder,einfo);
        WrStr(ProgEXEname+" : ");WrStr(S);WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

(* taken almost verbatim from OLDNEW v1.1f code, out of weariness *)

TYPE
    pFname = POINTER TO fnameType;
    fnameType = RECORD
        next      : pFname;
        slen      : CARDINAL; (* a SHORTCARD could do but who knows ? *)
        str       : CHAR;
    END;

PROCEDURE initList (VAR anchor : pFname );
BEGIN
    anchor := NIL;
END initList;

PROCEDURE freeList (anchor : pFname);
VAR
    needed : CARDINAL;
    p      : pFname;
BEGIN
    (* p:=anchor; *)
    WHILE anchor # NIL DO
        needed := SIZE(fnameType) - SIZE(anchor^.str) + anchor^.slen;
        p := anchor^.next;
        DEALLOCATE(anchor,needed);
        anchor:=p;
    END
END freeList;

PROCEDURE buildNewPtr (VAR anchor,p:pFname; len:CARDINAL):BOOLEAN;
VAR
    needed : CARDINAL;
BEGIN
    needed := SIZE(fnameType) - SIZE(p^.str) + len;
    IF Available(needed)=FALSE THEN RETURN FALSE; END;
    IF anchor = NIL THEN
        ALLOCATE(anchor,needed);
        p:=anchor;
    ELSE
        p:=anchor;
        WHILE p^.next # NIL DO
            p:=p^.next;
        END;
        ALLOCATE(p^.next,needed);
        p:=p^.next;
    END;
    p^.next := NIL;
    RETURN TRUE;
END buildNewPtr;

(* assume p is valid *)

PROCEDURE getStr (VAR S : pathtype; p:pFname);
VAR
    len:CARDINAL;
BEGIN
    len := p^.slen;
    Lib.FastMove( ADR(p^.str),ADR(S),len);
    S[len] := nullchar; (* REQUIRED safety ! *)
END getStr;

(* ------------------------------------------------------------ *)

(* Str.Match is not case sensitive *)

PROCEDURE isReservedPattern (S,skipthem:ARRAY OF CHAR):BOOLEAN;
VAR
    e3 : str16;
    n:CARDINAL;
    rc:BOOLEAN;
BEGIN
    rc:=FALSE;
    n:=0;
    LOOP
        isoleItemS(e3, skipthem,delim,n);
        IF same(e3,"") THEN EXIT; END;
        Str.Prepend(e3,"*");
        IF Str.Match(S,e3) THEN rc:=TRUE;EXIT; END;
        INC(n);
    END;
    RETURN rc;
END isReservedPattern;

PROCEDURE isReservedEntry (S:ARRAY OF CHAR) : BOOLEAN;
BEGIN
    IF same(S,dot) THEN RETURN TRUE; END;
    RETURN same(S,dotdot);
END isReservedEntry;

PROCEDURE buildFileList (VAR anchor:pFname;
                        noskip,useLFN,DEBUG :BOOLEAN;
                        spec:pathtype):CARDINAL;
VAR
    count:CARDINAL; (* should do ! *)
    ok,found:BOOLEAN;
    unicodeconversion:unicodeConversionFlagType;
    w9Xentry : findDataRecordType;
    w9Xhandle,errcode:CARDINAL;
    entry : FIO.DirEntry;
    dosattr:FIO.FileAttr;
    entryname:pathtype;
    len : CARDINAL;
    pp:pFname;
    excludeme1,excludeme2:BOOLEAN;
BEGIN
    count:=0;
    IF useLFN THEN
        found := w9XfindFirst (spec,SHORTCARD(everything),SHORTCARD(w9XnothingRequired),
                              unicodeconversion,w9Xentry,w9Xhandle,errcode);
    ELSE
        found := FIO.ReadFirstEntry(spec,everything,entry);
    END;
    WHILE found DO
        IF useLFN THEN
            Str.Copy(entryname,w9Xentry.fullfilename);
        ELSE
            Str.Copy(entryname,entry.Name);
        END;
        excludeme1 := isReservedEntry   (entryname);  (* skip "." and ".." *)
        IF noskip THEN
            excludeme2 := FALSE;
        ELSE
            excludeme2 := isReservedPattern (entryname,skippedextensions );
        END;
        IF NOT(excludeme1 OR excludeme2) THEN
            IF useLFN THEN
                dosattr:=FIO.FileAttr(w9Xentry.attr AND 0FFH);
            ELSE
                dosattr:=entry.attr;
            END;
            IF NOT (aD IN dosattr) THEN
                (* if file has no extension, add it as a marker *)
                IF Str.RCharPos(entryname,".")=MAX(CARDINAL) THEN
                    Str.Append(entryname,".");
                END;
                IF DEBUG THEN WrStr("Included : ");WrStr(entryname);WrLn; END;
                len:=Str.Length(entryname);
                IF buildNewPtr(anchor,pp,len)=FALSE THEN
                    IF useLFN THEN ok:=w9XfindClose(w9Xhandle,errcode); END;
                    RETURN MAX(CARDINAL); (* errStorage *)
                END;
                INC(count);
                pp^.slen      := len;
                Lib.FastMove ( ADR(entryname),ADR(pp^.str),len );
            ELSE
                IF DEBUG THEN WrStr("Ignored  : ");WrStr(entryname);WrLn;END;
            END;
        ELSE
            IF DEBUG THEN WrStr("Excluded : ");WrStr(entryname);WrLn;END;
        END;
        IF useLFN THEN
            found :=w9XfindNext(w9Xhandle, unicodeconversion,w9Xentry,errcode);
        ELSE
            found :=FIO.ReadNextEntry(entry);
        END;
    END;
    IF useLFN THEN ok:=w9XfindClose(w9Xhandle,errcode); END;
    RETURN count;
END buildFileList;

(* ------------------------------------------------------------ *)

PROCEDURE doGetCurrent (useLFN:BOOLEAN;drive:SHORTCARD;
                       VAR unit:str2; VAR current:pathtype);
VAR
    rc:CARDINAL;
    longform:pathtype;
BEGIN
    Str.Concat(unit, CHR( ORD("A")-1+ORD(drive) ),colon);

    FIO.GetDir(drive,current); (* \path without u: nor trailing \ except at root *)
    IF current[1] # colon THEN Str.Prepend(current,unit); END; (* safety *)
    IF useLFN THEN
        IF w9XshortToLong(current,rc,longform) THEN (* if error, keep DOS current *)
            Str.Copy(current,longform);
        END;
    END;
    (* LFN function seems to always return "u:\*" form except at root *)
    IF current[1] = colon THEN Str.Delete(current,0,2);END; (* safety *)
    fixDirectory(current);
END doGetCurrent;

PROCEDURE makebase (useLFN:BOOLEAN;spec:pathtype;VAR basepath:pathtype);
VAR
    u,d,n,e,current:pathtype;
    unit:str2;
    drive:SHORTCARD;
BEGIN
    Lib.SplitAllPath(spec,u,d,n,e);
    Str.Concat(basepath,u,d);
    IF same(basepath,"") THEN
        drive:=FIO.GetDrive(); (* yes we could use 0 as default drive *)
        doGetCurrent(useLFN,drive,  unit,current); (* "u:" and "\" or "\*\" *)
        Str.Concat(basepath, unit,current);
    END;
END makebase;

PROCEDURE chkUD (S:pathtype):BOOLEAN ;
VAR
    u,d,n,e:pathtype;
    pb:CARDINAL;
BEGIN
    Lib.SplitAllPath(S , u,d,n,e);
    pb:=0;
    IF chkJoker(u) THEN INC(pb);END;
    IF chkJoker(d) THEN INC(pb);END;
    RETURN (pb=0);
END chkUD;

PROCEDURE WrFname (useLFN:BOOLEAN;F:pathtype );
BEGIN
    IF useLFN THEN
        WrStr(dquote);WrStr(F);WrStr(dquote);
    ELSE
        WrStr(F);
    END;
END WrFname;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    ioBufferSize    = (8 * 512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = ioBufferSize;
TYPE
    ioBuffer  = ARRAY [firstBufferByte..lastBufferByte] OF BYTE;
VAR
    bufferIn  : ioBuffer;
    bufferOut : ioBuffer;
    hugestr   : ARRAY[0..32768-1] OF CHAR; (* does not like to be local ! ... *)
    hugetmp   : ARRAY[0..32768-1] OF CHAR; (* ... possibly because of stack size *)

(* ------------------------------------------------------------ *)

(* we use global hugestr and hugetmp here as not to fill local procedure stack *)

PROCEDURE countDetabbedLeft (tabwidth:CARDINAL;createtmp:BOOLEAN):CARDINAL ;
VAR
    len,i,j,k,add: CARDINAL;
    c : CHAR;
BEGIN
    IF createtmp THEN Str.Copy(hugetmp,"");END;
    len:=Str.Length(hugestr);
    IF len=0 THEN RETURN MAX(CARDINAL); END; (* ignore empty lines *)
    j:=0; (* yes, 0 and not 1 ! *)
    i:=1;
    LOOP
        c:=hugestr[i-1];
        IF ORD(c) > ORD(space) THEN EXIT; END;
        IF c = tabul THEN
            add := tabwidth - (j MOD tabwidth);
            WHILE add > 0 DO
                IF createtmp THEN Str.Append(hugetmp,space);END;
                INC(j);
                DEC(add);
            END;
        ELSE
            IF createtmp THEN Str.Append(hugetmp,c);END;
            INC(j);
        END;
        INC(i);
        IF i > len THEN EXIT; END;
    END;
    IF createtmp THEN
        (* yes, we could do something smarter... *)
        FOR k:= i TO len DO
            Str.Append(hugetmp,hugestr[k-1]);
        END;
    END;
    RETURN j; (* yes, j and not i *)
END countDetabbedLeft;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE unquote(bothquotes,relaxmode:BOOLEAN ; VAR S:ARRAY OF CHAR);
VAR
    len,last,i,performed:CARDINAL;
    ch:CHAR;
    pat:str16;
BEGIN
    IF bothquotes THEN
        last:=2;
    ELSE
        last:=1;
    END;
    i:=1;
    LOOP
        CASE i OF
        | 1: ch:=doublequote;
        | 2: ch:=singlequote;
        END;
        IF relaxmode THEN
            performed:=0;
            Str.Concat(pat,ch,"*");
            IF Str.Match(S,pat) THEN Str.Delete(S,0,1); INC(performed); END;
            Str.Concat(pat,"*",ch);
            IF Str.Match(S,pat) THEN
                len:=Str.Length(S);
                Str.Delete(S,len-1,1);
                INC(performed);
            END;
            IF performed # 0 THEN EXIT; END;
        ELSE
            Str.Concat(pat,ch,"*");Str.Append(pat,ch);
            IF Str.Match(S,pat) THEN
                Str.Delete(S,0,1); (* first doublequote *)
                len:=Str.Length(S);
                Str.Delete(S,len-1,1);
                EXIT;
            END;
        END;
        INC(i);
        IF i > last THEN EXIT; END;
    END;
END unquote;

(* ------------------------------------------------------------ *)

PROCEDURE fmtbignum (v:LONGCARD; base:CARDINAL;wi:INTEGER; pad:CHAR ):str16;
VAR
    S:str16;
    ok:BOOLEAN;
    i : CARDINAL;
BEGIN
    Str.CardToStr( v, S, base,ok);
    IF base=16 THEN Str.Lows(S);END;
    FOR i:=Str.Length(S)+1 TO ABS(wi) DO
         IF wi < 0 THEN
             Str.Append(S,pad);
         ELSE
             Str.Prepend(S,pad);
         END;
    END;
    RETURN S;
END fmtbignum;

PROCEDURE fmtnum ( v:CARDINAL; base:CARDINAL;wi:INTEGER; pad:CHAR ):str16;
BEGIN
    RETURN fmtbignum( LONGCARD(v),base,wi,pad);
END fmtnum;

(* "*" OR '*' : we use hugetmp instead of hugestr *)

PROCEDURE hasValidString (  ):BOOLEAN;
VAR
    i,p,len,sfound,slen:CARDINAL;
    ch,sdelim:CHAR;
    state:(waiting,instring);
BEGIN
    Str.Copy(hugetmp,hugestr);
    len:=Str.Length(hugetmp);
    IF len=0 THEN RETURN FALSE;END;
    sfound:=0;
    FOR i:=1 TO 2 DO
        CASE i OF
        | 1 : sdelim:=doublequote;
        | 2 : sdelim:=singlequote;
        END;
        p:=0;
        state:=waiting;
        LOOP
            ch:=hugetmp[p];
            CASE state OF
            | waiting:
                IF ch=sdelim THEN slen:=0;state:=instring;END;
            | instring:
                IF ch=sdelim THEN
                    IF slen # 0 THEN INC(sfound);END; (* ignore empty string *)
                    state:=waiting;
                ELSE
                    INC(slen);
                END;
            END;
            INC(p);
            IF p >= len THEN EXIT; END;
        END;
    END;
    RETURN ( sfound # 0 );
END hasValidString;

(* ------------------------------------------------------------ *)

(* nonsense such as enquoting before extracting string is handled earlier *)

PROCEDURE doTrim (oldname,newname:pathtype;
                  doRight,doLeft,doFull,relaxmode,doNames,doPackNames,
                  doEnclose,doUnclose,doKillEmpty,doGrabStrings,showline,
                  useLFN:BOOLEAN;
                  tabwidth:CARDINAL);
CONST
    msgProcessing = "Processing ";
    msgOK         = " OK !";
VAR
    hin,hout : FIO.File;
    leading,smallest,i,len,currline : CARDINAL;
    S:str16;
    dmpme:BOOLEAN;
BEGIN
    WrStr(msgProcessing); WrStr(newname);
    Work(cmdInit);

    hin := fileOpenRead(useLFN,oldname);
    FIO.AssignBuffer(hin,bufferIn);
    hout:= fileCreate(useLFN,newname);
    FIO.AssignBuffer(hout,bufferOut);

    IF ((doLeft=TRUE) AND (doFull=FALSE)) THEN
        smallest := MAX(CARDINAL);
        FIO.EOF := FALSE;
        LOOP
            IF FIO.EOF THEN EXIT; END;
            Work(cmdShow);
            FIO.RdStr(hin,hugestr);
            IF FIO.EOF THEN EXIT; END;
            IF doRight THEN RtrimBlanks(hugestr); END;
            leading:=countDetabbedLeft(tabwidth,FALSE );
            IF leading < smallest THEN smallest := leading; END;
        END;
        IF smallest = 0 THEN doLeft := FALSE; END;
        FIO.Seek(hin,0);
    END;

    currline := 0;
    FIO.EOF := FALSE;
    LOOP
        IF FIO.EOF THEN EXIT; END;
        Work(cmdShow);
        FIO.RdStr(hin,hugestr);
        IF FIO.EOF THEN EXIT; END;
        IF doRight THEN RtrimBlanks(hugestr); END;
        IF doFull THEN (* precedence over doLeft which can be set to false *)
            LtrimBlanks(hugestr);

            (* caller checked doNames and doPackNames cannot be both true *)

            IF doNames THEN (* doFull, doRight, doLeft have been forced when parsing *)
                IF Str.CharPos(hugestr,dot) = MAX(CARDINAL) THEN (* safety *)
                    ReplaceChar(hugestr,tabul,space);
                    Str.Subst(hugestr,space,dot); (* first one *)
                    ReplaceChar(hugestr,space,"");
                END;
            END;
            IF doPackNames THEN (* doFull, doRight, doLeft have been forced when parsing *)
                FOR i:=1 TO 2 DO
                    CASE i OF
                    | 1: S:=space+dot;
                    | 2: S:=tabul+dot;
                    END;
                    LOOP
                        IF Str.Pos(hugestr,S)=MAX(CARDINAL) THEN EXIT; END;
                        Str.Subst(hugestr,S,dot);
                    END;
                END;
            END;
        ELSE
            IF doLeft THEN
                leading:=countDetabbedLeft(tabwidth,TRUE);
                Str.Delete(hugetmp,0,smallest);
                Str.Copy(hugestr,hugetmp);
            END;
        END;

        len:=Str.Length(hugestr);
        IF len = 0 THEN
            dmpme := NOT(doKillEmpty);
        ELSE

            (* caller check for logic *)

            IF doEnclose THEN
                unquote(FALSE,relaxmode,hugestr);
                Str.Prepend (hugestr,doublequote);
                Str.Append  (hugestr,doublequote);
            END;
            IF doUnclose THEN unquote(FALSE,relaxmode, hugestr); END;

            IF doGrabStrings THEN
                dmpme:=hasValidString();
            ELSE
                dmpme:=TRUE;
            END;
        END;
        INC(currline);
        IF dmpme THEN
            IF doGrabStrings THEN
                IF showline THEN
                    Str.Concat(S,fmtnum ( currline,10,5,space)," : ");
                    FIO.WrStr(hout,S);
                END;
            END;
            FIO.WrStr(hout,hugestr);FIO.WrLn(hout);
        END;
    END;

    fileFlush(useLFN,hout);
    fileClose(useLFN,hout);
    fileClose(useLFN,hin);

    Work(cmdStop);
    WrStr(msgOK); WrLn;
END doTrim;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE doJokeFix (oldname,newname:pathtype; useLFN:BOOLEAN);
CONST
    msgProcessing = "Fixing ";
    msgOK         = " OK !";
    patleading    = doublequote+"*";
    pattrailing   = "*"+doublequote;
VAR
    hin,hout : FIO.File;
    p:CARDINAL;
    delme:BOOLEAN;
BEGIN
    WrStr(msgProcessing); WrStr(newname);
    Work(cmdInit);

    hin := fileOpenRead(useLFN,oldname);
    FIO.AssignBuffer(hin,bufferIn);
    hout:= fileCreate(useLFN,newname);
    FIO.AssignBuffer(hout,bufferOut);

    FIO.EOF := FALSE;
    LOOP
        IF FIO.EOF THEN EXIT; END;
        Work(cmdShow);
        FIO.RdStr(hin,hugestr);
        IF FIO.EOF THEN EXIT; END;
        IF ODD ( CharCount(hugestr,doublequote) ) THEN (* 1, 3, ... *)
            Str.Copy(hugetmp,hugestr);
            LtrimBlanks(hugetmp);
            RtrimBlanks(hugetmp);
            delme := TRUE;
            IF    Str.Match(hugetmp,patleading) THEN
                p:=Str.CharPos(hugestr,doublequote);
            ELSIF Str.Match(hugetmp,pattrailing) THEN
                p:=Str.RCharPos(hugestr,doublequote);
            ELSE
                delme:=FALSE;
            END;
            IF delme THEN Str.Delete(hugestr,p,1);END;
        END;
        FIO.WrStr(hout,hugestr);FIO.WrLn(hout);
    END;

    fileFlush(useLFN,hout);
    fileClose(useLFN,hout);
    fileClose(useLFN,hin);

    Work(cmdStop);
    WrStr(msgOK); WrLn;
END doJokeFix;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    mintabwidth = 1;
    maxtabwidth = 64;
    noskip      = FALSE; (* no reason FOR this TO be TRUE here ! *)
VAR
    parmcount,i,opt : CARDINAL;
    S,R             : pathtype;
    state           : (waiting,gotparm1);
    v               : LONGCARD;
    filespec        : pathtype;
    basedir         : pathtype;
    oldname,newname : pathtype;

    doRight,doLeft,doFull,doNames,doPackNames,doGrabStrings:BOOLEAN;
    doEnclose,doUnclose,doKillEmpty,relaxmode,showline : BOOLEAN;
    DEBUG,useLFN,doRemoveOddQuote : BOOLEAN; (* for jokes processing *)
    tabwidth        : CARDINAL;
    p,count         : CARDINAL;
    anchor,ptr      : pFname;
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE;

    WrLn;

    DEBUG       := FALSE;
    useLFN      := TRUE;
    doRight     := FALSE;
    doLeft      := FALSE;
    doFull      := FALSE;
    doNames     := FALSE;
    doPackNames := FALSE;
    doEnclose   := FALSE;
    doUnclose   := FALSE;
    doKillEmpty := FALSE;
    relaxmode   := FALSE;
    doGrabStrings:=FALSE;
    showline    := FALSE;
    doRemoveOddQuote:=FALSE;
    tabwidth    := 8;

    state   := waiting;
    parmcount := Lib.ParamCount();
    IF parmcount=0 THEN abort(errHelp,"");END;
    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i); cleantabs(S);
        Str.Copy(R,S);
        UpperCase(R);
        IF isOption(R) THEN
            opt := GetOptIndex (R, "?"+delim+"H"+delim+"HELP"+delim+
                                   "R"+delim+"RIGHT"+delim+"TRAILING"+delim+
                                   "L"+delim+"LEFT"+delim+"LEADING"+delim+
                                   "LL"+delim+"FULL"+delim+
                                   "B"+delim+"BOTH"+delim+
                                   "T:"+delim+"TAB:"+delim+
                                   "N"+delim+"NAMES"+delim+
                                   "NN"+delim+
                                   "E"+delim+"ENCLOSE"+delim+
                                   "Q"+delim+"QUOTES"+delim+
                                   "P"+delim+"PACK"+delim+
                                   "QQ"+delim+
                                   "EE"+delim+
                                   "K"+delim+"RELAXED"+delim+
                                   "S"+delim+"STRINGS"+delim+
                                   "SS"+delim+"SA"+delim+
                                   "A"+delim+"SHOWLINE"+delim+
                                   "J"+delim+"JOKES"+delim+
                                   "X"+delim+"LFN"+delim+
                                   "DEBUG"
                               );
            CASE opt OF
            | 1,2,3 : abort(errHelp,"");
            | 4,5,6 : doRight := TRUE;
            | 7,8,9 : doLeft  := TRUE;
            | 10,11 : doLeft  := TRUE; doFull := TRUE;
            | 12,13:  doRight := TRUE; doLeft := TRUE; doFull := TRUE;
            | 14,15:
                IF GetLongCard(S,v)=FALSE THEN abort(errBadCard,S);END;
                IF ((v < mintabwidth) OR (v > maxtabwidth)) THEN abort(errTabRange,S);END;
                tabwidth := CARDINAL(v);
            | 16,17:  doNames := TRUE;
                      doRight := TRUE; doLeft := TRUE; doFull := TRUE;
            | 18:     doPackNames:=TRUE;
                      doRight := TRUE; doLeft := TRUE; doFull := TRUE;
            | 19,20:  doEnclose:= TRUE;
            | 21,22:  doUnclose:= TRUE;
            | 23,24:  doKillEmpty:=TRUE;
            | 25:     doUnclose:= TRUE;
                      doRight := TRUE; doLeft := TRUE; doFull := TRUE;
            | 26:     doEnclose:= TRUE;
                      doRight := TRUE; doLeft := TRUE; doFull := TRUE;
            | 27,28:  relaxmode := TRUE;
            | 29,30:  doGrabStrings:=TRUE;
                      doKillEmpty := TRUE;
            | 31,32:  doGrabStrings:=TRUE; showline:=TRUE;
                      doKillEmpty := TRUE;
            | 33,34:  showline := TRUE;
            | 35,36:  doRemoveOddQuote := TRUE;
            | 37,38:  useLFN := FALSE;
            | 39:     DEBUG := TRUE;
            ELSE
                abort(errOption,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            CASE state OF
            | waiting : Str.Copy(filespec,S);
            | gotparm1: abort(errTooManyParms,S);
            END;
            INC(state);
        END;
    END;

    useLFN := ( useLFN AND fileSupportLFN() );

    (* check nonsense *)
    IF state=waiting THEN abort(errMissingSpec,"");END;

    IF (doNames AND doPackNames) THEN abort(errFileNames,"");END;
    IF (doEnclose AND doUnclose) THEN abort(errQuotes,"");END;
    IF (doEnclose OR doUnclose) THEN
        ; (* IF (doRight OR doLeft) THEN abort(errQuotesCmd,"");END; *)
    ELSE
        IF ((doRight=FALSE) AND (doLeft=FALSE)) THEN
            doRight := TRUE;
            doLeft  := TRUE;
            (* abort(errCmdNeeded,""); *)
        END;
    END;
    IF NOT(doUnclose) THEN
        IF relaxmode THEN abort(errRelaxMode,"");END;
    END;

    IF doGrabStrings THEN
        IF doEnclose THEN abort(errGrabNonsense,"-e");END;
    ELSE
        IF showline THEN abort(errGrabOnly,"");END;
    END;

    (* a few q&d sanity checks *)

    IF DEBUG THEN WrStr("Filespec 1 : ");WrFname(useLFN,filespec);WrLn;END;
    IF chkUD(filespec)=FALSE THEN abort(errBadSpec,filespec);END;

    IF same(filespec,dot) THEN Str.Copy(filespec,stardotstar); END;
    IF same(filespec,star) THEN Str.Copy(filespec,stardotstar); END;
    IF Str.Match(filespec,"*"+antislash) THEN Str.Append(filespec,stardotstar); END;
    IF chkJoker(filespec)=FALSE THEN
        IF fileIsDirectorySpec ( useLFN,filespec) THEN
            fixDirectory(filespec);
            Str.Append(filespec,stardotstar);
        END;
    END;
    IF DEBUG THEN WrStr("Filespec 2 : ");WrFname(useLFN,filespec);WrLn;END;
    makebase(useLFN,filespec,basedir);
    IF DEBUG THEN WrStr("Basedir    : ");WrFname(useLFN,basedir);WrLn;END;

    initList(anchor);
    count := buildFileList( anchor,noskip,useLFN,DEBUG,filespec);
    CASE count OF
    | MAX(CARDINAL): abort(errTooManyFiles,filespec);
    | 0 :            abort(errNotFound,filespec);
    END;

    WrStr(Banner);WrLn;
    WrLn;

    ptr:=anchor;
    WHILE ptr # NIL DO
        getStr(filespec,ptr);
        Str.Concat(oldname,basedir,filespec);

        p := Str.RCharPos(oldname,dot);
        IF p # MAX(CARDINAL) THEN
            Str.Slice(S,oldname,0,p);
            Str.Copy(oldname,S);
        END;
        Str.Append(oldname,extBAK);

        getStr(filespec,ptr);
        Str.Concat(newname,basedir,filespec);
        IF fileExists(useLFN,oldname) THEN
            IF fileIsRO(useLFN,oldname) THEN fileSetRW(useLFN,oldname);END;
            fileErase(useLFN,oldname);
        END; (* don't keep previous backup, whatever it was *)
        fileRename(useLFN, newname,oldname);

        IF doRemoveOddQuote THEN
            doJokeFix(oldname,newname,useLFN);
        ELSE
            doTrim (oldname,newname,
                   doRight,doLeft,doFull,relaxmode,doNames,doPackNames,
                   doEnclose,doUnclose,doKillEmpty,doGrabStrings,showline,useLFN,
                   tabwidth);
        END;
        ptr:=ptr^.next;
    END;
    freeList(anchor);

    abort(errNone,"");
END Trim.


