(* ---------------------------------------------------------------
Title         Q&D Text Reformatter
Notes
Bugs          ignore tags must have both before and after CRLFCRLF !
Wish List

--------------------------------------------------------------- *)

MODULE TxtFmt;

IMPORT IO;
IMPORT Lib;
IMPORT Str;
IMPORT FIO;

FROM IO IMPORT WrStr,WrLn;

FROM Storage IMPORT Available,ALLOCATE,DEALLOCATE;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits;

FROM QD_LFN IMPORT path9X, huge9X, findDataRecordType,
unicodeConversionFlagType, w9XchangeDir,
w9XgetDOSversion, w9XgetTrueDOSversion, w9XisWindowsEnh, w9XisMSDOS7,
w9XfindFirst, w9XfindNext, w9XfindClose, w9XgetCurrentDirectory,
w9XlongToShort, w9XshortToLong, w9XtrueName, w9XchangeDir,
w9XmakeDir, w9XrmDir, w9Xrename, w9XopenFile, w9XcloseFile,
w9XsupportLFN;

FROM QD_File IMPORT pathtype, w9XnothingRequired,
fileOpenRead, fileOpen, fileExists, fileIsRO, fileSetRW, fileSetRO,
fileErase, fileCreate, fileRename, fileGetFileSize, fileIsDirectorySpec,
fileClose;

(* ------------------------------------------------------------ *)

CONST
    ProgEXEname   = "TXTFMT";
    ProgTitle     = "Q&D Text Reformatter";
    ProgVersion   = "v1.2";
    ProgCopyright = "by PhG";
    Banner        = ProgTitle+" "+ProgVersion+" "+ProgCopyright;

CONST
    dot        = ".";
    CR         = CHR(13); (* 0D *)
    LF         = CHR(10); (* 0A *)
    Tab        = CHR(09); (* 09 *)
    Espace     = CHR(32); (* 20 *)
    FormFeed   = CHR(12); (* 0C *)
    CtrlZ      = CHR(26); (* 1A *)
    NL         = CR + LF;
    nl         = NL;
    nullchar   = 0C; (* yep, CHR(0) *)
    star       = "*";
    dash       = "-";
    colon      = ":";
    antislash  = "\";
    dashdash   = dash+dash;
    dotdot     = dot+dot;
    stardotstar= star+dot+star;
    DEFTAG1       = "$";
    DEFTAG2       = "*";
    DEFTAG3       = "!";
    DEFIGNORETAG  = DEFTAG1+DEFTAG2+DEFTAG3; (* chars must be different ! *)
    TAGLEN        = 3;
CONST
    msgWait         = " ..."; (* yep, leading space *)
    defaulttmpfile  = "~DEL_ME~.TMP";
CONST
    extBAK          = ".BK!";
    extCOM          = ".COM";
    extEXE          = ".EXE";
    extDLL          = ".DLL";
    extOVR          = ".OVR";
    extOVL          = ".OVL";
    extDRV          = ".DRV";
    extZIP          = ".ZIP";
    extARJ          = ".ARJ";
    extLZH          = ".LZH";
    skippedextensions = extBAK+delim+extCOM+delim+extEXE+delim+
                        extDLL+delim+extOVR+delim+extOVL+delim+extDRV+delim+
                        extZIP+delim+extARJ+delim+extLZH;
CONST
    errNone          = 0;
    errHelp          = 1;
    errTooMany       = 2;
    errNoMatch       = 3;
    errAbort         = 4;
    errBadNumber     = 5;
    errDuplicate     = 6;
    errOption        = 7;
    errTooManyParms  = 8;
    errMissingSpec   = 9;
    errNonsense      = 10;
    errPageLength    = 11;
    errBadTag        = 12;
    errBadSpec       = 13;

PROCEDURE abort (e : CARDINAL;einfo:ARRAY OF CHAR);
CONST
(*
 0        1         2         3         4         5         6         7         8
 12345678901234567890123456789012345678901234567890123456789012345678901234567890
*)
    msgHelp=
Banner+nl+
nl+
"Syntax 1 : "+ProgEXEname+" <file(s)> [-t:#] [-t] [-g:$] [-s] [-q]"+nl+
"Syntax 2 : "+ProgEXEname+" <file(s)> <-w:#> [-t] [-g:$] [-s] [-q]"+nl+
"Syntax 3 : "+ProgEXEname+" <file(s)> <-p:#> [-c:#] [-i] [-c] [-s] [-q]"+nl+
"Syntax 4 : "+ProgEXEname+" <file(s)> <-u[:#]> [-s] [-q]"+nl+
"Syntax 5 : "+ProgEXEname+" <file(s)> <-f[f]> [-a] [-s] [-q]"+nl+
nl+
"This program reads an ASCII file then builds paragraphs from lines (syntax 1),"+nl+
"breaks paragraphs into lines (syntax 2), skips lines/columns (syntax 3),"+nl+
"expands tabs (syntax 4), or processes a fortune file (syntax 5)."+nl+
nl+
"  -w:#   break paragraphs into lines counting at most # characters"+nl+
"  -p:#   page length (lines to keep together)"+nl+
"  -u[:#] expand tabs (default tab width is 8 unless specified with -u:#)"+nl+
"  -f[f]  process fortune file (-ff = start each paragraph with a tabulation)"+nl+
"  -t:#   tab width (default is 1)"+nl+
"  -t     syntax 1 and 2 : reformat text even between marker strings"+nl+
"         (marker string is a toggle and it must be on its own single line)"+nl+
'  -g:$   redefine marker string (3 symbols required, default is "'+DEFIGNORETAG+'")'+nl+
"  -c:#   count of lines to skip after page length (default is 1)"+nl+
"  -c     process columns instead of lines"+nl+
'  -a     start a paragraph for author (line beginning with "'+dash+'" or "'+dashdash+'")'+nl+
"  -i     inverse selection written to file"+nl+
"  -s     send output to screen instead of file (test mode)"+nl+
"  -q     quiet processing"+nl+
"  -lfn   disable LFN support even if available"+nl+
nl+
skippedextensions+" files will not be processed."+nl+
"Each original file will be backuped with "+extBAK+" extension."+nl;

VAR
    S : str1024;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(msgHelp);
    | errTooMany :
        Str.Concat(S,"Too many files match ",einfo);Str.Append(S," specification !");
    | errNoMatch :
        Str.Concat(S,"No valid file matches ",einfo);Str.Append(S," specification !");
    | errBadNumber:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," number !");
    | errDuplicate:
        Str.Concat(S,einfo," has already been specified !");
    | errOption:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," option !");
    | errTooManyParms:
        Str.Concat(S,einfo," is one file specification too many !");
    | errMissingSpec:
        S := "No file was specified !";
    | errNonsense:
        S:="At least one option irrelevant to specified command !";
    | errPageLength:
        S := "-p:# was not specified !";
    | errBadTag:
        Str.Concat(S,'Illegal "',einfo);Str.Append(S,'" tag !');
    | errBadSpec:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," specification !");
    ELSE
        S := "How did you get HERE ?";
    END;
    CASE  e OF
    | errNone, errHelp, errAbort: ;
    ELSE
        WrStr(ProgEXEname+" : "); WrStr(S); WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

TYPE
    pFname = POINTER TO fnameType;
    fnameType = RECORD
        next      : pFname;
        slen      : CARDINAL; (* a SHORTCARD could do but who knows ? *)
        str       : CHAR;
    END;

PROCEDURE initList (VAR anchor : pFname );
BEGIN
    anchor := NIL;
END initList;

PROCEDURE freeList (anchor : pFname);
VAR
    needed : CARDINAL;
    p      : pFname;
BEGIN
    (* p:=anchor; *)
    WHILE anchor # NIL DO
        needed := SIZE(fnameType) - SIZE(anchor^.str) + anchor^.slen;
        p := anchor^.next;
        DEALLOCATE(anchor,needed);
        anchor:=p;
    END
END freeList;

PROCEDURE buildNewPtr (VAR anchor,p:pFname; len:CARDINAL):BOOLEAN;
VAR
    needed : CARDINAL;
BEGIN
    needed := SIZE(fnameType) - SIZE(p^.str) + len;
    IF Available(needed)=FALSE THEN RETURN FALSE; END;
    IF anchor = NIL THEN
        ALLOCATE(anchor,needed);
        p:=anchor;
    ELSE
        p:=anchor;
        WHILE p^.next # NIL DO
            p:=p^.next;
        END;
        ALLOCATE(p^.next,needed);
        p:=p^.next;
    END;
    p^.next := NIL;
    RETURN TRUE;
END buildNewPtr;

(* assume p is valid *)

PROCEDURE getStr (VAR S : pathtype; p:pFname);
VAR
    len:CARDINAL;
BEGIN
    len := p^.slen;
    Lib.FastMove( ADR(p^.str),ADR(S),len);
    S[len] := nullchar; (* REQUIRED safety ! *)
END getStr;

(* ------------------------------------------------------------ *)

(* Str.Match is not case sensitive *)

PROCEDURE isReservedPattern (S,skipthem:ARRAY OF CHAR):BOOLEAN;
VAR
    e3 : str16;
    n:CARDINAL;
    rc:BOOLEAN;
BEGIN
    rc:=FALSE;
    n:=0;
    LOOP
        isoleItemS(e3, skipthem,delim,n);
        IF same(e3,"") THEN EXIT; END;
        Str.Prepend(e3,"*");
        IF Str.Match(S,e3) THEN rc:=TRUE;EXIT; END;
        INC(n);
    END;
    RETURN rc;
END isReservedPattern;

PROCEDURE isReservedEntry (S:ARRAY OF CHAR) : BOOLEAN;
BEGIN
    IF same(S,dot) THEN RETURN TRUE; END;
    RETURN same(S,dotdot);
END isReservedEntry;

PROCEDURE buildFileList (VAR anchor:pFname;
                        useLFN,DEBUG :BOOLEAN;spec:pathtype):CARDINAL;
VAR
    count:CARDINAL; (* should do ! *)
    ok,found:BOOLEAN;
    unicodeconversion:unicodeConversionFlagType;
    w9Xentry : findDataRecordType;
    w9Xhandle,errcode:CARDINAL;
    entry : FIO.DirEntry;
    dosattr:FIO.FileAttr;
    entryname:pathtype;
    len : CARDINAL;
    pp:pFname;
    excludeme1,excludeme2:BOOLEAN;
BEGIN
    count:=0;
    IF useLFN THEN
        found := w9XfindFirst (spec,SHORTCARD(everything),SHORTCARD(w9XnothingRequired),
                              unicodeconversion,w9Xentry,w9Xhandle,errcode);
    ELSE
        found := FIO.ReadFirstEntry(spec,everything,entry);
    END;
    WHILE found DO
        IF useLFN THEN
            Str.Copy(entryname,w9Xentry.fullfilename);
        ELSE
            Str.Copy(entryname,entry.Name);
        END;
        excludeme1 := isReservedEntry   (entryname);  (* skip "." and ".." *)
        excludeme2 := isReservedPattern (entryname,skippedextensions );
        IF NOT(excludeme1 OR excludeme2) THEN
            IF useLFN THEN
                dosattr:=FIO.FileAttr(w9Xentry.attr AND 0FFH);
            ELSE
                dosattr:=entry.attr;
            END;
            IF NOT (aD IN dosattr) THEN
                (* if file has no extension, add it as a marker *)
                IF Str.RCharPos(entryname,".")=MAX(CARDINAL) THEN
                    Str.Append(entryname,".");
                END;
                IF DEBUG THEN WrStr("Included : ");WrStr(entryname);WrLn; END;
                len:=Str.Length(entryname);
                IF buildNewPtr(anchor,pp,len)=FALSE THEN
                    IF useLFN THEN ok:=w9XfindClose(w9Xhandle,errcode); END;
                    RETURN MAX(CARDINAL); (* errStorage *)
                END;
                INC(count);
                pp^.slen      := len;
                Lib.FastMove ( ADR(entryname),ADR(pp^.str),len );
            ELSE
                IF DEBUG THEN WrStr("Ignored  : ");WrStr(entryname);WrLn;END;
            END;
        ELSE
            IF DEBUG THEN WrStr("Excluded : ");WrStr(entryname);WrLn;END;
        END;
        IF useLFN THEN
            found :=w9XfindNext(w9Xhandle, unicodeconversion,w9Xentry,errcode);
        ELSE
            found :=FIO.ReadNextEntry(entry);
        END;
    END;
    IF useLFN THEN ok:=w9XfindClose(w9Xhandle,errcode); END;
    RETURN count;
END buildFileList;

(* ------------------------------------------------------------ *)

PROCEDURE doGetCurrent (useLFN:BOOLEAN;drive:SHORTCARD;
                       VAR unit:str2; VAR current:pathtype);
VAR
    rc:CARDINAL;
    longform:pathtype;
BEGIN
    Str.Concat(unit, CHR( ORD("A")-1+ORD(drive) ),colon);

    FIO.GetDir(drive,current); (* \path without u: nor trailing \ except at root *)
    IF current[1] # colon THEN Str.Prepend(current,unit); END; (* safety *)
    IF useLFN THEN
        IF w9XshortToLong(current,rc,longform) THEN (* if error, keep DOS current *)
            Str.Copy(current,longform);
        END;
    END;
    (* LFN function seems to always return "u:\*" form except at root *)
    IF current[1] = colon THEN Str.Delete(current,0,2);END; (* safety *)
    fixDirectory(current);
END doGetCurrent;

PROCEDURE makebase (useLFN:BOOLEAN;spec:pathtype;VAR basepath:pathtype);
VAR
    u,d,n,e,current:pathtype;
    unit:str2;
    drive:SHORTCARD;
BEGIN
    Lib.SplitAllPath(spec,u,d,n,e);
    Str.Concat(basepath,u,d);
    IF same(basepath,"") THEN
        drive:=FIO.GetDrive(); (* yes we could use 0 as default drive *)
        doGetCurrent(useLFN,drive,  unit,current); (* "u:" and "\" or "\*\" *)
        Str.Concat(basepath, unit,current);
    END;
END makebase;

PROCEDURE WrQuoted (S:ARRAY OF CHAR);
BEGIN
    WrStr('"');
    WrStr(S);
    WrStr('"');
END WrQuoted;

PROCEDURE WrFname (useLFN:BOOLEAN;S:pathtype );
BEGIN
    IF useLFN THEN
        WrQuoted(S);
    ELSE
        WrStr(S);
    END;
END WrFname;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    IObufferSize = (8 * 512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = IObufferSize;
VAR
    IObufferIn : ARRAY [1..IObufferSize] OF BYTE;
    IObufferOut: ARRAY [1..IObufferSize] OF BYTE;

PROCEDURE donewname (useLFN:BOOLEAN;source,cible:pathtype);
VAR
    tmpfile : pathtype;
BEGIN
    Str.Copy(tmpfile,defaulttmpfile);
    fileRename (useLFN,cible,tmpfile);
    fileRename (useLFN,source,cible);
    fileRename (useLFN,tmpfile,source);
END donewname;

(* ------------------------------------------------------------ *)

PROCEDURE BuildParagraphs (source,cible:pathtype;IGNORETAG : ARRAY OF CHAR;
                          tabwidth:CARDINAL;
                          usetag,verbose,sendtoscreen,useLFN:BOOLEAN );
CONST
    MaxNL    = 3;
VAR
    wasCR       : BOOLEAN;
    wasLF       : BOOLEAN;
    wasNL       : BOOLEAN;
    nlnl        : CARDINAL;
    c,precedent : CHAR;
    got         : CARDINAL;
    htab,n      : LONGCARD; (* we may have very, very long paragraphs ! ;-) *)
    i           : CARDINAL;
    reformat    : BOOLEAN;
    tagcount    : CARDINAL;
    TAG1,TAG2,TAG3:CHAR;
    hndin    : FIO.File;
    hndout   : FIO.File;
    fsize,portion,lastportion,currportion,addr : LONGCARD;
    steps    : CARDINAL;
    wrtofile : BOOLEAN;
BEGIN
    IF sendtoscreen THEN
        verbose:=FALSE;
        wrtofile:=FALSE;
    ELSE
        wrtofile:=TRUE;
    END;

    IF wrtofile THEN WrStr("Reformatting ");WrFname(useLFN,source); END;

    hndin:=fileOpenRead(useLFN,source);
    FIO.AssignBuffer(hndin,IObufferIn);
    IF wrtofile THEN
        hndout := fileCreate(useLFN,cible);
        FIO.AssignBuffer(hndout,IObufferOut);
    ELSE
        hndout:=FIO.StandardOutput;
    END;

    TAG1:=IGNORETAG[0];
    TAG2:=IGNORETAG[1];
    TAG3:=IGNORETAG[2];

    wasCR :=TRUE;
    wasLF :=TRUE;
    wasNL :=TRUE;
    nlnl := 1;
    precedent := "";
    htab      := 0;

    reformat  := TRUE;
    tagcount  := 0;

    (* Work(cmdInit); *)
    addr:=0;
    steps := 10;
    fsize := fileGetFileSize(useLFN,source);
    portion:=fsize DIV LONGCARD(steps); INC(portion); (* avoid DIV 0 ! *)
    lastportion := LONGCARD(steps+1);
    IF verbose THEN
        (* animInit(steps, " ", "", CHR(249), "", "\/" ); *)
        animInit(steps, " ", "", "-", "+", "-\|/" );
    ELSE
        IF wrtofile THEN video(msgWait,TRUE);END;
    END;

    FIO.EOF := FALSE;

    LOOP
        IF FIO.EOF=TRUE THEN EXIT; END; (* pb at pass after first one! *)
        (* Work(cmdShow); *)
        IF verbose THEN
            anim(animShow);
            currportion:=addr DIV portion;
            IF currportion # lastportion THEN
                anim(animAdvance);
                lastportion:=currportion;
            END;
        END;

        got := FIO.RdBin(hndin,c,1);
        IF got = 0 THEN EXIT; END;
        INC(addr);
        CASE c OF
        | CR :
          IF reformat=FALSE THEN
            FIO.WrBin(hndout,c,1);
          ELSE
            IF wasCR THEN
                wasNL := TRUE;
                nlnl := nlnl + 1;
                IF nlnl < MaxNL THEN FIO.WrBin (hndout, NL,2); htab:=0; END;
            END;
            IF wasLF THEN wasNL := TRUE; END;
            IF wasNL THEN
                nlnl := nlnl + 1;
                IF nlnl < MaxNL THEN FIO.WrBin(hndout,NL,2); htab:=0; END;
            END;
          END;
            wasCR := TRUE;
            wasLF := FALSE;
        | LF :
          IF reformat=FALSE THEN
            FIO.WrBin(hndout,c,1);
          ELSE
            IF wasLF THEN
                wasNL := TRUE;
                nlnl := nlnl + 1;
                IF nlnl < MaxNL THEN FIO.WrBin(hndout,NL,2); htab:=0; END;
            END;
            IF wasNL THEN
                nlnl := nlnl + 1;
                IF nlnl < MaxNL THEN FIO.WrBin(hndout,NL,2); htab:=0; END;
            END;
          END;
            wasLF := TRUE;
            wasCR := FALSE;
        | Espace :
          IF reformat=FALSE THEN
            FIO.WrBin(hndout,c,1);
          ELSE
            IF ((wasCR = FALSE) AND (wasLF = FALSE)) THEN
                IF precedent <> Espace THEN
                    FIO.WrBin(hndout,Espace,1); INC(htab);
                END;
            END;
          END;
        | Tab :
          IF reformat=FALSE THEN
            FIO.WrBin(hndout,c,1);
          ELSE
            IF tabwidth = 0 THEN
                c := Espace;
                IF ((wasCR = FALSE) AND (wasLF = FALSE)) THEN
                   IF precedent <> Espace THEN
                       FIO.WrBin(hndout,Espace,1); INC(htab);
                   END;
                END;
            ELSE
                n := ( ( htab DIV LONGCARD(tabwidth) ) + 1 ) * LONGCARD(tabwidth);
                DEC(n,htab);
                FOR i := 1 TO CARDINAL(n) DO
                    FIO.WrBin(hndout,Espace,1); INC(htab);
                END;
            END;
          END;
        | FormFeed :
          IF reformat=FALSE THEN
            FIO.WrBin(hndout,c,1);
          END ;
        | CtrlZ :
          IF reformat=FALSE THEN
            FIO.WrBin(hndout,c,1);
          END;
        ELSE
            IF usetag THEN
                IF c = TAG1 THEN
                    tagcount:=1;
                ELSIF c = TAG2 THEN
                    IF precedent = TAG1 THEN INC(tagcount);END;
                ELSIF c = TAG3 THEN
                    IF precedent = TAG2 THEN INC(tagcount);END;
                    IF tagcount=TAGLEN THEN
                        reformat := NOT(reformat);
                        tagcount:=0;
                    END;
                END;
            END;
            IF reformat THEN
                IF wasNL = FALSE THEN
                    IF (wasCR OR wasLF ) THEN
                        FIO.WrBin(hndout,Espace,1); INC(htab);
                    END;
                END;
            END;
            (* remember here char is anything but already filtered ones ! *)
            FIO.WrBin(hndout,c,1);INC(htab);
            wasCR := FALSE;
            wasLF := FALSE;
            wasNL := FALSE;
            nlnl := 0;
        END;
        precedent := c;
    END;
    (* Work(cmdStop); *)
    IF verbose THEN
        anim(animEnd);anim(animClear);
    ELSE
        IF wrtofile THEN video(msgWait,FALSE);END;
    END;

    fileClose(useLFN,hndin);
    FIO.WrBin (hndout,NL,2);
    IF wrtofile THEN
        FIO.Flush(hndout);
        fileClose(useLFN,hndout);
        donewname(useLFN,source,cible);
        WrLn;
    END;
END BuildParagraphs;

(* ------------------------------------------------------------ *)

CONST
    hugestrsize = 0C000H; (* is 49152, was 24576, should be more than enough *)
VAR
    hugestr : ARRAY [0..hugestrsize-1] OF CHAR;
    basePos : CARDINAL;
    baseLen : CARDINAL;

PROCEDURE dmpTTXfile (VAR paragraph:ARRAY OF CHAR;width:CARDINAL;
                      VAR S:ARRAY OF CHAR;firstcall:BOOLEAN) : BOOLEAN;
CONST
    blank=" ";
VAR
    len,count:CARDINAL;
BEGIN
    IF firstcall THEN
        basePos := 0;
        baseLen := Str.Length(paragraph); (* do not recalc it each time *)
    END;

    IF basePos = baseLen THEN RETURN FALSE; END; (* we're done now *)

    count:=width;
    IF (basePos+count) < baseLen THEN (* more than 'width' remaining chars *)
        LOOP
            IF paragraph[basePos+count]=blank THEN
                EXIT;
            END;
            DEC(count);
            IF count=0 THEN (* no space at all in string to get *)
                count:=width;
                EXIT;
            END;
        END;
        Str.Slice(S,paragraph,basePos,count); (* slice required portion *)
        INC(basePos,count); (* advance *)
        IF paragraph[basePos] = blank THEN INC(basePos); END; (* skip possible first space *)
    ELSE (* remaining chars are just 'width' or less *)
        Str.Slice(S,paragraph,basePos,baseLen-basePos);
        basePos:=baseLen; (* right on nullchar *)
    END;
    RETURN TRUE;
END dmpTTXfile;

PROCEDURE BreakLines (source,cible:pathtype;IGNORETAG : ARRAY OF CHAR;
                     count:CARDINAL;
                     usetag,verbose,sendtoscreen,useLFN:BOOLEAN );
VAR
    S : str256; (* oversized ! *)
    continue,reformat : BOOLEAN;
    wrtofile : BOOLEAN;
    hndin    : FIO.File;
    hndout   : FIO.File;
    fsize,portion,lastportion,currportion,addr : LONGCARD;
    steps    : CARDINAL;
BEGIN
    IF sendtoscreen THEN
        verbose:=FALSE;
        wrtofile:=FALSE;
    ELSE
        wrtofile:=TRUE;
    END;

    IF wrtofile THEN WrStr("Breaking ");WrFname(useLFN,source); END;

    hndin := fileOpenRead(useLFN,source);
    FIO.AssignBuffer(hndin,IObufferIn);
    IF wrtofile THEN
        hndout := fileCreate(useLFN,cible);
        FIO.AssignBuffer(hndout,IObufferOut);
    ELSE
        hndout:=FIO.StandardOutput;
    END;

    reformat := TRUE;

    (* Work(cmdInit); *)
    addr:=0;
    steps := 10;
    fsize := fileGetFileSize(useLFN,source);
    portion:=fsize DIV LONGCARD(steps); INC(portion); (* avoid DIV 0 ! *)
    lastportion := LONGCARD(steps+1);
    IF verbose THEN
        (* animInit(steps, " ", "", CHR(249), "", "\/" ); *)
        animInit(steps, " ", "", "-", "+", "-\|/" );
    ELSE
        IF wrtofile THEN video(msgWait,TRUE);END;
    END;

    FIO.EOF := FALSE;
    LOOP
        IF FIO.EOF THEN EXIT; END;
        (* Work(cmdShow); *)
        IF verbose THEN
            anim(animShow);
            currportion:=addr DIV portion;
            IF currportion # lastportion THEN
                anim(animAdvance);
                lastportion:=currportion;
            END;
        END;

        FIO.RdStr(hndin,hugestr);
        INC(addr, LONGCARD(Str.Length(hugestr)) );
        IF hugestr[0] = CHR(0) THEN
            FIO.WrLn(hndout);
        ELSE
            IF usetag THEN
                IF same(IGNORETAG,hugestr) THEN
                    reformat := NOT(reformat); (* if toggled to TRUE, will autohandle dump of IGNORETAG *)
                END;
            END;
            IF reformat THEN
                continue:=dmpTTXfile(hugestr,count,S,TRUE);
                WHILE continue DO
                    FIO.WrStr(hndout,S);
                    FIO.WrLn(hndout);
                    continue:=dmpTTXfile(hugestr,count,S,FALSE);
                END;
            ELSE
                FIO.WrStr(hndout,hugestr);
                FIO.WrLn(hndout);
            END;
        END;
    END;
    (* Work(cmdStop); *)
    IF verbose THEN
        anim(animEnd);anim(animClear);
    ELSE
        IF wrtofile THEN video(msgWait,FALSE);END;
    END;

    fileClose(useLFN,hndin);
    IF wrtofile THEN
        FIO.Flush(hndout);
        fileClose(useLFN,hndout);
        donewname(useLFN,source,cible);
        WrLn;
    END;
END BreakLines;

(* ------------------------------------------------------------ *)

PROCEDURE SkipLines (source,cible : pathtype;
                     pagelength,count:CARDINAL;
                     inverse,verbose,sendtoscreen,useLFN:BOOLEAN );
VAR
    i,j:CARDINAL;
    state : (keeping,skipping);
    wrtofile : BOOLEAN;
    hndin    : FIO.File;
    hndout   : FIO.File;
    fsize,portion,lastportion,currportion,addr : LONGCARD;
    steps    : CARDINAL;
BEGIN
    IF sendtoscreen THEN
        verbose:=FALSE;
        wrtofile:=FALSE;
    ELSE
        wrtofile:=TRUE;
    END;

    IF wrtofile THEN WrStr("Cleaning ");WrFname(useLFN,source); END;

    hndin := fileOpenRead(useLFN,source);
    FIO.AssignBuffer(hndin,IObufferIn);
    IF wrtofile THEN
        hndout := fileCreate(useLFN,cible);
        FIO.AssignBuffer(hndout,IObufferOut);
    ELSE
        hndout := FIO.StandardOutput; (* DOS standard output will do here *)
    END;

    state := keeping;
    i:=0;

    (* Work(cmdInit); *)
    addr:=0;
    steps := 10;
    fsize := fileGetFileSize(useLFN,source);
    portion:=fsize DIV LONGCARD(steps); INC(portion); (* avoid DIV 0 ! *)
    lastportion := LONGCARD(steps+1);
    IF verbose THEN
        (* animInit(steps, " ", "", CHR(249), "", "\/" ); *)
        animInit(steps, " ", "", "-", "+", "-\|/" );
    ELSE
        IF wrtofile THEN video(msgWait,TRUE);END;
    END;

    FIO.EOF := FALSE;
    LOOP
        IF FIO.EOF THEN EXIT; END;
        (* Work(cmdShow); *)
        IF verbose THEN
            anim(animShow);
            currportion:=addr DIV portion;
            IF currportion # lastportion THEN
                anim(animAdvance);
                lastportion:=currportion;
            END;
        END;

        FIO.RdStr(hndin,hugestr);
        INC(addr, LONGCARD(Str.Length(hugestr)) );
        IF (hugestr[0]=CHR(0)) AND FIO.EOF THEN EXIT; END;
        CASE state OF
        | keeping:
            INC(i);
            IF i > pagelength THEN
                IF inverse THEN
                    FIO.WrStr(hndout,hugestr);
                    FIO.WrLn(hndout);
                END;
                state:=skipping;
                j:=0;
                INC(j); (* eh, we just skipped our first line ! *)
            ELSE
                IF NOT (inverse) THEN
                    FIO.WrStr(hndout,hugestr);
                    FIO.WrLn(hndout);
                END;
            END;
        | skipping:
            INC(j);
            IF j > count THEN
                IF NOT (inverse) THEN
                    FIO.WrStr(hndout,hugestr);
                    FIO.WrLn(hndout);
                END;
                state:=keeping;
                i := 0;
                INC(i); (* eh, we just kept our first line ! *)
            ELSE
                IF inverse THEN
                    FIO.WrStr(hndout,hugestr);
                    FIO.WrLn(hndout);
                END;
            END;
        END;
    END;
    (* Work(cmdStop); *)
    IF verbose THEN
        anim(animEnd);anim(animClear);
    ELSE
        IF wrtofile THEN video(msgWait,FALSE);END;
    END;

    fileClose(useLFN,hndin);
    IF wrtofile THEN
        FIO.Flush(hndout);
        fileClose(useLFN,hndout);
        donewname(useLFN,source,cible);
        WrLn;
    END;
END SkipLines;

(* ------------------------------------------------------------ *)

PROCEDURE SkipColumns (source,cible :pathtype;
                       collength,count:CARDINAL;
                       inverse,verbose,sendtoscreen,useLFN:BOOLEAN );
VAR
    i,j,p,len:CARDINAL;
    state : (keeping,skipping);
    ch:CHAR;
    wrtofile : BOOLEAN;
    hndin    : FIO.File;
    hndout   : FIO.File;
    fsize,portion,lastportion,currportion,addr : LONGCARD;
    steps    : CARDINAL;
BEGIN
    IF sendtoscreen THEN
        verbose:=FALSE;
        wrtofile:=FALSE;
    ELSE
        wrtofile:=TRUE;
    END;

    IF wrtofile THEN WrStr("Cleaning ");WrFname(useLFN,source); END;

    hndin := fileOpenRead(useLFN,source);
    FIO.AssignBuffer(hndin,IObufferIn);
    IF wrtofile THEN
        hndout := fileCreate(useLFN,cible);
        FIO.AssignBuffer(hndout,IObufferOut);
    ELSE
        hndout := FIO.StandardOutput;
    END;

    (* Work(cmdInit); *)
    addr:=0;
    steps := 10;
    fsize := fileGetFileSize(useLFN,source);
    portion:=fsize DIV LONGCARD(steps); INC(portion); (* avoid DIV 0 ! *)
    lastportion := LONGCARD(steps+1);
    IF verbose THEN
        (* animInit(steps, " ", "", CHR(249), "", "\/" ); *)
        animInit(steps, " ", "", "-", "+", "-\|/" );
    ELSE
        IF wrtofile THEN video(msgWait,TRUE);END;
    END;

    FIO.EOF := FALSE;
    LOOP
        IF FIO.EOF THEN EXIT; END;
        (* Work(cmdShow); *)
        IF verbose THEN
            anim(animShow);
            currportion:=addr DIV portion;
            IF currportion # lastportion THEN
                anim(animAdvance);
                lastportion:=currportion;
            END;
        END;

        FIO.RdStr(hndin,hugestr);
        len := Str.Length(hugestr);
        INC(addr, LONGCARD( len ) );
        IF (hugestr[0]=CHR(0)) AND FIO.EOF THEN EXIT; END;
        p:=0;
        i:=0;
        state := keeping;
        LOOP
            IF p=len THEN EXIT; END;
            ch:=hugestr[p];
            CASE state OF
            | keeping:
                INC(i);
                IF i > collength THEN
                    IF inverse THEN FIO.WrChar(hndout,ch); END;
                    state:=skipping;
                    j:=0;
                    INC(j); (* eh, we just skipped our first char ! *)
                ELSE
                    IF NOT (inverse) THEN FIO.WrChar(hndout,ch); END;
                END;
            | skipping:
                INC(j);
                IF j > count THEN
                    IF NOT (inverse) THEN FIO.WrChar(hndout,ch); END;
                    state:=keeping;
                    i := 0;
                    INC(i); (* eh, we just kept our first char ! *)
                ELSE
                    IF inverse THEN FIO.WrChar(hndout,ch); END;
                END;
            END;
            INC(p);
        END;
        FIO.WrLn(hndout);
    END;
    (* Work(cmdStop); *)
    IF verbose THEN
        anim(animEnd);anim(animClear);
    ELSE
        IF wrtofile THEN video(msgWait,FALSE);END;
    END;

    fileClose(useLFN,hndin);
    IF wrtofile THEN
        FIO.Flush(hndout);
        fileClose(useLFN,hndout);
        donewname(useLFN,source,cible);
        WrLn;
    END;
END SkipColumns;

(* ------------------------------------------------------------ *)

PROCEDURE procFortune (source,cible : pathtype;
                      paratab,author,verbose,sendtoscreen,useLFN:BOOLEAN );
TYPE
    skindtype = (sep,para,str,empty);
VAR
    separator : CHAR;
    S,S2:str4096; (* well, we could use hugestr *)
    skindold:skindtype;
    hndin    : FIO.File;
    hndout   : FIO.File;
    fsize,portion,lastportion,currportion,addr : LONGCARD;
    steps    : CARDINAL;
    wrtofile : BOOLEAN;
BEGIN
    hndin := fileOpenRead(useLFN,source);
    FIO.AssignBuffer(hndin,IObufferIn);

    FIO.RdStr(hndin,S);
    Str.Copy(S2,S);
    LtrimBlanks(S2); RtrimBlanks(S2);
    IF Str.Length(S2) # 1 THEN
        fileClose(useLFN,hndin);
        WrStr("Skipping ");WrFname(useLFN,source);WrStr(" : probably not a fortune file !");WrLn;
        RETURN;
    END;
    separator:=S2[0]; (* assume it's ok *)

    IF sendtoscreen THEN
        verbose:=FALSE;
        wrtofile:=FALSE;
    ELSE
        wrtofile:=TRUE;
    END;

    IF wrtofile THEN
        hndout := fileCreate(useLFN,cible);
        FIO.AssignBuffer(hndout,IObufferOut);
    ELSE
        hndout:=FIO.StandardOutput;
    END;

    IF wrtofile THEN WrStr("Reformatting ");WrFname(useLFN,source);WrStr(" fortune file"); END;

    steps := 10;
    fsize := fileGetFileSize(useLFN,source);
    portion:=fsize DIV LONGCARD(steps); INC(portion); (* avoid DIV 0 ! *)
    lastportion := LONGCARD(steps+1);

    IF verbose THEN
        (* animInit(steps, " ", "", CHR(249), "", "\/" ); *)
        animInit(steps, " ", "", "-", "+", "-\|/" );
    ELSE
        IF wrtofile THEN video(msgWait,TRUE);END;
    END;

    FIO.WrStr(hndout,separator);FIO.WrLn(hndout);

    skindold:=sep;
    LOOP
        addr := FIO.GetPos(hndin);
        IF addr >= fsize THEN EXIT;END;

        (* Work(cmdShow); *)
        IF verbose THEN
            anim(animShow);
            currportion:=addr DIV portion;
            IF currportion # lastportion THEN
                anim(animAdvance);
                lastportion:=currportion;
            END;
        END;

        FIO.RdStr(hndin,S); RtrimBlanks(S); (* always remove trailing blanks *)
        Str.Copy(S2,S);
        LtrimBlanks(S2);

        IF author THEN
            (* assume authorship if line begins verbatim with a dash or a double dash *)
            IF ( (Str.Match(S2,dashdash+"*")) OR (Str.Match(S2,dash+"*")) ) THEN (* redundant ! *)
                (* force a fake paragraph so that S # S2 now *)
                IF same(S,S2) THEN Str.Prepend(S2,Tab);END;
            END;
        END;

        IF    same(S2,separator) THEN
            CASE skindold OF
            | sep,empty:
                FIO.WrStr(hndout,S);FIO.WrLn(hndout);
            | para,str:
                FIO.WrLn(hndout);
                FIO.WrStr(hndout,S);FIO.WrLn(hndout);
            END;
            skindold:=sep;
        ELSIF same(S2,"") THEN
            CASE skindold OF
            | sep:
                skindold:=sep;
            | para,str:
                skindold:=empty;
            | empty:
                ;
            END;
        ELSIF same(S2,S) THEN
            CASE skindold OF
            | sep:
                FIO.WrStr(hndout,S);
            | para,str:
                FIO.WrStr(hndout," ");FIO.WrStr(hndout,S);
            | empty:
                FIO.WrLn(hndout);
                FIO.WrLn(hndout);
                FIO.WrStr(hndout,S);
            END;
            skindold:=str;
        ELSE
            IF paratab THEN Str.Concat(S,Tab,S2);END;
            CASE skindold OF
            | sep:
                FIO.WrStr(hndout,S);
            | para,str:
                FIO.WrLn(hndout);
                FIO.WrStr(hndout,S);
            | empty:
                FIO.WrLn(hndout);
                FIO.WrLn(hndout);
                FIO.WrStr(hndout,S);
            END;
            skindold:=para;
        END;

    END;
    (* we could add a trailing separator *)
    CASE skindold OF
    | sep: ;
    | para,str: FIO.WrLn(hndout);
    | empty: ;
    END;

    (* Work(cmdStop); *)
    IF verbose THEN
        anim(animEnd);anim(animClear);
    ELSE
        IF wrtofile THEN video(msgWait,FALSE);END;
    END;

    fileClose(useLFN,hndin);
    IF wrtofile THEN
        FIO.Flush(hndout);
        fileClose(useLFN,hndout);
        donewname(useLFN,source,cible);
        WrLn;
    END;
END procFortune;

(* ------------------------------------------------------------ *)

PROCEDURE detabit (tabwidth:CARDINAL;S:ARRAY OF CHAR; VAR R : ARRAY OF CHAR  );
CONST
    tabchar = Tab;
    space   = Espace;
VAR
    i,j,add: CARDINAL;
    c : CHAR;
BEGIN
    Str.Copy(R,"");
    j:=0; (* yes, 0 and not 1 ! *)
    FOR i:=1 TO Str.Length(S) DO
        c := S[i-1];
        IF c = tabchar THEN
            add := tabwidth - (j MOD tabwidth);
            WHILE add > 0 DO
                Str.Append(R,space); INC(j);
                DEC(add);
            END;
        ELSE
            Str.Append(R,c); INC(j);
        END;
    END;
END detabit;

PROCEDURE expTabs (source,cible : pathtype;tabwi:CARDINAL;
                  verbose,sendtoscreen,useLFN:BOOLEAN);
VAR
    S,S2:str4096; (* well, we could use hugestr *)
    hndin    : FIO.File;
    hndout   : FIO.File;
    fsize,portion,lastportion,currportion,addr : LONGCARD;
    steps    : CARDINAL;
    wrtofile : BOOLEAN;
    len:CARDINAL;
BEGIN
    IF sendtoscreen THEN
        verbose:=FALSE;
        wrtofile:=FALSE;
    ELSE
        wrtofile:=TRUE;
    END;

    IF wrtofile THEN WrStr("Expanding ");WrFname(useLFN,source); END;

    hndin := fileOpenRead(useLFN,source);
    FIO.AssignBuffer(hndin,IObufferIn);
    IF wrtofile THEN
        hndout := fileCreate(useLFN,cible);
        FIO.AssignBuffer(hndout,IObufferOut);
    ELSE
        hndout := FIO.StandardOutput;
    END;

    (* Work(cmdInit); *)
    addr:=0;
    steps := 10;
    fsize := fileGetFileSize(useLFN,source);
    portion:=fsize DIV LONGCARD(steps); INC(portion); (* avoid DIV 0 ! *)
    lastportion := LONGCARD(steps+1);
    IF verbose THEN
        (* animInit(steps, " ", "", CHR(249), "", "\/" ); *)
        animInit(steps, " ", "", "-", "+", "-\|/" );
    ELSE
        IF wrtofile THEN video(msgWait,TRUE);END;
    END;

    FIO.EOF := FALSE;
    LOOP
        IF FIO.EOF THEN EXIT; END;
        (* Work(cmdShow); *)
        IF verbose THEN
            anim(animShow);
            currportion:=addr DIV portion;
            IF currportion # lastportion THEN
                anim(animAdvance);
                lastportion:=currportion;
            END;
        END;

        FIO.RdStr(hndin,S);
        len := Str.Length(S);
        INC(addr, LONGCARD( len ) );
        IF (S[0]=CHR(0)) AND FIO.EOF THEN EXIT; END;

        detabit(tabwi,S, S2);

        FIO.WrStr(hndout,S2);
        FIO.WrLn(hndout);
    END;
    (* Work(cmdStop); *)
    IF verbose THEN
        anim(animEnd);anim(animClear);
    ELSE
        IF wrtofile THEN video(msgWait,FALSE);END;
    END;

    fileClose(useLFN,hndin);
    IF wrtofile THEN
        FIO.Flush(hndout);
        fileClose(useLFN,hndout);
        donewname(useLFN,source,cible);
        WrLn;
    END;

END expTabs;

(* ------------------------------------------------------------ *)

PROCEDURE newExtension (VAR R:pathtype;S:pathtype;e3:ARRAY OF CHAR);
VAR
    p : CARDINAL;
BEGIN
    Str.Copy(R,S);
    p := Str.RCharPos(R,dot);
    IF p = MAX(CARDINAL) THEN
        ;
    ELSE
        R[p]:=CHR(0);
    END;
    Str.Append(R,e3);
END newExtension;

PROCEDURE chkUD (S:pathtype):BOOLEAN ;
VAR
    u,d,n,e:pathtype;
    pb:CARDINAL;
BEGIN
    Lib.SplitAllPath(S , u,d,n,e);
    pb:=0;
    IF chkJoker(u) THEN INC(pb);END;
    IF chkJoker(d) THEN INC(pb);END;
    RETURN (pb=0);
END chkUD;

(* ------------------------------------------------------------ *)

PROCEDURE msg2 (S1,S2:ARRAY OF CHAR );
BEGIN
    WrStr(S1);WrStr(" : ");WrStr(S2);WrLn;
END msg2;

PROCEDURE msgb (S:ARRAY OF CHAR;tf:BOOLEAN);
BEGIN
    WrStr(S);WrStr(" : ");IO.WrBool(tf,8);WrLn;
END msgb;

PROCEDURE msgc (S:ARRAY OF CHAR;v:CARDINAL );
BEGIN
    WrStr(S);WrStr(" : ");IO.WrCard(v,8);WrLn;
END msgc;

(* tag is already uppercased because we except symbols *)

PROCEDURE validTag (S:ARRAY OF CHAR):BOOLEAN;
VAR
    i,pb,len:CARDINAL;
BEGIN
    len:=Str.Length(S);
    IF len # TAGLEN THEN RETURN FALSE; END;
    pb:=0;
    FOR i:=1 TO len DO
        CASE S[i-1] OF
        | "A".."Z" : INC(pb);
        | "0".."9" : INC(pb);
        | '"'      : INC(pb);
        END;
    END;
    RETURN (pb=0);
END validTag;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    minwidth    = 1;
    maxwidth    = 132;
    YesStr      = "Y";
    NoStr       = "N";
    mintab      = 1;
    maxtab      = 64; (* was 16 *)
    mininterval = 1;
    maxinterval = 1024; (* really oversized ! *)
    mincount    = 1;
    maxcount    = 1024; (* same *)
    unspecified = MAX(CARDINAL);
    defaultdetabwidth = 8;
VAR
    countFile   : CARDINAL;
    key         : str2;
VAR
    parmcount   : CARDINAL;
    i           : CARDINAL;
    opt         : CARDINAL;
    v           : LONGCARD;
VAR
    overwrite   : BOOLEAN;
    cmd         : (glue,break,skip,cookie,detab);
    linewidth, tabwidth,interval,count,detabwidth : CARDINAL;
    usetag,inverse,verbose,columns,sendtoscreen,tabpara,author : BOOLEAN;
    DEBUG       : BOOLEAN;
    S,R,filespec,basedir    : pathtype; (* S and R get parms, and are reused as source/target *)
    IGNORETAG   : str16;
    useLFN      : BOOLEAN;
    ptr,anchor  : pFname;
    state : (waiting,gotspec);
    skipit:BOOLEAN;
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE;
    WrLn;

    DEBUG         := FALSE;
    useLFN        := TRUE;
    overwrite     := TRUE; (* was FALSE *)
    cmd           := glue; (* default *)
    usetag        := TRUE;
    tabwidth      := unspecified; (* was 0 *)
    interval      := unspecified;
    count         := unspecified; (* should be 1 or mincount *)
    detabwidth    := unspecified;
    inverse       := FALSE;
    verbose       := TRUE;
    columns       := FALSE;
    sendtoscreen  := FALSE;
    tabpara       := FALSE;
    author        := FALSE;
    Str.Copy(IGNORETAG,DEFIGNORETAG);
    state := waiting;

    parmcount := Lib.ParamCount();
    IF parmcount=0 THEN abort(errHelp,"");END;

    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i);
        Str.Copy(R,S);
        UpperCase(R);
        cleantabs(R); (* YATB ! *)
        IF isOption(R) THEN
            opt := GetOptIndex(R,"?"+delim+"H"+delim+"HELP"+delim+
                                 "A"+delim+"AUTHOR"+delim+
                                 "T"+delim+"TAGS"+delim+
                                 "W:"+delim+"WIDTH:"+delim+
                                 "T:"+delim+"TAB:"+delim+
                                 "P:"+delim+"PAGE:"+delim+
                                 "C:"+delim+"COUNT:"+delim+
                                 "I"+delim+"INVERSE"+delim+
                                 "Q"+delim+"QUIET"+delim+
                                 "C"+delim+"COLUMNS"+delim+
                                 "S"+delim+"SCREEN"+delim+
                                 "F"+delim+"FORTUNE"+delim+
                                 "FF"+delim+
                                 "G:"+delim+"TAG:"+delim+
                                 "U"+delim+"DETAB"+delim+
                                 "U:"+delim+"DETAB:"+delim+
                                 "L"+delim+"LFN"+delim+
                                 "DEBUG"
                              );
            CASE opt OF
            | 1,2,3 :
                abort(errHelp,"");
            | 4,5 :
                author:=TRUE;
            | 6,7:
                usetag:=FALSE;
            | 8,9:
                CASE cmd OF
                | glue: ;
                | break : IF linewidth # unspecified THEN  abort(errDuplicate,"Line width"); END;
                | skip:   abort(errNonsense,"");
                | detab:  abort(errNonsense,"");
                | cookie: abort(errNonsense,"");
                END;
                IF GetLongCard(S,v)=FALSE THEN abort(errBadNumber,S); END;
                IF ( (v < minwidth) OR (v > maxwidth) ) THEN abort(errBadNumber,S); END;
                linewidth :=CARDINAL(v);
                cmd := break;
            | 10,11:
                CASE cmd OF
                | glue: IF tabwidth # unspecified THEN abort(errDuplicate,"Tab width"); END;
                | break:  abort(errNonsense,"");
                | skip:   abort(errNonsense,"");
                | detab:  abort(errNonsense,"");
                | cookie: abort(errNonsense,"");
                END;
                IF GetLongCard(S,v)=FALSE THEN abort(errBadNumber,S); END;
                IF ( (v < mintab) OR (v > maxtab) ) THEN abort(errBadNumber,S); END;
                tabwidth :=CARDINAL(v);
            | 12,13:
                CASE cmd OF
                | glue: ;
                | break : abort(errNonsense,"");
                | skip : IF interval # unspecified THEN abort(errDuplicate,"Page length"); END;
                | detab:  abort(errNonsense,"");
                | cookie: abort(errNonsense,"");
                END;
                IF GetLongCard(S,v)=FALSE THEN abort(errBadNumber,S); END;
                IF ( (v < mininterval) OR (v > maxinterval) ) THEN abort(errBadNumber,S); END;
                interval :=CARDINAL(v);
                cmd := skip;
            | 14,15:
                CASE cmd OF
                | glue: ;
                | break:  abort(errNonsense,"");
                | skip: IF count # unspecified THEN abort(errDuplicate,"Count");END;
                | detab:  abort(errNonsense,"");
                | cookie: abort(errNonsense,"");
                END;
                IF GetLongCard(S,v)=FALSE THEN abort(errBadNumber,S); END;
                IF ( (v < mincount) OR (v > maxcount) ) THEN abort(errBadNumber,S); END;
                count :=CARDINAL(v);
                cmd := skip;
            | 16,17:
                inverse := TRUE;
            | 18,19:
                verbose:=FALSE;
            | 20,21:
                columns := TRUE;
            | 22,23:
                sendtoscreen:=TRUE;
            | 24,25:
                CASE cmd OF
                | glue: ; (* remember it's default command after all ! *)
                | break : abort(errNonsense,"");
                | skip:   abort(errNonsense,"");
                | detab:  abort(errNonsense,"");
                | cookie : ;
                END;
                cmd:=cookie;
                tabpara:=FALSE;
            | 26:
                CASE cmd OF
                | glue: ; (* remember it's default command after all ! *)
                | break : abort(errNonsense,"");
                | skip:   abort(errNonsense,"");
                | detab:  abort(errNonsense,"");
                | cookie : ;
                END;
                cmd:=cookie;
                tabpara:=TRUE;
            | 27,28:
                GetString(R,IGNORETAG); (* use uppercased tag *)
                IF validTag(IGNORETAG)=FALSE THEN abort(errBadTag,IGNORETAG);END;
            | 29,30:
                CASE cmd OF
                | glue: ;
                | break:  abort(errNonsense,"");
                | skip:   abort(errNonsense,"");
                | detab:  IF detabwidth # unspecified THEN abort(errDuplicate,"Expand tab width"); END;
                | cookie: abort(errNonsense,"");
                END;
                cmd:=detab;
                detabwidth:=defaultdetabwidth;
            | 31,32:
                CASE cmd OF
                | glue: ;
                | break:  abort(errNonsense,"");
                | skip:   abort(errNonsense,"");
                | detab:  IF detabwidth # unspecified THEN abort(errDuplicate,"Expand tab width"); END;
                | cookie: abort(errNonsense,"");
                END;
                IF GetLongCard(S,v)=FALSE THEN abort(errBadNumber,S); END;
                IF ( (v < mintab) OR (v > maxtab) ) THEN abort(errBadNumber,S); END;
                detabwidth :=CARDINAL(v);
                cmd:=detab;
            | 33,34:
                useLFN := FALSE;
            | 35:
                DEBUG := TRUE;
            ELSE
                abort(errOption,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            CASE state OF
            | waiting : Str.Copy(filespec,S); (* org case *)
            | gotspec : abort(errTooManyParms,S);
            END;
            INC(state);
        END;
    END;
    (* check nonsense *)
    IF state=waiting THEN abort(errMissingSpec,""); END;

    useLFN:=( useLFN AND w9XsupportLFN() );
    IF NOT(useLFN) THEN UpperCase(filespec);END;

    CASE cmd OF
    | glue: (* sendtoscreen *)
        IF (inverse OR columns) THEN abort(errNonsense,"");END;
        IF tabwidth = unspecified THEN tabwidth:=0; END;
        IF author THEN abort(errNonsense,"");END;
        IF DEBUG THEN
            msg2("cmd","glue");
            msgb("usetag",usetag);
            msg2("tag",IGNORETAG);
            msgc("tabwidth",tabwidth);
        END;
    | break:
        IF (inverse OR columns) THEN abort(errNonsense,"");END;
        IF tabwidth # unspecified THEN abort(errNonsense,"");END;
        IF author THEN abort(errNonsense,"");END;
        IF DEBUG THEN
            msg2("cmd","break");
            msgb("usetag",usetag);
            msg2("tag",IGNORETAG);
            msgc("linewidth",linewidth);
            msgb("sendtoscreen",sendtoscreen);
        END;
    | skip:
        IF tabwidth # unspecified THEN abort(errNonsense,"");END;
        IF interval = unspecified THEN abort(errPageLength,"");END;
        IF count    = unspecified THEN count := 1;  END;
        IF usetag=FALSE THEN abort(errNonsense,"");END;
        IF author THEN abort(errNonsense,"");END;
        IF DEBUG THEN
            msg2("cmd","skip");
            msgc("page length",interval);
            msgc("count",count);
            msgb("inverse",inverse);
            msgb("columns",columns);
            msgb("sendtoscreen",sendtoscreen);
        END;
    | cookie: (* sendtoscreen *)
        IF (inverse OR columns) THEN abort(errNonsense,"");END;
        IF tabwidth # unspecified THEN abort(errNonsense,"");END;
        IF DEBUG THEN
            msg2("cmd","cookie");
            msgb("tab",usetag);
            msgb("author",author);
            msgb("sendtoscreen",sendtoscreen);
        END;
    | detab:
        IF (inverse OR columns) THEN abort(errNonsense,"");END;
        IF tabwidth # unspecified THEN abort(errNonsense,"");END;
        IF interval # unspecified THEN abort(errNonsense,"");END;
        IF count    # unspecified THEN abort(errNonsense,"");END;
        IF author THEN abort(errNonsense,"");END;
        IF DEBUG THEN
            msg2("cmd","detab");
            msgc("detabwidth",detabwidth);
        END;
    END;

        (* a few q&d sanity checks *)

        IF DEBUG THEN WrStr("Filespec : ");WrFname(useLFN,filespec);WrLn;END;
        IF chkUD(filespec)=FALSE THEN abort(errBadSpec,filespec);END;
        IF same(filespec,dot) THEN Str.Copy(filespec,stardotstar); END;
        IF same(filespec,star) THEN Str.Copy(filespec,stardotstar); END;
        IF Str.Match(filespec,"*"+antislash) THEN Str.Append(filespec,stardotstar); END;
        IF chkJoker(filespec)=FALSE THEN
            IF fileIsDirectorySpec ( useLFN,filespec) THEN
                fixDirectory(filespec);
                Str.Append(filespec,stardotstar);
            END;
        END;
        IF DEBUG THEN WrStr("Filespec : ");WrFname(useLFN,filespec);WrLn;END;
        makebase(useLFN,filespec,basedir);
        IF DEBUG THEN WrStr("Basedir  : ");WrFname(useLFN,basedir);WrLn;END;


    initList(anchor);
    countFile := buildFileList(anchor,useLFN,DEBUG,filespec);
    CASE countFile OF
    | MAX(CARDINAL): abort(errTooMany,filespec);
    | 0 :            abort(errNoMatch,filespec);
    END;

    IF DEBUG THEN abort(errNone,"");END; (* LFN check, nothing more *)

    WrStr(Banner);WrLn;
    WrLn;

    ptr:=anchor;
    WHILE ptr # NIL DO
        getStr(filespec,ptr);
        Str.Concat(S,basedir,filespec);

        newExtension(R,S,extBAK); (* S=source,R=backup *)

        skipit := FALSE;
        IF fileExists(useLFN,R) THEN
            IF overwrite THEN
		        fileErase(useLFN,R);
		        (*
		        WrStr("Existing ");WrStr(R);
		        WrStr(" has just been deleted");WrLn;
		        WrLn;
		        *)
            ELSE
                IF fileIsRO(useLFN,R) THEN
                    WrStr("Read-only b");
                ELSE
                    WrStr("B");
                END;
                WrStr("ackup file ");WrFname(useLFN,R);
                WrStr(" already exists !");WrLn;
                WrLn;
                WrStr("Hit '"+YesStr+"' to erase it, '"+NoStr+"' to skip it, or any other key to abort : ");
		        Flushkey;
		        key :=Waitkey();
		        UpperCase(key);
		        IF same(key,YesStr) THEN
		            WrStr("Yes");WrLn;
		            WrLn;
		            fileSetRW(useLFN,R); (* how ungraceful just to avoid YAB Yet Another Boolean *)
		            fileErase(useLFN,R);
		        ELSIF same(key,NoStr) THEN
		            WrStr("Skipped !");WrLn;
		            WrLn;
		            skipit := TRUE;
		        ELSE
		            WrStr("Exit !");WrLn;
		            abort(errAbort,"");
		        END;
            END;
        END;
        IF NOT(skipit) THEN
            (* the cut/paste/adapt syndrom strikes again *)
            CASE cmd OF
            | glue:  BuildParagraphs(S,R,IGNORETAG,tabwidth,usetag,verbose,sendtoscreen,useLFN);
            | break: BreakLines(S,R,IGNORETAG,linewidth,usetag,verbose,sendtoscreen,useLFN);
            | skip:  IF columns THEN
                         SkipColumns(S,R,interval,count,inverse,verbose,sendtoscreen,useLFN);
                     ELSE
                         SkipLines(S,R,interval,count,inverse,verbose,sendtoscreen,useLFN);
                     END;
            | cookie: procFortune(S,R, tabpara,author,verbose,sendtoscreen,useLFN);
            | detab: expTabs(S,R,detabwidth,verbose,sendtoscreen,useLFN);
            END;
            WrLn;
        END;

        ptr:=ptr^.next;
    END;
    freeList(anchor);

    abort(errNone,"");
END TxtFmt.




