(* ---------------------------------------------------------------
Title         Q&D Byte Processor
Author        PhG
Overview      see help
Usage         see help
Notes         existing f8e3 target with matching LFN lets LFN alive
Bugs
Wish List     allow user-specified operation with pseudo assembler ?
              allow XOR with a string ?

--------------------------------------------------------------- *)

MODULE ByteProc;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;
IMPORT IO;
IMPORT SYSTEM;

FROM IO IMPORT WrStr,WrLn;

FROM FIO IMPORT FIXEDLIBS;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite,
verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs,
completedInit, completedShow, completedSHOW, completedEnd, completed;

FROM Storage IMPORT ALLOCATE,DEALLOCATE,Available;

FROM QD_LFN IMPORT path9X, huge9X, findDataRecordType,
unicodeConversionFlagType, w9XchangeDir,
w9XgetDOSversion, w9XgetTrueDOSversion, w9XisWindowsEnh, w9XisMSDOS7,
w9XfindFirst, w9XfindNext, w9XfindClose, w9XgetCurrentDirectory,
w9XlongToShort, w9XshortToLong, w9XtrueName, w9XchangeDir,
w9XmakeDir, w9XrmDir, w9Xrename, w9XopenFile, w9XcloseFile,
w9XsupportLFN;

FROM QD_File IMPORT pathtype, w9XnothingRequired,
fileOpenRead, fileOpen, fileExists, fileIsRO, fileSetRW, fileSetRO,
fileErase, fileCreate, fileRename, fileGetFileSize, fileIsDirectorySpec,
fileClose;

(* ------------------------------------------------------------ *)

CONST
    cr          = CHR(13);
    lf          = CHR(10);
    dot         = ".";
    star        = "*";
    stardotstar = star+dot+star;
    blank       = " ";
    dotdot      = dot+dot;
    nullchar    = 0C;
    dquote      = '"';
    coma        = ",";
    dollar      = "$";
    digits      = "0123456789";
    hexadigits  =           "ABCDEF";

(* ------------------------------------------------------------ *)

CONST
    ProgEXEname   = "BYTEPROC";
    ProgTitle     = "Q&D Byte Processor";
    ProgVersion   = "v1.1d";
    ProgCopyright = "by PhG";
    Banner        = ProgTitle+" "+ProgVersion+" "+ProgCopyright;
    byterange     = "*";
    fposmodulo    = "!";
    extNEW        = ".NEW";
CONST
    errNone             = 0;
    errHelp             = 1;
    errUnknownOption    = 2;
    errTooManyParms     = 3;
    errNotEnough        = 4;
    errIllegal          = 5;
    errBadValue         = 6;
    errByteTooBig       = 7;
    errTooManyMatches   = 8;
    errNoMatch          = 9;
    errAborted          = 10;
    errBadValueRangeL   = 11;
    errByteTooBigRangeL = 12;
    errBadValueRangeU   = 13;
    errByteTooBigRangeU = 14;
    errBadRange         = 15;
    errBadRangeForOp    = 16;
    errXORonly          = 17;
    errRange            = 18;

(* ------------------------------------------------------------ *)

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);

    MODULE message;
    IMPORT Str;
    EXPORT msg2,msg3;

    PROCEDURE msg2 (VAR R:ARRAY OF CHAR;S1,S2:ARRAY OF CHAR);
    BEGIN
        Str.Concat(R,S1,S2);
    END msg2;

    PROCEDURE msg3 (VAR R:ARRAY OF CHAR;S1,S2,S3:ARRAY OF CHAR);
    BEGIN
        Str.Concat(R,S1,S2);Str.Append(R,S3);
    END msg3;

    END message;

CONST
    nl = cr+lf;
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)
    helpmsg =
Banner+nl+
nl+
"Syntax : "+ProgEXEname+" [option]... <operation> <value|range|"+byterange+"|"+fposmodulo+"> <filespec>"+nl+
nl+
"-p:#<,|..># specify start,count or start..end file positions range"+nl+
"-n          exclude CR and LF from operation"+nl+
"-c          exclude CR from operation"+nl+
"-f          exclude LF from operation"+nl+
"-k          exclude all control characters ([0..31]) from operation"+nl+
"-q          quiet processing"+nl+
"-a          audio warning at program completion"+nl+
"-x          disable LFN support even if available"+nl+
nl+
"a) Note any existing target will be overwritten without warning."+nl+
"b) Operation to perform may be XOR, INC, DEC, INCR, DECR, PLUS or MINUS."+nl+
"c) Value to use for operation is a byte in the [0..255] or [1..128] range."+nl+
"d) "+'"'+byterange+'" means [$00..$FF] processing for XOR, else it means [$01..$80].'+nl+
"e) "+'"'+fposmodulo+'" means value is XORed with file position modulo 256 (XOR required).'+nl+
'f) New files are created with ".$##" (hexadecimal value) extension :'+nl+
"   therefore, files with such an extension will not be processed."+nl+
"g) Range can be specified in either decimal or hexadecimal ("+dquote+dollar+dquote+" prefix)."+nl+
nl+
"Examples : "+ProgEXEname+" x 0..$10 *.bin"+nl+
"           "+ProgEXEname+" i "+byterange+" mystery.dat"+nl+
"           "+ProgEXEname+" x $ff selectit.cfg"+nl+
"           "+ProgEXEname+" x "+fposmodulo+" install.inf"+nl
(*%F FIXEDLIBS  *)
+nl+
"Unfortunately, thanks (!) to a fatal bug in TopSpeed Modula-2 FIO library,"+nl+
"paths are limited to 65 characters (longer ones will NOT be found). :-("+nl
(*%E *)
    ;
VAR
    S : str1024;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(helpmsg);
    | errUnknownOption :
        msg3(S,"Unknown ",einfo," option !");
    | errTooManyParms :
        msg2(S,einfo," is just one parameter too far !");
    | errNotEnough:
        S:="Not enough parameters in command line !";
    | errIllegal:
        msg3(S,"Illegal ",einfo," parameter !");
    | errBadValue :
        msg2(S,einfo," would not convert to a number !");
    | errByteTooBig :
        msg2(S,einfo," is not a valid byte value !");
    | errTooManyMatches:
        msg3(S,"Too many files match ",einfo," specification !");
    | errNoMatch :
        msg3(S,"No match for ",einfo," specification !");
    | errAborted:
        S := "Aborted by user !";
    | errBadValueRangeL :
        msg3(S,"Lower value of ",einfo," range would not convert to a number !");
    | errByteTooBigRangeL :
        msg3(S,"Lower value of ",einfo," range is not a valid byte value !");
    | errBadValueRangeU :
        msg3(S,"Upper value of ",einfo," range would not convert to a number !");
    | errByteTooBigRangeU :
        msg3(S,"Upper value of ",einfo," range is not a valid byte value !");
    | errBadRange:
        msg3(S,"Illogical ",einfo," range !");
    | errBadRangeForOp :
        msg3(S,"Illegal ",einfo," range for operation specified !");
    | errXORonly:
        S:='XOR operation is required with "'+fposmodulo+'" command !';
    | errRange:
        msg3(S,"Illegal ",einfo," file positions range !");
    ELSE
        S:="This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp :
        ; (* nada *)
    ELSE
        WrStr(ProgEXEname+" : "); WrStr(S); WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE hasReservedExtension ( S:ARRAY OF CHAR ):BOOLEAN;
CONST
    hexnums   = "ABCDEFabcdef"; (* should be already upper but... *)
    hexdigits = digits+hexnums;
VAR
    p : CARDINAL;
BEGIN
    (* we only care about ".$##" *)
    p := Str.RCharPos(S,dot);
    IF p = MAX(CARDINAL) THEN RETURN FALSE; END;
    Str.Delete(S,0,p+1);
    IF Str.Length(S) # 3 THEN RETURN FALSE; END;
    p:=0;
    IF S[0]                    = "$" THEN INC(p);END;
    IF verifyString (S[1],hexdigits) THEN INC(p);END;
    IF verifyString (S[2],hexdigits) THEN INC(p);END;
    (* we could also check for .new here *)
    RETURN (p=3);
END hasReservedExtension;

(* ------------------------------------------------------------ *)

TYPE
    pFname = POINTER TO fnameType;
    fnameType = RECORD
        next      : pFname;
        slen      : SHORTCARD;
        str       : CHAR;
    END;

PROCEDURE initList (VAR anchor : pFname );
BEGIN
    anchor := NIL;
END initList;

PROCEDURE freeList (anchor : pFname);
VAR
    needed : CARDINAL;
    p      : pFname;
BEGIN
    (* p:=anchor; *)
    WHILE anchor # NIL DO
        needed := SIZE(fnameType) - SIZE(anchor^.str) + CARDINAL(anchor^.slen);
        p := anchor^.next;
        DEALLOCATE(anchor,needed);
        anchor:=p;
    END
END freeList;

PROCEDURE buildNewPtr (VAR anchor,p:pFname; len:CARDINAL):BOOLEAN;
VAR
    needed : CARDINAL;
BEGIN
    needed := SIZE(fnameType) - SIZE(p^.str) + len;
    IF Available(needed)=FALSE THEN RETURN FALSE; END;
    IF anchor = NIL THEN
        ALLOCATE(anchor,needed);
        p:=anchor;
    ELSE
        p:=anchor;
        WHILE p^.next # NIL DO
            p:=p^.next;
        END;
        ALLOCATE(p^.next,needed);
        p:=p^.next;
    END;
    p^.next := NIL;
    RETURN TRUE;
END buildNewPtr;

(* assume p is valid *)

PROCEDURE getStr (VAR S : pathtype; p:pFname);
VAR
    len:CARDINAL;
BEGIN
    len := CARDINAL(p^.slen);
    Lib.FastMove( ADR(p^.str),ADR(S),len);
    S[len] := nullchar; (* REQUIRED safety ! *)
END getStr;

(* ------------------------------------------------------------ *)

PROCEDURE isReservedEntry (S:ARRAY OF CHAR) : BOOLEAN;
BEGIN
    IF same(S,dot) THEN RETURN TRUE; END;
    RETURN same(S,dotdot);
END isReservedEntry;

PROCEDURE buildFileList (VAR anchor:pFname;
                        useLFN:BOOLEAN;spec:pathtype):CARDINAL;
VAR
    count:CARDINAL; (* should do ! *)
    ok,found:BOOLEAN;
    unicodeconversion:unicodeConversionFlagType;
    w9Xentry : findDataRecordType;
    w9Xhandle,errcode:CARDINAL;
    entry : FIO.DirEntry;
    dosattr:FIO.FileAttr;
    entryname:pathtype;
    len : CARDINAL;
    pp:pFname;
    includeme:BOOLEAN;
BEGIN
    count:=0;
    IF useLFN THEN
        found := w9XfindFirst (spec,SHORTCARD(everything),SHORTCARD(w9XnothingRequired),
                              unicodeconversion,w9Xentry,w9Xhandle,errcode);
    ELSE
        found := FIO.ReadFirstEntry(spec,everything,entry);
    END;
    WHILE found DO
        IF useLFN THEN
            Str.Copy(entryname,w9Xentry.fullfilename);
        ELSE
            Str.Copy(entryname,entry.Name);
        END;
        includeme := NOT( isReservedEntry(entryname) ); (* skip "." and ".." *)
        includeme := includeme AND (hasReservedExtension(entryname)=FALSE);
        IF includeme THEN
            IF useLFN THEN
                dosattr:=FIO.FileAttr(w9Xentry.attr AND 0FFH);
            ELSE
                dosattr:=entry.attr;
            END;
            IF NOT (aD IN dosattr) THEN
                (* if file has no extension, add it as a marker *)
                IF Str.RCharPos(entryname,".")=MAX(CARDINAL) THEN
                    Str.Append(entryname,".");
                END;
                len:=Str.Length(entryname);
                IF buildNewPtr(anchor,pp,len)=FALSE THEN
                    IF useLFN THEN ok:=w9XfindClose(w9Xhandle,errcode); END;
                    RETURN MAX(CARDINAL); (* errStorage *)
                END;
                INC(count);
                pp^.slen      := SHORTCARD(len);
                Lib.FastMove ( ADR(entryname),ADR(pp^.str),len );
            END;
        END;
        IF useLFN THEN
            found :=w9XfindNext(w9Xhandle, unicodeconversion,w9Xentry,errcode);
        ELSE
            found :=FIO.ReadNextEntry(entry);
        END;
    END;
    IF useLFN THEN ok:=w9XfindClose(w9Xhandle,errcode); END;
    RETURN count;
END buildFileList;

PROCEDURE buildPath (VAR path:pathtype; spec:pathtype);
VAR
    u,d,n,e:pathtype;
BEGIN
    Lib.SplitAllPath(spec, u,d,n,e);
    Str.Concat(path, u,d);
    fixDirectory(path); (* safety *)
END buildPath;

PROCEDURE wrQ (useLFN:BOOLEAN;S:ARRAY OF CHAR   );
BEGIN
    IF useLFN THEN WrStr(dquote);END;
    WrStr(S);
    IF useLFN THEN WrStr(dquote);END;
END wrQ;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

TYPE
    proctype = PROCEDURE(WORD,WORD):WORD;

PROCEDURE xor (b1,b2 : WORD) : WORD;
BEGIN
    RETURN WORD ( (BITSET (b1) / BITSET (b2)) );
END xor;

PROCEDURE plus (b1,b2 : WORD) : WORD;
BEGIN
    RETURN (b1+b2);
END plus;

PROCEDURE minus (b1,b2 : WORD) : WORD;
BEGIN
    RETURN (b1-b2);
END minus;

(* ------------------------------------------------------------ *)

CONST
    ioBufferSize    = (8 * 512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = ioBufferSize;
TYPE
    ioBuffer  = ARRAY [firstBufferByte..lastBufferByte] OF BYTE;
VAR
    bufferIn  : ioBuffer;
    bufferOut : ioBuffer;

CONST
    dataBufferSize      = 1024;
    firstDataBufferByte = 1-1;
    lastDataBufferByte  = dataBufferSize-1;
VAR
    databuffer : ARRAY [firstDataBufferByte..lastDataBufferByte] OF BYTE;

(* ------------------------------------------------------------ *)

PROCEDURE getlc (VAR v:LONGCARD;S:ARRAY OF CHAR):BOOLEAN;
VAR
    base:CARDINAL;
    ok:BOOLEAN;
BEGIN
    IF S[0]=dollar THEN
        Str.Delete(S,0,1);
        base := 16;
    ELSE
        base := 10;
    END;
    v:=Str.StrToCard(S,base,ok);
    RETURN ok;
END getlc;

PROCEDURE parserange (VAR startpos,endpos:LONGCARD;S:ARRAY OF CHAR ):BOOLEAN;
VAR
    ok,iscount:BOOLEAN;
    p,l:CARDINAL;
    Z:str16;
BEGIN
    Str.Caps(S); (* [$]#<,|..>[$]# *)
    ok:=verifyString(S,digits+hexadigits+dot+coma+dollar);
    IF NOT(ok) THEN RETURN ok;END;

    ok:=FALSE;
    IF CharCount(S,coma)=1 THEN
        p:=Str.CharPos(S,coma); l:=1;    iscount:=TRUE;
    ELSIF Str.Pos(S,dotdot) # MAX(CARDINAL) THEN
        p:=Str.Pos(S,dotdot);   l:=2;    iscount:=FALSE;
    ELSE
        RETURN ok;
    END;
    Str.Slice(Z,S,0,p);
    Str.Delete(S,0,p+l);
(* WrStr(Z);WrLn; *)
    ok:=getlc(startpos,Z);
    IF NOT(ok) THEN RETURN ok;END;

(* WrStr(S);WrLn; *)
    ok:=getlc(endpos,S);
    IF NOT(ok) THEN RETURN ok;END;

    IF iscount THEN INC(endpos,startpos);DEC(endpos,1); END;

(* IO.WrLngCard(startpos,1);WrStr("  ..  ");IO.WrLngCard(endpos,1);WrLn; *)

    ok:=(startpos <= endpos);
    RETURN ok;
END parserange;

PROCEDURE inrange (startpos,endpos,fpos:LONGCARD;index:CARDINAL   ):BOOLEAN;
BEGIN
    INC(fpos, LONGCARD(index) ); (* index is 0-based *)
    IF fpos < startpos THEN RETURN FALSE;END;
    IF fpos > endpos   THEN RETURN FALSE;END;
    RETURN TRUE;
END inrange;

PROCEDURE vtostr (v:LONGCARD;base:CARDINAL):str16;
VAR
    R:str16;
    ok:BOOLEAN;
BEGIN
    Str.CardToStr(v,R,base,ok);
    IF base=16 THEN Str.Lows(R);END;
    RETURN R;
END vtostr;

(* ------------------------------------------------------------ *)

CONST
    firstcode    = 0;
    lastcontrol  = ORD(" ")-1; (* i.e. 31 *)
    lastcode     = 255;
VAR
    doop : ARRAY [firstcode..lastcode] OF BOOLEAN;

(* ------------------------------------------------------------ *)

CONST
    chFiller = "."; (* handle nasty vindoze OEM charset ! *)
CONST
    opmodulo  = MAX(BYTE)+1;
VAR
    parmcount : CARDINAL;
    state     : (waiting,gotop,gotvalue,gotspec);
    S,R,T,spec,path : pathtype;
    i,opt,n   : CARDINAL;
    operation : (doxor,doinc,dodec,domodulo);
    firstopvalue,lastopvalue  : CARDINAL;
    audiowarn,verbose : BOOLEAN;
    proc      : proctype;
    source,target,targetbase : pathtype;
    index     : CARDINAL;
    v         : LONGCARD;
    valeur    : CARDINAL;
    DEBUG,useLFN,ok        : BOOLEAN;
    anchor,ptr:pFname;
    countFile:CARDINAL;
VAR
    fpos,addr,fsize,portion,lastportion,currportion : LONGCARD;
    startpos,endpos:LONGCARD;
    steps    : CARDINAL;
    hndIn      : FIO.File;
    hndOut     : FIO.File;
    got        : CARDINAL;
    e3         : str16; (* oversized *)
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE;
    WrLn;

    parmcount := Lib.ParamCount();
    IF parmcount = 0 THEN abort(errHelp,"");END;

    FOR i:=firstcode TO lastcode DO doop[i]:=TRUE; END;
    audiowarn := FALSE;
    verbose   := TRUE;
    useLFN    := TRUE;
    DEBUG     := FALSE;
    startpos  := 0;
    endpos    := MAX(LONGCARD);
    state     := waiting;

    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i); cleantabs(S);
        Str.Copy(R,S);
        UpperCase(R);
        IF isOption(R) THEN
            opt := GetOptIndex(R, "?"+delim+"H"+delim+"HELP"+delim+
                                  "A"+delim+"AUDIO"+delim+
                                  "Q"+delim+"QUIET"+delim+
                                  "X"+delim+"LFN"+delim+
                                  "N"+delim+"NL"+delim+
                                  "C"+delim+"CR"+delim+
                                  "F"+delim+"LF"+delim+
                                  "K"+delim+"CTRL"+delim+
                                  "DEBUG"+delim+
                                  "P:"+delim+"POS:"
                              );
            CASE opt OF
            |  1,  2,  3: abort(errHelp,"");
            |  4,  5    : audiowarn := TRUE;
            |  6,  7    : verbose   := FALSE;
            |  8,  9    : useLFN    := FALSE;
            | 10, 11    : doop[ ORD(cr) ] := FALSE;
                          doop[ ORD(lf) ] := FALSE;
            | 12, 13    : doop[ ORD(cr) ] := FALSE;
            | 14, 15    : doop[ ORD(lf) ] := FALSE;
            | 16, 17    : FOR n:= firstcode TO lastcontrol DO doop[n]:=FALSE; END;
            | 18        : DEBUG:=FALSE;
            | 19,20     : GetString(S,T);
                          IF parserange(startpos,endpos,T)=FALSE THEN abort(errRange,S);END;
            ELSE
                abort(errUnknownOption,S);
            END;
        ELSE
            CASE state OF
            | waiting :
                n := getStrIndex(delim,R, "XOR"+delim+"X"+delim+
                                          "INC"+delim+"INCR"+delim+"I"+delim+"PLUS"+delim+
                                          "DEC"+delim+"DECR"+delim+"D"+delim+"MINUS"
                                );
                CASE n OF
                | 1,2   :    operation := doxor; proc := xor;
                | 3,4,5,6 :  operation := doinc; proc := plus;
                | 7,8,9,10 : operation := dodec; proc := minus;
                ELSE abort(errIllegal,"<operation>");
                END;
            | gotop   :
                IF same(R,byterange) THEN
                    IF operation=doxor THEN
                        firstopvalue := MIN(BYTE);
                        lastopvalue  := MAX(BYTE);
                    ELSE
                        firstopvalue := MIN(BYTE)+1;
                        lastopvalue  := 128;
                    END;
                ELSIF same(R,fposmodulo) THEN
                    IF operation=doxor THEN        (* fake 256..256 *)
                        firstopvalue := opmodulo;
                        lastopvalue  := opmodulo;
                    ELSE
                        abort(errXORonly,"");
                    END;
                ELSE
                    n := Str.Pos(R,dotdot);
                    CASE n OF
                    | 0, MAX(CARDINAL):     (* if "..", force an error *)
                        ok:=getlc(v,R);
                        IF ok =FALSE THEN abort(errBadValue,S); END;
                        IF v > MAX(BYTE) THEN abort(errByteTooBig,S);END;
                        firstopvalue := CARDINAL(v);
                        lastopvalue := firstopvalue;
                    ELSE
                        Str.Slice(T,R,0,n); (* n-1+1 *)
                        ok:=getlc(v,T);
                        IF ok =FALSE THEN abort(errBadValueRangeL,S); END;
                        IF v > MAX(BYTE) THEN abort(errByteTooBigRangeL,S);END;
                        firstopvalue := CARDINAL(v);
                        Str.Delete(R,0,n);
                        Str.Subst(R,dotdot,"");
                        Str.Copy(T,R);
                        ok:=getlc(v,T);
                        IF ok =FALSE THEN abort(errBadValueRangeU,S); END;
                        IF v > MAX(BYTE) THEN abort(errByteTooBigRangeU,S);END;
                        lastopvalue := CARDINAL(v);
                        IF lastopvalue < firstopvalue THEN abort(errBadRange,S);END;
                        IF operation=doxor THEN
                            IF NOT ( (firstopvalue >= MIN(BYTE)) AND (lastopvalue <= MAX(BYTE)) ) THEN
                                abort(errBadRangeForOp,S);
                            END;
                        ELSE
                            IF NOT ( (firstopvalue >= (MIN(BYTE)+1)) AND (lastopvalue <= 128) ) THEN
                                abort(errBadRangeForOp,S);
                            END;
                        END;
                    END;
                END;
            | gotvalue:
                Str.Copy(spec,S);
            | gotspec : abort(errTooManyParms,R);
            END;
            INC(state);
        END;
    END;
    IF state # gotspec THEN abort(errNotEnough,"");END;

    useLFN := ( useLFN AND w9XsupportLFN() );

    IF same(spec,".") THEN
        spec:=stardotstar;
    ELSIF Str.Match(spec,"*\") THEN
        Str.Append(spec,stardotstar);
    ELSIF Str.Match(spec,"*\.") THEN
        i:=Str.Length(spec);
        spec[i-1]:=nullchar;
        Str.Append(spec,stardotstar);
    ELSIF same(spec,stardotstar) THEN
        ;
    ELSIF same(spec,"*.") THEN (* ugly *)
        ;
    ELSE
        IF fileIsDirectorySpec(useLFN,spec) THEN
            fixDirectory(spec);
            Str.Append(spec,stardotstar);
        END;
    END;

    initList(anchor);
    countFile:=buildFileList(anchor,useLFN,spec);
    IF countFile=MAX(CARDINAL) THEN abort(errTooManyMatches,spec);END; (* errStorage *)
    IF countFile=0 THEN abort(errNoMatch,spec); END;

    buildPath(path,  spec);

    WrStr(Banner);WrLn;
    WrLn;
    IF ( (startpos = 0) AND (endpos = MAX(LONGCARD)) ) THEN
        ;
    ELSE
        S:="::: Decimal operating range     : ~..~";
        Str.Subst(S,"~",vtostr(startpos,10));
        Str.Subst(S,"~",vtostr(endpos,10));
        WrStr(S);WrLn;
        S:="::: Hexadecimal operating range : $~..$~";
        Str.Subst(S,"~",vtostr(startpos,16));
        Str.Subst(S,"~",vtostr(endpos,16));
        WrStr(S);WrLn;
        WrLn;
    END;

    IF firstopvalue=opmodulo THEN operation:=domodulo;END;

    ptr:=anchor;
    WHILE ptr # NIL DO
        getStr(R,ptr);
        i := Str.RCharPos(R,dot);
        IF i # MAX(CARDINAL) THEN
            Str.Slice(S,R,0,i);
            Str.Copy(R,S);
        END;
        Str.Concat(targetbase,path,R);
        Str.Append(targetbase,dot);

        getStr(source,ptr);
        Str.Prepend(source,path);

        WrStr("+++ ");wrQ(useLFN,source);

        FOR n := firstopvalue TO lastopvalue DO
            IF operation=domodulo THEN
                Str.Copy(e3,"!");
            ELSE
                Str.CardToStr(LONGCARD(n),e3,16,ok);
                UpperCase(e3);
                IF n < 16 THEN Str.Prepend(e3,"0"); END;
            END;

            Str.Prepend(e3,"$");
            Str.Concat(target,targetbase,e3);

            Str.Concat(S,blank,e3);Str.Append(S,blank);
            video(S,TRUE);

            steps := 10;
            fsize := fileGetFileSize(useLFN,source);
            portion:=fsize DIV LONGCARD(steps); INC(portion); (* avoid DIV 0 ! *)
            lastportion := LONGCARD(steps+1);
            addr  := 0;

            IF verbose THEN
                (* AltAnimation(cmdInit); *)
                animInit(steps, "[", "]", chFiller, "", "\/" );
            END;

            hndIn :=fileOpenRead(useLFN,source);
            FIO.AssignBuffer(hndIn,bufferIn);
            hndOut :=fileCreate(useLFN,target);
            FIO.AssignBuffer(hndOut,bufferOut);
            FIO.EOF:=FALSE; (* useless safety *)

            LOOP
                IF verbose THEN
                    (* AltAnimation(cmdSHOW); *)
                    anim(animShow);
                    currportion:=addr DIV portion;
                    IF currportion # lastportion THEN
                        anim(animAdvance);
                        lastportion:=currportion;
                    END;
                END;
                got:=FIO.RdBin(hndIn,databuffer,dataBufferSize);
                IF got = 0 THEN EXIT; END;
                fpos:=addr; (* for domodulo() *)
                INC(addr, LONGCARD(got) );

                FOR index:= firstDataBufferByte TO (got-1) DO
                    IF doop[ ORD (databuffer[index]) ] THEN
                        IF operation=domodulo THEN
                            v := (fpos+LONGCARD(index)) MOD 256;
                            valeur:=CARDINAL(v);
                        ELSE
                            valeur:=n;
                        END;
                        IF inrange(startpos,endpos,fpos,index) THEN
                            databuffer[index] := BYTE(proc(WORD(databuffer[index]),valeur));
                        END;
                    END;
                END;

                FIO.WrBin(hndOut,databuffer,got);
                IF got # dataBufferSize THEN EXIT; END;
            END;
            FIO.Flush(hndOut);
            fileClose(useLFN,hndOut);
            fileClose(useLFN,hndIn);

            IF verbose THEN
                (* AltAnimation(cmdStop); *)
                anim(animEnd);anim(animClear);
            END;
            video(S,FALSE);

            IF ChkEscape() THEN
                WrLn;
                WrLn;
                abort(errAborted,"");
            END;
        END;

        WrLn;

        ptr:=ptr^.next;

    END;

    freeList(anchor);

    IF audiowarn THEN
        Lib.Sound (100);
        Lib.Delay (100);
        Lib.NoSound;
    END;

    abort(errNone,"");
END ByteProc.

