(* ---------------------------------------------------------------
Title         Q&D Binary File Compare
Overview      try and correct DR6/ND7 buggy COMP utility
Usage         see help
Notes         check all batches using FCOMP before v1.3
              scan for FIXME and CHKME

              -mem -fake is UNWISE and should therefore be UNDOCUMENTED ;-)

Bugs          - we're lucky doGetDir() found in PCOPY and MAKEPATH prepends u:
                which is not ALWAYS returned by shortToLong (at least at root)
              - yet another M$ one (will they ever stop ?) :
                vindoze does NOT like frequent calls to BIOS chkEscape() !
              - well, side-effect : when comparing with subdirs,
                duplicate filenames can be flagged as different files
Wish List     include DCOMP functions
              some day (but why ?), use a more clever entries storage method
              (dir entries is really ugly for now)
              (more) clever match for jokers ?

--------------------------------------------------------------- *)

MODULE fComp;

FROM IO IMPORT WrStr, WrLn;

IMPORT Str;
IMPORT Lib;
IMPORT FIO;
IMPORT SYSTEM;
IMPORT IO;

FROM FIO IMPORT FIXEDLIBS;

FROM Storage IMPORT ALLOCATE,DEALLOCATE,Available;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits,
getAllLegalUnits;

FROM QD_LFN IMPORT path9X, huge9X, findDataRecordType,
unicodeConversionFlagType, w9XchangeDir,
w9XgetDOSversion, w9XgetTrueDOSversion, w9XisWindowsEnh, w9XisMSDOS7,
w9XfindFirst, w9XfindNext, w9XfindClose, w9XgetCurrentDirectory,
w9XlongToShort, w9XshortToLong, w9XtrueName, w9XchangeDir,
w9XmakeDir, w9XrmDir, w9Xrename, w9XopenFile, w9XcloseFile,
w9XsupportLFN;

FROM QD_File IMPORT pathtype, w9XnothingRequired,
fileOpenRead, fileOpen, fileExists, fileExistsAlt,
fileIsRO, fileSetRW, fileSetRO,
fileErase, fileCreate, fileRename, fileGetFileSize, fileGetFileStamp,
fileIsDirectorySpec, fileClose, fileSupportLFN;

(* ------------------------------------------------------------ *)

CONST
    cr            = CHR(13);
    lf            = CHR(10);
    nl            = cr+lf;
    dquote        = '"';
    singlequote   = "'";
    colon         = ":";
    nullchar      = 0C; (* CHR(0) *)
    blank         = " ";
    tab           = CHR(9);
    pound         = "#";
    semicolon     = ";";
CONST
    dot           = ".";
    dotdot        = "..";
CONST
    sTMP          = "TMP";
    sTEMP         = "TEMP";
    sTMPDIR       = "TMPDIR";
    sTEMPDIR      = "TEMPDIR";
    sNDX1         = "~DELME1~.TMP"; (* very unlikely to exist ! *)
    sNDX2         = "~DELME2~.TMP";
CONST
    extBAT        = ".BAT";
    extSYS        = ".SYS"; (* alas, there's config.sys stupid name ! *)

    extCOM        = ".COM";
    extEXE        = ".EXE";
    extDLL        = ".DLL";
    extOVR        = ".OVR";
    extOVL        = ".OVL";
    extDRV        = ".DRV";
    extZIP        = ".ZIP";
    extARJ        = ".ARJ";
    extLZH        = ".LZH";
    sSkippedExtensions = extCOM+delim+extEXE+delim+
                         extDLL+delim+extOVR+delim+extOVL+delim+extDRV+delim+
                         extZIP+delim+extARJ+delim+extLZH;
CONST
    sREMarkChar   = ";";
    sRemark       = "rem";
    sEcho         = "ECHO";
    sThem         = "THEM"+extBAT; (* will be prefixed with del/copy *)
    sDELcmd       = "DEL";
    sCOPYcmd      = "COPY";
    sUSERcmd      = "PROC";
CONST
    sDOSDELcli    = "DEL"; (* won't like hidden files *)
    sDOSCOPYcli   = "COPY";
    sDDcli        = "DD -k -y -r"; (* really delete including RO *)
    sPCOPYcli     = "PCOPY -p -o"; (* don't recreate path and overwrite RO target *)
CONST
    msgINFO       = "::: ";
    msgOK         = "+++ ";
    msgPB         = "--- ";
    msgDIFF       = "### ";
    msgNADA       = "    ";
    msgWARN       = "*** ";
    msgDEBUG      = "/// ";
CONST
    CHKEVERY      = 256; (* let's call chkEscape every CHKEVERY loop *)
VAR
    DEBUG : BOOLEAN; (* globerk for ease of use *)

(* ------------------------------------------------------------ *)

CONST
    progEXEname   = "FCOMP";
    progTitle     = "Q&D Binary/Text File Compare";
    progVersion   = "v1.3g";
    progCopyright = "by PhG";
    Banner        = progTitle+" "+progVersion+" "+progCopyright;

CONST
    errNone           = 0;
    errHelp           = 1;
    errOption         = 2;
    errParmOverflow   = 3;
    errExpecting      = 4;
    errColon          = 5;
    errInnerParent    = 6;
    errNoParent       = 7;
    errDirJoker       = 8;
    errNetSlash       = 9;
    errBadUnit        = 10;
    errPhantomUnit    = 11;
    errSameSpecs      = 12;
    errStorage        = 13;
    errBadValue       = 14;
    errListMode       = 15;
    errNonsenseBin    = 16;
    errNonsenseText   = 17;
    errNonsenseList   = 18;
    errSpecFileForm   = 19;
    errAborted        = 20;
    errNonsenseTest   = 21;
    errMacLeod        = 22;
    errNotFound       = 23;
    errListOnly       = 24;
    errOpcode         = 25;
    errOpcodeList     = 26;
    errBatchPrefix    = 27;
    errNonsenseRedirection=28;
    errUserCmd        = 29;
    errListJoker      = 30;
    errListNotFound   = 31;

    errCheckPB        = 64;
    errCheckOK        = 96;

    errMismatch       = 128;
    errBothFilesMatch = 255;

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);

    PROCEDURE msg3 (VAR S:ARRAY OF CHAR;S1,S2,S3:ARRAY OF CHAR );
    BEGIN
        Str.Concat(S, S1,S2); Str.Append(S, S3);
    END msg3;

CONST
(*
"12345678901234567890123456789012345678901234567890123456789012345678901234567890"
*)
    helpmsg = Banner+nl+
nl+
"Syntax 1 : "+progEXEname+" <spec1> <spec2> [option]..."+nl+
"Syntax 2 : "+progEXEname+" <-c[c]> <list>"+nl+
nl+
"If specification contains any joker, program assumes it applies to files."+nl+
'Directory specification should have a trailing "\".'+nl+
'<spec2> file part must match one of "$.$", "*.*", "$.*" or "*.$" forms.'+nl+
'Each <list> entry is a line matching "<#|;>< ><digest>< ><path>" format.'+nl+
nl+
"-c[c]   check <list> created with DUPLINES (-cc = create "+sDELcmd+sThem+")"+nl+
"-e      check entry existence if <path> is canonical (-c[c] only)"+nl+
"-m:#    maximum mismatch count (0=unlimited, default is 10)"+nl+
"-z      force comparison (shorter size is taken as reference)"+nl+
"-i      return error code 255 if two existing files are identical"+nl+
"-t      force text lines comparison instead of default binary (-v forced)"+nl+
"-k      case-sensitive (-t required)"+nl+
"-w      take tabs and spaces into account (-t required)"+nl+
"-l[a|l] list <spec1> files with identical matching <spec2> files (-ll = -l -a)"+nl+
"-d[a|d] list <spec1> files with different matching <spec2> files (-dd = -d -a)"+nl+
"-a      list all matching files in both <spec1> and <spec2> (-l or -d required)"+nl+
'-del    prefix files with "'+sDOSDELcli+'" (-l[l] or -d[d] required)'+nl+
'-copy   prefix files with "'+sDOSCOPYcli+'" (-l[l] or -d[d] required)'+nl+
'-kill   prefix files with "'+sDDcli+'" (-l[l] or -d[d] required)'+nl+
'-pcopy  prefix files with "'+sPCOPYcli+'" (-l[l] or -d[d] required)'+nl+
'-u:$    prefix files with user command (-l[l] or -d[d] required)'+nl+
"-bat    create batch file based on prefix operation (-db[a|b], -kb[a|b])"+nl+
"-s      process subdirectories"+nl+
"-v      verbose (slower display, forced with -t)"+nl+
"-n      alternate display (names without paths)"+nl+
"-lfn    disable LFN support even if available"+nl+
"-mem    ignore any Storage.ALLOCATE() error (unwise !)"+nl+
nl+
"a) About -t option (DOS ASCII) : older file is taken as reference ;"+nl+
"   temporary files are created in directory specified by, in that order,"+nl+
"   "+sTMP+", "+sTEMP+", "+sTMPDIR+", "+sTEMPDIR+" environment variables"+nl+
"   (if none of these variables is defined, current directory will be used)."+nl+
"   "+sSkippedExtensions+" files are ignored."+nl+
"b) About -n option : in default mode, listing is terser ;"+nl+
"   in list mode (-l or -d options), path is removed from <spec1> source ;"+nl+
"   this option is ignored with -ll, -dd and -t options."+nl+
"c) To be effective, all prefix commands require that output is redirected."+nl+
"d) -<d|k>b[a|b] = <-del|-kill> <-bat> [-all]."+nl+
"e) About -c[c] option (syntax 2) : any option other than -e is ignored."+nl+
'   An entry <path> is said canonical if matching "[?]\*" pattern.'+nl+
"   -c[c] returns error code 96 if groups contain identical files, else 64."+nl+
"f) Note joker matching includes extension if extension is omitted."+nl
(*
"For mysterious reasons which are beyond Programming Gods and Code Police,"+nl+
"-q option prevents program from working in *any* Windows 3.x/9x DOS box ! :-("+nl
*)
(*%F FIXEDLIBS  *)
+ (* yep ! *)
nl+
"Unfortunately, thanks to a fatal bug in TopSpeed Modula-2 FIO library,"+nl+
"paths are limited to 65 characters (longer ones will NOT be found). :-("+nl
(*%E  *)
; (* yep ! *)

VAR
    S : str1024; (* oversized just in case we' use a LFN *)
BEGIN
    CASE e OF
    | errHelp :
        WrStr(helpmsg);
    | errOption:
        msg3(S, 'Illegal "',einfo,'" option !');
    | errParmOverflow:
        msg3(S, '"',einfo,'" is one parameter too many !');
    | errExpecting:
        msg3(S, "Expecting ",einfo," !");
    | errColon:
        msg3(S, 'Unexpected ":" in "',einfo,'" specification !');
    | errInnerParent:
        msg3(S, 'Illegal ".." in "',einfo,'" specification !');
    | errNoParent:
        msg3(S, 'Unresolvable ".." in "',einfo,'" specification !');
    | errDirJoker:
        msg3(S, 'Illegal "',einfo,'" directory joker(s) !');
    | errNetSlash:
        msg3(S, 'Illegal server "\\" in "',einfo,'" specification !');
    | errBadUnit:
        msg3(S, 'Illegal unit in "',einfo,'" specification !');
    | errPhantomUnit:
        msg3(S, 'Unavailable unit in "',einfo,'" specification !');
    | errSameSpecs:
        S:= "Specifying identical <spec1> and <spec2> is a nonsense !";
    | errStorage:
        msg3(S, 'Storage.ALLOCATE() failure while processing "',einfo,'" specification !');
    | errBadValue:
        msg3(S, 'Illegal "',einfo,'" value !');
    | errListMode:
        S:="-l[l], -d[d] and -c[c] options are mutually exclusive !"; (* //V13F *)
    | errNonsenseBin:
        msg3(S,einfo," option is a nonsense without -t option"," !");
    | errNonsenseText:
        msg3(S,einfo," option is a nonsense with -t option"," !");
    | errNonsenseList:
        msg3(S,"-l[l] and -d[d] options are not compatible with ",einfo," option !");
    | errSpecFileForm:
        msg3(S,einfo,' file form must be "$.$", "*.*", "*.$" or "$.*"', " !");
    | errAborted:
        S:="Aborted by user !";
    | errNonsenseTest:
        msg3(S,einfo," option is a nonsense with -i option"," !");
    | errMacLeod: (* "There can be only one !" *)
        msg3(S,einfo," must match exactly one existing file with -i option"," !");
    | errNotFound:
        msg3(S,'No file matches "',einfo,'" specification !');
    | errListOnly:
        msg3(S,einfo," option requires -l[l] or -d[d] options"," !");
    | errOpcode:
        S:="All prefix command options are mutually exclusive !";
    | errOpcodeList:
        S:="All prefix command options require -l[l] or -d[d] options !";
    | errBatchPrefix:
        S:="-bat option requires a prefix command option !";
    | errNonsenseRedirection:
        S:="-bat option is a nonsense with redirected output !";
    | errUserCmd:
        S:="User command specified with -u:$ option cannot be an empty string !";
    | errListJoker:
        msg3(S,'"',einfo,'" should not contain any joker !');
    | errListNotFound:
        msg3(S,'"',einfo,'" does not exist !');

    | errCheckPB:
        S:="At least one difference found in entries groups !"; (* never displayed *)
    | errCheckOK:
        S:="No difference found in entries groups !"; (* never displayed *)

    | errMismatch:
        S:="Files do not match !"; (* never displayed *)
    | errBothFilesMatch:
        S:="Both files match !"; (* never displayed *)

    ELSE
        S := "How did you get here ?"; (* or legal error not taken into account ;-) *)
    END;
    CASE e OF
    | errNone, errHelp, errMismatch,errBothFilesMatch,errCheckPB,errCheckOK:
        ;
    ELSE
        WrStr(progEXEname+" : "); WrStr(S); WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

(* as usual, buffers must be outside procedure, else program bombs ! *)

CONST
    IObufferSize      = (16 * 512) + FIO.BufferOverhead;
    firstIObufferByte = 1;
    lastIObufferByte  = IObufferSize;
TYPE
    ioBufferType      = ARRAY [firstIObufferByte..lastIObufferByte] OF BYTE; (* for private M2 IO *)
VAR
    ioBuffer1,ioBuffer2 : ioBufferType;
VAR
    ioBufferNdxOld,ioBufferNdxNew:ioBufferType; (* out of proc in order to avoid crash *)

CONST
    dataBufferSize    = (16 * 512) ;
    firstBufferChar   = 1;
    lastBufferChar    = dataBufferSize;
VAR
    sourceBuffer,targetBuffer : ARRAY [firstBufferChar..lastBufferChar] OF CHAR;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE fmtlc (v:LONGCARD;base:CARDINAL;wi:INTEGER;ch,prefix:CHAR) : str80;
VAR
    R : str80;
    ok: BOOLEAN;
    i : CARDINAL;
BEGIN
    Str.CardToStr(v,R,base,ok);
    FOR i:= Str.Length(R)+1 TO ABS(wi) DO
        IF wi < 0 THEN
            Str.Append(R,ch);
        ELSE
            Str.Prepend(R,ch);
        END;
    END;
    IF base=16 THEN Str.Lows(R);END;
    Str.Prepend(R,prefix);
    RETURN R;
END fmtlc;

PROCEDURE fmt (v:CARDINAL;base:CARDINAL;wi:INTEGER;ch,prefix:CHAR) : str80;
BEGIN
    RETURN fmtlc(LONGCARD(v),base,wi,ch,prefix);
END fmt;

PROCEDURE fmtchar (enclose:BOOLEAN;c : BYTE) : str16;
VAR
    S : str16;
    enclosechar:CHAR;
BEGIN
    CASE ORD(c) OF
    | 0..ORD(blank)-1 : S:=".";
    | 255 :             S:=".";
    ELSE
                        Str.Copy(S,CHR(c));
    END;
    IF enclose THEN
        CASE ORD(c) OF
        | ORD(dquote) :
            enclosechar := singlequote;
        ELSE
            enclosechar := dquote;
        END;
        Str.Prepend(S,enclosechar);Str.Append(S,enclosechar);
    END;
    RETURN S;
END fmtchar;

PROCEDURE smaller (a,b:CARDINAL):CARDINAL;
BEGIN
    IF a < b THEN
        RETURN a;
    ELSE
        RETURN b;
    END;
END smaller;

PROCEDURE smallerlc (a,b:LONGCARD):LONGCARD;
BEGIN
    IF a < b THEN
        RETURN a;
    ELSE
        RETURN b;
    END;
END smallerlc;

PROCEDURE nice (useLFN:BOOLEAN;S:pathtype):pathtype;
VAR
    R:pathtype;
BEGIN
    IF useLFN THEN
        Str.Concat(R,dquote,S);
        Str.Append(R,dquote);
    ELSE
        Str.Copy(R,S);
    END;
    RETURN R;
END nice;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

(* current is "\" or "\*\" *)

PROCEDURE buildParent (VAR parent:pathtype; current:pathtype):BOOLEAN;
VAR
    p:CARDINAL;
BEGIN
    IF Str.Match(current,"\") THEN RETURN FALSE; END;
    unfixDirectory(current);
    p:=Str.RCharPos(current,"\");
    Str.Slice(parent,current,0,p+1); (* keep final "\" *)
    RETURN TRUE;
END buildParent;

PROCEDURE doGetCurrent (useLFN:BOOLEAN;drive:SHORTCARD;
                       VAR unit:str2; VAR current:pathtype);
VAR
    rc:CARDINAL;
    longform:pathtype;
BEGIN
    Str.Concat(unit, CHR( ORD("A")-1+ORD(drive) ),colon);

    FIO.GetDir(drive,current); (* \path without u: nor trailing \ except at root *)
    IF current[1] # colon THEN Str.Prepend(current,unit); END; (* safety *)
    IF useLFN THEN
        IF w9XshortToLong(current,rc,longform) THEN (* if error, keep DOS current *)
            Str.Copy(current,longform);
        END;
    END;
    (* LFN function seems to always return "u:\*" form except at root *)
    IF current[1] = colon THEN Str.Delete(current,0,2);END; (* safety *)
    fixDirectory(current);
END doGetCurrent;

(*
    remember dirs can have an extension, and LFNs can have inner dots
    we handle (whether u: or not) :

    .        current
    ..       parent
    .\xxx    current\xxx       F/D
    ..\xxx   parent\xxx        F/D

    xxx\     current\xxx\
    xxx\.    current\xxx\
    xxx      current\xxx       F/D

    \xxx\    \xxx\
    \xxx     \xxx              F/D
*)

PROCEDURE chkfixSpec (VAR base,spec:pathtype;
                     useLFN:BOOLEAN;orgS:pathtype;HDletters:ARRAY OF CHAR):CARDINAL;
VAR
    p,len,rc:CARDINAL;
    drive:SHORTCARD;
    u:CHAR;
    unit:str2;
    current,parent,S:pathtype;
    ok:BOOLEAN;
BEGIN
    Str.Copy(S,orgS);
    IF Str.Pos(S,"\\") # MAX(CARDINAL) THEN RETURN errNetSlash;END;

    (* process u: in S *)

    CASE CharCount(S,colon) OF
    | 0 :
        drive := FIO.GetDrive();
        rc:=errNone;
    | 1 :
        IF Str.CharPos(S,colon) = 1 THEN
            u:=CAP( S[0] );
            CASE u OF
            | "A".."Z" :
                IF verifyString(u,HDletters) THEN
                    drive := SHORTCARD( ORD(u) - ORD("A") +1 );
                    Str.Delete(S,0,2); (* remove u: *)
                    rc:=errNone;
                ELSE
                    rc:=errPhantomUnit;
                END;
            ELSE
                rc:=errBadUnit;
            END;
        ELSE
            rc:=errColon;
        END;
    ELSE
        rc:=errColon;
    END;
    IF rc # errNone THEN RETURN rc; END;

    (* note S no longer has u: *)

    doGetCurrent(useLFN,drive, unit,current); (* "u:" and "\" or "\*\" *)
    ok:=buildParent(parent, current);

    IF same(S,".") THEN Str.Copy(S,current);END;
    IF same(S,"..") THEN
        IF ok THEN Str.Copy(S,parent) ELSE RETURN errNoParent; END;
    END;
    IF Str.Match(S,".\*") THEN Str.Subst(S,".\",current); END;
    IF Str.Match(S,"..\*") THEN
        IF ok THEN Str.Subst(S,"..\",parent) ELSE RETURN errNoParent;END;
    END;
    IF Str.Match(S,"\*")=FALSE THEN Str.Prepend(S,current);END;
    IF Str.Match(S,"*\.") THEN
        (* S[Str.Length(S)-1]:=0C; *)
        len:=Str.Length(S);
        Str.Delete(S,len-1,1);
    END;
    (* we don't want inner or trailing ".." now *)
    IF Str.Pos(S,"..") # MAX(CARDINAL) THEN RETURN errInnerParent;END;

    (* base = "u:\xxx\" and spec = "xxx" *)

    IF Str.Match(S,"*\") THEN
        Str.Concat(base,unit,S);
        Str.Copy(spec,"*.*");
    ELSE
        Str.Prepend(S,unit);
        (* S is u:[\xxx]... without trailing "\" *)
        len:=Str.Length(S);
        p:=Str.RCharPos(S,"\");
        Str.Slice(base,S,0,p+1);
        Str.Slice(spec,S,p+1,len-p);
        IF chkJoker(spec)=FALSE THEN (* if spec has joker(s), assume files *)
            (* spec has no joker : dir or file ? *)
            IF fileIsDirectorySpec(useLFN,S) THEN
                Str.Copy(base,S);
                fixDirectory(base); (* safety *)
                Str.Copy(spec,"*.*");
            END;
        END;
    END;
    IF chkJoker(base) THEN RETURN errDirJoker; END;
    RETURN errNone;
END chkfixSpec;

(* minimalist *)

PROCEDURE chkBothSpecs (b1,s1,b2,s2:pathtype ):BOOLEAN;
VAR
    p1,p2:pathtype;
BEGIN
    Str.Concat(p1,b1,s1);
    Str.Concat(p2,b2,s2);
    UpperCaseAlt(p1); (* keep accents *)
    UpperCaseAlt(p2);
    RETURN NOT( same(p1,p2) );
END chkBothSpecs;

PROCEDURE splitFileExt (VAR f,e:pathtype;S:pathtype):BOOLEAN;
VAR
   p:CARDINAL;
BEGIN
    p:=Str.RCharPos(S,".");
    CASE p OF
    | 0 :
        RETURN FALSE; (* nonsense : extension without file ! *)
    | MAX(CARDINAL) :
        Str.Copy(f,S);
        Str.Copy(e,"");
    ELSE
        Str.Slice(f,S,0,p);
        Str.Copy(e,S);
        Str.Delete(e,0,p+1);
    END;
    RETURN TRUE;
END splitFileExt;

(* we only support "$.$", "*.*", "*.$" and "$.*" for now *)

TYPE
    patType = (patUnsupported,patExtAlone,
              patStarStar,patStarExt,patFileStar,patFileExt);
              (*    "*.*"      "*.$"       "$.*"      "$.$" *)

PROCEDURE getPattern (S:pathtype):patType;
VAR
    f,e:pathtype;
BEGIN
    IF Str.CharPos(S,"?") # MAX(CARDINAL) THEN RETURN patUnsupported; END;
    IF Str.CharPos(S,"*") = MAX(CARDINAL) THEN RETURN patFileExt; END;
    IF same(S,"*.*") THEN RETURN patStarStar; END;
    IF splitFileExt(f,e, S)= FALSE THEN RETURN patExtAlone; END; (* extension without file ! *)
    IF same(f,"*") THEN
        IF Str.CharPos(e,"*") = MAX(CARDINAL) THEN
            RETURN patStarExt;
        ELSE
            RETURN patUnsupported;
        END;
    END;
    IF same(e,"*") THEN
        IF Str.CharPos(f,"*") = MAX(CARDINAL) THEN
            RETURN patFileStar;
        ELSE
            RETURN patUnsupported;
        END;
    END;
    IF Str.CharPos(f,"*") # MAX(CARDINAL) THEN RETURN patUnsupported; END;
    IF Str.CharPos(e,"*") # MAX(CARDINAL) THEN RETURN patUnsupported; END;
    RETURN patFileExt;
END getPattern;

TYPE
    listmodeType = (lstDefault,lstIdentical,lstDifferent,lstCheck);

PROCEDURE chkList (VAR listmode:listmodeType;wanted:listmodeType ):BOOLEAN;
VAR
    ok:BOOLEAN;
BEGIN
    ok:=TRUE;
    CASE listmode OF
    | lstDefault :
        listmode := wanted;
    ELSE
        IF listmode # wanted THEN ok:=FALSE; END;
    END;
    RETURN ok;
END chkList;

(* we don't check for TMP validity *)

PROCEDURE buildWorkDir (useLFN,forcecurrent:BOOLEAN; VAR workdir:pathtype);
CONST
    numvars = 4;
VAR
    i:CARDINAL;
    D:str128;
    drive:SHORTCARD;
    unit:str2;
    current:pathtype;
BEGIN
    IF forcecurrent THEN
        i:=numvars+1;
    ELSE
        i:=1;
        LOOP
            CASE i OF
            | 1: D:=sTMP;
            | 2: D:=sTEMP;
            | 3: D:=sTMPDIR;
            | 4: D:=sTEMPDIR;
            END;
            Lib.EnvironmentFind(D,workdir);
            IF same(workdir,"")=FALSE THEN EXIT; END;
            INC(i);
            IF i > numvars THEN EXIT; END;
        END;
    END;
    IF i > numvars THEN
        drive := FIO.GetDrive();
        doGetCurrent(useLFN,drive, unit,current); (* "u:" and "\" or "\*\" *)
        Str.Concat(workdir,unit,current);
    END;
    fixDirectory(workdir);
END buildWorkDir;

TYPE
    opcodeType = (opDefault,opDel,opCopy,opDD,opPCOPY,opUser);

PROCEDURE chkOp (VAR opcode:opcodeType;wanted:opcodeType ):BOOLEAN;
VAR
    ok:BOOLEAN;
BEGIN
    ok:=TRUE;
    CASE opcode OF
    | opDefault :
        opcode := wanted;
    ELSE
        IF opcode # wanted THEN ok:=FALSE; END;
    END;
    RETURN ok;
END chkOp;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

(* dir/file list assume :
   a) dirs are stored first
   b) dircont won't be more than 65535
   c) list will contain at least two dirs
*)

CONST
    firstindex = 1;
    FULLPATH   = firstindex-1; (* 0 = dir *)
TYPE
    pEntry = POINTER TO entryType;
    entryType = RECORD
        next      : pEntry;
        specindex : SHORTCARD; (* 1=spec1, 2=spec2 *)
        ID        : LONGCARD;  (* entry unique ID *)
        baseindex : CARDINAL;  (* FULLPATH for dirs, [1.. is ID of base dir for files *)
        slen      : SHORTCARD;
        str       : CHAR;
    END;

PROCEDURE initList (VAR anchor : pEntry );
BEGIN
    anchor := NIL;
END initList;

PROCEDURE freeList (anchor : pEntry);
VAR
    needed : CARDINAL;
    p      : pEntry;
BEGIN
    (* p:=anchor; *)
    WHILE anchor # NIL DO
        needed := SIZE(entryType) - SIZE(anchor^.str) + CARDINAL(anchor^.slen);
        p := anchor^.next;
        DEALLOCATE(anchor,needed);
        anchor:=p;
    END
END freeList;

PROCEDURE buildNewPtr (VAR anchor,p:pEntry; len:CARDINAL):BOOLEAN;
VAR
    needed : CARDINAL;
BEGIN
    needed := SIZE(entryType) - SIZE(p^.str) + len;
    IF Available(needed)=FALSE THEN RETURN FALSE; END;
    IF anchor = NIL THEN
        ALLOCATE(anchor,needed);
        p:=anchor;
    ELSE
        p:=anchor;
        WHILE p^.next # NIL DO
            p:=p^.next;
        END;
        ALLOCATE(p^.next,needed);
        p:=p^.next;
    END;
    p^.next := NIL;
    RETURN TRUE;
END buildNewPtr;

(* ------------------------------------------------------------ *)

(* assume p is valid *)

PROCEDURE getStr (VAR S : pathtype; p:pEntry);
VAR
    len:CARDINAL;
BEGIN
    len := CARDINAL(p^.slen);
    Lib.FastMove( ADR(p^.str),ADR(S),len);
    S[len] := nullchar; (* REQUIRED safety ! *)
END getStr;

PROCEDURE findMatchID (VAR p : pEntry;
                      anchor:pEntry;wanted:CARDINAL;IDwanted:LONGCARD):BOOLEAN;
BEGIN
    p:=anchor;
    WHILE anchor # NIL DO
        IF p^.specindex = SHORTCARD(wanted) THEN
            IF p^.ID = IDwanted THEN RETURN TRUE; END;
        END;
        p:=anchor^.next;
        anchor:=p;
    END;
    p:=NIL; (* added safety *)
    RETURN FALSE;
END findMatchID;

(* pathtypes have been UpperCaseAlt-ed : d+f+e = source *)
(* we don't use hash because we have stored paths/files as they were *)

PROCEDURE findMatchName (VAR p:pEntry; VAR e2 : pathtype;
                        anchor,rootanchor:pEntry;wanted:CARDINAL;patform:patType;
                        fpat,epat,d,f,e:pathtype):BOOLEAN;
VAR
    ok,idnames:BOOLEAN;
    pdir:pEntry;
    file,d2,f2:pathtype;
BEGIN
    p:=anchor;
    WHILE anchor # NIL DO
        ok:=(p^.baseindex # FULLPATH);
        ok:=(ok AND (p^.specindex = SHORTCARD(wanted) ) );
        IF ok THEN
            getStr(file,p);
            ok:=splitFileExt(f2,e2, file);
            UpperCaseAlt(f2);
            UpperCaseAlt(e2);
            idnames:= same(f,f2) AND same(e,e2);
IF DEBUG THEN
WrStr(msgDEBUG+"f2+e2     : ");WrStr(nice(TRUE,f2));WrStr(" + ");
WrStr(nice(TRUE,e2));WrLn;
END;
            ok:=FALSE;
            CASE patform OF
            | patStarStar: ok:= idnames; (* don't waste time with meaningless comparisons *)
            | patFileStar: ok:= same(fpat,f2);
            | patStarExt:  ok:= same(f   ,f2) AND same(epat,e2);
            | patFileExt:  ok:= TRUE;    (* we specified a unique filename *)
            END;
IF DEBUG THEN
WrStr(msgDEBUG+"pattern and file ");
IF NOT(ok) THEN
WrStr("DO NOT ");
END;
WrStr("match");WrLn;
END;
            IF ok THEN
                IF idnames THEN (* same filenames, check directories *)
                    ok:=findMatchID(pdir, rootanchor,wanted,LONGCARD(p^.baseindex) ); (* assume always TRUE *)
                    getStr(d2,pdir);
                    UpperCaseAlt(d2);

                    ok:=NOT( same(d,d2) ); (* avoid comparing a file with itself ! *)
                END;
            END;
            IF ok THEN RETURN ok; END; (* TRUE ! *)
            (* try another entry *)
        END;
        p:=anchor^.next;
        anchor:=p;
    END;
    p:=NIL; (* added safety *)
    RETURN FALSE;
END findMatchName;

(* ------------------------------------------------------------ *)

PROCEDURE dmpList (anchor:pEntry;wanted:CARDINAL;useLFN:BOOLEAN;showdirs:BOOLEAN);
VAR
    p,pdir,rootanchor : pEntry;
    S,base,R: pathtype;
    len,baseindex:CARDINAL;
    found,ok:BOOLEAN;
BEGIN
    WrStr("<spec> baseindex        ID");WrLn;
    rootanchor:=anchor;
    p:=anchor;
    WHILE anchor # NIL DO
        ok:=(p^.baseindex # FULLPATH);
        IF showdirs THEN ok:=NOT(ok);END;
        ok:=(ok AND (p^.specindex = SHORTCARD(wanted) ) );
        IF ok THEN
            getStr(S,p);
            IF p^.baseindex = FULLPATH THEN
                Str.Copy(R,S);
            ELSE
                found:=findMatchID(pdir, rootanchor,wanted,LONGCARD(p^.baseindex) ); (* assume always TRUE *)
                getStr(base,pdir);
                Str.Concat(R,base,S);
            END;
            IO.WrShtCard(p^.specindex,6);
            IO.WrCard(p^.baseindex,10);
            IO.WrLngCard(p^.ID,10);
            WrStr("  ");
            WrStr(nice(useLFN,R));
            WrLn;
        END;
        p:=anchor^.next;
        anchor:=p;
    END;
END dmpList;

(* ------------------------------------------------------------ *)

PROCEDURE isReservedEntry (S:ARRAY OF CHAR) : BOOLEAN;
BEGIN
    IF same(S,dot) THEN RETURN TRUE; END;
    RETURN same(S,dotdot);
END isReservedEntry;

PROCEDURE buildDirList (VAR currID:LONGCARD; VAR count:LONGCARD;VAR anchor:pEntry;
                       useLFN,recurse:BOOLEAN;
                       whatspec:CARDINAL;base:pathtype):CARDINAL;
VAR
    p: pEntry;
    len : CARDINAL;
    root,rootspec,entryname,newroot : pathtype;
    entry : FIO.DirEntry;
    found : BOOLEAN;
    w9Xentry : findDataRecordType;
    unicodeconversion:unicodeConversionFlagType;
    dosattr:FIO.FileAttr;
    w9Xhandle,errcode:CARDINAL;
    rc : CARDINAL;
    ok:BOOLEAN;
BEGIN
    Str.Copy(root,base);
    fixDirectory(root); (* safety *)

    (* assume base exists ! *)

    len:=Str.Length(root);
    IF buildNewPtr(anchor,p,len)=FALSE THEN RETURN errStorage; END;
    INC(count);
    INC(currID);
    p^.specindex := SHORTCARD (whatspec);
    p^.ID        := currID;
    p^.baseindex := FULLPATH; (* 0 ! *)
    p^.slen      := SHORTCARD(len);
    Lib.FastMove ( ADR(root),ADR(p^.str),len );

    IF NOT(recurse) THEN RETURN errNone; END;

    Str.Concat(rootspec,root,"*.*");
    IF useLFN THEN
        found := w9XfindFirst (rootspec,SHORTCARD(everything),SHORTCARD(w9XnothingRequired),
                              unicodeconversion,w9Xentry,w9Xhandle,errcode);
    ELSE
        found := FIO.ReadFirstEntry(rootspec,everything,entry);
    END;
    WHILE found DO
        IF useLFN THEN
            Str.Copy(entryname,w9Xentry.fullfilename);
        ELSE
            Str.Copy(entryname,entry.Name);
        END;
        IF isReservedEntry (entryname) = FALSE THEN (* skip "." AND ".." *)
            IF useLFN THEN
                dosattr:=FIO.FileAttr(w9Xentry.attr AND 0FFH);
            ELSE
                dosattr:=entry.attr;
            END;
            IF (aD IN dosattr) THEN
                Str.Concat(newroot,root,entryname); (* u:\xx\ + xxx *)
                fixDirectory(newroot);
                rc:= buildDirList (currID,count,anchor,
                                  useLFN,recurse,whatspec,newroot);
                IF rc # errNone THEN
                    IF useLFN THEN ok:=w9XfindClose(w9Xhandle,errcode); END;
                    RETURN rc;
                END;
            END;
        END;
        IF useLFN THEN
            found :=w9XfindNext(w9Xhandle, unicodeconversion,w9Xentry,errcode);
        ELSE
            found :=FIO.ReadNextEntry(entry);
        END;
    END;
    IF useLFN THEN ok:=w9XfindClose(w9Xhandle,errcode); END;
    RETURN errNone;
END buildDirList;

PROCEDURE buildFileList (VAR currID:LONGCARD; VAR count:LONGCARD;
                        anchor:pEntry;
                        useLFN:BOOLEAN;
                        wanted:CARDINAL;spec:pathtype):CARDINAL;
VAR
    pdir,pp:pEntry;
    base,rootspec, entryname:pathtype;
    len : CARDINAL;
    ok,found:BOOLEAN;
    w9Xentry : findDataRecordType;
    unicodeconversion:unicodeConversionFlagType;
    dosattr:FIO.FileAttr;
    w9Xhandle,errcode:CARDINAL;
    entry : FIO.DirEntry;
BEGIN
    count:=0;
    pdir:=anchor;
    WHILE anchor # NIL DO
        ok:=(pdir^.baseindex=FULLPATH);
        ok:=(ok AND ( pdir^.specindex = SHORTCARD(wanted) ) );
        IF ok THEN
            getStr(base,pdir);
            Str.Concat(rootspec,base,spec);
            IF useLFN THEN
                found := w9XfindFirst (rootspec,SHORTCARD(everything),SHORTCARD(w9XnothingRequired),
                                      unicodeconversion,w9Xentry,w9Xhandle,errcode);
            ELSE
                found := FIO.ReadFirstEntry(rootspec,everything,entry);
            END;
            WHILE found DO
                IF useLFN THEN
                    Str.Copy(entryname,w9Xentry.fullfilename);
                ELSE
                    Str.Copy(entryname,entry.Name);
                END;
                IF isReservedEntry (entryname) = FALSE THEN (* skip "." AND ".." *)
                    IF useLFN THEN
                        dosattr:=FIO.FileAttr(w9Xentry.attr AND 0FFH);
                    ELSE
                        dosattr:=entry.attr;
                    END;
                    IF NOT (aD IN dosattr) THEN
                        (* if file has no extension, add it as a marker *)
                        IF Str.RCharPos(entryname,".")=MAX(CARDINAL) THEN
                            Str.Append(entryname,".");
                        END;
                        len:=Str.Length(entryname);
                        IF buildNewPtr(anchor,pp,len)=FALSE THEN
                            IF useLFN THEN ok:=w9XfindClose(w9Xhandle,errcode); END;
                            RETURN errStorage;
                        END;
                        INC(count);
                        INC(currID);
                        pp^.specindex := SHORTCARD (wanted);
                        pp^.ID        := currID;
                        pp^.baseindex := CARDINAL(pdir^.ID);
                        pp^.slen      := SHORTCARD(len);
                        Lib.FastMove ( ADR(entryname),ADR(pp^.str),len );
                    END;
                END;
                IF useLFN THEN
                    found :=w9XfindNext(w9Xhandle, unicodeconversion,w9Xentry,errcode);
                ELSE
                    found :=FIO.ReadNextEntry(entry);
                END;
            END;
            IF useLFN THEN ok:=w9XfindClose(w9Xhandle,errcode); END;
        END;
        pdir:=anchor^.next;
        anchor:=pdir;
    END;
    RETURN errNone;
END buildFileList;

(* ------------------------------------------------------------ *)

PROCEDURE nopath (VAR R:pathtype);
VAR
    p:CARDINAL;
BEGIN
    p:=Str.RCharPos(R,"\");
    IF p # MAX(CARDINAL) THEN Str.Delete(R,0,p+1); END;
END nopath;

TYPE
    cmpType = (binfile,textfile);

PROCEDURE headerComp (useLFN:BOOLEAN;
                     cmp:cmpType;source,target:pathtype);
VAR
    S:str1024; (* two LFNs can be very long *)
BEGIN
    CASE cmp OF
    | binfile:  Str.Copy(S,msgINFO+"Binary");
    | textfile: Str.Copy(S,msgINFO+"Line");
    END;
    Str.Append(S," differences between ");
    Str.Append(S, nice(useLFN,source) );
    Str.Append(S," and ");
    Str.Append(S, nice(useLFN,target) );
    WrStr(S);WrLn;
END headerComp;

TYPE
    resultType = (rcSame,rcSameForced,rcSize,rcMismatch,rcAborted);

PROCEDURE showDiff (addr : LONGCARD;ref,now : BYTE);
VAR
    S:str128;
BEGIN
    Str.Concat(S, fmtlc(addr,16,8,"0","$"), "  :  " );
    Str.Append(S, fmt(CARDINAL(ref),16,2,"0","$"));
    Str.Append(S," --> ");
    Str.Append(S, fmt(CARDINAL(now),16,2,"0","$"));
    Str.Append(S,"  :  ");
    Str.Append(S,fmtchar(TRUE,ref));
    Str.Append(S," --> ");
    Str.Append(S,fmtchar(TRUE,now));
    Str.Append(S,"  :  ");
    Str.Append(S, fmt(CARDINAL(ref),10,3," ",""));
    Str.Append(S," --> ");
    Str.Append(S, fmt(CARDINAL(now),10,3," ",""));
    WrStr(msgNADA);WrStr(S);WrLn;
END showDiff;

(* assume src and target exist *)

PROCEDURE doBinCompare (useLFN,verbose,candy,ignorefsize:BOOLEAN;
                       maxmismatches:LONGCARD;
                       source,target:pathtype):resultType;
CONST
    msgWorking = "Processing... ";
VAR
    hnd1,hnd2:FIO.File;
    fsize1,fsize2,fpos,mismatches:LONGCARD;
    i,got1,got2,got:CARDINAL;
    alcatraz,done,samesize:BOOLEAN;
    ref,now:BYTE;
    S:str128;
    result:resultType;
    lastfpos:LONGCARD;
    chkrounds:CARDINAL;
BEGIN
    hnd1:= fileOpenRead (useLFN, source);
    FIO.AssignBuffer(hnd1,ioBuffer1);
    hnd2:= fileOpenRead (useLFN, target);
    FIO.AssignBuffer(hnd2,ioBuffer2);

    fsize1:=FIO.Size(hnd1);
    fsize2:=FIO.Size(hnd2);
    samesize := (fsize1 = fsize2);
    IF NOT (samesize) THEN
        IF NOT(ignorefsize) THEN
            FIO.Close(hnd2);
            FIO.Close(hnd1);
            RETURN rcSize;
        END;
    END;
    lastfpos   :=smallerlc(fsize1,fsize2);
    fpos       :=0;
    mismatches :=0;
    done       :=FALSE;
    alcatraz   :=FALSE;
    chkrounds  :=0;

    IF candy THEN video(msgWorking,TRUE);completed (completedInit, lastfpos);END;
    LOOP
        got1:=FIO.RdBin(hnd1,sourceBuffer, dataBufferSize );
        got2:=FIO.RdBin(hnd2,targetBuffer, dataBufferSize );
        got := smaller(got1,got2);
        IF got = 0 THEN EXIT; END; (* size was a multiple of dataBufferSize *)

        i:=firstBufferChar-1; (* 1-1=0 *)
        LOOP
            INC(i); IF i > got THEN EXIT; END;
            IF candy THEN completed(completedShow,fpos); END;
            ref := sourceBuffer[i];
            now := targetBuffer[i];
            IF ref # now THEN
                INC(mismatches);
                done:=( mismatches > maxmismatches ); IF done THEN EXIT; END;
                IF verbose THEN
                    IF candy THEN
                        completed(completedEnd,0); video(msgWorking,FALSE);
                    END;
                    IF mismatches=1 THEN WrLn; END;
                    showDiff(fpos+LONGCARD(i-firstBufferChar),ref,now);
                    IF candy THEN
                        video(msgWorking,TRUE);completed(completedInit, lastfpos);
                    END;
                END;
            END;
            INC(chkrounds);
            IF (chkrounds MOD CHKEVERY) = 0 THEN alcatraz:=ChkEscape(); END;
            IF alcatraz THEN EXIT; END;
        END;
        IF done THEN EXIT; END;
        IF got # dataBufferSize THEN EXIT; END; (* < *)
        IF alcatraz THEN EXIT; END;
        INC(fpos, LONGCARD(got));
    END;
    IF candy THEN completed(completedEnd,0); video(msgWorking,FALSE);END;
    FIO.Close(hnd2);
    FIO.Close(hnd1);

    IF alcatraz THEN RETURN rcAborted;END;

    IF mismatches=0 THEN
        IF samesize THEN
            result:=rcSame;
        ELSE
            result:=rcSameForced;
        END;
    ELSE
        IF verbose THEN
            IF done THEN
                Str.Concat(S,"More than ", fmtlc(maxmismatches,10,0,"","") );
                DEC(mismatches); (* we were already at +1 *)
            ELSE
                Str.Copy(S, fmtlc(mismatches,10,0,"","") );
            END;
            Str.Append(S," mismatch");
            IF mismatches > 1 THEN Str.Append(S,"es"); END;
            Str.Append(S," found.");
            WrLn;
            WrStr(msgDIFF);WrStr(S);WrLn;
        END;
        result:=rcMismatch;
    END;
    RETURN result;
END doBinCompare;

(* ------------------------------------------------------------ *)

(*
   buildFileList() takes care of adding a trailing dot to file without extension
   e1 and e2 are already UpperCaseAlt-ed but they don't have dot
*)

PROCEDURE validExtensions (skipem:ARRAY OF CHAR;e,e2:pathtype):BOOLEAN;
VAR
    ext:str16; (* oversized *)
    n,pb:CARDINAL;
BEGIN
    pb:=0;
    n:=0;
    LOOP
        isoleItemS(ext, skipem,delim,n);
        Str.Subst(ext,".",""); (* remove leading dot *)
        IF same(ext,"") THEN EXIT; END;
        IF same(ext,e ) THEN INC(pb);END;
        IF same(ext,e2) THEN INC(pb);END;
        INC(n);
    END;
    RETURN (pb=0);
END validExtensions;

PROCEDURE getStamp (useLFN:BOOLEAN; S:pathtype):LONGREAL;
VAR
    hnd:FIO.File;
    stamp:FIO.FileStamp;
    ok:BOOLEAN;
    y,m,d,hh,mm,ss:SHORTCARD;
    seconds,days:LONGCARD;
BEGIN
    hnd:=fileOpenRead(useLFN,S);
    ok:=FIO.GetFileStamp(hnd,stamp);
    FIO.Close(hnd);
    y  := stamp.Year ;
    m  := stamp.Month;
    d  := stamp.Day  ;
    hh := stamp.Hour ;
    mm := stamp.Min  ;
    ss := stamp.Sec  ;

    (* seconds ok, days approximate : months of 32 days, years of 367 days *)

    seconds := LONGCARD(ss) + 60*LONGCARD(mm) + 60*60*LONGCARD(hh); (*  90060 *)
    days    := LONGCARD(d ) + 32*LONGCARD(m ) + 367  *LONGCARD(y ); (* 7xxxxx *)

    RETURN LONGREAL(days) + LONGREAL(seconds) / 100000.0;
END getStamp;

PROCEDURE olderNewer (useLFN:BOOLEAN; VAR source,target:pathtype);
VAR
    tmp:pathtype;
BEGIN
    IF getStamp(useLFN,target) < getStamp(useLFN,source) THEN
        Str.Copy(tmp,source);
        Str.Copy(source,target);
        Str.Copy(target,tmp);
    END;
END olderNewer;

CONST
    MAXHASH = MAX(CARDINAL)-1;

PROCEDURE doTxtCompare (useLFN,verbose,candy,keepcase,keepblanks:BOOLEAN;
                       source,target:pathtype):resultType;
TYPE
    lineType = RECORD
        linenum : LONGCARD;
        fpos    : LONGCARD;
        hashID  : CARDINAL;
    END;
CONST
    msgIndexing  = "Please wait while indexing ";
    msgWorking   = "Processing... ";
    tagDeleted   = " -- ";
    tagAdded     = " ++ ";
    tagCommon    = " == "; (* never used *)
    tagInfo      = msgINFO;
    winum        = 6;
VAR
    result:resultType;
    F,FNDX,workdir:pathtype;
    msg:str1024;
    hin,hndx,hndxold,hndxnew,hold,hnew:FIO.File;
    wanted,got,i : CARDINAL;
    mismatches,matches,currline,fpos:LONGCARD;
    S,SS:str4096;
    entry,oldentry,newentry:lineType;
    alcatraz:BOOLEAN;
    lastfpos:LONGCARD;
    chkrounds:CARDINAL;
BEGIN
    buildWorkDir(useLFN,FALSE,workdir);

    FOR i:=1 TO 2 DO
        CASE i OF
        | 1: Str.Copy(F,source); Str.Copy(FNDX,sNDX1);
        | 2: Str.Copy(F,target); Str.Copy(FNDX,sNDX2);
        END;
        Str.Concat(msg,msgIndexing,F);
        video(msg,TRUE);

        Str.Prepend(FNDX,workdir);
        IF fileExists(useLFN,FNDX) THEN
            IF fileIsRO(useLFN,FNDX) THEN fileSetRW(useLFN,FNDX);END;
        END;

        hin:=fileOpenRead(useLFN,F);
        FIO.AssignBuffer(hin,ioBuffer1);
        hndx:=FIO.Create(FNDX);           (* CHKME LFN function here *)
        FIO.AssignBuffer(hndx,ioBuffer2);

        currline:=0;
        LOOP
            INC(currline);
            fpos:=FIO.GetPos(hin);
            FIO.RdStr(hin,S);
            IF FIO.EOF THEN EXIT; END;
            IF NOT(keepcase) THEN LowerCaseAlt(S);END; (* keep accents *)
            IF NOT(keepblanks) THEN
                ReplaceChar(S,blank,"");
                ReplaceChar(S,tab,"");
            END;
            IF S[0] # CHR(0) THEN
                entry.linenum := currline;
                entry.fpos    := fpos;
                entry.hashID  := Lib.HashString(S,MAXHASH);
                FIO.WrBin(hndx,entry,SIZE(entry));
            END;
        END;

        FIO.Close(hndx);
        FIO.Close(hin);
        video(msg,FALSE);
    END;

    (* if hash1=hash2, we have to retrieve and compare both strings *)

    (* list lines deleted from source : hashold without match in newndx -- yes, it's not exactly the same *)

    WrStr(tagInfo+"Line(s) deleted from ");WrStr(source);WrLn;
    WrLn;

    Str.Concat(FNDX,workdir,sNDX1);
    hndxold:=fileOpenRead(useLFN,FNDX);
    FIO.AssignBuffer(hndxold,ioBufferNdxOld);

    Str.Concat(FNDX,workdir,sNDX2);
    hndxnew:=fileOpenRead(useLFN,FNDX);
    FIO.AssignBuffer(hndxnew,ioBufferNdxNew);

    hold:=fileOpenRead(useLFN,source);
    FIO.AssignBuffer(hold,ioBuffer1);
    hnew:=fileOpenRead(useLFN,target);
    FIO.AssignBuffer(hnew,ioBuffer2);

    mismatches:=0;
    alcatraz:=FALSE;
    chkrounds:=0;
    wanted:=SIZE(lineType);

    lastfpos:=FIO.Size(hndxold);
    IF candy THEN video(msgWorking,TRUE);completed (completedInit, lastfpos);END;
    LOOP
        IF candy THEN completed(completedShow,FIO.GetPos(hndxold)); END;
        IF alcatraz THEN EXIT; END;
        got:=FIO.RdBin(hndxold,oldentry,wanted);
        IF got # wanted THEN EXIT; END;
        FIO.Seek(hndxnew,0);
        matches:=0;
        LOOP
            got:=FIO.RdBin(hndxnew,newentry,wanted);
            IF got # wanted THEN EXIT; END;
            IF oldentry.hashID = newentry.hashID THEN
                FIO.Seek(hold,oldentry.fpos);
                FIO.RdStr(hold,S);
                FIO.Seek(hnew,newentry.fpos);
                FIO.RdStr(hnew,SS);

                IF NOT(keepcase) THEN LowerCaseAlt(S);END;
                IF NOT(keepblanks) THEN
                    ReplaceChar(S,blank,"");
                    ReplaceChar(S,tab,"");
                END;

                IF NOT(keepcase) THEN LowerCaseAlt(SS);END;
                IF NOT(keepblanks) THEN
                    ReplaceChar(SS,blank,"");
                    ReplaceChar(SS,tab,"");
                END;

                IF same(S,SS) THEN INC(matches);END;
            END;
        END;
        IF matches=0 THEN
            INC(mismatches);
            FIO.Seek(hold,oldentry.fpos);
            FIO.RdStr(hold,S);
            IF candy THEN
                completed(completedEnd,0);video(msgWorking,FALSE);
            END;
            WrStr( fmtlc(oldentry.linenum,10,winum," ","") );
            WrStr( tagDeleted );
            WrStr(S);WrLn;
            IF candy THEN
                video(msgWorking,TRUE);completed (completedInit, lastfpos);
            END;
        END;

        INC(chkrounds);
        IF (chkrounds MOD CHKEVERY) = 0 THEN alcatraz:=ChkEscape(); END;
        IF alcatraz THEN EXIT; END;
    END;
    IF candy THEN completed(completedEnd,0);video(msgWorking,FALSE);END;

    (* list lines added to source : hashnew without match in oldndx -- yes, it's not exactly the same *)

    WrLn;
    WrStr(tagInfo+"Line(s) added to ");WrStr(target);WrLn;
    WrLn;

    FIO.Seek(hndxnew,0);
    (* don't reinit alcatraz *)

    lastfpos:=FIO.Size(hndxnew);
    IF candy THEN video(msgWorking,TRUE);completed (completedInit, lastfpos);END;
    LOOP
        IF candy THEN completed(completedShow,FIO.GetPos(hndxnew)); END;
        IF alcatraz THEN EXIT; END;
        got:=FIO.RdBin(hndxnew,newentry,wanted);
        IF got # wanted THEN EXIT; END;
        FIO.Seek(hndxold,0);
        matches:=0;
        LOOP
            got:=FIO.RdBin(hndxold,oldentry,wanted);
            IF got # wanted THEN EXIT; END;
            IF newentry.hashID = oldentry.hashID THEN
                FIO.Seek(hnew,newentry.fpos);
                FIO.RdStr(hnew,S);
                FIO.Seek(hold,oldentry.fpos);
                FIO.RdStr(hold,SS);

                IF NOT(keepcase) THEN LowerCaseAlt(S);END;
                IF NOT(keepblanks) THEN
                    ReplaceChar(S,blank,"");
                    ReplaceChar(S,tab,"");
                END;

                IF NOT(keepcase) THEN LowerCaseAlt(SS);END;
                IF NOT(keepblanks) THEN
                    ReplaceChar(SS,blank,"");
                    ReplaceChar(SS,tab,"");
                END;

                IF same(S,SS) THEN INC(matches);END;
            END;
        END;
        IF matches=0 THEN
            INC(mismatches);
            FIO.Seek(hnew,newentry.fpos);
            FIO.RdStr(hnew,S);
            IF candy THEN
                completed(completedEnd,0);video(msgWorking,FALSE);
            END;
            WrStr( fmtlc(newentry.linenum,10,winum," ","") );
            WrStr( tagAdded );
            WrStr(S);WrLn;
            IF candy THEN
                video(msgWorking,TRUE);completed (completedInit, lastfpos);
            END;
        END;

        INC(chkrounds);
        IF (chkrounds MOD CHKEVERY) = 0 THEN alcatraz:=ChkEscape(); END;
        IF alcatraz THEN EXIT; END;
    END;
    IF candy THEN completed(completedEnd,0);video(msgWorking,FALSE);END;

    FIO.Close(hnew);
    FIO.Close(hold);
    FIO.Close(hndxnew);
    FIO.Close(hndxold);

    Str.Concat(FNDX,workdir,sNDX2);
    fileErase(useLFN,FNDX);
    Str.Concat(FNDX,workdir,sNDX1);
    fileErase(useLFN,FNDX);

    IF alcatraz THEN RETURN rcAborted; END;
    IF mismatches = 0 THEN
        result:=rcSame;
    ELSE
        IF verbose THEN
            Str.Copy(S, fmtlc(mismatches,10,0,"","") );
            Str.Append(S," line mismatch");
            IF mismatches > 1 THEN Str.Append(S,"es"); END;
            Str.Append(S," found.");
            WrLn;
            WrStr(msgDIFF);WrStr(S);WrLn;
        END;
        result:=rcMismatch;
    END;
    RETURN result;
END doTxtCompare;

(* ------------------------------------------------------------ *)

CONST
    msgSAME       = msgOK+  "Files match."+nl;
    msgSAMEFORCED = msgOK+  "Files match (comparison limited to smaller filesize)."+nl;
    msgSIZE       = msgPB+  "Sizes are not identical."+nl;
    msgNOTHINGTODO= msgWARN+"Nothing to do."+nl;           (* was msgINFO *)
    msgSAMELINES  = msgOK+  "Lines match."+nl;
    msgNOMATCH    = msgPB+  "No match for ";               (* was msgINFO *)
    msgNOTID      = msgPB+  "Mismatch(es) found."+nl;

PROCEDURE procComp (anchor:pEntry; cmp:cmpType;
                   useLFN,
                   altdisplay,candy,keepcase,keepblanks,
                   ignorefsize,testsame:BOOLEAN;
                   maxmismatches:LONGCARD; wanted,icomp:CARDINAL;
                   patref,patcomp:pathtype):CARDINAL;
VAR
    patform:patType;
    i:CARDINAL;
    matched,found,ok:BOOLEAN;
    rootanchor,p,pdir,pmatch:pEntry;
    file,d,source,target,f,e,fpat,epat,base:pathtype;
    mismatch,nothingtodo:BOOLEAN;
    rcode:resultType;
    rc:CARDINAL;
    e2:pathtype; (* target ext uppercasealt-ed *)
    S:str1024; (* oversized because of LFN *)
    Ssrc,Stgt:pathtype;
    verbose:BOOLEAN;
BEGIN
    mismatch:=FALSE;
    nothingtodo:=TRUE;

    patform:=getPattern(patcomp); (* already checked for errors *)
    ok:=splitFileExt(fpat,epat, patcomp);
    UpperCaseAlt(fpat);
    UpperCaseAlt(epat);
IF DEBUG THEN
WrStr(msgDEBUG+"patform   : ");
CASE patform OF
| patStarStar: WrStr("*.*");
| patFileExt : WrStr("$.$");
| patFileStar: WrStr("$.*");
| patStarExt : WrStr("*.$");
ELSE
               WrStr("???"); (* cannot happen *)
END;
WrLn;
WrStr(msgDEBUG+"fpat+epat : ");WrStr(nice(TRUE,fpat));WrStr(" + ");
WrStr(nice(TRUE,epat));WrLn;
END;
    rootanchor:=anchor;
    p:=anchor;
    WHILE anchor # NIL DO
        ok:=(p^.baseindex # FULLPATH);
        ok:=(ok AND (p^.specindex = SHORTCARD(wanted) ) );
        IF ok THEN
            getStr(file,p);
            ok:=findMatchID(pdir, rootanchor,wanted,LONGCARD(p^.baseindex) ); (* assume always TRUE *)
            getStr(d,pdir);
            Str.Concat(source,d,file);

            ok:=splitFileExt(f,e, file);
            UpperCaseAlt(d);
            UpperCaseAlt(f);
            UpperCaseAlt(e);
IF DEBUG THEN
WrStr(msgDEBUG+"d+f+e     : ");WrStr(nice(TRUE,d));WrStr(" + ");
WrStr(nice(TRUE,f));WrStr(" + ");
WrStr(nice(TRUE,e));WrLn;
END;
            found:=findMatchName (pmatch, e2,
                                 rootanchor,rootanchor,icomp,patform,
                                 fpat,epat,d,f,e);
            matched:=found;
            WHILE found DO
                getStr(file,pmatch);
                ok:=findMatchID(pdir, rootanchor,icomp,LONGCARD(pmatch^.baseindex) ); (* assume always TRUE *)
                getStr(base,pdir);
                Str.Concat(target,base,file);

                CASE cmp OF
                | binfile:
                    IF NOT(altdisplay) THEN
                        headerComp(useLFN, binfile,source,target);
                    END;
                    IF testsame THEN
                        verbose:=FALSE; (* whatever display *)
                    ELSE
                        verbose:=NOT(altdisplay);
                    END;
                    rcode:=doBinCompare (useLFN, verbose, candy,
                                        ignorefsize,maxmismatches,
                                        source,target);
                    CASE rcode OF
                    | rcSize:       mismatch:=TRUE; (* well... *)
                    | rcMismatch:   mismatch:=TRUE;
                    | rcAborted:    RETURN errAborted;
                    END;

                    (* nl included *)
                    IF altdisplay THEN
                        (* don't use nopath here *)
                        Str.Copy(Ssrc,nice(useLFN,source));
                        Str.Copy(Stgt,nice(useLFN,target));

                        CASE rcode OF
                        | rcSame:       Str.Concat(S,msgOK+" | and |",nl);
                                        Str.Subst(S,"|",Ssrc);
                                        Str.Subst(S,"|",Stgt);
                                        WrStr(S);
                        | rcSameForced: Str.Concat(S,msgWARN+" | and |",nl);
                                        Str.Subst(S,"|",Ssrc);
                                        Str.Subst(S,"|",Stgt);
                                        WrStr(S);
                        | rcSize:       Str.Concat(S,msgPB+"Size mismatch between | and |",nl);
                                        Str.Subst(S,"|",Ssrc);
                                        Str.Subst(S,"|",Stgt);
                                        WrStr(S);
                        | rcMismatch:   Str.Concat(S,msgPB+"Mismatch(es) between | and |",nl);
                                        Str.Subst(S,"|",Ssrc);
                                        Str.Subst(S,"|",Stgt);
                                        WrStr(S);
                        END;
                    ELSE
                        CASE rcode OF
                        | rcSame:       WrStr(msgSAME);
                        | rcSameForced: WrStr(msgSAMEFORCED);
                        | rcSize:       WrStr(msgSIZE);
                        | rcMismatch:   IF testsame THEN
                                            WrStr(msgNOTID);
                                        END; (* else, already done while comparing *)
                        END;
                    END;

                    nothingtodo:=FALSE;
                | textfile:
                    (* we skip forbidden binary extensions the m$ way : without telling user about it *)
                    IF validExtensions(sSkippedExtensions,e,e2) THEN
                        olderNewer(useLFN,source,target);

                        headerComp(useLFN, textfile,source,target);
                        rcode:=doTxtCompare (useLFN, TRUE ,candy,keepcase,keepblanks,
                                            source,target);

                        CASE rcode OF
                        | rcSame:     WrStr(msgSAMELINES);
                        | rcMismatch: mismatch:=TRUE;
                        | rcAborted:  RETURN errAborted;
                        END;
                        nothingtodo:=FALSE;
                    END;
                END;
                found:=findMatchName (pmatch, e2,
                                     pmatch^.next,rootanchor,icomp,patform,
                                     fpat,epat,d,f,e);
            END;
            IF NOT (matched) THEN
                Str.Concat(S, nice (useLFN,source) , nl);
                Str.Prepend(S, msgNOMATCH);
                WrStr(S);
            END;
        END;
        p:=anchor^.next;
        anchor:=p;
    END;
    IF mismatch THEN
        rc:=errMismatch;
    ELSE
        IF nothingtodo THEN WrStr(msgNOTHINGTODO); END;
        rc:=errNone;
    END;
    RETURN rc;
END procComp;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE initBatch (useLFN:BOOLEAN; opcode:opcodeType);
VAR
    batchfile:pathtype;
    hout:FIO.File;
BEGIN
    CASE opcode OF
    | opDel,opDD    : Str.Concat(batchfile,sDELcmd ,sThem);
    | opCopy,opPCOPY: Str.Concat(batchfile,sCOPYcmd,sThem);
    | opUser        : Str.Concat(batchfile,sUSERcmd,sThem);
    ELSE
        RETURN; (* should never happen *)
    END;
    IF fileExists(useLFN,batchfile) THEN
        IF fileIsRO(useLFN,batchfile) THEN fileSetRW(useLFN,batchfile);END;
    END;
    hout:=FIO.Create(batchfile);
    FIO.WrStr(hout,"@"+sEcho+blank+"ON");
    FIO.Close(hout);
END initBatch;

PROCEDURE show (VAR sourceshown:BOOLEAN;
               useLFN,altdisplay,showallmatches,bat,redirected:BOOLEAN;hout:FIO.File;
               op:ARRAY OF CHAR;source,target:pathtype);
VAR
    S:str1024;
BEGIN
    IF NOT(showallmatches) THEN
        IF altdisplay THEN nopath(source);END;
    END;
    IF sourceshown = FALSE THEN
        Str.Concat(S, nice(useLFN,source), nl);
        IF redirected THEN WrStr(op); END;
        WrStr(S);
        IF bat THEN FIO.WrStr(hout,op);FIO.WrStr(hout,S);END;
        sourceshown:=TRUE;
    END;
    IF showallmatches THEN
        Str.Concat(S, nice(useLFN,target), nl);
        IF redirected THEN WrStr(op); END;
        WrStr(S);
        IF bat THEN FIO.WrStr(hout,op);FIO.WrStr(hout,S);END;
    END;
END show;

PROCEDURE shownomatch (useLFN,altdisplay,bat:BOOLEAN;hout:FIO.File;
                      source:pathtype);
VAR
    S:str1024;
BEGIN
    IF altdisplay THEN nopath(source);END;
    Str.Concat(S, nice (useLFN,source), nl);
    Str.Prepend(S,msgNOMATCH);
    (* IF redirected THEN WrStr(sRemark+blank);END; *)
    WrStr(S);
    IF bat THEN FIO.WrStr(hout,sRemark+blank);FIO.WrStr(hout,S);END;
END shownomatch;

(* modified procComp *)

PROCEDURE procList (anchor:pEntry; lst:listmodeType; opcode:opcodeType;
                   useLFN,
                   altdisplay,candy,showallmatches,makebatch,redirected:BOOLEAN;
                   iref,icomp:CARDINAL;
                   patref,patcomp:pathtype;
                   usercli:ARRAY OF CHAR):CARDINAL;
VAR
    patform:patType;
    i:CARDINAL;
    matched,found,ok,sourceshown:BOOLEAN;
    rootanchor,p,pdir,pmatch:pEntry;
    file,d,source,target,f,e,fpat,epat,base:pathtype;
    rcode:resultType;
    rc:CARDINAL;
    e2:pathtype; (* target ext uppercasealt-ed *)
    S,S2:str128;
    pass:CARDINAL;
    op:str16;
    hout:FIO.File;
    batchfile:pathtype;
    bat:BOOLEAN;
BEGIN
    (* say we don't report files without match name *)
    CASE lst OF
    | lstIdentical: S:=msgINFO+"List of matching identical files"+nl;
    | lstDifferent: S:=msgINFO+"List of matching different files"+nl;
    END;
    CASE opcode OF
    | opDefault: op:="";
    | opDel:     op:=sDOSDELcli+blank;  Str.Concat(batchfile,sDELcmd,sThem);
    | opCopy:    op:=sDOSCOPYcli+blank; Str.Concat(batchfile,sCOPYcmd,sThem);
    | opDD:      op:=sDDcli+blank;      Str.Concat(batchfile,sDELcmd,sThem);
    | opPCOPY:   op:=sPCOPYcli+blank;   Str.Concat(batchfile,sCOPYcmd,sThem);
    | opUser:                           Str.Concat(batchfile,sUSERcmd,sThem);
                 Str.Concat(op,usercli,blank);
    END;
    bat:=makebatch AND (redirected=FALSE) AND (opcode # opDefault);
    IF bat THEN
        Str.Concat(S2,sEcho+blank,S); (* "@ECHO ON"+nl was removed *)
    ELSIF redirected THEN
        Str.Concat(S2,sREMarkChar+blank,S); (* "@ECHO ON"+nl was removed *)
    ELSE
        Str.Copy(S2,S);
    END;
    IF bat THEN
        IF fileExists(useLFN,batchfile) THEN
            IF fileIsRO(useLFN,batchfile) THEN fileSetRW(useLFN,batchfile);END;
        END;
        hout:=FIO.Create(batchfile);
    ELSE
        hout:=FIO.StandardOutput; (* safety *)
    END;
    IF redirected THEN
        WrStr(S2);
    ELSE
        WrStr(S);
    END;
    IF bat THEN FIO.WrStr(hout,S2);END;

    (* done with damn header, let's go now ! *)

    patform:=getPattern(patcomp); (* already checked for errors *)
    ok:=splitFileExt(fpat,epat, patcomp);
    UpperCaseAlt(fpat);
    UpperCaseAlt(epat);

    rootanchor:=anchor;
    p:=anchor;
    WHILE anchor # NIL DO
        ok:=(p^.baseindex # FULLPATH);
        ok:=(ok AND (p^.specindex = SHORTCARD(iref) ) );
        IF ok THEN
            getStr(file,p);
            ok:=findMatchID(pdir, rootanchor,iref,LONGCARD(p^.baseindex) ); (* assume always TRUE *)
            getStr(d,pdir);
            Str.Concat(source,d,file);

            ok:=splitFileExt(f,e, file);
            UpperCaseAlt(d);
            UpperCaseAlt(f);
            UpperCaseAlt(e);

            sourceshown:=FALSE; (* we haven't shown source file yet *)

            found:=findMatchName (pmatch, e2,
                                 rootanchor,rootanchor,icomp,patform,
                                 fpat,epat,d,f,e);
            matched:=found;
            WHILE found DO
                getStr(file,pmatch);
                ok:=findMatchID(pdir, rootanchor,icomp,LONGCARD(pmatch^.baseindex) ); (* assume always TRUE *)
                getStr(base,pdir);
                Str.Concat(target,base,file);

                (* verbose FALSE, ignorefsize FALSE, maxmismatches 1 *)

                rcode:=doBinCompare (useLFN, FALSE,candy, FALSE,1,
                                    source,target);

                CASE rcode OF
                | rcSame,rcSameForced:
                    IF lst=lstIdentical THEN
                        show (sourceshown,
                             useLFN,  altdisplay,showallmatches,
                             bat,redirected,hout,op,
                             source,target);
                    END;
                | rcSize:
                    IF lst=lstDifferent THEN
                        show (sourceshown,
                             useLFN,  altdisplay,showallmatches,
                             bat,redirected,hout,op,
                             source,target);
                    END;
                | rcMismatch:
                    IF lst=lstDifferent THEN
                        show (sourceshown,
                             useLFN,  altdisplay,showallmatches,
                             bat,redirected,hout,op,
                             source,target);
                    END;
                | rcAborted:
                    IF bat THEN FIO.Close(hout);END;
                    RETURN errAborted;
                END;

                found:=findMatchName (pmatch, e2,
                                     pmatch^.next,rootanchor,icomp,patform,
                                     fpat,epat,d,f,e);
            END;
            IF NOT(matched) THEN
                IF lst = lstDifferent THEN
                    shownomatch(useLFN,  altdisplay,bat,hout,source);
                END;
            END;
        END;
        p:=anchor^.next;
        anchor:=p;
    END;
    IF bat THEN
        FIO.Close(hout);
        Str.Concat(S,msgINFO,batchfile);Str.Append(S," has been created.");
        WrStr(S);WrLn;
    END;
    rc:=errNone;
    RETURN rc;
END procList;

PROCEDURE fixStorageError (VAR rc, ignoredstoragepb:CARDINAL;
                          fakeokstorage:BOOLEAN);
BEGIN
    IF rc=errStorage THEN
        IF fakeokstorage THEN
           rc:=errNone;
           INC(ignoredstoragepb);
        END;
    END;
END fixStorageError;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

(* "? digest file" *)

PROCEDURE grabname (VAR R:pathtype;S:ARRAY OF CHAR);
VAR
    p:CARDINAL;
BEGIN
    Str.Copy(R,"");
    ReplaceChar(S,tab,blank);
    p:=Str.CharPos(S,blank);
    IF p = MAX(CARDINAL) THEN RETURN; END;
    Str.Delete(S,0,p+1);
    LtrimBlanks(S);

    p:=Str.CharPos(S,blank);
    IF p = MAX(CARDINAL) THEN RETURN; END;
    Str.Delete(S,0,p+1);
    LtrimBlanks(S);
    IF S[0]="*" THEN Str.Delete(S,0,1);END;
    Str.Copy(R,S);
END grabname;

CONST
    opdelme     = 0;
    opremme     = 1;
    opflagme    = 2; (* FNF *)

PROCEDURE dmpname (useLFN,makebatch:BOOLEAN;whatop:CARDINAL;h:FIO.File;F:pathtype);
VAR
    S:str1024;
BEGIN
    IF NOT (makebatch) THEN RETURN; END;
    CASE whatop OF
    | opdelme : S:=sDELcmd;
    | opremme : S:=sRemark;
    | opflagme: S:=msgPB+"File not found !";
    END;
    Str.Append(S,blank);
    IF useLFN THEN Str.Append(S,dquote);END;
    Str.Append(S,F);
    IF useLFN THEN Str.Append(S,dquote);END;
    FIO.WrStr(h,S);FIO.WrLn(h);
    IF whatop=opflagme THEN WrStr(S);WrLn;END;
END dmpname;

PROCEDURE chkCanon (useLFN,bat,chkExistence:BOOLEAN;
                   hout:FIO.File;F:pathtype):BOOLEAN;
VAR
    ok,canonical:BOOLEAN;
BEGIN
    ok:=TRUE;
    IF chkExistence THEN
        IF F[0] = "\" THEN
            canonical := TRUE;
        ELSE
            canonical := Str.Match(F,"?:\*");
        END;
        IF canonical THEN
            ok:= fileExists(useLFN,F);
            IF NOT(ok) THEN dmpname(useLFN,bat,opflagme,hout,F);END;
        END;
    END;
    RETURN ok;
END chkCanon;

(* #=ref, ;=dup, skip <digest> *)

PROCEDURE doCheckList ( bat,useLFN,chkExistence:BOOLEAN;
                        rcOK,rcPB:CARDINAL;F:pathtype):CARDINAL;
CONST
    msgPBAT         = msgPB+"Problem at line ";
    msgGOODchk      = msgOK+"No mismatch found while checking file content.";
    msgGOODnochk    = msgOK+"No mismatch found but file content was not checked !";
    msgBAD          = msgPB+"Mismatch(es) and/or problem(s) found.";
VAR
    ciaobye:BOOLEAN;
    pb,currline:LONGCARD;
    rc:CARDINAL;
    hin,hout:FIO.File;
    refentry,currentry,batchfile:pathtype;
    S:str1024;
    state:(waiting,gotref,ingroup);
    result:resultType;
    verbose,candy,keepEOF:BOOLEAN;
BEGIN
    verbose:=FALSE; (* don't show address and byte values *)
    candy  :=TRUE;
    IF bat THEN
        batchfile:=sDELcmd+sThem;
        IF fileExists(useLFN,batchfile) THEN
            IF fileIsRO(useLFN,batchfile) THEN fileSetRW(useLFN,batchfile);END;
        END;
        hout:=fileCreate(useLFN,batchfile);
        S:="@ECHO ON"+nl+"PAUSE"+nl;
        FIO.WrStr(hout,S);
    END;
    pb:=0;
    currline:=0;
    ciaobye:=FALSE;
    hin:=fileOpenRead(useLFN,F);

    FIO.EOF:=FALSE;
    state:=waiting;
    LOOP
        IF FIO.EOF THEN EXIT; END;
        FIO.RdStr(hin,S); INC(currline);
        LtrimBlanks(S);
        RtrimBlanks(S);
        CASE S[0] OF
        | nullchar:
            IF FIO.EOF THEN EXIT; END;
        | pound :
            CASE state OF
            | waiting: ;
            | gotref:  ciaobye:=TRUE; (* # then # again *)
            | ingroup: ;
            END;
            grabname(refentry,S);
            state:=gotref;
            IF bat THEN FIO.WrLn(hout);END;
            dmpname(useLFN,bat,opremme,hout,refentry);

            IF chkCanon(useLFN,bat,chkExistence,hout,refentry)=FALSE THEN (* //V13G *)
                ciaobye:=TRUE;
            END;
        | semicolon :
            CASE state OF
            | waiting: ciaobye:=TRUE; (* first entry is not ref *)
            | gotref:  ;
            | ingroup: ;
            END;

            IF NOT(ciaobye) THEN
                grabname(currentry,S);
                IF chkCanon(useLFN,bat,chkExistence,hout,currentry)=FALSE THEN (* //V13G *)
                    ciaobye:=TRUE;
                END;
            END;

            IF NOT(ciaobye) THEN
                state:=ingroup;
                (* grabname(currentry,S); *)
                keepEOF:=FIO.EOF; (* required because of silly TS design ! *)
                (* headerComp (useLFN,binfile,refentry,currentry); *)

                IF chkExistence THEN
                    result:=doBinCompare (useLFN, verbose, candy,
                                         FALSE,1,
                                         refentry,currentry);
                ELSE
                    result:=rcSame;
                END;
                FIO.EOF:=keepEOF; (* restore *)

                CASE result OF
                | rcSame :
                    dmpname(useLFN,bat,opdelme,hout,currentry);
                | rcAborted:
                    ciaobye:=TRUE;
                    S:=msgINFO+"Aborted by user !";
                ELSE
                    dmpname(useLFN,bat,opremme,hout,currentry);
                    headerComp (useLFN,binfile,refentry,currentry);
                    WrStr(msgNOTID);
                    INC(pb);
                END;
            END;
        ELSE
            ciaobye:=TRUE; (* unexpected *)
            Str.Prepend(S,msgPB);
        END;
        IF ciaobye THEN
            INC(pb); WrStr(msgPBAT);WrStr( fmtlc(currline,10,1,"",""));WrLn;
            EXIT;
        END;
    END;
    fileClose(useLFN,hin);
    CASE state OF
    | waiting: ;
    | gotref:  IF NOT(ciaobye) THEN  (* # without ; *)
                   INC(pb); WrStr(msgPBAT);WrStr( fmtlc(currline,10,1,"",""));WrLn;
               END;
    | ingroup: ;
    END;

    IF bat THEN
        fileClose(useLFN,hout);
        WrStr(msgOK);WrStr(batchfile);WrStr(" has been created.");WrLn;
    END;
    IF pb = 0 THEN
        rc:=rcOK; IF chkExistence THEN S:=msgGOODchk; ELSE S:=msgGOODnochk; END;
    ELSE
        rc:=rcPB; S:=msgBAD;
    END;
    WrStr(S);WrLn;
    RETURN rc;
END doCheckList;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    firstparm  = 1;
    maxparm    = 2;
    ndxref     = firstparm;
    ndxcomp    = firstparm+1; (* darn, 1+1=2 ! <g> *)
VAR
    S,R:pathtype;
    lastparm,i,parmcount,opt:CARDINAL;
    parm : ARRAY[firstparm..maxparm] OF pathtype;
    sLegalUnits:str80; (* more than enough *)
    anchor : pEntry;
    currID:LONGCARD;
    rc:CARDINAL;
    base     : ARRAY [ndxref..ndxcomp] OF pathtype;
    spec     : ARRAY [ndxref..ndxcomp] OF pathtype;
    dircount : ARRAY [ndxref..ndxcomp] OF LONGCARD;
    filecount: ARRAY [ndxref..ndxcomp] OF LONGCARD;
    fcount:LONGCARD;
    useLFN : BOOLEAN;
    recurse,ignorefsize:BOOLEAN;
    altdisplay,keepcase,keepblanks,testsame,chkExistence:BOOLEAN;
    showallmatches,makebatch,candy: BOOLEAN;
    maxmismatches : LONGCARD;
    cmpmode       : cmpType;
    listmode      : listmodeType;
    opcode        : opcodeType;
    redirected    : BOOLEAN;
    usercli       : str128; (* should DO ! *)
    FAKEOKSTORAGE : BOOLEAN;
    IGNOREDSTORAGEPB:CARDINAL;
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE;
    FIO.ShareMode:=FIO.ShareDenyNone; (* very, very important ! *)

    WrLn;

    useLFN        := TRUE;
    recurse       := FALSE;
    maxmismatches := 0;         (* not yet redefined *)
    cmpmode       := binfile;
    keepcase      := FALSE;
    keepblanks    := FALSE;
    ignorefsize   := FALSE;
    testsame      := FALSE;
    listmode      := lstDefault;
    opcode        := opDefault;
    makebatch     := FALSE;
    candy         := FALSE;
    showallmatches:= FALSE;
    altdisplay    := FALSE;
    chkExistence  := FALSE;
    FAKEOKSTORAGE := FALSE;
    IGNOREDSTORAGEPB:=0;
    DEBUG         := FALSE;

    parmcount:=Lib.ParamCount();
    IF parmcount = 0 THEN abort(errHelp,""); END;

    lastparm:=firstparm-1;
    FOR i:=1 TO parmcount DO
        Lib.ParamStr(R,i); cleantabs(R);
        Str.Copy(S,R);
        UpperCase(S);
        IF isOption(S) THEN
            opt := GetOptIndex (S, "?"+delim+"H"+delim+"HELP"+delim+
                                   "LFN"+delim+"X"+delim+
                                   "S"+delim+"SUB"+delim+
                                   "M:"+delim+"MAX:"+delim+
                                   "T"+delim+"TEXT"+delim+"TXT"+delim+"LINES"+delim+
                                   "K"+delim+"KEEPCASE"+delim+
                                   "W"+delim+"KEEPBLANKS"+delim+
                                   "Z"+delim+"SIZE"+delim+
                                   "L"+delim+"IDENTICAL"+delim+
                                   "D"+delim+"DIFFERENT"+delim+
                                   "I"+delim+"IDENTICAL"+delim+

                                   "LL"+delim+"LA"+delim+
                                   "DD"+delim+"DA"+delim+
                                   "BAT"+delim+"BATCH"+delim+
                                   "V"+delim+"VERBOSE"+delim+
                                   "N"+delim+"ALTERNATE"+delim+"NAMES"+delim+
                                   "DB"+delim+
                                   "DBB"+delim+"DBA"+delim+
                                   "A"+delim+"ALL"+delim+"ALLMATCHES"+delim+

                                   "DEL"+delim+
                                   "COPY"+delim+
                                   "KILL"+delim+
                                   "PCOPY"+delim+
                                   "KB"+delim+
                                   "KBB"+delim+"KBA"+delim+
                                   "U:"+delim+"USER:"+delim+"CMD:"+delim+
                                   "SDB"+delim+"DBS"+delim+
                                   "MEM"+delim+"FAKE"+delim+
                                   "C"+delim+"CHECK"+delim+
                                   "CC"+delim+"CHECKDEL"+delim+
                                   "E"+delim+"EXIST"+delim+"EXISTENCE"+delim+
                                   "CE"+delim+
                                   "CCE"+delim+

                                   (* "SDBB"+delim+"DBBS"+delim+ *)
                                   "DEBUG"
                               );
            CASE opt OF
            | 1,2,3: abort(errHelp,"");
            | 4,5  : useLFN := FALSE;
            | 6,7  : recurse:= TRUE;
            | 8,9  : IF GetLongCard(S,maxmismatches)=FALSE THEN
                         abort(errBadValue,R);
                     END;
                     IF maxmismatches=0 THEN maxmismatches:=MAX(LONGCARD);END;
            | 10,11,12,13: cmpmode:= textfile; candy:=TRUE;
            | 14,15: keepcase:=TRUE;
            | 16,17: keepblanks:=TRUE;
            | 18,19: ignorefsize:=TRUE;
            | 20,21: IF chkList(listmode,lstIdentical)=FALSE THEN abort(errListMode,"");END;
            | 22,23: IF chkList(listmode,lstDifferent)=FALSE THEN abort(errListMode,"");END;
            | 24,25: testsame:=TRUE;

            | 26,27: IF chkList(listmode,lstIdentical)=FALSE THEN abort(errListMode,"");END;
                     showallmatches:=TRUE;
            | 28,29: IF chkList(listmode,lstDifferent)=FALSE THEN abort(errListMode,"");END;
                     showallmatches:=TRUE;
            | 30,31: makebatch:=TRUE;
            | 32,33: candy:=TRUE; (* "V" *)
            | 34,35,36: altdisplay:=TRUE;
            | 37   : IF chkOp(opcode,opDel)=FALSE THEN abort(errOpcode,"");END;
                     makebatch:=TRUE;
            | 38,39: IF chkOp(opcode,opDel)=FALSE THEN abort(errOpcode,"");END;
                     makebatch:=TRUE;
                     showallmatches:=TRUE;
            | 40,41,42:showallmatches:=TRUE;

            | 43   : IF chkOp(opcode,opDel)=FALSE THEN abort(errOpcode,"");END;
            | 44   : IF chkOp(opcode,opCopy)=FALSE THEN abort(errOpcode,"");END;
            | 45   : IF chkOp(opcode,opDD)=FALSE THEN abort(errOpcode,"");END;
            | 46   : IF chkOp(opcode,opPCOPY)=FALSE THEN abort(errOpcode,"");END;
            | 47   : IF chkOp(opcode,opDD)=FALSE THEN abort(errOpcode,"");END;
                     makebatch:=TRUE;
            | 48,49: IF chkOp(opcode,opDD)=FALSE THEN abort(errOpcode,"");END;
                     makebatch:=TRUE;
                     showallmatches:=TRUE;
            | 50,51,52: GetString(R,usercli);
                     IF same(usercli,"") THEN abort(errUserCmd,"");END;
                     IF chkOp(opcode,opUser)=FALSE THEN abort(errOpcode,"");END;
            | 53,54: IF chkOp(opcode,opDel)=FALSE THEN abort(errOpcode,"");END;
                     makebatch:=TRUE;
                     recurse:=TRUE;
            | 55,56: FAKEOKSTORAGE := TRUE;
            | 57,58: IF chkList(listmode,lstCheck) =FALSE THEN abort(errListMode,"");END;
            | 59,60: IF chkList(listmode,lstCheck) =FALSE THEN abort(errListMode,"");END;
                     makebatch:=TRUE;
            | 61,62,63:chkExistence:=TRUE;
            | 64   : IF chkList(listmode,lstCheck) =FALSE THEN abort(errListMode,"");END;
                     chkExistence:=TRUE;
            | 65   : IF chkList(listmode,lstCheck) =FALSE THEN abort(errListMode,"");END;
                     makebatch:=TRUE;
                     chkExistence:=TRUE;
            (*
            | 66,67: IF chkOp(opcode,opDel)=FALSE THEN abort(errOpcode,"");END;
                     makebatch:=TRUE;
                     showallmatches:=TRUE;
                     recurse:=TRUE;
            *)
            | 66    : DEBUG:=TRUE;
            ELSE
                abort(errOption,R);
            END;
        ELSE
            INC(lastparm);
            IF lastparm > maxparm THEN abort(errParmOverflow,R);END;
            parm[lastparm]:=R;
        END;
    END;

    redirected := IsRedirected();
    useLFN := ( useLFN AND w9XsupportLFN() );

    getAllLegalUnits(TRUE,TRUE,TRUE,sLegalUnits); (* floppy,hd,CDROM *)

    CASE listmode OF
    | lstCheck:
        CASE lastparm OF
        | 0: abort(errExpecting,"<list>");
        | 1: ;
        ELSE
             abort(errExpecting,"only <list>");
        END;
    ELSE
        CASE lastparm OF
        | 0: abort(errExpecting,"<spec1> <spec2>");
        | 1: abort(errExpecting,        "<spec2>");
        ELSE
            ;
        END;
    END;

    CASE listmode OF
    | lstDefault:
        IF showallmatches            THEN abort(errListOnly,"-a");END;
        IF makebatch                 THEN abort(errListOnly,"-bat");END;
        CASE cmpmode OF
        | binfile:
            IF keepcase              THEN abort(errNonsenseBin,"-k");END;
            IF keepblanks            THEN abort(errNonsenseBin,"-w");END;
        | textfile:
            IF ignorefsize           THEN abort(errNonsenseText,"-z");END;
            IF maxmismatches # 0     THEN abort(errNonsenseText,"-m:#");END;
        END;
        IF testsame THEN
            IF recurse               THEN abort(errNonsenseTest,"-s");END;
            IF ignorefsize           THEN abort(errNonsenseTest,"-z");END;
            IF maxmismatches # 0     THEN abort(errNonsenseTest,"-m:#");END;
        END;
        IF opcode # opDefault        THEN abort(errOpcodeList,""); END;
    | lstIdentical,lstDifferent:
        (* better safe than sorry with text files *)
        IF cmpmode = textfile    THEN abort(errNonsenseList,"-t");END;
        (* this test now makes it useless to check for dircount later *)
        (* CHKME IF recurse               THEN abort(errNonsenseList,"-s");END; *)
        IF testsame              THEN abort(errNonsenseList,"-i");END;
        (* reason dictates these options are a nonsense with list modes *)
        IF keepcase              THEN abort(errNonsenseList,"-k");END;
        IF keepblanks            THEN abort(errNonsenseList,"-w");END;
        IF ignorefsize           THEN abort(errNonsenseList,"-z");END;
        IF maxmismatches # 0     THEN abort(errNonsenseList,"-m:#");END;
        IF makebatch THEN
            IF opcode=opDefault  THEN abort(errBatchPrefix,"");END;
            IF redirected        THEN abort(errNonsenseRedirection,"");END;
        END;
    | lstCheck :
        S:=parm[lastparm];
        IF chkJoker(S) THEN abort(errListJoker,S);END;
        IF fileExists(useLFN,S)=FALSE THEN abort(errListNotFound,S);END;
        rc:=doCheckList( makebatch,useLFN,chkExistence,  errCheckOK, errCheckPB, S);
        abort(rc,"");
    END;

    FOR i:=ndxref TO ndxcomp DO
        rc:= chkfixSpec( base[i],spec[i], useLFN,parm[i],sLegalUnits );
        IF rc # errNone THEN abort(rc,parm[i]);END;
IF DEBUG THEN
WrStr(msgDEBUG+"command line <spec> : ");WrStr(parm[i]);WrLn;
WrStr(msgDEBUG+"base                : ");WrStr(base[i]);WrLn;
WrStr(msgDEBUG+"spec                : ");WrStr(spec[i]);WrLn;
END;
    END;
    CASE getPattern(spec[ndxcomp]) OF
    | patUnsupported : abort(errSpecFileForm,"<spec2>");
    | patExtAlone    : abort(errSpecFileForm,"<spec2>");
    END;
    IF chkBothSpecs(base[ndxref],spec[ndxref],base[ndxcomp],spec[ndxcomp])=FALSE THEN
        abort(errSameSpecs,"");
    END;

    CASE listmode OF
    | lstDefault:
        IF testsame THEN (* there can be only one... well, one+one ;-) *)
            IF getPattern(spec[ndxref])  # patFileExt THEN abort(errMacLeod,"<spec1>");END;
            IF getPattern(spec[ndxcomp]) # patFileExt THEN abort(errMacLeod,"<spec2>");END;
        END;
    END;

    (* step 1 : let's build lists *)

    initList(anchor);
    currID := FULLPATH; (* i.e. firstindex-1 i.e. 0 *)

    FOR i:=ndxref TO ndxcomp DO
        dircount[i]:=0; (* we can't init count in recursive buildDirList() *)
        rc:= buildDirList (currID,dircount[i],anchor,
                          useLFN,recurse,i,base[i]);

        fixStorageError (rc,IGNOREDSTORAGEPB,  FAKEOKSTORAGE);

        IF rc # errNone THEN abort(rc, parm[i] ); END;
IF DEBUG THEN dmpList(anchor,i,useLFN,TRUE); END;
    END;

    (* from now on, currID will change for files *)

    FOR i:=ndxref TO ndxcomp DO
        rc:= buildFileList (currID,filecount[i],
                           anchor,
                           useLFN,i,spec[i]);

        fixStorageError (rc,IGNOREDSTORAGEPB,  FAKEOKSTORAGE);

        IF rc # errNone THEN abort(rc, parm[i] ); END;
IF DEBUG THEN dmpList (anchor,i,useLFN,FALSE);END;
    END;

IF DEBUG THEN
WrStr(msgDEBUG+"dircount <spec1>  : ");IO.WrLngCard(dircount[ndxref],10);WrLn;
WrStr(msgDEBUG+"dircount <spec2>  : ");IO.WrLngCard(dircount[ndxcomp],10);WrLn;
WrStr(msgDEBUG+"filecount <spec1> : ");IO.WrLngCard(filecount[ndxref],10);WrLn;
WrStr(msgDEBUG+"filecount <spec2> : ");IO.WrLngCard(filecount[ndxcomp],10);WrLn;
END;

    IF makebatch THEN initBatch(useLFN,opcode); END; (* for compatibility with v1.2, always create a dummy empty batch *)

    IF testsame THEN (* there can be only one... well, one+one ;-) *)
        fcount:=filecount[ndxref];
        IF fcount # 1 THEN
            IF fcount=0 THEN
                abort(errNotFound,spec[ndxref]);
            ELSE
                abort(errMacLeod,"<spec1>");
            END;
        END;
        fcount:=filecount[ndxcomp];
        IF fcount # 1 THEN
            IF fcount=0 THEN
                abort(errNotFound,spec[ndxcomp]);
            ELSE
                abort(errMacLeod,"<spec2>");
            END;
        END;
    END;

    (* step 2 : let's compare *)

    (*
    WrStr(Banner);WrLn;
    WrLn;
    *)

    IF maxmismatches = 0 THEN maxmismatches := 10; END; (* default *)
    IF testsame          THEN maxmismatches := 1;  END;

    rc:=errNone;
    CASE listmode OF
    | lstDefault :
        IF filecount[ndxref ] = 0 THEN abort(errNotFound,parm[ndxref]);END;
        IF filecount[ndxcomp] = 0 THEN abort(errNotFound,parm[ndxcomp]);END;

        rc:=procComp (anchor, cmpmode,
                     useLFN,
                     altdisplay,candy, keepcase,keepblanks,
                     ignorefsize,testsame,
                     maxmismatches,
                     ndxref,ndxcomp, spec[ndxref],spec[ndxcomp]);

    | lstIdentical,lstDifferent:
        (* here we don't test filecount *)

        rc:=procList (anchor, listmode, opcode,
                     useLFN,
                     altdisplay,candy, showallmatches, makebatch, redirected,
                     ndxref,ndxcomp, spec[ndxref],spec[ndxcomp],usercli);
    END;

    freeList(anchor);

    IF testsame THEN
        IF rc=errNone THEN rc:=errBothFilesMatch;END; (* not an error, in fact ! *)
    END;
    abort(rc,"");
END fComp.

