(* ---------------------------------------------------------------
Title         Q&D total
Author        PhG
Overview      size of files in a directory
Usage         see help
Notes         very, very, very quick & dirty... :-(
              minimal error messages and checking, etc.
              how silly, to duplicate TREE functionality...
              we handle floppy/hd/CDROM
Bugs          -w does not like 2Gb+ units and FAT32
Wish List     better ?:\ check/trap
              accept @filelist containing dirs ?

--------------------------------------------------------------- *)

MODULE total;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;
IMPORT SYSTEM;
IMPORT DOSErr;

FROM Storage IMPORT ALLOCATE,DEALLOCATE,Available;

(* because of QD_Text duplicates, we have disabled cr lf bs nl *)

FROM QD_ASCII IMPORT dash, slash, nullchar, tabchar, (* cr, lf, nl, *)
space, dot, deg, doublequote, quote, colon, percent, vbar,
(* bs, *) blank, equal, dquote, charnull, singlequote, antislash, dollar,
star, backslash, coma, question, underscore, tabul, hbar,
comma, semicolon,
stardotstar, dotdot, escCh, escSet, letters, digits;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits,
getAllLegalUnits, metaproc;

FROM QD_Text IMPORT
colortype, cursorshapetype, scrolltype,
ff, cr, lf, bs, tab, nl, mincolor, maxcolor,
BW40, CO40, BW80, CO80, CO80x43, CO80x50, MONO,
vesa80x60, vesa132x25, vesa132x43, vesa132x50, vesa132x60,
selectCursorEmulation,
setCursorShape,
handleVesa, setBrightPaper, setBlinkMode,
setFillChar, setFillInk, setFillPaper,
setFillInkPaper, setTxtInk, setTxtPaper, setTxtInkPaper, setWrapMode,
setUseBiosMode, setTabWidth, getScreenData, setWindow,
setMode, restoreMode,
gotoXY, xyToHtabVtab, home, setVisualPage, setActivePage,
scrollWindow, fillTextAttr, cls, writeStr, writeLn, getScreenWidth,
getScreenHeight, getScreenMinX, getScreenMinY, getScreenMaxX, getScreenMaxY,
getMinHtab, getMaxHtab, getMinVtab, getMaxVtab, getUseBiosMode,
getMinHtab, getMaxHtab, getMinVtab, getMaxVtab, getHtab, getVtab,
getWindowWidth, getWindowHeight,
initScreenConsole,
findInkPaperAtStartup, getInkAtStartup, getPaperAtStartup,
setFullScreenWindow, gotoFullScreenXY;

FROM QD_LFN IMPORT path9X, huge9X, findDataRecordType,
unicodeConversionFlagType, w9XchangeDir,
w9XgetDOSversion, w9XgetTrueDOSversion, w9XisWindowsEnh, w9XisMSDOS7,
w9XfindFirst, w9XfindNext, w9XfindClose, w9XgetCurrentDirectory,
w9XlongToShort, w9XshortToLong, w9XtrueName, w9XchangeDir,
w9XmakeDir, w9XrmDir, w9Xrename, w9XsupportLFN;

(* ------------------------------------------------------------ *)

TYPE
    pathtype = path9X;

TYPE
    HUGECARD = LONGREAL;
CONST
    HUGEZERO = 0.0;
    HUGEONE  = 1.0;

(* ------------------------------------------------------------ *)

CONST
    CONSOLE = TRUE ; (* if FALSE, do not use QD_Text library *)

(*%F CONSOLE  *)

PROCEDURE vidinit ( useBios:BOOLEAN );
BEGIN
END vidinit;

PROCEDURE print (S : ARRAY OF CHAR  );
BEGIN
    IO.WrStr(S);
END print;

PROCEDURE newline (  );
BEGIN
    IO.WrLn;
END newline;

PROCEDURE colorheader ();
BEGIN
END colorheader;

PROCEDURE colorhelp ();
BEGIN
END colorhelp;

PROCEDURE colortext ();
BEGIN
END colortext;

PROCEDURE colorwarning ();
BEGIN
END colorwarning;

PROCEDURE getRows (  ):CARDINAL ;
VAR
    vrows  [00040H:0084H] : SHORTCARD; (* add 1 *)
BEGIN
    RETURN ( CARDINAL(vrows) +1 ); (* 25, 43, 50, etc. *)
END getRows;

PROCEDURE getColumns (  ):CARDINAL ;
VAR
    vrows  [00040H:004AH] : CARDINAL; (* 40, 80, 132 *)
BEGIN
    RETURN ( CARDINAL(vrows) +1 ); (* 25, 43, 50, etc. *)
END getColumns;

(*%E  *)

(*%T CONSOLE  *)

PROCEDURE vidinit ( useBios:BOOLEAN  );
BEGIN
    (* handleVesa; *) (* useless, because we won't change video mode *)
    setUseBiosMode ( useBios );
    findInkPaperAtStartup();
END vidinit;

PROCEDURE print (S : ARRAY OF CHAR  );
BEGIN
    writeStr(S);
END print;

PROCEDURE newline (  );
BEGIN
    writeLn;
END newline;

PROCEDURE colorhelp ();
BEGIN
    setTxtInk(getInkAtStartup());      (* was green *)
    setTxtPaper(getPaperAtStartup());  (* was black *)
    setFillInkPaper(getInkAtStartup(),getPaperAtStartup());
END colorhelp;

PROCEDURE colorheader ();
BEGIN
    setTxtInk(yellow);
    setTxtPaper(black);
    setFillInkPaper(getInkAtStartup(),getPaperAtStartup());
END colorheader;

PROCEDURE colortext ();
BEGIN
    setTxtInk(cyan);
    setTxtPaper(black);
    setFillInkPaper(getInkAtStartup(),getPaperAtStartup());
END colortext;

PROCEDURE colorwarning ();
BEGIN
    setTxtInk(red);
    setTxtPaper(white);
    setFillInkPaper(getInkAtStartup(),getPaperAtStartup());
END colorwarning;

PROCEDURE getRows (  ):CARDINAL ;
BEGIN
    RETURN getWindowHeight();
END getRows;

PROCEDURE getColumns (  ):CARDINAL ;
BEGIN
    RETURN getWindowWidth();
END getColumns;

(*%E  *)

(* ------------------------------------------------------------ *)

CONST
    upper      = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    lower      = "abcdefghijklmnopqrstuvwxyz";

    filenameset= "&~#{-_}$!";

    legalset   = upper+lower+digits+filenameset+colon+backslash+dot;

    netslash   = backslash+backslash;

    sMaxDirs   = "16000";
TYPE
    scanmodetype = (normalandspecial,normalscan,specialscan);

(* ------------------------------------------------------------ *)

CONST
    ProgEXEname   = "TOTAL";
    ProgTitle     = "Q&D Total";
    ProgVersion   = "v1.0s";
    ProgCopyright = "by PhG";
    Banner        = ProgTitle+" "+ProgVersion+" "+ProgCopyright;
CONST
    errNone             = 0;
    errHelp             = 1;
    errUnknownOption    = 2;
    errParmOverflow     = 3;
    errMissingDirSpec   = 4;
    errJokerDir         = 5;
    errBadNameDir       = 6;
    errBadPath          = 7;
    errBadDir           = 8;
    errTooManyDirs      = 9;
    errNonsense         = 10;
    errRootOnly         = 11;
    errInvalid          = 12;
    errNonsensical      = 13;
    errPhantomUnit      = 14;
    errExclusive        = 15;
    errCmdEmpty         = 16;

    errEmptyDirHere     = 128;
    errNoEmptyDir       = 255;

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)
    helpmsg =
Banner+nl+
nl+
"Syntax : "+ProgEXEname+" <directory> [option]..."+nl+
nl+
"  -d    list cumulative results for each subdirectory from <directory>"+nl+
"  -x    exclude subdirectories, analyzing only current directory"+nl+
"  -n    list only normal files (H and S excluded, default is -n -z)"+nl+
"  -z    list only special files (H or S required, default is -n -z)"+nl+
"  -e    list only empty subdirectories (forced with -k and -r)"+nl+
"  -k    delete empty subdirectories including <directory> itself if empty"+nl+
"  -r    delete empty subdirectories except <directory> even if empty"+nl+
"  -p    sort by path (ignored with -k and -r)"+nl+
"  -i    sort by LFN path (ignored with -k and -r)"+nl+
"  -a    do not use graphics characters in separator lines"+nl+
"  -u    display paths in uppercase (real DOS or -l required)"+nl+
"  -c    shorten paths according to columns count"+nl+
"  -t[t] terse mode showing total only (-tt = alternate terser display)"+nl+
'  -w    show "wasted" space when analyzing root directory'+nl+
"  -f    english group separator (default is french group separator)"+nl+
"  -l    disable LFN support even if available"+nl+
(*%T CONSOLE  *)
"  -b    monochrome BIOS output (no colors)"+nl+
(*%E  *)
nl+
"a) Program will not handle more than "+sMaxDirs+" directories."+nl+
"b) LFN support is limited to display : internally, program handles DOS paths"+nl+
"   (-i option is therefore required in order to sort LFN paths)."+nl+
(* "   (thus, even with LFN support, -p option does not apply to LFN paths)."+nl+ *)
"c) -w option results will not be significant with FAT32 or 2Gb+ drives."+nl+
"d) If <directory> is empty, -k option will delete it while -r option will not !"+nl+
"e) By design, -k and -r options remove deepest empty directories only :"+nl+
"   DD utility is better suited to recursive empty directories processing."+nl+
"f) -e, -k and -r options return one of the following codes :"+nl+
"   128 (at least one empty directory) or 255 (no empty directory)."+nl+
"g) Results will be unreliable (i.e. weird and wrong) with NFTS."+nl;
VAR
    S : str256;
BEGIN
    colorhelp;
    CASE e OF
    | errHelp :
        print(helpmsg);
    | errUnknownOption :
        Str.Concat(S,"Unknown ",einfo); Str.Append(S," option !");
    | errParmOverflow :
        Str.Concat(S,einfo," parameter is just one too many !");
    | errMissingDirSpec:
        S:="Missing directory specification !";
    | errJokerDir :
        S := "Jokers not allowed in directory specification !";
    | errBadNameDir :
        S := "Illegal characters in directory specification !";
    | errBadPath :
        Str.Concat(S,"Illegal ",einfo); Str.Append(S," directory specification !");
    | errBadDir :
        Str.Concat(S,einfo," directory does not exist !");
    | errTooManyDirs:
        (*
        Str.Concat(S,"Too many subdirectories in ",einfo);
        Str.Append(S," !");
        *)
        S:="More than "+sMaxDirs+" directories !";
    | errNonsense:
        Str.Concat(S,einfo," option is a nonsense with -w option !");
    | errRootOnly:
        Str.Concat(S,einfo," is not the root directory required with -w option !");
    | errInvalid:
        Str.Concat(S,"Invalid ",einfo); Str.Append(S," drive !");
    | errNonsensical:
        Str.Concat(S,einfo," option is a nonsense with -d option !");
    | errPhantomUnit:
        Str.Concat(S,einfo,": unit does not exist !");
    | errExclusive:
        S := "-n and -z options are mutually exclusive !";
    | errCmdEmpty:
        Str.Concat(S,einfo," option is a nonsense with -e, -k or -r options !");

    | errEmptyDirHere :
        S := "At least one empty directory found !";
    | errNoEmptyDir :
        S := "No empty directory found !";

    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp :
        ; (* nada *)
    | errEmptyDirHere, errNoEmptyDir :
        ;
    ELSE
        print(ProgEXEname+" : "); print(S); newline;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)

PROCEDURE getCurrentPath (VAR R : ARRAY OF CHAR);
VAR
    drive : SHORTCARD;
    unit  : CHAR;
BEGIN
    drive := FIO.GetDrive();
    unit  := CHR(drive + ORD("A") -1 );
    FIO.GetDir(drive,R); (* \path without trailing slash nor leading u: *)
    Str.Prepend(R,colon);
    Str.Prepend(R,unit); (* u:\path *)
    fixDirectory(R);     (* u:\path\ *)
END getCurrentPath;

(*
    assume path is in UPPERCASE ! legal variations upon u:\...\...

    .       use current path
      \xxx  prepend current unit
       xxx  prepend current path
    .\xxx   prepend current path

    u:\xxx  ok
    u:xxx   insert current path for u:
    u:.     use current path for u:
    u:      use current path for u: too even though it's ugly because no directory explicitely specified
    u:.\xxx insert current path for u:

*)

PROCEDURE buildCanonicalPath (base:ARRAY OF CHAR; VAR R : ARRAY OF CHAR):BOOLEAN;
VAR
    p         : CARDINAL;
    drive     : SHORTCARD;
    unit      : CHAR;
    S         : str128;
BEGIN
    p := Str.CharPos(R,colon);
    CASE p OF
    | 1 :                               (* unit specified *)
        IF CharCount(R,colon) # 1 THEN RETURN FALSE; END; (* more than one ":" *)

        IF R[0]=base[0] THEN
            Str.Copy(S,base);           (* current path of current unit again *)
        ELSE
            drive := SHORTCARD( ORD(R[0])-ORD("A")+1 );
            unit  := CHR(drive + ORD("A") -1 );
            FIO.GetDir(drive,S);
            Str.Prepend(S,colon);
            Str.Prepend(S,unit);
            fixDirectory(S);            (* u:\path\ *)
        END;

        CASE R[2] OF
        | dot :
            IF Str.Length(R)=3 THEN                      (* "U:." *)
                Str.Copy(R,S);
                RETURN TRUE;
            END;
            IF R[3] # backslash THEN RETURN FALSE; END;
            Str.Delete(R,0,4);                           (* "U:.\XXX" *)
            Str.Prepend(R,S);
        | backslash :                                    (* "U:\XXX" *)
            (* do nothing ! *)
        ELSE                                             (* "U:XXX" or "U:" *)
            Str.Delete(R,0,2);
            Str.Prepend(R,S);
        END;
    | MAX(CARDINAL):                    (* no unit specified *)
        CASE R[0] OF
        | dot :
            IF Str.Length(R)=1 THEN                      (* "." *)
                Str.Copy(R,base);
                RETURN TRUE;
            END;
            IF R[1] # backslash THEN RETURN FALSE; END;
            Str.Delete(R,0,2);                           (* ".\XXX" *)
            Str.Prepend(R,base);
        | backslash :                                    (* "\XXX" *)
            Str.Prepend(R,colon);
            Str.Prepend(R,base[0]);
        ELSE
            Str.Prepend(R,base);                         (* "XXX" *)
        END;
    ELSE
        RETURN FALSE;                   (* illegal, for ":" is either absent or at 1 *)
    END;
    IF Str.Pos(R,netslash) # MAX(CARDINAL) THEN RETURN FALSE;END;
    fixDirectory(R);                    (* just in case *)
    RETURN TRUE;
END buildCanonicalPath;

(* ah, we should have found a better name for QD_Box chkJoker ! ;-) *)

PROCEDURE chkJokersHere (S:ARRAY OF CHAR ) : BOOLEAN;
BEGIN
    IF Str.CharPos(S,star) # MAX(CARDINAL) THEN RETURN TRUE; END;
    IF Str.CharPos(S,question) # MAX(CARDINAL) THEN RETURN TRUE; END;
    RETURN FALSE;
END chkJokersHere;

PROCEDURE chkValidName (S:ARRAY OF CHAR ) : BOOLEAN;
VAR
    i : CARDINAL;
BEGIN
    IF Str.Pos(S,dotdot) # MAX(CARDINAL) THEN RETURN FALSE; END;
    IF Str.Pos(S,netslash) # MAX(CARDINAL) THEN RETURN FALSE; END;

    (* we no longer check other characters, for win95fr uses weirdoes ! *)
    (*
    FOR i := 0 TO (Str.Length(S)-1) DO
        IF Str.CharPos(legalset,S[i])=MAX(CARDINAL) THEN RETURN FALSE; END;
    END;
    *)
    RETURN TRUE;
END chkValidName;

PROCEDURE fixUnit (wasted:BOOLEAN;VAR S:ARRAY OF CHAR);
BEGIN
    IF wasted THEN
        IF Str.Match(S,"?:") THEN
            Str.Append(S,backslash); (* force "u:" to becomes "u:\" *)
        END;
    END;
END fixUnit;

(* ------------------------------------------------------------ *)

PROCEDURE fmtlc (v : LONGCARD; pad:CHAR; sep:CHAR; field:INTEGER) : str80;
VAR
    S,R   : str80;
    len,i : CARDINAL;
    ok  : BOOLEAN;
    ch  : CHAR;
BEGIN
    Str.CardToStr(v,S,10,ok);
    len:=Str.Length(S);
    R := "";
    FOR i := 1 TO len DO
        Str.Prepend(R,S[len-i]);
        IF i < len THEN
            IF (i MOD 3) = 0 THEN
                Str.Prepend(R,sep);
            END;
        END;
    END;
    LOOP
        IF INTEGER(Str.Length(R)) >= ABS(field) THEN EXIT; END;
        IF field < 0 THEN
            Str.Append(R,pad);  (* right alignment *)
        ELSE
            Str.Prepend(R,pad);
        END;
    END;
    RETURN R;
END fmtlc;

PROCEDURE fmtc (v : CARDINAL; pad:CHAR; sep:CHAR; field:INTEGER) : str80;
BEGIN
    RETURN fmtlc (LONGCARD(v),pad,sep,field);
END fmtc;

PROCEDURE fmtstring (S:ARRAY OF CHAR; pad:CHAR; n:INTEGER ) : str80;
VAR
    R : str80;
BEGIN
    Str.Copy(R,S);
    LOOP
        IF INTEGER(Str.Length(R)) >= ABS(n) THEN EXIT; END;
        IF n < 0 THEN
            Str.Append(R,pad); (* right alignment *)
        ELSE
            Str.Prepend(R,pad);
        END;
    END;
    RETURN R;
END fmtstring;

PROCEDURE fmtpercent (v:LONGREAL;pad:CHAR;sep:CHAR;field:INTEGER):str80;
CONST
    digits = 1; (* after dot/coma *)
VAR
    R : str80;
    ok: BOOLEAN;
BEGIN
    Str.FixRealToStr(v,digits,R,ok);
    Str.Subst(R,dot,sep);
    RETURN fmtstring(R,pad,field);
END fmtpercent;

PROCEDURE HUGEINC (VAR r:HUGECARD;n:HUGECARD );
BEGIN
    r := r + n;
END HUGEINC;

PROCEDURE fmthc (v : HUGECARD; pad:CHAR; sep:CHAR; field:INTEGER) : str80;
VAR
    S,R   : str80;
    len,i : CARDINAL;
    ok  : BOOLEAN;
    ch  : CHAR;
BEGIN
    Str.FixRealToStr( LONGREAL(v),0,S,ok);
    len:=Str.Length(S);
    R := "";
    FOR i := 1 TO len DO
        Str.Prepend(R,S[len-i]);
        IF i < len THEN
            IF (i MOD 3) = 0 THEN
                Str.Prepend(R,sep);
            END;
        END;
    END;
    LOOP
        IF INTEGER(Str.Length(R)) >= ABS(field) THEN EXIT; END;
        IF field < 0 THEN
            Str.Append(R,pad);  (* right alignment *)
        ELSE
            Str.Prepend(R,pad);
        END;
    END;
    RETURN R;
END fmthc;

(* ------------------------------------------------------------ *)

CONST
    firstPath = 1;     (* was 0, but 1 required by Lib.QSort and by firstPath-1 of course ! *)
    maxPath   = 16000; (* should do even with obscene win9X systems *)

TYPE
    ptrToEntry = POINTER TO entrytype;
    entrytype  = RECORD
        slen : SHORTCARD;
        string:CHAR; (* variable length *)
    END;
VAR
    Path : ARRAY[firstPath..maxPath] OF ptrToEntry; (* yes, we could use an index in entrytype too... *)

(* ------------------------------------------------------------ *)

PROCEDURE getentry (i:CARDINAL; VAR R : ARRAY OF CHAR);
CONST
    nullchar=CHR(0);
VAR
    len:CARDINAL;
BEGIN
    len := CARDINAL(Path[i]^.slen);
    Lib.FastMove( ADR(Path[i]^.string),ADR(R),len);
    R[len]:=nullchar; (* REQUIRED safety ! *)
END getentry;

PROCEDURE setentry ( i:CARDINAL; S : ARRAY OF CHAR  ) : BOOLEAN ;
VAR
    len,needed:CARDINAL;
BEGIN
    len    := Str.Length(S);
    needed := SIZE(entrytype)-SIZE(CHAR)+len;
    IF Available(needed) THEN
        ALLOCATE(Path[i],needed);
        Path[i]^.slen := SHORTCARD(len);
        Lib.FastMove( ADR(S),ADR(Path[i]^.string),len);
    ELSE
        Path[i]:=NIL;
    END;
    RETURN (Path[i] # NIL);
END setentry;

(* ------------------------------------------------------------ *)

VAR
    gForceLFNsort:BOOLEAN; (* globerk *)

PROCEDURE isLessAlpha (i,j : CARDINAL) : BOOLEAN;
VAR
    SI,SJ:pathtype; (* str128 *)
    shortform,longform:pathtype;
    errcode:CARDINAL;
BEGIN
    getentry(i,SI);
    getentry(j,SJ);

    IF gForceLFNsort THEN
        shortform:=SI;
        IF w9XshortToLong(shortform,errcode,longform) THEN SI:=longform; END;
        shortform:=SJ;
        IF w9XshortToLong(shortform,errcode,longform) THEN SJ:=longform; END;
    END;

    LowerCase(SI); (* ignore accents when sorting *)
    LowerCase(SJ);
    IF Str.Compare(SI,SJ) < 0 THEN (* -1 is less *)
        RETURN TRUE;
    ELSE
        RETURN FALSE;
    END;
END isLessAlpha;

PROCEDURE doSwap (i,j : CARDINAL);
VAR
    P : ptrToEntry;
BEGIN
    P:=Path[i];
    Path[i]:=Path[j];
    Path[j]:=P;
END doSwap;

(* ------------------------------------------------------------ *)

PROCEDURE fixDirPath (VAR S : ARRAY OF CHAR);
BEGIN
    IF Str.Length(S)=0 THEN Str.Copy(S,backslash); END;
    fixDirectory(S);
END fixDirPath;

PROCEDURE isDirEntry (S : ARRAY OF CHAR) : BOOLEAN;
BEGIN
    IF same(S,dot) THEN RETURN TRUE; END;
    RETURN same(S,dotdot);
END isDirEntry;

(* ------------------------------------------------------------ *)

PROCEDURE doDir (DEBUG:BOOLEAN; root : ARRAY OF CHAR;
                 VAR index:CARDINAL;VAR err:BOOLEAN);
VAR
    path    : str128;
    S       : str128;
    entry   : FIO.DirEntry;
    found   : BOOLEAN;
BEGIN
    IF err THEN RETURN; END;
    IF index > maxPath THEN
        err:=TRUE;
        RETURN;
    END;
    (*
    Str.Copy(Path[index],root);
    fixDirPath(Path[index]); (* add required \ because it will be REQUIRED *)
    *)
    fixDirPath(root); (* add required \ because it will be REQUIRED *)
    IF DEBUG THEN print(root);newline;END;
    IF setentry(index,root)=FALSE THEN
        err:=TRUE;
        RETURN;
    END;

    Str.Copy(path,root);
    fixDirPath(path); (* add required \ *)
    Str.Append(path,stardotstar); (* root\*.* *)

    found := FIO.ReadFirstEntry(path,everything,entry);
    WHILE found DO
        Work(cmdShow);
        Str.Copy(S,root);
        fixDirPath(S);
        Str.Append(S,entry.Name); (* root\f8e3 *)
        IF isDirEntry(entry.Name)=FALSE THEN (* skip . and .. *)
            IF aD IN entry.attr THEN
                INC(index);
                doDir(DEBUG,S,index,err);
            END;
        END;
        found :=FIO.ReadNextEntry(entry);
    END;
END doDir;

PROCEDURE buildPathList (rootdir : ARRAY OF CHAR;
                         flagRecurse,DEBUG : BOOLEAN;
                         VAR lastpath:CARDINAL) : BOOLEAN;
VAR
    i        : CARDINAL;
    error,rc : BOOLEAN;
BEGIN
    error := FALSE;
    i     := firstPath;
    IF flagRecurse THEN
        doDir(DEBUG,rootdir,i,error);
    ELSE
        fixDirPath(rootdir); (* just in case ! but should not be necessary here *)
        (* Str.Copy(Path[i],rootdir); *)
        IF DEBUG THEN print(rootdir);newline;END;
        IF setentry(i,rootdir)=FALSE THEN RETURN FALSE;END;
    END;
    lastpath := i;
    rc := NOT ( error ); (* reverse flag and return true if no error *)
    RETURN rc;
END buildPathList;

(* ------------------------------------------------------------ *)

PROCEDURE isRoot (S:ARRAY OF CHAR):BOOLEAN ;
BEGIN
    RETURN Str.Match(S,"?"+colon+backslash);
END isRoot;

(* assume u is UPPERCASE *)

PROCEDURE getUnitFreeTotal (u:CHAR; VAR freebytes,totalbytes:HUGECARD):BOOLEAN;
VAR
    r : SYSTEM.Registers;
    sectorsPerCluster : CARDINAL;
    freeClusters      : CARDINAL;
    bytesPerSector    : CARDINAL;
    totalClusters     : CARDINAL;
BEGIN
    freebytes  :=HUGEZERO;
    totalbytes :=HUGEZERO;
    r.AH := 036H;  (* DOS 2+ - GET FREE DISK SPACE *)
    r.DL := BYTE( ORD(u)-ORD("A")+1 ); (* $00=default, $01=A:, etc. *)
    Lib.Dos(r);
    IF r.AX = 0FFFFH THEN RETURN FALSE; END;
    sectorsPerCluster := r.AX; (* IF $FFFF, invalid drive *)
    freeClusters      := r.BX;
    bytesPerSector    := r.CX;
    totalClusters     := r.DX;
    freebytes  := HUGECARD(sectorsPerCluster) * HUGECARD(bytesPerSector);
    totalbytes := HUGECARD(sectorsPerCluster) * HUGECARD(bytesPerSector);
    freebytes  := freebytes  * HUGECARD(freeClusters);
    totalbytes := totalbytes * HUGECARD(totalClusters);
    RETURN TRUE;
END getUnitFreeTotal;

(* ------------------------------------------------------------ *)

(*
333333333333333333333333333
111111111111111122222222222222
*)

CONST
    ENCLOSEME = TRUE;
    NOENCLOSE = FALSE;
    FORCENL   = TRUE;
    NONL      = FALSE;
    NOTSHORTENED=FALSE;

(*
we brutally start with "..." but we could perform an inner cut too
addNL is NONL when showing final stats with subdirs, else is FORCENL i.e. TRUE
0123456789
c:\z\x      ok
c:\z\xxxx   ok
c:\z\xxxxx  shorten or leave as is without newline ?
c:\z\xxxxxx shorten
*)

PROCEDURE dmppath (addNL,shortenpaths,enclose:BOOLEAN;wi:CARDINAL;S:ARRAY OF CHAR);
CONST
    prefix    = "...";
    prefixlen = 3;
VAR
    le,colcount:CARDINAL;
    R:str1024;
BEGIN
    Str.Copy(R,S);
    IF shortenpaths THEN
        colcount:=getColumns();
        le:=Str.Length(R);
        IF enclose THEN INC(le,2);END;
        IF (wi+le) >= colcount THEN
            Str.Delete(R,0,wi+le-colcount+1);
            Str.Delete(R,0,prefixlen);
            Str.Prepend(R,prefix);
        END;
    END;
    IF enclose THEN print(dquote);END;
    print(R);
    IF enclose THEN print(dquote);END;
    IF addNL THEN newline; END;
END dmppath;

PROCEDURE dumpDirStats (lowercase,useLFN,addNL,shortenpaths:BOOLEAN; sepgroup:CHAR;
                        wisize,wicount,wdir:CARDINAL;
                        dirsize:HUGECARD;nfiles:LONGCARD;ndirs:CARDINAL;
                        sep,sDir:ARRAY OF CHAR    );
VAR
    valstr:str80;
    shortform,longform:pathtype;
    errcode:CARDINAL;
    wi,seplen:CARDINAL;
BEGIN
    seplen:=Str.Length(sep);
    wi:=0;
    valstr:=fmthc(dirsize,blank,sepgroup,wisize);  print(valstr);print(sep);
    IF shortenpaths THEN INC(wi, Str.Length(valstr));INC(wi,seplen);END;
    valstr:=fmtlc(nfiles,blank,sepgroup,wicount);  print(valstr);print(sep);
    IF shortenpaths THEN INC(wi, Str.Length(valstr));INC(wi,seplen);END;
    valstr:=fmtc(ndirs,blank,sepgroup,wdir);       print(valstr);print(sep);
    IF shortenpaths THEN INC(wi, Str.Length(valstr));INC(wi,seplen);END;
    IF useLFN THEN
        Str.Copy(shortform,sDir);
        IF w9XshortToLong(shortform,errcode,longform) THEN
            dmppath(addNL,shortenpaths,ENCLOSEME,wi,longform);
        ELSE
            dmppath(addNL,shortenpaths,NOENCLOSE,wi,sDir); (* should never happen *)
        END;
    ELSE
        IF lowercase THEN LowerCase(sDir);END;
        dmppath(addNL,shortenpaths,NOENCLOSE,wi,sDir);
    END;
END dumpDirStats;

PROCEDURE dumpEntry (lowercase,useLFN,addNL,shortenpaths,onlyempty:BOOLEAN; sepgroup:CHAR;
                     wisize,wicount:CARDINAL;
                     vsize:HUGECARD;count:LONGCARD;
                     sep,sDir:ARRAY OF CHAR  );
VAR
    valstr:str80;
    shortform,longform:pathtype;
    errcode,wi,seplen:CARDINAL;
BEGIN
    seplen:=Str.Length(sep);
    wi:=0;
    IF NOT(onlyempty) THEN
        valstr:=fmthc(vsize,blank,sepgroup,wisize);  print(valstr);print(sep);
        IF shortenpaths THEN INC(wi, Str.Length(valstr));INC(wi,seplen);END;
        valstr:=fmtlc(count,blank,sepgroup,wicount); print(valstr);print(sep);
        IF shortenpaths THEN INC(wi, Str.Length(valstr));INC(wi,seplen);END;
    END;
    IF useLFN THEN
        Str.Copy(shortform,sDir);
        IF w9XshortToLong(shortform,errcode,longform) THEN
            dmppath(addNL,shortenpaths,ENCLOSEME,wi,longform);
        ELSE
            dmppath(addNL,shortenpaths,NOENCLOSE,wi,sDir); (* should never happen *)
        END;
    ELSE
        IF lowercase THEN LowerCase(sDir);END;
        dmppath(addNL,shortenpaths,NOENCLOSE,wi,sDir);
    END;
END dumpEntry;

(* ------------------------------------------------------------ *)

PROCEDURE fmtAboutSubdirs (VAR R:ARRAY OF CHAR;
                          showcount:BOOLEAN;sepgroup:CHAR;count:CARDINAL);
CONST
    widircount=1;
VAR
    valstr:str80;
BEGIN
    IF NOT(showcount) THEN
        Str.Copy(R," and subdirectories (if any)");
    ELSE
        IF count < 2 THEN
            Str.Copy(R, " and ~ subdirectory");
        ELSE
            Str.Copy(R, " and ~ subdirectories");
        END;
        valstr:=fmtc(count,blank,sepgroup,widircount);
        Str.Subst(R,"~",valstr);
    END;
END fmtAboutSubdirs;

PROCEDURE fmtplural (VAR R:ARRAY OF CHAR; morethan1:BOOLEAN;S1,S2:ARRAY OF CHAR);
VAR
    S:str128;
BEGIN
    Str.Copy(R,S1);
    IF morethan1 THEN Str.Append(R,S2);END;
END fmtplural;

(* ------------------------------------------------------------ *)

CONST
    filler          = "$";
    sepa            = "  ";
    graphdash       = CHR(196);  txtdash  = "-";
    graphbar        = CHR(205);  txtbar   = "=";
CONST
    fsizefiller     = "$$$,$$$,$$$,$$$";
    filecountfiller =         "$$$,$$$";
    dircountfiller  =          "$$,$$$";
    pathfiller      = "$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$"; (* 40 *)
CONST
    fsizeheader     = "           Size";
    filecountheader =         "  Files";
    dircountheader  =          "  Dirs";
    pathheader      = "Path";
    msgHSincluded   = " (HSRA)";
    msgHSexcluded   = " (RA)";
    msgHSonly       = " (HS)";

(* ------------------------------------------------------------ *)

PROCEDURE showStats (VAR rcode:CARDINAL;
                    lastpath:CARDINAL;scanmode:scanmodetype;sepgroup,sepdec:CHAR;
                    graphline,lowercase,useLFN,shortenpaths,verbose,terser,wasted:BOOLEAN;
                    onlyempty,killempty,sparebase,DEBUG :BOOLEAN;
                    freebytes,usedbytes,totalbytes:HUGECARD );
CONST
    header1 = fsizeheader+sepa+filecountheader+sepa+pathheader;
    header2 = fsizefiller+sepa+filecountfiller+sepa+pathfiller;
    placeholder="|";
VAR
    i        : CARDINAL;
    base     : str128;
    S,spec   : str256;  (* if VERY long a path *)
    R,bartop,barbot,sdyn: str128;
    found    : BOOLEAN;
    entry    : FIO.DirEntry;
    filecount,entrycount: LONGCARD;
    dirsize  : HUGECARD;
    totalcount:LONGCARD;
    totalsize :HUGECARD;
    ch,ch2    : CHAR;
    valstr    : str80;
    delta     : HUGECARD;
    percentwasted:LONGREAL;
    wifsize,wifilecount,widircount,errcode:CARDINAL;
    ok:BOOLEAN;
    countdirempty,countdirkilled:CARDINAL;
    Z:str1024; (* oversized but who knows ? *)
    shortform,longform:pathtype;
BEGIN
    countdirempty  := 0;
    countdirkilled := 0;

    wifsize    :=Str.Length(fsizefiller);
    wifilecount:=Str.Length(filecountfiller);
    widircount :=Str.Length(dircountfiller);
    IF graphline THEN
        ch := graphdash; ch2 := graphbar;
    ELSE
        ch := txtdash;   ch2 := txtbar;
    END;
    Str.Copy(bartop,header2);
    Str.Copy(barbot,header2);
    ReplaceChar(bartop,filler,ch);
    ReplaceChar(bartop,comma,ch);
    ReplaceChar(barbot,filler,ch2);
    ReplaceChar(barbot,comma,ch2);

    colorheader;

    ok:=( NOT(onlyempty) AND NOT(terser) );
    IF ok THEN
        IF NOT(wasted) THEN
            print(header1);
            CASE scanmode OF
            | normalscan:       print(msgHSexcluded);
            | normalandspecial: print(msgHSincluded);
            | specialscan:      print(msgHSonly);
            END;
            newline;
        END;

        IF verbose THEN
            IF lastpath > firstPath THEN
                print(bartop);
            ELSE
                print(barbot);
            END;
            newline;
        END;
    END;

    colortext;

    FIO.EOF := FALSE; (* safety *)

    totalcount := 0;
    totalsize  := HUGEZERO;
    i := firstPath;
    IF (onlyempty AND sparebase) THEN INC(i);END; (* we won't show empty base directory *)
    LOOP
        IF i > lastpath THEN EXIT; END;
        (* Str.Copy(base,Path[i]); *)
        getentry(i,base);
        Str.Concat(spec,base,stardotstar);                  (* assume u:\xxx\ *)
        entrycount:= 0;
        filecount := 0;
        dirsize   := HUGEZERO;

        found := FIO.ReadFirstEntry(spec,everything,entry);   (* H S R A D *)
        WHILE found DO
            IF (aD IN entry.attr) THEN
                IF isDirEntry(entry.Name)=FALSE THEN (* skip . and .. *)
                    INC(entrycount); (* it's a directory *)
                END;
            ELSE
                CASE scanmode OF
                | specialscan: ok:=((aH IN entry.attr) OR (aS IN entry.attr));
                | normalscan : ok:=NOT ((aH IN entry.attr) OR (aS IN entry.attr));
                | normalandspecial:ok:=TRUE ;
                END;
                IF ok THEN
                    Str.Concat(S,base,entry.Name); (* uppercase *)
                    INC(filecount);
                    HUGEINC(dirsize, HUGECARD(entry.size) );
                END;
                INC(entrycount);
            END;
            found:=FIO.ReadNextEntry(entry);
        END;

        IF verbose THEN
            IF onlyempty THEN
                IF entrycount = 0 THEN (* dirsize may be 0 with empty files *)
                    INC(countdirempty);
                    IF killempty THEN
                        print ("+++ ");
                    END;
                    dumpEntry (lowercase,useLFN,FORCENL,shortenpaths,onlyempty,sepgroup,wifsize,wifilecount,
                              dirsize,filecount,sepa,base);
                    IF killempty THEN
                        INC(countdirkilled); (* assume everything will go fine *)
                        unfixDirectory(base); (* RmDir does not like trailing "\" ! *)
                        FIO.RmDir(base); (* DOS path *)
                        errcode:=FIO.IOresult();
                        IF errcode # DOSErr.NO_ERROR THEN
                            print("--- ERROR CODE ");
                            print( fmtc(errcode,"","",1) );
                            print(" !");
                            newline;
                        END;
                    END;
                END;
            ELSE
                    dumpEntry (lowercase,useLFN,FORCENL,shortenpaths,onlyempty, sepgroup,wifsize,wifilecount,
                              dirsize,filecount,sepa,base);
            END;
        END;
        INC(totalcount,filecount);
        HUGEINC(totalsize,dirsize);

        INC(i);
    END;
    ok:=( NOT(onlyempty) AND NOT(terser) );
    IF ok THEN
        IF wasted THEN
            valstr:=fmthc(totalbytes,blank,sepgroup,wifsize);
            print(valstr);
            fmtplural(R, (totalbytes > HUGEONE), " byte" , "s");
            print(R);
            print(" available on drive ");
            getentry(firstPath,sdyn);
            print(sdyn[0]);print(colon);
            newline;
            valstr:=fmtlc(totalcount,blank,sepgroup,wifsize); (* use longest here *)
            print(valstr);
            fmtplural(R, (totalcount > 1), " file" , "s");
            print(R);
            CASE scanmode OF
            |normalscan :     print(msgHSexcluded); (* never *)
            |normalandspecial:print(msgHSincluded);
            |specialscan:     print(msgHSonly);     (* never *)
            END;
            newline;

            valstr:=fmthc(freebytes,blank,sepgroup,wifsize);
            print(valstr);
            fmtplural (R, (freebytes > HUGEONE), " byte" , "s");
            print(R);
            print(" free"); newline;

            valstr:=fmthc(totalsize,blank,sepgroup,wifsize);
            print(valstr);
            fmtplural (R, (totalsize > HUGEONE), " byte" , "s");
            print(R);
            print(" used"); newline;

            valstr:=fmthc(usedbytes,blank,sepgroup,wifsize);
            print(valstr);
            fmtplural (R, (usedbytes > HUGEONE), " byte" , "s");
            print(R);
            print(" really used (including slack space)"); newline;

            delta := usedbytes-totalsize;
            valstr:=fmthc(delta,blank,sepgroup,wifsize);
            print(valstr);
            fmtplural (R, (delta > HUGEONE), " byte" , "s");
            print(R);
            print(' "wasted" in slack space (');

            IF totalsize = HUGEZERO THEN (* avoid nasty ERRORINF.$$$ ! *)
                percentwasted := 0.0;
            ELSE
                percentwasted := (LONGREAL(delta) / LONGREAL(totalsize) ) * 100.0;
            END;
            Str.Copy(R,fmtpercent(percentwasted,blank,sepdec,5));
            LtrimBlanks(R);
            Str.Append(R,"%)");
            print(R); newline;
        ELSE
            colorheader;

            IF verbose THEN
                IF lastpath > firstPath THEN
                    print(barbot);newline;

                    (* pre v1.0n
                    valstr:=fmthc(totalsize,blank,sepgroup,wifsize);
                    print(valstr);
                    print(sepa);
                    valstr:=fmtlc(totalcount,blank,sepgroup,wifilecount);
                    print(valstr);
                    *)

                    getentry(firstPath,sdyn);
                    dumpEntry (lowercase,useLFN,NONL,NOTSHORTENED,onlyempty, sepgroup,wifsize,wifilecount,
                              totalsize,totalcount,sepa,sdyn);
                    fmtAboutSubdirs(R,TRUE,sepgroup,lastpath-firstPath);
                    print(R);

                    newline;
                END;
            ELSE
                print(barbot);newline;
                getentry(firstPath,sdyn);

                colortext;

                dumpEntry (lowercase,useLFN,NONL,NOTSHORTENED,onlyempty, sepgroup,wifsize,wifilecount,
                          totalsize,totalcount,sepa,sdyn);
                IF lastpath > firstPath THEN
                    fmtAboutSubdirs(R,TRUE,sepgroup,lastpath-firstPath);
                    print(R);
                END;
                newline;
            END;
        END;
    END;

    colortext;

    ok:=( onlyempty AND NOT(terser) );
    IF ok THEN
        IF countdirempty = 0 THEN
            rcode:=errNoEmptyDir;
            IF killempty THEN print ("::: ");END;
            print ("No empty directory found !");newline;  (* show something *)

            IF DEBUG THEN print ("::: rc = 255");newline; END;
        ELSE
            rcode:=errEmptyDirHere;

            IF DEBUG THEN print ("::: rc = 128");newline; END;
        END;
    ELSE
        IF terser THEN
            Z:='o=|  f=|  r=|  |  "|"'; (* fr *)
            Z:='b=|  f=|  d=|  |  "|"';

            valstr:=fmthc(totalsize,blank,sepgroup,wifsize);
            Str.Subst(Z,placeholder,valstr);

            valstr:=fmtlc(totalcount,blank,sepgroup,wifilecount);
            Str.Subst(Z,placeholder,valstr);

            valstr:=fmtc(lastpath-firstPath,blank,sepgroup,widircount);
            Str.Subst(Z,placeholder,valstr);

            CASE scanmode OF
            |normalscan :     valstr:="..RA";
            |normalandspecial:valstr:="HSRA";
            |specialscan:     valstr:="HS..";
            END;
            Str.Subst(Z,placeholder,valstr);

            getentry(firstPath,sdyn);
            IF useLFN THEN
                Str.Copy(shortform,sdyn);
                IF w9XshortToLong(shortform,errcode,longform) THEN
                    Str.Subst(Z,placeholder,longform);
                ELSE
                    Str.Subst(Z,placeholder,sdyn); (* should never happen *)
                END;
            ELSE
                IF lowercase THEN LowerCase(sdyn);END;
                Str.Subst(Z,placeholder,sdyn);
            END;
            print(Z);newline;
        END;
        rcode:=errNone; (* for all *)
    END;

END showStats;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

(* assume trailing "\" *)

PROCEDURE getDirStats (scanmode:scanmodetype; S:ARRAY OF CHAR;
                      VAR totalsize:HUGECARD;
                      VAR nfiles:LONGCARD; VAR ndirs:CARDINAL );
VAR
    entry    : FIO.DirEntry;
    spec     : str256; (* safety *)
    found,ok : BOOLEAN;
BEGIN
    totalsize :=HUGEZERO;
    nfiles    :=0;
    ndirs     :=0;

    Str.Concat(spec,S,stardotstar);
    found := FIO.ReadFirstEntry(spec,everything,entry);
    WHILE found DO
        IF (aD IN entry.attr) THEN
            ok:=same(entry.Name,dot);
            ok:=ok OR (same(entry.Name,dotdot));
            IF NOT(ok) THEN INC(ndirs);END;
        ELSE
            CASE scanmode OF
            | specialscan: ok:=((aH IN entry.attr) OR (aS IN entry.attr));
            | normalscan : ok:=NOT ((aH IN entry.attr) OR (aS IN entry.attr));
            | normalandspecial:ok:=TRUE ;
            END;
            IF ok THEN
                HUGEINC(totalsize, HUGECARD(entry.size) );
                INC(nfiles);
            END;
        END;
        found:=FIO.ReadNextEntry(entry);
    END;
END getDirStats;

PROCEDURE showCumul (VAR rcode:CARDINAL;
                    lastpath:CARDINAL;scanmode:scanmodetype;sepgroup:CHAR;
                    graphline,lowercase,useLFN,shortenpaths:BOOLEAN;
                    dirspec:ARRAY OF CHAR);
CONST
    header1 = fsizeheader+sepa+filecountheader+sepa+dircountheader+sepa+pathheader;
    header2 = fsizefiller+sepa+filecountfiller+sepa+dircountfiller+sepa+pathfiller;
VAR
    bartop,barbot : str128;
    ch,ch2   : CHAR;
    R,S,S0,base : str128;
    totalsize:HUGECARD;
    totalfiles,nfiles,znfiles:LONGCARD;
    totaldirs,ndirs,zndirs:CARDINAL;
    dirsize:HUGECARD;
    zdirsize:HUGECARD;
    refslashes,slashes:CARDINAL;
    i,j:CARDINAL;
    wifsize,wifilecount,widircount:CARDINAL;
BEGIN
    wifsize:=Str.Length(fsizefiller);
    wifilecount:=Str.Length(filecountfiller);
    widircount:=Str.Length(dircountfiller);
    IF graphline THEN
        ch := graphdash; ch2 := graphbar;
    ELSE
        ch := txtdash;   ch2 := txtbar;
    END;
    Str.Copy(bartop,header2);
    Str.Copy(barbot,header2);
    ReplaceChar(bartop,filler,ch);
    ReplaceChar(bartop,comma,ch);
    ReplaceChar(barbot,filler,ch2);
    ReplaceChar(barbot,comma,ch2);

    colorheader;

    print(header1);
    CASE scanmode OF
    |normalscan:       print(msgHSexcluded);
    |normalandspecial: print(msgHSincluded);
    |specialscan:      print(msgHSonly);
    END;
    newline;

    IF lastpath > firstPath THEN
        print(bartop);
    ELSE
        print(barbot);
    END;
    newline;

    colortext;

    Str.Copy(base,dirspec);
    fixDirectory(base);
    refslashes:=CharCount(base,backslash) +1;    (* "base\"+"*\" *)

    totalsize  :=HUGEZERO;
    totalfiles :=0;
    totaldirs  :=0;

    FOR i:=firstPath TO lastpath DO
        getentry(i,S);
        slashes:=CharCount(S,backslash);
        IF slashes < refslashes THEN    (* base *)
            getDirStats(scanmode,S,  dirsize,nfiles,ndirs);
            dumpDirStats (lowercase,useLFN,FORCENL,shortenpaths, sepgroup,wifsize,wifilecount,widircount,
                         dirsize,nfiles,ndirs,sepa,S);
        ELSIF slashes = refslashes THEN (* dir in base *)
            getDirStats(scanmode,S,  dirsize,nfiles,ndirs);
            FOR j:=firstPath TO lastpath DO
                getentry(j,S0);
                slashes:=CharCount(S0,backslash);
                IF ( (slashes > refslashes) AND (Str.Pos(S0,S)=0) ) THEN
                    getDirStats(scanmode,S0,  zdirsize,znfiles,zndirs);
                    HUGEINC(dirsize,zdirsize);
                    INC(nfiles,znfiles);
                    INC(ndirs,zndirs);
                END;
            END;
            dumpDirStats (lowercase,useLFN,FORCENL,shortenpaths, sepgroup,wifsize,wifilecount,widircount,
                         dirsize,nfiles,ndirs,sepa,S);
        ELSE                            (* subdir in dir in base *)
            dirsize:=HUGEZERO;
            nfiles :=0;
            ndirs  :=0;
        END;
        HUGEINC(totalsize,dirsize);
        INC(totalfiles,nfiles);
        INC(totaldirs,ndirs);
    END;

    colorheader;

    IF lastpath > firstPath THEN
        print(barbot);newline;
        dumpDirStats (FALSE, useLFN,NONL,NOTSHORTENED,sepgroup,wifsize,wifilecount,widircount,
                     totalsize,totalfiles,totaldirs,sepa,base);
        IF lastpath > firstPath THEN
            fmtAboutSubdirs(R,FALSE,sepgroup,lastpath-firstPath);
            print(R);
        END;
        newline;
    END;

    colortext;

    rcode:=errNone;
END showCumul;

PROCEDURE getsepgroup ( fr:BOOLEAN  ):CHAR ;
BEGIN
    IF fr THEN RETURN dot; END;
    RETURN comma;
END getsepgroup;

PROCEDURE getsepdec ( fr:BOOLEAN  ):CHAR ;
BEGIN
    IF fr THEN RETURN comma; END;
    RETURN dot;
END getsepdec;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    msgWait = "Working, please wait... ";
VAR
    parmcount,i,opt : CARDINAL;
    S,R             : pathtype;
    state           : (waiting,gotdir);
    recurse,onlynormal,graphline,lowercase,shortenpaths,verbose,terser:BOOLEAN;
    useLFN          : BOOLEAN;
    quiet,wasted,grandtotal,alphasort,onlyempty,killempty,sparebase : BOOLEAN;
    DEBUG           : BOOLEAN;
    dirspec,base    : str128;
    lastdirndx      : CARDINAL;
    filecount       : CARDINAL;
    totalsize       : LONGCARD;
    freebytes,totalbytes,usedbytes : HUGECARD;
    sHD,sCD,sValidUnits:str80;
    cdcount:CARDINAL;
    cdfirst:CHAR;
    shortform,orgdirspec:pathtype;
    errcode:CARDINAL;
    scanmode : scanmodetype;
    sepgroup,sepdec : CHAR;
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE;

    vidinit ( IsRedirected() );
    newline; (* here for pretty output *)
    colortext;

    recurse   := TRUE;
    graphline := TRUE;
    lowercase := TRUE;
    verbose   := TRUE;
    terser    := FALSE;
    wasted    := FALSE;
    grandtotal:= FALSE;
    alphasort := FALSE;
    scanmode  := normalandspecial;
    useLFN    := TRUE;
    onlyempty := FALSE;
    killempty := FALSE;
    sparebase := FALSE;
    quiet     := FALSE;
    sepgroup  := getsepgroup(TRUE);
    sepdec    := getsepdec(TRUE);
    gForceLFNsort:=FALSE;
    shortenpaths := FALSE;
    DEBUG     := FALSE;

    parmcount := Lib.ParamCount();
    IF parmcount=0 THEN abort(errHelp,""); END;

    state := waiting;

    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i);
        Str.Copy(R,S);
        UpperCase(R);
        cleantabs(R);
        IF isOption(R) THEN
            opt := GetOptIndex(R,"?"+delim+"H"+delim+"HELP"+delim+
                                 "X"+delim+"EXCLUDE"+delim+
                                 "N"+delim+"NORMAL"+delim+
                                 "A"+delim+"ASCII"+delim+
                                 "U"+delim+"UPPERCASE"+delim+
                                 "T"+delim+"TERSE"+delim+
                                 "W"+delim+"WASTED"+delim+
                                 "D"+delim+"DD"+delim+
                                 "P"+delim+"PATH"+delim+
                                 "L"+delim+"LFN"+delim+
                                 "E"+delim+"EMPTY"+delim+
                                 "K"+delim+"KILL"+delim+"DEL"+delim+
                                 "R"+delim+"KILLSUBDIRS"+delim+
                                 "DEBUG"+delim+
                                 "Q"+delim+"QUIET"+delim+
                                 "F"+delim+"US"+delim+"UK"+delim+
                                 "Z"+delim+"SPECIAL"+delim+
                                 "I"+delim+"LFNPATH"+delim+"PP"+delim+
                                 "C"+delim+"COLUMNS"+delim+
                                 "TT"
(*%T CONSOLE  *)
                                                    +delim+
                                 "B"+delim+"BIOS"
(*%E  *)
                              );
            CASE opt OF
            | 1,2,3 :   abort(errHelp,"");
            | 4,5:      recurse    := FALSE;
            | 6,7:      CASE scanmode OF
                        | normalandspecial,normalscan:
                            scanmode:=normalscan;
                        ELSE
                            abort(errExclusive,"");
                        END;
            | 8,9:      graphline  := FALSE;
            | 10,11:    lowercase  := FALSE;
            | 12,13:    verbose    := FALSE;
            | 14,15:    wasted     := TRUE;
            | 16,17:    grandtotal := TRUE;
            | 18,19:    alphasort  := TRUE;
            | 20,21:    useLFN     := FALSE;
            | 22,23:    onlyempty  := TRUE;
            | 24,25,26: killempty  := TRUE; onlyempty:=TRUE;
            | 27,28:    killempty  := TRUE; onlyempty:=TRUE; sparebase:=TRUE;
            | 29:       DEBUG      := TRUE;
            | 30,31:    quiet      := TRUE;
            | 32,33,34: sepgroup   := getsepgroup(FALSE);
                        sepdec     := getsepdec(FALSE);
            | 35,36:    CASE scanmode OF
                        | normalandspecial,specialscan:
                            scanmode:=specialscan;
                        ELSE
                            abort(errExclusive,"");
                        END;
            | 37,38,39: alphasort  := TRUE; gForceLFNsort:=TRUE;
            | 40,41:    shortenpaths := TRUE;
            | 42:       verbose :=FALSE; terser:=TRUE;
(*%T CONSOLE  *)
            | 43,44:    vidinit( TRUE);
(*%E  *)
            ELSE
                abort(errUnknownOption,S);
            END;
        ELSE
            CASE state OF
            | waiting   : Str.Copy(orgdirspec,S);
            | gotdir    : abort(errParmOverflow,S);
            END;
            INC(state);
        END;
    END;
    IF state = waiting THEN abort(errMissingDirSpec,"");END;

    useLFN:=( useLFN AND w9XsupportLFN() );

    getCurrentPath(base); (* "u:\path\" *)
    IF same(orgdirspec,dot) THEN Str.Copy(orgdirspec,base);END;

    IF chkJokersHere(orgdirspec) THEN abort(errJokerDir,""); END;
    IF chkValidName(orgdirspec)=FALSE THEN abort(errBadNameDir,"");END;
    IF useLFN THEN
        IF w9XlongToShort(orgdirspec, errcode, shortform) THEN
            Str.Copy(dirspec,shortform);
        ELSE
            Str.Copy(dirspec,orgdirspec); (* we'll force an error *)
        END;
    ELSE
        Str.Copy(dirspec,orgdirspec);
    END;
    UpperCaseAlt(dirspec); (* //V10Q fix : was UpperCase *)
    fixUnit(wasted,dirspec);
    IF buildCanonicalPath(base,dirspec)=FALSE THEN abort(errBadPath,dirspec);END;
    IF isDirectory(dirspec)=FALSE THEN abort(errBadDir,dirspec);END;

    getCDROMletters(cdcount,cdfirst,sCD);
    getAllHDunits(sHD);
    Str.Concat(sValidUnits,"AB",sHD);Str.Append(sValidUnits,sCD);

    IF Str.CharPos( sValidUnits, dirspec[0] ) = MAX(CARDINAL) THEN
        abort(errPhantomUnit,dirspec[0]);
    END;

    totalbytes := HUGEZERO; (* safety *)
    usedbytes  := HUGEZERO; (* safety *)
    freebytes  := HUGEZERO; (* safety *)
    IF wasted THEN
        IF isRoot(dirspec)=FALSE THEN abort(errRootOnly,dirspec);END;
        IF NOT(recurse) THEN abort(errNonsense,"-x");END;
        CASE scanmode OF
        | normalscan: abort(errNonsense,"-n");
        | specialscan:abort(errNonsense,"-z");
        END;
        IF grandtotal THEN abort(errNonsense,"-d");END;
        IF NOT(verbose) THEN abort(errNonsense,"-t[t]");END; (* useless in fact *)
        IF onlyempty    THEN abort(errNonsense,"-e");END;
        verbose:=FALSE;

        IF getUnitFreeTotal (dirspec[0],freebytes,totalbytes)=FALSE THEN
            abort(errInvalid,dirspec);
        END;
        usedbytes := totalbytes - freebytes;
    END;
    IF grandtotal THEN
        IF NOT(recurse) THEN abort(errNonsensical,"-x");END;
        IF NOT(verbose) THEN abort(errNonsensical,"-t[t]");END;
        IF onlyempty    THEN abort(errNonsensical,"-e");END;
    END;
    IF onlyempty THEN
        IF NOT(verbose) THEN abort(errCmdEmpty,"-t[t]");END;
        IF grandtotal   THEN abort(errCmdEmpty,"-d");END;
        CASE scanmode OF
        | normalscan:        abort(errCmdEmpty,"-n");
        | specialscan:       abort(errCmdEmpty,"-z");
        END;
    END;

    IF NOT(quiet) THEN video(msgWait,TRUE); END;
    IF buildPathList(dirspec,recurse,DEBUG,lastdirndx)=FALSE THEN
        abort(errTooManyDirs,dirspec);
    END;
    IF ( killempty = FALSE ) THEN
        IF alphasort THEN Lib.QSort(lastdirndx,isLessAlpha,doSwap); END;
    END;
    IF NOT(quiet) THEN video(msgWait,FALSE); END;

    IF grandtotal THEN
        showCumul (errcode,
                  lastdirndx,scanmode,sepgroup,
                  graphline,lowercase,useLFN,shortenpaths,
                  dirspec);
    ELSE
        showStats (errcode,
                  lastdirndx,scanmode,sepgroup,sepdec,
                  graphline,lowercase,useLFN,shortenpaths,verbose,terser,wasted,
                  onlyempty,killempty,sparebase,DEBUG,
                  freebytes,usedbytes,totalbytes);
    END;

    abort(errcode,"");
END total.



