(* ---------------------------------------------------------------
Title         Q&D set video text mode (VGA OR VESA)
Author        PhG
Overview      self-explanatory !
Usage         see help
Notes         very, very, very quick & dirty... :-(
              minimal error messages and checking, etc.
              VGA assumed !
Bugs          cursor is not always correct !
Wish List     handle w*h*bpp in addition to $mode ? bah, what for ?

--------------------------------------------------------------- *)

MODULE SetVmode;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetString, CharCount, same,
aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, cleantabs, str16;

IMPORT Lib;
IMPORT Str;
IMPORT IO;
IMPORT SYSTEM;

FROM IO IMPORT WrStr, WrLn;

(* ------------------------------------------------------------ *)

CONST
    ALTERNATE     = TRUE;
    dash          = "-";
    star          = "*";
    mul           = "X";
    space         = " ";
    cr = CHR(13);
    lf = CHR(10);
    nl = cr+lf;
CONST
    sFMAM = "$81ff";
    fullMemoryAccessMode = 081FFH; (* as defined by VESA *)
    showAllModes         = MAX(CARDINAL);

CONST
    ProgEXEname   = "SETVMODE";
    ProgTitle     = "Q&D set video text mode (VGA or VESA)";
    ProgVersion   = "v1.0j";
    ProgCopyright = "by PhG";
    Banner        = ProgTitle+" "+ProgVersion+" "+ProgCopyright;
CONST
    errNone         = 0;
    errHelp         = 1;
    errOption       = 2;
    errBadParm      = 3;
    errNoVesa       = 4;
    errNotAvailable = 5;
    errVesaProblem  = 6;
    errVGA          = 7;
    errSyntax       = 8;
    errUnset        = 9;
    errParmOverflow = 10;
    errIllegalValue = 11;
    errBadModeValue = 12;
    errBadShape     = 13;
    errNotVesaMode  = 14;
    errNonsense     = 15;

(* ------------------------------------------------------------ *)

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)
    msgHelp =
nl+
Banner+nl+
nl+
"Syntax 1 : "+ProgEXEname+" < 80x25 |  80x28 |  80x30 |  80x43 |  80x50 >"+nl+
"           "+ProgEXEname+" < 80x60 | 132x25 | 132x43 | 132x50 | 132x60 >"+nl+
"           "+ProgEXEname+" <  $108 |   $109 |   $10a |   $10b |   $10c >"+nl+
nl+
"Syntax 2 : "+ProgEXEname+" <[$|0x]mode[h]> [-c|-c:start,end]"+nl+
nl+
"Syntax 3 : "+ProgEXEname+" <-i[vbe2] [-v[v[v]]] [-x] | -v:[$|0x]mode[h]>"+nl+
nl+
"Syntax 4 : "+ProgEXEname+" <-list>"+nl+
nl+
"Note that a VESA text mode may be available, and yet be unsupported by BIOS."+nl+
"Standard VGA BIOS is used for [$00..$ff], VESA BIOS for [$100..$1ff,"+sFMAM+"]."+nl;
VAR
    S : str256;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(msgHelp);
    | errOption :
        Str.Concat(S,"Unknown ",einfo);Str.Append(S," option !");
    | errBadParm :
        Str.Concat(S,"Unsupported ",einfo);Str.Append(S," parameter !");
    | errNoVesa :
        S := "VESA BIOS not found !";
    | errNotAvailable :
        S := "Required VESA text mode not available !";
    | errVesaProblem :
        S := "A problem occurred while trying to set required VESA text mode !";
    | errVGA :
        Str.Concat(S,einfo,"-lines mode does not seem to be available !");
    | errSyntax:
        S := "-i option cannot be specified when resetting video mode !";
    | errUnset:
        S := "Neither command nor video text mode setting were specified !";
    | errParmOverflow:
        Str.Concat(S,"Useless ",einfo);Str.Append(S," parameter !");
    | errIllegalValue:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," value !");
    | errBadModeValue:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," mode !");
    | errBadShape:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," cursor shape !");
    | errNotVesaMode:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," video mode !");
    | errNonsense:
        S:="-x and -vv options are mutually exclusive !";
    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone, errHelp :
        ;
    ELSE
        WrLn;
        WrStr(ProgEXEname+" : ");WrStr(S);WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)

PROCEDURE getCardValue (VAR v:CARDINAL;S:ARRAY OF CHAR):BOOLEAN;
VAR
    n:LONGCARD;
    p,base:CARDINAL;
    ok:BOOLEAN;
BEGIN
    Str.Caps(S); (* just in case ! *)
    IF Str.Match(S,"$*") THEN
        Str.Delete(S,0,1);
        base:=16;
    ELSIF Str.Match(S,"0X*") THEN
        Str.Delete(S,0,2);
        base:=16;
    ELSIF Str.Match(S,"*H") THEN
        p:=Str.Length(S);
        Str.Delete(S,p-1,1);
        base:=16;
    ELSE
        base:=10;
    END;
    n:=Str.StrToCard(S,base,ok);
    IF ok=FALSE THEN RETURN FALSE; END;
    IF n > MAX(CARDINAL) THEN RETURN FALSE; END;
    v:=CARDINAL(n);
    RETURN TRUE;
END getCardValue;

(* "#,#" "#:#" "#-#" "#!#" "#..#" *)

PROCEDURE getBoth (VAR nstart,nend:CARDINAL;S:ARRAY OF CHAR):BOOLEAN;
VAR
    n : LONGCARD;
    ok : BOOLEAN;
    p : CARDINAL;
    V : str16;
    tst,sep:str2;
BEGIN
    sep:="";
    FOR p:=1 TO 5 DO
        IF same(sep,"") THEN
            CASE p OF
            | 1: tst:= ",";
            | 2: tst:= ":";
            | 3: tst:= "-";
            | 4: tst:= "!";
            | 5: tst:= "..";
            END;
            IF Str.Pos(S,tst) # MAX(CARDINAL) THEN sep:=tst; END;
        END;
    END;
    IF same(sep,"") THEN RETURN FALSE; END;
    p:=Str.Pos(S,sep);
    Str.Slice(V,S,0,p);
    n:=Str.StrToCard(V,10,ok);
    IF ok=FALSE THEN RETURN FALSE; END;
    IF n > MAX(CARDINAL) THEN RETURN FALSE; END;
    nstart:=CARDINAL(n);

    Str.Delete(S,0,p+Str.Length(sep));
    Str.Copy(V,S);
    n:=Str.StrToCard(V,10,ok);
    IF ok=FALSE THEN RETURN FALSE; END;
    IF n > MAX(CARDINAL) THEN RETURN FALSE; END;
    nend:=CARDINAL(n);
    RETURN TRUE;
END getBoth;

(* ------------------------------------------------------------ *)

PROCEDURE dump ( addr : ADDRESS ; count:CARDINAL);
CONST
    n = 16;
    w4= 4;
    w2= 2;
VAR
    i,offset : CARDINAL;
    S,S2 : str16;
    ok: BOOLEAN;
    b : SHORTCARD;
    ch: CHAR;
BEGIN
    WrStr("{");WrLn;
    offset := 0;
    LOOP
        IF (offset MOD n) = 0 THEN
            Str.CardToStr( LONGCARD(offset), S, 16, ok);
            FOR i:=(Str.Length(S)+1) TO w4 DO Str.Prepend(S,"0"); END;
            Str.Append(S," :");
            Str.Lows(S);
            WrStr(S);
            S2:="";
        END;

        Lib.Move( addr, ADR(b), SIZE(b));
        Str.CardToStr( LONGCARD(b), S, 16, ok);
        FOR i:=(Str.Length(S)+1) TO w2 DO Str.Prepend(S,"0");END;
        Str.Lows(S);
        WrStr(" ");WrStr(S);
        Lib.IncAddr(addr,SIZE(b));

        CASE CARDINAL(b) OF
        | 0..(ORD(" ")-1), 255 : ch:=".";
        ELSE                   ch:=CHR(b);
        END;
        Str.Append(S2,ch);

        IF (offset MOD n) = (n-1) THEN
            WrStr(" | ");WrStr(S2);
            WrLn;
        END;
        INC(offset);
        DEC(count);
        IF count = 0 THEN EXIT; END;
    END;
    WrStr("}");WrLn;
END dump;

(* ------------------------------------------------------------ *)

CONST
    CO80 = BYTE(3); (* color 80x25 *)
    MONO = BYTE(7); (* monochrome *)
CONST
    iv = 010H; (* video interrupt is $10 *)
    sb = 040H; (* segBiosData *)
VAR
    biosCurrentVideoMode  [sb:049H]  : BYTE;
    columnsOnScreen       [sb:004AH] : CARDINAL; (* a WORD *)

PROCEDURE isMonoMode ():BOOLEAN;
BEGIN
    RETURN (biosCurrentVideoMode = MONO);
END isMonoMode;

PROCEDURE setVideoMode (mode:SHORTCARD);
VAR
    R : SYSTEM.Registers;
BEGIN
    (* set video mode *)
    R.AH := 00H;
    R.AL := mode;
    Lib.Intr(R,iv);
END setVideoMode;

PROCEDURE set25LineMode () : BOOLEAN;
VAR
    R : SYSTEM.Registers;
BEGIN
    IF isMonoMode() THEN RETURN FALSE; END;
    (* select vertical resolution vga *)
    R.AH := 12H;
    R.BL := 30H;
    R.AL := 02H;        (* 0=200, 1=350, 2=400 *)
    Lib.Intr(R,iv);     (* al=$12 if function supported *)
    IF R.AL # 12H THEN RETURN FALSE; END;
    setVideoMode(CO80);
    RETURN TRUE;
END set25LineMode;

PROCEDURE set28LineMode (  ) : BOOLEAN;
VAR
    R : SYSTEM.Registers;
BEGIN
    IF isMonoMode() THEN RETURN FALSE; END;
    (* select vertical resolution vga *)
    R.AH := 12H;
    R.BL := 30H;
    R.AL := 02H;        (* 0=200, 1=350, 2=400 *)
    Lib.Intr(R,iv);     (* al=$12 if function supported *)
    IF R.AL # 12H THEN RETURN FALSE; END;
    setVideoMode (CO80);
    (* load 8x14 ROM monochrome patterns EGA/VGA *)
    R.AH := 11H;
    R.AL := 11H;
    R.BL := 00H;        (* load block 0 *)
    Lib.Intr(R,iv);
    RETURN TRUE;
END set28LineMode;

PROCEDURE set43LineMode (  ) : BOOLEAN;
VAR
    R : SYSTEM.Registers;
BEGIN
    IF isMonoMode() THEN RETURN FALSE; END;
    (* select vertical resolution vga *)
    R.AH := 12H;
    R.BL := 30H;
    R.AL := 01H;        (* 0=200, 1=350, 2=400 *)
    Lib.Intr(R,iv);     (* al=$12 if function supported *)
    IF R.AL # 12H THEN RETURN FALSE; END;
    setVideoMode(CO80);
    (* load rom 8x8 dbl-dot patterns *)
    R.AH := 11H;
    R.AL := 12H;
    R.BL := 00H;        (* load block 0 *)
    Lib.Intr(R,iv);
    RETURN TRUE;
END set43LineMode;

PROCEDURE set50LineMode (  ) : BOOLEAN;
VAR
    R : SYSTEM.Registers;
BEGIN
    IF isMonoMode() THEN RETURN FALSE; END;
    (* select vertical resolution vga *)
    R.AH := 12H;
    R.BL := 30H;
    R.AL := 02H;        (* 0=200, 1=350, 2=400 *)
    Lib.Intr(R,iv);     (* al=$12 if function supported *)
    IF R.AL # 12H THEN RETURN FALSE; END;
    setVideoMode(CO80);
    (* load rom 8x8 dbl-dot patterns *)
    R.AH := 11H;
    R.AL := 12H;
    R.BL := 00H;        (* load block 0 *)
    Lib.Intr(R,iv);
    RETURN TRUE;
END set50LineMode;

(* ------------------------------------------------------------ *)

(*%F ALTERNATE *)

(* port from ASM code found on Internet : hardly understandable ! *)
(* slightly checked in Ralf Brown's Interrupt List *)

PROCEDURE set30LineMode (  ):BOOLEAN;
VAR
    R : SYSTEM.Registers;
    biosCurrentRows     [sb:084H] : BYTE;
    biosCurrentPageSize [sb:04CH] : WORD; (* PAGE (REGEN BUFFER) SIZE IN BYTES  *)
    biosCRTCport        [sb:063H] : WORD; (* CRT CONTROLLER BASE I/O PORT ADDRESS *)
    addr : CARDINAL;
    n:SHORTCARD;
BEGIN
    IF set25LineMode()=FALSE THEN RETURN FALSE; END; (* safety ! *)

    R.AX := 1A00H;      (* GET DISPLAY COMBINATION CODE (PS,VGA/MCGA) *)
    Lib.Intr(R,iv);     (* al=$1A if function supported *)
    IF R.AL # 1AH THEN RETURN FALSE; END;
    CASE R.BL OF (* active display code *)
    | 7 : ; (* VGA monochrome *)
    | 8 : ; (* VGA color *)
    ELSE
        RETURN FALSE;
    END;

    biosCurrentPageSize := 8192; (* 80x50x2=8000 ! *)
    biosCurrentRows     := 30-1;
    addr := biosCRTCport;

    SYSTEM.DI;

    SYSTEM.OutW(addr,00C11H); (* register $11 : end vertical retrace *)
    SYSTEM.OutW(addr,00D06H); (* register $06 : vertical displayed *)
    SYSTEM.OutW(addr,03E07H); (* register $07 : vertical sync pulse width-1 *)
    SYSTEM.OutW(addr,0EA10H); (* register $10 : start vertical retrace *)
    SYSTEM.OutW(addr,08C11H); (* register $11 : end vertical retrace *)
    SYSTEM.OutW(addr,0DF12H); (* register $12 : vertical display end register *)
    SYSTEM.OutW(addr,0E715H); (* register $15 : start vertical blanking-1 *)
    SYSTEM.OutW(addr,00616H); (* register $16 : end vertical blanking register *)

    addr := 003CCH; (* miscellaneous output register  *)
    n:=SYSTEM.In(addr);
    n:=n AND 033H;
    n:=n  OR 0C4H;
    addr := 003C2H; (* miscellaneous output register *)
    SYSTEM.Out(addr,n);

    SYSTEM.EI;

    R.AH := 012H; (* ALTERNATE FUNCTION SELECT (PS,EGA,VGA,MCGA) - ALTERNATE PRTSC *)
    R.BL := 020H;
    Lib.Intr(R,iv);
    RETURN TRUE;
END set30LineMode;

(*%E  *)

(*%T ALTERNATE  *)

PROCEDURE set30LineMode (  ):BOOLEAN;
VAR
    biosCRTCport        [sb:063H] : WORD; (* CRT CONTROLLER BASE I/O PORT ADDRESS *)
    biosCurrentRows     [sb:084H] : BYTE;
    R : SYSTEM.Registers;
    addr : CARDINAL;
    n : SHORTCARD;
BEGIN
    IF set25LineMode()=FALSE THEN RETURN FALSE; END; (* safety ! *)
    (* load rom 8x16 dbl-dot patterns *)
    R.AH := 11H;
    R.AL := 14H;
    R.BL := 00H;        (* load block 0 *)
    Lib.Intr(R,iv);

    SYSTEM.DI;

    addr := 003CCH; (* miscellaneous output register  *)
    n:=SYSTEM.In(addr);

    n:=n OR 3*64;   (* set scanline type 3 - 480 *)
    addr := 003C2H; (* miscellaneous output register *)
    SYSTEM.Out(addr,n);

    addr := biosCRTCport;
    SYSTEM.Out(addr,11H); (* register $11 : end vertical retrace *)
    INC(addr);
    n:=011H AND 070H; (* 11 AND (NOT(15+128)) *)
    n:=n OR 12;
    SYSTEM.Out(addr, n);

    DEC(addr); (* now set vertical 480 parameters *)
    SYSTEM.OutW(addr, 00B06H);
    SYSTEM.OutW(addr, 03E07H);
    SYSTEM.OutW(addr, 0EA10H);
    SYSTEM.OutW(addr, 0DF12H);
    SYSTEM.OutW(addr, 0E715H);
    SYSTEM.OutW(addr, 00416H);

    SYSTEM.Out (addr,11H);
    INC(addr);
    SYSTEM.Out(addr,n); (* enable protection *)

    biosCurrentRows     := 30-1;

    SYSTEM.EI;

RETURN TRUE;
END set30LineMode;

(*%E  *)

(* ------------------------------------------------------------ *)

(* bugfix *)

(*# save *)
(*# data(near_ptr => off)  *)
TYPE
    vesapointer = POINTER TO WORD; (* well, glad M2 is case-sensitive ! *)
    pchar       = POINTER TO CHAR;
(*# restore *)

TYPE
    vesamainbuffertype = RECORD
        id             : ARRAY [0..3] OF CHAR;
        majmin         : WORD;
        pOEMname       : LONGWORD;  (* asciiz *)
        capabilities   : LONGWORD;
        pList          : LONGWORD;
        VRAM64KBblocks :WORD;
        OEMmajmin      : WORD;
        pOEM           : LONGWORD;
        pProduct       : LONGWORD;
        pRevision      : LONGWORD;
        VBEAFversion   : WORD;
        pAcceleratedModes:LONGWORD;
        dummy          : ARRAY [1..216] OF BYTE;
        OEMdummy       : ARRAY [1..256] OF BYTE;
    END;

VAR
    Buffer : vesamainbuffertype;

(*
CONST
    firstBufferByte = 0;
    lastBufferByte = 768-1; (* 511 should be enough but... *)
VAR
    Buffer : ARRAY [firstBufferByte..lastBufferByte] OF BYTE;
*)

PROCEDURE CheckVesaHere (forceVBE2:BOOLEAN;VAR listPtr : vesapointer) : BOOLEAN;
VAR
    R : SYSTEM.Registers;
BEGIN
    IF forceVBE2 THEN
        Buffer.id[0]:="V";
        Buffer.id[1]:="B";
        Buffer.id[2]:="E";
        Buffer.id[3]:="2";
    END;
    R.AX := 4F00H;
    R.ES := Seg(Buffer); (* buffer segment *)
    R.DI := Ofs(Buffer); (* buffer offset *)
    Lib.Intr(R,10H);
    IF R.AL # 4FH THEN RETURN FALSE; END;
    IF R.AH # 00H THEN RETURN FALSE; END;
    IF Buffer.id[0] # "V" THEN RETURN FALSE; END;
    IF Buffer.id[1] # "E" THEN RETURN FALSE; END;
    IF Buffer.id[2] # "S" THEN RETURN FALSE; END;
    IF Buffer.id[3] # "A" THEN RETURN FALSE; END;
    (* Lib.Move (ADR(Buffer[0EH]),ADR(listPtr),4); (* set pointer to list ADDRESS *) *)
    Lib.Move (ADR(Buffer.pList),ADR(listPtr),SIZE(Buffer.pList));
    RETURN TRUE;
END CheckVesaHere;

(* warning ! a mode may be supposed "available" while not being implemented ! matrox idiosyncrasy ! *)

PROCEDURE ModeAvailable (listPtr : vesapointer; mode : CARDINAL) : BOOLEAN;
BEGIN
    LOOP
        IF listPtr^ = WORD(0FFFFH) THEN EXIT; END;
        IF listPtr^ = WORD(mode) THEN RETURN TRUE; END;
        Lib.IncFarAddr(listPtr,SIZE(WORD));
    END;
    RETURN FALSE;
END ModeAvailable;

PROCEDURE SetVesaMode (mode : CARDINAL) : BOOLEAN;
VAR
    R : SYSTEM.Registers;
BEGIN
    R.AX := 4F02H;
    R.BX := mode;
    Lib.Intr(R,10H);
    IF R.AL # 4FH THEN RETURN FALSE; END;
    IF R.AH # 00H THEN RETURN FALSE; END;
    RETURN TRUE;
END SetVesaMode;

TYPE
    capflagtype = SET OF [0..31]; (* LONGWORD *)

PROCEDURE infvesa (flag:capflagtype;bit:CARDINAL;msg:ARRAY OF CHAR);
BEGIN
    WrStr("bit ");IO.WrCard(bit,2);WrStr(" ");
    IF bit IN flag THEN
        WrStr("ON  : ");
    ELSE
        WrStr("off : ");
    END;
    WrStr(msg);WrLn;
END infvesa;

(* ------------------------------------------------------------ *)

CONST
    prefixbin = "%"; padbin = "0";
    prefixdec =  ""; paddec = ""; (* was " " but we no longer pad decimal values *)
    prefixhex = "$"; padhex = "0";
    prefixhuh = "?"; padhuh = "?";

PROCEDURE fmtit (wi,base:CARDINAL;v:LONGCARD;VAR N:ARRAY OF CHAR);
VAR
    i:CARDINAL;
    ok:BOOLEAN;
    prefix,pad:CHAR;
BEGIN
    CASE base OF
    | 0: prefix:=prefixdec; pad:=space;  base:=10; (* special decimal *)
    | 2: prefix:=prefixbin; pad:=padbin;
    |10: prefix:=prefixdec; pad:=paddec;
    |16: prefix:=prefixhex; pad:=padhex;
    ELSE prefix:=prefixhuh; pad:=padhuh;
    END;
    Str.CardToStr(v,N,base,ok);
    IF base=16 THEN Str.Lows(N);END;
    FOR i:=1 TO wi DO
        IF Str.Length(N) < wi THEN Str.Prepend(N,pad);END;
    END;
    Str.Prepend(N,prefix);
END fmtit;

CONST
    header     = "  ";
    middle     = ": ";
    middleplus = "+ ";
    maxlen     = 49;

PROCEDURE reminf (S:ARRAY OF CHAR);
VAR
    i :CARDINAL;
BEGIN
    WrStr(header);
    WrStr(header);
    WrStr(middleplus);
    WrStr(S);WrLn;
END reminf;

PROCEDURE reminfvesa (flag:capflagtype;bit:CARDINAL;msg:ARRAY OF CHAR);
CONST
    w2 = 2;
VAR
    S:str128;
    i:CARDINAL;
    ok:BOOLEAN;
BEGIN
    Str.CardToStr( LONGCARD(bit), S, 10, ok);
    FOR i:=(Str.Length(S)+1) TO w2 DO Str.Prepend(S," ");END;
    Str.Prepend(S,"bit ");
    Str.Append(S," ");
    IF bit IN flag THEN
        Str.Append(S,"ON  : ");
    ELSE
        Str.Append(S,"off : ");
    END;
    Str.Append(S,msg);
    reminf(S);
END reminfvesa;

PROCEDURE binf (base:CARDINAL;S:ARRAY OF CHAR;v:SHORTCARD);
VAR
    R : str128;
    N : str16;
    i,wi : CARDINAL;
BEGIN
    Str.Concat(R,header,S);
    FOR i:= 1 TO maxlen DO
        IF Str.Length(R) < maxlen THEN Str.Append(R," ");END;
    END;

    CASE base OF
    | 2: wi:=8;
    |10: wi:=3;
    |16: wi:=2;
    ELSE wi:=8;
    END;
    fmtit (wi,base,LONGCARD(v),N);

    WrStr(R);WrStr(middle);WrStr(N);WrLn;
END binf;

PROCEDURE winf (base:CARDINAL;S:ARRAY OF CHAR;v:CARDINAL);
VAR
    R : str128;
    N : str16;
    i,wi : CARDINAL;
BEGIN
    Str.Concat(R,header,S);
    FOR i:= 1 TO maxlen DO
        IF Str.Length(R) < maxlen THEN Str.Append(R," ");END;
    END;

    CASE base OF
    | 2: wi:=16;
    |10: wi:=5;
    |16: wi:=4;
    ELSE wi:=16;
    END;
    fmtit (wi,base,LONGCARD(v),N);

    WrStr(R);WrStr(middle);WrStr(N);WrLn;
END winf;

PROCEDURE dwinf (base:CARDINAL;S:ARRAY OF CHAR;v:LONGCARD);
VAR
    R : str128;
    N : str16;
    i,wi : CARDINAL;
BEGIN
    Str.Concat(R,header,S);
    FOR i:= 1 TO maxlen DO
        IF Str.Length(R) < maxlen THEN Str.Append(R," ");END;
    END;

    CASE base OF
    |10: wi:=12;
    |16: wi:=8;
    ELSE wi:=8;
    END;
    fmtit (wi,base,LONGCARD(v),N);

    WrStr(R);WrStr(middle);WrStr(N);WrLn;
END dwinf;

PROCEDURE getasciiz (a:LONGWORD;VAR R:ARRAY OF CHAR);
VAR
    p  : pchar;
    ch : CHAR;
    i  : CARDINAL;
BEGIN
    p  :=a;
    Str.Copy(R,"");
    LOOP
        ch:=p^;
        Str.Append(R,ch);
        IF ch=CHR(0) THEN EXIT; END;
        Lib.IncFarAddr( p,1 );
    END;
END getasciiz;

(* ------------------------------------------------------------ *)

(* vesamodeinfobuffertype MUST be 256 bytes *)

TYPE
    vesamodeinfobuffertype = RECORD
        modeattributes      : WORD; (* 79 *)
        wA                  : BYTE; (* 81 *)
        wB                  : BYTE; (* id *)
        wgranularityKB      : WORD;
        wsizeKB             : WORD;
        segmentwA           : WORD; (* 0 if unsupported *)
        segmentwB           : WORD; (* id *)
        wpositioningfunction: LONGWORD;
        bytesperscanline    : WORD;
    (* ---remainder is optional for VESA modes in v1.0/1.1, needed for OEM modes--- *)
        width               : WORD; (* pixels or chars *)
        height              : WORD; (* id *)
        wcell               : BYTE;
        hcell               : BYTE;
        memplanes           : BYTE;
        bpp                 : BYTE;
        banks               : BYTE;
        memorymodeltype     : BYTE; (* 82 *)
        banksizeKB          : BYTE;
        imagepages          : BYTE; (* -1 *)
        reservedflag        : BYTE; (* 1 for VBE 3.0, ELSE 0 *)
    (* ---VBE v1.2+ --- *)
        rmasksize           : BYTE;
        rfield              : BYTE;
        gmasksize           : BYTE;
        gfield              : BYTE;
        bmasksize           : BYTE;
        bfield              : BYTE;
        reservedmasksize    : BYTE;
        reservedmaskposition: BYTE;
        directcolormode     : BYTE; (* bits 0 and 1 *)
    (* ---VBE v2.0+ --- *)
        linearvideobufferaddr: LONGWORD;
        offscreenmemoryaddr : LONGWORD;
        offscreenmemoryKB   : WORD;
    (* ---VBE v3.0 --- *)
        bytesperscanlinelinear:WORD;
        imagesforbankedmodes: BYTE; (* -1 *)
        imagesforlinearmodes: BYTE; (* -1 *)
        rmasklinear         : BYTE;
        rmaskLSBlinear      : BYTE;
        gmasklinear         : BYTE;
        gmaskLSBlinear      : BYTE;
        bmasklinear         : BYTE;
        bmaskLSBlinear      : BYTE;
        reservedmasklinear  : BYTE;
        reservedmaskLSBlinear:BYTE;
        maxclockHz          : LONGWORD;
        unused              : ARRAY [1..190] OF BYTE;
    END;

PROCEDURE showVESAmodeInfos (mode:CARDINAL;hexdump,shortfmt:BOOLEAN;strMode:ARRAY OF CHAR);
VAR
    R : SYSTEM.Registers;
    b : vesamodeinfobuffertype;
    S,S1,msg: str128;
    fl: capflagtype;
    bitsum:CARDINAL;
BEGIN
    IF shortfmt THEN
        Str.Concat(msg,strMode," :");
    ELSE
        Str.Concat(msg,nl+"VESA mode ",strMode);Str.Append(msg," :"); (* nl+nl *)
    END;

    Lib.Fill(ADR(b),SIZE(b),00H);

    R.AX := 4F01H;      (* GET SuperVGA MODE INFORMATION *)
    R.CX := WORD(mode);
    R.ES := Seg(b); (* buffer segment *)
    R.DI := Ofs(b); (* buffer offset *)
    Lib.Intr(R,10H);
IF shortfmt THEN (* don't betray modes which are listed but not supported *)
    IF R.AL # 4FH THEN
        (* WrStr("listed, yet unsupported");WrLn; *)
        RETURN;
    END;
    IF R.AH # 00H THEN
        (* WrStr("listed, yet unsupported");WrLn; *)
        RETURN;
    END;
    WrStr(msg);
ELSE
    WrStr(msg);
    IF R.AL # 4FH THEN
        WrStr(nl+nl+header+"104F01 is an unsupported function !"+nl);
        RETURN;
    END;
    IF R.AH # 00H THEN
        WrStr(nl+nl+header+"104F01 function failed !"+nl);
        RETURN;
    END;
END;
    WrStr(" ");
    fmtit (4,0,LONGCARD(b.width),S);     WrStr(S);WrStr(" x ");
    fmtit (4,0,LONGCARD(b.height),S);    WrStr(S);WrStr(" x ");
    fmtit (2,0,LONGCARD(b.bpp),S);       WrStr(S);
    CASE b.bpp OF
    | 4:
        S:=     " -  16 colors";
    | 8:
        S:=     " - 256 colors";
    | 16,32:
        S:=     " - ";
        bitsum:=CARDINAL(b.rmasksize+b.gmasksize+b.bmasksize);
        CASE bitsum OF
        | 15: S1:=  "32K colors (";
        | 16: S1:=  "64K colors (";
        | 24: S1:=  "16M colors (";
        | 32: S1:=  " 4G colors (";
        ELSE
              S1:=" (";
        END;
        Str.Append(S,S1);
        fmtit(1,0,LONGCARD(b.rmasksize),S1); Str.Append(S,S1);Str.Append(S,":");
        fmtit(1,0,LONGCARD(b.gmasksize),S1); Str.Append(S,S1);Str.Append(S,":");
        fmtit(1,0,LONGCARD(b.bmasksize),S1); Str.Append(S,S1);Str.Append(S,")");
    ELSE
        S:="";
    END;
    WrStr(S);WrStr(" ");
    CASE b.memorymodeltype OF
    | 00H:
        S:="text";
    ELSE
        S:="graphics";
    END;
    WrStr(S);WrLn;

    IF shortfmt THEN RETURN; END;

    WrLn;

    winf(16,"mode attributes"                      ,b.modeattributes); (* 80 *)
    winf( 2,"mode attributes"                      ,b.modeattributes); (* 80 *)
    fl:=capflagtype(b.modeattributes);
    reminfvesa ( fl, 0, "mode supported by present hardware configuration");
    reminfvesa ( fl, 1, "optional information available (must be =1 for VBE v1.2+)");
    reminfvesa ( fl, 2, "BIOS output supported");
    reminfvesa ( fl, 3, "set if color, clear if monochrome");
    reminfvesa ( fl, 4, "set if graphics mode, clear if text mode");
    reminfvesa ( fl, 5, "mode is not VGA-compatible");
    reminfvesa ( fl, 6, "bank-switched mode not supported");
    reminfvesa ( fl, 7, "linear framebuffer mode supported");
    reminfvesa ( fl, 8, "double-scan mode available (e.g. 320x200 and 320x240)");
    reminfvesa ( fl, 9, "interlaced mode available (VBE 3.0)");
    reminfvesa ( fl,10, "hardware supports triple buffering");
    reminfvesa ( fl,11, "hardware supports stereoscopic display");
    reminfvesa ( fl,12, "dual display start address support");
    reminfvesa ( fl, 9, "application must call EnableDirectAccess before calling bank-switching functions (VBE/AF v1.0P)");

    binf( 2,"window A attributes"                  ,b.wA);             (* 81 *)
    fl:=capflagtype(b.wA);
    reminfvesa ( fl, 0, "exists");
    reminfvesa ( fl, 1, "readable");
    reminfvesa ( fl, 2, "writable");
    binf( 2,"window B attributes"                  ,b.wB);             (* 81 *)
    fl:=capflagtype(b.wB);
    reminfvesa ( fl, 0, "exists");
    reminfvesa ( fl, 1, "readable");
    reminfvesa ( fl, 2, "writable");
    winf(10,"window granularity in KB"             ,b.wgranularityKB);
    winf(10,"window size in KB"                    ,b.wsizeKB);
    winf(16,"start segment of window A"            ,b.segmentwA); (* 0 if unsupported *)
    CASE b.segmentwA OF
    | 0  : S:="unsupported";
    ELSE   S:="supported";
    END;
    reminf(S);
    winf(16,"start segment of window B"            ,b.segmentwB); (* 0 if unsupported *)
    CASE b.segmentwB OF
    | 0  : S:="unsupported";
    ELSE   S:="supported";
    END;
    reminf(S);

   dwinf(16,"FAR window positioning function"      ,b.wpositioningfunction);

    winf(10,"bytes per scan line"                  ,b.bytesperscanline);

    winf(10,"width in pixels or characters"        ,b.width);
    winf(10,"height in pixels or characters"       ,b.height);
    binf(10,"width of character cell in pixels"    ,b.wcell);
    binf(10,"height of character cell in pixels"   ,b.hcell);
    binf(10,"number of memory planes"              ,b.memplanes);
    binf(10,"number of bits per pixel"             ,b.bpp);
    binf(10,"number of banks"                      ,b.banks);
    binf(16,"memory model type"                    ,b.memorymodeltype); (* 82 *)
    CASE b.memorymodeltype OF
    | 00H : S:="text";
    | 01H : S:="CGA graphics";
    | 02H : S:="HGC graphics";
    | 03H : S:="16-color (EGA) graphics";
    | 04H : S:="packed pixel graphics";
    | 05H : S:="''sequ 256'' (non-chain 4) graphics";
    | 06H : S:="direct color (HiColor, 24-bit color)";
    | 07H : S:="YUV (luminance-chrominance, also called YIQ)";
    | 08H..00FH : S:="reserved for VESA";
    | 10H..0FFH : S:="OEM memory models";
    END;
    reminf(S);

    binf(10,"size of bank in KB"                   ,b.banksizeKB);
    binf(10,"number of image pages fitting VRAM"   ,b.imagepages+BYTE(1) );
    binf(10,"reserved flag"                        ,b.reservedflag); (* 00h for VBE 1.0-2.0, 01h for VBE 3.0 *)
    CASE b.reservedflag OF
    | 0 : S:="VBE 1.0-2.0";
    | 1 : S:="VBE 3.0";
    ELSE  S:="Unexpected VBE value !";
    END;
    reminf(S);
    binf(10,"red mask size"                        ,b.rmasksize);
    binf(10,"red field position"                   ,b.rfield);
    binf(10,"green mask size"                      ,b.gmasksize);
    binf(10,"green field position"                 ,b.gfield);
    binf(10,"blue mask size"                       ,b.bmasksize);
    binf(10,"blue field position"                  ,b.bfield);
    binf(10,"reserved mask size"                   ,b.reservedmasksize);
    binf(10,"reserved mask position"               ,b.reservedmaskposition);
    binf( 2,"direct color mode info"               ,b.directcolormode); (* bit 0: color ramp is programmable -- bit 1: bytes in reserved field may be used by application *)
    fl:=capflagtype(b.directcolormode);
    reminfvesa ( fl, 0, "color ramp is programmable");
    reminfvesa ( fl, 1, "bytes in reserved field may be used by application");

   dwinf(16,"physical address of linear video buffer",b.linearvideobufferaddr);
   dwinf(16,"pointer to start of offscreen memory" ,b.offscreenmemoryaddr);
    winf(10,"KB of offscreen memory"               ,b.offscreenmemoryKB);

    winf(10,"bytes per scan line in linear modes"  ,b.bytesperscanlinelinear);
    binf(10,"number of images for banked video modes",b.imagesforbankedmodes+BYTE(1));
    binf(10,"number of images for linear video modes",b.imagesforlinearmodes+BYTE(1));
    (* removed "linear modes : ", shortened "position" and "maximum" *)
    binf(10,"size of direct color red mask, in bits",b.rmasklinear);
    binf(10,"bit pos of red mask LSB (shift count)",b.rmaskLSBlinear);
    binf(10,"size of direct color green mask, in bits",b.gmasklinear);
    binf(10,"bit pos of green mask LSB (shift count)",b.gmaskLSBlinear);
    binf(10,"size of direct color blue mask, in bits",b.bmasklinear);
    binf(10,"bit pos of blue mask LSB (shift count)",b.bmaskLSBlinear);
    binf(10,"size of direct color reserved mask, in bits",b.reservedmasklinear);
    binf(10,"bit pos of reserved mask LSB",b.reservedmaskLSBlinear);
   dwinf(10,"max pixel clock for graphics video mode, in Hz",b.maxclockHz);

    IF hexdump THEN WrLn;dump ( ADR(b), SIZE(b) ); END;

END showVESAmodeInfos;

(* ------------------------------------------------------------ *)

(* we use "Buffer" globerk *)

PROCEDURE showVESA (justlist,forceVBE2,verbose,veryverbose,hexdump,shortfmt:BOOLEAN;
                   vmode:CARDINAL):BOOLEAN; (* well, very, very terse *)
VAR
    S : str128;
    v,major,minor,i : CARDINAL;
    ok,dmp:BOOLEAN;
    capflag:capflagtype;
    listPtr  : vesapointer; (* pointer to list of available VESA modes *)
    mode:CARDINAL;
    htab,maxhtab,nexthtab:CARDINAL;
BEGIN
    IF CheckVesaHere(forceVBE2,listPtr)=FALSE THEN RETURN FALSE;END;

IF justlist=FALSE THEN

    WrLn;

    (* Lib.Move (ADR(Buffer[004H]),ADR(v),SIZE(v)); (* majorminor : 0102 = 1.2 *) *)
    v := Buffer.majmin; (* majorminor : 0102 = 1.2 *)
    major:=(v >> 8);
    minor:=(v AND 00FFH);
    WrStr("VESA ");IO.WrCard(major,1);WrStr(".");IO.WrCard(minor,1);

    (* Lib.Move (ADR(Buffer[012H]),ADR(v),SIZE(v)); (* video memory in 64K blocks *) *)
    v:=Buffer.VRAM64KBblocks;

    Str.CardToStr(64*LONGCARD(v),S,10,ok);
    LOOP
        IF Str.CharPos(S," ")=MAX(CARDINAL) THEN EXIT; END;
        Str.Subst(S," ","");
    END;
    WrStr(", "); WrStr(S);WrStr(" Kb video RAM");

    getasciiz(Buffer.pOEMname,S);
    WrStr(", ");WrStr(S);WrLn;
    WrLn;

    (* Lib.Move (ADR(Buffer[00AH]),ADR(capflag),SIZE(capflag)); (* capabilities flag *) *)
    capflag:=Buffer.capabilities;
    S:="";
    FOR i:=31 TO 0 BY -1 DO
        IF i IN capflag THEN
            Str.Append(S,"1");
        ELSE
            Str.Append(S,"0");
        END;
    END;

       (* "bit ## off : " *)
    WrStr("flag       : %");WrStr(S);WrLn;
    WrLn;

    infvesa(capflag,0,"DAC can be switched into 8-bit mode");
    infvesa(capflag,1,"non-VGA controller");
    infvesa(capflag,2,"programmed DAC with blank bit (i.e. only during blanking interval)");
    WrLn;

    infvesa(capflag,3,"(VESA 3.0) controller supports VBE/AF v1.0P extensions");
    infvesa(capflag,4,"(VESA 3.0) stereo signalling via VESA EVC (no=via external stereo)");
    WrLn;

    infvesa(capflag,3,"(VBE/AF 1.0P) controller supports hardware stereoscopic signalling");
    infvesa(capflag,4,"(VBE/AF 1.0P) must call EnableDirectAccess to access framebuffer");
    infvesa(capflag,5,"(VBE/AF 1.0P) controller supports hardware mouse cursor");
    infvesa(capflag,6,"(VBE/AF 1.0P) controller supports hardware clipping");
    infvesa(capflag,7,"(VBE/AF 1.0P) controller supports transparent BitBLT");

    IF hexdump THEN WrLn;dump( ADR(Buffer), SIZE(Buffer) ); END;

END;

    IF verbose THEN
        WrLn;
        IF vmode=showAllModes THEN
            S:="List of video modes supported by VESA BIOS :";
        ELSE
            S:="Informations about specified video mode :";
        END;
        WrStr(S);WrLn; IF shortfmt THEN WrLn;END;
        maxhtab:=columnsOnScreen;
        htab:=maxhtab;
        LOOP
            IF listPtr^ = WORD(0FFFFH) THEN EXIT; END;
            mode:=listPtr^;
            IF vmode=showAllModes THEN
                dmp:=TRUE;
            ELSE
                dmp:=(mode = vmode);
            END;
            IF dmp THEN
                fmtit( 4, 16 , LONGCARD(mode), S);
                IF veryverbose THEN
                    showVESAmodeInfos(mode,hexdump,shortfmt,S);
                ELSE
                    nexthtab:=htab+Str.Length("  ")+Str.Length(S);
                    IF nexthtab > maxhtab THEN
                        WrLn;
                        WrStr(S);
                        htab:=Str.Length(S);
                    ELSE
                        WrStr("  ");WrStr(S);
                        htab:=nexthtab;
                    END;
                END;
            END;
            Lib.IncFarAddr(listPtr,SIZE(WORD));
        END;
        IF vmode=fullMemoryAccessMode THEN
            showVESAmodeInfos(vmode,hexdump,shortfmt,sFMAM+" : "); (* full memory access *)
        END;
        IF justlist=FALSE THEN WrLn; END;
    END;

    RETURN TRUE;
END showVESA;

(* ------------------------------------------------------------ *)

VAR
    fontheight [sb:0085H] : CARDINAL;

PROCEDURE getcursordata ( VAR scanlines:CARDINAL;
                          VAR cstart,cend:SHORTCARD);
VAR
    R : SYSTEM.Registers;
    columns,displaymode,activepage:SHORTCARD;
    cursorstart,cursorend,cursorcolumn,cursorrow:SHORTCARD;
    oldscanlines : CARDINAL;
BEGIN
    oldscanlines := fontheight;

    R.AH := 0FH;           (* get current video mode *)
    Lib.Intr(R,iv);
    columns     := R.AH;
    displaymode := R.AL;   (* bit 7 can be ON if previous mode setting was so *)
    activepage  := R.BH;

    R.AH := 03H;           (* get cursor position and size *)
    R.BH := activepage;
    Lib.Intr(R,iv);
    cursorstart := R.CH;
    cursorend   := R.CL;
    cursorcolumn:= R.DL;
    cursorrow   := R.DH;

    scanlines := oldscanlines;
    cstart    := cursorstart;
    cend      := cursorend;
END getcursordata;

PROCEDURE newcursorshape (newcursorstart,newcursorend:CARDINAL   );
VAR
    R : SYSTEM.Registers;
BEGIN
    R.AH := 01H;           (* set text-mode cursor shape *)
    R.CH := SHORTCARD(newcursorstart);
    R.CL := SHORTCARD(newcursorend);
    Lib.Intr(R,iv);
END newcursorshape;

(* ------------------------------------------------------------ *)

CONST
    unset=  MAX(CARDINAL);
    C25  =  0003H; (* color 80x25 *)
    C28  =  1003H;
    C30  =  8003H; (* direct hardware assuming VGA *)
    C43  =  2003H;
    C50  =  3003H;
VAR
    parmcount,i,opt,mode : CARDINAL;
    S,R                  : str128;
    state                : (waiting,gotmode);
    modeset              : (byprogram,byvgabios,byvesabios);
    changeshape          : (nochange,autochange,forcedchange);
    oldscanlines,newscanlines,newcursorstart,newcursorend:CARDINAL;
    cursorstart,cursorend:SHORTCARD;
    nstart,nend:CARDINAL;
    INFOS,INFOSPLUS,INFOSPLUSPLUS,INFOSHEX,forcevbe2,SHORTFMT,justlist:BOOLEAN;
    DEBUG : BOOLEAN;
    videomode:CARDINAL;
VAR
    listPtr  : vesapointer; (* pointer to list of available VESA modes *)
BEGIN
    Lib.DisableBreakCheck();

    parmcount := Lib.ParamCount();
    IF parmcount=0 THEN abort(errHelp,"");END;

    mode        := unset;
    changeshape := nochange;
    INFOS       := FALSE;
    DEBUG       := FALSE;
    INFOSPLUS   := FALSE;
    INFOSPLUSPLUS:=FALSE;
    INFOSHEX    := FALSE;
    SHORTFMT    := FALSE;
    forcevbe2   := FALSE;
    justlist    := FALSE;
    videomode   := showAllModes;
    modeset     := byprogram;

    state       := waiting;

    FOR i := 1 TO parmcount DO (* for future extension ! *)
        Lib.ParamStr(S,i); cleantabs(S);
        Str.Copy(R,S);
        UpperCase(R);
        IF isOption(R) THEN
            opt := GetOptIndex(R, "?"+delim+"H"+delim+"HELP"+delim+
                                  "C"+delim+"CURSOR"+delim+
                                  "I"+delim+"INFOS"+delim+
                                  "C:"+delim+"CURSOR:"+delim+
                                  "IV"+delim+"IVBE2"+delim+"II"+delim+
                                  "V"+delim+"VERBOSE"+delim+
                                  "VVV"+delim+"VERYVERBOSE"+delim+
                                  "X"+delim+"HEX"+delim+
                                  "V:"+delim+"VESAMODE:"+delim+
                                  "VV"+delim+"VERBOSER"+delim+
                                  "L"+delim+"LIST"+delim+
                                  "DEBUG");
            CASE opt OF
            | 1,2,3 : abort(errHelp,"");
            | 4,5:    changeshape:=autochange;
            | 6,7:    INFOS := TRUE; forcevbe2:=FALSE;
            | 8,9:    GetString(S,R);
                      IF getBoth(nstart,nend,R)=FALSE THEN abort(errBadShape,R);END;
                      changeshape:=forcedchange;
            | 10,11,12:  INFOS := TRUE; forcevbe2:=TRUE;
            | 13,14:  INFOSPLUS:=TRUE;
            | 15,16:  INFOSPLUS:=TRUE; INFOSPLUSPLUS:=TRUE;
            | 17,18:  INFOSHEX:=TRUE;
            | 19,20:  GetString(S,R);
                      IF getCardValue(videomode,R)=FALSE THEN abort(errIllegalValue,S);END;
                      CASE videomode OF
                      | 100H..1FFH,fullMemoryAccessMode:
                          INFOS:=TRUE; INFOSPLUS:=TRUE;INFOSPLUSPLUS:=TRUE;
                      ELSE
                          abort(errNotVesaMode,S);
                      END;
            | 21,22:  INFOSPLUS:=TRUE; INFOSPLUSPLUS:=TRUE;  SHORTFMT:=TRUE;
            | 23,24:  INFOS:=TRUE; forcevbe2:=TRUE; (* -iv *)
                      INFOSPLUS:=TRUE; INFOSPLUSPLUS:=TRUE;  SHORTFMT:=TRUE; (* -vv *)
                      justlist:=TRUE;
            | 25:     DEBUG := TRUE;
            ELSE
                abort(errOption,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            IF state = gotmode THEN abort(errParmOverflow,S); END;
            INC(state);
            Str.Prepend(R,dash); (* fake option *)
            IF CharCount(R,star)=1 THEN Str.Subst(R,star,mul);END;
            opt := GetOptIndex(R,"80X60"+delim+
                                 "132X25"+delim+
                                 "132X43"+delim+
                                 "132X50"+delim+
                                 "132X60"+delim+
                                 "80X25"+delim+
                                 "80X43"+delim+
                                 "80X50"+delim+
                                 "80X28"+delim+
                                 "80X30");
            CASE opt OF
            | 1      : mode := 108H;
            | 2      : mode := 109H;
            | 3      : mode := 10AH;
            | 4      : mode := 10BH;
            | 5      : mode := 10CH;
            | 6      : mode := C25;
            | 7      : mode := C43;
            | 8      : mode := C50;
            | 9      : mode := C28;
            | 10     : mode := C30;
            ELSE
                (* abort(errBadParm,S); *)
                (* use original param without dash trick ! *)
                IF getCardValue(mode,S)=FALSE THEN abort(errIllegalValue,S);END;
                CASE mode OF
                | 000H..0FFH:
                    (* 3 is normally C25 but we won't use Set25LineMode here *)
                    modeset:=byvgabios;
                | 100H..1FFH:
                    modeset:=byvesabios;
                ELSE
                    abort(errBadModeValue,S);
                END;
            END;
        END;
    END;

    IF INFOS THEN
        IF mode # unset THEN abort(errSyntax,"");END;
        IF INFOSHEX AND SHORTFMT THEN abort(errNonsense,"");END;
        IF showVESA(justlist,forcevbe2,INFOSPLUS,INFOSPLUSPLUS,INFOSHEX,SHORTFMT,videomode)=FALSE THEN
            abort(errNoVesa,"");
        END;
    ELSE
        IF mode=unset THEN abort(errUnset,"");END;

        IF ( (changeshape # nochange) OR DEBUG) THEN
            getcursordata(oldscanlines,cursorstart,cursorend);
        END;

        CASE modeset OF
        | byprogram :
            CASE mode OF
            | C25  : IF set25LineMode()=FALSE THEN abort(errVGA,"25");END;
            | C28  : IF set28LineMode()=FALSE THEN abort(errVGA,"28");END;
            | C43  : IF set43LineMode()=FALSE THEN abort(errVGA,"43");END;
            | C50  : IF set50LineMode()=FALSE THEN abort(errVGA,"50");END;
            | C30  : IF set30LineMode()=FALSE THEN abort(errVGA,"30");END;
            ELSE
                IF CheckVesaHere(FALSE,listPtr)=FALSE THEN abort(errNoVesa,""); END;
                IF ModeAvailable(listPtr,mode)=FALSE THEN abort(errNotAvailable,""); END;
                IF SetVesaMode(mode) = FALSE THEN abort(errVesaProblem,""); END;
            END;
        | byvgabios:
            setVideoMode( SHORTCARD(mode) );
        | byvesabios:
            IF CheckVesaHere(FALSE,listPtr)=FALSE THEN abort(errNoVesa,""); END;
            IF ModeAvailable(listPtr,mode)=FALSE THEN abort(errNotAvailable,""); END;
            IF SetVesaMode(mode) = FALSE THEN abort(errVesaProblem,""); END;
        END;

        IF ( (changeshape # nochange) OR DEBUG) THEN
            newscanlines   := fontheight;
            newcursorstart := (newscanlines * CARDINAL(cursorstart)) DIV oldscanlines;
            newcursorend   := (newscanlines * CARDINAL(cursorend  )) DIV oldscanlines;
        END;

        CASE changeshape OF
        | autochange : newcursorshape(newcursorstart,newcursorend);
        | forcedchange:newcursorshape(nstart,nend);
        END;

        IF DEBUG THEN
            WrStr("oldscanlines   ");IO.WrCard(oldscanlines,3);
            WrStr("    oldcursorstart ");IO.WrShtCard(cursorstart,3);
            WrStr("    oldcursorend   ");IO.WrShtCard(cursorend,3);WrLn;
            WrStr("newscanlines   ");IO.WrCard(newscanlines,3);
            WrStr("    newcursorstart ");IO.WrCard(newcursorstart,3);
            WrStr("    newcursorend   ");IO.WrCard(newcursorend,3);WrLn;
        END;

    END;

    abort(errNone,"");
END SetVmode.

