(* ---------------------------------------------------------------
Title         Q&D File Data Extractor
Author        PhG
Overview      Extract and save portion of a file to another file
Usage         see help
Notes         YACWOT
              weird : without cache, M2 code is about 50% slower than BASIC
              yet, with cache, it is about 8 times FASTER !
              as for other programs (see infra), even though a LONGCARD
              can hold 4 Gb, filesize will be 0 with huge 2 Gb files
              when using win9X
              yep, we should $216c00 extended open file function,
              but we could risk weird problems thanks to ms crap
              interfering more or less willingly with TS libraries
              so, let this for now and for ever
Bugs          program won't process files 2.6 Gb long, thinking it's 0

Wish list     no, why ?

--------------------------------------------------------------- *)

MODULE Xtract;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;
IMPORT IO;
IMPORT SYSTEM;

FROM IO IMPORT WrStr,WrLn;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits,
getAllLegalUnits;

FROM QD_LFN IMPORT path9X, huge9X, findDataRecordType,
unicodeConversionFlagType, w9XchangeDir,
w9XgetDOSversion, w9XgetTrueDOSversion, w9XisWindowsEnh, w9XisMSDOS7,
w9XfindFirst, w9XfindNext, w9XfindClose, w9XgetCurrentDirectory,
w9XlongToShort, w9XshortToLong, w9XtrueName, w9XchangeDir,
w9XmakeDir, w9XrmDir, w9Xrename, w9XopenFile, w9XcloseFile,
w9XsupportLFN;

FROM QD_File IMPORT pathtype, w9XnothingRequired,
fileOpenRead, fileOpen, fileExists, fileIsRO, fileSetRW, fileSetRO,
fileErase, fileCreate, fileRename, fileGetFileSize, fileIsDirectorySpec;

(* ------------------------------------------------------------ *)

CONST
    cr          = CHR(13);
    lf          = CHR(10);
    nl          = cr+lf;
    nullchar    = CHR(0);
    diese       = "#";
    semicolon   = ";";
    blank       = " ";      (* 32 = $20 *)
    tab         = CHR(9);
    dquote      = '"';      (* 34 = $22 *)
    dollar      = "$";
    dot         = ".";
    star        = "*";
    equal       = "=";
    colon       = ":";
    minus       = "-";
    extBAK      = ".BK!";
CONST
    ProgEXEname   = "XTRACT";
    ProgTitle     = "Q&D File Data Extractor";
    ProgVersion   = "v1.2i";
    ProgCopyright = "by PhG";
    Banner        = ProgTitle+" "+ProgVersion+" "+ProgCopyright;
CONST
    errNone             = 0;
    errHelp             = 1;
    errUnknownOption    = 2;
    errUneededParm      = 3;
    errCmd              = 4;
    errMissingSource    = 5;
    errMissingTarget    = 6;
    errMissingStart     = 7;
    errMissingEnd       = 8;
    errNotFound         = 9;
    errAlreadyExists    = 10;
    errBadHex           = 11;
    errBadDec           = 12;
    errStrToCard        = 13;
    errBeforeStart      = 14;
    errEndPastSize      = 15;
    errStartPastSize    = 16;
    errSameSourceTarget = 17;
    errNonsense         = 18;
    errRenameFailure    = 19;
    errAnotherNonsense  = 20;
    errBadOffset        = 21;

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)
    helpmsg =
Banner+nl+
nl+
"Syntax : "+ProgEXEname+" [option]... <source> <target> <start> <end|L[=]length|*[-#]>"+nl+
nl+
"  -o    overwrite existing <target>"+nl+
"  -t[t] create text <target> formatted for XD (-tt = -t -8)"+nl+
"  -8    output 8 hexadecimal bytes instead of 16 (-t[t])"+nl+
"  -n    do not display data in ASCII alongside data in hexadecimal (-t[t])"+nl+
"  -u    hexadecimal data in uppercase"+nl+
"  -q    quiet (no eyecandy)"+nl+
"  -r    rename existing <target>, if any"+nl+
'  -b    do not create backup (only if <target> was specified as "'+star+'" or "'+equal+'")'+nl+
"  -z    abort data extraction at first encountered $00 byte (C-style string)"+nl+
"  -lfn  disable LFN support even if available"+nl+
nl+
"a) <start> address is 0-based."+nl+
'b) <end> address can be specified as "*[-#]" for "end of file" ;'+nl+
"   optional concatenated value is relative negative offset."+nl+
'c) Hexadecimal values are prefixed with "$".'+nl+
'd) If <target> is specified as "'+star+'" or "'+equal+'", it will become <source>.'+nl+
nl+
"Examples : "+ProgEXEname+" -t c:\win31\system\user.exe guesswho $29e2c L30"+nl+
"           "+ProgEXEname+" xpsp3.nrg xpsp3.iso 307200 *-156"+nl;

VAR
    S : str256;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(helpmsg);
    | errUnknownOption :
        Str.Concat(S,"Unknown ",einfo);Str.Append(S," option !");
    | errUneededParm :
        Str.Concat(S,"Uneeded ",einfo);Str.Append(S," parameter !");
    | errMissingSource :
        S:="Missing <source> specification !";
    | errMissingTarget :
        S:="Missing <target> specification !";
    | errMissingStart :
        S:="Missing <[$]start> specification !";
    | errMissingEnd :
        S:="Missing <[$]end|L[$]length|*> specification !";
    | errNotFound :
        Str.Concat(S,einfo," <source> does not exist !");
    | errAlreadyExists :
        Str.Concat(S,einfo," <target> already exists !");
    | errBadHex:
        Str.Concat(S,"Illegal characters in hexadecimal ",einfo);Str.Append(S," value !");
    | errBadDec:
        Str.Concat(S,"Illegal characters in decimal ",einfo);Str.Append(S," value !");
    | errStrToCard:
        Str.Concat(S,"Illegal (",einfo);Str.Append(S,") value !");
    | errBeforeStart:
        S := "End position is before start position !";
    | errEndPastSize:
        S := "End position is beyond end of <source> !";
    | errStartPastSize:
        S := "Start position is beyond end of <source> !";
    | errSameSourceTarget:
        S := "<source> and <target> must be different !";
    | errNonsense:
        S := "-o and -r are mutually exclusive !";
    | errRenameFailure:
        Str.Concat(S,"No new name could be created for ",einfo);Str.Append(S," <target> !");
    | errAnotherNonsense:
        Str.Concat(S,einfo," option is a nonsense when <target> is <source> !");
    | errBadOffset:
        Str.Concat(S,einfo," relative offset must be smaller than file size !");

    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp :
        ; (* nada *)
    ELSE
        WrStr(ProgEXEname+" : "); WrStr(S); WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    ioBufferSize    = (8 * 512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = ioBufferSize;
    tmpbuffersize   = ioBufferSize * 8;
    lasttmpbuffer   = tmpbuffersize;
TYPE
    ioBuffer  = ARRAY [firstBufferByte..lastBufferByte] OF BYTE;
VAR
    bufferIn  : ioBuffer;
    bufferOut : ioBuffer;
    tmpbuffer : ARRAY [firstBufferByte..lasttmpbuffer] OF BYTE;

(* ------------------------------------------------------------ *)

PROCEDURE PadHex (v : LONGCARD; digits : CARDINAL) : str16;
CONST
    padStr = "0000000000000000"; (* 16 digits *)
VAR
    S : str16;
    R : str16;
    ok : BOOLEAN;
    delta : CARDINAL;
BEGIN
    digits := digits MOD 16; (* better safe than sorry! *)
    Str.CardToStr (v,S,16,ok);
    delta := digits - Str.Length(S);
    Str.Slice (R,padStr,0,delta);
    Str.Append (R,S);
    Str.Lows(R);
    RETURN R;
END PadHex;

PROCEDURE PadDec (v : LONGCARD; digits : CARDINAL) : str16;
CONST
    padStr = "                "; (* 16 digits *)
VAR
    S : str16;
    R : str16;
    ok : BOOLEAN;
    delta : CARDINAL;
BEGIN
    digits := digits MOD 16; (* better safe than sorry! *)
    Str.CardToStr (v,S,10,ok);
    delta := digits - Str.Length(S);
    Str.Slice (R,padStr,0,delta);
    Str.Append (R,S);
    RETURN R;
END PadDec;

PROCEDURE Filter (c : CHAR) : CHAR;
BEGIN
    IF ( (ORD(c) < ORD(blank)) OR (ORD(c) = 255) ) THEN
        RETURN ".";
    ELSE
        RETURN c;
    END;
END Filter;

(* ------------------------------------------------------------ *)

PROCEDURE newname (useLFN:BOOLEAN; VAR S : pathtype):BOOLEAN ;
CONST
    maxtries = 999;
    maxf8    = 8;
    max9X    = 128; (* should be more than enough *)
VAR
    u,d,n,e,newfile,fname : pathtype;
    i   : LONGCARD;
    num : str16;
    ok  : BOOLEAN;
    len,maxlen : CARDINAL;
BEGIN
    IF useLFN THEN
        maxlen := max9X;
    ELSE
        maxlen := maxf8;
    END;
    Lib.SplitAllPath(S,u,d,n,e);
    len:=Str.Length(n);
    i:=0;
    LOOP
        Str.CardToStr(i,num,10,ok);
        WHILE (len+Str.Length(num)) > maxlen DO
            Str.Delete(n,len-1,1); (* delete last char of name *)
            DEC(len);
        END;
        Str.Concat(fname,n,num); (* build new name *)
        Lib.MakeAllPath(newfile,u,d,fname,e);
        IF fileExists(useLFN,newfile)=FALSE THEN
            Str.Copy(S,newfile);
            RETURN TRUE;
        END;
        INC(i);
        IF i > maxtries THEN EXIT; END;
    END;
    RETURN FALSE;
END newname;

PROCEDURE newExtension (VAR R:pathtype;S:pathtype;e3:ARRAY OF CHAR);
VAR
    p : CARDINAL;
BEGIN
    Str.Copy(R,S);
    p := Str.RCharPos(R,dot);
    IF p = MAX(CARDINAL) THEN
        Str.Append(R,e3);
    ELSE
        R[p]:=CHR(0);
        Str.Append(R,e3);
    END;
END newExtension;

(* ------------------------------------------------------------ *)

PROCEDURE verify (S,allowed:ARRAY OF CHAR):BOOLEAN;
VAR
    i,mismatches : CARDINAL;
    ch : CHAR;
BEGIN
    mismatches := 0;
    FOR i := 1 TO Str.Length(S) DO
        ch := S[i-1];
        IF Belongs(allowed,ch)=FALSE THEN INC(mismatches);END;
    END;
    RETURN (mismatches=0);
END verify;

(* assume already trimmed and in uppercase *)

PROCEDURE StrToLong (S:ARRAY OF CHAR; VAR v : LONGCARD) : CARDINAL;
CONST
    digits    = "0123456789";
    hexdigits = "ABCDEF"+digits;
VAR
    ok : BOOLEAN;
    base : CARDINAL;
BEGIN
    IF S[0]=dollar THEN
        Str.Delete(S,0,1);
        IF verify(S,hexdigits)=FALSE THEN RETURN errBadHex;END;
        base := 16;
    ELSIF ( (S[0]="0") AND (S[1]="X") ) THEN
        Str.Delete(S,0,2);
        IF verify(S,hexdigits)=FALSE THEN RETURN errBadHex;END;
        base := 16;
    ELSE
        IF verify(S,digits)=FALSE THEN RETURN errBadDec;END;
        base := 10;
    END;
    v:=Str.StrToCard(S,base,ok);
    IF ok THEN
        RETURN errNone;
    ELSE
        RETURN errStrToCard;
    END;
END StrToLong;

(* ------------------------------------------------------------ *)

PROCEDURE header( useLFN:BOOLEAN;src,dest:pathtype;pos,lastpos,count:LONGCARD;
                  renamed : BOOLEAN; orgdest:pathtype;
                  VAR S1,S2,S3,S1BIS:ARRAY OF CHAR);
CONST
    wihex      = 8;
    widec      = 10; (* ########## *)
    performing = "Extracting data";
    performed  = "Data extracted";
CONST
    headhex    = "[ $";
    midhex     = ".. $";
    trailhex   = "], count= $";
    headdec    = "[";
    middec     = "..";
    traildec   = "], count=";
VAR
    newdest : str1024;
BEGIN
    Str.Copy(newdest,dest);
    IF useLFN THEN
        Str.Prepend(newdest,dquote);
        Str.Append(newdest,dquote);
    END;
    IF renamed THEN
        Str.Append(newdest," (");
        IF useLFN THEN Str.Append(newdest,dquote);END;
        Str.Append(newdest,orgdest);
        IF useLFN THEN Str.Append(newdest,dquote);END;
        Str.Append(newdest,")");
    END;

    Str.Concat(S1,performing," from ");
    IF useLFN THEN Str.Append(S1,dquote);END;
    Str.Append(S1,src);
    IF useLFN THEN Str.Append(S1,dquote);END;
    Str.Append(S1," to ");
    Str.Append(S1,newdest);
    Str.Append(S1,nl);

    Str.Concat(S2,headhex,PadHex(pos,wihex));
    Str.Append(S2,midhex);
    Str.Append(S2,PadHex(lastpos,wihex));
    Str.Append(S2,trailhex);
    Str.Append(S2,PadHex(count,wihex));
    Str.Append(S2,nl);

    Str.Concat(S3,headdec,PadDec(pos,widec));
    Str.Append(S3,middec);
    Str.Append(S3,PadDec(lastpos,widec));
    Str.Append(S3,traildec);
    Str.Append(S3,PadDec(count,widec));
    Str.Append(S3,nl);

    Str.Concat(S1BIS,performed," from ");
    IF useLFN THEN Str.Append(S1BIS,dquote);END;
    Str.Append(S1BIS,src);
    IF useLFN THEN Str.Append(S1BIS,dquote);END;
    Str.Append(S1BIS,nl);
END header;

(* we use tmpbuffer *)

PROCEDURE dumpxd (hnd:FIO.File;offset:LONGCARD;total:CARDINAL;
                  showascii,lowercase:BOOLEAN;row:CARDINAL);
CONST
    sepaddrbeg  = " :";
    sepaddr     = sepaddrbeg+" ";
    sepasciibeg = " |"; (* for safety, must be different from sepaddr *)
    sepascii    = sepasciibeg+" ";
    sep         = "  ";
VAR
    i,j : CARDINAL;
    V : str16;
    O : str256;
    R : str128;
    c : BYTE;
BEGIN
    i:=0;
    LOOP
        IF i >= total THEN EXIT; END;
        V := PadHex(offset+LONGCARD(i),8);
        Str.Concat(O,V,sepaddr);

        Str.Copy(R,"");
        FOR j := 1 TO row DO
            IF (i+j-1) < total THEN
                c := tmpbuffer[i+j];
                V := PadHex(LONGCARD(ORD(c)),2); (* $$ *)
                Str.Append(R,V);
            ELSE
                Str.Append(R,sep);
            END;
            IF j < row THEN Str.Append(R," "); END;
        END;
        Str.Append(O,R);
        IF lowercase THEN Str.Lows(O);END;
        IF showascii THEN
            Str.Copy(R,sepascii);
            FOR j := 1 TO row DO
                IF (i+j-1) < total THEN
                    c := Filter(tmpbuffer[i+j]);
                    Str.Append(R,c);
                END;
            END;
            Str.Append(O,R);
        END;

        Str.Append(O,nl);
        FIO.WrStr(hnd,O);
        INC(i,row);
    END;
END dumpxd;

(* ------------------------------------------------------------ *)

PROCEDURE doExtract (useLFN:BOOLEAN;src,dest:pathtype; pos,lastpos,count:LONGCARD;
                     forXD,showascii,lowercase,eyecandy,cstring:BOOLEAN;
                     bytesPerLine:CARDINAL;
                     renamed:BOOLEAN;orgtarget:pathtype);
CONST
    msg = "Working "; (* space needed *)
VAR
    hin,hout:FIO.File;
    S1,S2,S3,S1BIS:str1024; (* were str128 but with LFNs... *)
    remaining : LONGCARD;
    wanted,got : CARDINAL;
    i,p : CARDINAL;
    cstringfound:BOOLEAN;
BEGIN
    cstringfound :=FALSE;

    hin:=fileOpenRead(useLFN,src);
    FIO.AssignBuffer(hin,bufferIn);
    hout:=fileCreate(useLFN,dest);
    FIO.AssignBuffer(hout,bufferOut);

    header ( useLFN,src,dest,pos,lastpos,count,
             renamed,orgtarget,
             S1,S2,S3,S1BIS );

    WrStr(S1);
    WrLn;
    WrStr(S2);
    WrStr(S3);

    IF forXD THEN
        FIO.WrStr(hout,semicolon+nl);
        FIO.WrStr(hout,semicolon+blank);FIO.WrStr(hout,S1BIS);
        FIO.WrStr(hout,semicolon+nl);
        FIO.WrStr(hout,semicolon+blank);FIO.WrStr(hout,S2);
        FIO.WrStr(hout,semicolon+blank);FIO.WrStr(hout,S3);
        FIO.WrStr(hout,semicolon+nl+nl);
    END;

    FIO.Seek(hin,pos);
    remaining:=count;

    IF eyecandy THEN
        video(msg,TRUE);
        Animation(cmdInit);
    END;

    LOOP
        IF eyecandy THEN Animation(cmdShow); END;
        IF remaining = 0 THEN EXIT; END;
        IF remaining > tmpbuffersize THEN
            wanted := tmpbuffersize;
            DEC(remaining,tmpbuffersize);
        ELSE
            wanted := CARDINAL(remaining);
            remaining := 0;
        END;
        got := FIO.RdBin(hin,tmpbuffer,wanted);
        IF cstring THEN
            IF got # 0 THEN (* safety ! *)
                p:=Lib.ScanR( ADR(tmpbuffer), got, BYTE(00H) );
                IF p # got THEN (* found, truncate data and force next EXIT *)
                    got := p;
                    remaining:=0;
                    cstringfound:=TRUE;
                END;
            END;
        END;
        IF forXD THEN
            dumpxd (hout,pos+count-remaining-LONGCARD(wanted),got,showascii,lowercase,bytesPerLine);
        ELSE
            FIO.WrBin(hout,tmpbuffer,got);
        END;
    END;

    FIO.Flush(hout);
    FIO.Close(hout);
    FIO.Close(hin);

    IF eyecandy THEN
        Animation(cmdStop);
        video(msg,FALSE);
    END;

    IF (cstring AND cstringfound) THEN
        WrLn;
        WrStr("Data truncated at first encountered $00 byte.");WrLn;
    END;

END doExtract;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

VAR
    parmcount : CARDINAL;
    i,opt     : CARDINAL;
    S,R       : pathtype;
    source,target : pathtype;
    sStartaddr, sEndaddr : str128;
    status    : (waiting,gotsource,gottarget,gotstartaddr,gotendaddr);

    overwrite, autorename, forXD : BOOLEAN;
    bytesPerLine:CARDINAL;
    showascii, lowercase, eyecandy, samename, nobackup, cstring : BOOLEAN;
    nullcount,DEBUG : BOOLEAN;

    useLFN,renamed,wasRO   : BOOLEAN;
    orgtarget : pathtype;

    fsize,pos,lastpos,count,negofs : LONGCARD;
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE;
    WrLn;

    DEBUG        := FALSE;
    overwrite    := FALSE;
    autorename   := FALSE;
    forXD        := FALSE;
    bytesPerLine := 16;
    showascii    := TRUE;
    lowercase    := TRUE;
    eyecandy     := TRUE;
    samename     := FALSE;
    nobackup     := FALSE;
    renamed      := FALSE;
    cstring      := FALSE;
    useLFN       := TRUE;
    Str.Copy(orgtarget,"");

    parmcount := Lib.ParamCount();
    IF parmcount = 0 THEN abort(errHelp,"");END;

    status := waiting;
    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i);
        Str.Copy(R,S);
        UpperCase(R);  cleantabs(R); (* YATB ! *)
        IF isOption(R)=TRUE THEN
            opt := GetOptIndex(R, "?"+delim+"H"+delim+"HELP"+delim+
                                  "O"+delim+"OVERWRITE"+delim+
                                  "T"+delim+"TEXT"+delim+
                                  "8"+delim+
                                  "N"+delim+"NOASCII"+delim+
                                  "U"+delim+"UPPERCASE"+delim+
                                  "Q"+delim+"QUIET"+delim+
                                  "R"+delim+"RENAME"+delim+
                                  "B"+delim+"NOBACKUP"+delim+
                                  "Z"+delim+"ZEROTERMINATED"+delim+
                                  "LFN"+delim+"X"+delim+
                                  "TT"+delim+
                                  "DEBUG"

                              );
            CASE opt OF
            |  1,  2,  3 : abort(errHelp,"");
            |  4,  5     : overwrite    := TRUE;
            |  6,  7     : forXD        := TRUE;
            |  8         : bytesPerLine := 8;
            |  9, 10     : showascii    := FALSE;
            | 11, 12     : lowercase    := FALSE;
            | 13, 14     : eyecandy     := FALSE;
            | 15, 16     : autorename   := TRUE;
            | 17, 18     : nobackup     := TRUE;
            | 19, 20     : cstring      := TRUE;
            | 21, 22     : useLFN       := FALSE;
            | 23         : forXD        := TRUE;
                           bytesPerLine := 8;
            | 24         : DEBUG        := TRUE;
            ELSE
                abort(errUnknownOption,S);
            END;
        ELSE
            CASE status OF
            | waiting      : Str.Copy(source,R);
            | gotsource    : Str.Copy(target,R);
            | gottarget    : Str.Copy(sStartaddr,R);
            | gotstartaddr : Str.Copy(sEndaddr,R);
            | gotendaddr   : abort(errUneededParm,S);
            END;
            INC(status);
        END;
    END;

    CASE status OF
    | waiting      : abort(errMissingSource,"");
    | gotsource    : abort(errMissingTarget,"");
    | gottarget    : abort(errMissingStart,"");
    | gotstartaddr : abort(errMissingEnd,"");
    | gotendaddr   : ; (* ok for now *)
    END;

    useLFN := (useLFN AND w9XsupportLFN() );

    IF fileExists(useLFN,source)=FALSE THEN abort(errNotFound,source); END;

    IF (overwrite AND autorename) THEN abort(errNonsense,"");END;

    IF ( same(target,star) OR same(target,equal) ) THEN
        IF forXD THEN abort(errAnotherNonsense,"-t");END;
        IF autorename THEN abort(errAnotherNonsense,"-r");END;
        samename:=TRUE;
    END;

    fsize := fileGetFileSize(useLFN,source);

    IF StrToLong (sStartaddr,pos) # errNone THEN abort(errStrToCard,sStartaddr);END;
    INC(pos); (* 1-based now *)

    nullcount:=FALSE;
    CASE sEndaddr[0] OF
    | star :
        IF same(sEndaddr,star) THEN
            lastpos:=fsize;
        ELSE
            Str.Delete(sEndaddr,0,1);
            IF Str.Match(sEndaddr,minus+"*") THEN
                Str.Delete(sEndaddr,0,1);
                IF StrToLong(sEndaddr,negofs) # errNone THEN abort(errStrToCard,sEndaddr);END;
                IF negofs >= fsize THEN abort(errBadOffset,sEndaddr);END;
                lastpos:=fsize-negofs;
            ELSE
                abort(errStrToCard,sEndaddr);
            END;
        END;
    | "L": (* we kept uppercase *)
        Str.Delete(sEndaddr,0,1);
        CASE sEndaddr[0] OF
        | equal, colon :
            Str.Delete(sEndaddr,0,1); (* kludge ! *)
        END;
        IF same(sEndaddr,star) THEN
            lastpos:=fsize;
        ELSE
            IF StrToLong(sEndaddr,count) # errNone THEN abort(errStrToCard,sEndaddr);END;
            IF count=0 THEN nullcount:=TRUE;END;
            lastpos:=pos+count-1; (* at worst (0+1)+0-1=0 *)
        END;
    ELSE
        IF StrToLong(sEndaddr,lastpos) # errNone THEN abort(errStrToCard,sEndaddr);END;
        INC(lastpos);
        IF lastpos = 0 THEN abort(errEndPastSize,sEndaddr); END; (* v1.2h check rollover *)
    END;

    IF fsize = 0 THEN
        IF pos     > 1 THEN abort(errStartPastSize,sStartaddr);END;
        IF lastpos > 1 THEN abort(errEndPastSize,sEndaddr);END;
        IF NOT(nullcount) THEN abort(errEndPastSize,sEndaddr);END;
        lastpos:=pos;
        count:=0;
    ELSE
        IF pos     > fsize THEN abort(errStartPastSize,sStartaddr);END;
        IF lastpos > fsize THEN abort(errEndPastSize,sEndaddr);END;
        IF nullcount THEN
            lastpos:=pos;
            count:=0; (* done already ! for beauty ! *)
        ELSE
            IF lastpos < pos THEN abort(errBeforeStart,sEndaddr);END;
            count:=lastpos-pos+1;
        END;
    END;

    DEC(pos);
    DEC(lastpos);

    CASE samename OF
    | TRUE:
        Str.Copy(target,source);
        newExtension(source,target,extBAK);
        fileRename(useLFN,target,source);
        IF fileIsRO(useLFN,target)=FALSE  THEN
            wasRO:=FALSE;
        ELSE
            wasRO:=TRUE;
            fileSetRW(useLFN,target);
        END;
        doExtract( useLFN,source,target,pos,lastpos,count,
                   forXD,showascii,lowercase,eyecandy,cstring,bytesPerLine,
                   renamed,orgtarget);
        IF wasRO THEN fileSetRO(useLFN,target);END;
        IF nobackup THEN
            IF fileIsRO(useLFN,source) THEN fileSetRW(useLFN,source);END;
            fileErase(useLFN,source);
            WrLn;
            WrStr('"');WrStr(source);WrStr('" backup has been deleted !');WrLn;
        END;
    | FALSE:
        IF fileExists(useLFN,target) THEN
            IF same(source,target) THEN abort(errSameSourceTarget,""); END;
            IF overwrite THEN
                IF fileIsRO(useLFN,target) THEN
                    fileSetRW(useLFN,target); (* was abort(errReadonly,target) *)
                END;
            ELSE
                IF autorename THEN
                    Str.Copy(orgtarget,target);
                    IF newname(useLFN,target)=FALSE THEN
                        abort(errRenameFailure,target);
                    END;
                    renamed:=TRUE;
                ELSE
                    abort(errAlreadyExists,target);
                END;
           END;
        END;
        doExtract( useLFN,source,target,pos,lastpos,count,
                   forXD,showascii,lowercase,eyecandy,cstring,bytesPerLine,
                   renamed,orgtarget);
    END;

    abort(errNone,"");
END Xtract.



(*
--------D-216C00-----------------------------
INT 21 - DOS 4.0+ - EXTENDED OPEN/CREATE
	AX = 6C00h
	BL = open mode as in AL for normal open (see also AH=3Dh)
	    bit 7: inheritance
	    bits 4-6: sharing mode
	    bit 3 reserved
	    bits 0-2: access mode
		100 read-only, do not modify file's last-access time (DOS 7.0)
	BH = flags
	    bit 6 = auto commit on every write (see also AH=68h)
	    bit 5 = return error rather than doing INT 24h
	    bit 4 = (FAT32) extended size (allow 4GB files instead of 2GB)
	CX = create attribute (see #01769)
	DL = action if file exists/does not exist (see #01770)
	DH = 00h (reserved)
	DS:SI -> ASCIZ file name
Return: CF set on error
	   AX = error code (see #01680 at AH=59h/BX=0000h)
	CF clear if successful
	   AX = file handle
	   CX = status (see #01768)
Notes:	the PC LAN Program only supports existence actions (in DL) of 01h,
	  10h with sharing=compatibility, and 12h
	DR DOS reportedly does not support this function and does not return
	  an "invalid function call" error when this function is used.
	the documented bits of BX are stored in the SFT when the file is opened
	  (see #01641,#01642)
BUG:	this function has bugs (at least in DOS 5.0 and 6.2) when used with
	  drives handled via the network redirector (INT 2F/AX=112Eh):
	    - CX (attribute) is not passed to the redirector if DL=11h,
	    - CX does not return the status, it is returned unchanged because
	      DOS does a PUSH CX/POP CX when calling the redirector.
SeeAlso: AH=3Ch,AH=3Dh,AX=6C01h,AX=7160h/CL=00h,INT 2F/AX=112Eh

(Table 01768)
Values for extended open function status:
 01h	file opened
 02h	file created
 03h	file replaced

Bitfields for file create attribute:
Bit(s)	Description	(Table 01769)
 6-15	reserved
 5	archive
 4	reserved
 3	volume label
 2	system
 1	hidden
 0	readonly

Bitfields for action:
Bit(s)	Description	(Table 01770)
 7-4	action if file does not exist
	0000 fail
	0001 create
 3-0	action if file exists
	0000 fail
	0001 open
	0010 replace/open

*)


