
(* ---------------------------------------------------------------
Title         Q&D TV Off
Author        PhG
Overview      clear text screen
Notes         assume VGA and page 0
              cursor disappears at return to the prompt : block char ? paper attribute ?
              well, seems cured by forcing "Far" when moving screen !
Bugs          note text screen is not square, hence the crux visual quirk in 80x25
Wish List     hide cursor, fade, sound ?

--------------------------------------------------------------- *)

MODULE TVoff;

IMPORT SYSTEM;
IMPORT Lib;
IMPORT Str;
IMPORT MATHLIB;

FROM IO IMPORT WrStr,WrLn;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits;

(* ------------------------------------------------------------ *)

CONST
    ProgEXEname   = "TVOFF";
    ProgTitle     = "Q&D TV off";
    ProgVersion   = "v1.1";
    ProgCopyright = "by PhG";
CONST
    errNone         = 0;
    errHelp         = 1;
    errOption       = 2;
    errParm         = 3;
    errTooManyParms = 4;
    errUnexpectedCols = 5;
    errUnexpectedRows = 6;
    errUnexpectedCells= 7;
    errBadMethod      = 8;

CONST
    dash  = "-";
    slash = "/";

(* ------------------------------------------------------------ *)

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
    nl = CHR(13)+CHR(10);
    help = nl+
ProgTitle+" "+ProgVersion+" "+ProgCopyright+nl+
nl+
"This program clears VGA text screen in a way similar to old TV sets."+nl+
nl+
"Syntax : "+ProgEXEname+" [[-|/]<0..6>]"+nl+
nl+
"0  persistence of hline, bright white twinkling (default)"+nl+
"1  grey twinkling"+nl+
"2  smoother motion (wand effect), persistence of hline, bright white twinkling"+nl+
"3  smoother motion (wand effect), grey twinkling"+nl+
"4  same as 0 with a cross effect"+nl+
"5  same as 1 with a cross effect"+nl+
"6  same as 0 in 320x200 graphics mode"+nl;

VAR
    S : str256;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(help);
    | errOption  :          S := "Unknown option !";
    | errParm  :            S := "Illegal parameter !";
    | errTooManyParms:      S := "Too many parameters !";
    | errUnexpectedCols:    S := "Weird number of columns !";
    | errUnexpectedRows:    S := "Weird number of rows !";
    | errUnexpectedCells:   S := "Weird number of screen cells !";
    | errBadMethod:         S := "Illegal method !";
    ELSE
        S := "This is illogical, Captain !!!";
    END;
    CASE e OF
    | errNone, errHelp : ;
    ELSE
        WrLn;
        WrStr(ProgEXEname+" : "); WrStr(S); WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)

PROCEDURE waitVGAretrace ();
BEGIN
    WHILE (SYSTEM.In(03DAH) AND 08H) # 0 DO
    END;
    WHILE (SYSTEM.In(03DAH) AND 08H) = 0 DO
    END;
END waitVGAretrace;

PROCEDURE pause (n:CARDINAL);
VAR
    i:CARDINAL;
BEGIN
    FOR i := 1 TO n DO
        waitVGAretrace();
    END;
END pause;

(* ------------------------------------------------------------ *)

VAR
    bioscols  [00040H:004AH] : CARDINAL;
    biosrows  [00040H:0084H] : SHORTCARD; (* add 1 *)

TYPE
    inktype = (black,blue,green,cyan,red,magenta,brown,white,
               gray,bblue,bgreen,bcyan,bred,bmagenta,yellow,bwhite);
CONST
    minCol    = 1;
    maxCol    = 132;
    minRow    = 1;
    maxRow    = 60;
    firstcell = 0;
    maxcell   = (maxCol * maxRow) -1;
TYPE
    vcell = RECORD
        ch   : CHAR;
        attr : SHORTCARD;
    END;
VAR
    screen [0B800H:0000H] : ARRAY [firstcell..maxcell] OF vcell;
    savscr                : ARRAY [firstcell..maxcell] OF vcell;
    lastCol               : CARDINAL;
    lastRow               : CARDINAL;
    lastcell  : CARDINAL;
    Ybase     : ARRAY [minRow..maxRow] OF CARDINAL;

PROCEDURE initYbase ();
VAR
    i,p:CARDINAL;
BEGIN
    p := 0;
    FOR i := minRow TO lastRow DO
        Ybase[i]:=p;
        INC(p,lastCol);
    END;
END initYbase;

(* force far here ! *)

PROCEDURE pushscreen ();
BEGIN
    Lib.FarWordMove(FarADR(screen),FarADR(savscr),lastcell); (* each cell is a WORD *)
END pushscreen;

PROCEDURE popattributes ();
VAR
    i:CARDINAL;
BEGIN
    FOR i := firstcell TO lastcell DO
        screen[i].attr := (savscr[i].attr AND 0FH);
    END;
END popattributes;

(* ------------------------------------------------------------ *)

PROCEDURE plot (x,y:CARDINAL;ch:CHAR;attr:SHORTCARD);
VAR
    p : CARDINAL;
BEGIN
    p := Ybase[y]+x-minCol;    (* remember first column is 1, not 0 *)
    screen[p].ch   := ch;
    screen[p].attr := attr;
END plot;

PROCEDURE hline (x,y:CARDINAL;ch:CHAR;attr:SHORTCARD;xx:CARDINAL);
VAR
    n : CARDINAL;
BEGIN
    FOR n := x TO xx DO
        plot(n,y,ch,attr);
    END;
END hline;

PROCEDURE vline (x,y:CARDINAL;ch:CHAR;attr:SHORTCARD;yy:CARDINAL);
VAR
    n : CARDINAL;
BEGIN
    FOR n := y TO yy DO
        plot(x,n,ch,attr);
    END;
END vline;

(* frame : assume x<xx and y<yy, and now ignore overlap h/v (in case frame is a line) *)

PROCEDURE fplot (x,y:CARDINAL;ch:CHAR;attr:SHORTCARD;xx,yy:CARDINAL);
BEGIN
    hline( x,y  ,ch,attr,xx  );
    hline( x,yy ,ch,attr,xx  );
    (*
    vline( x,y+1,ch,attr,yy-1);
    vline(xx,y+1,ch,attr,yy-1);
    *)
    vline( x,y  ,ch,attr,yy  );
    vline(xx,y  ,ch,attr,yy  );
END fplot;

(* ------------------------------------------------------------ *)

PROCEDURE limit (n:CARDINAL):CARDINAL;
VAR
    r : CARDINAL;
BEGIN
    r := n DIV 2;
    IF ODD(n) THEN INC(r); END;
    RETURN r;
END limit;

PROCEDURE paperink (color:inktype) : SHORTCARD;
BEGIN
    RETURN SHORTCARD(ORD(color));
END paperink;

VAR (* those were constants ! *)
    gdash,gdashv,gdashx, dim, med, pure : CHAR;
    blank, blankhalfupper, blankhalflower : CHAR;
    sAnim : str16;
    maxindex :CARDINAL;

PROCEDURE initChars (  );
BEGIN
    IF warning95() THEN (* redefine some chars ? 249 is no longer a dot here *)
        gdash          := "";
        gdashv         := CHR(179);
        gdashx         := CHR(197);
        dim            := "";
        med            := "";
        pure           := "";
        blank          := " ";
        blankhalfupper := "";
        blankhalflower := "";
        sAnim          := CHR(250)+CHR(7);
        maxindex       := 2-1;
    ELSE
        gdash          := ""; (* 196 : a bigger one is ugly ! *)
        gdashv         := CHR(179);
        gdashx         := CHR(197);
        dim            := ""; (* 176 *)
        med            := ""; (* 177 *)
        pure           := ""; (* 219 *)
        blank          := " ";
        blankhalfupper := ""; (* 220 *)
        blankhalflower := ""; (* 223 *)
        sAnim          := ""+CHR(7); (* 250 249 7 -- "*" is not pretty *)
        maxindex       := 3-1;
    END;
END initChars;

(* ------------------------------------------------------------ *)

TYPE
    cleartype = (normal,implode,explode,
                 updowncenter,updowncentersmooth,
                 cruxcenter,cruxcentersmooth);

PROCEDURE cls (zechar : CHAR; ink : inktype; clear : cleartype;
              delay : CARDINAL;crux:BOOLEAN);
VAR
    x,y,n     : CARDINAL;
    attribute : SHORTCARD;
    limitCol,limitRow : CARDINAL;
BEGIN
    IF crux THEN
        CASE clear OF
        | updowncenter       : clear:=cruxcenter;
        | updowncentersmooth : clear:=cruxcentersmooth;
        END;
    END;

    attribute := paperink(ink);
    limitRow  := limit(lastRow);
    limitCol  := limit(lastCol);
    CASE clear OF
    | normal :
        FOR y := minRow TO lastRow DO
            FOR x := minCol TO lastCol DO
                plot(x,y,zechar,attribute);
            END;
        END;
    | implode :
        IF lastRow < lastCol THEN
            x := minCol;
            FOR y := minRow TO limitRow DO
                fplot(x,y,zechar,attribute,lastCol-x+1,lastRow-y+1);
                INC(x);
                pause(delay);
            END;
        ELSE
            y := minRow;
            FOR x := minCol TO limitCol DO
                fplot(x,y,zechar,attribute,lastCol-x+1,lastRow-y+1);
                INC(y);
                pause(delay);
            END;
        END;
    | explode :
        IF lastRow < lastCol THEN
            FOR y := limitRow TO minRow BY -1 DO
                x:=y;
                fplot(x,y,zechar,attribute,lastCol-x+1,lastRow-y+1);
                pause(delay);
            END;
        ELSE
            FOR x := limitCol TO minCol BY -1 DO
                y := x;
                fplot(x,y,zechar,attribute,lastCol-x+1,lastRow-y+1);
                pause(delay);
            END;
        END;
    | updowncenter:
        FOR y := minRow TO limitRow DO
            hline(minCol,y,zechar,attribute,lastCol);
            hline(minCol,lastRow-y+1,zechar,attribute,lastCol);
            pause(delay);
        END;
    | updowncentersmooth: (* ignore char : always assume blank instead *)
        FOR y := minRow TO limitRow DO
            hline(minCol,y,blankhalfupper,attribute,lastCol);
            hline(minCol,lastRow-y+1,blankhalflower,attribute,lastCol);
            pause(delay);
            zechar := blank;
            hline(minCol,y,zechar,attribute,lastCol);
            hline(minCol,lastRow-y+1,zechar,attribute,lastCol);
            pause(delay);
        END;
    | cruxcenter,cruxcentersmooth:
        IF limitRow < limitCol THEN
            n:=limitCol-limitRow;
            FOR y:=minRow TO limit(lastRow) DO
                fplot(minCol    ,minRow,zechar,attribute,minCol+n,y);
                fplot(lastCol-n ,minRow,zechar,attribute,lastCol ,y);

                fplot(minCol    ,lastRow-y+1,zechar,attribute,minCol+n,lastRow);
                fplot(lastCol-n ,lastRow-y+1,zechar,attribute,lastCol,lastRow);

                INC(n);
                pause(delay);
            END;
        ELSE
            (* don't bother : rows < columns *)
        END;
    END;
END cls;

(* ------------------------------------------------------------ *)

PROCEDURE showlastlines (zechar,zecharv,zecharx:CHAR;ink:inktype;crux:BOOLEAN);
VAR
    attribute : SHORTCARD;
    limitRow,limitCol:CARDINAL;
BEGIN
    attribute := paperink(ink);
    limitRow  := limit(lastRow);
    limitCol  := limit(lastCol);

    hline(minCol,limitRow,zechar,attribute,lastCol);
    IF crux THEN
       vline(limitCol,minRow,zecharv,attribute,lastRow);
       plot(limitCol,limitRow,zecharx,attribute);
    END;
END showlastlines;

PROCEDURE eraselastlines (zechar : CHAR; inkeraser : inktype;
                         delayeraser : CARDINAL;crux:BOOLEAN);
VAR
    attribute : SHORTCARD;
    limitRow,limitCol:CARDINAL;
    x,y,n,ndx,k : CARDINAL;
BEGIN
    attribute := paperink(inkeraser);
    limitRow  := limit(lastRow);
    limitCol  := limit(lastCol);

    IF crux THEN
        n   := limitCol-limitRow;
        ndx := 0;
        k   := 0;
    END;

    FOR x := minCol TO limitCol DO
        plot(x           ,limitRow,zechar,attribute);
        plot(lastCol-x+1 ,limitRow,zechar,attribute);
        IF crux THEN
            INC(ndx);
            IF ndx > n THEN
                plot(limitCol,minRow+k  ,zechar,attribute);
                plot(limitCol,lastRow-k ,zechar,attribute);
                INC(k);
            END;
        END;
        pause(delayeraser);
    END;
END eraselastlines;

PROCEDURE twinkle (inkchar : inktype;delaychar : CARDINAL;
                   inkeraser:inktype);
VAR
    attribute : SHORTCARD;
    x         : CARDINAL;
    y         : CARDINAL;
    i         : CARDINAL;
BEGIN
    y         := limit(lastRow);
    attribute := paperink(inkchar);
    x         := limit(lastCol);
    FOR i := 0 TO maxindex DO
        plot(x,y,sAnim[i],attribute);
        pause(delaychar);
    END;
    FOR i := maxindex-1 TO 0 BY -1 DO
        plot(x,y,sAnim[i],attribute);
        pause(delaychar);
    END;

    attribute := paperink(inkeraser);
    plot(x,y,blank,attribute); (* was pure *)

END twinkle;

(* ------------------------------------------------------------ *)

CONST
    iv = 010H; (* video interrupt is $10 *)
    sb = 040H; (* segBiosData *)

PROCEDURE isMonoMode ():BOOLEAN;
CONST
    MONO = BYTE(7); (* monochrome *)
VAR
    biosCurrentVideoMode  [sb:049H] : BYTE;
BEGIN
    RETURN (biosCurrentVideoMode = MONO);
END isMonoMode;

(* ------------------------------------------------------------ *)

VAR
    oldCursorShape : CARDINAL;

PROCEDURE setTextCursorShape (scanlines:CARDINAL );
VAR
    R : SYSTEM.Registers;
BEGIN
    R.CX := scanlines;
    R.AH := 01H; (* set text-mode cursor shape *)
    Lib.Intr(R,iv);
END setTextCursorShape;

TYPE
    cursorshapetype = (saveoldcursor, invisiblecursor, oldcursor);

PROCEDURE setCursorShape (shape:cursorshapetype);
VAR
    biosCursorType [sb:060H] : WORD;
    scanlines : CARDINAL;
BEGIN
    CASE shape OF
    | invisiblecursor: (* bits 6,5 : 01 *)
        IF isMonoMode () THEN
            scanlines := 02D0EH; (* 2b0c *)
        ELSE
            scanlines := 02607H;
        END;
    | oldcursor:
        scanlines := oldCursorShape;
    | saveoldcursor:
        oldCursorShape := biosCursorType;
        RETURN; (* force it here *)    END;
    setTextCursorShape(scanlines);
END setCursorShape;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    minalpha = 0;
    maxalpha = 90;
    ka       = 255;
VAR
    tcos : ARRAY [minalpha..maxalpha] OF INTEGER;
    tsin : ARRAY [minalpha..maxalpha] OF INTEGER;

PROCEDURE initcostable (  );
VAR
    pi,deg2rad,angle,d1,d2:LONGREAL;
    i : CARDINAL;
BEGIN
    pi := 4.0 * MATHLIB.ATan(1.0);
    deg2rad := pi / 180.0;
    FOR i := 0 TO 90 DO
        angle := LONGREAL(i) * deg2rad;
        d1 := MATHLIB.Cos(angle);
        tcos[i] := INTEGER(d1 * LONGREAL(ka) );

        angle := LONGREAL(i) * deg2rad;
        d2 := MATHLIB.Sin(angle);
        tsin[i] := INTEGER(d2 * LONGREAL(ka) );
    END;
END initcostable;


CONST
    hires       = 13H; (* 320x200x256 *)
    text        = 03H; (* 80x25 *)
    xcount      = 320;
    ycount      = 200;
    xmin        = 0;
    xmax        = xcount-1;
    ymin        = 0;
    ymax        = ycount-1;
    cx          = xmax DIV 2;
    cy          = ymax DIV 2;
    screensize  = xcount * ycount;
    screensizeW =(screensize DIV 2);

VAR
    videoscreen [0A000H:0000H] : ARRAY [0..screensize-1] OF BYTE;
    grYbase                    : ARRAY [ymin..ymax] OF CARDINAL;

PROCEDURE clearvideoscreen (inkndx:CARDINAL);
BEGIN
    Lib.FarWordFill(FarADR(videoscreen),screensizeW,(inkndx << 8 + inkndx));
END clearvideoscreen;

PROCEDURE clearvideoscreenline (y,inkndx:CARDINAL);
CONST
    hlinesizeW = (xcount DIV 2);
VAR
    p:CARDINAL;
BEGIN
    p:=grYbase[y];
    Lib.FarWordFill(FarADR(videoscreen[p]),hlinesizeW,(inkndx << 8 + inkndx));
END clearvideoscreenline;

PROCEDURE grInitYbase ();
VAR
    i,p:CARDINAL;
BEGIN
    p := 0;
    FOR i := ymin TO ymax DO
        grYbase[i]:=p;
        INC(p,xcount);
    END;
END grInitYbase;

PROCEDURE vplot (x,y:CARDINAL;ink:BYTE);
BEGIN
    videoscreen[grYbase[y]+x]:=ink;
END vplot;

PROCEDURE zline(x1, y1, x2, y2 : CARDINAL; ink:BYTE);
VAR
    i, deltax, deltay, numpixels,
    d, dinc1, dinc2,
    x, xinc1, xinc2,
    y, yinc1, yinc2 : INTEGER;
BEGIN
    (* Calculate deltax and deltay for initialisation *)
    deltax := ABS( INTEGER(x2) - INTEGER(x1) );
    deltay := ABS( INTEGER(y2) - INTEGER(y1) );

    (* Initialize all vars based on which is the independent variable *)
    IF deltax >= deltay THEN
        (* x is independent variable *)
        numpixels := deltax + 1;
        d := (2 * deltay) - deltax;
        dinc1 := deltay << 1;
        dinc2 := (deltay - deltax) << 1;
        xinc1 := 1;
        xinc2 := 1;
        yinc1 := 0;
        yinc2 := 1;
    ELSE
        (* y is independent variable *)
        numpixels := deltay + 1;
        d := (2 * deltax) - deltay;
        dinc1 := deltax << 1;
        dinc2 := (deltax - deltay) << 1;
        xinc1 := 0;
        xinc2 := 1;
        yinc1 := 1;
        yinc2 := 1;
    END;

    (* Make sure x and y move in the right directions *)
    IF x1 > x2 THEN
        xinc1 := - xinc1;
        xinc2 := - xinc2;
    END;
    IF y1 > y2 THEN
        yinc1 := - yinc1;
        yinc2 := - yinc2;
    END;

    (* Start drawing at <x1, y1> *)
    x := CARDINAL(x1);
    y := CARDINAL(y1);

    (* Draw the pixels *)
    FOR i := 1 TO numpixels DO
        vplot(x,y,ink);
        IF d < 0 THEN
            d := d + dinc1;
            x := x + xinc1;
            y := y + yinc1;
        ELSE
            d := d + dinc2;
            x := x + xinc2;
            y := y + yinc2;
        END;
    END;
END zline;


PROCEDURE clipxy (VAR x,y:INTEGER);
BEGIN
    IF x < xmin THEN
        x:=xmin;
    ELSIF x > xmax THEN
        x:=xmax;
    END;
    IF y < ymin THEN
        y:=ymin;
    ELSIF y > ymax THEN
        y:=ymax;
    END;
END clipxy;

PROCEDURE zplot (x,y:CARDINAL;ink:BYTE);
BEGIN
    IF x < xmin THEN RETURN; END;
    IF x > xmax THEN RETURN; END;
    IF y < ymin THEN RETURN; END;
    IF y > ymax THEN RETURN; END;
    vplot(x,y,ink);
END zplot;

PROCEDURE plotsym (filled:BOOLEAN; cx,cy,dx,dy:INTEGER;ink:BYTE);
VAR
    x1,y1,x2,y2:INTEGER;
BEGIN
  IF filled THEN
    x1:=cx-dx;
    y1:=cy-dy;
    x2:=cx+dx;
    y2:=cy-dy;
    clipxy(x1,y1);
    clipxy(x2,y2);
    zline(x1,y1,x2,y2,ink);

    x1:=cx-dx;
    y1:=cy+dy;
    x2:=cx+dx;
    y2:=cy+dy;
    clipxy(x1,y1);
    clipxy(x2,y2);
    zline(x1,y1,x2,y2,ink);
  ELSE
    x1:=cx-dx;
    y1:=cy-dy;
    zplot(x1,y1,ink);
    x2:=cx+dx;
    y2:=cy-dy;
    zplot(x2,y2,ink);

    x1:=cx-dx;
    y1:=cy+dy;
    zplot(x1,y1,ink);
    x2:=cx+dx;
    y2:=cy+dy;
    zplot(x2,y2,ink);
  END;
END plotsym;

PROCEDURE aellipse (filled:BOOLEAN; cx,cy,rx,ry:INTEGER;ink:BYTE);
VAR
    i:CARDINAL;
    dx,dy,cosinus,sinus:INTEGER;
BEGIN
    FOR i:=minalpha TO maxalpha DO
        cosinus := tcos[i];
        dx := (cosinus * rx) DIV ka;
        sinus   := tsin[i];
        dy := (  sinus * ry) DIV ka;
        plotsym(filled,cx,cy,dx,dy,ink);
    END;
END aellipse;

PROCEDURE zcircle (filled:BOOLEAN; cx,cy,r:INTEGER;ink:BYTE);
BEGIN
    aellipse(filled,cx,cy,r,r,ink);
END zcircle;

PROCEDURE setVideoMode (mode:CARDINAL);
VAR
    R : SYSTEM.Registers;
BEGIN
    R.AH := 00H;
    R.AL := BYTE(mode);
    Lib.Intr(R,10H);
END setVideoMode;

CONST
    gblack       = LONGCARD(0000000H);

CONST
    egarange     = 40H;
    mininkindex  = 0;
    maxinkindex  = 256-1;
    ndxblack     = mininkindex;
    maxcolorindex=egarange-1; (* we use 64 colors palette here *)
    ndxwhite     = maxinkindex;

TYPE
    triplet = RECORD
        r,g,b:SHORTCARD;
    END;
VAR
    orgpal     : ARRAY [0..255] OF triplet;

PROCEDURE setDAC(index:CARDINAL;red,green,blue:BYTE);
CONST
    DACWriteIndex  = 03C8H;
    DACDataRegister= 03C9H;
BEGIN
    SYSTEM.Out (DACWriteIndex,SHORTCARD(index));
    SYSTEM.Out (DACDataRegister, red);
    SYSTEM.Out (DACDataRegister, green);
    SYSTEM.Out (DACDataRegister, blue);
END setDAC;

PROCEDURE resetdac(slow:BOOLEAN;index,red,green,blue:CARDINAL);
BEGIN
    IF slow THEN pause(1);END;
    setDAC(index,BYTE(red),BYTE(green),BYTE(blue));
END resetdac;

PROCEDURE getDAC (index:CARDINAL;VAR r,g,b:SHORTCARD);
CONST
    DACReadIndex   = 03C7H;
    DACDataRegister= 03C9H;
BEGIN
    SYSTEM.Out (DACReadIndex,SHORTCARD(index));
    r:=SYSTEM.In (DACDataRegister);
    g:=SYSTEM.In (DACDataRegister);
    b:=SYSTEM.In (DACDataRegister);
END getDAC;

PROCEDURE readDACrgb (index:CARDINAL):LONGCARD;
VAR
    r,g,b:SHORTCARD;
    rgb : LONGCARD;
BEGIN
    getDAC(index,r,g,b);
    rgb := (LONGCARD(r) << 16) + (LONGCARD(g) << 8) + LONGCARD(b);
    RETURN rgb;
END readDACrgb;

PROCEDURE savesystempal (  );
VAR
    i:CARDINAL;
BEGIN
    FOR i := 0 TO 255 DO
        getDAC(i,orgpal[i].r,orgpal[i].g,orgpal[i].b);
    END;
END savesystempal;

PROCEDURE blend (slow:BOOLEAN;ndx,count:CARDINAL; startink, endink:LONGCARD);
CONST
    rshift = LONGCARD(16);
    gshift = LONGCARD(8);
VAR
    r1,g1,b1:INTEGER;
    r2,g2,b2:INTEGER;
    r,g,b:INTEGER;
    i : CARDINAL;
BEGIN
    IF count=0 THEN count := 1; END; (* safety to avoid division by 0 ! *)

    r1 := INTEGER (startink >> rshift) MOD egarange;
    g1 := INTEGER (startink >> gshift) MOD egarange;
    b1 := INTEGER (startink          ) MOD egarange;
    r2 := INTEGER (endink   >> rshift) MOD egarange;
    g2 := INTEGER (endink   >> gshift) MOD egarange;
    b2 := INTEGER (endink            ) MOD egarange;

    FOR i := 1 TO count DO
        r := r1 + ((r2 - r1) * INTEGER(i) ) DIV INTEGER(count);
        g := g1 + ((g2 - g1) * INTEGER(i) ) DIV INTEGER(count);
        b := b1 + ((b2 - b1) * INTEGER(i) ) DIV INTEGER(count);
        IF r < 0 THEN r:=0; END;
        IF g < 0 THEN g:=0; END;
        IF b < 0 THEN b:=0; END;
        IF r >= egarange THEN r:=egarange-1; END;
        IF g >= egarange THEN g:=egarange-1; END;
        IF b >= egarange THEN b:=egarange-1; END;
        IF slow THEN pause(1);END;
        setDAC(ndx+i-1,BYTE(r),BYTE(g),BYTE(b));
    END;
END blend;

TYPE
    palettetype = (systempal);

PROCEDURE newPalette64 (pal:palettetype;slow:BOOLEAN);
VAR
    ndx,n,i,j   : CARDINAL;
    r,g,b:BYTE;
    ir,ig,ib:CARDINAL;
BEGIN
    waitVGAretrace(); (* reduce noise on screen *)
    CASE pal OF
    | systempal: (* assume it was saved ! *)
        FOR i := 0 TO 255 DO
            setDAC(i,orgpal[i].r,orgpal[i].g,orgpal[i].b);
        END;
    END;
    setDAC(ndxblack,00H,00H,00H);    (* safety 0=black *)
    (* setDAC(ndxwhite,3FH,3FH,3FH);    (* safety 63=white *) *)
    n := maxinkindex-egarange+1;
    blend (FALSE,egarange,n,gblack,gblack);  (* 64..255=black *)
END newPalette64;

PROCEDURE grTVOFF (init:BOOLEAN);
CONST
    blanc= maxinkindex;
    gris = blanc-1;
    noir = ndxblack;
    xbytes=(xcount DIV 2);
    speed = 6;
    disquiet = 32;
VAR
    i,x,y:CARDINAL;
BEGIN
    i:=0;
    waitVGAretrace;
    resetdac(FALSE, blanc,i,i,i); (* en fait, redfini en noir *)

    i:=3*(egarange DIV 4); (* 015H *)
    waitVGAretrace;
    resetdac(FALSE,gris,i,i,i);

    IF init THEN
        clearvideoscreen(blanc); (* cran noir *)

        FOR i:= 0 TO egarange-1 DO
            pause(1);
            resetdac(FALSE,blanc,i,i,i);
        END;
        (* maintenant, l'cran est vraiment blanc *)
        pause(disquiet);
    END;

    (* from topbot to center *)

    FOR i:=ymin TO (ymax DIV 2) DO
        IF (i MOD speed) = 0 THEN waitVGAretrace; END;
        clearvideoscreenline(i,noir);
        clearvideoscreenline(ymax-i,noir);

        clearvideoscreenline(i+1,gris);
        clearvideoscreenline(ymax-i-1,gris);
    END;
    i:=(ymax DIV 2)+1;
    clearvideoscreenline(i,noir);

    (* from leftright to center *)

    y:=(ymax DIV 2);
    FOR i:=xmin TO (xmax DIV 2) DO
        IF (i MOD speed) = 0 THEN waitVGAretrace; END;
        vplot(i,y,noir);
        vplot(xmax-i,y,noir);
    END;
    vplot(cx,cy,gris);

    (* twinkle *)

    FOR i:=1 TO 4 DO
        pause(4);
        zcircle (FALSE, cx,cy, i  ,gris);
        zcircle (TRUE,  cx,cy, i-1,blanc);
    END;

    FOR i:=4 TO 2 BY -1 DO
        pause(4);
        zcircle (FALSE, cx,cy, i  ,noir);
        zcircle (TRUE,  cx,cy, i-1,gris);
        zcircle (TRUE,  cx,cy, i-2,blanc);
    END;

    pause(disquiet );

    FOR i:=2 TO 0 BY -1 DO
        pause(4);
        zcircle (FALSE, cx,cy, i  ,noir);
    END;

    pause(disquiet << 1);

END grTVOFF;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE getmethod (ch:CHAR; VAR method:CARDINAL;VAR crux:BOOLEAN):BOOLEAN;
VAR
    p : CARDINAL;
BEGIN
    p:=Str.CharPos("0123456",ch);
    crux:=FALSE;
    CASE p OF
    | 0 : method:=0;
    | 1 : method:=1;
    | 2 : method:=2;
    | 3 : method:=3;
    | 4 : method:=0; crux:=TRUE;
    | 5 : method:=1; crux:=TRUE;
    | 6 : method:=100;
    ELSE
          RETURN FALSE;
    END;
    RETURN TRUE;
END getmethod;

PROCEDURE adj (k:CARDINAL; VAR v:CARDINAL);
BEGIN
    IF k # 1 THEN
        IF v=0 THEN v:=1; END;
    END;
    v:=v*k;
END adj;

VAR
    parmcount,i,opt:CARDINAL;
    S,R : str128;
    method,k : CARDINAL;
    implodeDelay,explodeDelay,updownDelay,lineDelay,charDelay,dashDelay:CARDINAL;
    ok,fixcursor,crux : BOOLEAN;
    state:(waiting,gotparm);
BEGIN
    Lib.DisableBreakCheck();

    fixcursor := TRUE;
    ok:=getmethod("0", method,crux);

    state:=waiting;
    parmcount := Lib.ParamCount();
    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i); cleantabs(S);
        Str.Copy(R,S);
        UpperCase(R);
        IF isOption(R) THEN
            opt := GetOptIndex (R,"?"+delim+"H"+delim+"HELP"+delim+
                               "0"+delim+
                               "1"+delim+
                               "2"+delim+
                               "3"+delim+
                               "4"+delim+
                               "5"+delim+
                               "6"
                              );
            CASE opt OF
            | 1, 2, 3   : abort(errHelp,"");
            | 4..11     : ok:=getmethod(R[0+1], method, crux); (* skip dash/slash *)
            ELSE
                abort(errOption,S);
            END;
        ELSE
    	    CASE state OF
	        | waiting   :
	            IF getmethod(R[0], method, crux)=FALSE THEN
	                abort(errBadMethod,S);
	            END;
            | gotparm   :
                abort(errTooManyParms,S);
            END;
            INC(state);
        END;
    END;

CASE method OF
| 100:
    grInitYbase;
    initcostable;
    setVideoMode(hires);
    savesystempal();
    setDAC(ndxblack,00H,00H,00H);    (* safety 0=black *)
    clearvideoscreen(ndxblack); (* cran noir *)
    pause(4);
    grTVOFF(TRUE );
    newPalette64(systempal,FALSE);
    setVideoMode(text);
ELSE

    lastCol  := bioscols;
    lastRow  := CARDINAL(biosrows)+1;
    lastcell := (lastCol * lastRow) -1;

    IF lastCol > maxCol THEN abort(errUnexpectedCols,"");END;
    IF lastRow > maxRow THEN abort(errUnexpectedRows,"");END;
    IF lastcell > maxcell THEN abort(errUnexpectedCells,"");END;

    initChars;

    IF fixcursor THEN
        setCursorShape (saveoldcursor);
        setCursorShape (invisiblecursor);
    END;

    initYbase();

    IF lastRow > 25 THEN
        implodeDelay := 0;
        explodeDelay := 0;
        updownDelay  := 1;
        lineDelay    := 1;
        charDelay    := 12;
        dashDelay    := 9;
    ELSE
        implodeDelay := 0;
        explodeDelay := 0;
        updownDelay  := 2; (* force pause a little *)
        lineDelay    := 1;
        charDelay    := 12;
        dashDelay    := 9;
    END;

    k:=1; (* [1..8], default is 1, 8 is very slow *)
    adj(k,implodeDelay);
    adj(k,explodeDelay);
    adj(k,updownDelay);
    adj(k,lineDelay);
    adj(k,charDelay);
    adj(k,dashDelay);

    pushscreen();
    CASE method OF
    | 0 :
        cls(blank,black,implode,implodeDelay,crux);
        cls(pure,white,explode,explodeDelay,crux);
        cls(pure,bwhite,explode,explodeDelay,crux);
        cls(blank,black,updowncenter,updownDelay,crux);
        showlastlines(gdash,gdashv,gdashx,white,crux);
        pause(dashDelay);                               (* specific *)
        eraselastlines(blank,black,lineDelay,crux);
        twinkle(bwhite,charDelay,black);                (* bright white *)
    | 1 :
        cls(blank,black,implode,implodeDelay,crux);
        cls(pure,white,explode,explodeDelay,crux);
        cls(pure,bwhite,explode,explodeDelay,crux);
        cls(blank,black,updowncenter,updownDelay,crux);
        showlastlines(gdash,gdashv,gdashx,white,crux);
        eraselastlines(blank,black,lineDelay,crux);
        twinkle(white,charDelay,black);                 (* greyish white *)
    | 2 :
        updownDelay := 1;
        cls(blank,black,implode,implodeDelay,crux);
        cls(pure,white,explode,explodeDelay,crux);
        cls(pure,bwhite,explode,explodeDelay,crux);
        cls(blank,black,updowncentersmooth,updownDelay,crux); (* smooth *)
        showlastlines(gdash,gdashv,gdashx,white,crux);
        pause(dashDelay);                               (* specific *)
        eraselastlines(blank,black,lineDelay,crux);
        twinkle(bwhite,charDelay,black);                (* bright white *)
    | 3 :
        updownDelay := 1;
        cls(blank,black,implode,implodeDelay,crux);
        cls(pure,white,explode,explodeDelay,crux);
        cls(pure,bwhite,explode,explodeDelay,crux);
        cls(blank,black,updowncentersmooth,updownDelay,crux); (* smooth *)
        showlastlines(gdash,gdashv,gdashx,white,crux);
        eraselastlines(blank,black,lineDelay,crux);
        twinkle(white,charDelay,black);                 (* greyish white *)
    END;
    popattributes(); (* restore blinking cursor now *)

    IF fixcursor THEN setCursorShape (oldcursor); END;

END;

    abort(errNone,"");
END TVoff.
