(* ---------------------------------------------------------------
Title         Q&D File Usage
Author        PhG
Overview      show disk usage
Usage         see help
Notes         very, very, very quick & dirty... :-(
              minimal error messages and checking, etc.
              we handle floppy/hd/CDROM
Bugs          yes, TopSpeed did it again !
              if a tab ends command line (and therefore last parameter),
              it is included by Lib.ParamStr() !!!
              we must only specify legal existing drives !
Wish List     "cosmetic" display quirk : long paths trash paging
              (seen with win9X systems)

--------------------------------------------------------------- *)

MODULE fu;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;
IMPORT IO;

FROM FIO IMPORT FIXEDLIBS;

FROM Storage IMPORT ALLOCATE,DEALLOCATE,Available;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite,
getFileSize, verifyString, str4096, unfixDirectory, cleantabs,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits,
getAllLegalUnits;

FROM QD_Text IMPORT
colortype, cursorshapetype, scrolltype,
ff, cr, lf, bs, tab, nl, mincolor, maxcolor,
BW40, CO40, BW80, CO80, CO80x43, CO80x50, MONO,
vesa80x60, vesa132x25, vesa132x43, vesa132x50, vesa132x60,
selectCursorEmulation, setCursorShape,
handleVesa, setBrightPaper, setBlinkMode,
setFillChar, setFillInk, setFillPaper,
setFillInkPaper, setTxtInk, setTxtPaper, setTxtInkPaper, setWrapMode,
setUseBiosMode, setTabWidth, getScreenData, setWindow,
setMode, restoreMode,
gotoXY, xyToHtabVtab, home, setVisualPage, setActivePage,
scrollWindow, fillTextAttr, cls, writeStr, writeLn, getScreenWidth,
getScreenHeight, getScreenMinX, getScreenMinY, getScreenMaxX, getScreenMaxY,
getMinHtab, getMaxHtab, getMinVtab, getMaxVtab, getUseBiosMode,
getMinHtab, getMaxHtab, getMinVtab, getMaxVtab, getHtab, getVtab,
getWindowWidth, getWindowHeight, initScreenConsole,
findInkPaperAtStartup, getInkAtStartup, getPaperAtStartup;

FROM QD_LFN IMPORT path9X, huge9X, findDataRecordType,
unicodeConversionFlagType, w9XchangeDir,
w9XgetDOSversion, w9XgetTrueDOSversion, w9XisWindowsEnh, w9XisMSDOS7,
w9XfindFirst, w9XfindNext, w9XfindClose, w9XgetCurrentDirectory,
w9XlongToShort, w9XshortToLong, w9XtrueName, w9XchangeDir,
w9XmakeDir, w9XrmDir, w9Xrename, w9XsupportLFN;

(* ------------------------------------------------------------ *)

TYPE
    pathtype = path9X;
    str3     = ARRAY [0..2] OF CHAR; (* DOS extension *)

(* ------------------------------------------------------------ *)

TYPE
    HUGECARD = LONGREAL;
CONST
    HUGEZERO = 0.0;

CONST
    CONSOLE = TRUE; (* if FALSE, do not use QD_Text library *)

(*%F CONSOLE  *)

PROCEDURE vidinit ( useBios:BOOLEAN );
BEGIN
END vidinit;

PROCEDURE print (S : ARRAY OF CHAR  );
BEGIN
    IO.WrStr(S);
END print;

PROCEDURE newline (  );
BEGIN
    IO.WrLn;
END newline;

PROCEDURE colorheader ();
BEGIN
END colorheader;

PROCEDURE colorhelp ();
BEGIN
END colorhelp;

PROCEDURE colortext ();
BEGIN
END colortext;

PROCEDURE colorwarning ();
BEGIN
END colorwarning;

PROCEDURE getRows (  ):CARDINAL ;
CONST
    segBIOSdata = 040H;
VAR
    vrows  [segBIOSdata:0084H] : SHORTCARD; (* add 1 *)
BEGIN
    RETURN ( CARDINAL(vrows) +1 ); (* 25, 43, 50, etc. *)
END getRows;

PROCEDURE getColumns (  ):CARDINAL ;
CONST
    segBIOSdata = 040H;
VAR
    vcolumns [segBIOSdata:004AH] : CARDINAL; (* a word *)
BEGIN
    RETURN vcolumns;
END getColumns;

(*%E  *)

(*%T CONSOLE  *)

PROCEDURE vidinit ( useBios:BOOLEAN  );
BEGIN
    (* handleVesa; *) (* useless, because we won't change video mode *)
    setUseBiosMode ( useBios );
    findInkPaperAtStartup();
END vidinit;

PROCEDURE print (S : ARRAY OF CHAR  );
BEGIN
    writeStr(S);
END print;

PROCEDURE newline (  );
BEGIN
    writeLn;
END newline;

PROCEDURE colorhelp ();
BEGIN
    setTxtInk(getInkAtStartup());      (* was green *)
    setTxtPaper(getPaperAtStartup());  (* was black *)
    setFillInkPaper(getInkAtStartup(),getPaperAtStartup());
END colorhelp;

PROCEDURE colorheader ();
BEGIN
    setTxtInk(yellow);
    setTxtPaper(black);
    setFillInkPaper(getInkAtStartup(),getPaperAtStartup());
END colorheader;

PROCEDURE colortext ();
BEGIN
    setTxtInk(cyan);
    setTxtPaper(black);
    setFillInkPaper(getInkAtStartup(),getPaperAtStartup());
END colortext;

PROCEDURE colorwarning ();
BEGIN
    setTxtInk(red);
    setTxtPaper(white);
    setFillInkPaper(getInkAtStartup(),getPaperAtStartup());
END colorwarning;

PROCEDURE getRows (  ):CARDINAL ;
BEGIN
    RETURN getWindowHeight();
END getRows;

PROCEDURE getColumns (  ):CARDINAL ;
BEGIN
    RETURN getWindowWidth();
END getColumns;

(*%E  *)

(* ------------------------------------------------------------ *)

CONST
    ProgEXEname   = "FU";
    ProgTitle     = "Q&D File Usage";
    ProgVersion   = "v1.0o";
    ProgCopyright = "by PhG";
    Banner        = ProgTitle+" "+ProgVersion+" "+ProgCopyright;
CONST
    colon         = ":";
    backslash     = "\";
    star          = "*";
    question      = "?";
    blank         = " ";
    coma          = ",";
    dot           = ".";
    dotdot        = dot+dot;
    stardotstar   = star+dot+star;
    nullchar      = CHR(0);
    dquote        = '"';
    extEXE        = ".EXE";
    extINI        = ".INI";
CONST
    initpagingcounter = 1; (* account for message *)
    kbdmsg = "Hit (almost) any key to continue or Escape to abort listing : ";
    cancel = CHR(27);
    msgsort = "Sorting, please wait (possibly a few minutes)... ";
CONST
    errNone             = 0;
    errHelp             = 1;
    errOption           = 2;
    errBadUnitSpec      = 3;
    errTooManyUnits     = 4;
    errAlreadySpecified = 5;
    errNotFound         = 6;
    errNoUnitSpecified  = 7;
    errSortConflict     = 8;
    errTooManyDirs      = 9;
    errAllocate         = 10;
    errAbortedByUser    = 11;
    errAttr             = 12;
    errBadFileMask      = 13;
    errPhantomUnit      = 14;
    errOnePath          = 15;
    errBadPath          = 16;
    errJoker            = 17;
    errFile             = 18;
TYPE
    scantype = (normalandspecial,normal,special);

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
(*
CONST
    cr = CHR(13);
    lf = CHR(10);
    nl = cr+lf;
*)
CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)
    msgHelp =
Banner+nl+
nl+
"Syntax : "+ProgEXEname+" <u:|path>... <-d[d]|-e[e]|-y[y]|-i> [option]..."+nl+
nl+
"-d[d]  by directory (default), -dd = by root directory"+nl+
"-e[e]  by extension (use yearly total or just extension)"+nl+
"-y[y]  by year (use yearly total or just year)"+nl+
"-i[i]  by case-insensitive initial (-ii = sort by count instead of character)"+nl+
"-f[f]  by path (-d[d] forced accordingly, case-insensitive, accents ignored)"+nl+
"-f:$   file mask (????????.??? format, default is "+stardotstar+")"+nl+
"-r     reverse order"+nl+
"-s     do not show sum of all sizes and counts for specified files"+nl+
"-c     do not show count of files"+nl+
"-p     show percentage of used file space"+nl+
"-g[g]  show bar graph (-gg = -g -t)"+nl+
"-a[a]  do not use graphics separator lines (-aa = -a -t)"+nl+
"-t     do not use graphics characters in bar graph"+nl+
"-n     list only normal files (H and S excluded, default is -n -z)"+nl+
"-z     list only special files (H or S required, default is -n -z)"+nl+
"-u     paths and extensions in uppercase (real DOS or -l required)"+nl+
"-j     do not recurse"+nl+
"-k     paging (ignored if output redirected)"+nl+
"-l     disable LFN support even if available"+nl+
(*%T CONSOLE  *)
"-b     monochrome BIOS output (no colors)"+nl+
(*%E  *)
nl+
"a) If LFN support is available, and if -lfn option was NOT specified,"+nl+
"   directory display will use LFN paths instead of DOS paths."+nl+
"b) If program is run without any parameter, command line will be read from"+nl+
"   "+ProgEXEname+extINI+", if such a file exists in "+ProgEXEname+extEXE+" directory."+nl+
"c) Note R and A attributes are always ignored when building the list of files."+nl+
'd) Internally, program calls DOS functions : -e will only use ".???" format.'+nl+
"e) Warning : results will NOT be significant for CD-ROM and network drives."+nl;
VAR
    S : str256;
BEGIN
    colorhelp;
    CASE e OF
    | errHelp :
        print(msgHelp);
    | errOption :
        Str.Concat(S,"Unknown ",einfo);Str.Append(S," option !");
    | errBadUnitSpec:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," unit specification !");
    | errTooManyUnits:
        Str.Concat(S,"Overflow with ",einfo);Str.Append(S," unit specification !");
    | errAlreadySpecified:
        Str.Concat(S,"Duplicated ",einfo);Str.Append(S," unit specification !");
    | errNotFound:
        Str.Concat(S,"File ",einfo);Str.Append(S," does not exist !");
    | errNoUnitSpecified:
        S := "No unit specified !";
    | errSortConflict:
        S := "-d[d][f], -e[e], -y[y] and -i[i] options are mutually exclusive !";
    | errTooManyDirs:
        (*
        Str.Concat(S,"Too many subdirectories in ",einfo);
        Str.Append(S," !");
        *)
        S:="Too many directories while limit is 16000 !";
    | errAllocate:
        Str.Concat(S,"Storage.ALLOCATE() failure while processing ",einfo);
        Str.Append(S," directory !");
    | errAbortedByUser:
        S:="Aborted by user !";
    | errAttr:
        S := "-n and -z options are mutually exclusive !";
    | errBadFileMask:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," file mask !");
    | errPhantomUnit:
        Str.Concat(S,einfo,": unit does not exist !");
    | errOnePath:
        S:="Only one path may be specified !";
    | errBadPath:
         Str.Concat(S,"Illegal ",einfo);Str.Append(S," path !");
    | errJoker:
         Str.Concat(S,"At least one illegal joker in ",einfo);Str.Append(S," path !");
    | errFile:
         Str.Concat(S,einfo," is a file !");
    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp: ;
    ELSE
        print(ProgEXEname+" : "); print(S);
        newline;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

(* we expect a canonical path u:\xxx...xxx\ *)

PROCEDURE shortenVid (VAR R:ARRAY OF CHAR;
                     wiscreen,wimsg:CARDINAL;S:ARRAY OF CHAR );
VAR
    len:CARDINAL;
BEGIN
    len:=Str.Length(S);
    IF (wimsg+len) < (wiscreen DIV 2) THEN
        Str.Copy(R, S);
    ELSE
        Str.Slice(R, S,0,wiscreen-wimsg-3-1);
        Str.Append(R,"..."); (* not pretty but functional *)
    END;
END shortenVid;

(* ------------------------------------------------------------ *)

(* we use VAR as to go a little faster : nay, not even really faster in fact ! *)

PROCEDURE argc (cli : ARRAY OF CHAR; clean : BOOLEAN) : CARDINAL;
CONST
    quote = dquote;
VAR
    state    : (empty,intoken,instring);
    cliLen   : CARDINAL;
    cliPos   : CARDINAL;
    argCount : CARDINAL;
    char     : CHAR;
    code     : CARDINAL;
    arg      : str128;
BEGIN
    IF clean THEN
        LtrimBlanks(cli);
        RtrimBlanks(cli);
    END;
    cliLen   := Str.Length(cli);
    cliPos   := 1;
    argCount := 0;
    state    := empty;
    LOOP
        IF cliPos > cliLen THEN EXIT; END;
        char := cli[cliPos-1];
        code := ORD(char);
        CASE state OF
        | empty :
            IF code = ORD(quote) THEN
                state := instring;        (* begin new string *)
                Str.Copy(arg,char);
                INC(argCount);
            ELSIF code > ORD(blank) THEN  (* quote already trapped *)
                state := intoken;         (* begin new token *)
                Str.Copy(arg,char);
                INC(argCount);
            END;
        | intoken :
            IF code = ORD(quote) THEN     (* quote in token *)
                state := instring;        (* if string in parameter *)
                Str.Append(arg,char);
            ELSIF code > ORD(blank) THEN
                Str.Append(arg,char);
            ELSE
                state := empty;           (* end of token *)
            END;
        | instring :
            IF code = ORD(quote) THEN
                Str.Append(arg,char);
                state := empty;           (* end of string *)
            ELSIF code > ORD(blank) THEN
                Str.Append(arg,char);
            ELSE
                Str.Append(arg,blank);    (* remove TAB and controls if any *)
            END;
        END;
        INC(cliPos);
    END;
    RETURN argCount;
END argc;

PROCEDURE argv (VAR argument : ARRAY OF CHAR;
                cli : ARRAY OF CHAR; n : CARDINAL; clean:BOOLEAN);
CONST
    quote = dquote;
VAR
    state    : (empty,intoken,instring);
    cliLen   : CARDINAL;
    cliPos   : CARDINAL;
    argCount : CARDINAL;
    char     : CHAR;
    code     : CARDINAL;
    arg      : str128;
BEGIN
    IF ( (n < 1) OR (n > argc(cli,clean)) ) THEN
        Str.Copy(argument,"");
        RETURN;
    END;
    IF clean THEN
        LtrimBlanks(cli);
        RtrimBlanks(cli);
    END;
    cliLen   := Str.Length(cli);
    cliPos   := 1;
    argCount := 0;
    state    := empty;
    LOOP
        IF cliPos > cliLen THEN EXIT; END;
        char := cli[cliPos-1];
        code := ORD(char);
        CASE state OF
        | empty :
            IF n = argCount THEN EXIT; END; (* argV$ test *)
            IF code = ORD(quote) THEN
                state := instring;        (* begin new string *)
                Str.Copy(arg,char);
                INC(argCount);
            ELSIF code > ORD(blank) THEN  (* quote already trapped *)
                state := intoken;         (* begin new token *)
                Str.Copy(arg,char);
                INC(argCount);
            END;
        | intoken :
            IF code = ORD(quote) THEN     (* quote in token *)
                state := instring;        (* if string in parameter *)
                Str.Append(arg,char);
            ELSIF code > ORD(blank) THEN
                Str.Append(arg,char);
            ELSE
                state := empty;           (* end of token *)
            END;
        | instring :
            IF code = ORD(quote) THEN
                Str.Append(arg,char);
                state := empty;           (* end of string *)
            ELSIF code > ORD(blank) THEN
                Str.Append(arg,char);
            ELSE
                Str.Append(arg,blank);    (* remove TAB and controls if any *)
            END;
        END;
        INC(cliPos);
    END;
    Str.Copy(argument,arg);
END argv;

(* ------------------------------------------------------------ *)

PROCEDURE HUGEINC (VAR r:HUGECARD;n:HUGECARD );
BEGIN
    r := r + n;
END HUGEINC;

PROCEDURE fmthc (v : HUGECARD; pad:CHAR; sep:CHAR; field:INTEGER) : str80;
VAR
    S,R   : str80;
    len,i : CARDINAL;
    ok  : BOOLEAN;
    ch  : CHAR;
BEGIN
    Str.FixRealToStr( LONGREAL(v),0,S,ok);
    len:=Str.Length(S);
    R := "";
    FOR i := 1 TO len DO
        Str.Prepend(R,S[len-i]);
        IF i < len THEN
            IF (i MOD 3) = 0 THEN
                Str.Prepend(R,sep);
            END;
        END;
    END;
    LOOP
        IF INTEGER(Str.Length(R)) >= ABS(field) THEN EXIT; END;
        IF field < 0 THEN
            Str.Append(R,pad);  (* right alignment *)
        ELSE
            Str.Prepend(R,pad);
        END;
    END;
    RETURN R;
END fmthc;

PROCEDURE fmtlc (v : LONGCARD; pad:CHAR; sep:CHAR; field:INTEGER) : str80;
VAR
    S,R   : str80;
    len,i : CARDINAL;
    ok  : BOOLEAN;
    ch  : CHAR;
BEGIN
    Str.CardToStr(v,S,10,ok);
    len:=Str.Length(S);
    R := "";
    FOR i := 1 TO len DO
        Str.Prepend(R,S[len-i]);
        IF i < len THEN
            IF (i MOD 3) = 0 THEN
                Str.Prepend(R,sep);
            END;
        END;
    END;
    LOOP
        IF INTEGER(Str.Length(R)) >= ABS(field) THEN EXIT; END;
        IF field < 0 THEN
            Str.Append(R,pad);  (* right alignment *)
        ELSE
            Str.Prepend(R,pad);
        END;
    END;
    RETURN R;
END fmtlc;

PROCEDURE fmtstring (S:ARRAY OF CHAR; pad:CHAR; n:INTEGER ) : str80;
VAR
    R : str80;
BEGIN
    Str.Copy(R,S);
    LOOP
        IF INTEGER(Str.Length(R)) >= ABS(n) THEN EXIT; END;
        IF n < 0 THEN
            Str.Append(R,pad); (* right alignment *)
        ELSE
            Str.Prepend(R,pad);
        END;
    END;
    RETURN R;
END fmtstring;

PROCEDURE fmtpercent (v:LONGREAL;pad:CHAR;sep:CHAR;field:INTEGER):str80;
CONST
    digits = 1; (* after dot/coma *)
VAR
    R : str80;
    ok: BOOLEAN;
BEGIN
    Str.FixRealToStr(v,digits,R,ok);
    Str.Subst(R,dot,sep);
    RETURN fmtstring(R,pad,field);
END fmtpercent;

PROCEDURE fmtbar (percentused:LONGREAL;used,free:CHAR;field:CARDINAL):str80;
VAR
    R : str80;
    p,i : CARDINAL;
BEGIN
    percentused := (percentused / 100.0 ) * LONGREAL(field);
    p := CARDINAL(percentused + 0.5 ); (* round ! *)
    R := "";
    FOR i := 1 TO field DO
        IF i <= p THEN
           Str.Append(R,used);
        ELSE
           Str.Append(R,free);
        END;
    END;
    RETURN R;
END fmtbar;

(*
Year stored relative to 1980 (ex. 1988 stores as 8)
    year      month    day   

 F E D C B A 9 8 7 6 5 4 3 2 1 0   <-- Bit Number
*)

CONST
    yyMask=BITSET{9..15};
    yyShft=9;
    mmMask=BITSET{5..8};
    mmShft=5;
    ddMask=BITSET{0..4};
    ddShft=0;
CONST
    BIOSbaseYear = 1980;

PROCEDURE unpackdate (ymd:CARDINAL;VAR y,m,d:SHORTCARD);
BEGIN
    y := SHORTCARD( CARDINAL(BITSET(ymd) * yyMask) >> yyShft );
    m := SHORTCARD( CARDINAL(BITSET(ymd) * mmMask) >> mmShft );
    d := SHORTCARD( CARDINAL(BITSET(ymd) * ddMask) >> ddShft );
END unpackdate;

(* ------------------------------------------------------------ *)

CONST
    maxhash = 65535;

TYPE
    ptrLogEntry = POINTER TO LogEntry;
    LogEntry = RECORD
        next         : ptrLogEntry;
        rank         : CARDINAL;
        year         : SHORTCARD;
        totalforyear : HUGECARD; (* was LONGCARD *)
        countforyear : CARDINAL;
        hash         : CARDINAL;
        extension    : str3; (* ignore LFN : we use DOS directory FIO sub *)
        totalforext  : HUGECARD; (* was LONGCARD *)
        countforext  : CARDINAL;
        totalforinitial:HUGECARD;
        countforinitial:CARDINAL;
        initial      : CHAR;
    END;

CONST
    firstPath = 1;     (* was 0, but 1 required by Lib.QSort and by firstPath-1 of course ! *)
    maxPath   = 16000; (* could be 16383... *)

TYPE
    ptrPathInfoType = POINTER TO pathinfotype;
    pathinfotype  = RECORD
        totalsize : HUGECARD; (* was LONGCARD *)
        filecount : CARDINAL;
        showme    : BOOLEAN;
        slen      : SHORTCARD;
        string    : CHAR; (* variable length *)
    END;

VAR
    Path : ARRAY[firstPath..maxPath] OF ptrPathInfoType;

(* ------------------------------------------------------------ *)

PROCEDURE getentry (i:CARDINAL; VAR R : ARRAY OF CHAR);
VAR
    len:CARDINAL;
BEGIN
    len := CARDINAL(Path[i]^.slen);
    Lib.FastMove( ADR(Path[i]^.string),ADR(R),len);
    R[len]:=nullchar; (* REQUIRED safety ! *)
END getentry;

PROCEDURE setentry (i:CARDINAL; S : ARRAY OF CHAR  ) : BOOLEAN ;
VAR
    len,needed:CARDINAL;
BEGIN
    len    := Str.Length(S);
    needed := SIZE(pathinfotype)-SIZE(CHAR)+len;
    IF Available(needed) THEN
        ALLOCATE(Path[i],needed);
        Path[i]^.slen := SHORTCARD(len);
        Lib.FastMove( ADR(S),ADR(Path[i]^.string),len);
    ELSE
        Path[i]:=NIL;
    END;
    RETURN (Path[i] # NIL);
END setentry;

(* ------------------------------------------------------------ *)

PROCEDURE isLessPath (i,j : CARDINAL) : BOOLEAN;
VAR
    SI,SJ:str128;
BEGIN
    getentry(i,SI);
    getentry(j,SJ);
    RETURN ( Str.Compare(SI,SJ) < 0 );  (* -1 is less *)
END isLessPath;

PROCEDURE doSwapPath (i,j : CARDINAL);
VAR
    S : ptrPathInfoType;
BEGIN
    S:=Path[i];
    Path[i]:=Path[j];
    Path[j]:=S;
END doSwapPath;

PROCEDURE fixDirPath (VAR S : ARRAY OF CHAR);
BEGIN
    IF Str.Length(S)=0 THEN Str.Copy(S,backslash); END;
    fixDirectory(S);
END fixDirPath;

PROCEDURE isDirEntry (S : ARRAY OF CHAR) : BOOLEAN;
BEGIN
    IF same(S,dot) THEN RETURN TRUE; END;
    RETURN same(S,dotdot);
END isDirEntry;

PROCEDURE doDir (root : ARRAY OF CHAR;
                 VAR index:CARDINAL;VAR err:BOOLEAN);
VAR
    path    : str128;
    S       : str128;
    entry   : FIO.DirEntry;
    found   : BOOLEAN;
BEGIN
    IF index > maxPath THEN
        err:=TRUE;
        RETURN;
    END;
    (*
    Str.Copy(Path[index],root);
    fixDirPath(Path[index]); (* add required \ because it will be REQUIRED *)
    *)
    fixDirPath(root); (* add required \ because it will be REQUIRED *)
    IF setentry(index,root)=FALSE THEN
        err:=TRUE;
        RETURN;
    END;

    Str.Copy(path,root);
    fixDirPath(path); (* add required \ *)
    Str.Append(path,stardotstar); (* root\*.* *)

    found := FIO.ReadFirstEntry(path,everything,entry);
    WHILE found DO
        Work(cmdShow);
        Str.Copy(S,root);
        fixDirPath(S);
        Str.Append(S,entry.Name); (* root\f8e3 *)
        IF isDirEntry(entry.Name)=FALSE THEN (* skip . and .. *)
            IF aD IN entry.attr THEN
                INC(index);
                doDir(S,index,err);
            END;
        END;
        found :=FIO.ReadNextEntry(entry);
    END;
END doDir;

PROCEDURE buildPathList (rootdir : ARRAY OF CHAR;
                         flagRecurse,flagSorted:BOOLEAN;
                         VAR lastpath:CARDINAL) : BOOLEAN;
VAR
    i        : CARDINAL;
    error    : BOOLEAN;
BEGIN
    error := FALSE;
    (* i     := firstPath; *)
    i := lastpath;

    IF flagRecurse THEN
        Work(cmdInit);
        doDir(rootdir,i,error);
        Work(cmdStop);
        IF error=FALSE THEN
            IF flagSorted THEN Lib.QSort(i,isLessPath,doSwapPath); END;
        END;
    ELSE
        fixDirPath(rootdir); (* just in case ! but should not be necessary here *)
        (* Str.Copy(Path[i],rootdir); *)
        IF setentry(i,rootdir)=FALSE THEN RETURN FALSE;END;
    END;
    lastpath := i;
    error := NOT error; (* reverse flag and return true if no error *)
    RETURN error;
END buildPathList;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE foundYear (year: SHORTCARD; anchor:ptrLogEntry;
                     VAR newInList:ptrLogEntry) : BOOLEAN;
BEGIN
    newInList := anchor;
    WHILE newInList <> NIL DO
        IF newInList^.year = year THEN RETURN TRUE; END;
        newInList := newInList^.next;
    END;
    RETURN FALSE;
END foundYear;

PROCEDURE foundExt (hash:CARDINAL;ext:ARRAY OF CHAR;anchor:ptrLogEntry;
                    VAR newInList:ptrLogEntry  ) : BOOLEAN ;
BEGIN
    newInList := anchor;
    WHILE newInList <> NIL DO
        IF newInList^.hash = hash THEN
            IF same(newInList^.extension,ext) THEN RETURN TRUE; END;
        END;
        newInList := newInList^.next;
    END;
    RETURN FALSE;
END foundExt;

PROCEDURE foundInitial (c: CHAR;anchor:ptrLogEntry;
                    VAR newInList:ptrLogEntry  ) : BOOLEAN ;
BEGIN
    newInList := anchor;
    WHILE newInList <> NIL DO
        IF newInList^.initial = c THEN RETURN TRUE; END;
        newInList := newInList^.next;
    END;
    RETURN FALSE;
END foundInitial;

(* ------------------------------------------------------------ *)

PROCEDURE buildDataDir (scanmode:scantype;
                        spec,filemask:ARRAY OF CHAR;currdirndx:CARDINAL):BOOLEAN ;
VAR
    S : str128;
    found,ok: BOOLEAN;
    entry : FIO.DirEntry;
    total : HUGECARD;
    n     : CARDINAL ;
BEGIN

    (* Work(cmdInit); *)
    Str.Concat(S,spec,filemask); (* assume u:\xxx\ -- was stardotstar *)
    total := HUGEZERO;
    n     := 0;
    found := FIO.ReadFirstEntry(S,allfiles,entry); (* // was attr *)
    WHILE found DO
        (* Work(cmdShow); *)
        CASE scanmode OF
        | special: ok:=((aH IN entry.attr) OR (aS IN entry.attr));
        | normal : ok:=NOT ((aH IN entry.attr) OR (aS IN entry.attr));
        | normalandspecial:ok:=TRUE ;
        END;
      IF ok THEN
            HUGEINC (total, HUGECARD(entry.size) );
            INC(n);
      END;
        found:=FIO.ReadNextEntry(entry);
    END;
    Path[currdirndx]^.totalsize := total;
    Path[currdirndx]^.filecount := n;
    Path[currdirndx]^.showme    := TRUE;
    (* Work(cmdStop); *)
    RETURN TRUE;
END buildDataDir;

PROCEDURE buildDataYear (scanmode:scantype;
                         spec,filemask:ARRAY OF CHAR;currdirndx:CARDINAL;
                         VAR anchor,newInList:ptrLogEntry;
                         VAR count:CARDINAL ):BOOLEAN;
VAR
    S : str128;
    found,ok: BOOLEAN;
    entry : FIO.DirEntry;
    len,needed : CARDINAL;
    year,month,day:SHORTCARD;
    ptr : ptrLogEntry;
    fsize:HUGECARD;
BEGIN
    (* Work(cmdInit); *)
    Str.Concat(S,spec,filemask); (* assume u:\xxx\ -- was stardotstar *)
    found := FIO.ReadFirstEntry(S,everything,entry); (* // was attr *)
    WHILE found DO
        (* Work(cmdShow); *)
        CASE scanmode OF
        | special: ok:=((aH IN entry.attr) OR (aS IN entry.attr));
        | normal : ok:=NOT ((aH IN entry.attr) OR (aS IN entry.attr));
        | normalandspecial:ok:=TRUE ;
        END;
        IF ( (aD) IN entry.attr ) THEN ok:=FALSE;END; (* v1.0l fix *)
      IF ok THEN
        needed := SIZE(LogEntry);
        IF Available(needed)=FALSE THEN
             (* Work(cmdStop); *)
             RETURN FALSE;
        END;
        fsize:=HUGECARD(entry.size);
        unpackdate(entry.date,year,month,day); (* we'll add BIOSbaseYear later ! *)
        IF anchor=NIL THEN
            count                       := 1;
            ALLOCATE(newInList,needed);
            newInList^.rank             := count;
            newInList^.year             := year;
            newInList^.totalforyear     := fsize;
            newInList^.countforyear     := 1;
            newInList^.next             := NIL;

            anchor    := newInList;
        ELSE
            IF foundYear(year,anchor,ptr) THEN
                HUGEINC(ptr^.totalforyear, fsize );
                INC(ptr^.countforyear);
            ELSE
                INC(count);
                ALLOCATE(newInList^.next,needed);
                newInList                   := newInList^.next;
                newInList^.rank             := count;
                newInList^.year             := year;
                newInList^.totalforyear     := fsize;
                newInList^.countforyear     := 1;
                newInList^.next             := NIL;
            END;
        END;
      END;
        found:=FIO.ReadNextEntry(entry);
    END;
    (* Work(cmdStop); *)
    RETURN TRUE;
END buildDataYear;

PROCEDURE buildDataExt (scanmode:scantype;
                        spec,filemask:ARRAY OF CHAR;currdirndx:CARDINAL;
                        VAR anchor,newInList:ptrLogEntry;
                        VAR count:CARDINAL ):BOOLEAN;
VAR
    S,u,d : str128;
    n,e : str16;
    found,ok: BOOLEAN;
    entry : FIO.DirEntry;
    needed,hash : CARDINAL;
    ptr : ptrLogEntry;
    fsize:HUGECARD;
BEGIN
    (* Work(cmdInit); *)
    Str.Concat(S,spec,filemask); (* assume u:\xxx\ -- was stardotstar *)
    found := FIO.ReadFirstEntry(S,everything,entry); (* // was attr *)
    WHILE found DO
        (* Work(cmdShow); *)
        CASE scanmode OF
        | special: ok:=((aH IN entry.attr) OR (aS IN entry.attr));
        | normal : ok:=NOT ((aH IN entry.attr) OR (aS IN entry.attr));
        | normalandspecial:ok:=TRUE ;
        END;
        IF ( (aD) IN entry.attr ) THEN ok:=FALSE;END; (* v1.0l fix *)
      IF ok THEN
        needed := SIZE(LogEntry);
        IF Available(needed)=FALSE THEN
            (* Work(cmdStop); *)
            RETURN FALSE;
        END;
        fsize:=HUGECARD(entry.size);
        Lib.SplitAllPath(entry.Name,u,d,n,e);
        Str.Delete(e,0,1); (* remove leading "." *)
        Str.Lows(e);

        hash := Lib.HashString(e,maxhash);
        IF anchor=NIL THEN
            count                       := 1;
            ALLOCATE(newInList,needed);
            newInList^.rank             := count;
            Str.Copy(newInList^.extension,e);
            newInList^.hash             := hash;
            newInList^.totalforext      := fsize;
            newInList^.countforext      := 1;
            newInList^.next             := NIL;

            anchor    := newInList;
        ELSE
            IF foundExt(hash,e,anchor,ptr) THEN
                HUGEINC(ptr^.totalforext, fsize);
                INC(ptr^.countforext);
            ELSE
                INC(count);
                ALLOCATE(newInList^.next,needed);
                newInList                   := newInList^.next;
                newInList^.rank             := count;
                Str.Copy(newInList^.extension,e);
                newInList^.hash             := hash;
                newInList^.totalforext      := fsize;
                newInList^.countforext      := 1;
                newInList^.next             := NIL;
            END;
        END;
      END;
        found:=FIO.ReadNextEntry(entry);
    END;
    (* Work(cmdStop); *)
    RETURN TRUE;
END buildDataExt;

PROCEDURE buildDataInitial (scanmode:scantype;
                        spec,filemask:ARRAY OF CHAR;currdirndx:CARDINAL;
                        VAR anchor,newInList:ptrLogEntry;
                        VAR count:CARDINAL ):BOOLEAN;
VAR
    S,u,d : str128;
    n,e : str16;
    found,ok: BOOLEAN;
    entry : FIO.DirEntry;
    needed,hash : CARDINAL;
    ptr : ptrLogEntry;
    fsize:HUGECARD;
    c:CHAR;
BEGIN
    (* Work(cmdInit); *)
    Str.Concat(S,spec,filemask); (* assume u:\xxx\ -- was stardotstar *)
    found := FIO.ReadFirstEntry(S,everything,entry); (* // was attr *)
    WHILE found DO
        (* Work(cmdShow); *)
        CASE scanmode OF
        | special: ok:=((aH IN entry.attr) OR (aS IN entry.attr));
        | normal : ok:=NOT ((aH IN entry.attr) OR (aS IN entry.attr));
        | normalandspecial:ok:=TRUE ;
        END;
        IF ( (aD) IN entry.attr ) THEN ok:=FALSE;END; (* v1.0l fix *)
      IF ok THEN
        needed := SIZE(LogEntry);
        IF Available(needed)=FALSE THEN
            (* Work(cmdStop); *)
            RETURN FALSE;
        END;
        fsize:=HUGECARD(entry.size);
        Lib.SplitAllPath(entry.Name,u,d,n,e);
        Str.Copy(c,n[0]);
        LowerCase(c);
(* print(c);print(" -> ");print(n);newline; *)
        IF anchor=NIL THEN
            count                       := 1;
            ALLOCATE(newInList,needed);
            newInList^.rank             := count;
            Str.Copy(newInList^.initial,c);
            newInList^.totalforinitial  := fsize;
            newInList^.countforinitial  := 1;
            newInList^.next             := NIL;

            anchor    := newInList;
        ELSE
            IF foundInitial(c,anchor,ptr) THEN
                HUGEINC(ptr^.totalforinitial, fsize);
                INC(ptr^.countforinitial);
            ELSE
                INC(count);
                ALLOCATE(newInList^.next,needed);
                newInList                   := newInList^.next;
                newInList^.rank             := count;
                Str.Copy(newInList^.initial,c);
                newInList^.totalforinitial  := fsize;
                newInList^.countforinitial  := 1;
                newInList^.next             := NIL;
            END;
        END;
      END;
        found:=FIO.ReadNextEntry(entry);
    END;
    (* Work(cmdStop); *)
    RETURN TRUE;
END buildDataInitial;

(* ------------------------------------------------------------ *)

PROCEDURE freeData (newInList:ptrLogEntry  );
VAR
    nextInList:ptrLogEntry;
    needed:CARDINAL;
BEGIN
    WHILE newInList <> NIL DO
        needed := SIZE(LogEntry);
        nextInList := newInList^.next;
        DEALLOCATE(newInList,needed);
        newInList := nextInList;
    END
END freeData;

PROCEDURE freePaths (lastpath:CARDINAL);
VAR
    i,len,needed:CARDINAL ;
    newInList,nextInList : ptrLogEntry;
    R : str16;
BEGIN
    FOR i := firstPath TO lastpath DO
        len := CARDINAL( Path[i]^.slen);
        needed := SIZE(pathinfotype)-SIZE(CHAR)+len;
        DEALLOCATE(Path[i],needed);
        Path[i]:=NIL;
    END;
END freePaths;

(* ------------------------------------------------------------ *)

CONST
    graphdash  = CHR(196);  txtdash  = "-";
    graphbar   = CHR(205);  txtbar   = "=";
    chUsed     = CHR(176);  chUsedTxt = "+";
    chFree     = CHR(219);  chFreeTxt = "-";
    filler     = "$";
    comma      = ",";
    sepa       = "  ";
CONST
    modeDir = 0;
    modeYear= 1;
    modeExt = 2;
    modeInitial=3;
CONST
    usedfiller      =       "$$,$$$,$$$,$$$";
    filecountfiller =              "$$$,$$$";
    percentfiller   =                "$$$,$";
    barfiller       = "                    "; (* 20 spaces *)
    lastfiller      = "$$$$$$$$$$$$$$$$$$$$$$$$";  (* 2 chars more and we hit 80 columns ! *)
                   (*  path (hsra ????????.???) *)
CONST
    usedheader      =       "    Used bytes";
    filecountheader =              "  Files";
    percentheader   =                "%Used";
    barheader       = 'Used="$"   Total="$"';
    pathheader      = "Path";
    yearheader      = "Year";
    extensionheader = "Extension"; (* was "Ext." check hit 80 columns wall *)
    initialheader   = "Initial";
CONST
    norecurseheader = "-j ";
    recurseheader   = "";
    norecursetrailer= "-j, ";
    recursetrailer  = "";
    HSRAheader      = "HSRA";
    RAheader        = "RA";
    HSheader        = "HS";
    HSRAlong        = sepa+"(~all files, including system and/or hidden)";
    RAlong          = sepa+"(~all files, excluding system and/or hidden)";
    HSlong          = sepa+"(~only system and/or hidden files)";

PROCEDURE getWidths (VAR  wi,wic,wip,wib:CARDINAL   );
BEGIN
    wi := Str.Length(usedfiller);
    wic:= Str.Length(filecountfiller);
    wip:= Str.Length(percentfiller);
    wib:= Str.Length(barfiller);
END getWidths;


(* bytes, count, percentage, bar, sort_type (mask, files) *)

PROCEDURE doHeader (bymode:CARDINAL;filemask:ARRAY OF CHAR;
                    showcount,showpercent,showgraph,
                    ascii,semigraphics,flagRecurse:BOOLEAN;
                    scanmode:scantype);
VAR
    H1,H2:str128; (* 80 should do but... *)
    T1,T2:str80;
    i:CARDINAL;
    ch:CHAR;
BEGIN
    H1:=usedheader;
    H2:=usedfiller;
    IF showcount THEN
        Str.Append(H1,sepa);Str.Append(H1,filecountheader);
        Str.Append(H2,sepa);Str.Append(H2,filecountfiller);
    END;
    IF showpercent THEN
        Str.Append(H1,sepa);Str.Append(H1,percentheader);
        Str.Append(H2,sepa);Str.Append(H2,percentfiller);
    END;
    IF showgraph THEN
        Str.Append(H1,sepa);Str.Append(H1,barheader);
        Str.Append(H2,sepa);Str.Append(H2,barfiller);
    END;

    CASE bymode OF
    | modeDir:    T1:=pathheader;
    | modeYear:   T1:=yearheader;
    | modeExt:    T1:=extensionheader;
    | modeInitial:T1:=initialheader;
    END;
    Str.Append(H1,sepa);Str.Append(H1,T1);
    Str.Append(H2,sepa);Str.Append(H2,lastfiller);

    CASE scanmode OF
    | normalandspecial:  T1:=HSRAheader;
    | normal:            T1:=RAheader;
    | special:           T1:=HSheader;
    END;
    Str.Append(H1," (~~ ~)");
    IF flagRecurse THEN
        Str.Subst(H1,"~",recurseheader);
    ELSE
        Str.Subst(H1,"~",norecurseheader);
    END;
    Str.Subst (H1,"~",T1);
    Str.Subst (H1,"~",filemask);

    IF semigraphics THEN
        Str.Subst(H1,filler,chUsed);
        Str.Subst(H1,filler,chFree);
    ELSE
        Str.Subst(H1,filler,chUsedTxt);
        Str.Subst(H1,filler,chFreeTxt);
    END;

    IF ascii THEN
        ch := txtdash;
    ELSE
        ch := graphdash;
    END;
    ReplaceChar(H2,filler,ch);
    ReplaceChar(H2,comma,ch);

    colorheader;

    print(H1);newline;
    print(H2);newline;

    colortext;
END doHeader;

(* bytes, count, percentage, bar, sort_type (mask, files) *)

PROCEDURE doTrailer (sum:HUGECARD;files:CARDINAL;
                    showcount,showpercent,showgraph,ascii,flagRecurse:BOOLEAN;
                    scanmode:scantype);
VAR
    H1,H2:str128;
    SC:str80;
    i,wisep,n,wi,wic,wip,wib : CARDINAL;
    ch:CHAR;
BEGIN
    wisep:=Str.Length(sepa);
    getWidths(wi,wic,wip,wib);
    Str.Copy(H1, fmthc (sum, blank, dot, wi) );
    Str.Copy(H2,usedfiller);

    n:=0;
    IF showcount THEN
        SC := fmtlc ( LONGCARD(files), blank, dot, wic);
        Str.Append(H1,sepa);Str.Append(H1,SC);
        INC(n,wisep);INC(n,wic);
    END;
    IF showpercent THEN INC(n,wisep);INC(n,wip); END;
    IF showgraph THEN INC(n,wisep);INC(n,wib); END;
    INC(n,wisep);INC(n,Str.Length(lastfiller));
    IF n > wisep THEN
        Str.Append(H2,sepa);
        FOR i:=wisep+1 TO n DO Str.Append(H2,filler);END;
    END;

    IF ascii THEN
        ch := txtbar;
    ELSE
        ch := graphbar;
    END;
    ReplaceChar(H2,filler,ch);
    ReplaceChar(H2,comma,ch);

    CASE scanmode OF
    | normalandspecial: SC:=HSRAlong;
    | normal:           SC:=RAlong;
    | special:          SC:=HSlong;
    END;
    IF flagRecurse THEN
        Str.Subst(SC,"~",recursetrailer);
    ELSE
        Str.Subst(SC,"~",norecursetrailer);
    END;
    Str.Append(H1,SC);

    colorheader;
    print(H2);newline; (* bar first *)
    print(H1);newline; (* then values *)
    colortext;
END doTrailer;

(* ------------------------------------------------------------ *)

VAR
    globalAnchor : ptrLogEntry;

PROCEDURE getlogentry (n:CARDINAL):ptrLogEntry;
VAR
    newInList:ptrLogEntry;
BEGIN
    newInList := globalAnchor;
    WHILE newInList <> NIL DO
        IF newInList^.rank = n THEN RETURN newInList; END;
        newInList := newInList^.next;
    END;
    RETURN NIL;
END getlogentry;

(* ------------------------------------------------------------ *)

PROCEDURE isLessDir (i,j:CARDINAL   ):BOOLEAN ;
VAR
    si,sj:pathtype;
BEGIN
    getentry(i,si);
    getentry(j,sj);
    UpperCase(si);
    UpperCase(sj);
    RETURN ( Str.Compare(si,sj) < 0 );
END isLessDir;

PROCEDURE isLessDirReversed (i,j:CARDINAL   ):BOOLEAN ;
VAR
    si,sj:pathtype;
BEGIN
    getentry(i,si);
    getentry(j,sj);
    UpperCase(si);
    UpperCase(sj);
    RETURN ( Str.Compare(si,sj) > 0 );
END isLessDirReversed;

PROCEDURE isLessDirSize (i,j:CARDINAL   ):BOOLEAN ;
BEGIN
    RETURN ( Path[i]^.totalsize < Path[j]^.totalsize );
END isLessDirSize;

PROCEDURE isLessDirSizeReversed (i,j:CARDINAL   ):BOOLEAN ;
BEGIN
    RETURN ( Path[i]^.totalsize > Path[j]^.totalsize );
END isLessDirSizeReversed;

PROCEDURE doSwapDirSize (i,j:CARDINAL);
VAR
    tmp : ptrPathInfoType;
BEGIN
    tmp       := Path[i];
    Path[i]   := Path[j];
    Path[j]   := tmp;
END doSwapDirSize;

PROCEDURE doByDir (filemask:ARRAY OF CHAR;lastpath:CARDINAL;
                   reversed,showcount,showpercent,showgraph,
                   ascii,semigraphics,useLFN,uppercase,
                   showsum,paging,flagRecurse,justdir:BOOLEAN;
                   scanmode:scantype);
VAR
    i,wi,wic,wip,wib : CARDINAL;
    R:str1024; (* oversized *)
    S:str128;
    udne,shortform,longform : pathtype;
    S1,S2,S3,SC : str80;
    dirsize,sum : HUGECARD;
    percentused:LONGREAL;
    rc,filesperdir,totalfiles  : CARDINAL;
    charUsed,charFree:CHAR;
VAR
    lastRow               : CARDINAL;
    rowcount              : CARDINAL;
    key                   : str2;
BEGIN
    IF semigraphics THEN
        charUsed:=chUsed;     charFree:=chFree;
    ELSE
        charUsed:=chUsedTxt;  charFree:=chFreeTxt;
    END;

    video(msgsort,TRUE);
    IF justdir THEN
        IF reversed THEN
            Lib.QSort(lastpath,isLessDirReversed,doSwapDirSize);
        ELSE
            Lib.QSort(lastpath,isLessDir,doSwapDirSize);
        END;
    ELSE
        IF reversed THEN
            Lib.QSort(lastpath,isLessDirSizeReversed,doSwapDirSize);
        ELSE
            Lib.QSort(lastpath,isLessDirSize,doSwapDirSize);
        END;
    END;
    video(msgsort,FALSE);

    sum:=HUGEZERO;
    FOR i:=firstPath TO lastpath DO
        IF Path[i]^.showme THEN HUGEINC(sum,Path[i]^.totalsize); END;
    END;

    doHeader (modeDir,filemask,
             showcount,showpercent,showgraph,ascii,semigraphics,flagRecurse,
             scanmode);
    getWidths(wi,wic,wip,wib);

    IF paging THEN
        lastRow  := getWindowHeight();
        rowcount := initpagingcounter;
    END;

    totalfiles := 0;

    FOR i:=firstPath TO lastpath DO
      IF Path[i]^.showme THEN
        getentry(i,S);
        IF useLFN THEN
            Str.Copy(shortform,S);
            IF w9XshortToLong(shortform,rc,longform) THEN
                Str.Concat(udne,dquote,longform);Str.Append(udne,dquote);
            ELSE
                Str.Copy(udne,S);
            END;
        ELSE
            IF uppercase THEN
                UpperCase(S);
            ELSE
                LowerCase(S);
            END;
            Str.Copy(udne,S);
        END;

        dirsize := Path[i]^.totalsize;
        filesperdir:=Path[i]^.filecount;
        INC(totalfiles,filesperdir);
        percentused := (LONGREAL(dirsize) / LONGREAL(sum) ) * 100.0;

        S1 := fmthc (dirsize, blank, dot, wi);
        SC := fmtlc ( LONGCARD(filesperdir), blank, dot, wic);
        S2 := fmtpercent(percentused,blank,coma,wip);
        S3 := fmtbar(percentused,charUsed,charFree,wib);
        Str.Copy(R,S1);
        IF showcount THEN Str.Append(R,sepa);Str.Append(R,SC);END;
        IF showpercent THEN Str.Append(R,sepa);Str.Append(R,S2);END;
        IF showgraph THEN Str.Append(R,sepa);Str.Append(R,S3);END;
        Str.Append(R,sepa);Str.Append(R,udne);
        print(R);newline;

        IF paging THEN
            INC(rowcount);
            IF rowcount >= lastRow THEN (* = is enough but who knows what evil lurks in the heart of BIOS *)
                rowcount := initpagingcounter;
                video(kbdmsg,TRUE);
                Flushkey;
                key:=Waitkey();
                video(kbdmsg,FALSE);
                IF same(key,cancel) THEN abort(errAbortedByUser,"");END;
            END;
        END;
      END;
    END;
    IF showsum THEN doTrailer(sum,totalfiles,showcount,showpercent,showgraph,ascii,flagRecurse,scanmode); END;
END doByDir;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE isLessYear (i,j:CARDINAL   ):BOOLEAN ;
VAR
    pi,pj:ptrLogEntry;
BEGIN
    pi:=getlogentry(i);
    pj:=getlogentry(j);
    RETURN ( pi^.year < pj^.year );
END isLessYear;

PROCEDURE isLessYearReversed (i,j:CARDINAL   ):BOOLEAN ;
VAR
    pi,pj:ptrLogEntry;
BEGIN
    pi:=getlogentry(i);
    pj:=getlogentry(j);
    RETURN ( pi^.year > pj^.year );
END isLessYearReversed;

PROCEDURE isLessYearSize (i,j:CARDINAL   ):BOOLEAN ;
VAR
    pi,pj:ptrLogEntry;
BEGIN
    pi:=getlogentry(i);
    pj:=getlogentry(j);
    RETURN ( pi^.totalforyear < pj^.totalforyear );
END isLessYearSize;

PROCEDURE isLessYearSizeReversed (i,j:CARDINAL   ):BOOLEAN ;
VAR
    pi,pj:ptrLogEntry;
BEGIN
    pi:=getlogentry(i);
    pj:=getlogentry(j);
    RETURN ( pi^.totalforyear > pj^.totalforyear );
END isLessYearSizeReversed;

PROCEDURE doSwapYearSize (i,j:CARDINAL);
VAR
    pi,pj:ptrLogEntry;
    tmp : CARDINAL;
BEGIN
    pi:=getlogentry(i);
    pj:=getlogentry(j);
    tmp       := pi^.rank;
    pi^.rank  := pj^.rank;
    pj^.rank  := tmp;
END doSwapYearSize;

PROCEDURE doByYear (filemask:ARRAY OF CHAR;anchor:ptrLogEntry;
                    count,lastpath:CARDINAL;
                    reversed,showcount,showpercent,showgraph,
                    ascii,semigraphics,showsum,paging,flagRecurse,justyear:BOOLEAN;
                    scanmode:scantype);
VAR
    i,wi,wic,wip,wib : CARDINAL;
    H1,H2,R : str128;
    S,S1,S2,S3,SC : str80;
    yearsize,sum : HUGECARD;
    yyyy : LONGCARD;
    percentused:LONGREAL;
    ptr : ptrLogEntry;
    filesperyear,totalfiles : CARDINAL;
    charUsed,charFree:CHAR;
VAR
    lastRow               : CARDINAL;
    rowcount              : CARDINAL;
    key                   : str2;
BEGIN
    IF semigraphics THEN
        charUsed:=chUsed;     charFree:=chFree;
    ELSE
        charUsed:=chUsedTxt;  charFree:=chFreeTxt;
    END;

    globalAnchor := anchor;
    video(msgsort,TRUE);
    IF justyear THEN
        IF reversed THEN
            Lib.QSort(count,isLessYearReversed,doSwapYearSize);
        ELSE
            Lib.QSort(count,isLessYear,doSwapYearSize);
        END;
    ELSE
        IF reversed THEN
            Lib.QSort(count,isLessYearSizeReversed,doSwapYearSize);
        ELSE
            Lib.QSort(count,isLessYearSize,doSwapYearSize);
        END;
    END;
    video(msgsort,FALSE);

    sum:=HUGEZERO;
    FOR i:=1 TO count DO
        ptr:=getlogentry(i);
        HUGEINC(sum,ptr^.totalforyear);
    END;

    doHeader (modeYear,filemask,
             showcount,showpercent,showgraph,ascii,semigraphics,flagRecurse,
             scanmode);
    getWidths(wi,wic,wip,wib);

    IF paging THEN
        lastRow  := getWindowHeight();
        rowcount := initpagingcounter;
    END;

    totalfiles := 0;

    FOR i:=1 TO count DO
        ptr:=getlogentry(i);
        yyyy := BIOSbaseYear+LONGCARD ( ptr^.year);
        S:=fmtlc(yyyy,blank,"",4);
        yearsize := ptr^.totalforyear;
        filesperyear := ptr^.countforyear;
        INC(totalfiles,filesperyear);
        percentused := (LONGREAL(yearsize) / LONGREAL(sum) ) * 100.0;
        S1 := fmthc (yearsize, blank, dot, wi);
        SC := fmtlc ( LONGCARD(filesperyear), blank,dot,wic);
        S2 := fmtpercent(percentused,blank,coma,wip);
        S3 := fmtbar(percentused,charUsed,charFree,wib);
        Str.Copy(R,S1);
        SC := fmtlc ( LONGCARD(filesperyear), blank, dot, wic);
        IF showcount THEN Str.Append(R,sepa);Str.Append(R,SC);END;
        IF showpercent THEN Str.Append(R,sepa);Str.Append(R,S2);END;
        IF showgraph THEN Str.Append(R,sepa);Str.Append(R,S3);END;
        Str.Append(R,sepa);Str.Append(R,S);
        print(R);newline;

        IF paging THEN
            INC(rowcount);
            IF rowcount >= lastRow THEN (* = is enough but who knows what evil lurks in the heart of BIOS *)
                rowcount := initpagingcounter;
                video(kbdmsg,TRUE);
                Flushkey;
                key:=Waitkey();
                video(kbdmsg,FALSE);
                IF same(key,cancel) THEN abort(errAbortedByUser,"");END;
            END;
        END;

    END;
    IF showsum THEN doTrailer(sum,totalfiles,showcount,showpercent,showgraph,ascii,flagRecurse,scanmode); END;
END doByYear;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE isLessExt (i,j:CARDINAL   ):BOOLEAN ;
VAR
    pi,pj:ptrLogEntry;
    ei,ej:str3;
BEGIN
    pi:=getlogentry(i);
    pj:=getlogentry(j);
    ei:=pi^.extension;
    ej:=pj^.extension;
    (*
    Str.Caps(ei);
    Str.Caps(ej);
    *)
    RETURN ( Str.Compare (ei,ej) < 0 );
END isLessExt;

PROCEDURE isLessExtReversed (i,j:CARDINAL   ):BOOLEAN ;
VAR
    pi,pj:ptrLogEntry;
    ei,ej:str3;
BEGIN
    pi:=getlogentry(i);
    pj:=getlogentry(j);
    ei:=pi^.extension;
    ej:=pj^.extension;
    (*
    Str.Caps(ei);
    Str.Caps(ej);
    *)
    RETURN ( Str.Compare (ei,ej) > 0 );
END isLessExtReversed;

PROCEDURE isLessExtSize (i,j:CARDINAL   ):BOOLEAN ;
VAR
    pi,pj:ptrLogEntry;
BEGIN
    pi:=getlogentry(i);
    pj:=getlogentry(j);
    RETURN ( pi^.totalforext < pj^.totalforext );
END isLessExtSize;

PROCEDURE isLessExtSizeReversed (i,j:CARDINAL   ):BOOLEAN ;
VAR
    pi,pj:ptrLogEntry;
BEGIN
    pi:=getlogentry(i);
    pj:=getlogentry(j);
    RETURN ( pi^.totalforext > pj^.totalforext );
END isLessExtSizeReversed;

PROCEDURE doSwapExtSize (i,j:CARDINAL);
VAR
    pi,pj:ptrLogEntry;
    tmp : CARDINAL;
BEGIN
    pi:=getlogentry(i);
    pj:=getlogentry(j);
    tmp       := pi^.rank;
    pi^.rank  := pj^.rank;
    pj^.rank  := tmp;
END doSwapExtSize;

PROCEDURE doByExt  (filemask:ARRAY OF CHAR;anchor:ptrLogEntry;
                    count,lastpath:CARDINAL;
                    reversed,showcount,showpercent,showgraph,ascii,
                    semigraphics,uppercase,showsum,paging,flagRecurse,justext:BOOLEAN;
                    scanmode:scantype);
VAR
    i,wi,wic,wip,wib : CARDINAL;
    H1,H2,R : str128;
    S1,S2,S3,SC : str80;
    extsize,sum : HUGECARD;
    percentused:LONGREAL;
    ptr : ptrLogEntry;
    e : str16;
    filesperext,totalfiles:CARDINAL;
    charUsed,charFree:CHAR;
VAR
    lastRow               : CARDINAL;
    rowcount              : CARDINAL;
    key                   : str2;
BEGIN
    IF semigraphics THEN
        charUsed:=chUsed;     charFree:=chFree;
    ELSE
        charUsed:=chUsedTxt;  charFree:=chFreeTxt;
    END;

    globalAnchor := anchor;
    video(msgsort,TRUE);
    IF justext THEN
        IF reversed THEN
            Lib.QSort(count,isLessExtReversed,doSwapExtSize);
        ELSE
            Lib.QSort(count,isLessExt,doSwapExtSize);
        END;
    ELSE
        IF reversed THEN
            Lib.QSort(count,isLessExtSizeReversed,doSwapExtSize);
        ELSE
            Lib.QSort(count,isLessExtSize,doSwapExtSize);
        END;
    END;
    video(msgsort,FALSE);

    sum:=HUGEZERO;
    FOR i:=1 TO count DO
        ptr:=getlogentry(i);
        HUGEINC(sum,ptr^.totalforext);
    END;

    doHeader (modeExt,filemask,
             showcount,showpercent,showgraph,ascii,semigraphics,flagRecurse,
             scanmode);
    getWidths(wi,wic,wip,wib);

    IF paging THEN
        lastRow  := getWindowHeight();
        rowcount := initpagingcounter;
    END;

    totalfiles:=0;

    FOR i:=1 TO count DO
        ptr:=getlogentry(i);
        Str.Copy(e,ptr^.extension);
        IF uppercase THEN
            UpperCase(e);
        ELSE
            LowerCase(e);
        END;
        Str.Prepend(e,dquote);Str.Append(e,dquote);

        extsize := ptr^.totalforext;
        filesperext:= ptr^.countforext;
        INC(totalfiles,filesperext);
        percentused := (LONGREAL(extsize) / LONGREAL(sum) ) * 100.0;
        S1 := fmthc (extsize, blank, dot, wi);
        SC := fmtlc ( LONGCARD(filesperext), blank,dot,wic);
        S2 := fmtpercent(percentused,blank,coma,wip);
        S3 := fmtbar(percentused,charUsed,charFree,wib);
        Str.Copy(R,S1);
        IF showcount THEN Str.Append(R,sepa);Str.Append(R,SC);END;
        IF showpercent THEN Str.Append(R,sepa);Str.Append(R,S2);END;
        IF showgraph THEN Str.Append(R,sepa);Str.Append(R,S3);END;
        Str.Append(R,sepa);Str.Append(R,e);
        print(R);newline;

        IF paging THEN
            INC(rowcount);
            IF rowcount >= lastRow THEN (* = is enough but who knows what evil lurks in the heart of BIOS *)
                rowcount := initpagingcounter;
                video(kbdmsg,TRUE);
                Flushkey;
                key:=Waitkey();
                video(kbdmsg,FALSE);
                IF same(key,cancel) THEN abort(errAbortedByUser,"");END;
            END;
        END;

    END;
    IF showsum THEN doTrailer(sum,totalfiles,showcount,showpercent,showgraph,ascii,flagRecurse,scanmode); END;
END doByExt;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE isLessInitialChar (i,j:CARDINAL   ):BOOLEAN ;
VAR
    pi,pj:ptrLogEntry;
BEGIN
    pi:=getlogentry(i);
    pj:=getlogentry(j);
    (* RETURN ( pi^.totalforinitial < pj^.totalforinitial ); *)
    RETURN ( ORD(pi^.initial) < ORD(pj^.initial) );
END isLessInitialChar;

PROCEDURE isLessInitialCharReversed (i,j:CARDINAL   ):BOOLEAN ;
VAR
    pi,pj:ptrLogEntry;
BEGIN
    pi:=getlogentry(i);
    pj:=getlogentry(j);
    (* RETURN ( pi^.totalforinitial > pj^.totalforinitial ); *)
    RETURN ( ORD(pi^.initial) > ORD(pj^.initial) );
END isLessInitialCharReversed;

PROCEDURE isLessInitialCount (i,j:CARDINAL   ):BOOLEAN ;
VAR
    pi,pj:ptrLogEntry;
BEGIN
    pi:=getlogentry(i);
    pj:=getlogentry(j);
    RETURN ( pi^.countforinitial < pj^.countforinitial );
END isLessInitialCount;

PROCEDURE isLessInitialCountReversed (i,j:CARDINAL   ):BOOLEAN ;
VAR
    pi,pj:ptrLogEntry;
BEGIN
    pi:=getlogentry(i);
    pj:=getlogentry(j);
    RETURN ( pi^.countforinitial > pj^.countforinitial );
END isLessInitialCountReversed;

PROCEDURE doSwapInitial (i,j:CARDINAL);
VAR
    pi,pj:ptrLogEntry;
    tmp : CARDINAL;
BEGIN
    pi:=getlogentry(i);
    pj:=getlogentry(j);
    tmp       := pi^.rank;
    pi^.rank  := pj^.rank;
    pj^.rank  := tmp;
END doSwapInitial;

PROCEDURE doByInitial  (filemask:ARRAY OF CHAR;anchor:ptrLogEntry;
                    count,lastpath:CARDINAL;
                    reversed,showcount,showpercent,showgraph,ascii,
                    semigraphics,uppercase,showsum,paging,flagRecurse,bychar:BOOLEAN;
                    scanmode:scantype);
VAR
    i,wi,wic,wip,wib : CARDINAL;
    H1,H2,R : str128;
    S1,S2,S3,SC : str80;
    initialsize,sum : HUGECARD;
    percentused:LONGREAL;
    ptr : ptrLogEntry;
    sc : str16;
    filesperinitial,totalfiles:CARDINAL;
    charUsed,charFree,c:CHAR;
VAR
    lastRow               : CARDINAL;
    rowcount              : CARDINAL;
    key                   : str2;
BEGIN
    IF semigraphics THEN
        charUsed:=chUsed;     charFree:=chFree;
    ELSE
        charUsed:=chUsedTxt;  charFree:=chFreeTxt;
    END;

    globalAnchor := anchor;
    video(msgsort,TRUE);
    IF reversed THEN
        IF bychar THEN
        Lib.QSort(count,isLessInitialCharReversed,doSwapInitial);
        ELSE
        Lib.QSort(count,isLessInitialCountReversed,doSwapInitial);
        END;
    ELSE
        IF bychar THEN
        Lib.QSort(count,isLessInitialChar,doSwapInitial);
        ELSE
        Lib.QSort(count,isLessInitialCount,doSwapInitial);
        END;
    END;
    video(msgsort,FALSE);

    sum:=HUGEZERO;
    FOR i:=1 TO count DO
        ptr:=getlogentry(i);
        HUGEINC(sum,ptr^.totalforinitial);
    END;

    doHeader (modeInitial,filemask,
             showcount,showpercent,showgraph,ascii,semigraphics,flagRecurse,
             scanmode);
    getWidths(wi,wic,wip,wib);

    IF paging THEN
        lastRow  := getWindowHeight();
        rowcount := initpagingcounter;
    END;

    totalfiles:=0;

    FOR i:=1 TO count DO
        ptr:=getlogentry(i);
        Str.Copy(sc,ptr^.initial);
        IF uppercase THEN
            UpperCase(sc);
        ELSE
            LowerCase(sc); (* useless *)
        END;
        Str.Prepend(sc,dquote);Str.Append(sc,dquote);

        initialsize := ptr^.totalforinitial;
        filesperinitial:= ptr^.countforinitial;
        INC(totalfiles,filesperinitial);
        percentused := (LONGREAL(initialsize) / LONGREAL(sum) ) * 100.0;
        S1 := fmthc (initialsize, blank, dot, wi);
        SC := fmtlc ( LONGCARD(filesperinitial), blank,dot,wic);
        S2 := fmtpercent(percentused,blank,coma,wip);
        S3 := fmtbar(percentused,charUsed,charFree,wib);
        Str.Copy(R,S1);
        IF showcount THEN Str.Append(R,sepa);Str.Append(R,SC);END;
        IF showpercent THEN Str.Append(R,sepa);Str.Append(R,S2);END;
        IF showgraph THEN Str.Append(R,sepa);Str.Append(R,S3);END;
        Str.Append(R,sepa);Str.Append(R,sc);
        print(R);newline;

        IF paging THEN
            INC(rowcount);
            IF rowcount >= lastRow THEN (* = is enough but who knows what evil lurks in the heart of BIOS *)
                rowcount := initpagingcounter;
                video(kbdmsg,TRUE);
                Flushkey;
                key:=Waitkey();
                video(kbdmsg,FALSE);
                IF same(key,cancel) THEN abort(errAbortedByUser,"");END;
            END;
        END;

    END;
    IF showsum THEN doTrailer(sum,totalfiles,showcount,showpercent,showgraph,ascii,flagRecurse,scanmode); END;
END doByInitial;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE sumroot (VAR lastpath : CARDINAL);
VAR
    i,j,filesperdir,count : CARDINAL;
    S,R : str128;
    dirsize,total:HUGECARD;
BEGIN
    (*
        note to the Code Police : we sum up sizes for each root directory "?:\*\"
        the Q&D way, where count of "\" is 2
    *)
    FOR i:=firstPath TO lastpath DO
        getentry(i,S);
        total := Path[i]^.totalsize;
        count := Path[i]^.filecount;
        IF CharCount(S,backslash)=2 THEN (* for each "?:\*\" *)
            FOR j:=firstPath TO lastpath DO
                IF j # i THEN            (* ignore oneself *)
                    IF Path[j]^.showme THEN
                        getentry(j,R);
                        IF CharCount(R,backslash) > 2 THEN (* ignore "?:\" *)
                            IF Str.Pos(R,S)=0 THEN (* find root *)
                                dirsize    := Path[j]^.totalsize;
                                filesperdir:= Path[j]^.filecount;
                                HUGEINC( total, dirsize);
                                INC( count, filesperdir);
                                Path[j]^.showme := FALSE;
                            END;
                        END;
                    END;
                END;
            END;
        END;
        Path[i]^.totalsize:=total;
        Path[i]^.filecount:=count;
    END;
    (* now we force disk to only have the root directories *)

END sumroot;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    letters = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";

(* assume S is already in uppercase *)

PROCEDURE isUnitSpec ( S : ARRAY OF CHAR ) : BOOLEAN;
BEGIN
    IF Str.Length(S) # 2 THEN RETURN FALSE; END; (* not xx *)
    IF S[1] # colon THEN RETURN FALSE; END;      (* not x: *)
    RETURN Str.CharPos(letters,S[0]) # MAX(CARDINAL);
END isUnitSpec;

(* assume S is already in uppercase, minimalist f8e3 check *)

PROCEDURE chkF8E3 (S:ARRAY OF CHAR):BOOLEAN;
CONST
    maxspeclen = 8+1+3;
    legalset = letters+dot+question+star;
VAR
    pb,i,len:CARDINAL;
BEGIN
    pb:=0;
    IF CharCount(S,dot) > 1 THEN INC(pb);END;
    len:=Str.Length(S);
    IF len > maxspeclen THEN INC(pb);END;
    FOR i:=1 TO len DO
        IF Str.CharPos(legalset, S[i-1])= MAX(CARDINAL) THEN INC(pb);END;
    END;
    RETURN (pb = 0);
END chkF8E3;

PROCEDURE getUnitChar (u:SHORTCARD):CHAR;
BEGIN
    RETURN CHR(u+ORD("A")-1);
END getUnitChar;

PROCEDURE getAnchor (VAR current:ARRAY OF CHAR);
VAR
    unit:SHORTCARD;
    rc:BOOLEAN;
    errcode:CARDINAL;
BEGIN
    unit:=FIO.GetDrive();
    FIO.GetDir(unit, current); (* 0=default, 1=A:, 2=B:, etc. *)
    IF Str.CharPos(current,colon)=MAX(CARDINAL) THEN (* always true, but who knows with win9X ? *)
        Str.Prepend(current,colon);
        Str.Prepend(current,getUnitChar(unit));
    END;
END getAnchor;

(* ------------------------------------------------------------ *)

CONST
    placeholder = "$";
    msgScanningDirs = "Scanning "+placeholder+" directories..."; (* $ is unit placeholder *)
    msgScanningFiles= "Scanning files in ";
TYPE
    sorttype  = SET OF (byextension,bydirectory,byyear,
                        byinitialchar,byinitialcount);
CONST
    firstUnit = 1;
    maxUnit   = 26; (* A..Z at most *)
VAR
    unit      : ARRAY [firstUnit..maxUnit] OF CHAR;
    lastUnit  : CARDINAL;
    msg,sHD,sCD,sValidUnits,sUnits    : str80;
    cdcount:CARDINAL;
    cdfirst:CHAR;
    sortcmd   : sorttype;
    DEBUG,reversed,showcount,showpercent,showgraph,ascii,semigraphics:BOOLEAN;
    onlyroot,scanOnePath,flagRecurse : BOOLEAN;
    useLFN,uppercase,showsum,paging,justyear,justext,justdir : BOOLEAN;
    scanmode : scantype;
    lastdirndx,count                   : CARDINAL;
    anchor,newInList : ptrLogEntry;
    rc                : BOOLEAN;
    filemask          : str16; (* f8e3=8+1+3=12 *)
    wimsg,wiscreen:CARDINAL;
VAR
    parmcount,i,opt : CARDINAL;
    S,R,zeOnePath   : str128;
    useCLI          : BOOLEAN;
    cli             : str256; (* false cli, oversized just in case *)
    hnd             : FIO.File;
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE;

    vidinit ( IsRedirected() );
    newline; (* here for pretty output *)
    colortext;

    parmcount := Lib.ParamCount();
    useCLI    := TRUE;

    IF parmcount = 0 THEN
        Lib.ParamStr(S,0); (* argh ! no longer okay when qd_text is used ! weird... *)
        UpperCase(S); (* useless ! *)
        Str.Subst(S,extEXE,extINI);
        IF FIO.Exists(S)=FALSE THEN
            (* abort(errNotFound,S); *)
            abort(errHelp,"");
        END;
        hnd:=FIO.OpenRead(S);
        FIO.RdStr(hnd,cli);
        FIO.Close(hnd);
        LtrimBlanks(cli);
        RtrimBlanks(cli);
        parmcount := argc(cli,FALSE); (* FALSE because already trimed *)
        useCLI := FALSE;
    END;

    DEBUG := FALSE;

    lastUnit  := firstUnit;
    Str.Copy(sUnits,"");
    sortcmd     := sorttype{};
    reversed    := FALSE;
    showcount   := TRUE;
    showpercent := FALSE;
    showgraph   := FALSE;
    ascii       := FALSE;
    semigraphics:= TRUE;
    uppercase   := FALSE;
    showsum     := TRUE;
    paging      := FALSE;
    onlyroot    := FALSE;
    scanmode    := normalandspecial;
    filemask    := stardotstar;
    scanOnePath := FALSE;
    flagRecurse := TRUE;
    justyear    := FALSE;
    justext     := FALSE;
    justdir     := FALSE;
    useLFN      := TRUE;

    FOR i := 1 TO parmcount DO
        IF useCLI THEN
            Lib.ParamStr(S,i);
        ELSE
            argv(S,cli,i,FALSE); (* FALSE because already trimed *)
        END;
        Str.Copy(R,S);
        UpperCase(R);
        cleantabs(R); (* YATB : Yet Another TopSpeed Bug ! *)
        IF isOption(R) THEN
            opt := GetOptIndex(R, "?"+delim+"H"+delim+"HELP"+delim+
                                  "DEBUG"+delim+"DEBUG"+delim+
                                  "D"+delim+"DIRECTORY"+delim+
                                  "E"+delim+"EXTENSION"+delim+
                                  "Y"+delim+"YEAR"+delim+
                                  "R"+delim+"REVERSE"+delim+
                                  "P"+delim+"PERCENTAGE"+delim+
                                  "G"+delim+"BARGRAPH"+delim+
                                  "A"+delim+"ASCII"+delim+
                                  "N"+delim+"NORMAL"+delim+
                                  "U"+delim+"UPPERCASE"+delim+
                                  "C"+delim+"COUNT"+delim+
                                  "S"+delim+"SUM"+delim+
                                  "K"+delim+"P"+delim+"PAGING"+delim+
                                  "DD"+delim+"ROOT"+delim+
                                  "F:"+delim+"FILE:"+delim+
                                  "L"+delim+"LFN"+delim+
                                  "Z"+delim+"SPECIAL"+delim+
                                  "T"+delim+"TEXT"+delim+
                                  "GG"+delim+
                                  "J"+delim+"NORECURSE"+delim+
                                  "I"+delim+"INITIAL"+delim+
                                  "II"+delim+
                                  "AA"+delim+
                                  "YY"+delim+
                                  "EE"+delim+
                                  "F"+delim+"PATH"+delim+
                                  "FF"+delim+
                                  "DF"+delim+
                                  "DDF"
(*%T CONSOLE  *)
                                                    +delim+
                                  "B"+delim+"BIOS"
(*%E  *)
                              );
            CASE opt OF
            | 1,2,3 : abort(errHelp,"");
            | 4,5   : DEBUG := TRUE;
            | 6,7   : INCL (sortcmd,bydirectory);
            | 8,9   : INCL (sortcmd,byextension);
            | 10,11 : INCL (sortcmd,byyear);
            | 12,13 : reversed :=TRUE;
            | 14,15 : showpercent  :=TRUE;
            | 16,17 : showgraph    :=TRUE;
            | 18,19 : ascii    :=TRUE;
            | 20,21 : CASE scanmode OF
                      | normalandspecial,normal:
                          scanmode:=normal;
                      ELSE
                          abort(errAttr,"");
                      END;
            | 22,23 : uppercase:=TRUE;
            | 24,25 : showcount:=FALSE;
            | 26,27 : showsum  :=FALSE;
            | 28,29,30:paging  :=TRUE;
            | 31,32 : INCL (sortcmd,bydirectory);onlyroot :=TRUE;
            | 33,34 : GetString(R,filemask);
                      IF chkF8E3(filemask)=FALSE THEN abort(errBadFileMask,R);END;
            | 35,36 : useLFN:=FALSE;
            | 37,38 : CASE scanmode OF
                      | normalandspecial,special:
                          scanmode:=special;
                      ELSE
                          abort(errAttr,"");
                      END;
            | 39,40: semigraphics := FALSE;
            | 41:    showgraph :=TRUE; semigraphics:=FALSE;
            | 42,43: flagRecurse:=FALSE;
            | 44,45: INCL (sortcmd,byinitialchar);
            | 46:    INCL (sortcmd,byinitialcount);
            | 47:    ascii :=TRUE; semigraphics:=FALSE;
            | 48:    INCL (sortcmd,byyear);       justyear:=TRUE;
            | 49:    INCL (sortcmd,byextension);  justext :=TRUE;
            | 50,51: INCL (sortcmd,bydirectory);  justdir :=TRUE; (* -f forces -d *)
            | 52:    INCL (sortcmd,bydirectory);  justdir :=TRUE; onlyroot:=TRUE; (* -ff forces -dd *)
            | 53:    INCL (sortcmd,bydirectory);  justdir :=TRUE; (* -df *)
            | 54:    INCL (sortcmd,bydirectory);  justdir :=TRUE; onlyroot :=TRUE; (* -ddf *)
(*%T CONSOLE  *)
            | 55,56: vidinit( TRUE);
(*%E  *)
            ELSE
                abort(errOption,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            IF scanOnePath THEN
                abort(errOnePath,"");
            ELSE
                IF isUnitSpec(R) THEN (* "u:" *)
                    IF lastUnit > maxUnit THEN abort(errTooManyUnits,S); END; (* extremely unlikely and even impossible ! *)
                    IF Str.CharPos(sUnits,R[0]) # MAX(CARDINAL) THEN
                        abort(errAlreadySpecified,S); (* we could just ignore it, but... *)
                    END;
                    unit[lastUnit] := R[0]; (* keep uppercase *)
                    INC(lastUnit);
                    Str.Append(sUnits,R[0]);
                ELSE
                    CASE Str.CharPos(R,colon) OF
                    | 1,MAX(CARDINAL):
                        ;
                    ELSE
                        abort(errBadPath,R);
                    END;
                    IF same(R,stardotstar) THEN R:=dot; END;
                    IF same(R,dot) THEN getAnchor(R);END;
                    IF chkJoker(R) THEN abort(errJoker,R);END;
                    IF same(R,backslash)=FALSE THEN
                        IF isDirectory(R)=FALSE THEN abort(errFile,R);END;
                    END;
                    fixDirectory(R);
                    Str.Copy(zeOnePath,R);
                    scanOnePath:=TRUE;
                    (* abort(errBadUnitSpec,S); *)
                END;
            END;
        END;
    END;

    IF scanOnePath=FALSE THEN
        IF lastUnit = firstUnit THEN abort(errNoUnitSpecified,""); END;
        DEC (lastUnit);
    END;

    IF sortcmd=sorttype{} THEN
        INCL(sortcmd,bydirectory);
    ELSIF sortcmd=sorttype{byextension} THEN
        ;
    ELSIF sortcmd=sorttype{bydirectory} THEN
        ;
    ELSIF sortcmd=sorttype{byyear}      THEN
        ;
    ELSIF sortcmd=sorttype{byinitialchar}   THEN
        ;
    ELSIF sortcmd=sorttype{byinitialcount}  THEN
        ;
    ELSE
        abort(errSortConflict,"");
    END;

    useLFN := ( useLFN AND w9XsupportLFN() );

    (* print(Banner);newline;
    newline;
    *)

    IF scanOnePath THEN
        Str.Copy(S,zeOnePath);
        lastdirndx:=firstPath;
        Str.Copy(msg,msgScanningDirs);
        Str.Subst(msg,placeholder, S);
        video(msg,TRUE);
        (* recurse, sort *)
        IF buildPathList (S, flagRecurse, TRUE, lastdirndx)=FALSE THEN
            video(msg,FALSE);
            abort(errTooManyDirs,S);
        END;
        video(msg,FALSE);
    ELSE
        getCDROMletters(cdcount,cdfirst,sCD);
        getAllHDunits(sHD);
        Str.Concat(sValidUnits,"AB",sHD);Str.Append(sValidUnits,sCD);

        FOR i := firstUnit TO lastUnit DO
            IF Str.CharPos( sValidUnits, unit[i] ) = MAX(CARDINAL) THEN
                abort(errPhantomUnit,unit[i]);
            END;
        END;

        lastdirndx:=firstPath;
        FOR i := firstUnit TO lastUnit DO
            Str.Copy(msg,msgScanningDirs);
            Str.Subst(msg,placeholder, unit[i]);
            video(msg,TRUE);
            Str.Concat(S,unit[i],colon);
            Str.Append(S,backslash);
            (* recurse, sort *)
            IF buildPathList (S, flagRecurse, TRUE, lastdirndx)=FALSE THEN
                video(msg,FALSE);
                abort(errTooManyDirs,S);
            END;
            video(msg,FALSE);
        END;
    END;

    IF DEBUG THEN
        FOR i:=firstPath TO lastdirndx DO
            getentry(i,S);
            print(S);newline;
        END;
        newline;
    END;

    wiscreen:=getColumns();
    wimsg:=Str.Length(msgScanningFiles);

    video(msgScanningFiles,TRUE);

    anchor := NIL; (* required default *)

    FOR i:=firstPath TO lastdirndx DO
        getentry(i,S);
        shortenVid(R, wiscreen,wimsg,S);
        video(R,TRUE );
        IF sortcmd=sorttype{byextension} THEN
            rc:=buildDataExt    (scanmode,S,filemask,i,anchor,newInList,count);
        ELSIF sortcmd=sorttype{bydirectory} THEN
            rc:=buildDataDir    (scanmode,S,filemask,i);
        ELSIF sortcmd=sorttype{byyear}     THEN
            rc:=buildDataYear   (scanmode,S,filemask,i,anchor,newInList,count);
        ELSIF sortcmd=sorttype{byinitialchar}  THEN
            rc:=buildDataInitial(scanmode,S,filemask,i,anchor,newInList,count);
        ELSIF sortcmd=sorttype{byinitialcount}  THEN
            rc:=buildDataInitial(scanmode,S,filemask,i,anchor,newInList,count);
        END;
        IF rc=FALSE THEN
             video(S,FALSE);
             video(msgScanningFiles,FALSE);
             abort(errAllocate,S);
        END;
        video(R,FALSE);
    END;
    video(msgScanningFiles,FALSE);

    IF paging THEN
        IF IsRedirected() THEN paging:=FALSE;END;
    END;

    IF sortcmd=sorttype{byextension}    THEN
        doByExt (filemask,anchor,count,lastdirndx,
                 reversed,showcount,showpercent,showgraph,ascii,semigraphics,
                 uppercase,showsum,paging,flagRecurse,justext,scanmode);
    ELSIF sortcmd=sorttype{bydirectory} THEN
        IF onlyroot THEN sumroot(lastdirndx);END;
        doByDir (filemask,lastdirndx,
                 reversed,showcount,showpercent,showgraph,ascii,semigraphics,useLFN,
                 uppercase,showsum,paging,flagRecurse,justdir,scanmode);
    ELSIF sortcmd=sorttype{byyear}      THEN
        doByYear (filemask,anchor,count,lastdirndx,
                  reversed,showcount,showpercent,showgraph,ascii,semigraphics,
                  showsum,paging,flagRecurse,justyear,scanmode);
    ELSIF sortcmd=sorttype{byinitialchar}   THEN
        doByInitial (filemask,anchor,count,lastdirndx,
                 reversed,showcount,showpercent,showgraph,ascii,semigraphics,
                 uppercase,showsum,paging,flagRecurse,TRUE,scanmode);
    ELSIF sortcmd=sorttype{byinitialcount}   THEN
        doByInitial (filemask,anchor,count,lastdirndx,
                 reversed,showcount,showpercent,showgraph,ascii,semigraphics,
                 uppercase,showsum,paging,flagRecurse,FALSE,scanmode);
    END;

    freeData(anchor); (* no problemo for bydirectory because anchor will be NIL *)
    freePaths(lastdirndx);

    abort(errNone,"");
END fu.






(*

@echo off
if "%1" == "" goto syntax
if "%2" == "" goto syntax
if "%3" == "" goto syntax
if "%4" == "" goto ok
:syntax
echo.
echo Syntax : %0 exe dir report
echo.
echo Example : %0 c:\bat\tools\fu  c:\mesdoc~1  g:\z\test.rpt
echo.
echo           (nul: to dump to screen)
echo.
goto end
:ok
echo. > %3

%1 %2 /c /p /g          /f:*.arj   >> %3
%1 %2 /c    /g          /f:*.arj    >> %3
%1 %2 /c /p             /f:*.arj    >> %3

%1 %2 /c /p /g /s       /f:*.arj    >> %3
%1 %2 /c    /g /s       /f:*.arj    >> %3
%1 %2 /c /p    /s       /f:*.arj    >> %3

%1 %2 /c /p /g     /dd  /f:*.arj   >> %3
%1 %2 /c    /g     /dd  /f:*.arj   >> %3
%1 %2 /c /p        /dd  /f:*.arj   >> %3

%1 %2 /c /p /g /s  /dd  /f:*.arj   >> %3
%1 %2 /c    /g /s  /dd  /f:*.arj   >> %3
%1 %2 /c /p    /s  /dd  /f:*.arj   >> %3


%1 %2 /c /p /g     /e   /f:*.arj   >> %3
%1 %2 /c    /g     /e   /f:*.arj   >> %3
%1 %2 /c /p        /e   /f:*.arj   >> %3

%1 %2 /c /p /g /s  /e   /f:*.arj   >> %3
%1 %2 /c    /g /s  /e   /f:*.arj   >> %3
%1 %2 /c /p    /s  /e   /f:*.arj   >> %3

%1 %2 /c /p /g     /y   /f:*.arj   >> %3
%1 %2 /c    /g     /y   /f:*.arj   >> %3
%1 %2 /c /p        /y   /f:*.arj   >> %3

%1 %2 /c /p /g /s  /y   /f:*.arj   >> %3
%1 %2 /c    /g /s  /y   /f:*.arj   >> %3
%1 %2 /c /p    /s  /y   /f:*.arj   >> %3

:end

*)

