(* ---------------------------------------------------------------
Title         Q&D Date/Time Check
Overview      see help
Usage         see help
Notes         very, very, very quick & dirty... :-(
              minimal error messages and checking, etc.

              user is responsible for consistency :
              if reference file other than exe is used,
              it must always be used ! (are we clear enough ?)

Bugs          yes, in TopSpeed runtime : with Lib.SetDate and Lib.SetTime,
              ignore return code : they check if AX=0 when it should be AL !
Wish List     an interactive way to change date and time
              (as in the old XT days with ddate or edate)

--------------------------------------------------------------- *)

MODULE DTchk;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;
IMPORT SYSTEM;
IMPORT IO;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, cleantabs;

FROM QD_Text IMPORT
colortype, cursorshapetype, scrolltype,
ff, cr, lf, bs, tab, nl, mincolor, maxcolor,
BW40, CO40, BW80, CO80, CO80x43, CO80x50, MONO,
vesa80x60, vesa132x25, vesa132x43, vesa132x50, vesa132x60,
selectCursorEmulation,
setCursorShape,
handleVesa, setBrightPaper, setBlinkMode,
setFillChar, setFillInk, setFillPaper,
setFillInkPaper, setTxtInk, setTxtPaper, setTxtInkPaper, setWrapMode,
setUseBiosMode, setTabWidth, getScreenData, setWindow,
setMode, restoreMode,
gotoXY, xyToHtabVtab, home, setVisualPage, setActivePage,
scrollWindow, fillTextAttr, cls, writeStr, writeLn, getScreenWidth,
getScreenHeight, getScreenMinX, getScreenMinY, getScreenMaxX, getScreenMaxY,
getMinHtab, getMaxHtab, getMinVtab, getMaxVtab, getUseBiosMode,
getMinHtab, getMaxHtab, getMinVtab, getMaxVtab, getHtab, getVtab,
getWindowWidth, getWindowHeight,
initScreenConsole,
findInkPaperAtStartup, getInkAtStartup, getPaperAtStartup,
setFullScreenWindow, gotoFullScreenXY;

(* ------------------------------------------------------------ *)

CONST
    (* aliases for our own procedures *)
    WrStr ::= writeStr;
    WrLn  ::= writeLn;

CONST
    defaultink          = white;
    defaultpaper        = black;
    defaultinkwarning   = red;
    defaultpaperwarning = white;
    defaultinkok        = cyan;
    defaultpaperok      = darkblue;

PROCEDURE colorhelp (  );
BEGIN
    setTxtInk(getInkAtStartup());      (* was green *)
    setTxtPaper(getPaperAtStartup());  (* was black *)
    setFillInkPaper(getInkAtStartup(),getPaperAtStartup());
END colorhelp;

PROCEDURE color (ink,paper:CARDINAL );
BEGIN
    setTxtInk   (VAL(colortype,ink));
    setTxtPaper (VAL(colortype,paper));
    setFillInkPaper(getInkAtStartup(),getPaperAtStartup());
END color;

(* ------------------------------------------------------------ *)

CONST
    ProgEXEname     = "DTCHK";
    ProgTitle       = "Q&D Date/Time Check";
    ProgVersion     = "v1.0j";
    ProgCopyright   = "by PhG";
    Banner          = ProgTitle+" "+ProgVersion+" "+ProgCopyright;
CONST
    extEXE             = ".EXE";
    extINI             = ".INI";
    defaultCompDate    = "01-Jan-1980"; (* beginning of messdos epoch *)
    defaultCompTime    = "00:30:00";    (* allow a very generous 30 minutes booting time ! *)
    defaultCompTimeZero= "00:00:00";
    defaultFormat      = "$0d-$0m-$yyyy, %0h:%0m:%0s";
    defaultFormatDATE  = "$mm $d, $yyyy "; (* do not use $dd here, and force cursor pos *)
    defaultFormatTIME  = "%0h:%0m ";
CONST
    century         = 1900;
    baseyear        = 1980;
    sDefaultCentury = "1900";
CONST
    dot             = ".";
    antislash       = "\";
    dash            = "-";
    slash           = "/";
    colon           = ":";
    alphabet        = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    digits          = "0123456789";
    dotdot          = dot+dot;
    netslash        = antislash+antislash;
CONST
    errNone         = 0;
    errHelp         = 1;
    errOption       = 2;
    errTooManyParms = 3;
    errDOSversion   = 4;
    errJoker        = 5;
    errBadName      = 6;
    errNoMatch      = 7;
    errWriteProtected=8;
    errBadDate       =9;
    errBadTime       =10;
    errDateNeeded    =11;
    errOpening       =12;
    errGetting       =13;
    errClockBeforeOrigin =14;
    errClockBeforeStored =15;
    errSettingDate   =16;
    errSettingTime   =17;
    errBadNumber     =18;
    errPauseRange    =19;
    errRedirected    =20;
    errConflict      =21;
    errRange         =22;

    errNoProblemo    =128;
    errProblem       =255;

(* ------------------------------------------------------------ *)

PROCEDURE sound (freq,duration,pause:CARDINAL);
BEGIN
    Lib.Sound(freq);
    Lib.Delay(duration);
    Lib.NoSound();
    Lib.Delay(pause);
END sound;

PROCEDURE alarm (  );
BEGIN
    sound(55,55,100);
    sound(55,55,10);
END alarm;

PROCEDURE alert (  );
BEGIN
    sound(550,55,100);
    sound(550,55,10);
END alert;

VAR
    AUDIO : BOOLEAN; (* audio warn *)

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)

helpmsg =
Banner+nl+
nl+
"Syntax : "+ProgEXEname+" [reference_file] [option]..."+nl+
nl+
"-c    do not update system clock with reference date/time stamp"+nl+
"-s    do not update reference date/time stamp with current date/time"+nl+
"-a    do not ask confirmation before system clock update"+nl+
"-i    interactive d/t change if current d/t is earlier than reference d/t"+nl+
"-f[f] force interactive d/t change (-ff : use system d/t as base d/t)"+nl+
"-w    audio warning"+nl+
"-t    terse"+nl+
"-r    ignore file read-only status"+nl+
"-d:$  origin dd-mm-yy[yy] system clock date (default century is "+sDefaultCentury+")"+nl+
"-t:$  origin hh:mm[:ss] system clock time"+nl+
'-f:$  format ("$[0]<d|m>", "$dd", "$mm", "$yy", "$yyyy", "%[0]<h|m|s>")'+nl+
"-o    older display"+nl+
"-i:#  ink [0..15] -- default is white"+nl+
"-p:#  paper [0..15] -- default is black"+nl+
"-iw:# warning ink [0..15] -- default is red"+nl+
"-pw:# warning paper [0..15] -- default is white"+nl+
"-ii:# ok ink [0..15] -- default is cyan"+nl+
"-pp:# ok paper [0..15] -- default is dark blue"+nl+
"-b    monochrome BIOS output (no colors)"+nl+
nl+
"Default origin system clock date and time is "+defaultCompDate+", "+defaultCompTime+"."+nl+
'Default format string is "'+defaultFormat+'".'+nl+
"If -d:$ is specified and -t:$ is not, -t:"+defaultCompTimeZero+" is assumed."+nl+
"If -t:$ is specified, -d:$ must be specified too."+nl+
"-i and -f options enable -r option, while ignoring -c, -s and -a options."+nl+
"If not specified, reference file is searched for in executable directory"+nl+
"first as "+ProgEXEname+extINI+" then as "+ProgEXEname+extEXE+" itself ;"+nl+
"if specified, file must exist ; if specified in 8+3 format (without a path),"+nl+
"it is searched for first in executable directory, then in current directory."+nl+
"Program returns code 128 if everything went fine, else 255."+nl+
nl+
"Dark [0..7] : black, blue, green, cyan, red, magenta, brown and gray."+nl+
"Bright [8..15] : gray, blue, green, cyan, red, magenta, yellow and white."+nl+
nl+
"With DR-DOS 6.0 or Novell DOS 7.0, "+ProgEXEname+" can be called after each"+nl+
"external command by setting SET PEXEC=path\"+ProgEXEname+extEXE+" -w and PROMPT $x$p$g"+nl;

VAR
    S : str256;
BEGIN
    colorhelp;
    CASE e OF
    | errHelp :
        WrStr(helpmsg);
    | errOption:
        Str.Concat(S,"Unknown ",einfo);Str.Append(S," option !");
    | errTooManyParms:
        Str.Concat(S,einfo," parameter is one too many !");
    | errDOSversion:
        Str.Concat(S,"DOS ",einfo);Str.Append(S," or later is required !");
    | errJoker:
        Str.Concat(S,"No joker allowed in ",einfo);Str.Append(S," reference file !");
    | errBadName :
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," name !");
    | errNoMatch :
        Str.Concat(S,einfo," does not exist !");
    | errWriteProtected:
        Str.Concat(S,einfo," is read-only !");
    | errBadDate:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," date format !");
    | errBadTime:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," time format !");
    | errDateNeeded:
        S := "-t:$ option requires -d:$ option !";
    | errOpening :
        Str.Concat(S,"Problem while opening ",einfo);Str.Append(S," !");
    | errGetting :
        Str.Concat(S,"Problem while retrieving date/time stamp from ",einfo);
        Str.Append(S," !");
    | errClockBeforeOrigin:
        S := "System clock is earlier than origin date and time !";
    | errClockBeforeStored:
        S := "System clock is earlier than stored date and time !";
    | errSettingDate:
        S := "Problem when setting system date to saved date !";
    | errSettingTime:
        S := "Problem when setting system time to saved time !";
    | errBadNumber:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," number !");
    | errPauseRange:
        S:="Pause range is [1..15000] !";
    | errRedirected:
        S:="Redirection is a nonsense here !";
    | errConflict:
        S:="Specifying -i and -f options together is a nonsense !";
    | errRange:
        Str.Concat(S,einfo," range is [0..15] !");
    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp :
        ;
    ELSE
        WrStr(ProgEXEname+" : ");WrStr(S);WrLn;
    END;
    CASE e OF
    | errNone:
        e := errNoProblemo;
    | errClockBeforeOrigin, errClockBeforeStored:
        IF AUDIO THEN alert();END;
        e := errProblem;
    | errSettingDate, errSettingTime : (* audio already handled in calling code *)
        e := errProblem;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE AtLeastDosVersion (minmajor,minminor:CARDINAL) : BOOLEAN;
VAR
    R             : SYSTEM.Registers;
    minDosVersion : CARDINAL;
    major         : CARDINAL;
    minor         : CARDINAL;
    thisDosVersion: CARDINAL;
BEGIN
    minDosVersion := (minmajor << 8) + minminor;
    R.AX := 3000H;
    Lib.Dos(R);
    major := CARDINAL(R.AL);
    minor := CARDINAL(R.AH);
    thisDosVersion := (major << 8) + minor;
    IF thisDosVersion < minDosVersion THEN RETURN FALSE; END;
    RETURN TRUE;
END AtLeastDosVersion;

PROCEDURE chkValidName (S:ARRAY OF CHAR ) : BOOLEAN;
VAR
    i : CARDINAL;
BEGIN
    IF same(S,dot) THEN RETURN FALSE;END;
    IF Str.Pos(S,dotdot) # MAX(CARDINAL) THEN RETURN FALSE; END;
    IF Str.Pos(S,netslash) # MAX(CARDINAL) THEN RETURN FALSE; END;
    (* yes, there are lost of other things to check... but who cares ? *)
    RETURN TRUE;
END chkValidName;

PROCEDURE procReference (VAR R:ARRAY OF CHAR ):BOOLEAN;
VAR
    u,d,n,e,exe,f8,e3:str128; (* oversized because of user possible creativity *)
BEGIN
    Lib.ParamStr(exe,0);
    Str.Caps(exe); (* useless safety *)
    IF same(R,"") THEN
        Str.Copy(R,exe);
        Str.Subst(R,extEXE,extINI);
        IF FIO.Exists(R)=FALSE THEN Str.Copy(R,exe);END;
        (* ini may not exist but exe always will ;-) *)
    ELSE
        Str.Caps(R);
        Lib.SplitAllPath(R, u,d,n,e);
        IF ( same(u,"") AND same (d,"") ) THEN (* looks like a f8e3 *)
            Lib.SplitAllPath(exe, u,d,f8,e3);
            Lib.MakeAllPath (R, u,d,n,e);
            IF FIO.Exists(R)=FALSE THEN Lib.MakeAllPath(R, "","",n,e);END;
        END;
    END;
    RETURN FIO.Exists(R);
END procReference;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE SetDate(Year,Month,Day:CARDINAL):BOOLEAN;
VAR
    R : SYSTEM.Registers;
BEGIN
    R.AX := 2B00H;
    R.CX := Year;
    R.DH := SHORTCARD(Month);
    R.DL := SHORTCARD(Day);
(*%T _WINDOWS *)
    R.DS := Seg(R);
    R.ES := Seg(R);
(*%E *)
    Lib.Dos(R);
    (* silly bug in Lib tests AX !!! *)
    IF R.AL=0 THEN
        RETURN TRUE;
    ELSE
        RETURN FALSE;
    END;
END SetDate;

PROCEDURE SetTime(Hrs,Mins,Secs,Hsecs:CARDINAL):BOOLEAN;
VAR
   R : SYSTEM.Registers;
BEGIN
   R.AH := 2DH;
   R.CH := SHORTCARD(Hrs);
   R.CL := SHORTCARD(Mins);
   R.DH := SHORTCARD(Secs);
   R.DL := SHORTCARD(Hsecs);
   Lib.Dos(R);
   (* silly bug in Lib tests AX !!! *)
   IF R.AL=0 THEN
       RETURN TRUE;
   ELSE
       RETURN FALSE;
   END;
END SetTime;

PROCEDURE FixSeconds (VAR t : FIO.FileStamp );
BEGIN
    t.Sec:=t.Sec*2; (* this damn GetFileStamp lets seconds DIV 2 = 0..31 ! *)
    IF t.Sec > 59 THEN (* in case it's a weird seconds field : 61 seconds ! *)
        t.Sec := 59;
    END;
END FixSeconds;

TYPE
    datetype = RECORD
        day   : CARDINAL;
        month : CARDINAL;
        year  : CARDINAL; (* 1980 is already added *)
        dayOfWeek : Lib.DayType;
    END;
    timetype = RECORD
        hours   : CARDINAL;
        minutes : CARDINAL;
        seconds : CARDINAL;
    END;
    dttype = RECORD
        CASE : BOOLEAN OF
        | TRUE  :
            t : CARDINAL; (* hms is low  *)
            d : CARDINAL; (* ymd is high *)
        | FALSE :
            dt : LONGCARD;
        END;
    END;

PROCEDURE getDateNow (VAR d : datetype);
VAR
    dayOfWeek : Lib.DayType;
BEGIN
    Lib.GetDate(d.year,d.month,d.day,dayOfWeek);
END getDateNow;

PROCEDURE getTimeNow (VAR t : timetype);
VAR
    hundredth : CARDINAL;
BEGIN
    Lib.GetTime(t.hours,t.minutes,t.seconds,hundredth);
END getTimeNow;

(* yes, we could check if date has not changed after GetTime... Bah ! *)

PROCEDURE GetDateTimeNow (VAR d : datetype;VAR t : timetype);
VAR
    hundredth : CARDINAL;
BEGIN
    Lib.GetDate(d.year,d.month,d.day,d.dayOfWeek);
    (* we do not care and do not round hundredth of seconds *)
    Lib.GetTime(t.hours,t.minutes,t.seconds,hundredth);
END GetDateTimeNow;

PROCEDURE parseDate (S : ARRAY OF CHAR;
                     VAR date : datetype) : BOOLEAN;
CONST
    separator=dash;
    legaldateset = digits+separator+alphabet;
    mindd=1;
    maxdd=31;
    minmm=1;
    maxmm=12;
    minyy=1980; (* base year for messdos *)
    maxyy=2099;
VAR
    i : CARDINAL;
    R : str80;
    v : LONGCARD;
    ok: BOOLEAN;
BEGIN
    UpperCase(S); (* in case months would be letters *)
    ReplaceChar(S,slash,separator);
    FOR i := 0 TO (Str.Length(S)-1) DO
        IF Str.CharPos(legaldateset,S[i])=MAX(CARDINAL) THEN RETURN FALSE; END;
    END;
    IF CharCount(S,separator) # 2 THEN RETURN FALSE; END;

    Str.ItemS(R,S,separator,0);
    v := Str.StrToCard(R,10,ok);
    IF ok=FALSE THEN RETURN FALSE; END;
    IF (v < mindd) OR (v > maxdd) THEN RETURN FALSE; END;
    date.day := CARDINAL(v);

    Str.ItemS(R,S,separator,1);
    v := Str.StrToCard(R,10,ok);
    IF ok=FALSE THEN
        Str.Prepend(R,dash); (* fake command line parameter ! *)
        i := GetOptIndex(R,"JAN"+delim+"JAN"+delim+
                           "FEB"+delim+"FEV"+delim+
                           "MAR"+delim+"MAR"+delim+
                           "APR"+delim+"AVR"+delim+
                           "MAY"+delim+"MAI"+delim+
                           "JUN"+delim+"JUN"+delim+
                           "JUL"+delim+"JUI"+delim+
                           "AUG"+delim+"AOU"+delim+
                           "SEP"+delim+"SEP"+delim+
                           "OCT"+delim+"OCT"+delim+
                           "NOV"+delim+"NOV"+delim+
                           "DEC"+delim+"DEC");
        CASE i OF
        | 1..24 :
            v := LONGCARD(i+1) DIV 2;
        ELSE
            RETURN FALSE;
        END;
    END;
    IF (v < minmm) OR (v > maxmm) THEN RETURN FALSE; END;
    date.month := CARDINAL(v);

    Str.ItemS(R,S,separator,2);
    v := Str.StrToCard(R,10,ok);
    IF ok=FALSE THEN RETURN FALSE; END;
    IF v < 100 THEN INC(v,century); END;
    IF (v < minyy) OR (v > maxyy) THEN RETURN FALSE; END;
    date.year := CARDINAL(v);
    RETURN TRUE;
END parseDate;

PROCEDURE parseTime (S : ARRAY OF CHAR;
                     VAR time : timetype) : BOOLEAN;
CONST
    separator=colon;
    legaltimeset = digits+separator;
    minhh=0;
    maxhh=23;
    minmm=0;
    maxmm=59;
    minss=0;
    maxss=59;
VAR
    i : CARDINAL;
    R : str80;
    v : LONGCARD;
    ok: BOOLEAN;
BEGIN
    FOR i := 0 TO (Str.Length(S)-1) DO
        IF Str.CharPos(legaltimeset,S[i])=MAX(CARDINAL) THEN RETURN FALSE; END;
    END;
    i := CharCount(S,separator);
    IF (i # 1) AND (i # 2) THEN RETURN FALSE; END;

    Str.ItemS(R,S,separator,0);
    v := Str.StrToCard(R,10,ok);
    IF ok=FALSE THEN RETURN FALSE; END;
    IF (v < minhh) OR (v > maxhh) THEN RETURN FALSE; END;
    time.hours := CARDINAL(v);

    Str.ItemS(R,S,separator,1);
    v := Str.StrToCard(R,10,ok);
    IF ok=FALSE THEN RETURN FALSE; END;
    IF (v < minmm) OR (v > maxmm) THEN RETURN FALSE; END;
    time.minutes := CARDINAL(v);

    CASE i OF
    | 1 :
        time.seconds := minss;
    | 2 :
        Str.ItemS(R,S,separator,2);
        v := Str.StrToCard(R,10,ok);
        IF ok=FALSE THEN RETURN FALSE; END;
        IF (v < minss) OR (v > maxss) THEN RETURN FALSE; END;
        time.seconds := CARDINAL(v);
    END;

    RETURN TRUE;
END parseTime;

PROCEDURE DTconvertFIOtoPrivate (stamp:FIO.FileStamp;
                                 VAR d : datetype; VAR t : timetype  );
BEGIN
    d.day     := CARDINAL( stamp.Day);
    d.month   := CARDINAL( stamp.Month);
    d.year    := CARDINAL( stamp.Year)    +   century;
    t.hours   := CARDINAL( stamp.Hour);
    t.minutes := CARDINAL( stamp.Min);
    t.seconds := CARDINAL( stamp.Sec);
END DTconvertFIOtoPrivate;

(*
    Y M D hh mm ss
    comparison with longreal julian dates was not always secure !
    makeMyDay does not always give the same result for the same data
    dd hh mm ss !
*)

PROCEDURE compDT (d1:datetype;t1:timetype;
                  d2:datetype;t2:timetype) : INTEGER;
CONST
    k4 = 10000;
    k2 =   100;
    secondsPerHour = 60 * 60;
    secondsPerMinute=60;
VAR
    yyyymmdd1,yyyymmdd2:LONGCARD;
    s1,s2:LONGCARD;
BEGIN
    yyyymmdd1 := LONGCARD(d1.year) * k4 + LONGCARD(d1.month) * k2 + LONGCARD(d1.day);
    yyyymmdd2 := LONGCARD(d2.year) * k4 + LONGCARD(d2.month) * k2 + LONGCARD(d2.day);
    IF yyyymmdd1 < yyyymmdd2 THEN RETURN -1; END;
    IF yyyymmdd1 > yyyymmdd2 THEN RETURN  1; END;
    (* older version used hh*k4+mm*k2+ss *)
    s1 := LONGCARD(t1.hours) * secondsPerHour + LONGCARD(t1.minutes) * secondsPerMinute + LONGCARD(t1.seconds);
    s2 := LONGCARD(t2.hours) * secondsPerHour + LONGCARD(t2.minutes) * secondsPerMinute + LONGCARD(t2.seconds);
    IF s1 < s2 THEN RETURN -1; END;
    IF s1 > s2 THEN RETURN  1; END;
    RETURN 0;
END compDT;

(*
Year stored relative to 1980 (ex. 1988 stores as 8)
    year      month    day   

 F E D C B A 9 8 7 6 5 4 3 2 1 0   <-- Bit Number
*)

PROCEDURE PackDMY (d,m,y : CARDINAL  ) : CARDINAL;
CONST
    yyMask=BITSET{9..15};
    yyShft=9;
    mmMask=BITSET{5..8};
    mmShft=5;
    ddMask=BITSET{0..4};
    ddShft=0;
BEGIN
    y := (y - baseyear) << yyShft; (* 1980 *)
    m :=              m << mmShft;
    RETURN (y + m + d);
END PackDMY;

(*
Seconds are 0 to 29 -- DOS stores nearest even / 2
  hours    minutes   seconds 

 F E D C B A 9 8 7 6 5 4 3 2 1 0   <-- Bit Number
*)

PROCEDURE PackHMS (h,m,s:CARDINAL  ) : CARDINAL;
CONST
    hhMask=BITSET{11..15};
    hhShft=11;
    mmMask=BITSET{5..10};
    mmShft=5;
    ssMask=BITSET{0..4};
    ssShft=0;
BEGIN
    h := h << hhShft;
    m := m << mmShft;
    s := s >> 1;
    RETURN (h + m + s);
END PackHMS;

PROCEDURE PackDateTime (d:datetype;t:timetype) : dttype;
VAR
    dt : dttype;
BEGIN
    dt.d := PackDMY(d.day,d.month,d.year);
    dt.t := PackHMS(t.hours,t.minutes,t.seconds);
    RETURN dt;
END PackDateTime;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE substring (S1,S2:ARRAY OF CHAR) : BOOLEAN;
BEGIN
    Str.Caps(S2);
    IF Str.Pos(S1,S2) # MAX(CARDINAL) THEN RETURN TRUE; END;
    Str.Lows(S2);
    IF Str.Pos(S1,S2) # MAX(CARDINAL) THEN RETURN TRUE; END;
    RETURN FALSE;
END substring;

PROCEDURE newstring (VAR R:ARRAY OF CHAR; old,new:ARRAY OF CHAR);
BEGIN
    Str.Caps(old);
    IF Str.Pos(R,old) # MAX(CARDINAL) THEN
        Str.Subst(R,old,new);
    ELSE
        Str.Lows(old);
        IF Str.Pos(R,old) = MAX(CARDINAL) THEN RETURN; END;
        Str.Subst(R,old,new);
    END;
END newstring;

PROCEDURE using (n : CARDINAL; digits : CARDINAL; pad : CHAR) : str80;
VAR
    ok   : BOOLEAN;
    v    : LONGCARD;
    len  : CARDINAL;
    S    : str80;
BEGIN
    v := LONGCARD(n);
    Str.CardToStr(v,S,10,ok);
    IF pad="" THEN RETURN S; END;
    len := Str.Length(S);
    LOOP
        IF Str.Length(S) >= digits THEN EXIT; END;
        Str.Prepend(S,pad);
    END;
    RETURN S;
END using;

PROCEDURE oldnew (sOld,sNew:ARRAY OF CHAR;VAR R:ARRAY OF CHAR);
BEGIN
    LOOP
        IF substring(R,sOld)=FALSE THEN EXIT;END;
        newstring(R,sOld,sNew);
    END;
END oldnew;

(*
    $dd     lundi..dimanche / monday..sunday
    $d      1..31
    $0d     01..31
    $mm     janvier..dcembre / january..december
    $m      1..12
    $0m     01..12

    $yyyy   ####
    $yy     00..99

    !d      1er..31
    !0d     1er..31

    %h      0..23
    %0h     00..23
    %m      0..59
    %0m     00..59
    %s      0..59
    %0s     00..59
*)

TYPE
    languagetype = (french, english);

CONST
    jours   = "dimanche lundi mardi mercredi jeudi vendredi samedi";
    mois    = "janvier fvrier mars avril mai juin "+
              "juillet aot septembre octobre novembre dcembre";
    joursUS = "Sunday Monday Tuesday Wednesday Thursday Friday Saturday";
    moisUS  = "January February March April May June "+
              "July August September October November December";

PROCEDURE formatDT (dmy:datetype;hms:timetype;
                    format:ARRAY OF CHAR;language:languagetype;
                    VAR R : ARRAY OF CHAR  );
VAR
    S,T    : str16;
    pad    : CHAR;
    v      : CARDINAL;
BEGIN
    Str.Copy(R,format);

    v := ORD(dmy.dayOfWeek);
    CASE language OF
    | french:   Str.ItemS(T,jours," ",v);
    | english:  Str.ItemS(T,joursUS," ",v);
    END;
    oldnew("$dd",T,R);

    v:=dmy.day;
    pad := "";   oldnew("$d",    using(v,2,pad), R);
    pad := "0";  oldnew("$0d",   using(v,2,pad), R);

    IF v=1 THEN
    pad := "";   oldnew("!d",    "1er"         , R);
    pad := "0";  oldnew("!0d",   "1er"         , R);
    ELSE
    pad := "";   oldnew("!d",    using(v,2,pad), R);
    pad := "0";  oldnew("!0d",   using(v,2,pad), R);
    END;

    v := dmy.month-1;
    CASE language OF
    | french:   Str.ItemS(T,mois," ",v);
    | english:  Str.ItemS(T,moisUS," ",v);
    END;
    oldnew("$mm",T,R);

    v := dmy.month;
    pad := "";  oldnew("$m",    using(v,2,pad), R);
    pad := "0"; oldnew("$0m",   using(v,2,pad), R);

    v := dmy.year;
    pad := "";  oldnew("$yyyy", using(v,4,pad), R);
    pad := "0";
    IF v >= 1900 THEN
        v  := dmy.year - 1900;IF v > 100 THEN DEC(v,100);END;
    END;
                oldnew("$yy",   using(v,2,pad), R);

    v := hms.hours;
    pad := "";   oldnew("%h",    using(v,2,pad), R);
    pad := "0";  oldnew("%0h",   using(v,2,pad), R);

    v := hms.minutes;
    pad := "";   oldnew("%m",    using(v,2,pad), R);
    pad := "0";  oldnew("%0m",   using(v,2,pad), R);

    v := hms.seconds;
    pad := "";   oldnew("%s",    using(v,2,pad), R);
    pad := "0";  oldnew("%0s",   using(v,2,pad), R);

END formatDT;

PROCEDURE answerYes (prompt : ARRAY OF CHAR) : BOOLEAN;
CONST
    yes = str2("Y");
    no  = str2("N");
    oui = str2("O");
    non = str2("N");
VAR
    key : str2;
    rc  : BOOLEAN;
BEGIN
    WrStr(prompt);
    WrStr(" (yes/no) ? ");
    Flushkey;
    key := Waitkey();
    UpperCase(key);
    IF same(key,yes) OR same(key,oui) THEN
        WrStr("Yes");
        rc := TRUE;
    ELSE
        WrStr("No");
        rc := FALSE;
    END;
    WrLn;
    RETURN rc;
END answerYes;

(* ------------------------------------------------------------ *)

PROCEDURE bip (  );
BEGIN
    sound (55,50,10);
END bip;

(*
    gregorian calendar rule : leap 366 years if divisible by 4
    but centurial years NOT divisible by 4 are common
    Leap years are divisible by 400, or by 4 and not by 100
    1900 is common, 2000 is leap
*)

PROCEDURE getDaysInFebruary (annee:CARDINAL):CARDINAL;
CONST
    common = 28;
    leap   = 29;
BEGIN
    IF (annee MOD 400) = 0 THEN RETURN leap; END;
    IF (((annee MOD 4) = 0) AND ((annee MOD 100) # 0)) THEN RETURN leap; END;
    RETURN common;
END getDaysInFebruary;

PROCEDURE getDaysInMonth (mois,annee:INTEGER):INTEGER;
TYPE
    daysinmonthtype = ARRAY [1..12] OF INTEGER;
CONST
    (*                    JanFevMarAprMayJunJulAugSepOctNovDec *)
    (*                     1  2  3  4  5  6  7  8  9 10 11 12  *)
    dpm = daysinmonthtype(31, 0,31,30,31,30,31,31,30,31,30,31);
VAR
    m : INTEGER;
BEGIN
    m:= dpm[mois];
    IF m=0 THEN m:=getDaysInFebruary(annee);END;
    RETURN m;
END getDaysInMonth;

(* we can adjust only one parameter out of three here *)
(* assume abs(plusday) < halfmonth, abs(plusmonth/year) <= 1 *)
(* the procedure I did not bother to write for Q&D AstroTools CARTE*.* ! *)

PROCEDURE adjDMY (VAR d : datetype ; plusday,plusmonth,plusyear : INTEGER  );
CONST
    minD = 1;    (* maxD depends upon month and year *)
    minM = 1;    maxM = 12;
    minY = 1980; maxY = 2099;
VAR
    j,m,a,lastday:INTEGER;
BEGIN
    j := INTEGER(d.day);
    m := INTEGER(d.month);
    a := INTEGER(d.year);

    INC(j,plusday);
    IF j < minD THEN
        DEC(m);
        IF m < minM THEN
            DEC(a);
            IF a < minY THEN
                RETURN;
            ELSE
                m:=maxM;
                INC( j, getDaysInMonth(m,a) );
            END;
        ELSE
            INC( j, getDaysInMonth(m,a) );
        END;
    ELSIF j > getDaysInMonth(m,a) THEN
        INC(m);
        IF m > maxM THEN
            INC(a);
            IF a > maxY THEN
                RETURN;
            ELSE
                m := minM;
                DEC ( j, getDaysInMonth(maxM,a) );
            END;
        ELSE
            DEC (j, getDaysInMonth(m-1,a) );
        END;
    END;

    INC(m, plusmonth);
    IF m < minM THEN
        DEC(a);
        IF a < minY THEN
            RETURN;
        ELSE
            m := maxM;
        END;
    ELSIF m > maxM THEN
        INC(a);
        IF a > maxY THEN
            RETURN;
        ELSE
            m := minM;
        END;
    END;

    INC(a, plusyear);
    IF a < minY THEN
        RETURN;
    ELSIF a > maxY THEN
        RETURN;
    END;

    IF j > getDaysInMonth(m,a) THEN j:=getDaysInMonth(m,a);END; (* adjust for leap years *)

    d.day := CARDINAL(j);
    d.month:=CARDINAL(m);
    d.year:= CARDINAL(a);
END adjDMY;

(* pass integers because value can be negative ! *)

PROCEDURE hmsTOseconds (h,m,s:INTEGER):LONGINT ;
CONST
    secondsPerMinute=60;
    secondsPerHour  =60*60;
BEGIN
    RETURN LONGINT(h)*secondsPerHour + LONGINT(m)*secondsPerMinute + LONGINT(s);
END hmsTOseconds;

(* we may adjust several parms together here because time is autowrap *)

PROCEDURE adjHMS(VAR t : timetype ; plushh,plusmm,plusss : INTEGER  );
CONST
    secondsPerDay = 24*60*60;
    secondsPerHour=    60*60;
VAR
    now,plus : LONGINT;
BEGIN
    now := hmsTOseconds( INTEGER(t.hours),INTEGER(t.minutes),INTEGER(t.seconds) );
    plus:= hmsTOseconds( plushh,plusmm,plusss);
    INC( now,plus);
    WHILE now < 0 DO
        INC(now,secondsPerDay);
    END;
    WHILE now >= secondsPerDay DO
        DEC(now,secondsPerDay);
    END;
    t.seconds := CARDINAL (  now MOD 60);
    t.minutes := CARDINAL ( (now DIV 60 ) MOD 60);
    t.hours   := CARDINAL (  now DIV secondsPerHour);
END adjHMS;

PROCEDURE xt (asuserlikesit,orgnow:BOOLEAN;
              VAR date1:datetype; VAR time1:timetype;
              VAR date2:datetype; VAR time2:timetype):BOOLEAN ;
CONST
    msgD     = 'Update dd-mm [arrows, "+", "-", Return, Esc] : ';
    msgT     = "Update hh:mm [arrows, Return, Esc] : "; (* year seems useless ! *)
    msgERR   = "*** Date and time must be later than reference date and time !";
    msgERRALT= "*** Date and time are earlier than reference date and time !";
    msgND    = ">>> New date : ";
    msgNT    = ">>> New time : ";
VAR
    S,S2 : str128;
    d : datetype;
    t : timetype;
    c1,c2:CHAR;
    shifted:BOOLEAN;
    cmd: (continue,cr,esc);
BEGIN
LOOP

    IF orgnow THEN GetDateTimeNow(date1,time1); END;

    d := date1;
    t := time1;
    (* t.seconds := 0; *)
    cmd := continue;
    LOOP
        formatDT (d,t, defaultFormatDATE,english,S2);
        Str.Concat(S,msgD,S2);
        video(S,TRUE);
        BiosFlushkey;
        BiosWaitkeyShifted(c1,c2,shifted);
        c1:=CAP(c1);
        CASE c1 OF
        | "+" :          adjDMY (d,  0, 0, 1);
        | "-" :          adjDMY (d,  0, 0,-1);
        | CHR(13): cmd:=cr;
        | CHR(27): cmd:=esc;
        | CHR(0) :
            CASE c2 OF
             | CHR(77) : adjDMY (d,  1, 0, 0); (* right *)
             | CHR(75) : adjDMY (d, -1, 0, 0); (* left *)
             | CHR(72) : adjDMY (d,  0, 1, 0); (* up *)
             | CHR(80) : adjDMY (d,  0,-1, 0); (* down *)
             | CHR(73) : adjDMY (d, 10, 0, 0); (* pageup *)
             | CHR(81) : adjDMY (d,-10, 0, 0); (* pagedown *)

             | CHR(132): adjDMY (d,  0, 0, 1); (* ctr-pageup *)
             | CHR(118): adjDMY (d,  0, 0,-1); (* ctrl-pagedown *)

             ELSE
                 bip;
             END;
        ELSE
             bip;
        END;
        video(S,FALSE);
        IF cmd=esc THEN RETURN FALSE; END;
        IF cmd=cr THEN EXIT; END;
    END;
    WrStr(msgND);WrStr(S2);WrLn;
    cmd := continue;

    IF orgnow THEN GetDateTimeNow(date1,time1); t:=time1; END;

    LOOP
        formatDT (d,t, defaultFormatTIME,english,S2);
        Str.Concat(S,msgT,S2);
        video(S,TRUE);
        BiosFlushkey;
        BiosWaitkeyShifted(c1,c2,shifted);
        c1:=CAP(c1);
        CASE c1 OF
        | CHR(13): cmd:=cr;
        | CHR(27): cmd:=esc;
        | CHR(0) :
            CASE c2 OF
             | CHR(77) : adjHMS(t,  0,  1, 0); (* right *)
             | CHR(75) : adjHMS(t,  0, -1, 0); (* left *)
             | CHR(72) : adjHMS(t,  1,  0, 0); (* up *)
             | CHR(80) : adjHMS(t, -1,  0, 0); (* down *)
             | CHR(73) : adjHMS(t,  0, 10, 0); (* pageup *)
             | CHR(81) : adjHMS(t,  0,-10, 0); (* pagedown *)
             ELSE
                 bip;
             END;
        ELSE
             bip;
        END;
        video(S,FALSE);
        IF cmd=esc THEN RETURN FALSE; END;
        IF cmd=cr THEN EXIT; END;
    END;
    WrStr(msgNT);WrStr(S2);WrLn;

    IF orgnow THEN EXIT; END; (* anything will go ! *)

    (* check if new date/time is after reference ! *)
    IF compDT(d,t, date1,time1) >= 0 THEN EXIT;END;
    IF asuserlikesit THEN
        (* user has been warned, but he has the sacred right to be wrong ! *)
        WrStr(msgERRALT);WrLn; (* slightly modified msgERR *)
        alarm;
        EXIT;
    END;
    WrStr(msgERR);WrLn;
    alarm;
END;
(* ok, update d/t for clock and for file *)
date1 := d;
time1 := t;
date2 := d;
time2 := t;
RETURN TRUE;
END xt;

(* ------------------------------------------------------------ *)

CONST
    msgFix          = "--- ";
    msgInfo         = "::: ";
    msgGood         = "+++ ";
    msgSPC          = "    ";
    msgCurrent      = msgInfo+"Current date/time : ";
    msgOrigin       = msgInfo+"Origin date/time  : ";
    msgSaved        = msgInfo+"Stored date/time  : ";
    msgOKtest1      = msgGood+"System clock is later than origin date/time.";
    msgOKtest2      = msgGood+"System clock is later than stored date/time.";
    msgPBtest2      =  msgFix+"System clock is earlier than stored date/time !";
    msgUpdated      =        " date/time stamp was successfully updated.";
    msgUntouched    =        " date/time stamp was NOT updated.";
    msgClockUpdated =msgInfo+"System clock was updated with stored date/time."; (* was msgFix *)
    msgSureClock    = msgSPC+"Are you sure you want to update system clock";
    msgUserEscape   = msgFix+"User did not reset date/time !";
    (*
    msgOK               = " OK";
    msgClockBeforeOrigin= " <-- System clock is earlier than origin !";
    msgClockBeforeStored= " <-- System clock is earlier than stored !";
    *)

PROCEDURE doTest (ink,paper,inkwarning,paperwarning,inkok,paperok:CARDINAL);
BEGIN
    color(ink,paper);
    WrStr("Normal  : "+msgOKtest2);
    color(ink,paper);
    WrLn;

    color(inkok,paperok);
    WrStr("OK      : "+msgOKtest2);
    color(ink,paper);
    WrLn;

    color(inkwarning,paperwarning);
    WrStr("Warning : "+msgPBtest2);
    color(ink,paper);
    WrLn;
END doTest;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

VAR
    dateref,datenow,datesaved   : datetype;
    timeref,timenow,timesaved   : timetype;
    dtnow     : dttype;
VAR
    DEBUG,UPDATEFILE,UPDATECLOCK,VERBOSE:BOOLEAN;
    IGNORERO,QUERY,INTERACTIVE,FORCE,BASENOW,OLDERDISPLAY: BOOLEAN;
    TESTMODE,wasReadOnly : BOOLEAN;
    compDate,compTime,format : str128; (* oversized ! *)
    newDMY,newHMS:BOOLEAN;
    hnd : FIO.File;
    dtfile : FIO.FileStamp;
    Sref,Ssav,Snow:str128;
VAR
    parmcount,i,opt : CARDINAL;
    S,R             : str128;
    state           : (waiting,gotspec);
    spec           : str128;
    rc,changedOK    : BOOLEAN;
    v               : LONGCARD;
    ink,paper,inkwarning,paperwarning,inkok,paperok : CARDINAL;
BEGIN
    Lib.DisableBreakCheck();

    (* handleVesa; *) (* useless, because we won't change video mode *)
    setUseBiosMode ( IsRedirected() );
    findInkPaperAtStartup();
    ink          := ORD(defaultink);
    paper        := ORD(defaultpaper);
    inkwarning   := ORD(defaultinkwarning);
    paperwarning := ORD(defaultpaperwarning);
    inkok        := ORD(defaultinkok);
    paperok      := ORD(defaultpaperok);

    WrLn; (* here is ok *)

    FIO.IOcheck := FALSE;

    IF AtLeastDosVersion(3,20)=FALSE THEN abort(errDOSversion,"3.20"); END;
    (* IF IsRedirected() THEN abort(errRedirected,"");END; *)
    DEBUG      := FALSE;
    UPDATEFILE := TRUE;
    UPDATECLOCK:= TRUE;
    AUDIO      := FALSE;
    VERBOSE    := TRUE;
    IGNORERO   := FALSE;
    QUERY      := TRUE;
    INTERACTIVE:= FALSE;
    Str.Copy(compDate,defaultCompDate);
    Str.Copy(compTime,defaultCompTime);
    Str.Copy(format,defaultFormat);
    newDMY     := FALSE;
    newHMS     := FALSE;
    FORCE      := FALSE;
    BASENOW    := FALSE; (* useless initialization ! ;-) *)
    OLDERDISPLAY:=FALSE;
    TESTMODE   := FALSE;

    state   := waiting;
    parmcount := Lib.ParamCount();
    (* IF parmcount=0 THEN abort(errHelp,"");END; *)
    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i); cleantabs(S); (* YATB ! *)
        Str.Copy(R,S);
        UpperCase(R);
        IF isOption(R) THEN
            opt := GetOptIndex (R, "?"+delim+"H"+delim+"HELP"+delim+
                                   "DEBUG"+delim+
                                   "S"+delim+"NOSTAMP"+delim+
                                   "C"+delim+"NOCLOCK"+delim+
                                   "W"+delim+"AUDIOWARN"+delim+
                                   "T"+delim+"TERSE"+delim+
                                   "R"+delim+"IGNORERO"+delim+
                                   "D:"+delim+"DATE:"+delim+
                                   "T:"+delim+"TIME:"+delim+
                                   "F:"+delim+"FORMAT:"+delim+
                                   "A"+delim+"AUTO"+delim+
                                   "I"+delim+"INTERACTIVE"+delim+
                                   "F"+delim+"FORCE"+delim+
                                   "FF"+delim+"FORCENOW"+delim+
                                   "O"+delim+"OLDERDISPLAY"+delim+
                                   "B"+delim+"BIOS"+delim+
                                   "I:"+delim+"INK:"+delim+
                                   "P:"+delim+"PAPER:"+delim+
                                   "IW:"+delim+
                                   "PW:"+delim+
                                   "II:"+delim+
                                   "PP:"+delim+
                                   "TEST"
                               );
            CASE opt OF
            | 1,2,3 : abort(errHelp,"");
            | 4     : DEBUG      := TRUE;
            | 5,6   : UPDATEFILE := FALSE;
            | 7,8   : UPDATECLOCK:= FALSE;
            | 9,10  : AUDIO      := TRUE;
            | 11,12 : VERBOSE    := FALSE;
            | 13,14 : IGNORERO   := TRUE;
            | 15,16 : GetString(R,compDate); newDMY:=TRUE;
            | 17,18 : GetString(R,compTime); newHMS:=TRUE;
            | 19,20 : GetString(S,format);   (* keep case... just in case ! *)
            | 21,22 : QUERY      := FALSE;
            | 23,24 : INTERACTIVE:= TRUE;
            | 25,26 : FORCE      := TRUE; BASENOW:=FALSE;
            | 27,28 : FORCE      := TRUE; BASENOW:=TRUE;
            | 29,30 : OLDERDISPLAY:=TRUE;
            | 31,32:
                setUseBiosMode ( TRUE );
            | 33,34:
                IF GetLongCard(S,v)=FALSE THEN abort(errBadNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errRange,"Ink");END;
                ink:=CARDINAL(v);
            | 35,36:
                IF GetLongCard(S,v)=FALSE THEN abort(errBadNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errRange,"Paper");END;
                paper:=CARDINAL(v);
            | 37:
                IF GetLongCard(S,v)=FALSE THEN abort(errBadNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errRange,"Warning ink");END;
                inkwarning:=CARDINAL(v);
            | 38:
                IF GetLongCard(S,v)=FALSE THEN abort(errBadNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errRange,"Warning paper");END;
                paperwarning:=CARDINAL(v);
            | 39:
                IF GetLongCard(S,v)=FALSE THEN abort(errBadNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errRange,"OK ink");END;
                inkok:=CARDINAL(v);
            | 40:
                IF GetLongCard(S,v)=FALSE THEN abort(errBadNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errRange,"OK paper");END;
                paperok:=CARDINAL(v);
            | 41:
                TESTMODE:=TRUE;
            ELSE
                abort(errOption,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            CASE state OF
            | waiting : Str.Copy(spec,S);
            | gotspec: abort(errTooManyParms,S);
            END;
            INC(state);
        END;
    END;

    IF IsRedirected() THEN abort(errRedirected,"");END;

    IF TESTMODE THEN
        doTest(ink,paper,inkwarning,paperwarning,inkok,paperok);
        abort(errNone,"");
    END;

    (* check nonsense *)
    CASE state OF
    | waiting :
        spec:=""; (* in exe dir, try ini then exe itself *)
    | gotspec:
        IF chkJoker(spec) THEN abort(errJoker,spec);END;
        IF chkValidName(spec)=FALSE THEN abort(errBadName,spec); END;
    END;
    IF procReference(spec)=FALSE THEN abort(errNoMatch,spec);END;

    IF (INTERACTIVE AND FORCE) THEN abort(errConflict,"");END;
    IF FORCE THEN INTERACTIVE := TRUE; END; (* trick ! *)
    IF INTERACTIVE THEN
        IGNORERO    := TRUE;  (* added safety *)
        UPDATECLOCK := TRUE;  (* force system clock update *)
    END;

    wasReadOnly := isReadOnly(spec);
    IF wasReadOnly THEN
        IF IGNORERO=FALSE THEN abort(errWriteProtected,spec);END;
    END;

    IF newDMY THEN
        IF newHMS=FALSE THEN Str.Copy(compTime,defaultCompTimeZero);END;
    ELSE
        IF newHMS THEN abort(errDateNeeded,"");END;
    END;

    IF parseDate(compDate,dateref)=FALSE THEN abort(errBadDate,compDate);END;
    IF parseTime(compTime,timeref)=FALSE THEN abort(errBadTime,compTime);END;

    color(ink,paper);

    CASE OLDERDISPLAY OF
    | TRUE:
        GetDateTimeNow(datenow,timenow);
        IF VERBOSE THEN
            formatDT (dateref,timeref, format,english,S);
            WrStr(msgOrigin);WrStr(S);WrLn;
            formatDT (datenow,timenow, format,english,S);
            WrStr(msgCurrent);WrStr(S);WrLn;
        END;
        IF compDT(datenow,timenow, dateref,timeref) < 0 THEN
            abort(errClockBeforeOrigin,""); (* severe problem *)
        END;
        color(inkok,paperok);
        WrStr(msgOKtest1);
        color(ink,paper);
        WrLn;

        hnd:=FIO.OpenRead(spec);
        IF hnd = MAX(CARDINAL) THEN abort(errOpening,spec); END;
        rc:=FIO.GetFileStamp(hnd,dtfile);
        FIO.Close(hnd);
        IF rc=FALSE THEN abort(errGetting,spec); END;
        FixSeconds(dtfile);
        DTconvertFIOtoPrivate(dtfile,datesaved,timesaved);

        GetDateTimeNow(datenow,timenow);

        IF VERBOSE THEN
            formatDT (datesaved,timesaved, format,english,S);
            WrStr(msgSaved);WrStr(S);WrLn;
            formatDT (datenow,timenow, format,english,S);
            WrStr(msgCurrent);WrStr(S);WrLn;
        END;
    | FALSE: (* newer display, order of checks was modified *)
        hnd:=FIO.OpenRead(spec);
        IF hnd = MAX(CARDINAL) THEN abort(errOpening,spec); END;
        rc:=FIO.GetFileStamp(hnd,dtfile);
        FIO.Close(hnd);
        IF rc=FALSE THEN abort(errGetting,spec); END;
        FixSeconds(dtfile);
        DTconvertFIOtoPrivate(dtfile,datesaved,timesaved);

        GetDateTimeNow(datenow,timenow);

        IF VERBOSE THEN
            formatDT (dateref,timeref, format,english,S);
            WrStr(msgOrigin);WrStr(S);WrLn;
            formatDT (datesaved,timesaved, format,english,S);
            WrStr(msgSaved);WrStr(S);WrLn;
            formatDT (datenow,timenow, format,english,S);
            WrStr(msgCurrent);WrStr(S);WrLn;
        END;

        IF compDT(datenow,timenow, dateref,timeref) < 0 THEN
            abort(errClockBeforeOrigin,""); (* severe problem *)
        END;
        color(inkok,paperok);
        WrStr(msgOKtest1);
        color(ink,paper);
        WrLn;
    END;

    IF DEBUG THEN
        changedOK:=xt(FORCE,BASENOW,datesaved,timesaved, datenow,timenow);
        IF changedOK THEN
            formatDT (datenow,timenow, format,english,S);
            (* WrStr("New date and time : ");WrStr(S);WrLn; *)
            WrStr("User has modified date and time !");WrLn;
        ELSE
            WrStr("User has not modified date and time !");WrLn;
        END;
        abort(errNone,"");
    END;

    IF ( FORCE OR (compDT(datenow,timenow, datesaved,timesaved) < 0) ) THEN
        IF UPDATECLOCK THEN
            IF FORCE THEN
                WrLn;
            ELSE
                color(inkwarning,paperwarning);
                WrStr(msgPBtest2);
                color(ink,paper);
                WrLn;
            END;
            IF AUDIO THEN alert();END;
            IF INTERACTIVE THEN
                changedOK:=xt(FORCE,BASENOW,datesaved,timesaved, datenow,timenow);
                rc         := changedOK;
                UPDATEFILE := changedOK;
            ELSE
                IF QUERY THEN
                    rc := answerYes(msgSureClock);
                ELSE
                    rc := TRUE;
                END;
            END;
            IF rc THEN
                rc := SetDate(datesaved.year,datesaved.month,datesaved.day);
                IF rc = FALSE THEN abort(errSettingDate,""); END;
                rc := SetTime(timesaved.hours,timesaved.minutes,timesaved.seconds,0);
                IF rc = FALSE THEN abort(errSettingTime,""); END;
                IF FORCE THEN
                    WrLn;
                ELSE
                    WrStr(msgClockUpdated);WrLn;
                END;
            END;
            IF INTERACTIVE THEN
                IF changedOK=FALSE THEN
                    WrStr(msgUserEscape);WrLn;
                END;
            END;
        ELSE
            abort(errClockBeforeStored,"");
        END;
    ELSE
        color(inkok,paperok);
        WrStr(msgOKtest2);
        color(ink,paper);
        WrLn;
    END;

    IF UPDATEFILE THEN
        dtnow := PackDateTime (datenow,timenow);
        IF wasReadOnly THEN setReadWrite(spec);END;
        hnd := FIO.Open(spec);
        FIO.SetFileDate(hnd, dtnow.dt);
        FIO.Close(hnd);
        IF wasReadOnly THEN setReadOnly(spec);END;
        WrStr(msgGood);WrStr(spec);WrStr(msgUpdated);WrLn;
    ELSE
        WrStr(msgInfo);WrStr(spec);WrStr(msgUntouched);WrLn;
    END;
    colorhelp;

    abort(errNone,"");
END DTchk.


