
(* ---------------------------------------------------------------
Title         Q&D console library
Author        PhG
Overview
Notes         a DOS text-mode library... we MUST be kidding !
              look for // in comments
Bugs          in SMALL model only, lib.paramstr(0) was garbled.
              something to do with vesa check ???
Wish List

--------------------------------------------------------------- *)

IMPLEMENTATION MODULE QD_Text;

IMPORT SYSTEM;
IMPORT Lib;
IMPORT Str;
FROM IO IMPORT WrStr;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

TYPE
    biosCursorPos = RECORD
        xcursor : SHORTCARD;
        ycursor : SHORTCARD;
    END;
    videocell = RECORD
        ch   : CHAR;
        attr : SHORTCARD;
    END;
    screentype     = ARRAY [0..16384-1] OF videocell; (* should be enough *)
    (* always force a far pointer *)
    (*# save, data(near_ptr=> off) *)
    screenptrtype  = POINTER TO screentype;
    (*# restore *)
CONST
    vi = 010H; (* video interrupt *)
    sb = 040H; (* segBiosData *)
CONST
    segMono  = 0B000H;
    segColor = 0B800H;
VAR
    biosColumnsOnScreen   [sb:04AH] : WORD;
    biosRowsOnScreen      [sb:084H] : BYTE; (* rows-1 in fact *)
    biosCurrentVideoMode  [sb:049H] : BYTE;
    biosPageStart         [sb:04EH] : WORD; (* offset from screen segment *)
    biosCurrentPage       [sb:062H] : BYTE;
    biosCursorPosition    [sb:050H] : ARRAY [0..7] OF biosCursorPos;
    biosCursorType        [sb:060H] : WORD;

(* ------------------------------------------------------------ *)

CONST
    (* relative to window, not absolute to screen !!! *)
    minimumvtab = 0;
    maximumvtab = 64-1; (* 60 rows should be a maximum but who knows ? *)
VAR
    basevtab : ARRAY[minimumvtab..maximumvtab] OF CARDINAL;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

VAR
    segScreen,ofsScreen                      : CARDINAL;
    screen                                   : screenptrtype;
    visualPage                               : CARDINAL;
    activePage                               : CARDINAL;
    cursorX,cursorY                          : CARDINAL;
    screenWidth,screenHeight                 : CARDINAL;
    wWidth,wHeight                           : CARDINAL;
    wLeft,wTop,wRight,wBottom                : CARDINAL;
    htab,vtab                                : CARDINAL;
    maxhtab,maxvtab                          : CARDINAL; (* relative ! *)
    fillChar                                 : CHAR;
    fillAttr                                 : SHORTCARD;
    fillInk                                  : colortype;
    fillPaper                                : colortype;
    txtAttr                                  : SHORTCARD;
    txtInk                                   : colortype;
    txtPaper                                 : colortype;
    egavgahere                               : BOOLEAN;
    vgahere                                  : BOOLEAN;
    vesahere                                 : BOOLEAN;
    brightPaper                              : BOOLEAN;
    blinkMode                                : BOOLEAN; (* ignored if brightPaper enabled *)
    wrapMode                                 : BOOLEAN;
    useBiosMode                              : BOOLEAN;
    cursorShape                              : cursorshapetype;
    tabWidth                                 : CARDINAL;
VAR
    pageAtStartup                            : CARDINAL;
    DOSinkAtStartup                          : colortype;
    DOSpaperAtStartup                        : colortype;
VAR
    oldModeSaved   : BOOLEAN;
    oldMode        : BYTE;
    oldWidth       : CARDINAL;
    oldHeight      : CARDINAL;
    oldPage        : BYTE;
    oldCursorShape : CARDINAL;
CONST
    minx    = 0; (* absolute left of screen *)
    miny    = 0; (* absolute top of screen *)
    minhtab = 0; (* relative to wLeft *)
    minvtab = 0; (* relative to wTop *)
    defaultFillChar     = " ";
    defaultFillInk      = gray;   (* assume ugly BIOS defaults *)
    defaultFillPaper    = black;

    defaultTxtInk       = gray;   (* assume ugly BIOS defaults *)
    defaultTxtPaper     = black;

    defaultBrightPaper  = TRUE; (* if possible *)
    defaultBlinkMode    = FALSE;
    defaultWrapMode     = TRUE;
    defaultUseBiosMode  = FALSE;

    defaultCursorShape  = defaultcursor;
    defaultCursorEmulation=TRUE;
    defaultTabWidth     = 8; (* could be 4 *)

	defaultDetectVesaToo= FALSE;
VAR
    R : SYSTEM.Registers; (* global *)

(* ------------------------------------------------------------ *)

PROCEDURE initFlags(  );
BEGIN
    activePage   := 0;
    visualPage   := 0;
    oldModeSaved := FALSE;
    egavgahere   := FALSE;
    vgahere      := FALSE;
    vesahere     := FALSE;
    brightPaper  := FALSE;
    blinkMode    := FALSE;
    wrapMode     := TRUE;
    useBiosMode  := FALSE;
    cursorShape  := defaultcursor;
    tabWidth     := defaultTabWidth;
END initFlags;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE isMonoMode ():BOOLEAN;
BEGIN
    RETURN (biosCurrentVideoMode = BYTE(MONO));
END isMonoMode;

PROCEDURE selectCursorEmulation (state:BOOLEAN );
BEGIN
    IF NOT(vgahere) THEN RETURN; END;
    R.AH := 12H;
    R.BL := 34H;
    IF state THEN
        R.AL := 00H; (* enable *)
    ELSE
        R.AL := 01H; (* disable *)
    END;
    Lib.Intr(R,vi);     (* al=$12 if function supported : vga only *)
END selectCursorEmulation;

PROCEDURE setTextCursorShape (scanlines:CARDINAL );
BEGIN
    R.CX := scanlines;
    R.AH := 01H; (* set text-mode cursor shape *)
    Lib.Intr(R,vi);
END setTextCursorShape;

PROCEDURE setCursorShape (shape:cursorshapetype);
VAR
    scanlines : CARDINAL;
BEGIN
    cursorShape := shape;
    CASE shape OF
    | defaultcursor :
        IF isMonoMode () THEN
            scanlines := 00D0EH; (* 0b0c *)
        ELSE
            scanlines := 00607H;
        END;
    | halfcursor :
        IF isMonoMode () THEN
            scanlines := 0060EH;
        ELSE
            scanlines := 00307H;
        END;
    | blockcursor :
        IF isMonoMode () THEN
            scanlines := 0000EH;
        ELSE
            scanlines := 00007H;
        END;
    | invisiblecursor: (* bits 6,5 : 01 *)
        IF isMonoMode () THEN
            scanlines := 02D0EH; (* 2b0c *)
        ELSE
            scanlines := 02607H;
        END;
    | oldcursor:
        scanlines := oldCursorShape;
    END;
    setTextCursorShape(scanlines);
END setCursorShape;

(* ------------------------------------------------------------ *)

(* bugfix *)

(*# save *)
(*# data(near_ptr => off)  *)
TYPE
    vesapointer = POINTER TO WORD;
(*# restore *)

CONST
    firstVesaBufferByte = 0;
    lastVesaBufferByte  = 768-1; (* 511 should be enough but... *)
VAR
    vesaBuffer  : ARRAY [firstVesaBufferByte..lastVesaBufferByte] OF BYTE;
    vesaListPtr : vesapointer;

PROCEDURE checkVesaHere () : BOOLEAN;
BEGIN
    R.AX := 4F00H;
    R.ES := Seg(vesaBuffer); (* buffer segment *)
    R.DI := Ofs(vesaBuffer); (* buffer offset *)
    Lib.Intr(R,vi);
    IF R.AL # 4FH THEN RETURN FALSE; END;
    IF R.AH # 00H THEN RETURN FALSE; END;
    IF vesaBuffer[0] # BYTE("V") THEN RETURN FALSE; END;
    IF vesaBuffer[1] # BYTE("E") THEN RETURN FALSE; END;
    IF vesaBuffer[2] # BYTE("S") THEN RETURN FALSE; END;
    IF vesaBuffer[3] # BYTE("A") THEN RETURN FALSE; END;
    Lib.Move (ADR(vesaBuffer[0EH]),ADR(vesaListPtr),4); (* set pointer to list address *)
    RETURN TRUE;
END checkVesaHere;

PROCEDURE isVesaModeAvailable (listPtr : vesapointer; mode : CARDINAL) : BOOLEAN;
BEGIN
    LOOP
        IF listPtr^ = WORD(0FFFFH) THEN EXIT; END;
        IF listPtr^ = WORD(mode) THEN RETURN TRUE; END;
        Lib.IncFarAddr(listPtr,SIZE(WORD));
    END;
    RETURN FALSE;
END isVesaModeAvailable;

PROCEDURE setVesaMode (mode : CARDINAL) : BOOLEAN;
BEGIN
    R.AX := 4F02H;
    R.BX := mode;
    Lib.Intr(R,vi);
    IF R.AL # 4FH THEN RETURN FALSE; END;
    IF R.AH # 00H THEN RETURN FALSE; END;
    RETURN TRUE;
END setVesaMode;

PROCEDURE possibleVesaMode (width,height:CARDINAL;VAR mode:CARDINAL ):BOOLEAN;
VAR
    ok : BOOLEAN;
BEGIN
    ok := TRUE; (* default is yes *)
    CASE width OF
    | 80 :
        CASE height OF
        | 60 : mode:= vesa80x60;
        ELSE
            ok:=FALSE;
        END;
    | 132 :
        CASE height OF
        | 25 : mode:= vesa132x25;
        | 43 : mode:= vesa132x43;
        | 50 : mode:= vesa132x50;
        | 60 : mode:= vesa132x60;
        ELSE
            ok:=FALSE;
        END;
    ELSE
        ok:=FALSE;
    END;
    RETURN ok;
END possibleVesaMode;

(* ------------------------------------------------------------ *)

PROCEDURE checkAtLeastEGA (  ) : BOOLEAN ;
BEGIN
    R.AX := 01200H; (* alternate function select *)
    R.BX := 0FF10H;
    Lib.Intr(R,vi);
    RETURN (R.BX # 0FFH);
END checkAtLeastEGA;

PROCEDURE checkVGA (  ) : BOOLEAN;
BEGIN
    R.AX := 01A00H; (* get display combination code *)
    Lib.Intr(R,vi);
    IF R.AL # 01AH THEN RETURN FALSE; END;
    (* another safety check *)
    R.AX := 01C00H; (* save/restore video state : ask state buffer size *)
    R.CX := 00001H; (* ask about bios data areas *)
    Lib.Intr(R,vi);
    RETURN (R.AL = 01CH);
END checkVGA;

PROCEDURE getVideoCard (detectVesaToo:BOOLEAN);
BEGIN
    egavgahere := checkAtLeastEGA();
    vgahere    := checkVGA();
    IF detectVesaToo THEN
        vesahere := checkVesaHere();   (* logically, we could avoid this test if not ega/vga *)
    ELSE
        vesahere := FALSE; (* safety *)
    END;
END getVideoCard;

(* required if we are to set or reset vesa video modes *)

PROCEDURE handleVesa();
BEGIN
    getVideoCard(TRUE);
END handleVesa;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE setBrightPaper (state:BOOLEAN);
BEGIN
    IF NOT(egavgahere) THEN
        brightPaper := FALSE;
        RETURN;
    END;
    brightPaper := state;
    R.AX := 1003H; (* toggle intensity/blinking bit *)
    IF state THEN
        R.BX := 0000H; (* background intensity enabled *)
    ELSE
        R.BX := 0001H; (* blink enabled *)
    END;
    Lib.Intr(R,vi);
END setBrightPaper;

PROCEDURE setBlinkMode (state:BOOLEAN );
BEGIN
    blinkMode := state; (* ignored if brightPaper is enabled *)
END setBlinkMode;

PROCEDURE setFillChar (ch:CHAR);
BEGIN
    fillChar := ch;
END setFillChar;

PROCEDURE fixAttr (VAR Attr:SHORTCARD);
BEGIN
    IF NOT(brightPaper) THEN (* only 0..7 allowed *)
        Attr := (Attr AND 7FH); (* clear reserved blink bit 7 : 01111111b *)
        IF blinkMode THEN
            Attr := (Attr OR 80H); (* force blink bit 7 *)
        END;
    ELSE
        (* obviously ignore blinkMode setting here ! *)
    END;
END fixAttr;

PROCEDURE newInk (ink:colortype; VAR Attr : SHORTCARD; VAR kept:colortype);
BEGIN
    kept := ink;
    Attr := (Attr AND 0F0H); (* preserve paper bits 4..6 and blink bit 7 : 11110000b *)
    Attr := Attr OR SHORTCARD(ORD(ink));
    fixAttr(Attr);
END newInk;

PROCEDURE newPaper (paper:colortype; VAR Attr: SHORTCARD; VAR kept:colortype);
BEGIN
    kept := paper;
    Attr := (Attr AND 00FH); (* preserve ink bits 0..3 : 00001111b *)
    Attr := Attr OR SHORTCARD(ORD(paper) << 4);
    fixAttr(Attr);
END newPaper;

PROCEDURE setFillInk (ink:colortype); (* 0..15 *)
BEGIN
    newInk (ink,fillAttr,fillInk);
END setFillInk;

PROCEDURE setFillPaper (paper:colortype); (* normally, 0..7 *)
BEGIN
    newPaper(paper,fillAttr,fillPaper);
END setFillPaper;

PROCEDURE setFillInkPaper (ink,paper:colortype);
BEGIN
    setFillInk(ink);
    setFillPaper(paper);
END setFillInkPaper;

PROCEDURE setTxtInk (ink:colortype); (* 0..15 *)
BEGIN
    newInk(ink,txtAttr,txtInk);
END setTxtInk;

PROCEDURE setTxtPaper (paper:colortype); (* normally, 0..7 *)
BEGIN
    newPaper(paper,txtAttr,txtPaper);
END setTxtPaper;

PROCEDURE setTxtInkPaper (ink,paper:colortype);
BEGIN
    setTxtInk(ink);
    setTxtPaper(paper);
END setTxtInkPaper;

PROCEDURE setWrapMode (state:BOOLEAN  );
BEGIN
    wrapMode := state;
END setWrapMode;

PROCEDURE setUseBiosMode (state:BOOLEAN   );
BEGIN
    useBiosMode := state;
END setUseBiosMode;

PROCEDURE setTabWidth (tab:CARDINAL);
BEGIN
    tabWidth := tab; (* no check is done ! *)
END setTabWidth;

(* ------------------------------------------------------------ *)

PROCEDURE getScreenData ();
BEGIN
    screenWidth  := biosColumnsOnScreen;
    screenHeight := CARDINAL(biosRowsOnScreen) +1;

    IF isMonoMode() THEN
        segScreen := segMono;
    ELSE
        segScreen := segColor;
    END;
    ofsScreen := biosPageStart;
    screen    := [segScreen:ofsScreen];

    pageAtStartup    := CARDINAL(biosCurrentPage);
    activePage       := pageAtStartup;
    visualPage       := pageAtStartup;

    IF NOT (oldModeSaved) THEN
        oldModeSaved   := TRUE;
        oldWidth       := screenWidth;
        oldHeight      := screenHeight;
        oldMode        := biosCurrentVideoMode;
        oldPage        := biosCurrentPage;
        oldCursorShape := biosCursorType;
    END;
END getScreenData;

PROCEDURE setWindow (left,top,width,height:CARDINAL);
VAR
    v : CARDINAL;
BEGIN
    wLeft  := (left MOD screenWidth);
    wTop   := (top  MOD screenHeight);

    wRight := wLeft + width;
    IF wRight > screenWidth THEN wRight := screenWidth; END;
    DEC(wRight);

    wBottom := wTop + height;
    IF wBottom > screenHeight THEN wBottom := screenHeight; END;
    DEC(wBottom);

    wWidth  := (1 + wRight ) - wLeft;
    wHeight := (1 + wBottom) - wTop;

    maxhtab := wWidth  -1;
    maxvtab := wHeight -1;
    (* remember cursor positioning will be relative, not absolute, to defined window *)
    FOR v := minvtab TO maxvtab DO
        basevtab[v] :=  (wTop+v)*screenWidth + wLeft ;
    END;
END setWindow;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE loadROM8x8 (  );
BEGIN
    (* load rom 8x8 dbl-dot patterns *)
    R.AH := 11H;
    R.AL := 12H;
    R.BL := 00H;        (* load block 0 *)
    Lib.Intr(R,vi);
END loadROM8x8;

PROCEDURE selectVerticalResolution (lines:CARDINAL);
VAR
    code : BYTE;
BEGIN
    CASE lines OF
    | 200 : code := 00H;
    | 350 : code := 01H;
    | 400 : code := 02H;
    ELSE
        RETURN; (* do nothing *)
    END;
    (* select vertical resolution vga *)
    R.AH := 12H;
    R.BL := 30H;
    R.AL := code;
    Lib.Intr(R,vi);     (* al=$12 if function supported : vga only *)
END selectVerticalResolution;

PROCEDURE setVideoMode (mode:CARDINAL);
BEGIN
    (* set video mode *)
    R.AH := 00H;
    R.AL := BYTE(mode);
    Lib.Intr(R,vi);
END setVideoMode;

PROCEDURE set25LineMode ():BOOLEAN;
BEGIN
    selectVerticalResolution(400);
    setVideoMode(CO80);
    RETURN TRUE;
END set25LineMode;

PROCEDURE set43LineMode (  ) : BOOLEAN;
BEGIN
    IF isMonoMode() THEN RETURN FALSE; END;
    IF NOT(egavgahere) THEN RETURN FALSE; END;
    selectVerticalResolution(350);
    setVideoMode (CO80);
    loadROM8x8;
    RETURN TRUE;
END set43LineMode;

PROCEDURE set50LineMode (  ) : BOOLEAN;
BEGIN
    IF isMonoMode() THEN RETURN FALSE; END;
    IF NOT(vgahere) THEN RETURN FALSE; END;
    selectVerticalResolution (400);
    setVideoMode (CO80);
    loadROM8x8;
    RETURN TRUE;
END set50LineMode;

PROCEDURE setMode (mode : CARDINAL ) : BOOLEAN;
VAR
    ok : BOOLEAN;
BEGIN
    CASE mode OF
    | BW40,CO40,BW80 :
        setVideoMode(mode);
        ok := TRUE; (* assume ok *)
    | MONO :
        IF isMonoMode() THEN
            setVideoMode(mode);
            ok := TRUE; (* assume ok *)
        ELSE
            ok:=FALSE; (* matrox bios does NOT like monochrome mode setting ! *)
        END;
    | CO80 :
        ok:=set25LineMode();
    | CO80x43 :
        ok:=set43LineMode();
    | CO80x50 :
        ok:=set50LineMode();
    | vesa80x60,vesa132x25,vesa132x43,vesa132x50,vesa132x60 :
        IF vesahere THEN
            ok:=isVesaModeAvailable(vesaListPtr,mode);
            IF ok THEN
                ok:=set25LineMode(); (* safety *)
                ok:=setVesaMode(mode);
            END;
        ELSE
            ok:=FALSE;
        END;
    ELSE
        setVideoMode(mode);
        ok:=TRUE; (* assume ok *)
    END;
    (* we could reset co80 if not ok *)
    setBrightPaper(brightPaper); (* // MUST be reset after a mode change ! *)
    setCursorShape(cursorShape); (* // MUST be reset after a mode change ! *)
    RETURN ok;
END setMode;

PROCEDURE biosSelectActiveDisplayPage (page:BYTE);
BEGIN
    R.AH := 05; (* select active display page *)
    R.AL := page;
    Lib.Intr(R,vi);
END biosSelectActiveDisplayPage;

PROCEDURE restoreMode ():BOOLEAN;
VAR
    ok : BOOLEAN;
    mode,vmode : CARDINAL;
BEGIN
    IF NOT (oldModeSaved) THEN RETURN FALSE; END;
    mode := CARDINAL (oldMode);
    ok := set25LineMode(); (* just in CASE *)
    CASE mode OF
    | BW40,CO40,BW80 :
        setVideoMode( mode );
        ok := TRUE; (* assume ok *)
    | MONO :
        IF isMonoMode() THEN
            setVideoMode( mode );
            ok := TRUE; (* assume ok *)
        ELSE
            ok:=FALSE; (* matrox bios does NOT like monochrome mode setting ! *)
        END;
    | CO80 :
        CASE oldHeight OF
        | 25 :
            ; (* done *)
        | 43 :
            ok:=set43LineMode();
        | 50 :
            ok:=set50LineMode();
        ELSE
            setVideoMode( mode );
            ok := TRUE; (* assume ok *)
        END;
    ELSE
        IF possibleVesaMode(oldWidth,oldHeight,vmode) THEN
            IF vesahere THEN
                ok:=isVesaModeAvailable(vesaListPtr,vmode);
                IF ok THEN
                    ok:=setVesaMode(vmode);
                END;
            ELSE
                setVideoMode( mode );
                ok := TRUE; (* assume ok *)
            END;
        ELSE
            setVideoMode( mode );
            ok := TRUE; (* assume ok *)
        END;
    END;
    IF ok THEN biosSelectActiveDisplayPage(oldPage); END;
    setCursorShape(oldcursor);
    RETURN ok;
END restoreMode;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE gotoXY (x,y:CARDINAL );
VAR
    R : SYSTEM.Registers;
BEGIN
    htab    := (x MOD wWidth);
    vtab    := (y MOD wHeight);
    cursorX := wLeft + htab;
    cursorY := wTop  + vtab;
    R.AH := 02H; (* set cursor position *)
    R.BH := BYTE(activePage);
    R.DL := BYTE(cursorX);
    R.DH := BYTE(cursorY);
    Lib.Intr(R,vi);
END gotoXY;

(*
    should be used only if window exactly maps full screen
    anyway, if cursor was outside window, it will be repositioned
    within window boundaries, preserving x and y as far as possible
*)
PROCEDURE xyToHtabVtab();
VAR
    reposition:CARDINAL;
BEGIN
    (* current absolute cursor screen position *)
    cursorX := CARDINAL (biosCursorPosition[activePage].xcursor);
    cursorY := CARDINAL (biosCursorPosition[activePage].ycursor);
    reposition:=0;
    IF cursorX <= wLeft THEN
        htab := minhtab;
        INC(reposition);
    ELSIF cursorX > wRight THEN
        htab := maxhtab;
        INC(reposition);
    ELSE
        htab := cursorX - wLeft;
    END;
    IF cursorY <= wTop THEN
        vtab := minvtab;
        INC(reposition);
    ELSIF cursorY > wBottom THEN
        vtab := maxvtab;
        INC(reposition);
    ELSE
        vtab := cursorY - wTop;
    END;
    IF reposition > 0 THEN gotoXY(htab,vtab); END;
END xyToHtabVtab;

PROCEDURE home ();
BEGIN
    gotoXY(minhtab,minvtab);
END home;

(* ------------------------------------------------------------ *)

PROCEDURE setVisualPage (page : CARDINAL );
BEGIN
    IF isMonoMode() THEN RETURN; END;
    visualPage := page;
    biosSelectActiveDisplayPage( BYTE(visualPage) );
END setVisualPage;

PROCEDURE setActivePage (page : CARDINAL);
VAR
    offset : CARDINAL;
BEGIN
    IF isMonoMode() THEN RETURN; END;
    activePage := page;
    biosSelectActiveDisplayPage( BYTE(activePage) );

    (* ripped from getScreenData() *)
    ofsScreen := biosPageStart;
    screen    := [segScreen:ofsScreen];

    biosSelectActiveDisplayPage( BYTE(visualPage) );
    xyToHtabVtab();
END setActivePage;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE scrollWindow (motion:scrolltype);
VAR
    x,y,base,baseto:CARDINAL;
BEGIN
    CASE motion OF
    | moveup:
        FOR y := (minvtab+1) TO maxvtab DO
            baseto := basevtab[y-1];
            base   := basevtab[y];
            FOR x := minhtab TO maxhtab DO
                screen^[baseto+x] := screen^[base+x];
            END;
        END;
        base := basevtab[maxvtab];
        FOR x := minhtab TO maxhtab DO
            screen^[base+x].ch   := fillChar;
            screen^[base+x].attr := fillAttr;
        END;
    | movedown:
        FOR y := (maxvtab-1) TO minvtab BY -1 DO
            baseto := basevtab[y+1];
            base   := basevtab[y];
            FOR x := minhtab TO maxhtab DO
                screen^[baseto+x] := screen^[base+x];
            END;
        END;
        base := basevtab[minvtab];
        FOR x := minhtab TO maxhtab DO
            screen^[base+x].ch   := fillChar;
            screen^[base+x].attr := fillAttr;
        END;
    | moveleft:
        FOR x := (minhtab+1) TO maxhtab DO
            FOR y := minvtab TO maxvtab DO
                base   := basevtab[y];
                screen^[base+x-1] := screen^[base+x];
            END;
        END;
        FOR y := minvtab TO maxvtab DO
            base := basevtab[y];
            screen^[base+maxhtab].ch   := fillChar;
            screen^[base+maxhtab].attr := fillAttr;
        END;
    | moveright:
        FOR x := (maxhtab-1) TO minhtab BY -1 DO
            FOR y := minvtab TO maxvtab DO
                base   := basevtab[y];
                screen^[base+x+1] := screen^[base+x];
            END;
        END;
        FOR y := minvtab TO maxvtab DO
            base := basevtab[y];
            screen^[base+minhtab].ch   := fillChar;
            screen^[base+minhtab].attr := fillAttr;
        END;

    END;
END scrollWindow;

PROCEDURE fillTextAttr (x1,y1,x2,y2:CARDINAL;ch:CHAR;attr:SHORTCARD);
VAR
    x,y,base:CARDINAL;
BEGIN
    FOR y := y1 TO y2 DO
        base := basevtab[y];
        FOR x := x1 TO x2 DO
            screen^[base+x].ch  :=ch;
            screen^[base+x].attr:=attr;
        END;
    END;
END fillTextAttr;

PROCEDURE cls ();
BEGIN
    fillTextAttr(minhtab,minvtab,maxhtab,maxvtab,fillChar,fillAttr);
    home;
END cls;

PROCEDURE writeStr (S:ARRAY OF CHAR);
VAR
    len,i,base:CARDINAL;
    ch:CHAR;
    n:CARDINAL;
BEGIN
    IF useBiosMode THEN WrStr(S); RETURN; END;
    len := Str.Length(S);
    IF len=0 THEN RETURN; END;
    FOR i := 0 TO len-1 DO
        ch:=S[i];
        CASE ch OF
        | cr  :
            gotoXY( minhtab,vtab );
        | lf  :
            INC(vtab);
            IF vtab > maxvtab THEN
                scrollWindow(moveup);
                vtab:=maxvtab;
            END;
            gotoXY(htab,vtab);
        | ff  :
            cls;
        | bs  :
            IF htab > minhtab THEN
                DEC(htab);
            ELSE
                IF vtab > minvtab THEN
                    DEC(vtab);
                    htab := maxhtab;
                ELSE
                    (* cannot move here, for we're already at top of screen *)
                END;
            END;
            gotoXY(htab,vtab);
        | tab :
            n := ( ( htab DIV tabWidth ) + 1 ) * tabWidth; (* this is next tab *)
            DEC(n,htab); (* compute number of spaces needed to go to next tab *)
            WHILE (n > 0) DO
                base := basevtab[vtab];
                screen^[base+htab].ch   := fillChar; (* could be space and txtAttr ? *)
                screen^[base+htab].attr := fillAttr;
                INC(htab);
                IF htab > maxhtab THEN
                    htab := minhtab;
                    IF wrapMode THEN
                        INC(vtab);
                        IF vtab > maxvtab THEN
                            scrollWindow(moveup);
                            vtab:=maxvtab;
                        END;
                    END;
                    n := tabWidth; (* handle new tab after crlf *)
                    INC(n); (* handle the fact we're about to DEC n ! *)
                END;
                DEC (n);
            END;
        ELSE
            base := basevtab[vtab];
            screen^[base+htab].ch   := ch;
            screen^[base+htab].attr := txtAttr;
            INC(htab);
            IF htab > maxhtab THEN
                htab := minhtab;
                IF wrapMode THEN
                    INC(vtab);
                    IF vtab > maxvtab THEN
                        scrollWindow(moveup);
                        vtab:=maxvtab;
                    END;
                END;
            END;
        END;

    END;
    gotoXY(htab,vtab);
END writeStr;

PROCEDURE writeLn ();
BEGIN
    writeStr(cr+lf);
END writeLn;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE getScreenWidth (  ):CARDINAL ;
BEGIN
    RETURN screenWidth;
END getScreenWidth;

PROCEDURE getScreenHeight (  ):CARDINAL ;
BEGIN
    RETURN screenHeight;
END getScreenHeight;

PROCEDURE getScreenMinX (  ):CARDINAL ;
BEGIN
    RETURN minx;
END getScreenMinX;

PROCEDURE getScreenMinY (  ):CARDINAL ;
BEGIN
    RETURN miny;
END getScreenMinY;

PROCEDURE getScreenMaxX (  ):CARDINAL ;
VAR
    v : CARDINAL;
BEGIN
    v := screenWidth;
    IF v > 0 THEN DEC(v); END;
    RETURN v;
END getScreenMaxX;

PROCEDURE getScreenMaxY (  ):CARDINAL ;
VAR
    v : CARDINAL;
BEGIN
    v := screenHeight;
    IF v > 0 THEN DEC(v); END;
    RETURN v;
END getScreenMaxY;

PROCEDURE getMinHtab (  ):CARDINAL ;
BEGIN
    RETURN minhtab;
END getMinHtab;

PROCEDURE getMaxHtab (  ):CARDINAL ;
BEGIN
    RETURN maxhtab;
END getMaxHtab;

PROCEDURE getMinVtab (  ):CARDINAL ;
BEGIN
    RETURN minvtab;
END getMinVtab;

PROCEDURE getMaxVtab (  ):CARDINAL ;
BEGIN
    RETURN maxvtab;
END getMaxVtab;

PROCEDURE getUseBiosMode (  ):BOOLEAN;
BEGIN
    RETURN useBiosMode;
END getUseBiosMode;

PROCEDURE getHtab (  ):CARDINAL ;
BEGIN
    RETURN htab;
END getHtab;

PROCEDURE getVtab (  ):CARDINAL ;
BEGIN
    RETURN vtab;
END getVtab;

PROCEDURE getWindowWidth (  ):CARDINAL ;
BEGIN
    RETURN wWidth;
END getWindowWidth;

PROCEDURE getWindowHeight (  ):CARDINAL ;
BEGIN
    RETURN wHeight;
END getWindowHeight;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE initScreenConsole ();
BEGIN
    initFlags(); (* make no assumption : no ega, no vga, no vesa, no blinking *)
    getVideoCard (defaultDetectVesaToo); (* default is ignore vesa modes set/reset *)

    setBrightPaper(defaultBrightPaper);
    setBlinkMode(defaultBlinkMode);

    setFillChar(defaultFillChar);
    setFillInk(defaultFillInk);
    setFillPaper(defaultFillPaper);
    setTxtInk(defaultTxtInk);
    setTxtPaper(defaultTxtPaper);

    (* just in case we would not call initInkPaperAtStartup() ! *)
    DOSinkAtStartup   := defaultTxtInk;
    DOSpaperAtStartup := defaultTxtPaper;

    setWrapMode(defaultWrapMode);
    setUseBiosMode(defaultUseBiosMode);
    setTabWidth(defaultTabWidth);
    (* //
    selectCursorEmulation(defaultCursorEmulation);
    setCursorShape(defaultCursorShape);
    *)

    getScreenData();
    setWindow(minx,miny,screenWidth,screenHeight);
    xyToHtabVtab();
END initScreenConsole;

(* ------------------------------------------------------------ *)

(* do not import IsRedirected() FROM QD_Box ! *)

PROCEDURE DOSredirected() : BOOLEAN;
VAR
    flag : CARDINAL; (* wants bitwise M2 operator ON, here *)
BEGIN
    R.AX := 4400H;   (* get ioctl info *)
    R.BX := 01H;     (* FIO.StandardOutput is always 1 *)
    Lib.Dos(R);
    flag := R.DX;
    IF (flag AND 80H ) = 80H THEN (* bit 7 set = periph, else file *)
       RETURN FALSE;
    ELSE
       RETURN TRUE;
    END;
END DOSredirected;

(* should be called AFTER initScreenConsole and setBiosMode *)

PROCEDURE findInkPaperAtStartup();
CONST
    blank     = BYTE(' '); (* space character *)
    backspace = BYTE(8);   (* backspace *)
VAR
    DOSattrAtStartup : SHORTCARD;
BEGIN
    IF useBiosMode THEN RETURN; END;      (* if setBiosMode TRUE *)
    IF DOSredirected() THEN RETURN; END;  (* just in case... *)

    (* space with DOS *)

    R.DL := blank;
    R.AH := 02H; (* write character to standard output : alternate would be 06H direct console output *)
    Lib.Dos(R);

    (* backspace with BIOS *)

    R.BH := BYTE(pageAtStartup); (* // *)
    R.AL := backspace;
    R.AH := 0EH; (* video teletype output *)
    Lib.Intr(R,vi);

    R.BH := BYTE(pageAtStartup); (* // *)
    R.AH := 08H; (* video read character and attribute at cursor position *)
    Lib.Intr(R,vi);
    DOSattrAtStartup := R.AH; (* AL should be the space we had sent... *)

    (*
       Bitfields for character's display attribute:
       7	foreground blink or (alternate) background bright (see also AX=1003h)
       6-4	background color (see #00015)
       3	foreground bright or (alternate) alternate character set (see AX=1103h)
       2-0	foreground color (see #00015)
       Values for character color:
	   Normal		    Bright
       000b	black		dark gray
       001b	blue		light blue
       010b	green		light green
       011b	cyan		light cyan
       100b	red		    light red
       101b	magenta		light magenta
       110b	brown		yellow
       111b	light gray	white
    *)
    DOSinkAtStartup   := colortype( DOSattrAtStartup AND 00FH)    ; (* 00001111b *)
    DOSpaperAtStartup := colortype((DOSattrAtStartup AND 0F0H)>>4); (* 11110000b *)
END findInkPaperAtStartup;

PROCEDURE getInkAtStartup():colortype;
BEGIN
    RETURN DOSinkAtStartup;
END getInkAtStartup;

PROCEDURE getPaperAtStartup():colortype;
BEGIN
    RETURN DOSpaperAtStartup;
END getPaperAtStartup;

PROCEDURE setFullScreenWindow ();
BEGIN
    (*
    setWindow( getScreenMinX() , getScreenMinY() ,
               getScreenWidth() , getScreenHeight() );
    *)
    setWindow(minx,miny,screenWidth,screenHeight);
END setFullScreenWindow;

(* use with caution ! *)

PROCEDURE gotoFullScreenXY (x,y:CARDINAL );
VAR
    R : SYSTEM.Registers;
BEGIN
    htab    := (x MOD screenWidth);
    vtab    := (y MOD screenHeight);
    cursorX := minx + htab;
    cursorY := miny + vtab;
    R.AH := 02H; (* set cursor position *)
    R.BH := BYTE(activePage);
    R.DL := BYTE(cursorX);
    R.DH := BYTE(cursorY);
    Lib.Intr(R,vi);
END gotoFullScreenXY;

(* ------------------------------------------------------------ *)

BEGIN
    initScreenConsole();
END QD_Text.
