(* ---------------------------------------------------------------
Title         Q&D Calendar
Author        PhG
Overview      see help
Notes
Bugs
Wish List     add ini file with special dates ?
              (first weekday, fixed date, etc.)
              calendar by day (weird idea seen in yearcal.c by a Paul Sittler)
              other calendars ?
              keyboard pause (bah, redirection was not created for nothing)

--------------------------------------------------------------- *)

MODULE Cal;

IMPORT Lib;
IMPORT Str;
IMPORT MATHLIB;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite, setReadOnly,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs, UpperCaseAlt, LowerCaseAlt,
completedInit, completedShow, completedSHOW, completedEnd, completed,
removeDups, isValidHDunit, removePhantoms, removeFloppies,
getCDROMunits, getCDROMletters, removeCDROMs, getAllHDunits,
getAllLegalUnits;

FROM QD_Text IMPORT
colortype, cursorshapetype, scrolltype,
ff, cr, lf, bs, tab, nl, mincolor, maxcolor,
BW40, CO40, BW80, CO80, CO80x43, CO80x50, MONO,
vesa80x60, vesa132x25, vesa132x43, vesa132x50, vesa132x60,
selectCursorEmulation,
setCursorShape,
handleVesa, setBrightPaper, setBlinkMode,
setFillChar, setFillInk, setFillPaper,
setFillInkPaper, setTxtInk, setTxtPaper, setTxtInkPaper, setWrapMode,
setUseBiosMode, setTabWidth, getScreenData, setWindow,
setMode, restoreMode,
gotoXY, xyToHtabVtab, home, setVisualPage, setActivePage,
scrollWindow, fillTextAttr, cls, writeStr, writeLn, getScreenWidth,
getScreenHeight, getScreenMinX, getScreenMinY, getScreenMaxX, getScreenMaxY,
getMinHtab, getMaxHtab, getMinVtab, getMaxVtab, getUseBiosMode,
getMinHtab, getMaxHtab, getMinVtab, getMaxVtab, getHtab, getVtab,
getWindowWidth, getWindowHeight,
initScreenConsole,
findInkPaperAtStartup, getInkAtStartup, getPaperAtStartup,
setFullScreenWindow, gotoFullScreenXY;

(* ------------------------------------------------------------ *)

CONST
    (* aliases for our own procedures *)
    WrStr  ::= writeStr;
    WrLn   ::= writeLn;
    WrChar ::= writeStr;

PROCEDURE colorhelp ();
BEGIN
    setTxtInk(getInkAtStartup());      (* was green *)
    setTxtPaper(getPaperAtStartup());  (* was black *)
    setFillInkPaper(getInkAtStartup(),getPaperAtStartup());
END colorhelp;

PROCEDURE color (ink,paper:colortype);
BEGIN
    setTxtInk   (ink);
    setTxtPaper (paper);
    setFillInkPaper(getInkAtStartup(),getPaperAtStartup());
END color;

(* ------------------------------------------------------------ *)

CONST
    extEXE        = ".EXE";
    extINI        = ".INI";
    cHilightleft  = ">";
    cHilightright = "<";
CONST
    progEXEname   = "CAL";
    progTitle     = "Q&D Calendar";
    progVersion   = "v1.0d";
    progCopyright = "by PhG";
    Banner        = progTitle+" "+progVersion+" "+progCopyright;
CONST
    errNone         = 0;
    errHelp         = 1;
    errUnknownOpt   = 2;
    errTooManyParms = 3;
    errNumber       = 4;
    errInk          = 5;
    errPaper        = 6;
    errYear         = 7;
    errMonth        = 8;
    errBadToday     = 9;
    errJulian       = 10;

(* ------------------------------------------------------------ *)

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
    helpmsg =
Banner+nl+
nl+
"Syntax : "+progEXEname+" [ <month> <year|*> | <year|*> ] [option]..."+nl+
nl+
"This program prints a monthly or yearly calendar."+nl+
nl+
"  -s      Sunday is first day of week (default is Monday)"+nl+
"  -u[k|s] English texts (default is French texts)"+nl+
"  -x|-p   special calendar"+nl+
"  -j[j]   show last four digits of Julian Day (-jj = show full JD in header)"+nl+
"  -n      show day in year (base is January 1st)"+nl+
"  -t:??   left and right markers to highlight today in monthly calendar"+nl+
'          (default is "'+cHilightleft+cHilightright+'")'+nl+
"  -q      do not highlight today"+nl+
"  -i?:#   ink (im=main, it=title, id=days, ic=current day, ij=julian)"+nl+
"  -p?:#   paper (pm=main, pt=title, pd=days, pc=current day, pj=julian)"+nl+
"  -b      monochrome BIOS output (no colors)"+nl+
nl+
"a) Month is [1..12], full month name or first three letters of month."+nl+
"b) Calendar is assumed julian for dates before October 15, 1582 :"+nl+
"   note this was not true in the whole Western world"+nl+
"   (a few countries did not adopt gregorian calendar until September, 1752 !)."+nl+
"c) Yearly calendars with any of -n or -j[j] options should be redirected."+nl+
"d) Julian Day begins at 12:00:00."+nl+
'e) "*" stands for current year.'+nl+
nl+
"Examples : "+progEXEname+nl+
"           "+progEXEname+" *"+nl+
"           "+progEXEname+" 1987"+nl+
"           "+progEXEname+" fvrier 1963"+nl;

VAR
    S : str256;
BEGIN
    colorhelp;
    CASE e OF
    | errHelp :
        WrStr(helpmsg);
    | errUnknownOpt :
        Str.Concat(S,"Unknown ",einfo); Str.Append(S," option !");
    | errTooManyParms :
        Str.Concat(S,"Unexpected ",einfo);Str.Append(S," parameter !");
    | errNumber:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," number !");
    | errInk:
        S := "Ink range is [0..15] !";
    | errPaper:
        S := "Paper range is [0..15] !";
    | errYear:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," year !");
    | errMonth:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," month !");
    | errBadToday:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," markers string !");
    | errJulian:
        S := "-j[j] and -n options are mutually exclusive !";
    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE  e OF
    | errNone, errHelp : ;
    ELSE
        WrStr(progEXEname+" : "); WrStr(S); WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

(* ripped/adapted from frozen QD_SKY *)

(* ---------------------------------------------------------------
   input  : value
   output : int(value) ie int(0.95) gives 0 (no rounding)

   TRUNC(n) gives only a CARDINAL without rounding either
--------------------------------------------------------------- *)

PROCEDURE int (v : LONGREAL) : LONGREAL;
VAR
    tmp : LONGINT; (* this should be enough! *)
BEGIN
    tmp := VAL(LONGINT,v); (* as intended, does NOT perform any rounding *)
    RETURN VAL(LONGREAL,tmp);
END int;

(* ---------------------------------------------------------------
   input  : value
   output : frac(value)

   frac keeps sign
--------------------------------------------------------------- *)

PROCEDURE frac (v : LONGREAL) : LONGREAL;
BEGIN
    RETURN (v - int (v));
END frac;

(* ---------------------------------------------------------------
   input  : value1, value2
   output : value1 mod value2

   mod keeps sign
--------------------------------------------------------------- *)

PROCEDURE mod (v1, v2 : LONGREAL) : LONGREAL;
BEGIN
    RETURN ( MATHLIB.Mod(v1, v2) );
END mod;

(* ---------------------------------------------------------------
   assume day with decimal, and whole date is ok
   valid only for JD >= 0 i.e. NOT before 1.5 Jan -4712 !
   input  : first gregorian date (yyyymmdd), day (with decimals), month, year
   output : JD
   thanks to Meeus !
--------------------------------------------------------------- *)

PROCEDURE dateToJD (Day,Month,Year:LONGREAL) : LONGREAL;
VAR
    YYYYMMDD  : LONGREAL;
    A,gregorian : LONGREAL; (* gregorian correction *)
    JD        : LONGREAL;
BEGIN
    YYYYMMDD := Year * 10000.0 + Month * 100.0 + Day;
    IF Month < 2.5 THEN
        Year  := Year - 1.0;
        Month := Month + 12.0;
    END;
    IF YYYYMMDD < LONGREAL(15821015.0) THEN (* before 15 october 1582, julian calendar *)
        gregorian := 0.0;
    ELSE
        A         := int (Year / 100.0);
        gregorian := 2.0 - A + int (A / 4.0);
    END;
    JD := int (365.25 * (Year + 4716.0)) + int (30.6001 * (Month + 1.0));
    JD := JD + Day + gregorian - 1524.5;
    RETURN JD;
END dateToJD;

(* ---------------------------------------------------------------
   assume JD >= 0
   input  : JD
   output : day (with decimals), month, year
--------------------------------------------------------------- *)

PROCEDURE JDtoDate (JD : LONGREAL; VAR Day,Month,Year : LONGREAL);
VAR
    Z,F,A,alf,B,C,D,E:LONGREAL;
BEGIN
    JD := JD + 0.5;
    Z := int(JD);
    F := frac(JD);
    IF Z < 2299161.0 THEN
        A := Z;
    ELSE
        alf := int( (Z - 1867216.25) / 36524.25);
        A := Z + 1.0 + alf - int( alf / 4.0 );
    END;
    B := A + 1524.0;
    C := int ( (B - 122.1) / 365.25);
    D := int ( 365.25 * C );
    E := int ( (B - D) / 30.6001 );
    Day := B - D - int( 30.6001 * E) + F;
    IF E < 13.5 THEN
        Month := E - 1.0;
    ELSE
        Month := E - 13.0;
    END;
    IF Month > 2.5 THEN
        Year := C - 4716.0;
    ELSE
        Year := C - 4715.0;
    END;
END JDtoDate;

(* ---------------------------------------------------------------
   find JD at 0h
   input  : JD
   output : JD for this day at 0H
--------------------------------------------------------------- *)

PROCEDURE JDtoJD0H (JD : LONGREAL) : LONGREAL;
VAR
    JD0H : LONGREAL;
BEGIN
    IF frac(JD) < 0.5 THEN
       JD0H := int(JD) - 0.5; (* was < 23H59 so take previous 0H *)
    ELSE
       JD0H := int(JD) + 0.5; (* was 0H or later so take current 0H *)
    END;
    RETURN JD0H;
END JDtoJD0H;

(* ---------------------------------------------------------------
   0=sunday, 1=monday... 6=saturday
   input  : JD
   output : day of week
--------------------------------------------------------------- *)

CONST
    firstdayofweek = 0; (* dimanche i.e. sunday *)
    lastdayofweek  = 6; (* samedi i.e. saturday *)

PROCEDURE JDtoWeekday (JD : LONGREAL) : CARDINAL;
VAR
    v : LONGREAL;
BEGIN
    v := mod ( (JDtoJD0H (JD) + 1.5), 7.0 );
    RETURN ( VAL(CARDINAL, v ) );
END JDtoWeekday;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

(* assume n is [1.. *)

PROCEDURE getElem (n:CARDINAL;delimiters,elements:ARRAY OF CHAR):str80;
VAR
    R:str80;
BEGIN
    Str.ItemS(R, elements, delimiters,n-1);
    RETURN R;
END getElem;

PROCEDURE inrange (n,mini,maxi:CARDINAL):BOOLEAN;
BEGIN
    IF n < mini THEN RETURN FALSE; END;
    IF n > maxi THEN RETURN FALSE; END;
    RETURN TRUE;
END inrange;

PROCEDURE fmtbignum (v:LONGCARD; base:CARDINAL;wi:INTEGER; pad:CHAR ):str16;
VAR
    S:str16;
    ok:BOOLEAN;
    i : CARDINAL;
BEGIN
    Str.CardToStr( v, S, base,ok);
    IF base=16 THEN Str.Lows(S);END;
    FOR i:=Str.Length(S)+1 TO ABS(wi) DO
         IF wi < 0 THEN
             Str.Append(S,pad);
         ELSE
             Str.Prepend(S,pad);
         END;
    END;
    RETURN S;
END fmtbignum;

PROCEDURE fmtnum ( v:CARDINAL; base:CARDINAL;wi:INTEGER; pad:CHAR ):str16;
BEGIN
    RETURN fmtbignum( LONGCARD(v),base,wi,pad);
END fmtnum;

PROCEDURE str2card (VAR n:CARDINAL;S:ARRAY OF CHAR):BOOLEAN;
VAR
    lc:LONGCARD;
    ok:BOOLEAN;
    base:CARDINAL;
BEGIN
    IF Str.Match(S,"$*") THEN
        base:=16;
        Str.Delete(S,0,1);
    ELSIF Str.Match(S,"0X*") THEN
        base:=16;
        Str.Delete(S,0,2);
    ELSE
        base:=10;
    END;
    lc:=Str.StrToCard(S,base,ok);
    IF NOT(ok) THEN RETURN FALSE; END;
    IF lc > MAX(CARDINAL) THEN RETURN FALSE; END;
    n:=CARDINAL(lc);
    RETURN TRUE;
END str2card;

(*
    gregorian calendar rule : leap 366 years if divisible by 4
    but centurial years NOT divisible by 4 are common
    Leap years are divisible by 400, or by 4 and not by 100
    1900 is common, 2000 is leap
*)

PROCEDURE getDaysInFebruary (annee:CARDINAL):CARDINAL;
CONST
    common = 28;
    leap   = 29;
BEGIN
    IF (annee MOD 400) = 0 THEN RETURN leap; END;
    IF (((annee MOD 4) = 0) AND ((annee MOD 100) # 0)) THEN RETURN leap; END;
    RETURN common;
END getDaysInFebruary;

(* assume month and year are correct *)

PROCEDURE getDaysInMonth (mois,annee:CARDINAL):CARDINAL;
TYPE
    daysinmonthtype = ARRAY [1..12] OF CARDINAL;
CONST
    (*                    JanFevMarAprMayJunJulAugSepOctNovDec *)
    (*                     1  2  3  4  5  6  7  8  9 10 11 12  *)
    dpm = daysinmonthtype(31, 0,31,30,31,30,31,31,30,31,30,31);
VAR
    m : CARDINAL;
BEGIN
    m:= dpm[mois];
    IF m=0 THEN m:=getDaysInFebruary(annee);END;
    RETURN m;
END getDaysInMonth;

(* ------------------------------------------------------------ *)

TYPE
    languagetype = (french,english);
    daytype      = (sunday,monday);
CONST
    space        = " ";
    abrevjours   = "Di Lu Ma Me Je Ve Sa";
    abrevjoursUS = "Su Mo Tu We Th Fr Sa";
    abrevmois    = "jan fv mar avr mai jui jul ao sep oct nov dc";
    abrevmoisUS  = "jan feb mar apr may jun jul aug sep oct nov dec";
    mois         = "Janvier Fvrier Mars Avril Mai Juin "+
                   "Juillet Aot Septembre Octobre Novembre Dcembre";
    moisUS       = "January February March April May June "+
                   "July August September October November December";
CONST
    firstmonth   = 1;
    lastmonth    = 12;
    firstweekday = 1;
    lastweekday  = 7;

(* ------------------------------------------------------------ *)

CONST
    safety    = 2;                   (* paranoiac ! *)
    minRow    = 1;                   (* must be 1 *)
    maxRow    = (1+1+6+1)+safety;
    minCol    = 1;                   (* must be 1 *)
    maxCol    = 3*(1+7*3+1)+safety;  (* "## " 2+1 *)
    (* maxCol    = 1+7*10+1+safety;     (* "[####] ## " 6+1+3 *) *)
VAR
    hline     : ARRAY [1..3],[minRow..maxRow] OF str80; (* will do for year (1+7*3+1) * 3 *)
    inkattr   : ARRAY [1..3],[minRow..maxRow],[minCol..maxCol] OF colortype;
    paperattr : ARRAY [1..3],[minRow..maxRow],[minCol..maxCol] OF colortype;

PROCEDURE storeline (ndx,n:CARDINAL;ink,paper:colortype;S:ARRAY OF CHAR);
VAR
    i:CARDINAL;
BEGIN
    Str.Copy( hline[ndx,n],S);
    FOR i:=1 TO Str.Length(S) DO
        inkattr[ndx,n,i]:=ink;
        paperattr[ndx,n,i]:=paper;
    END;
END storeline;

PROCEDURE storeattr (p,wi,ndx,n:CARDINAL;ink,papercolor:colortype);
VAR
    i:CARDINAL;
BEGIN
    FOR i:= p TO (p+wi-1) DO
        inkattr[ndx,n,i]:=ink;
        paperattr[ndx,n,i]:=papercolor;
    END;
END storeattr;

PROCEDURE storechar (ndx,n,i:CARDINAL;ink,paper:colortype;ch:CHAR);
BEGIN
    hline[ndx,n,i]:=ch;
    INC(i); (* string is 0-based while attr arrays are 1-based *)
    inkattr[ndx,n,i]:=ink;
    paperattr[ndx,n,i]:=paper;
END storechar;

PROCEDURE retrievechar (ndx,n,i:CARDINAL):CHAR;
BEGIN
    RETURN hline[ndx,n,i];
END retrievechar;

PROCEDURE dmpline (ndx,n:CARDINAL);
VAR
    i:CARDINAL;
BEGIN
    FOR i:=1 TO Str.Length( hline[ndx,n] ) DO
        color ( inkattr[ndx,n,i], paperattr[ndx,n,i] );
        WrStr ( hline[ndx,n,i-1]);
    END;
    WrLn;
END dmpline;

PROCEDURE dmp(WHICH,currline:CARDINAL);
VAR
    i:CARDINAL;
BEGIN
    i:=minRow-1;
    LOOP
        INC(i);
        IF i > currline THEN EXIT; END;
        dmpline(WHICH,i);
    END;
    colorhelp; (* restore default just in case *)
END dmp;

PROCEDURE merge (ndst,nsrc,idst,isrc:CARDINAL;ink,paper:colortype):CARDINAL;
VAR
    i,last,nextpos,len:CARDINAL;
    S:str80;
BEGIN
    IF ndst < nsrc THEN
        last:=nsrc;
        FOR i:= ndst+1 TO last DO
            S:="";
            FOR len:=1 TO Str.Length(hline[idst,minRow]) DO Str.Append(S,space);END;
            storeline(idst,i,ink,paper,S);
        END;
    ELSIF nsrc < ndst THEN
        last:=ndst;
        FOR i:= nsrc+1 TO last DO
            S:="";
            FOR len:=1 TO Str.Length(hline[isrc,minRow]) DO Str.Append(S,space);END;
            storeline(isrc,i,ink,paper,S);
        END;
    ELSE
        last:=nsrc;
    END;

    (* now merge columns and attributes *)

    FOR i:=minRow TO last DO
        Str.Append(hline[idst,i],space);

        nextpos:=Str.Length(hline[idst,i]);

        FOR len:=1 TO Str.Length(space) DO
            inkattr  [idst,i,nextpos]:=ink;
            paperattr[idst,i,nextpos]:=paper;
            INC(nextpos);
        END;

        Str.Append(hline[idst,i], hline[isrc,i]);

        FOR len:=1 TO Str.Length(hline[isrc,i]) DO
            inkattr  [idst,i,nextpos]:=inkattr[isrc,i,len];
            paperattr[idst,i,nextpos]:=paperattr[isrc,i,len];
            INC(nextpos);
        END;
    END;
    RETURN last;
END merge;

PROCEDURE centerstr (wi:CARDINAL;pad:CHAR; VAR R:ARRAY OF CHAR);
VAR
    i,len,op:CARDINAL;
BEGIN
    len:=Str.Length(R);
    op:=0;
    FOR i:=len TO wi DO
        IF Str.Length(R) < wi THEN
            IF op=0 THEN
                Str.Append(R,pad);
            ELSE
                Str.Prepend(R,pad);
            END;
            INC(op);
            op:=op MOD 2;
        END;
    END;
END centerstr;

PROCEDURE getfmtparms (programmer:BOOLEAN;
                      VAR base,Ydigits,digits:CARDINAL;VAR padchar:CHAR);
BEGIN
    IF programmer THEN
        base:=16; Ydigits := 4; digits := 2; padchar:="0";
    ELSE
        base:=10; Ydigits := 1; digits := 2; padchar:=" ";
    END;
END getfmtparms;

PROCEDURE getlngparms (language:languagetype; VAR TM,TJ:ARRAY OF CHAR);
BEGIN
    CASE language OF
    | french:  Str.Copy(TM,mois);   Str.Copy(TJ,abrevjours);
    | english: Str.Copy(TM,moisUS); Str.Copy(TJ,abrevjoursUS);
    END;
END getlngparms;

(* too many IFs force an isl342 if we use small model *)

PROCEDURE buildmonth (programmer,julian,showfullJD,numbered:BOOLEAN;
                     WHICH,daynow,monthnow,yearnow, month,year:CARDINAL;
                     language:languagetype;weekstart:daytype;
                     hilightleft,hilightright:CHAR;
                     inkmain,papermain,inktitle,papertitle,
                     inkdays,paperdays,inknow,papernow,
                     inkjulian,paperjulian:colortype):CARDINAL;
CONST
    sepjulian      = " -- "; (* title *)

    wijulian       = 4;
    wipadjulian    = wijulian+2+1;   (* +2=open/close +1=trailing space *)
    padjulian      = " "+"    "+" "+space; (* "(####) " *)

    widiy          = 3;
    wipaddiy       = widiy+2+1;      (* +2=open/close +1=trailing space *)
    paddiy         = " "+"   " +" "+space; (* "(###) " *)

    wiquantieme    = 2;
    wipadquantieme = wiquantieme+1;
    padquantieme   = "  "+space;
    openjulian     = "[";
    closejulian    = "]";
    opennumbered   = "{";
    closenumbered  = "}";
VAR
    TM,TJ : str128; (* str80 would not be enough for months *)
    T,D,S:str80;
    currline,i,count,wical,baseJDofWeek,dow:CARDINAL;
    hereline,herepos: CARDINAL;
    JDYEAR,JD,jj,mm,yy:LONGREAL;
    state:(waiting,inmonth,done);
    base,Ydigits,digits:CARDINAL;
    padchar:CHAR;
    vcurr,currjdbase,currjd:LONGCARD;
    p,ii : CARDINAL;
BEGIN
    getfmtparms(programmer, base,Ydigits,digits,padchar);
    getlngparms(language, TM, TJ );

    (* create days header *)

    CASE weekstart OF
    | sunday: baseJDofWeek:=firstdayofweek;
    | monday: baseJDofWeek:=firstdayofweek+1;
    END;
    i:=baseJDofWeek; (* init first day *)
    Str.Copy(D,space);
    FOR count:=firstweekday TO lastweekday DO
        IF julian THEN
            Str.Append(D,padjulian);
        ELSIF numbered THEN
            Str.Append(D,paddiy);
        END;
        Str.Append(D,getElem(i+1,space,TJ));
        Str.Append(D,space);
        INC(i);
        IF i > lastdayofweek THEN i:=firstdayofweek;END;
    END;
    wical:=Str.Length(D);

    (* first day of month : JD stuff now *)

    JD:=dateToJD(LONGREAL( 1 ),LONGREAL(month),LONGREAL(year));
    currjd    := LONGCARD(JD); INC(currjd);

    (* create title bar *)

    Str.Concat(T,getElem(month,space,TM), " ");
    IF programmer THEN Str.Append(T,"$");END; (* reminder we're in hex mode *)
    Str.Append(T,fmtnum(year,base,Ydigits,padchar));
    IF showfullJD THEN
        Str.Append(T,sepjulian);
        Str.Append(T,fmtbignum( currjd,base,1," "));
    END;
    centerstr(wical,space,T);

    (* store title then days header *)

    currline:=minRow;
    storeline(WHICH,currline,inktitle,papertitle,T);
    INC(currline);
    storeline(WHICH,currline,inkdays,paperdays,D);
    INC(currline);

    (* don't be clever here : nowadays, what's the use for such an obsolete quality ? :-( *)

    JDYEAR:=dateToJD(LONGREAL( 1 ),LONGREAL(1),LONGREAL(year));
    currjdbase:= LONGCARD(JDYEAR); INC(currjdbase); (* poor man's floor() = +0.5 *)

    (* first day of month : JD computed earlier for title *)

    (*
    JD:=dateToJD(LONGREAL( 1 ),LONGREAL(month),LONGREAL(year));
    currjd    := LONGCARD(JD); INC(currjd);
    *)

    dow:=JDtoWeekday ( JD );

    S:=space; (* we'll be able to do len-1 without problem *)
    count:=firstweekday;

    i:=baseJDofWeek;
    state:=waiting;
    LOOP
        CASE state OF
        | waiting:
            IF i=dow THEN
                state:=inmonth;
            ELSE
                IF julian THEN
                    Str.Append(S,padjulian);
                ELSIF numbered THEN
                    Str.Append(S,paddiy);
                END;
                Str.Append(S,padquantieme);
                INC(i);
                IF i > lastdayofweek THEN i:=firstdayofweek;END;
                INC(count);
            END;
        | inmonth:
            JDtoDate(JD,jj,mm,yy);
            IF CARDINAL(mm) = month THEN
                IF CARDINAL(jj)=daynow THEN
                    hereline:=currline;
                    herepos:=Str.Length(S)-1; (* S always at least one space *)
                    IF julian THEN
                        INC(herepos,wipadjulian);
                    ELSIF numbered THEN
                        INC(herepos,wipaddiy);
                    END;
                END;
                IF julian THEN
                    vcurr:=(currjd MOD 10000); DEC(vcurr); (* fix *)
                    Str.Append(S,openjulian);
                    Str.Append(S,fmtbignum(vcurr,base,wijulian,"0"));
                    Str.Append(S,closejulian);
                    Str.Append(S,space);
                ELSIF numbered THEN
                    vcurr:=currjd-currjdbase+1; DEC(vcurr); (* fix *)
                    Str.Append(S,opennumbered);
                    Str.Append(S,fmtbignum(vcurr,base,widiy," "));
                    Str.Append(S,closenumbered);
                    Str.Append(S,space);
                END;
                Str.Append(S,fmtnum( CARDINAL(jj),base,digits,padchar));
                Str.Append(S,space);
                JD := JD + LONGREAL(1);
                INC(count);
            ELSE
                state:=done;
            END;
        | done:
            IF julian THEN
                Str.Append(S,padjulian);
            ELSIF numbered THEN
                Str.Append(S,paddiy);
            END;
            Str.Append(S,padquantieme);
            INC(count);
        END;
        IF count > lastweekday THEN
            storeline(WHICH,currline,inkmain,papermain,S);
            IF julian THEN
                 p:=minCol+1; (* right after first space *)
                 FOR ii:=firstweekday TO lastweekday DO
                     storeattr(p,wipadjulian-1, WHICH,currline,inkjulian,paperjulian);
                     INC(p,wipadjulian);
                     INC(p,wipadquantieme);
                 END;
            ELSIF numbered THEN
                 p:=minCol+1; (* right after first space *)
                 FOR ii:=firstweekday TO lastweekday DO
                     storeattr(p,wipaddiy-1, WHICH,currline,inkjulian,paperjulian);
                     INC(p,wipaddiy);
                     INC(p,wipadquantieme);
                 END;
            END;
            INC(currline);
            IF state=done THEN EXIT; END;
            JDtoDate(JD,jj,mm,yy);
            IF CARDINAL(mm) # month THEN EXIT; END; (* don't start a blank line *)
            S:=space;
            count:=firstweekday;
        END;
        IF (state # waiting) THEN INC(currjd);END;
    END;

    (* highlight current day if needed *)

    IF ( (month=monthnow) AND (year=yearnow) ) THEN
        p:=herepos;
        ii:=hereline;
        storechar(WHICH,ii,p  ,inknow,papernow,hilightleft);
        storechar(WHICH,ii,p+3,inknow,papernow,hilightright);
        storechar(WHICH,ii,p+1,inknow,papernow,retrievechar(WHICH,ii,p+1));
        storechar(WHICH,ii,p+2,inknow,papernow,retrievechar(WHICH,ii,p+2));
    END;

    DEC(currline); (* we were already at next line *)
    RETURN currline;
END buildmonth;

PROCEDURE dmpyear (programmer,julian,showfullJD,numbered:BOOLEAN;
                  daynow,monthnow,yearnow,year:CARDINAL;
                  language:languagetype;weekstart:daytype;
                  hilightleft,hilightright:CHAR;
                  inkmain,papermain,inktitle,papertitle,
                  inkdays,paperdays,inknow,papernow,
                  inkjulian,paperjulian:colortype);
VAR
    nlines:ARRAY [1..3] OF CARDINAL;
    month,i,len,last : CARDINAL;
BEGIN
    month:=firstmonth;
    IF (julian OR numbered) THEN
        last := 1; (* wide display *)
    ELSE
        last := 3;
    END;
    LOOP
        FOR i:=1 TO last DO
            nlines[i]:= buildmonth ( programmer,julian,showfullJD,numbered,
                                   i, daynow,monthnow,yearnow,month,year,
                                   language,weekstart,
                                   hilightleft,hilightright,
                                   inkmain,papermain,inktitle,papertitle,
                                   inkdays,paperdays,inknow,papernow,
                                   inkjulian,paperjulian);
            INC(month);
        END;
        IF last # 1 THEN
            nlines[2]:=merge(nlines[2],nlines[3], 2,3, inkmain,papermain);
            nlines[1]:=merge(nlines[1],nlines[2], 1,2, inkmain,papermain);
        END;
        dmp(1,nlines[1]);
        IF month > lastmonth THEN EXIT; END;

        color(inkmain,papermain);
        FOR i:=1 TO Str.Length(hline[1,minRow]) DO WrStr(space);END;
        WrLn;
    END;
END dmpyear;

(* ------------------------------------------------------------ *)

CONST
    minparm = 1;
    maxparm = 2;
VAR
    parmcount,i,opt:CARDINAL;
    S,R:str128;
    parm : ARRAY [minparm..maxparm] OF str128;
    lastparm:CARDINAL;
    v : LONGCARD;
    DEBUG:BOOLEAN;
VAR
    language:languagetype;
    weekstart:daytype;
    inkmain,papermain:colortype;
    inktitle,papertitle,inkdays,paperdays,inknow,papernow:colortype;
    inkjulian,paperjulian:colortype;
    inkspecial,paperspecial:colortype;
    hilightleft,hilightright:CHAR;
VAR
    yearnow,monthnow,daynow,year,month:CARDINAL;
    DoW:Lib.DayType;
    calmode : (monthly,yearly);
    programmer,julian,numbered,showtoday,showfullJD:BOOLEAN;
BEGIN
    Lib.DisableBreakCheck();

    (* handleVesa; *) (* useless, because we won't change video mode *)
    setUseBiosMode ( IsRedirected() );
    findInkPaperAtStartup();

    WrLn; (* must be AFTER lib init ! *)

    lastparm  := minparm-1;
    DEBUG     := FALSE;
    language  := french;
    weekstart := monday;
    programmer:= FALSE;
    julian    := FALSE;
    showfullJD:= FALSE;
    numbered  := FALSE;
    showtoday := TRUE;

    hilightleft    :=cHilightleft;
    hilightright   :=cHilightright;

    inkmain   := cyan;          papermain   := darkblue;
    inktitle  := white;         papertitle  := darkblue;
    inkdays   := black;         paperdays   := cyan;
    inknow    := yellow;        papernow    := darkred;
    inkspecial:= red;           paperspecial:= white; (* for future use *)
    inkjulian := yellow;        paperjulian := darkblue;

    parmcount := Lib.ParamCount();
    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i); cleantabs(S);
        Str.Copy(R,S); UpperCase(R); (* don't preserve accents *)
        IF isOption(R) THEN
            opt := GetOptIndex (R, "?"+delim+"H"+delim+"HELP"+delim+
                                   "DEBUG"+delim+
                                   "B"+delim+"BIOS"+delim+
                                   "U"+delim+"US"+delim+"UK"+delim+"ENGLISH"+delim+
                                   "S"+delim+"SUNDAY"+delim+
                                   "IM:"+delim+
                                   "PM:"+delim+
                                   "IT:"+delim+
                                   "PT:"+delim+
                                   "ID:"+delim+
                                   "PD:"+delim+
                                   "IN:"+delim+"IC:"+delim+
                                   "PN:"+delim+"PC:"+delim+
                                   "IS:"+delim+
                                   "PS:"+delim+
                                   "T:"+delim+"TODAY:"+delim+
                                   "X"+delim+"PROGRAMMER"+delim+"P"+delim+
                                   "J"+delim+"JULIAN"+delim+
                                   "N"+delim+
                                   "IJ:"+delim+
                                   "PJ:"+delim+
                                   "Q"+delim+"NOHILIGHT"+delim+
                                   "JJ"
                               );
            CASE opt OF
            | 1,2,3 :
                abort(errHelp,"");
            | 4:
                DEBUG:=TRUE;
            |5,6:
                setUseBiosMode ( TRUE );
            |7,8,9,10 :
                language:=english;
            |11,12:
                weekstart:=sunday;
            |13:
                IF GetLongCard(R,v)=FALSE THEN abort(errNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errInk,"");END;
                inkmain:=VAL(colortype,CARDINAL(v));
            |14:
                IF GetLongCard(R,v)=FALSE THEN abort(errNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errPaper,"");END;
                papermain:=VAL(colortype,CARDINAL(v));
            |15:
                IF GetLongCard(R,v)=FALSE THEN abort(errNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errInk,"");END;
                inktitle:=VAL(colortype,CARDINAL(v));
            |16:
                IF GetLongCard(R,v)=FALSE THEN abort(errNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errPaper,"");END;
                papertitle:=VAL(colortype,CARDINAL(v));
            |17:
                IF GetLongCard(R,v)=FALSE THEN abort(errNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errInk,"");END;
                inkdays:=VAL(colortype,CARDINAL(v));
            |18:
                IF GetLongCard(R,v)=FALSE THEN abort(errNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errPaper,"");END;
                paperdays:=VAL(colortype,CARDINAL(v));
            |19,20:
                IF GetLongCard(R,v)=FALSE THEN abort(errNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errInk,"");END;
                inknow:=VAL(colortype,CARDINAL(v));
            |21,22:
                IF GetLongCard(R,v)=FALSE THEN abort(errNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errPaper,"");END;
                papernow:=VAL(colortype,CARDINAL(v));
            |23:
                IF GetLongCard(R,v)=FALSE THEN abort(errNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errInk,"");END;
                inkspecial:=VAL(colortype,CARDINAL(v));
            |24:
                IF GetLongCard(R,v)=FALSE THEN abort(errNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errPaper,"");END;
                paperspecial:=VAL(colortype,CARDINAL(v));
            |25,26:
                GetString(S,R);
                IF Str.Length(R) # 2 THEN abort(errBadToday,S);END;
                hilightleft:=R[0];
                hilightright:=R[1];
            |27,28,29:
                programmer:=TRUE;
            |30,31:
                julian := TRUE;
            |32:
                numbered:=TRUE;
            |33:
                IF GetLongCard(R,v)=FALSE THEN abort(errNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errInk,"");END;
                inkjulian:=VAL(colortype,CARDINAL(v));
            |34:
                IF GetLongCard(R,v)=FALSE THEN abort(errNumber,S); END;
                IF ( (v < mincolor) OR (v > maxcolor) ) THEN abort(errPaper,"");END;
                paperjulian:=VAL(colortype,CARDINAL(v));
            |35,36:
                showtoday := FALSE;
            |37:
                julian     := TRUE;
                showfullJD := TRUE;
            ELSE
                abort(errUnknownOpt,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            INC(lastparm);
            IF lastparm > maxparm THEN abort(errTooManyParms,S);END;
            Str.Copy(parm[lastparm],R);
        END;
    END;

    IF (julian AND numbered) THEN abort(errJulian,"");END;

    IF showtoday=FALSE THEN
        hilightleft  := space;
        hilightright := space;
        inknow       := inkmain;
        papernow     := papermain;
    END;

    Lib.GetDate(yearnow,monthnow,daynow,DoW);

    CASE lastparm OF
    | minparm-1: (* current month and current year *)
        month:=monthnow;
        year :=yearnow;
        calmode:=monthly;
    | minparm:   (* year *)
        Str.Copy(S, parm[minparm]);
        IF same(S,"*") THEN
            year:=yearnow;
        ELSE
            IF str2card(year,S)=FALSE THEN abort(errYear,S); END;
        END;
        calmode:=yearly;
    | maxparm:   (* month year *)
        Str.Copy(S, parm[minparm]);
        IF str2card(month,S)=FALSE THEN
            i:=1;
            LOOP
                CASE i OF
                | 1 : R:=mois;
                | 2 : R:=abrevmois;
                | 3 : R:=moisUS;
                | 4 : R:=abrevmoisUS;
                END;
                UpperCase(R); (* don't preserve accents *)
                month:=getStrIndex(space, S, R);
                IF inrange(month,firstmonth,lastmonth) THEN EXIT; END;
                INC(i);
                IF i > 4 THEN abort(errMonth,S); END;
            END;
        END;
        IF inrange(month,firstmonth,lastmonth)=FALSE THEN abort(errMonth,S);END;
        Str.Copy(S, parm[minparm+1]);
        IF same(S,"*") THEN
            year:=yearnow;
        ELSE
            IF str2card(year,S)=FALSE THEN abort(errYear,S); END;
        END;
        calmode:=monthly;
    END;

    (*
    IF DEBUG THEN
        Str.Concat(S,"day       : ",fmtnum(day,10,8,space));WrStr(S);WrLn;
        Str.Concat(S,"month     : ",fmtnum(month,10,8,space));
        Str.Append(S," = ");Str.Append(S,getElem(month,space,mois));
        Str.Append(S," = ");Str.Append(S,getElem(month,space,moisUS));
        WrStr(S);WrLn;
        Str.Concat(S,"year      : ",fmtnum(year ,10,8,space));WrStr(S);WrLn;
    END;
    *)

    CASE calmode OF
    | monthly:        (* 1 is minRow *)
        i:=buildmonth ( programmer,julian,showfullJD,numbered,
                      1, daynow,monthnow,yearnow,month,year,
                      language,weekstart,
                      hilightleft,hilightright,
                      inkmain,papermain,inktitle,papertitle,
                      inkdays,paperdays,inknow,papernow,inkjulian,paperjulian);
        dmp( 1, i);
    | yearly:
        dmpyear       ( programmer,julian,showfullJD,numbered,
                      daynow,monthnow,yearnow,year,
                      language,weekstart,
                      hilightleft,hilightright,
                      inkmain,papermain,inktitle,papertitle,
                      inkdays,paperdays,inknow,papernow,inkjulian,paperjulian);
    END;

    abort(errNone,"");
END Cal.

