
(* ---------------------------------------------------------------
Title         Q&D Morse
Overview      tsk tsk...
Usage         see help
Notes         tsk tsk...
Bugs          yes, TopSpeed did it again !
Wish List     tsk tsk...

--------------------------------------------------------------- *)

MODULE Morse;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;

FROM IO IMPORT WrStr,WrLn;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, getCurrentDirectory, setReadWrite,
getFileSize, verifyString, str4096, unfixDirectory,cleantabs;

(* ------------------------------------------------------------ *)

CONST
    ProgEXEname   = "MORSE";
    ProgTitle     = "Q&D Morse";
    ProgVersion   = "v1.0b";
    ProgCopyright = "by PhG";
    Banner        = ProgTitle+" "+ProgVersion+" "+ProgCopyright;
CONST
    extDAT        = ".DAT";
    dquote        = '"';
    tab           = CHR(9);
    blank         = " ";
    semicolon     = ";";
    pound         = "#";
    huh           = "<?>";
CONST
    errNone             = 0;
    errHelp             = 1;
    errOption           = 2;
    errRedirected       = 3;
    errOverflow         = 4;
    errNonsense         = 5;
    errBadNumber        = 6;
    errFreqRange        = 7;
    errDelayRange       = 8;
    errNotFound         = 9;

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
    cr = CHR(13);
    lf = CHR(10);
    nl = cr+lf;
CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)
    msgHelp =
Banner+nl+
nl+
"Syntax : "+ProgEXEname+" [option]... <quoted string>"+nl+
nl+
"  -f:#  frequency (default is 1000 Hz)"+nl+
"  -d:#  delay unit (default is 44 milliseconds)"+nl+
"  -n    no sound"+nl+
"  -v    verbose (-vv = very verbose)"+nl+
"  -t    terse"+nl+
"  -s    spell using international aviation code (-n is implicit then)"+nl+
"  -f    French alternate for -s"+nl+
"  -e    use data from external "+ProgEXEname+extDAT+" file"+nl+
nl+
"This is international Morse code. American Morse code is, of course, different."+nl+
'Data file format is "character word" (undefined codes are shown as "'+huh+'").'+nl+
nl+
"The characters that are most frequently used have the shortest code symbols,"+nl+
"and the characters that are infrequently used have the longer symbols :"+nl+
"therefore, international Morse code looks like a fixed Huffman tree."+nl;

VAR
    S : str256;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(msgHelp);
    | errOption :
        Str.Concat(S,"Unknown ",einfo);Str.Append(S," option !");
    | errRedirected:
        S:="Redirection is a nonsense here !";
    | errOverflow:
        S:="Use double quotes if string contains more than one single word !";
    | errNonsense:
        S:="-v and -t options are mutually exclusive !";
    | errBadNumber  :
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," number !");
    | errFreqRange  :
        S := "Frequency range is [1..60000] !";
    | errDelayRange :
        S := "Delay range is [1..1000] !";
    | errNotFound:
        S := ProgEXEname+extDAT+" does not exist !";
    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp: ;
    ELSE
        WrStr(ProgEXEname+" : ");WrStr(S);WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)

PROCEDURE cardToBin (i:CARDINAL; VAR S : ARRAY OF CHAR);
VAR
    ok:BOOLEAN;
BEGIN
    Str.CardToStr (  LONGCARD(i), S, 2, ok); (* always ok here *)
END cardToBin;

PROCEDURE reverseString (VAR S : ARRAY OF CHAR );
VAR
    i:CARDINAL;
    R:str128;
BEGIN
    Str.Copy(R,"");
    FOR i:=1 TO Str.Length(S) DO
        Str.Prepend(R, S[i-1] );
    END;
    Str.Copy(S,R);
END reverseString;

(* ------------------------------------------------------------ *)

CONST
    shortvalue = CHR(254);
    longvalue  = "-";
    firstcode = ORD(" ");
    lastcode  = ORD("Z");
    SPACEMASK = (1 << 15);
    BITMASK   = 0FEH; (* "11111110" *)
    defaultFREQUENCY = 1000; (* was 1500 *)
    defaultDELAYUNIT  = 44; (* was 18 *)
    minfreq = 1;
    maxfreq = 60000;
    mindelay = 1;
    maxdelay = 1000;
TYPE
    encodetable= ARRAY [firstcode..lastcode] OF CARDINAL;
CONST
    charToCode = encodetable (
        SPACEMASK,                                      (* space             *)
        000H, 000H, 000H, 000H, 000H, 000H, 000H, 000H, (* ! " # $  % & " (  *)
        000H, 000H, 000H, 073H, 031H, 06AH, 029H,       (* ) * + , - . /     *)
        03FH, 03EH, 03CH, 038H, 030H, 020H, 021H, 023H, (* 0 1 2 3 4 5 6 7   *)
        027H, 02FH, 000H, 000H, 000H, 000H, 000H, 04CH, (* 8 9 : ; < = > ?   *)
        000H, 006H, 011H, 015H, 009H, 002H, 014H, 00BH, (* @ A B C D E F G   *)
        010H, 004H, 01EH, 00DH, 012H, 007H, 005H, 00FH, (* H I J K L M N O   *)
        016H, 01BH, 00AH, 008H, 003H, 00CH, 018H, 00EH, (* P Q R S T U V W   *)
        019H, 01DH, 013H                                (* X Y Z             *)
        );

PROCEDURE filterForMorse (VAR S:ARRAY OF CHAR);
VAR
    R : str128;
    i,code : CARDINAL;
BEGIN
    UpperCase(S);
    Str.Copy(R,"");
    FOR i := 1 TO Str.Length(S) DO
        code :=ORD( S[i-1] );
        CASE code OF
        | firstcode..lastcode :
            Str.Append(R,CHR(code));
        END;
    END;
    Str.Copy(S,R);
END filterForMorse;

PROCEDURE pause (delay:CARDINAL);
BEGIN
    Lib.Delay(delay);
END pause;

PROCEDURE sound (frequency,delay:CARDINAL);
BEGIN
    Lib.Sound(frequency);
    pause(delay);
    Lib.NoSound();
END sound;

PROCEDURE senddash (frequency,delay:CARDINAL ; nosound:BOOLEAN);
BEGIN
    WrStr(longvalue);
    IF nosound THEN
        pause( delay * 3 );
    ELSE
        sound( frequency, delay * 3);
    END;
    pause( delay );
END senddash;

PROCEDURE senddot (frequency,delay:CARDINAL;  nosound:BOOLEAN);
BEGIN
	WrStr(shortvalue);
	IF nosound THEN
	    pause( delay);
	ELSE
	    sound(frequency,delay) ;
	END;
    pause (delay) ;
END senddot;

PROCEDURE sendcharspace (delay:CARDINAL);
BEGIN
    WrStr(blank);
    pause (delay * 2);
END sendcharspace;

PROCEDURE sendwordspace (delay:CARDINAL);
BEGIN
    WrStr(blank+blank);
    pause (delay * 4);
END sendwordspace;

(* assume S has already been filtered and uppercased *)

PROCEDURE emitMorse (frequency,delay:CARDINAL;
                     nosound:BOOLEAN; S:ARRAY OF CHAR   );
VAR
    i,code,v : CARDINAL;
BEGIN
    FOR i:=1 TO Str.Length(S) DO
        code:=ORD( S[i-1] );
        CASE code OF
        | firstcode..lastcode :
            v := charToCode[code];
            IF (v AND SPACEMASK)=0 THEN (* everything but space *)
                WHILE (v AND BITMASK) # 0 DO
                   IF (v AND 01H) = 0 THEN
                       senddot (frequency,delay,nosound);
                   ELSE
                       senddash (frequency,delay,nosound);
                   END;
                   v := (v >> 1);
                END;
                sendcharspace (delay) ;
            ELSE
                sendwordspace (delay);
            END;
        END;
    END;
    WrLn;
END emitMorse;

PROCEDURE dumpOutput (S:ARRAY OF CHAR   );
VAR
    i,code,v : CARDINAL;
    R : str80;
BEGIN
    FOR i:=1 TO Str.Length(S) DO
        code:=ORD( S[i-1] );
        CASE code OF
        | firstcode..lastcode :
            v := charToCode[code];
            IF (v AND SPACEMASK)=0 THEN (* everything but space *)
                cardToBin(v,R);
                (* first char must be the stop marker bit, or else it's an undefined char *)
                IF R[0]="1" THEN
                    Str.Delete(R,0,1); (* delete stop marker bit *)
                    Str.Delete(R,0,1); (* delete one char *)
                    WrStr( S[i-1]);
                    FOR v := 1 TO Str.Length(R) DO
                        WrStr(blank);
                    END;
                END;
                WrStr(blank);
            ELSE
                WrStr(blank+blank);
            END;
        END;
    END;
    WrLn;
END dumpOutput;

(* ------------------------------------------------------------ *)

TYPE
    spelltype = (default,french,external);
VAR
    spellflag: ARRAY [MIN(CHAR)..MAX(CHAR)] OF BOOLEAN;
    spellcode: ARRAY [MIN(CHAR)..MAX(CHAR)] OF str80;

PROCEDURE initSpell (mode:spelltype; F:ARRAY OF CHAR );
CONST
    spellcodes=
"A Alpha,"+
"B Bravo,"+
"C Charlie,"+
"D Delta,"+
"E Echo,"+
"F Foxtrot,"+
"G Golf,"+
"H Hotel,"+
"I India,"+
"J Juliet,"+
"K Kilo,"+
"L Lima,"+
"M Mike,"+
"N November,"+
"O Oscar,"+
"P Papa,"+
"Q Qubec,"+
"R Romo,"+
"S Sierra,"+
"T Tango,"+
"U Uniform,"+
"V Victor,"+
"W Whiskey,"+
"X X-Ray,"+
"Y Yankee,"+
"Z Zulu,"+
"1 One,"+
"2 Two,"+
"3 Three,"+
"4 Four,"+
"5 Five,"+
"6 Six,"+
"7 Seven,"+
"8 Eight,"+
"9 Nine,"+
"0 Zero";

    spellcodesFR=
"A Alpha,"+
"B Bravo,"+
"C Charlie,"+
"D Delta,"+
"E Echo,"+
"F Foxtrot,"+
"G Golf,"+
"H Hotel,"+
"I India,"+
"J Juliet,"+
"K Kilo,"+
"L Lima,"+
"M Mike,"+
"N November,"+
"O Oscar,"+
"P Papa,"+
"Q Qubec,"+
"R Romo,"+
"S Sierra,"+
"T Tango,"+
"U Uniform,"+
"V Victor,"+
"W Whiskey,"+
"X X-Ray,"+
"Y Yankee,"+
"Z Zanzibar,"+
"1 Un,"+
"2 Deux,"+
"3 Trois,"+
"4 Quatre,"+
"5 Cinq,"+
"6 Six,"+
"7 Sept,"+
"8 Huit,"+
"9 Neuf,"+
"0 Zro";

VAR
    c : CHAR;
    i : CARDINAL;
    S : str80;
    hnd:FIO.File;
BEGIN
FOR c:=MIN(CHAR) TO MAX(CHAR) DO
    spellflag[c]:=FALSE;
    spellcode[c]:="";
END;
CASE mode OF
| default :
    i:=0;
    LOOP
        Str.ItemS(S,spellcodes, "," ,i);
        IF same(S,"") THEN EXIT; END;
        c:=S[0];
        Str.Delete(S,0,1);
        LtrimBlanks(S);
        spellflag[c]:=TRUE;
        spellcode[c]:=S;
        INC(i);
    END;
| french:
    i:=0;
    LOOP
        Str.ItemS(S,spellcodesFR, "," ,i);
        IF same(S,"") THEN EXIT; END;
        c:=S[0];
        Str.Delete(S,0,1);
        LtrimBlanks(S);
        spellflag[c]:=TRUE;
        spellcode[c]:=S;
        INC(i);
    END;
| external:
    FOR c:=MIN(CHAR) TO MAX(CHAR) DO
        spellflag[c]:=TRUE;
        spellcode[c]:=huh;
    END;

    hnd:=FIO.OpenRead(F);
    i:=0;
    LOOP
        IF FIO.EOF THEN EXIT;END;
        FIO.RdStr(hnd,S);
        IF Str.Length(S) > 0 THEN
            c:=S[0];
            CASE c OF
            | semicolon,pound:
                ;
            ELSE
                UpperCase(c);
                Str.Delete(S,0,1);
                LtrimBlanks(S); RtrimBlanks(S);
                spellflag[c]:=TRUE;
                spellcode[c]:=S;
                INC(i);
            END;
        END;
    END;
    FIO.Close(hnd);
END;
END initSpell;

PROCEDURE getspell ( c:CHAR):str80;
BEGIN
    IF spellflag[c]=FALSE THEN RETURN ""; END;
    RETURN spellcode[c];
END getspell;

PROCEDURE filterForSpell (VAR S:ARRAY OF CHAR);
VAR
    R : str128;
    i,code : CARDINAL;
    c:CHAR;
BEGIN
    UpperCase(S);
    Str.Copy(R,"");
    FOR i := 1 TO Str.Length(S) DO
        c:=S[i-1];
        IF spellflag[c] THEN
            Str.Append(R,c);
        ELSE
            IF c=blank THEN Str.Append(R,blank);END;
        END;
    END;
    Str.Copy(S,R);
END filterForSpell;

PROCEDURE emitSpell (S:ARRAY OF CHAR);
VAR
    i,len:CARDINAL;
    c:CHAR;
BEGIN
    len:=Str.Length(S);
    FOR i := 1 TO len DO
        c:=S[i-1];
        IF spellflag[c] THEN
            WrStr(getspell(c));
            IF i < len THEN WrStr(" ");END;
        ELSE
            IF c=blank THEN WrStr(blank+blank);END;
            IF i < len THEN WrStr(" ");END;
        END;
    END;
    WrLn;
END emitSpell;

(* ------------------------------------------------------------ *)

PROCEDURE buildCurrentAndEXE (VAR incurrent,inexe:ARRAY OF CHAR);
VAR
    S,u,d,n,e:str128; (* oversized *)
BEGIN
    Lib.ParamStr(S,0);
    Str.Caps(S); (* useless *)
    Lib.SplitAllPath(S,u,d,n,e);
    Lib.MakeAllPath(incurrent,"","",n,extDAT);
    Lib.MakeAllPath(inexe,u,d,n,extDAT);
END buildCurrentAndEXE;

(* ------------------------------------------------------------ *)

CONST
    msgText  = "Text   : ";
    msgTextU = "TEXT   : ";
    msgOutput= "Output : ";
    msgMorse = "Morse  : ";
    msgSpell = "Spell  : ";
VAR
    parmcount,i,opt,v:CARDINAL;
    S,R,text:str128;
    nosound,verbose,terse,veryverbose,debug,spell:BOOLEAN;
    FREQUENCY,DELAYUNIT : CARDINAL;
    lc:LONGCARD;
    spellmode:spelltype;
BEGIN
    Lib.DisableBreakCheck();
    WrLn;
    parmcount := Lib.ParamCount();
    IF parmcount = 0 THEN abort(errHelp,""); END;
    Str.Copy(text,"");
    nosound:=FALSE;
    verbose:=FALSE;
    veryverbose:=FALSE;
    terse  :=FALSE;
    FREQUENCY := defaultFREQUENCY;
    DELAYUNIT  := defaultDELAYUNIT;
    debug  :=FALSE;
    spell := FALSE;
    spellmode:=default;
    FOR i:=1 TO parmcount DO
        Lib.ParamStr(S,i); cleantabs(S); (* because of YATB ! Yet Another TopSpeed Bug ! *)
        Str.Copy(R,S);
        UpperCase(R);
        IF isOption(R) THEN
            opt := GetOptIndex(R, "?"+delim+"H"+delim+"HELP"+delim+
                                  "N"+delim+"NOSOUND"+delim+
                                  "V"+delim+"VERBOSE"+delim+
                                  "T"+delim+"TERSE"+delim+
                                  "F:"+delim+"FREQUENCY:"+delim+
                                  "U:"+delim+"UNIT:"+delim+"D:"+delim+"DELAY:"+delim+
                                  "VV"+delim+"VERYVERBOSE"+delim+
                                  "DEBUG"+delim+
                                  "S"+delim+"SPELL"+delim+
                                  "F"+delim+"FRENCH"+delim+"FR"+delim+
                                  "E"+delim+"EXTERNAL"
                              );
            CASE opt OF
            | 1,2,3 : abort(errHelp,"");
            | 4,5   : nosound:=TRUE;
            | 6,7   : verbose:=TRUE;
            | 8,9   : terse  :=TRUE;
            | 10,11 :
                IF GetLongCard(S,lc)=FALSE THEN abort(errBadNumber,S); END;
                IF ( (lc < minfreq) OR (lc > maxfreq) ) THEN abort(errFreqRange,"");END;
                FREQUENCY :=CARDINAL(lc);
            | 12,13,14,15 :
                IF GetLongCard(S,lc)=FALSE THEN abort(errBadNumber,S); END;
                IF ( (lc < mindelay) OR (lc > maxdelay) ) THEN abort(errDelayRange,"");END;
                DELAYUNIT :=CARDINAL(lc);
            | 16,17 : verbose := TRUE; veryverbose := TRUE;
            | 18    : debug  :=TRUE;
            | 19,20 :    spell:=TRUE; nosound:=TRUE; spellmode:=default;
            | 21,22,23 : spell:=TRUE; nosound:=TRUE; spellmode:=french;
            | 24,25 :    spell:=TRUE; nosound:=TRUE; spellmode:=external;
            ELSE
                abort(errOption,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            IF same(text,"")=FALSE THEN abort(errOverflow,"");END;
            Str.Copy(text,S);
        END;
    END;

    IF verbose AND terse THEN abort(errNonsense,"");END;

    IF spell THEN
        buildCurrentAndEXE(S,R);
        IF FIO.Exists(S)=FALSE THEN
            IF FIO.Exists(R)=FALSE THEN abort(errNotFound,S); END; (* yes, S *)
            Str.Copy(S,R);
        END;
        initSpell(spellmode,S);
    END;

    IF (debug AND (spell=FALSE)) THEN
        FOR i:=firstcode TO lastcode DO
            v := charToCode[i]; (* always in legal range *)
            IF (v AND SPACEMASK)=0 THEN (* ok only if not space *)
                Str.Copy(text, CHR(i) );
                filterForMorse(text); (* utterly useless here ! *)
                cardToBin(v,S);
                (* first char must be the stop marker bit, or else it's an undefined char *)
                IF S[0]="1" THEN
                    Str.Delete(S,0,1);
                    reverseString(S);
                    Str.Copy(R,S);
                    ReplaceChar(R, "0", shortvalue );
                    ReplaceChar(R, "1", longvalue  );
                    Str.Append(text,tab);
                    Str.Append(text,S);
                    Str.Append(text,tab);
                    Str.Append(text,R);
                ELSE
                    (*
                    Str.Append(text,tab);
                    Str.Append(text,"???");
                    *)
                END;
                WrStr(text);WrLn;
            END;
        END;
        abort(errNone,"");
    END;

    IF (debug AND spell) THEN
        FOR i:=ORD(MIN(CHAR)) TO ORD(MAX(CHAR)) DO
            IF spellflag[CHR(i)] THEN
                Str.Concat(text,CHR(i),tab);
                Str.Append(text,getspell( CHR(i)) );
                WrStr(text);WrLn;
            END;
        END;
        abort(errNone,"");
    END;

    IF IsRedirected() THEN abort(errRedirected,"");END;

IF spell THEN
    IF verbose THEN
        WrStr(msgText); WrStr(dquote); WrStr(text); WrStr(dquote); WrLn;
    END;
    filterForSpell(text);
    IF verbose THEN
        WrStr(msgTextU);WrStr(dquote); WrStr(text); WrStr(dquote); WrLn;
    END;
    IF terse=FALSE THEN WrStr(msgSpell); END;
    emitSpell(text);
ELSE
    IF verbose THEN
        WrStr(msgText); WrStr(dquote); WrStr(text); WrStr(dquote); WrLn;
    END;
    filterForMorse(text); (* uppercase and legal chars only *)
    IF verbose THEN
        WrStr(msgTextU);WrStr(dquote); WrStr(text); WrStr(dquote); WrLn;
    END;
    (* only Morse code output *)
    IF veryverbose THEN
        WrStr(msgOutput);
        dumpOutput(text);
    END;

    IF terse=FALSE THEN WrStr(msgMorse); END;
    emitMorse(FREQUENCY,DELAYUNIT,nosound,text);
END;

    abort(errNone,"");
END Morse.

