
(* ---------------------------------------------------------------
Title         Q&D Yes/No Quiz
Overview      see help
Usage         see help
Notes         as usual, Q&D rules ! :-(
Bugs
Wish List     yes, we should use dynamic allocation... but what for ?

--------------------------------------------------------------- *)

MODULE QuizYN;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;
IMPORT SYSTEM;
IMPORT IO;

FROM IO IMPORT WrStr, WrLn;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim,
completedInit, completedShow, completedSHOW, completedEnd, completed;

(* ------------------------------------------------------------ *)

CONST
    cr              = CHR(13);
    lf              = CHR(10);
    nl              = cr+lf;
    extEXE          = ".EXE";
    extDAT          = ".DAT";
    extLOG          = ".LOG";
    star            = "*";
    semicolon       = ";";
    colon           = ":";
    doublequote     = '"';
    singlequote     = "'";
    dot             = ".";
    dollar          = "$";
CONST
    ProgEXEname     = "QUIZYN";
    ProgTitle       = "Q&D Yes/No Quiz";
    ProgVersion     = "v1.0";
    ProgCopyright   = "by PhG";
    Banner          = ProgTitle+" "+ProgVersion+" "+ProgCopyright;
CONST
    errNone             = 0;
    errHelp             = 1;
    errOption           = 2;
    errParameter        = 3;
    errDataNotFound     = 4;
    errTooManyQuestions = 5;
    errTooManyScorings  = 6;
    errBadQuestion      = 7;
    errBadScoring       = 8;
    errBadWeight        = 9;
    errBadUpper         = 10;
    errRedirected       = 11;
    errAborted          = 12;
    errNotWithResume    = 13;
    errBadWidth         = 14;
    errLogNotFound      = 15;
    errBadLog           = 16;

(* ------------------------------------------------------------ *)

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)

errmsg =
Banner+nl+
nl+
"Syntax : "+ProgEXEname+" [-d:datafile["+extDAT+"]] [-l:logfile[+extLOG+]] [option]..."+nl+
nl+
"This is a computerized Yes/No quiz."+nl+
nl+
"  -d:$ data file (default is "+ProgEXEname+extDAT+")"+nl+
"  -l:$ log file (default is data file with "+extLOG+" extension)"+nl+
"  -o   do not shuffle questions"+nl+
"  -s   save results to log file file"+nl+
"  -r   resume using data from log file"+nl+
"  -w:# line width (default is 74)"+nl;

VAR
    S : str256;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(errmsg);
    | errOption:
        Str.Concat(S,"Unknown ",einfo);Str.Append(S," option !");
    | errParameter:
        Str.Concat(S,"Useless ",einfo);Str.Append(S," parameter !");
    | errDataNotFound:
        Str.Concat(S,einfo," data file does not exist !");
    | errTooManyQuestions:
        S:="Too many questions in data file !";
    | errTooManyScorings:
        S:="Too many scoring entries in data file !";
    | errBadQuestion:
        Str.Concat(S,"Bad question format at line ",einfo);Str.Append(S," !");
    | errBadScoring:
        Str.Concat(S,"Bad scoring entry format at line ",einfo);Str.Append(S," !");
    | errBadWeight:
        Str.Concat(S,"Bad weight value at line ",einfo);Str.Append(S," !");
    | errBadUpper:
        Str.Concat(S,"Bad upper value at line ",einfo);Str.Append(S," !");
    | errRedirected:
        S:="Output redirection is a nonsense !";
    | errAborted:
        S:="Aborted by user !";
    | errNotWithResume:
        S:="-o option is a nonsense with -r option !";
    | errBadWidth :
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," line width !");
    | errLogNotFound:
        Str.Concat(S,einfo," log file does not exist !");
    | errBadLog:
        Str.Concat(S,"Corrupted ",einfo);Str.Append(S," log file !");

    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp: ;
    ELSE
        WrStr(ProgEXEname+" : ");WrStr(S);WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    ioBufferSize    = (8 * 512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = ioBufferSize;
VAR
    ioBuffer : ARRAY [firstBufferByte..lastBufferByte] OF BYTE;

CONST
    sep          = " : ";
    msgReading   = "Reading...";
    msgShuffling = "Shuffling...";
    msgSaving    = "Saving...";
    msgResuming  = "Loading...";
    undefined    = " ";
    reindexed    = "*";
CONST
    prefixinfo    = "### ";
    prefix        = "::: ";
    msgQuestion   = prefix+"Question ";
    msgOutOf      = " out of ";
    msgChoice     = " [Y/N/Q]";
    legalchoices  = "YNQ"+CHR(27);
    msgYes        = " Yes";
    msgNo         = " No";
    msgQuit       = " Quit";
    msgYourScore  = prefix+"Your score"+sep;
    msgCreated    = " has been created !";
    msgLoaded     = " has been loaded !";
TYPE
    questiontype = RECORD
        index  : CARDINAL;
        weight : INTEGER;
        text   : str256; (* should do *)
        answer : CHAR;
    END;
    scoringtype = RECORD
        upper : INTEGER;
        text  : str256;  (* should do *)
    END;
CONST
    firstQuestion = 1;
    maxQuestion   = 100;
    firstScoring  = 1;
    maxScoring    = 20;
VAR
    question     : ARRAY[firstQuestion..maxQuestion] OF questiontype;
    scoring      : ARRAY[firstScoring..maxScoring] OF scoringtype;
    lastQuestion : CARDINAL;
    lastScoring  : CARDINAL;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

PROCEDURE readentry (hnd:FIO.File;
                     VAR currline:CARDINAL;
                     VAR R:ARRAY OF CHAR):BOOLEAN ;
VAR
    len:CARDINAL;
BEGIN
    LOOP
        FIO.RdStr(hnd,R);
        INC(currline);
        LtrimBlanks(R);
        RtrimBlanks(R);
        CASE R[0] OF
        | CHR(0): ; (* empty string *)
        | semicolon : ; (* comment *)
        | dollar : RETURN TRUE; (* end of section *)
        | doublequote,singlequote:
            len:=Str.Length(R);
            IF len >= 2 THEN
                IF R[len-1]=R[0] THEN
                    R[len-1]:=0C;
                    Str.Delete(R,0,1);
                END;
            END;
            RETURN FALSE;
        ELSE
            RETURN FALSE; (* not done yet *)
        END;
        IF FIO.EOF THEN RETURN TRUE; END;
    END;
END readentry;

PROCEDURE dump (wi:CARDINAL; lastCR:BOOLEAN; para:ARRAY OF CHAR);
VAR
    ok:BOOLEAN;
    R:str128;
BEGIN
    ok:=dmpTTX (para,wi,R,TRUE);
    WHILE ok DO
        WrStr(R);
        ok:=dmpTTX(para,wi,R,FALSE);
        IF ok THEN
            WrLn;
        ELSE
            IF lastCR THEN WrLn;END;
        END;
    END;
END dump;

PROCEDURE getintval (S:ARRAY OF CHAR):INTEGER;
VAR
    lc:LONGINT;
    ok:BOOLEAN;
BEGIN
    lc:=Str.StrToInt(S,10,ok);
    IF ok=FALSE THEN RETURN MAX(INTEGER);END;
    IF lc>MAX(INTEGER) THEN RETURN MAX(INTEGER);END;
    RETURN INTEGER(lc);
END getintval;

PROCEDURE getcardval (S:ARRAY OF CHAR):CARDINAL;
VAR
    lc:LONGCARD;
    ok:BOOLEAN;
BEGIN
    lc:=Str.StrToCard(S,10,ok);
    IF ok=FALSE THEN RETURN MAX(CARDINAL);END;
    IF lc>MAX(CARDINAL) THEN RETURN MAX(CARDINAL);END;
    RETURN CARDINAL(lc);
END getcardval;

PROCEDURE getrndwithin (lowerlimit,upperlimit:CARDINAL):CARDINAL;
VAR
    lowerbound:REAL;
    upperbound:REAL;
    rnd:REAL;
BEGIN
    lowerbound:=REAL(lowerlimit);
    upperbound:=REAL(upperlimit);
    rnd :=  (upperbound-lowerbound+1.0) * Lib.RAND()+ lowerbound ;
    RETURN CARDINAL(rnd);
END getrndwithin;

PROCEDURE getchoice (legal:ARRAY OF CHAR):CHAR;
VAR
    ch:CHAR;
BEGIN
    LOOP
        ch:=IO.RdKey();
        IF ch=CHR(0) THEN
            ch:=IO.RdKey();
        ELSE
            IF Str.CharPos(legal,ch) # MAX(CARDINAL) THEN EXIT; END;
            ch:=CAP(ch);
            IF Str.CharPos(legal,ch) # MAX(CARDINAL) THEN EXIT; END;
        END;
    END;
    RETURN ch;
END getchoice;

PROCEDURE getCard (VAR wi:CARDINAL;
                   lower,upper:CARDINAL;S:ARRAY OF CHAR):BOOLEAN;
VAR
    lc:LONGCARD;
BEGIN
    IF GetLongCard(S,lc)=FALSE THEN RETURN FALSE;END;
    IF lc > MAX(CARDINAL) THEN RETURN FALSE; END;
    wi:=CARDINAL(lc);
    IF ( (wi < lower) OR (wi > upper) ) THEN RETURN FALSE; END;
    RETURN TRUE;
END getCard;

(* uses a few globerks ! ;-) *)

PROCEDURE loadData (debug:BOOLEAN;datafile:ARRAY OF CHAR;wi:CARDINAL;
                    VAR info:ARRAY OF CHAR):CARDINAL;
VAR
    hnd:FIO.File;
    i,currline,p,rc:CARDINAL;
    done:BOOLEAN;
    S:str128; (* oversized *)
    para:str1024; (* should do for intro longest line *)
BEGIN
    FIO.EOF:=FALSE;
    hnd:=FIO.OpenRead(datafile);
    FIO.AssignBuffer(hnd,ioBuffer);
    currline:=0;
    FOR i:=1 TO 2 DO (* title then intro *)
        LOOP
            done:=readentry(hnd,currline,para);
            IF done THEN EXIT;END;
            dump(wi,TRUE,para);
        END;
        WrLn;
    END;

    video(msgReading,TRUE);

    rc:=errNone;
    lastQuestion:=firstQuestion-1; (* read questions *)
    LOOP
        done:=readentry(hnd,currline,para);
        IF done THEN EXIT;END;
        INC(lastQuestion);
        IF lastQuestion > maxQuestion THEN
            rc:=errTooManyQuestions;
            EXIT;
        END;
        IF Str.Match(para,"*:*")=FALSE THEN
            rc:=errBadQuestion;
            EXIT;
        END;
        p:=Str.CharPos(para,colon);
        Str.Slice(S,para,0,p);
        question[lastQuestion].weight:=getintval(S);
        IF question[lastQuestion].weight=MAX(INTEGER) THEN
            rc:=errBadWeight;
            EXIT;
        END;
        Str.Delete(para,0,p+1);
        Str.Copy(question[lastQuestion].text,para);
        IF debug THEN
            IO.WrCard(lastQuestion,4);
            IO.WrInt(question[lastQuestion].weight,4);
            WrStr(sep);
            WrStr(question[lastQuestion].text);WrLn;
        END;
    END;
    IF rc # errNone THEN
        FIO.Close(hnd);
        video(msgReading,FALSE);
        Str.CardToStr(LONGCARD(currline),info,10,done);
        RETURN rc;
    END;

    rc:=errNone;
    lastScoring:=firstScoring-1; (* read scoring entries *)
    LOOP
        done:=readentry(hnd,wi,para);
        IF done THEN EXIT;END;
        INC(lastScoring);
        IF lastScoring > maxScoring THEN
            rc:=errTooManyScorings;
            EXIT;
        END;
        IF Str.Match(para,"*:*")=FALSE THEN
            rc:=errBadScoring;
            EXIT;
        END;
        p:=Str.CharPos(para,colon);
        Str.Slice(S,para,0,p);
        scoring[lastScoring].upper:=getintval(S);
        IF scoring[lastScoring].upper=MAX(INTEGER) THEN
            rc:=errBadUpper;
            EXIT;
        END;
        Str.Delete(para,0,p+1);
        Str.Copy(scoring[lastScoring].text,para);
        IF debug THEN
            IO.WrCard(lastScoring,4);
            IO.WrCard(scoring[lastScoring].upper,4);
            WrStr(sep);
            WrStr(scoring[lastScoring].text);WrLn;
        END;
    END;
    IF rc # errNone THEN
        FIO.Close(hnd);
        video(msgReading,FALSE);
        Str.CardToStr(LONGCARD(currline),info,10,done);
        RETURN rc;
    END;
    FIO.Close(hnd);
    video(msgReading,FALSE);

    RETURN errNone;
END loadData;

PROCEDURE saveLog (debug:BOOLEAN;logfile:ARRAY OF CHAR);
VAR
    hnd:FIO.File;
    i,p:CARDINAL;
BEGIN
    video(msgSaving,TRUE);
    hnd:=FIO.Create(logfile);
    FOR i:=firstQuestion TO lastQuestion DO
        p:=question[i].index;
        FIO.WrCard(hnd,p,4);
        FIO.WrStr(hnd,sep);
        FIO.WrStr(hnd,question[p].answer);
        FIO.WrLn(hnd);
        IF debug THEN
           IO.WrCard(p,4);
           WrStr(sep);
           WrStr(question[p].answer);
           WrLn;
        END;
    END;
    FIO.Close(hnd);
    video(msgSaving,FALSE);
    WrStr(prefixinfo);WrStr(logfile);WrStr(msgCreated);WrLn;
END saveLog;

PROCEDURE loadLog (debug:BOOLEAN;logfile:ARRAY OF CHAR):BOOLEAN ;
VAR
    hnd:FIO.File;
    i,p,len:CARDINAL;
    S,R:str128;
BEGIN
    len:=Str.Length(sep);
    video(msgResuming,TRUE);
    hnd:=FIO.OpenRead(logfile);
    FIO.AssignBuffer(hnd,ioBuffer);

    FOR i:=firstQuestion TO lastQuestion DO
        FIO.RdStr(hnd,S);
        IF debug THEN WrStr(S);WrLn;END;
        p:=Str.Pos(S,sep);
        IF p = MAX(CARDINAL) THEN
            FIO.Close(hnd);
            video(msgResuming,FALSE);
            RETURN FALSE;
        END;
        Str.Slice(R,S,0,p);
        LtrimBlanks(R);
        RtrimBlanks(R);
        Str.Delete(S,0,p+len);
        p:=getcardval(R);
        question[i].index:=p;
        IF S[0] = reindexed THEN S[0] := undefined; END; (* should never happen *)
        question[p].answer:=S[0];
        IF debug THEN
            IO.WrCard(question[i].index,4);
            WrStr(sep);
            WrStr(question[p].answer);
            WrLn;
        END;
    END;
    FIO.Close(hnd);
    video(msgResuming,FALSE);
    RETURN TRUE;
END loadLog;

PROCEDURE initQuestions (  );
VAR
    i:CARDINAL ;
BEGIN
    FOR i:=firstQuestion TO lastQuestion DO
        question[i].index  := i;
        question[i].answer := undefined;
    END;
END initQuestions;

PROCEDURE shuffleQuestions (  );
VAR
    i,p:CARDINAL;
BEGIN
    video(msgShuffling,TRUE);
    FOR i:=firstQuestion TO lastQuestion DO
        LOOP
            p:=getrndwithin(firstQuestion,lastQuestion);
            IF question[p].answer=" " THEN
                question[p].answer:=reindexed;
                question[i].index :=p;
                EXIT;
            END;
        END;
    END;
    FOR i:=firstQuestion TO lastQuestion DO
        question[i].answer := undefined;
    END;
    video(msgShuffling,FALSE);
END shuffleQuestions;

(* ------------------------------------------------------------ *)
(* ------------------------------------------------------------ *)

CONST
    minWidth     = 1;
    maxWidth     = 256; (* longest possible line, eh eh *)
    defaultWidth = 74;
VAR
    DEBUG,log,shuffle,resume:BOOLEAN;
    wi:CARDINAL;
    datafile,logfile,u,d,n,e:str128;
    score:INTEGER;
    para:str256; (* should DO *)
    ch:CHAR;
VAR
    parmcount,i,opt,p:CARDINAL;
    S,R:str128;
    state:(waiting);
BEGIN
    Lib.DisableBreakCheck();
    FIO.IOcheck := FALSE;
    WrLn;

    DEBUG  := FALSE;
    log    := FALSE;
    shuffle:= TRUE;
    resume := FALSE;
    wi     := defaultWidth;
    Str.Copy(datafile,"");
    Str.Copy(logfile,"");

    state:=waiting;
    parmcount := Lib.ParamCount();
    (* IF parmcount=0 THEN abort(errHelp,"");END; *)
    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i);
        Str.Copy(R,S);
        UpperCase(R);
        RtrimBlanks(R); (* try and fix Yet Another TopSpeed bug ! *)
        IF isOption(R) THEN
            opt := GetOptIndex (R, "?"+delim+"H"+delim+"HELP"+delim+
                                   "DEBUG"+delim+
                                   "S"+delim+"SAVE"+delim+
                                   "O"+delim+"ORDERED"+delim+
                                   "R"+delim+"RESUME"+delim+
                                   "W:"+delim+"WIDTH:"+delim+
                                   "L:"+delim+"LOG:"+delim+
                                   "D:"+delim+"DATA:"
                               );
            CASE opt OF
            | 1,2,3 : abort(errHelp,"");
            | 4     : DEBUG := TRUE;
            | 5,6   : log := TRUE;
            | 7,8   : shuffle:=FALSE;
            | 9,10  : resume:=TRUE;
            | 11,12 : IF getCard(wi,minWidth,maxWidth,S)=FALSE THEN
                          abort(errBadWidth,S);
                      END;
            | 13,14 : GetString(R,logfile);
            | 15,16 : GetString(R,datafile);
            ELSE
                abort(errOption,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            CASE state OF
            | waiting:
                abort(errParameter,S);
            END;
            INC(state);
        END;
    END;

    IF ( (DEBUG=FALSE) AND IsRedirected() ) THEN abort(errRedirected,"");END;

    IF (resume AND (shuffle=FALSE) ) THEN abort(errNotWithResume,"");END;

    IF same(datafile,"") THEN
        Lib.ParamStr(datafile,0);
        UpperCase(datafile); (* useless but just in case... *)
        Str.Subst(datafile,extEXE,extDAT);
    ELSE
        IF Str.CharPos(datafile,dot)=MAX(CARDINAL) THEN
            Str.Append(datafile,extDAT);
        END;
    END;
    IF FIO.Exists(datafile)=FALSE THEN abort(errDataNotFound,datafile);END;

    IF same(logfile,"") THEN
        Lib.SplitAllPath(datafile,u,d,n,e);
        Lib.MakeAllPath(logfile,u,d,n,extLOG);
    ELSE
        IF Str.CharPos(logfile,dot)=MAX(CARDINAL) THEN
            Str.Append(logfile,extLOG);
        END;
    END;

    Lib.RANDOMIZE;

    WrStr(Banner);WrLn;
    WrLn;

    i:=loadData(DEBUG,datafile,wi,R);
    IF i # errNone THEN abort(i,R);END;

    initQuestions;

    IF resume THEN
        IF FIO.Exists(logfile)=FALSE THEN abort(errLogNotFound,logfile);END;
        IF loadLog(DEBUG,logfile)=FALSE THEN abort(errBadLog,logfile);END;
        WrStr(prefixinfo);WrStr(logfile);WrStr(msgLoaded);WrLn;
        WrLn;
    ELSE
        IF shuffle THEN shuffleQuestions();END;
    END;

    IF DEBUG THEN
        FOR i:=firstQuestion TO lastQuestion DO
            p:=question[i].index;
            IO.WrCard(p,4);
            IO.WrCard(question[p].weight,4);
            WrStr(sep);
            WrStr(question[p].text);WrLn;
        END;
    END;

    score := 0;
    i:=firstQuestion;
    LOOP
        p:=question[i].index;
        WrStr(msgQuestion);IO.WrCard(i,3);
        WrStr(msgOutOf);IO.WrCard(lastQuestion,3);WrLn;
        WrLn;
        Str.Concat(para,question[p].text,msgChoice);
        dump(wi,FALSE,para);
        IF resume THEN
            ch := question[p].answer;
            CASE ch OF
            | undefined,reindexed: ch:=getchoice(legalchoices);
            END;
        ELSE
            ch:=getchoice(legalchoices);
        END;
        video(msgChoice,FALSE);
        question[p].answer := ch;
        CASE ch OF
        | "Y":
            INC(score,question[p].weight);
            WrStr(msgYes);WrLn;
        | "N":
            WrStr(msgNo);WrLn;
        | "Q",CHR(27):
            question[p].answer := undefined; (* reset it ! *)
            WrStr(msgQuit);WrLn;
            WrLn;
            IF log THEN saveLog(DEBUG,logfile); WrLn;END;
            abort(errAborted,"");
        END;
        WrLn;
        INC(i);
        IF i > lastQuestion THEN EXIT; END;
    END;
    WrStr(msgYourScore);
    IO.WrInt(score,4);WrLn;
    WrLn;
    p:=0;
    FOR i:=firstScoring TO lastScoring DO
        IF score <= scoring[i].upper THEN
            IF p=0 THEN
                dump(wi,TRUE,scoring[i].text);
                p:=MAX(CARDINAL);
            END;
        END;
    END;

    IF log THEN WrLn;saveLog(DEBUG,logfile);END;
    abort(errNone,"");
END QuizYN.
