(* ---------------------------------------------------------------
Title         see help
Overview      see help
Usage         see help
Notes         very, very, very quick & dirty... :-(
              minimal error messages and checking, etc.
Bugs
Wish List

--------------------------------------------------------------- *)

MODULE Columns;

IMPORT Lib;
IMPORT FIO;
IMPORT Str;
IMPORT SYSTEM;
IMPORT IO;

FROM QD_Box IMPORT str80, str2, cmdInit, cmdShow, cmdStop, delim,
Work, video, Ltrim, Rtrim, UpperCase, LowerCase, ReplaceChar,
ChkEscape, Waitkey, WaitkeyDelay, Flushkey, IsRedirected, chkJoker,
isOption, GetOptIndex, GetLongCard, GetLongInt, GetString, CharCount,
same, aR, aH, aS, aD, aA, everything, isDirectory, fixDirectory,
str128, str256, Animation, allfiles, Belongs, FixAE, CodePhonetic,
CodeSoundex, CodeSoundexOrg, isReadOnly, LtrimBlanks, RtrimBlanks,
getStrIndex, cmdSHOW,BiosWaitkey,BiosWaitkeyShifted,BiosFlushkey,
str1024, isoleItemS, dmpTTX, str2048, Elapsed, TerminalReadString,
getDosVersion, DosVersion, warning95, runningWindows,
aV, reallyeverything, chkClassicTextMode, setClassicTextMode,
AltAnimation, str16, getCurrentDirectory, setReadWrite,
getFileSize, verifyString, str4096, unfixDirectory,
animShow, animSHOW, animAdvance, animEnd, animClear,
animInit, animGetSdone, anim, cleantabs,
completedInit, completedShow, completedSHOW, completedEnd, completed;

FROM IO IMPORT WrStr, WrLn;

(* ------------------------------------------------------------ *)

CONST
    progEXEname     = "COLUMNS";
    progTitle       = "Q&D Columns Processor";
    progVersion     = "v1.0l";
    progCopyright   = "by PhG";
    banner          = progTitle+" "+progVersion+" "+progCopyright;

CONST
    cr              = CHR(13);
    lf              = CHR(10);
    nl              = cr+lf;
    space           = " ";
    tabchar         = CHR(9);
    EOLcmd          = "*";
    extBAK          = ".BK!";
    extCOM          = ".COM";
    extEXE          = ".EXE";
    extDLL          = ".DLL";
    extOVR          = ".OVR";
    extOVL          = ".OVL";
    extDRV          = ".DRV";
    extZIP          = ".ZIP";
    extARJ          = ".ARJ";
    extLZH          = ".LZH";
    extensions      = extBAK+delim+extCOM+delim+extEXE+delim+
                      extDLL+delim+extOVR+delim+extOVL+delim+extDRV+delim+
                      extZIP+delim+extARJ+delim+extLZH;
    coma            = ","; (* french fractional part separator *)
    dot             = "."; (* american id. *)
    star            = "*";
    stardotstar     = star+dot+star;
CONST
    doDelete = 1;
    doExtract= 2;
    doCard   = 3;
    doInt    = 4;
    doFloat  = 5;
CONST
    steps    = 10;
    chFiller = "."; (* because of vindoze ! *)
CONST
    errNone         = 0;
    errHelp         = 1;
    errOption       = 2;
    errTooManyParms = 3;
    errBadCard      = 4;
    errTabRange     = 5;
    errConflict     = 6;
    errColumnRange  = 7;
    errSyntax       = 8;
    errMissingCmd   = 9;
    errNonsenseRange= 10;
    errJoker        = 11;
    errNotFound     = 12;
    errAlready      = 13;
    errSame         = 14;
    errTooMany      = 15;
    errNonsense     = 16;
    errAddToTarget  = 17;
    errRawUntab     = 18;
    errTotalMode    = 19;
    errSubMode      = 20;
    errBadSourceExt = 21;
    errBadTargetExt = 22;
    errSubMethod    = 23;
    errAborted      = 24;

PROCEDURE abort (e : CARDINAL; einfo : ARRAY OF CHAR);
CONST
(*
 00000000011111111112222222222333333333344444444445555555555666666666677777777778
 1...'....0....'....0....'....0....'....0....'....0....'....0....'....0....'....0
*)

errmsg =
banner+nl+
nl+
"Syntax : "+progEXEname+" <command> <first> <last|"+EOLcmd+"> <source> [target] [option]..."+nl+
nl+
"This program deletes or keeps text from <first> column to <last> column,"+nl+
"or it adds values found in specified column range after triming."+nl+
nl+
"  -d[t]   delete (-dt = -d -t)"+nl+
"  -k|e[t] keep (-kt = -k -t = -et = -e -t)"+nl+
"  -a      add decimal LONGCARD values"+nl+
"  -ai     add decimal LONGINT values"+nl+
"  -af[f]  add decimal LONGREAL values (-aff = -af -us)"+nl+
"  -s[g]   show each subtotal when <first..last> is empty (-sg = -s -g)"+nl+
"  -z[g]   show each subtotal when out of <first..last> changes (-zg = -z -g)"+nl+
"  -i      insert subtotals in <source> dumped to standard output"+nl+
"  -g      show grand total"+nl+
"  -c      see <last> value as a count, not as an absolute position (default)"+nl+
"  -t:#    tabulation width ([1..64], default is 8)"+nl+
"  -r      do not expand tab into spaces, interpreting it as a raw single space"+nl+
"  -t      test mode (dump result to screen without changing file)"+nl+
"  -o      overwrite existing <target>"+nl+
"  -v      verbose (slower)"+nl+
'  -us     when adding, delimit fractional part with "'+dot+'" (default is "'+coma+'")'+nl+
"  -f[f]   when adding, format total (-ff = -f -us)"+nl+
nl+
"Each line is at most 4096 characters. End of line is specified with "+EOLcmd+" marker."+nl+
"Without [target] specification, backups are always created with "+extBAK+" extension."+nl+
"Note program defaults to tab expansion, unless -r is specified."+nl+
"When adding, each substring is filtered to [0..9+-"+coma+dot+"] character set."+nl+
"With -a and -ai, any character from [ "+coma+dot+"] set is a valid cosmetic separator."+nl+
extensions+" files will be ignored."+nl+
"Warning : when adding, invalid data will lead to weird and wrong results !"+nl;

VAR
    S : str256;
BEGIN
    CASE e OF
    | errHelp :
        WrStr(errmsg);
    | errOption:
        Str.Concat(S,"Unknown ",einfo);Str.Append(S," option !");
    | errTooManyParms:
        Str.Concat(S,einfo," parameter is one too many !");
    | errBadCard:
        Str.Concat(S,"Illegal ",einfo);Str.Append(S," value !");
    | errTabRange:
        S := "Tabulation count must be in the [1..64] range !";
    | errConflict:
        S := "-d, -e, -a, -ai and -af commands are mutually exclusive !";
    | errColumnRange:
        S := "Column must be in the [1..4096] range !";
    | errSyntax:
        S := "Syntax error !"; (* in fact, missing parms *)
    | errMissingCmd:
        S := "Missing <command> !";
    | errNonsenseRange:
        S := "<last> column is smaller than <first> column !";
    | errJoker    :
        Str.Concat(S,"At least one illegal joker in ",einfo);
        Str.Append(S," !");
    | errNotFound :
        (* Str.Concat(S,einfo," does not exist !"); *)
        Str.Concat(S,"No legal file matching ",einfo);Str.Append(S," !");
    | errAlready  :
        Str.Concat(S,einfo," already exist !");
    | errSame:
        S:="<source> and <target> are identical !";
    | errTooMany:
        S := "Too many files match <source> !";
    | errNonsense:
        S := "-t and -v commands are mutually exclusive !";
    | errAddToTarget:
        S := "Target specification is a nonsense with any add command !";
    | errRawUntab:
        S := "-r and -t:# options are mutually exclusive !";
    | errTotalMode:
        S := "-g option requires -s option !";
    | errSubMode:
        S := "-i option requires -s option !";
    | errBadSourceExt:
        Str.Concat(S,einfo," source has an illegal extension !");
    | errBadTargetExt:
        Str.Concat(S,einfo," target has an illegal extension !");
    | errSubMethod:
        S := "-s[g] and -z[g] options are mutually exclusive !";
    | errAborted:
        S := "Aborted by user !";
    ELSE
        S := "This is illogical, Captain !";
    END;
    CASE e OF
    | errNone,errHelp:
        ;
    ELSE
        WrStr(progEXEname+" : ");WrStr(S);WrLn;
    END;
    Lib.SetReturnCode(SHORTCARD(e));
    HALT;
END abort;

(* ------------------------------------------------------------ *)

PROCEDURE detab (tabwidth:CARDINAL;S:ARRAY OF CHAR; VAR R : ARRAY OF CHAR  );
VAR
    i,j,add: CARDINAL;
    c : CHAR;
BEGIN
    Str.Copy(R,"");
    j:=0; (* yes, 0 and not 1 ! *)
    FOR i:=1 TO Str.Length(S) DO
        c := S[i-1];
        IF c = tabchar THEN
            add := tabwidth - (j MOD tabwidth);
            WHILE add > 0 DO
                Str.Append(R,space); INC(j);
                DEC(add);
            END;
        ELSE
            Str.Append(R,c); INC(j);
        END;
    END;
END detab;

CONST
    ioBufferSize    = (8 * 512) + FIO.BufferOverhead;
    firstBufferByte = 1;
    lastBufferByte  = ioBufferSize;
TYPE
    ioBuffer  = ARRAY [firstBufferByte..lastBufferByte] OF BYTE;
VAR
    bufferIn  : ioBuffer;
    bufferOut : ioBuffer;

(* we already know test and verbose cannot be the same *)

PROCEDURE process (cmd,lower,upper,tabwidth:CARDINAL;
                  test,verbose,expandtab:BOOLEAN;
                  source,target:ARRAY OF CHAR );
VAR
    hin,hout : FIO.File;
    S,R      : str4096;
    pos,count    : CARDINAL;
    fsize,portion,lastportion,currportion,addr : LONGCARD;

BEGIN
    fsize := getFileSize(source);
    portion:=fsize DIV LONGCARD(steps); INC(portion); (* avoid DIV 0 ! *)
    lastportion := LONGCARD(steps+1);

    (* IF verbose THEN Work(cmdInit); END; *)
    IF verbose THEN animInit(steps, "[", "]", chFiller, "", "\/" ); END;

    pos   := lower -1;
    count := upper - lower + 1;

    hin := FIO.OpenRead(source);
    FIO.AssignBuffer(hin,bufferIn);
  IF NOT(test) THEN (* yes, yes, we could have used stdout or stderr handle *)
    hout:= FIO.Create(target);
    FIO.AssignBuffer(hout,bufferOut);
  END;

    FIO.EOF:=FALSE;
    LOOP
        IF FIO.EOF THEN EXIT; END;
        (* IF verbose THEN Work(cmdShow); END; *)
        IF verbose THEN
            addr:=FIO.GetPos(hin);
            anim(animShow);
            currportion:=addr DIV portion;
            IF currportion # lastportion THEN
                anim(animAdvance);
                lastportion:=currportion;
            END;
        END;

        FIO.RdStr(hin,S);
        IF FIO.EOF THEN EXIT; END;

        IF expandtab THEN detab(tabwidth,S,R); END;

        CASE cmd OF
        | doDelete:
            Str.Delete(R,pos,count);
            IF test THEN
                WrStr(R);WrLn;
            ELSE
                FIO.WrStr(hout,R);FIO.WrLn(hout);    (* R *)
            END;
        | doExtract:
            Str.Slice(S,  R,pos,count);
            IF test THEN
                WrStr(S);WrLn;
            ELSE
                FIO.WrStr(hout,S);FIO.WrLn(hout);    (* S ! *)
            END;
        END;

    END;
  IF NOT(test) THEN
    FIO.Flush(hout);
    FIO.Close(hout);
  END;
    FIO.Close(hin);

    (* IF verbose THEN Work(cmdStop); END; *)
    IF verbose THEN anim(animEnd);anim(animClear); END;
END process;

(* ------------------------------------------------------------ *)

PROCEDURE fmtlc (v : LONGCARD; pad:CHAR; sep:CHAR; field:INTEGER) : str80;
VAR
    S,R   : str80;
    len,i : CARDINAL;
    ok  : BOOLEAN;
    ch  : CHAR;
BEGIN
    Str.CardToStr(v,S,10,ok);
    len:=Str.Length(S);
    R := "";
    FOR i := 1 TO len DO
        Str.Prepend(R,S[len-i]);
        IF i < len THEN
            IF (i MOD 3) = 0 THEN
                Str.Prepend(R,sep);
            END;
        END;
    END;
    LOOP
        IF INTEGER(Str.Length(R)) >= ABS(field) THEN EXIT; END;
        IF field < 0 THEN
            Str.Append(R,pad);  (* right alignment *)
        ELSE
            Str.Prepend(R,pad);
        END;
    END;
    RETURN R;
END fmtlc;

PROCEDURE fmtli (v : LONGINT; pad:CHAR; sep:CHAR; field:INTEGER) : str80;
VAR
    S,R   : str80;
    len,i : CARDINAL;
    ok  : BOOLEAN;
    ch  : CHAR;
BEGIN
    Str.IntToStr(v,S,10,ok);
    len:=Str.Length(S);
    R := "";
    FOR i := 1 TO len DO
        Str.Prepend(R,S[len-i]);
        IF i < len THEN
            IF (i MOD 3) = 0 THEN
                Str.Prepend(R,sep);
            END;
        END;
    END;
    LOOP
        IF INTEGER(Str.Length(R)) >= ABS(field) THEN EXIT; END;
        IF field < 0 THEN
            Str.Append(R,pad);  (* right alignment *)
        ELSE
            Str.Prepend(R,pad);
        END;
    END;
    RETURN R;
END fmtli;

PROCEDURE fmthc (v : LONGREAL; pad:CHAR; sep,fracsep:CHAR; field,prec:INTEGER) : str80;
VAR
    S,SF,R   : str80;
    len,i,p : CARDINAL;
    ok,frachere  : BOOLEAN;
    ch  : CHAR;
BEGIN
    Str.FixRealToStr( v,prec,S,ok);
    p:=Str.RCharPos(S,dot);
    IF p=MAX(CARDINAL) THEN
        frachere:=FALSE;
    ELSE
        Str.Copy(SF,S); Str.Delete(SF,0,p+1);
        S[p]:=CHR(0); (* brutal ! *)
        frachere:=(prec # 0);
    END;

    len:=Str.Length(S);
    R := "";
    FOR i := 1 TO len DO
        Str.Prepend(R,S[len-i]);
        IF i < len THEN
            IF (i MOD 3) = 0 THEN
                Str.Prepend(R,sep);
            END;
        END;
    END;
    LOOP
        IF INTEGER(Str.Length(R)) >= ABS(field) THEN EXIT; END;
        IF field < 0 THEN
            Str.Append(R,pad);  (* right alignment *)
        ELSE
            Str.Prepend(R,pad);
        END;
    END;

    IF frachere THEN
        Str.Append(R,fracsep);Str.Append(R,SF);
    END;

    RETURN R;
END fmthc;

PROCEDURE dmptotal (beautify:BOOLEAN;errors,opcode:CARDINAL;
                    totalcard:LONGCARD;totalint:LONGINT;totalfloat:LONGREAL;
                    killfrac,fracsep:CHAR;fieldfmt,fracwi,field,lfield:CARDINAL;
                    prefix,suffix,source:ARRAY OF CHAR);
VAR
    ok:BOOLEAN;
    R:str128;
BEGIN
    IF errors = 0 THEN
        Str.Copy(R,"::: ");
    ELSE
        Str.Copy(R,"--- ");
    END;
    Str.Append(R,prefix);
    Str.Append(R," = ");
    WrStr(R);

    IF beautify THEN
        CASE opcode OF
        | doCard: Str.Copy(R,fmtlc (totalcard ," ",killfrac, fieldfmt ));
        | doInt:  Str.Copy(R,fmtli (totalint  ," ",killfrac, fieldfmt ));
        | doFloat:Str.Copy(R,fmthc (totalfloat," ",killfrac,fracsep, fieldfmt,fracwi));
        END;
        WrStr(R);
    ELSE
        CASE opcode OF
        | doCard: IO.WrLngCard(totalcard,field);
        | doInt:  IO.WrLngInt (totalint,field);
        | doFloat:Str.FixRealToStr(totalfloat,fracwi,R,ok);WrStr(R);
        END;
    END;

    WrStr(" (");WrStr(source);WrStr(")");
    IF errors = 0 THEN
        WrStr(suffix);
    ELSE
        WrStr(" --- ");
        IO.WrCard(errors,lfield);
        WrStr(" error(s)");
    END;
    WrLn;

END dmptotal;

(* ------------------------------------------------------------ *)

PROCEDURE DMPDBG ( prefix,S:ARRAY OF CHAR  );
BEGIN
    WrStr(prefix); WrStr('"'); WrStr(S); WrStr('"'); WrLn;
END DMPDBG;

PROCEDURE swap (VAR i,j:CARDINAL   );
VAR
    tmp:CARDINAL;
BEGIN
    tmp:=i;
    i:=j;
    j:=tmp;
END swap;

PROCEDURE addlines (opcode,lower,upper,tabwidth:CARDINAL;
                   test,verbose,expandtab,beautify,
                   subtotals,smarttotals,insertmode,grandtotal,DEBUG:BOOLEAN;
                   fracsep:CHAR;source:ARRAY OF CHAR);
CONST
    ndxfirst= 1;
    ndxlast = 2;
VAR
    orgS    :ARRAY [ndxfirst..ndxlast] OF str4096;
    category:ARRAY [ndxfirst..ndxlast] OF str4096;
CONST
    field    = 1+3*3+1;     (* " # ### ### ###" *)
    fieldfmt = 1+3*3+1+3;   (* "+#,###,###,###" *)
    lfield   = 5;           (* "#####" *)
    prefTotal      = "Total";
    prefGrandTotal = "Grand Total";
    prefSubTotal   = "Sub Total";
    prefSubGrand   = "Sub Total  ";
    remark         = " ; ";
VAR
    pos,count:CARDINAL;
    hin:FIO.File;
    S,R      : str4096;
    ndxcurr,ndxprev:CARDINAL;
VAR
    Z:str256;
    grandcard, vlc,totalcard : LONGCARD;
    grandint,  vli,totalint  : LONGINT;
    grandfloat,vlf,totalfloat: LONGREAL;
    fixvalue,reread,doswap,dmpsubtotal,dmpcurrline,ok:BOOLEAN;
    base:CARDINAL;
    currline,fracwi,p,wi,errors,granderrors:CARDINAL;
    killfrac:CHAR;
    prefix,suffix:str128;
    dmpmethod:(nothing,onlysubs,subsatsepempty,subsatsepchange);
    grabbing:(waiting,gotref,newref);
    fanchor:LONGCARD;
BEGIN
    IF DEBUG THEN verbose:=FALSE;END;

    CASE fracsep OF
    | coma: killfrac:=dot;
    | dot:  killfrac:=coma;
    END;

    IF subtotals THEN
        IF insertmode THEN
            IF smarttotals THEN
                dmpmethod:=subsatsepchange;
            ELSE
                dmpmethod:=subsatsepempty;
            END;
        ELSE
            dmpmethod:=onlysubs;
        END;
    ELSE
        dmpmethod:=nothing;
    END;
IF DEBUG THEN
CASE dmpmethod OF
| nothing: S:="nothing;"
| onlysubs: S:="onlysubs";
|subsatsepempty: S:="subs at sep empty";
|subsatsepchange: S:="subs at sep change";
END;
DMPDBG("??? ",S);
END;

    IF subtotals THEN
        IF grandtotal THEN
           prefix:=prefSubGrand;
        ELSE
           prefix:=prefSubTotal;
        END;
    ELSE
        prefix:=prefTotal;
    END;

    granderrors := 0;
    grandcard   := 0;
    grandint    := 0;
    grandfloat  := 0.0;

    errors:=0;

    CASE opcode OF
    | doCard:  totalcard :=0;   Z:="LONGCARD";
    | doInt:   totalint  :=0;   Z:="LONGINT";
    | doFloat: totalfloat:=0.0; Z:="LONGREAL";
    END;
    Str.Prepend(Z,"Adding "); Str.Append(Z,"s from "); Str.Append(Z,source);

    pos      := lower -1;
    count    := upper - lower + 1;

    base     := 10; (* for now *)
    currline := 0;
    fracwi   := 0;

    grabbing := waiting;
    ndxcurr  := ndxfirst;
    ndxprev  := ndxlast;
    FOR p:=ndxfirst TO ndxlast DO
         orgS[p]:="";
         category[p]:="";
    END;

    IF verbose THEN video(Z,TRUE); END;

    hin := FIO.OpenRead(source);
    FIO.AssignBuffer(hin,bufferIn);

    FIO.EOF:=FALSE;
    LOOP
        IF FIO.EOF THEN EXIT; END;
        IF dmpmethod = subsatsepchange THEN fanchor:=FIO.GetPos(hin);END;
        FIO.RdStr(hin,S);
IF DEBUG THEN DMPDBG("<<< ",S);END;
        IF FIO.EOF THEN EXIT; END;
        Str.Copy(orgS[ndxcurr],S);
        INC(currline);

        IF expandtab THEN
            detab(tabwidth,S,R);
        ELSE
            Str.Copy(R,S);
        END;
        Str.Slice(S,  R,pos,count);

IF DEBUG THEN DMPDBG("::: ",S);END;

        IF dmpmethod = subsatsepchange THEN
            Str.Copy(category[ndxcurr],R);
            Str.Delete(category[ndxcurr],pos,count);
            LtrimBlanks(category[ndxcurr]);
            RtrimBlanks(category[ndxcurr]);
            LowerCase(category[ndxcurr]);
        END;

        ReplaceChar(S,space,"");
        ReplaceChar(S,tabchar,"");
        IF S[0]="+" THEN Str.Delete(S,0,1);END;

        CASE opcode OF
        | doCard,doInt:
            ReplaceChar(S,dot,"");
            ReplaceChar(S,coma,"");
        | doFloat:
            ReplaceChar(S,killfrac,"");
            IF fracsep=coma THEN Str.Subst(S,coma,dot); END;
            p:=Str.RCharPos(S,dot);
            IF p # MAX(CARDINAL) THEN
                wi:=Str.Length(S)-(p+1);
                IF wi > fracwi THEN fracwi:=wi;END;
            END;
        END;

IF DEBUG THEN DMPDBG(">>> ",S);END; (* S is extracted number *)

        reread := FALSE;
        suffix:="";
        CASE dmpmethod OF
        | nothing:
             dmpsubtotal:=FALSE;
             dmpcurrline:=FALSE;
        | onlysubs:
             dmpsubtotal:=same(S,"");
             dmpcurrline:=FALSE;
        | subsatsepempty:
             dmpsubtotal:=same(orgS[ndxcurr],""); (* was S before fix v1.0l *)
             dmpcurrline:=NOT(dmpsubtotal);
        | subsatsepchange:

IF DEBUG THEN DMPDBG("PREV ",category[ndxprev]);END;
IF DEBUG THEN DMPDBG("CURR ",category[ndxcurr]);END;

            CASE grabbing OF
            | waiting: (* only once *)
                Str.Copy(orgS[ndxprev],orgS[ndxcurr]);
                Str.Copy(category[ndxprev],category[ndxcurr]);
                dmpsubtotal:=FALSE;
                dmpcurrline:=TRUE;
                grabbing:=gotref;
            | gotref:
                IF same(category[ndxcurr],category[ndxprev]) THEN
                    dmpsubtotal := FALSE;
                    dmpcurrline := TRUE;
                ELSE
                    dmpsubtotal := TRUE;
                    dmpcurrline := FALSE; (* not yet *)
                    grabbing:=newref;
                    reread  := TRUE; (* trick *)
                    Str.Concat(suffix,remark,category[ndxprev]); (* easier to read *)
                END;
            | newref : ; (* never here *)
            END;
        END;
IF DEBUG THEN
DMPDBG("+++ ",orgS[ndxcurr]);
END;
        IF dmpcurrline THEN
            IF verbose THEN video(Z,FALSE);END;
            WrStr(orgS[ndxcurr]);WrLn;
            IF verbose THEN video(Z,TRUE);END;
        END;

        IF reread THEN
            fixvalue:=TRUE; (* fake it... *)
        ELSE
            fixvalue:=same(S,"");
        END;
        IF fixvalue THEN Str.Copy(S,"0"); END; (* extract *)

        CASE opcode OF
        | doCard:
            IF S[0]="-" THEN
                ok:=FALSE;
            ELSE
                vlc:=Str.StrToCard(S,base,ok); IF ok THEN INC(totalcard,vlc);END;
            END;
        | doInt:
            vli:=Str.StrToInt(S,base,ok);  IF ok THEN INC(totalint,vli);END;
        | doFloat:
            vlf:=Str.StrToReal(S,ok); IF ok THEN totalfloat:=totalfloat+vlf;END;
        END;
        IF NOT(ok) THEN
            IF verbose THEN video(Z,FALSE); END;
            WrStr("--- Problem at line ");
            IO.WrCard(currline,1);
            WrStr(' ("');WrStr(S);WrStr('")');WrLn;
            IF verbose THEN video(Z,TRUE); END;
            INC(errors);
        END;
        IF dmpsubtotal THEN

            IF verbose THEN video(Z,FALSE); END;
            dmptotal (beautify,errors,opcode,
                      totalcard,totalint,totalfloat,
                      killfrac,fracsep,fieldfmt,fracwi,field,lfield,
                      prefix,suffix,source);
            IF verbose THEN video(Z,TRUE); END;

            INC(granderrors,errors);
            INC(grandcard,totalcard);
            INC(grandint,totalint);
            grandfloat:=grandfloat+totalfloat;

            errors:=0;

            CASE opcode OF
            | doCard:  totalcard :=0;
            | doInt:   totalint  :=0;
            | doFloat: totalfloat:=0.0;
            END;

        END;

        doswap:=FALSE;
        CASE dmpmethod OF
        | nothing,onlysubs: ;
        | subsatsepempty:
            IF dmpcurrline THEN dmpcurrline:=FALSE;END;
        | subsatsepchange:

            dmpcurrline:=NOT(dmpcurrline);
            CASE grabbing OF
            | waiting: ; (* never here *)
            | gotref :
                dmpcurrline:= FALSE;
            | newref :
                dmpcurrline:=FALSE; (* we'll reread it *)
                grabbing:=gotref;   (* waiting works too *)
                doswap:=TRUE;
                FIO.Seek(hin,fanchor); (* be sure to reread current line *)
            END;
        END;
        IF dmpcurrline THEN
            IF verbose THEN video(Z,FALSE);END;
            WrStr(orgS[ndxcurr]);WrLn;
            IF verbose THEN video(Z,TRUE);END;
        END;
        IF doswap THEN swap(ndxcurr,ndxprev);END;
    END;
    FIO.Close(hin);

    (* reread trick removes need to check subsatsepchange state *)

    IF dmpmethod = subsatsepchange THEN
        Str.Concat(suffix,remark,category[ndxcurr]);
    ELSE
        suffix:=""; (* useless safety *)
    END;

    IF verbose THEN video(Z,FALSE); END;

    dmptotal (beautify,errors,opcode,
              totalcard,totalint,totalfloat,
              killfrac,fracsep,fieldfmt,fracwi,field,lfield,
              prefix,suffix,source);

    INC(granderrors,errors);
    INC(grandcard,totalcard);
    INC(grandint,totalint);
    grandfloat:=grandfloat+totalfloat;

    suffix:="";
    IF grandtotal THEN
        dmptotal (beautify,granderrors,opcode,
                 grandcard,grandint,grandfloat,
                 killfrac,fracsep,fieldfmt,fracwi,field,lfield,
                 prefGrandTotal,suffix,source);
    END;

END addlines;

(* ------------------------------------------------------------ *)

PROCEDURE legalextension (S:ARRAY OF CHAR):BOOLEAN;
VAR
    e3 : str16;
    n:CARDINAL;
    rc:BOOLEAN;
BEGIN

    Str.Caps(S); (* ah, lowercase LFNs... *)

    rc:=TRUE;
    n:=0;
    LOOP
        isoleItemS(e3, extensions,delim,n);
        IF same(e3,"") THEN EXIT; END;
        IF Str.Pos(S,e3) # MAX(CARDINAL) THEN rc:=FALSE;EXIT; END;
        INC(n);
    END;
    RETURN rc;
END legalextension;

TYPE
    f8e3 = ARRAY [0..8+1+3-1] OF CHAR;
CONST
    firstFile = 1;
    lastFile  = 2000;
    maxFile   = lastFile-firstFile+1;
VAR
    fileArray : ARRAY [firstFile..lastFile] OF f8e3;

PROCEDURE buildList (spec:ARRAY OF CHAR):CARDINAL;
VAR
    countFile : CARDINAL;
    found     : BOOLEAN;
    entry     : FIO.DirEntry;
BEGIN
    FIO.IOcheck := FALSE;
    countFile := 0;
    found := FIO.ReadFirstEntry(spec,allfiles,entry);
    WHILE found DO
        IF countFile = maxFile THEN RETURN MAX(CARDINAL); END;
        IF legalextension(entry.Name) THEN (* skip *.bk!, *.com and *.exe entries *)
            fileArray[firstFile+countFile]:=f8e3(entry.Name);
            Str.Caps(fileArray[firstFile+countFile]); (* useless ! *)
            INC (countFile);
        END;
        found :=FIO.ReadNextEntry(entry);
    END;
    RETURN countFile;
END buildList;

(* ------------------------------------------------------------ *)

PROCEDURE chkRange (lc:LONGCARD;lower,upper:CARDINAL;VAR v:CARDINAL):BOOLEAN ;
BEGIN
    IF lc < LONGCARD (lower) THEN RETURN FALSE; END;
    IF lc > LONGCARD (upper) THEN RETURN FALSE; END;
    v := CARDINAL(lc);
    RETURN  TRUE;
END chkRange;

(* ------------------------------------------------------------ *)

CONST
    undefinedcard=MAX(CARDINAL);
    defaulttabwidth = 8;
    mintabwidth = 1;
    maxtabwidth = 64;
    mincolumn   = 1;
    maxcolumn   = 4096;
    ascmaxcol   = "4096";
VAR
    cmd         : (undefined,delete,extract,addcard,addint,addfloat);
    absolute    : BOOLEAN;
    tabwidth    : CARDINAL;
    test        : BOOLEAN;
    overwrite   : BOOLEAN;
    verbose     : BOOLEAN;
    fracsep     : CHAR;
    expandtab   : BOOLEAN;
    beautify    : BOOLEAN;
    subtotals   : BOOLEAN;
    smarttotals : BOOLEAN;
    insertmode  : BOOLEAN;
    grandtotal  : BOOLEAN;
    DEBUG       : BOOLEAN;
    firstcolumn : CARDINAL;
    lastcolumn  : CARDINAL;
    source,target,bakfile : str128;
    count       : CARDINAL;
    opcode      : CARDINAL;
    u,path,f8,e3: str128; (* oversized just in case *)
    basepath    : str128;
    redirON     : BOOLEAN;
VAR
    parmcount,i,opt : CARDINAL;
    S,R             : str128;
    state           : (waiting,gotfirst,gotlast,gotsource,gottarget);
    v               : LONGCARD;
BEGIN
    Lib.DisableBreakCheck(); (* if enabled, Ctrl-C forces an ERRORINF.$$$ creation ! *)
    WrLn;

    redirON := IsRedirected();

    cmd       := undefined;
    absolute  := TRUE;
    tabwidth  := undefinedcard;
    test      := FALSE;
    overwrite := FALSE;
    verbose   := FALSE;
    fracsep   := coma;
    expandtab := TRUE;
    beautify  := FALSE;
    subtotals := FALSE;
    smarttotals:=FALSE;
    insertmode:= FALSE;
    grandtotal:= FALSE;
    DEBUG     := FALSE;

    state     := waiting;
    parmcount := Lib.ParamCount();
    IF parmcount=0 THEN abort(errHelp,"");END;
    FOR i := 1 TO parmcount DO
        Lib.ParamStr(S,i);
        Str.Copy(R,S);
        UpperCase(R);
        cleantabs(R);
        IF isOption(R) THEN
            opt := GetOptIndex (R, "?"+delim+"H"+delim+"HELP"+delim+
                                   "C"+delim+"COUNT"+delim+
                                   "T:"+delim+"TAB:"+delim+
                                   "O"+delim+"OVERWRITE"+delim+
                                   "D"+delim+"DELETE"+delim+
                                   "E"+delim+"EXTRACT"+delim+"X"+delim+
                                   "V"+delim+"VERBOSE"+delim+
                                   "T"+delim+"TEST"+delim+
                                   "DT"+delim+
                                   "ET"+delim+"XT"+delim+
                                   "A"+delim+
                                   "AI"+delim+
                                   "AF"+delim+
                                   "AFF"+delim+
                                   "US"+delim+
                                   "R"+delim+"RAW"+delim+
                                   "F"+delim+"FORMAT"+delim+"BEAUTIFY"+delim+
                                   "S"+delim+"SUBTOTAL"+delim+
                                   "SG"+delim+
                                   "G"+delim+"GRANDTOTAL"+delim+
                                   "I"+delim+"INSERT"+delim+
                                   "Z"+delim+"SMARTSUBTOTAL"+delim+
                                   "ZG"+delim+
                                   "K"+delim+"KEEP"+delim+
                                   "KT"+delim+
                                   "DEBUG"
                               );
            CASE opt OF
            | 1,2,3 :
                abort(errHelp,"");
            | 4,5   :
                absolute := FALSE;
            | 6,7   :
                IF GetLongCard(S,v)=FALSE THEN abort(errBadCard,S);END;
                IF chkRange(v,mintabwidth,maxtabwidth,tabwidth)=FALSE THEN
                    abort(errTabRange,S);
                END;
            | 8,9   :
                overwrite := TRUE;
            | 10,11 :
                CASE cmd OF
                | undefined,delete: cmd:=delete;
                ELSE                abort(errConflict,"");END;
            | 12,13,14 :
                CASE cmd OF
                | undefined,extract:cmd:=extract;
                ELSE                abort(errConflict,"");END;
            | 15,16 :
                verbose := TRUE;
            | 17,18:
                test := TRUE;
            | 19:
                CASE cmd OF
                | undefined,delete: cmd:=delete; test:=TRUE;
                ELSE                abort(errConflict,"");END;
            | 20,21:
                CASE cmd OF
                | undefined,extract:cmd:=extract;test:=TRUE;
                ELSE                abort(errConflict,"");END;
            | 22:
                CASE cmd OF
                | undefined,addcard:cmd:=addcard;
                ELSE                abort(errConflict,"");END;
            | 23:
                CASE cmd OF
                | undefined,addint:cmd:=addint;
                ELSE                abort(errConflict,"");END;
            | 24:
                CASE cmd OF
                | undefined,addfloat:cmd:=addfloat;
                ELSE                abort(errConflict,"");END;
            | 25:
                CASE cmd OF
                | undefined,addfloat:cmd:=addfloat;fracsep:=dot;
                ELSE                abort(errConflict,"");END;
            | 26:       fracsep:=dot;
            | 27,28:    expandtab:=FALSE;
            | 29,30,31: beautify:=TRUE;
            | 32,33:    subtotals:=TRUE;
            | 34:       subtotals:=TRUE;grandtotal:=TRUE;
            | 35,36:    grandtotal:=TRUE;
            | 37,38:    insertmode:=TRUE;
            | 39,40:    smarttotals:=TRUE;
            | 41:       smarttotals:=TRUE;grandtotal:=TRUE;
            | 42,43 :
                CASE cmd OF
                | undefined,extract:cmd:=extract;
                ELSE                abort(errConflict,"");END;
            | 44:
                CASE cmd OF
                | undefined,extract:cmd:=extract;test:=TRUE;
                ELSE                abort(errConflict,"");END;
            | 45:       DEBUG:=TRUE;
            ELSE
                abort(errOption,S); (* could be errHelp, eh eh ! *)
            END;
        ELSE
            CASE state OF
            | waiting :
                Str.Prepend(R,"=");
                IF GetLongCard(R,v)=FALSE THEN abort(errBadCard,S);END;
                IF chkRange(v,mincolumn,maxcolumn,firstcolumn)= FALSE THEN
                    abort(errColumnRange,S);
                END;
            | gotfirst:
                IF same(R,EOLcmd) THEN R:=ascmaxcol; END;
                Str.Prepend(R,"=");
                IF GetLongCard(R,v)=FALSE THEN abort(errBadCard,S);END;
                IF chkRange(v,mincolumn,maxcolumn,lastcolumn)= FALSE THEN
                    abort(errColumnRange,S);
                END;
            | gotlast:
                Str.Copy(source,R); (* keep uppercase *)
            | gotsource:
                Str.Copy(target,R); (* keep uppercase *)
            | gottarget:
                abort(errTooManyParms,S);
            END;
            INC(state);
        END;
    END;
    (* check nonsense *)

    IF smarttotals THEN
        IF subtotals THEN abort(errSubMethod,"");END;
        subtotals:=TRUE;
    END;

    IF ( grandtotal AND (subtotals=FALSE) ) THEN abort(errTotalMode,"");END;
    IF ( insertmode AND (subtotals=FALSE) ) THEN abort(errSubMode,"");END;

    IF cmd = undefined THEN  abort(errMissingCmd,""); END;

    CASE state OF
    | gotsource: (* autobak *)
        IF chkJoker(source)=FALSE THEN
            IF legalextension(source)=FALSE THEN abort(errBadSourceExt,source);END;
        END;
    | gottarget: (* "There can be only one !" *)
        CASE cmd OF
        | extract,delete: ;
        ELSE
            abort(errAddToTarget,"");
        END;

        IF legalextension(source)=FALSE THEN abort(errBadSourceExt,source);END;
        IF legalextension(target)=FALSE THEN abort(errBadTargetExt,target);END;

        IF same(source,target) THEN abort(errSame,"");END; (* well, minimum check *)
        IF chkJoker(source) THEN abort(errJoker,source);END;
        IF FIO.Exists(source)=FALSE THEN abort(errNotFound,source);END;
        IF chkJoker(target) THEN abort(errJoker,target);END;
        IF FIO.Exists(target) THEN
            IF overwrite THEN
                IF isReadOnly(target) THEN setReadWrite(target);END;
                FIO.Erase(target);
            ELSE
                abort(errAlready,target);
            END;
        END;
    ELSE
        abort(errSyntax,"");
    END;

    IF expandtab = FALSE THEN
        IF tabwidth # undefinedcard THEN abort(errRawUntab,"");END;
    END;
    IF tabwidth = undefinedcard THEN tabwidth:=defaulttabwidth; END;

    IF absolute=FALSE THEN lastcolumn := firstcolumn + lastcolumn -1; END;
    IF lastcolumn < firstcolumn THEN abort(errNonsenseRange,"");END;

    IF test THEN verbose:=FALSE; END; (* cancel -v *)

    IF same (source,dot) THEN Str.Copy(source,stardotstar);END;

    count := buildList(source);
    CASE count OF
    | 0 : abort(errNotFound,source);
    | MAX(CARDINAL): abort(errTooMany,source);
    END;

    CASE cmd OF
    | delete:  i:=doDelete;
    | extract: i:=doExtract;
    | addcard: i:=doCard;
    | addint:  i:=doInt;
    | addfloat:i:=doFloat;
    END;
    opcode:=i;

    Lib.SplitAllPath(source,u,path,f8,e3);
    Str.Concat(basepath,u,path);

    IF NOT (redirON) THEN
        WrStr(banner);WrLn;
        WrLn;
    END;

FOR i := 1 TO count DO
    Str.Concat(source,basepath,fileArray[firstFile+i-1]);
    IF Str.CharPos(source,dot)=MAX(CARDINAL) THEN Str.Append(source,dot);END;
    CASE cmd OF
    | delete,extract:
        CASE state OF
        | gotsource:
            Str.Copy(target,source);
            Lib.SplitAllPath(source,u,path,f8,e3);
            Lib.MakeAllPath(bakfile,u,path,f8,extBAK);
            IF NOT(test) THEN
                IF FIO.Exists(bakfile) THEN
                    IF isReadOnly(bakfile) THEN setReadWrite(bakfile);END;
                    FIO.Erase(bakfile);
                END;
                FIO.Rename(source,bakfile);
                Str.Copy(source,bakfile);
            END;
            IF NOT(test) THEN
                WrStr("Processing ");WrStr(target);
            END;
        | gottarget:
            IF NOT(test) THEN
                WrStr("Creating ");WrStr(target);WrStr(" from ");WrStr(source);
            END;
        END;

        WrStr(" "); (* gap between filename and animation *)

        process (opcode,firstcolumn,lastcolumn,tabwidth,
                test,verbose,expandtab,
                source,target);
        IF NOT(test) THEN
            WrStr(": Done !");WrLn;
        END;
    ELSE
        addlines (opcode,firstcolumn,lastcolumn,tabwidth,
                 test,verbose,expandtab,beautify,
                 subtotals,smarttotals,insertmode,grandtotal,DEBUG,
                 fracsep,source);
        IF ChkEscape() THEN abort(errAborted,"");END;
    END;
END;

    abort(errNone,"");
END Columns.





(*

parano batch to check successive revisions

   c:\bat\columns %1 1 10 -a -g -s     empty.txt  > a
   c:\bat\columns %1 1 10 -a -g -s     change.txt > b
   c:\bat\columns %1 1 10 -a -g -s -i  empty.txt  > c
   c:\bat\columns %1 1 10 -a -g -s -i  change.txt > d

c:\modula\columns %1 1 10 -a -g -s     empty.txt  > aa
c:\modula\columns %1 1 10 -a -g -s     change.txt > bb
c:\modula\columns %1 1 10 -a -g -s -i  empty.txt  > cc
c:\modula\columns %1 1 10 -a -g -s -i  change.txt > dd

c:\modula\columns %1 1 10 -a -g -z     empty.txt  > aaa
c:\modula\columns %1 1 10 -a -g -z     change.txt > bbb
c:\modula\columns %1 1 10 -a -g -z -i  empty.txt  > ccc
c:\modula\columns %1 1 10 -a -g -z -i  change.txt > ddd

*)

