{
* scrquiz   (was crtquiz)
* screen/terminal oriented quiz program
*
* Copyright (c) 2003-2006 Andreas K. Foerster <akfquiz@akfoerster.de>
*
* Environment: FreePascal or GNU-Pascal
*
* This file is part of AKFQuiz
*
* AKFQuiz is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* AKFQuiz is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
*
*}

{ Conditional Defines: 
  NoHide : don't hide Cursor
}

{$X+}
{$R+} { Range checking }

{ compatiblity definition }
{$IfDef _WIN32} {$Define Win32} {$EndIf}

{$IfDef Win32}
  {$R w32/scrquiz}
{$EndIf}

{$IfDef FPC}
  {$Mode Delphi}

  {$IfDef Win32}
    {$AppType Console}
  {$EndIf}
{$EndIf}

{ FPC code for hiding the cursor causes trouble on BSD systems }
{$IfDef BSD} {$Define NoHide} {$EndIf}

program scrquiz(input, output);
uses qsys, qmsgs, uakfquiz, crt;

{$I-}

type keyset = set of char;

type TscreenPos = {$IfDef FPC} byte; {$Else} word; {$EndIf}
{ Byte is a bad choice, but FPC insists on it }

const 
  Esc   = chr(27);
  Enter = chr(13);

{$IfDef scrNoEscKey} { Esc may cause problems }
const 
  ExitKey     = 'Q';
  ExitKeyName = ExitKey;
{$Else}
const 
  ExitKey     = Esc;
  ExitKeyName = 'Esc';
{$EndIf}

const
  HeadTxtColor     = yellow;
  HeadTxtBack      = blue;
  HeadColor        = blue;
  HeadBackground   = lightgray;
  TxtColor         = black;
  TxtBackground    = lightgray;
  QuestionColor    = blue;
  waitcolor        = brown;
  falsecolor       = red;
  truecolor        = green;
  errorcolor       = red;
  LogoColor        = lightgray;
  LogoShadow       = black;

const MaxAnswers = 35; { Q might be reserved }
const AnswerKeys = [ '1'..'9', 'A'..'Z' ];

var endless : boolean;

var ScrWidth, ScrHeight : TscreenPos;

{ maximum Length and Height of textarea }
var MaxLength, MaxHeight: TscreenPos;

{ displaytype }
var display: DisplayType;

{ Line count after last wait }
var LineCnt: word;

type 
  Tscrquiz = 
    object(Takfquiz)
      readerror : boolean;

      { only temorarily used: }
      AnsPoints : array[1..MaxAnswers] of pointsType;

      destructor Done;                            virtual;

      procedure ResetQuiz;                        virtual;
      procedure StartQuiz;                        virtual;
      procedure setcharset(cs: string);           virtual;
      procedure processComment;                   virtual;
      procedure processHint;                      virtual;
      procedure processQuestion;                  virtual; 
      procedure processMulti;                     virtual;
      procedure processAnswer;                    virtual;
      procedure processMultiAnswer;               virtual;
      procedure processAssessment;                virtual;
      procedure processAssessmentPercent;         virtual;
      procedure evaluate;                         virtual;
      procedure EndQuiz;                          virtual; 
      procedure error;                            virtual;
      procedure NewScreen;
      procedure wait;
      procedure paragraph;
      procedure showanswers(showdefault: boolean);
      procedure nextanswer;
      function getanswer(keys: keyset): integer;
    end;

var infile: mystring;
var quiz: Tscrquiz;



procedure normscreen;
begin
window(1,1,ScrWidth,ScrHeight);
NormVideo;
ClrScr;
MaxLength := ScrWidth;
MaxHeight := ScrHeight;
LineCnt   := 0
end;

procedure errorFileNotFound;
begin
normscreen;
WriteLn(msg_filenotfound);
Halt(1)
end;
	  
procedure Ln;
begin
WriteLn;
inc(LineCnt)
end;

function toodeep: boolean;
begin
toodeep := LineCnt >= MaxHeight
end;

procedure clearbuffer;
var c: char;
begin
while KeyPressed do c := ReadKey
end;

function waitkey(const s: string): boolean;
var 
  c : char;
  oldcolor: byte;
begin

{ compatiblity GPC/FPC: }
{$IfDef __GPC__}
  {$Define CursorOff HideCursor}
  {$Define CursorOn  NormalCursor}
{$EndIf}

if WhereX<>1 then Ln;

{ clear buffer }
{ needed for some nasty compiler-bug for BSD ? }
clearbuffer;

oldcolor := TextAttr;
TextColor(waitcolor);
if length(s) <= MaxLength { avoid linebreak here }
  then Write(s)
  else Write(copy(s, 1, MaxLength));
TextAttr := oldcolor;

{$IfNdef NoHide}
  CursorOff;
{$EndIf}

c := ReadKey;
if c=#0 then ReadKey;
  
GotoXY(1, WhereY);
DelLine;
LineCnt := 0;

{$IfNdef NoHide}
  CursorOn;
{$EndIf}

waitkey := (Upcase(c) <> ExitKey)
end;

procedure centered(x: string);
begin
if length(x) > MaxLength then x := copy(x, 1, MaxLength);
GotoXY( (MaxLength div 2) - (Length(x) div 2), WhereY);
Write(x);
Ln
end;

procedure Logo(s: string);
var 
  x, y: TscreenPos;
  i: integer;
  wide: boolean;
const
  BlockChar = '*';
  LColor  = ((LogoColor  and $7F) shl 4) or (LogoColor  and $7F);
  LShadow = ((LogoShadow and $7F) shl 4) or (LogoShadow and $7F);
  { set color & background to the same value => block }
  { enforce lowvideo, because highvideo cannot be used for background }
begin
wide := ScrWidth >= 80; 

{ draw shadow }
TextAttr := LShadow;
y := WhereY + 1;
if wide 
  then x := (ScrWidth div 2) - (Length(s)+1) 
  else x := ((ScrWidth div 2) - (Length(s) div 2)+1);

for i := 1 to Length(s) do
  begin
  if s[i]<>' ' then
    begin
    GotoXY(x, y);
    Write(BlockChar);
    if wide then Write(BlockChar)
    end;
  if wide then inc(x, 2) else inc(x)
  end;

{ draw foreground }
TextAttr := LColor;
y := WhereY - 1;
if wide 
  then x := (ScrWidth div 2) - (Length(s)+1) 
  else x := ((ScrWidth div 2) - (Length(s) div 2)+1);

for i := 1 to Length(s) do
  begin
  if s[i]<>' ' then
    begin
    GotoXY(x, y);
    Write(BlockChar);
    if wide then Write(BlockChar)
    end;
  if wide then inc(x, 2) else inc(x)
  end;
WriteLn
end;

procedure titlescreen;
begin
normscreen;

TextColor(HeadTxtColor);
TextBackground(HeadTxtBack);

ClrScr;

{ is the display large enough? }
if (ScrHeight<21) or (ScrWidth<30) 
  then begin
       GotoXY(1, TscreenPos(Max((ScrHeight div 2)-1, 1)));
       centered(AKFQuizName + ' ' + AKFQuizVersion);
       WriteLn
       end
  else begin
       GotoXY(1, TscreenPos(Max((ScrHeight div 2)-8, 1)));
       Logo('   #    #    #  #######');
       Logo('  # #   #   #   #      ');
       Logo(' #   #  #  #    #      ');
       Logo('#     # ###     #####  ');
       Logo('####### #  #    #      ');
       Logo('#     # #   #   #      ');
       Logo('#     # #    #  #      ');
       WriteLn;
       WriteLn;
       Logo(' #####            #        ');
       Logo('#     #                    ');
       Logo('#     #  #    #   #  ######');
       Logo('#     #  #    #   #      # ');
       Logo('#   # #  #    #   #    ##  ');
       Logo('#    #   #    #   #   #    ');
       Logo(' #### #   #### #  #  ######');
       WriteLn;
       WriteLn
       end;

TextColor(HeadTxtColor);
TextBackground(HeadTxtBack);
centered('Copyright (C) '+AKFQuizCopyright);
centered(msg_GPL);
{$IfDef Advertisement}
  centered(msg_advertisement);
{$EndIf}
{$IfDef Win32}
  case lang of
    deutsch: centered('ALT + Enter = Vollbildmodus');
    otherwise centered('Alt + Enter = Fullscreen mode')
    end;
{$EndIf}

if not waitKey('') and not endless then 
   begin
   normscreen;
   Halt
   end
end;

procedure buildscreen(title: string);
var sideborder, topborder, bottomborder: Tscreenpos;
begin
normscreen;

TextColor(HeadTxtColor);
TextBackground(HeadTxtBack);
ClrScr;

if ScrHeight>17
   then begin topborder := 5; bottomborder := 1 end
   else begin topborder := 2; bottomborder := 0 end;

if ScrWidth<40 
   then begin sideborder:=0; bottomborder:=0 end
   else sideborder := 2;

if topborder=2
  then centered(title)
  else begin
       WriteLn;
       TextColor(HeadColor);
       TextBackground(HeadBackground);
       if Length(title)+6 <= MaxLength
          then centered('   '+title+'   ')
          else centered(title);
       TextColor(HeadTxtColor);
       TextBackground(HeadTxtBack);
       WriteLn;
       if ScrWidth>=70 
          then centered('Program: '+AKFQuizName+', version '+AKFQuizVersion+
                ' by AKFoerster')
          else centered(AKFQuizName)
       end;

{ wider visible border }
window(sideborder+1, topborder, ScrWidth-sideborder, ScrHeight-bottomborder);
TextColor(TxtColor);
TextBackground(TxtBackground);
ClrScr;
{ real window }
if sideborder<>0 then inc(sideborder);
window(sideborder+1, topborder, ScrWidth-sideborder, ScrHeight-bottomborder);
{ one column wider than MaxLength, to avoid automatic linebreaks }

MaxLength := ScrWidth - (2*sideborder) - 1; { see comment above }
MaxHeight := ScrHeight - topborder - bottomborder - 1;
LineCnt   := 0
end;

{---------------------------------------------------------------------}
procedure Tscrquiz.resetQuiz;
begin
inherited resetQuiz;

readerror := false
end;

destructor Tscrquiz.Done;
begin
inherited Done;

normscreen;
if readerror then
  begin
  TextColor(errorcolor);
  WriteLn(msg_fileerror);
  NormVideo
  end
end;

{ bug workaround }
procedure Tscrquiz.NewScreen;
begin
{ buildscreen(title); } { redraw everything }

ClrScr; { still problematic for GPC } {@@@}
LineCnt := 0
end;

procedure Tscrquiz.wait;
begin
if not quit then
  if ScrWidth<80 
     then quit := not waitkey(msg_anykey(''))
     else quit := not waitkey(msg_anykey(ExitKeyName))
end;

procedure Tscrquiz.StartQuiz;
begin
inherited StartQuiz;
buildscreen(title);

ClrScr;
Ln; Ln; Ln;

centered(msg_quiz + Title);
if author<>'' then 
  centered(msg_author + author);
if copyright<>'' then
  centered('Copyright: ' + copyright);
if authorURI<>'' then 
  centered(msg_authorURI + authorURI);
if translator<>'' then
  centered(msg_translator + translator);
if edited<>'' then
  centered(msg_edited + edited);
if license<>'' then
  centered(msg_license + license);
if licenseURI<>'' then
  centered(msg_licenseURI + licenseURI);

{$IfDef Advertisement}
  Ln;
  centered(msg_advertisement);
{$EndIf}

{$IfDef Win32}
  Ln;
  case lang of
    deutsch: centered('ALT + Enter = Vollbildmodus');
    otherwise centered('Alt + Enter = Fullscreen mode')
    end;
{$EndIf}

Ln;

wait
end;

procedure tscrquiz.setcharset(cs: string);
begin
inherited setcharset(cs);

cs := makeUpcase(cs); { to make checking easier }

{ Latin1 is set as default }
if (cs='IBM850') or (cs='IBM437') or
   (cs='CP850') or (cs='CP437') or
   (cs='850') or (cs='437')  
   then case display of
          OEMdisplay:  setconverter(noconversion);
          UTF8display: setconverter(OEMtoUTF8);
	  ISOdisplay:  setconverter(OEMtoLatin1);
          end;

if (cs='UTF-8') or (cs='UTF8') then
   case display of
     OEMdisplay:  setconverter(UTF8toOEM);
     UTF8display: setconverter(noconversion);
     ISOdisplay:  setconverter(UTF8toLatin1);
     end;

if (cs='ASCII') or (cs='US-ASCII') then setconverter(forceASCII)
end;

procedure newparagraph;
begin
Ln;
Ln
end;

procedure Tscrquiz.paragraph;
var 
  s, rest: mystring;
begin
s := readLine;
while s<>'' do
  begin
  if s='.'
    then newparagraph
    else begin
         rest := s;
         while rest<>'' do 
	   begin
	   Write(format(rest, MaxLength-WhereX+1, MaxLength), ' ');
	   if rest<>'' then Ln;
	   if toodeep then wait
	   end
	 end;
  if toodeep then wait;
  s := readLine
  end;
Ln
end;

procedure Tscrquiz.processComment;
begin
NewScreen;
Ln;
paragraph;
Ln;
wait
end;

procedure Tscrquiz.processHint;
begin
processComment { handle like a Comment }
end;

procedure Tscrquiz.processQuestion;
begin
inherited processQuestion;

NewScreen;
Ln;
TextColor(QuestionColor);
paragraph;
TextColor(TxtColor);

processAnswer
end;

procedure Tscrquiz.processMulti;
begin
inherited processMulti;
NewScreen;
Ln;
TextColor(QuestionColor);
paragraph;
TextColor(TxtColor);

processMultiAnswer
end;

procedure Tscrquiz.nextanswer;
begin
inc(answerNr);
if answerNr>MaxAnswers then 
   begin 
   error; 
   answerNr := MaxAnswers { to avoid range overruns }
   end
end;

procedure Tscrquiz.showanswers(showdefault: boolean);
var
  startpos: TscreenPos;
  s, ans: mystring;
  value: pointsType;
begin
Ln;
if toodeep then wait;

answerNr := 0;
readAnswer(value, s);
while s<>'' do
  begin
  ans := s;
  nextanswer;
  AnsPoints[answerNr] := value;  { points }
  if ScrWidth>=80 then Write('   ');
  Write(ValueToKey(answerNr), ') ');
  startpos := WhereX;
  while ans<>'' do 
    begin
    Write(format(ans, MaxLength-WhereX+1, MaxLength-startpos)); 
    if ans<>'' then 
       begin
       Ln;
       if toodeep then wait;
       if ScrWidth>=80 then Write('      ')
                       else Write('   ')
       end
    end;
  Ln;
  if toodeep then wait;
  readAnswer(value, s) { Answer line }
  end;

{ show default-Answer }
if showdefault and (defanswer<>'') then
  begin
  ans := defanswer;
  nextanswer;
  AnsPoints[answerNr] := 0;
  if ScrWidth>=80 then write('   ');
  Write(ValueToKey(answerNr), ') ');
  startpos := WhereX;
  while ans<>'' do 
    begin
    Write(format(ans, MaxLength-WhereX+1, MaxLength-startpos)); 
    if ans<>'' then 
       begin
       Ln;
       if toodeep then wait;
       if ScrWidth>=80 then Write('      ')
                       else Write('   ')
       end
    end;
  Ln; 
  if toodeep then wait;
  end
end;

function Tscrquiz.getanswer(keys: keyset): integer;
var 
  c : char;
  okay: boolean;
begin
repeat
  c := Upcase(ReadKey);
  okay := c in keys;
  if not okay then ErrorSignal;
  if c=#0 then ReadKey;
until okay;

if c=ExitKey then quit := true;

if c in AnswerKeys
  then begin write(c); getanswer := KeyToValue(c) end
  else getanswer := -1
end;

procedure Tscrquiz.processAnswer;
var ap : pointsType;
    i : integer;
    keys: keyset;
    maxkey: char;
begin
showanswers(true);

if toodeep then wait;
Ln;
Write('> ');

maxkey := ValueToKey(answerNr);
if answerNr<10 
   then keys := [ '1' .. maxkey, ExitKey ]
   else keys := [ '1' .. '9', 'A' .. maxkey, ExitKey ];
i := getanswer(keys);
if (i<>-1) and not quit then 
  begin
  ap := AnsPoints[i];
  inc(Points, ap);

  if not neutral then
    begin
    Write('  '); { distance }
    if ap > 0 
      then 
        begin { true }
          TextColor(TrueColor);
          Write(msg_right);
	  RightSignal
        end
      else
        begin { false }
            TextColor(FalseColor);
            Write(msg_wrong);
	    FalseSignal
        end
     end; { not neutral }

  TextColor(TxtColor);
  Ln; Ln;
  Write(msg_points, Points);
  Ln;
  wait
  end
end;

procedure Tscrquiz.processMultiAnswer;
var i: integer;
    keys: keyset;
    myPoints, myMax : pointsType;
    maxkey: char;
begin
myPoints := 0;
myMax := 0;

showanswers(false);

Ln; 
Write('>> ');

{ max points for this one question }
for i := 1 to answerNr do
  if AnsPoints[i] > 0 then inc(myMax, AnsPoints[i]);

maxkey := ValueToKey(answerNr);
if answerNr<10 
   then keys := [ '1'..maxkey, ExitKey, Enter ]
   else keys := [ '1'..'9', 'A'..maxkey, ExitKey, Enter ];

repeat
  i := getAnswer(keys);
  if i<>-1 then 
    begin
    inc(Points, AnsPoints[i]);   { absolute points }
    inc(myPoints, AnsPoints[i]); { points for this question }
    { remove that key from valid keys }
    keys := keys - [ ValueToKey(i) ]
    end
until i=-1;

Ln; Ln;
Write(msg_points, Points, 
      '  (', myPoints, '/', myMax, ')');
Ln;
wait
end;

procedure Tscrquiz.evaluate;
begin
if MaxPoints=0 then 
  begin
  inherited evaluate;
  exit
  end;
  
NewScreen;
Ln; Ln;
if ScrWidth>=80 then Write('   ');
Write(msg_sol1, Points, msg_sol2, MaxPoints);
if length(msg_sol3) > MaxLength-WhereX then Ln;
Write(msg_sol3);
Ln;
If Points > 0
  then begin
       if ScrWidth>=80 then write('   ');
       Write(msg_sol4, getPercentage, '%.');
       Ln
       end
  else if not neutral then 
          begin
	  if ScrWidth>=80 then write('   ');
          Write(msg_sol5);
          Ln
          end;
Ln;
inherited evaluate;

InfoSignal;
wait
end;

procedure Tscrquiz.processAssessment;
begin
processComment { is handled like a comment }
end;

procedure Tscrquiz.processAssessmentPercent;
var rest: mystring;
begin
rest := readAssessmentPercent;
while rest<>'' do 
   begin
   Write(format(rest, MaxLength-WhereX+1, MaxLength), ' ');
   Ln
   end;
wait
end;

procedure Tscrquiz.EndQuiz;
begin
if not evaluated and not quit 
  then evaluate
end;

procedure Tscrquiz.error;
begin
readerror := true;
quit := true
end;

{---------------------------------------------------------------------}

{$IfDef __GPC__}

  procedure FetchScreenSize;
  var x1, y1, x2, y2: integer;
  begin
  GetWindow(x1, y1, x2, y2);
  
  ScrWidth  := word(x2-x1+1);
  ScrHeight := word(y2-y1+1);

  MaxLength := x2-x1+1;
  MaxHeight := y2-y1+1
  end;

{$Else}

  procedure FetchScreenSize;
  begin
  { like in Borland (Turbo) Pascal }
  ScrWidth  := Lo(WindMax) - Lo(WindMin) + 1;
  ScrHeight := Hi(WindMax) - Hi(WindMin) + 1;

  { If nothing else works, use constants }
  { ScrWidth := 80; ScrHeight := 25; }

  MaxLength := ScrWidth;
  MaxHeight := ScrHeight
  end;
{$EndIf}

{ FPC has trouble with stderr when used with unit Crt }
procedure help;
begin
WriteLn(AKFQuizName + ', scrquiz, version ' + AKFQuizVersion);
WriteLn('(' + platform + ')');
WriteLn('for using akfquiz files on a textconsole');
WriteLn('Copyright (C) ', AKFQuizCopyright);
WriteLn(msg_License, msg_GPL);
{$IfDef Advertisement}
  WriteLn;
  WriteLn(msg_advertisement);
{$EndIf}
WriteLn;
WriteLn('Syntax:');
WriteLn('  scrquiz [options] [inputfile]');
WriteLn('  scrquiz [ -h | --help | /? ]');
WriteLn;
WriteLn('Options:');
WriteLn('-s          no sound');
WriteLn('-d <dir>    path to quizfiles');
WriteLn('-p          endless (use with care!)');
WriteLn('-OEM        display has OEM (IBM850/IBM437) charset');
WriteLn('-latin1     display has Latin1 charset');
WriteLn('-UTF8       display has UTF-8 charset');
{$IfDef FPC} {$IfDef Go32v2}
WriteLn('-lfn        use long filenames (DOS only)');
{$EndIf} {$EndIf}

WriteLn;
WriteLn('QUIZPATH: ', getQuizPath);
Halt
end;

procedure myshowentry(const s: string);
begin
Write(stripext(s));
Ln;
if toodeep then waitkey(msg_anykey(''))
end;

function askfile: mystring;
var 
  found: boolean;
  path, s: mystring;
begin
buildscreen('AKFQuiz');
Ln;

found := false;
path := getquizpath;
while path<>'' do
  begin
  s := getnextdir(path);
  if ListEntries(s, quizext, myshowentry)  then found := true;
  if ListEntries(s, quizext2, myshowentry) then found := true;
  end;

if not found then begin Write(msg_noquizfound); Ln end;

Ln;
clearbuffer;
QuestionSignal;
Write('Quiz: ');
ReadLn(s);
inc(LineCnt);
askfile := s
end;

procedure setmsgconv;
begin
case display of
  ISOdisplay:  setmsgconverter(UTF8toLatin1);
  OEMdisplay:  setmsgconverter(UTF8toOEM);
  UTF8display: setmsgconverter(noconversion)
  end
end;

procedure parameters;
var 
  i: integer;
  p: mystring;
begin
i := 0;
while i<ParamCount do
  begin
  inc(i);
  p := makeUpcase(ParamStr(i));
  if p='-LFN' then
      begin setLFNsupport; continue end;
  if p='-S' then
      begin DisableSignals; continue end;
  if p='-OEM' then
      begin display := OEMdisplay; setmsgconv; continue end;
  if (p='-LATIN1') or (p='-NOOEM') then
      begin display := ISOdisplay; setmsgconv; continue end;
  if (p='-UTF8') or (p='-UTF-8') then
      begin display := UTF8display; setmsgconv; continue end;
  if (p='-P') or (p='/P') then 
     begin endless := true; nobreak; checkbreak := false; continue end;
  if (p='-D') then
     begin inc(i); { handled in qsys } continue end;
  if (p='-H') or (p='--HELP') or (p='/?') then help;
  if p[1]='-'    { "/" might be used in a path }
     then help { unknown parameter }
     else infile := ParamStr(i) { not Upcase }
  end
end;

var myexitcode : byte;

begin { main }
myexitcode := 0;
endless := false;
useSystemLanguage;
useBeepSignals;

display := ISOdisplay;
if checkOEM then display := OEMdisplay;
if checkUTF8 then display := UTF8display;
setmsgconv;

{$IfDef __GPC__}
  CRTInit;
  {$IfDef __OS_DOS__}
     SetPCCharset(true);
     display := OEMdisplay;
  {$Else}
     SetPCCharset(false);
  {$EndIf}
{$EndIf}

FetchScreenSize;

repeat
  parameters;
  titlescreen;

  if infile<>'' then
     if not getquizfile(infile) then ErrorFileNotFound;

  if infile='' then 
    repeat
      infile := askfile;
      if not endless and (infile='') 
           then begin normscreen; Halt end;
    until (infile<>'') and getquizfile(infile);
  
  quiz.Init(infile);
  if IOResult<>0 then ErrorFileNotFound;

  { assume Latin1 as default charset }
  { may be changed by the "charset:" keyword }
  case display of
    ISOdisplay:  quiz.setconverter(noconversion);
    OEMdisplay:  quiz.setconverter(Latin1toOEM);
    UTF8display: quiz.setconverter(Latin1toUTF8)
    end;

  quiz.process;
  if quiz.readerror then myexitcode := 2;

  quiz.Done;

  infile := ''
until not endless;

Halt(myexitcode)
end.
