#include <alloc.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <dir.h>
#include <time.h>
#include "announce.h"
#include "dx.h"
#include "tools.h"
#include "cshow.h"
#include "node.h"
#include "define.h"
#include "buffers.h"
#include "message.h"
#include "streams.h"
#include "commands.h"
#include "protocol.h"
#include "database.h"
#include "language.h"
#include "hops.h"
#include "motd.h"
#include "time.h"
#include "dxcc.h"
#include "switch.h"
extern "C" {
#include "memory.h"
}
#include "system.h"
#include "wwv.h"
#include "filter.h"
#include "users.h"
#include "locator.h"
#include "ufilter.h"
#include "sun.h"
#include "wcy.h"
#include "password.h"
#include "rcmd.h"
#ifdef DOS
#include "malloc.h"
#include "term.h"
#endif

#define SPECIAL_DX_

/*------------------------------------------------------------------------
  ------ Rediriger les commandes SHOW ------------------------------------
  ------------------------------------------------------------------------
  La fonction return TRUE si la commande a ete executee                   */
#pragma argsused
int CSHOW_execute(int StreamNum, char * pCmd_p, int nParams, char **pArg, char **pStrArg)
{
	int Length;			/*Longueur de la commande*/

	if( *pArg[0] == SNULL )
		return FALSE;

	/*Longueur de la commande*/
	Length = strlen (pArg[0]);

	if( COMMANDS_check(pArg[0], "ann_age", Length, 5) )
		return CSHOW_annAge(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "announce", Length, 1) || ! strncmp(pArg[0], "a/", 2) )
		return CSHOW_announce(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "beam", Length, 1) )
		return CSHOW_beam(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "buffer_timeout", Length, 8) )
		return CSHOW_bufferTimeout(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "cluster", Length, 2) )
		return CSHOW_cluster(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "configuration", Length, 1) )
		return CSHOW_configuration(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "c/n", Length, 3) )
		return CSHOW_nodeConfiguration(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "date", Length, 4) )
		return CSHOW_date(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "memleft", Length, 3) )
		return CSHOW_memleft(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "database", Length, 4) )
		return CSHOW_database(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "dx_age", Length, 4) )
		return CSHOW_dxAge(StreamNum, nParams, pArg, pStrArg);

	if( ! strcmp(pArg[0], "dx") || ! strncmp(pArg[0], "dx/", 3) )
		return CSHOW_dx(StreamNum, nParams, pArg, pStrArg);

	if( ! strcmp(pArg[0], "fc") )
		return CSHOW_fileClosed(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "filter", Length, 1) )
		return CSHOW_filter(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "hops", Length, 3) )
		return CSHOW_hops(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "language", Length, 3) )
		return CSHOW_language(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "mail_age", Length, 6) )
		return CSHOW_mailAge(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "motd", Length, 3) )
		return CSHOW_motd(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "page_timeout", Length, 6) )
		return CSHOW_pageTimeout(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "page", Length, 2) )
		return CSHOW_pageLength(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "prefix", Length, 1) )
		return CSHOW_prefix(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "register", Length, 2) )
		return CSHOW_register(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "station", Length, 2) )
		return CSHOW_station(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "sun", Length, 2) )
		return CSHOW_sun(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "spot_age", Length, 4) )
		return CSHOW_dxAge(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "time", Length, 1) )
		return CSHOW_time(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "tnc_status", Length, 3) )
		return CSHOW_tnc_status(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "users", Length, 1) )
		return CSHOW_users(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "usercmd", Length, 5) )
		return CSHOW_usercmd(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "version", Length, 3) )
		return CSHOW_version(StreamNum, nParams, pArg, pStrArg);

	if( COMMANDS_check(pArg[0], "wwfilter", Length, 3) )
		return CSHOW_wwfilter(StreamNum, nParams, pArg, pStrArg);

	if( ! strcmp(pArg[0], "wwv") || ! strncmp(pArg[0], "wwv/", 4) )
		return CSHOW_wwv(StreamNum, nParams, pArg, pStrArg);

	if( ! strcmp(pArg[0], "wcy") || ! strncmp(pArg[0], "wcy/", 4) )
		return CSHOW_wcy(StreamNum, nParams, pArg, pStrArg);

	/*Interrogation d'une base de donnees locale ?*/
	if( DATABASE_local(StreamNum, pArg[0], pArg[1], DB_LOCAL) )
	{
		/*Commande database OK*/
		Prompt;
		return TRUE;
	}

	return FALSE;		/*La commande n'a pas ete trouvee*/
}

/*------------------------------------------------------------------------
  ------ Annonce ---------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_announce(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	int	iNbr = 5;       	/* Nombre d'annonce a afficher */
	int	Done = FALSE;		/* TRUE si une annonce a ete trouvee */
	int	index;
	char szUserCall[16];	/* Utilise par ANNOUNCE_get */
	char szNodeCall[16];	/* idem */
	char szString[256];		/* idem */
	char szToWho[32];		/* idem */
	char cFlag;				/* idem */
	char szDate[16];
	char szTime[16];
	char buffer[256];
	time_t tm;

	unsigned long lDateTime;

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/* Nombre d'annonces a renvoyer */
	if( pArg[0][1] == '/' )
	{
		if( pArg[0][2] != SNULL )	/*Proteger le atoi*/
		{
			iNbr = atoi (pArg[0] + 2);

			if( iNbr <= 0 )
				iNbr = 5;
		}
	}

	/* Pour une recherche non "case sensitive" */
	if( *pArg[1] != SNULL )
		strlwr(pArg[1]);

	/* Lire les iNbr annonces */
	tm = time(NULL);
	for(index = 1; index <= iNbr; index++)
	{
		if( ! ANNOUNCE_get(index, szNodeCall, szToWho, szUserCall, &cFlag, szString, &lDateTime) )
			break;	/* La fin de la liste a ete atteinte */

		/* Si des conditions ont ete donnees est qu'elles sont remplies - si pas de condx. */
		strcpy(buffer, szString);
		strlwr(buffer);
		if( *pArg[1] == SNULL || strstr(buffer, pArg[1]) )
		{
			/* Convertir la date */
			TOOLS_dateTime_Long2Str(szDate, szTime, lDateTime);

			/* Adresser l'annonce a l'utilisateur */
			BUFFERS_printBuff(StreamNum, OUT, "%-8s de %-9s %10s <%4sZ>:\n%s\n",
				szToWho, szUserCall, szDate, szTime, szString);

			Done = TRUE;
		}
		else
		{
			/* Si le programme arrive ici, c'est qu'une condition de recherche
			   n'a pas ete trouvee. Ajouter 1 a iNbr */
			iNbr++;

			/* Timeout de 30 secopndes pour la recherche */
			if( time(NULL) > tm + 30 )
				break;
		}
	}

	/* Afficher un message si aucune correspodance n'a ata trouvee */
	if( ! Done )
		MSG_send(StreamNum, MSG7);

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ Ann Age ---------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_annAge(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern long	PARAMS_lAnnDupeAllowed;
	extern int STREAMS_level[MAX_STREAMS];

	char	szTime[256];

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/* Aucun argument pour cette fonction */
	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);
		Prompt;
		return TRUE;
	}

	/* Niveau d'utilisateur */
	if( ! (STREAMS_level[StreamNum] & LEVEL_sysopV) )
		return FALSE;

	TOOLS_seconds2string(PARAMS_lAnnDupeAllowed,  szTime);

	BUFFERS_printBuff(StreamNum, OUT, "Annonce age : %s\n",
		szTime);

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/Beam ---------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_beam(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	char szToLocator[8];
	char szFromLocator[8];
	extern tUser   *pUser;

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	LOCATOR_COORDINATES from;
	LOCATOR_COORDINATES to;

	/* Un argument pour cette fonction */
	if( nParams != 2 )
	{
		MSG_send(StreamNum, MSG58);
		Prompt;
		return TRUE;
	}

	TOOLS_maxLength(pArg[1], 6);
	strcpy(szToLocator, pArg[1]);
	strupr(szToLocator);

	/* Le locator est il valide ? */
	if( ! LOCATOR_isValidWW(szToLocator) )
	{
		/* Locator incorrect */
		MSG_send(StreamNum, MSG18);
		Prompt;
		return TRUE;
	}

	/* Recuperer mon locator en memoire */
	USERS_getRecord(StreamNum);

	if( ! LOCATOR_isValidWW(pUser->sLocator) )
	{
		/* locator pas valide */
		MSG_send(StreamNum, MSG71);
		Prompt;
		return TRUE;
	}

	strcpy(szFromLocator, pUser->sLocator);

	/* Calcul de la distance et de l'azimut */
	LOCATOR_locator2coordinates(szFromLocator, &from);
	LOCATOR_locator2coordinates(szToLocator, &to);

	/* Trois appels (il faut s'assurer que azimut soit appele apres
	   distance */
	BUFFERS_printBuff(StreamNum, OUT, "%s -> %s : ",
			  pUser->sLocator,
			  szToLocator);
	BUFFERS_printBuff(StreamNum, OUT, "%d km, ",
			  LOCATOR_distance(from, to));
	BUFFERS_printBuff(StreamNum, OUT, "%d\n",
			  LOCATOR_azimut(from, to));

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ Buffer timeout --------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_bufferTimeout(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern long	PARAMS_userBuffTimeout;
	extern int STREAMS_level[MAX_STREAMS];

	char	szTime[256];

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/* Aucun argument pour cette fonction */
	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);
		Prompt;
		return TRUE;
	}

	/* Niveau d'utilisateur */
	if( ! (STREAMS_level[StreamNum] & LEVEL_sysopV) )
		return FALSE;

	TOOLS_seconds2string(PARAMS_userBuffTimeout,  szTime);

	BUFFERS_printBuff(StreamNum, OUT, "Buffer timeout : %s\n",
		szTime);

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/CLuster ------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_cluster(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern 	time_t	MAIN_timeStart;
	int	iNodes,
		iLocalUsers,
		iTotalUsers,
		iMaxUsers;
	int   	iDays,
		iHours,
		iMin;

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/*Pas d'arguments pour cette fonction*/
	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);
		Prompt;
		return TRUE;
	}

	/*Quelle est la config ?*/
	NODE_getConfig(iNodes, iLocalUsers, iTotalUsers, iMaxUsers);
	TOOLS_diffTime(MAIN_timeStart, time(NULL), &iDays, &iHours, &iMin);

	BUFFERS_printBuff(StreamNum, OUT,
		"DxNet v%s: %d nodes, %d local / %d total users - Max %d Uptime %dd %02d:%02d\n",
		VERSION, iNodes, iLocalUsers, iTotalUsers, iMaxUsers,
		iDays, iHours, iMin);


	Prompt;
  return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/Configuration ------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_configuration(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern int 	LastStream;	/*switch.cpp*/
	unsigned char cFlag;
	char		sNodeCall[16];
	char		sUserCall[16];
	char		sCallSign[16];
	char		sFilter[16];
	int		iFilterLength;
	int		iNb;	        /*Nombre d'utilisateurs affiches*/
	int		iStream;	/*Numero du stream*/
	int		iNodePos;	/*Position du node dans le stream*/
	int		iUserPos;	/*Position des utilisateurs*/

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/*Pas d'arguments pour cette fonction*/
	if( nParams > 2 )
	{
		MSG_send(StreamNum, MSG58);
		Prompt;
		return TRUE;
	}

	/*L'utilisateur a t-il specifie un filtre ?*/
	if( nParams == 2 )
	{
		TOOLS_maxLength(pArg[1], 15);
		strcpy(sFilter, pArg[1]);
		strupr(sFilter);
		iFilterLength = strlen (sFilter);
	}
	else
	{
		*sFilter = SNULL;
		iFilterLength = 0;
	}

	/*Afficher l'entete*/
	MSG_send(StreamNum, MSG6);	/*Cluster configuration*/

	/*------ Parcourir la table ------*/
	for(iStream = 0; iStream <= LastStream; iStream++)
	{
		for(iNodePos = 0; iNodePos < NODE_MAXANODE; iNodePos++ )
		{
			/*Indicatif du node sur le stream*/
			NODE_getNodeCall(iStream, iNodePos, sNodeCall);

			/*Pas d'indicatif -> iNodePos suivant*/
			if( ! *sNodeCall )
				continue;

			/*Ne rentre pas dans le filtre -> iNodePos suivant*/
			if( *sFilter && strncmp(sNodeCall, sFilter, iFilterLength) )
				continue;

			/* Flags sur ce cluster */
			cFlag = NODE_getNodeFlag(iStream, iNodePos);

			/* here / nohere */
			if( cFlag & BIT6 )
				strcpy(sCallSign, sNodeCall);
			else
				sprintf(sCallSign, "(%s)", sNodeCall);

			/* Cluster-wide conf */
			if( cFlag & BIT7 )
				strcat(sCallSign, "*");

			BUFFERS_printBuff(StreamNum, OUT, "%-14s", sCallSign);

			iNb = 0;

			/*Liste des utilisateurs*/
			for(iUserPos = 1; iUserPos <= 64; iUserPos++)
			{
				/*Indicatif de l'utilisateur*/
				NODE_getUserCall(iStream, iNodePos, iUserPos, sUserCall);

				/*Pas d'utilisateur -> iUserPos suivant*/
				if( ! *sUserCall )
					continue;

				/*Faut-il passer a la ligne ?*/
				if( iNb++ == 5 )
				{
					iNb = 1;
					BUFFERS_addBuff(StreamNum, "\n              ", OUT);
				}

				/*Quelle est la configuration de l'utilisateur ?*/
				cFlag =  NODE_getUserFlag(iStream, iNodePos, iUserPos);
				if( cFlag & PROTOCOL_USER_HERE )
					strcpy (sCallSign, sUserCall);		/*SET/HERE*/
				else                                            /*SET/NOHERE*/
					sprintf(sCallSign, "(%s)", sUserCall);

				if( cFlag & PROTOCOL_USER_CONF )		/*Conference ?*/
					strcat(sCallSign, "+");
				else if( cFlag & PROTOCOL_USER_CLUSTERCONF )
					strcat(sCallSign, "*");

				BUFFERS_printBuff(StreamNum, OUT, "%-12s", sCallSign);

			}/*End iUserPos FOR*/

			if( iNb == 0 ) /*Pas d'utilisateurs connectes*/
			{
				int iUserCount = NODE_getUserCount(iStream, sNodeCall);

				if( iUserCount != 0 )
					BUFFERS_printBuff(StreamNum, OUT, "(%d users)", iUserCount);
			}/*End IF*/

			BUFFERS_addBuff(StreamNum, "\n", OUT);

		}/*End iNodePos FOR*/
	}/*End iStream FOR*/

#ifdef SPECIAL_DX_
	{
		extern int	STREAMS_level[MAX_STREAMS];
		extern char	STREAMS_callsign[MAX_STREAMS][10];

		if( STREAMS_level[StreamNum] & LEVEL_sysopD )
		{
			for(iStream = 1; iStream <= LastStream; iStream++)
			{
				if( HOPS_fix(iStream, NULL, 0, PCAS_USER, IN, 100) )
					BUFFERS_printBuff(StreamNum, OUT, "<%s>\n",
				STREAMS_callsign[iStream]);
			}
		}
	}/*End IF*/
#endif

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/DATE ---------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_date(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern long   PARAMS_lTz;
	time_t t;
	struct tm tDate;
	char   szBuffer[256];
	extern int	STREAMS_level[MAX_STREAMS];

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/* RX only ? */
	if( STREAMS_level[StreamNum] & LEVEL_user_RXonly )
		return FALSE;

		if( nParams != 1 )
		{
			MSG_send(StreamNum, MSG58);		/*Pas d'argument pour cette cmd*/
			Prompt;
			return TRUE;
		}

		t     = time(NULL) - PARAMS_lTz;
		tDate = *localtime(&t);
		strftime(szBuffer, 255, "%a %b %d, %Y", &tDate);
		BUFFERS_printBuff(StreamNum, OUT, "Current date : %s (%04d.%02d.%02d)\n",
			szBuffer,
			tDate.tm_year + 1900,
			tDate.tm_mon + 1,
			tDate.tm_mday);

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ Ann Age ---------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_dxAge(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern int	STREAMS_level[MAX_STREAMS];
	extern long	PARAMS_lDxRejectBefore;
	extern long	PARAMS_lDxRejectAfter;

	char	szTime1[256];
	char	szTime2[256];

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/* Aucun argument pour cette fonction */
	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);
		Prompt;
		return TRUE;
	}

	/* Niveau d'utilisateur */
	if( ! (STREAMS_level[StreamNum] & LEVEL_sysopV) )
		return FALSE;

	TOOLS_seconds2string(PARAMS_lDxRejectBefore, szTime1);
	TOOLS_seconds2string(PARAMS_lDxRejectAfter,  szTime2);

	BUFFERS_printBuff(StreamNum, OUT, "DX spot age : -%s +%s\n",
		szTime1, szTime2);

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/MEMleft ------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_memleft(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

#ifdef DOS
	unsigned int  iHandler;
	unsigned long lConventional = 0L;
	unsigned long lExtended     = 0L;
	unsigned long lExpanded     = 0L;
	unsigned long lVirtual      = 0L;
	char          *pMemoryType;
	unsigned int	iCounter      = 0;

	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);		/*Pas d'argument pour cette cmd*/
		Prompt;
		return TRUE;
	}

	BUFFERS_printBuff(StreamNum, OUT, "%ld bytes free in far heap, %u bytes left in stack.\n",
		farcoreleft(), stackavail());

	/*Scanner la memoire utilise par le memory manager*/
	for(iHandler = 0; iHandler < 32767; iHandler++)
	{
		pMemoryType = mem_type( iHandler );

		if( *pMemoryType != 'N' )	/*Not allocated*/
		{
			iCounter = 0;

			if( *pMemoryType == 'C' )
				lConventional += (unsigned long) get_memsize(iHandler);
			else if( *(pMemoryType + 2) == 't' )
				lExtended     += (unsigned long) get_memsize(iHandler);
			else if( *(pMemoryType + 2) == 'p' )
				lExpanded     += (unsigned long) get_memsize(iHandler);
			else if( *pMemoryType == 'V' )
				lVirtual      += (unsigned long) get_memsize(iHandler);
		}
		else
		{
			/*Au dela de 500 ihandler vide consecutif, on suppose
			  qu'il n'y en aura pas d'autres utilises. */
			if( iCounter++ > 500 )
				break;
		}
	}/*End FOR*/

	BUFFERS_printBuff(StreamNum, OUT, "The memory manager uses : %4ld Kb of Conventional memory.\n",   lConventional / 1024L);
	BUFFERS_printBuff(StreamNum, OUT, "                          %4ld Kb of Extented (XMS) memory.\n", lExtended / 1024L);
	BUFFERS_printBuff(StreamNum, OUT, "                          %4ld Kb of Expanded (EMS) memory.\n", lExpanded / 1024L);
	BUFFERS_printBuff(StreamNum, OUT, "                          %4ld Kb of Virtual memory.\n",        lVirtual / 1024L);
#else
	BUFFERS_addBuff(StreamNum, "This command is implemented in the DOS version only.\n", OUT);
  /*BUFFERS_printBuff(StreamNum, OUT, "%ld bytes free.\n", farcoreleft() );*/
#endif

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/database -----------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_database(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);		/*Pas d'argument pour cette cmd*/
		Prompt;
		return TRUE;
	}

	DATABASE_showList(StreamNum);

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/DX -----------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_dx(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	unsigned long	lBand = 0xFFFFFFFFL; /*Bande sur laquelle effectuer la recherche*/
	int	 iNbr  = 5;
	char *   pPtr;
	int	 index;
	int 	 Done;
	time_t   tm;
	DXSPOT   DxSpot;

	extern int	PARAMS_iSetDxSsid;

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/*Nombre de DX a renvoyer*/
	if( pArg[0][2] == '/' )
	{
		if( pArg[0][3] != SNULL )	/*Proteger le atoi*/
		{
			iNbr = atoi (pArg[0] + 3);

			if( iNbr <= 0 )
				iNbr = 5;
		}
	}/*End IF*/

	/* S'il y a des arguments */
	if( nParams != 1 )
	{
		/*Recherche en majuscule uniquement*/
		strupr(pArg[1]);
		strupr(pArg[2]);

		/*Recherche sur une bande particuliere */
		pPtr = pArg[1];
		Done = TRUE;

		while( *pPtr )
		{
			if( ! isdigit(*pPtr) && *pPtr != '.' )
				Done = FALSE;	/*Autre chose de des chiffres ont ete trouve*/
			pPtr++;
		}/*End WHILE*/

		lBand = (unsigned long) (atof(pArg[1]) * 10000.0);
		if( lBand == 0L || Done == FALSE )
			lBand = 0xFFFFFFFFL;
		else
			strcpy(pArg[1], "###");	/*Copier qq chose que le test suivant ne pourra confondre avec un indicatif*/

		/* Cas particulier pour le 18 et le 24 MHz */
		if( lBand == 180000L )
			lBand = 180680L;
		else if( lBand == 240000L )
			lBand = 248900L;

		lBand = DX_checkFrequency(lBand);
	}/*End IF*/

   /*Done est utilise pour s'assurer qu'une correspondance a ete trouve
     lors de la recherche*/
	Done = FALSE;

	/*Parcourir le fichier*/
	tm = time(NULL);
	for(index = 1; index <= iNbr; index++)
	{
		if( ! DX_get(index, &DxSpot) )
			break;	/*La fin de la liste  t atteinte*/

		/*Si des conditions ont t donnes et qu'elles sont remplies - si
		  pas de condx, afficher */
		if( ! *pArg[1] || 				    		/*S'il n'y a aucune conditions*/
		   (! *pArg[2] && strstr(DxSpot.szDxCall, pArg[1]))   || 	/*Recherche sur une partie de l'indicatif*/
		   (  *pArg[2] && strstr(DxSpot.szDxCall, pArg[2] )   &&   /*Recherche sur une partie de l'indicatif*/
		      lBand == DX_checkFrequency(DxSpot.lFrequency) ) ||
		   (  lBand == DX_checkFrequency(DxSpot.lFrequency)   && 	/*Recherche sur la bande*/
		    ! *pArg[2] ))
		{
			char szLogger[16];

			/* Faut il enlever le SSID du logger ? */
			strcpy(szLogger, DxSpot.szLogger);
			if( PARAMS_iSetDxSsid == FALSE )
				TOOLS_removeSsid(szLogger);
			strcat(szLogger, ">");

			TOOLS_maxLength(DxSpot.szComments, 27);	/*Ne pas dpasser la limite sur l'cran*/
			TOOLS_maxLength(DxSpot.szDxCall, 12);	/*idem*/
			BUFFERS_printBuff(StreamNum, OUT, "%8ld.%ld %-12s %11s %4sZ %-27s<%-6s\n",
				DxSpot.lFrequency/10, DxSpot.lFrequency%10,
				DxSpot.szDxCall, DxSpot.szDate, DxSpot.szTime,
				DxSpot.szComments, szLogger);
			Done = TRUE;
		}
		else
		{
			/*Si le programme arrive ici, c'est qu'une condition de recherche
			  n'a pas t trouve. Ajouter 1  iNbr*/
			iNbr++;

			/* Timeout de 30 secopndes pour la recherche */
			if( time(NULL) > tm + 30 )
				break;
		}
	}/*End FOR*/

	/* Afficher un message si aucune correspodance n'a t trouvee */
	if( ! Done )
		MSG_send(StreamNum, MSG7);

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/FC -----------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_fileClosed(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	static int iFileClosed = 0;	/*Nombre de fichier qui n'ont pas ete ouvert*/

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_SYSOP) )
		return FALSE;

	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);		/*Pas d'argument pour cette cmd*/
		Prompt;
		return TRUE;
	}

	iFileClosed += TOOLS_fcloseall();

	BUFFERS_printBuff(StreamNum, OUT, "%d files have been abnormaly closed.\n", iFileClosed);

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/FILTER -------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_filter(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern tUser *pUser;
	extern char *MSG_pParams[10];
	char   szBuffer[254];

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58); 	/* Pas d'argument pour cette cmd */
		Prompt;
		return TRUE;
	}

	/* Recuperer les parametres de filtrage */
	USERS_getRecord(StreamNum);

	/* Affichage de confirmation de la commande */
	if( pUser->lBandsOff & USERS_ALLBANDS || pUser->lBandsOff & USERS_ALLMODES )
		MSG_get(StreamNum, MSG63, szBuffer); /* valide */
	else
		MSG_get(StreamNum, MSG64, szBuffer); /* valide */
	if( szBuffer[0] ) /* Si buffer non vide : enlever CR */
		szBuffer[strlen(szBuffer) - 1] = SNULL;
	MSG_pParams[9] = szBuffer;
	MSG_send(StreamNum, MSG75);

	if( pUser->lBandsOff & USERS_ALLBANDS || pUser->lBandsOff & USERS_ALLMODES )
		UFILTER_show(StreamNum, ~(pUser->lBandsOff));

	Prompt;
	return TRUE;
}


/*------------------------------------------------------------------------
  ------ SH/HOPs ---------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_hops(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	int	iPc16,
		iPc19,
		iPc50;
	int	iDone = FALSE;
	int	index;
	char	sBuffer[256];

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/*Si l'utilisateur a prcis un indicatif ...*/
	if( nParams > 1 )
	{
		strupr (pArg[1]); 	/*Convertir en majuscule pour la recherche*/

		/*Verifier que l'indicatif est valide*/
		if( ! TOOLS_isCall (pArg[1]) )
		{
			MSG_send(StreamNum, MSG18);
			Prompt;
			return TRUE;
		}

		if( HOPS_getHops(pArg[1], 0, iPc16, iPc19, iPc50) )
		{
			MSG_send(StreamNum, MSG30);
			COMMANDS_sendHops(StreamNum, pArg[1], iPc16, iPc19, iPc50, TRUE);  /*Envoyer le tableau  l'utilisateur*/
		}
		else
			MSG_send(StreamNum, MSG31);
	}
	else
	{
		int nCR = FALSE;

		/*Pas d'indicatif donn, parcourir la table*/
		for(index = 0; index < 5000; index++) /*5000 tant dj bcp !!*/
		{
			*sBuffer = SNULL;

			if( ! HOPS_getHops(sBuffer, index, iPc16, iPc19, iPc50) )
				break;

			if( iDone == FALSE )
			{
				char	szBuffer[256];

				/* Afficher l'entete */
				MSG_get(StreamNum, MSG30, szBuffer);
				TOOLS_removeNR(szBuffer);
				BUFFERS_printBuff(StreamNum, OUT, "%s           %s\n",
					szBuffer, szBuffer);
				iDone = TRUE;
			}

			/*Envoyer le tableau  l'utilisateur*/
			COMMANDS_sendHops(StreamNum, sBuffer, iPc16, iPc19, iPc50, nCR);

			/* Pour l'affichage sur deux colonnes */
			if( nCR == FALSE )
			{
				BUFFERS_printBuff(StreamNum, OUT, "            ");
				nCR = TRUE;
			}
			else
				nCR = FALSE;
		}/*End FOR*/

		if( nCR == TRUE )
			BUFFERS_addBuff(StreamNum, "\n", OUT);

	}/*End IF*/

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/LANguage -----------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_language(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	int 	index;
	char	sBuffer[256];

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_NOTALLOWED) )
		return FALSE;

	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);		/*Nbr d'arguments incorrect*/
		Prompt;
		return TRUE;
	}

	/*Afficher l'entete*/
	MSG_send(StreamNum, MSG9);

	/*Balayer la liste des langues disponibles*/
	for(index = 0; index < MAX_LANG; index ++)
	{
		extern char	STREAMS_lang[MAX_STREAMS];
		if( LANG_isDefined(index) )
			BUFFERS_printBuff(StreamNum, OUT, "%c%d : %s",
				(STREAMS_lang[StreamNum] == index ? '*' : ' '),
				index + 1,
				LANG_getMessage(index, LANGUAGE, sBuffer));
	}/*End FOR*/

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ Mail Age --------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_mailAge(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern long	PARAMS_lMailReject;
	extern int	STREAMS_level[MAX_STREAMS];

	char	szTime[256];

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/* Aucun argument pour cette fonction */
	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);
		Prompt;
		return TRUE;
	}

	/* Niveau d'utilisateur */
	if( ! (STREAMS_level[StreamNum] & LEVEL_sysopV) )
		return FALSE;

	TOOLS_seconds2string(PARAMS_lMailReject, szTime);

	BUFFERS_printBuff(StreamNum, OUT, "Mail age : %s\n",
		szTime);

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/MOTD ---------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_motd(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern char	STREAMS_lang[MAX_STREAMS];
	extern int	STREAMS_level[MAX_STREAMS];

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/*Niveau d'utilisateur*/
	if( ! (STREAMS_level[StreamNum] & LEVEL_sysopV) )
		return FALSE;

	if( nParams > 2 )
	{
		MSG_send(StreamNum, MSG58);		/*Nbr d'arguments incorrect*/
		Prompt;
		return TRUE;
	}

	if( nParams == 1 ) 	/*MOTD par defaut, si aucune valeur n'est precisee*/
		MOTD_send(StreamNum, STREAMS_lang[StreamNum]);
	else
	{
		int iNb = atoi (pArg[1]);

		if( iNb < 1 || iNb > MAX_LANG )
			MSG_send(StreamNum, MSG18);
		else
			MOTD_send(StreamNum, iNb - 1);
	}/*End IF*/

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/N/C ----------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_nodeConfiguration(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern int 	FirstStream;	/*switch.cpp*/
	extern int 	LastStream;	/*switch.cpp*/
	char		szNodeCall[16];
	char		szFilter[16];
	int		nFilterLength;
	int		nNb;	        /*Nombre d'utilisateurs affiches*/
	int		nStream;	/*Numero du stream*/
	int		nNodePos;	/*Position du node dans le stream*/

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/*Pas d'arguments pour cette fonction*/
	if( nParams > 2 )
	{
		MSG_send(StreamNum, MSG58);
		Prompt;
		return TRUE;
	}

	/*L'utilisateur a t-il specifie un filtre ?*/
	if( nParams == 2 )
	{
		TOOLS_maxLength(pArg[1], 15);
		strcpy(szFilter, pArg[1]);
		strupr(szFilter);
		nFilterLength = strlen (szFilter);
	}
	else
	{
		*szFilter = SNULL;
		nFilterLength = 0;
	}

  /*Afficher l'entete*/
//  MSG_send(StreamNum, MSG6);	/*Cluster configuration*/


	/*------ Parcourir la table ------*/
	for(nStream = FirstStream; nStream <= LastStream; nStream++)
	{
		/* Indicatif du node sur le stream */
		NODE_getNodeCall(nStream, 0, szNodeCall);

		/* Pas d'indicatif -> iNodePos suivant */
		if( ! *szNodeCall )
			continue;

		/* Ne rentre pas dans le filtre -> iNodePos suivant */
		if( *szFilter && strncmp(szNodeCall, szFilter, nFilterLength) )
			continue;

		BUFFERS_printBuff(StreamNum, OUT, "%-14s", szNodeCall);

		nNb = 0;

		for(nNodePos = 1; nNodePos < NODE_MAXANODE; nNodePos++ )
		{
			/* Indicatif du node sur le stream */
			NODE_getNodeCall(nStream, nNodePos, szNodeCall);

			/* Pas d'indicatif -> iNodePos suivant */
			if( ! *szNodeCall )
				continue;

			/* Ne rentre pas dans le filtre -> iNodePos suivant */
			if( *szFilter && strncmp(szNodeCall, szFilter, nFilterLength) )
				continue;

			/* Faut-il passer a la ligne ? */
			if( nNb++ == 5 )
			{
				nNb = 1;
				BUFFERS_addBuff(StreamNum, "\n              ", OUT);
			}

			BUFFERS_printBuff(StreamNum, OUT, "%-12s", szNodeCall);
		}

		BUFFERS_addBuff(StreamNum, "\n", OUT);
	}

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ Page timeout ----------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_pageTimeout(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern long	PARAMS_userPageTimeout;
	extern int	STREAMS_level[MAX_STREAMS];

	char	szTime[256];

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/* Aucun argument pour cette fonction */
	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);
		Prompt;
		return TRUE;
	}

	/* Niveau d'utilisateur */
	if( ! (STREAMS_level[StreamNum] & LEVEL_sysopV) )
		return FALSE;

	TOOLS_seconds2string(PARAMS_userPageTimeout,  szTime);

	BUFFERS_printBuff(StreamNum, OUT, "Page timeout : %s\n",
		szTime);

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/PAGE ---------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_pageLength(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern char *   MSG_pParams[10];
	extern tUser*   pUser;
	int		nPage;
	char		szBuffer[256];

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_NOTALLOWED) )
		return FALSE;

	/* L'utilisateur  t-il indiqu un numro ? */
	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);
		Prompt;
		return TRUE;
	}

	USERS_getRecord(StreamNum);
	nPage = (int) pUser->byPage;

	if( nPage == 0 )
		MSG_get(StreamNum, MSG64, szBuffer); /* devalide */
	else
		sprintf(szBuffer, "%d ", nPage);
	if( szBuffer[0] ) /* Si buffer non vide */
		szBuffer[strlen(szBuffer) - 1] = SNULL;
	MSG_pParams[9] = szBuffer;
	MSG_send(StreamNum, MSG79);

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/PREFIX -------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_prefix(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	char szCtyFound[256];
	int  nResult;

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	if( nParams != 2 )
	{
		MSG_send(StreamNum, MSG58);
		Prompt;
		return TRUE;
	}

	nResult = DXCC_search(strupr(pArg[1]), szCtyFound);

	if( nResult == -1 )
	{
		/* Info trouve, afficher le resultat */
		DXCC_showResult(StreamNum, pArg[1], szCtyFound);
	}
	else
	{
		/* Pas trouve, afficher le message d'erreur */
		MSG_send(StreamNum, nResult);
	}

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/STATION ------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_station(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern int STREAMS_level[MAX_STREAMS];
	extern char PARAMS_nNTWULogin;
	char szStation[16];
	char szDate[16],    szTime[16];
	char szNTWDate[16], szNTWTime[16];
	char szLanguage[256];
	tUser  user;

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_SYSOP) )
		return FALSE;

	/* RX only ? */
	if( STREAMS_level[StreamNum] & LEVEL_user_RXonly )
		return FALSE;

	if( nParams > 2 )
	{
		MSG_send(StreamNum, MSG58);
		Prompt;
		return TRUE;
	}

	if( nParams == 1 )
	{
		extern char	STREAMS_callsign[MAX_STREAMS][10];

		strcpy(szStation, STREAMS_callsign[StreamNum]);
	}
	else
	{
		TOOLS_maxLength(pArg[1], 9);
		strcpy(szStation, pArg[1]);
	}

	/* Mettre en forme l'indicatif */
	TOOLS_removeSsid(szStation);
	strupr(szStation);

	/* Rechercher */
	if( ! USERS_getConfig(szStation, &user) )
	{
		/* utilisateur inconnu */
		MSG_send(StreamNum, MSG31);
		Prompt;
		return TRUE;
	}

	/* Date et heure de la derniere connexion */
	TOOLS_dateTime_Long2Str(szDate, szTime, user.lLastConnect);
	TOOLS_dateTime_Long2Str(szNTWDate, szNTWTime, user.lLastNetwkConnect);
	LANG_getMessage(user.byLanguage, LANGUAGE, szLanguage);
	TOOLS_removeNR(szLanguage);

	BUFFERS_printBuff(StreamNum, OUT, "User          : %s\n", user.sCallSign);
	BUFFERS_printBuff(StreamNum, OUT, "Name          : %s\n", user.sName);
	BUFFERS_printBuff(StreamNum, OUT, "QTH           : %s\n", user.sQth);
	BUFFERS_printBuff(StreamNum, OUT, "Locator       : %s\n", user.sLocator);
	BUFFERS_printBuff(StreamNum, OUT, "Home node     : %s\n", user.sHomeNode);
	if( user.lLastConnect )
		BUFFERS_printBuff(StreamNum, OUT, "Last connect  : %s %sZ\n", szDate, szTime);
	else
		BUFFERS_printBuff(StreamNum, OUT, "Last connect  :\n");
	if( PARAMS_nNTWULogin )
	{
		if( user.lLastNetwkConnect )
			BUFFERS_printBuff(StreamNum, OUT, "Network       : %s %sZ @%s\n", szNTWDate, szNTWTime, user.szLastCluster);
		else
			BUFFERS_printBuff(StreamNum, OUT, "Network       :\n");
	}
	BUFFERS_printBuff(StreamNum, OUT, "Last msg read : %ld\n", user.lLastMsg);
	BUFFERS_printBuff(StreamNum, OUT, "Connections   : %ld\n", user.lLogins);
	BUFFERS_printBuff(StreamNum, OUT, "Language      : %s (#%d)\n", szLanguage, user.byLanguage);

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/PREFIX -------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_sun(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	char szCtyFound[256];
	int  nResult;

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	if( nParams != 2 )
	{
		MSG_send(StreamNum, MSG58);
		Prompt;
		return TRUE;
	}

	nResult = DXCC_search(strupr(pArg[1]), szCtyFound);

	if( nResult == -1 )
	{
		/* Info trouve, afficher le resultat */
		char szPrefix[256];
		char szCtyName[256];
		char szSunrise[256];
		char szSunset[256];
		char szBuffer[256];
		LOCATOR_COORDINATES to;

		DXCC_getField(szCtyFound, szBuffer, DXCC_LATITUDE_P, DXCC_LATITUDE_L);
		to.doLat = atof(szBuffer);
		DXCC_getField(szCtyFound, szBuffer, DXCC_LONGITUDE_P, DXCC_LONGITUDE_L);
		to.doLong = -atof(szBuffer);

		BUFFERS_printBuff(StreamNum, OUT,
			"%-5s %-25s Sunrise = %sZ   Sunset = %sZ\n",
			DXCC_getField(szCtyFound, szPrefix, DXCC_PREFIX_P, DXCC_PREFIX_L),
			DXCC_getField(szCtyFound, szCtyName, DXCC_CTYNAME_P, DXCC_CTYNAME_L),
			SUN_calculate(to.doLat, to.doLong, SUN_RISE, szSunrise),
			SUN_calculate(to.doLat, to.doLong, SUN_SET, szSunset));
	}
	else
	{
		/* Pas trouve, afficher le message d'erreur */
		MSG_send(StreamNum, nResult);
	}

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/TIME ---------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_time(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern long   PARAMS_lTz;
	time_t t;
	struct tm tTime;
	char   szBuffer[256];
	extern int	STREAMS_level[MAX_STREAMS];

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);		/*Pas d'argument pour cette cmd*/
		Prompt;
		return TRUE;
	}

	t     = time(NULL) - PARAMS_lTz;
	tTime = *localtime(&t);
	strftime(szBuffer, 255, "%X", &tTime);
	BUFFERS_printBuff(StreamNum, OUT, "Current time : %s\n", szBuffer);

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/REGISTER -----------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_register(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern int	STREAMS_level[MAX_STREAMS];

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_NOTALLOWED) )
		return FALSE;

	/*Niveau d'utilisateur*/
	if( ! (STREAMS_level[StreamNum] & LEVEL_sysopV) )
		return FALSE;

	if( nParams > 2 )
	{
		MSG_send(StreamNum, MSG58);
		Prompt;
		return TRUE;
	}

	if( ! PASSWORD_showFile(StreamNum, pArg[1]) )
		BUFFERS_addBuff(StreamNum, "*** No entry found !\n", OUT);

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/TNC_STATUS ---------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_tnc_status(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern int	STREAMS_level[MAX_STREAMS];
	extern char	STREAMS_callsign[MAX_STREAMS][10];
	extern		tUserCfg * pUserCfg;
#ifdef LINUX
	extern int	SWITCH_nSocket;
#endif
	int	index;
	char szLevel[64];
	char szPortName[64];
	char* pszConnectedCall;
	int	nCount = 0;
	int nConTime, nT1, nT2;
	char cSepar;

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/*Niveau d'utilisateur*/
	if( ! (STREAMS_level[StreamNum] & LEVEL_sysopV) )
		return FALSE;

	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);
		Prompt;
		return TRUE;
	}

#ifdef DOS
	if( SWITCH_isPcFlexNet() )
		BUFFERS_printBuff(StreamNum, OUT, "PC/FlexNet\n");
	else
		BUFFERS_printBuff(StreamNum, OUT, "G8BPQ switch - %d buffer(s) left\n", SWITCH_getBufferLeft());
#endif
#ifdef LINUX
	BUFFERS_printBuff(StreamNum, OUT, "Linux kernel\n");
#endif
#ifdef WIN32
	BUFFERS_printBuff(StreamNum, OUT, "SV2AGW packet engine\n");
#endif

	for(index = 1; index <= 64; index++)
	{
		int nBufferSize;

		if( STREAMS_level[index] == 0 )
			continue;

		if( nCount == 0 )
		{
			BUFFERS_addBuff(StreamNum, "chan  port    callsign   status   buffer  time \n", OUT);
			BUFFERS_addBuff(StreamNum, "-----------------------------------------------\n", OUT);
		}

		nCount++;
		pszConnectedCall = STREAMS_callsign[index];

		if( STREAMS_level[index] & LEVEL_cluster )
			strcpy(szLevel, "cluster");
		else if( STREAMS_level[index] & LEVEL_sysopV )
			strcpy(szLevel, "sysop");
		else if( STREAMS_level[index] & LEVEL_linkSetup )
			strcpy(szLevel, "link setup");
		else if( STREAMS_level[index] & LEVEL_bbs )
			strcpy(szLevel, "bbs");
		else if( STREAMS_level[index] & LEVEL_hiddenUser )
			strcpy(szLevel, "hidden");
#ifdef DOS
		else if( STREAMS_level[index] & LEVEL_term )
		{
			strcpy(szLevel, "terminal");
			pszConnectedCall = TERM_setCall(NULL);
		}
#endif
		else
			strcpy(szLevel, "user");

		/* Connexion time */
		if( STREAMS_level[index] & LEVEL_linkSetup )
		{
			nT1 = nT2 = 0;
			cSepar = ':';
		}
		else
		{
			USERS_getRecord(index);
			nConTime = (int) (difftime(time(NULL),
				TOOLS_dateTime_Long2time_t(pUserCfg->lConnectedDateTime)) / 60L);
			if( nConTime < 1440 )				/* les than a day */
			{
				nT1 = nConTime / 60;			/* hours */
				nT2 = nConTime % 60;			/* minutes */
				cSepar = ':';
			}
			else if( nConTime < 10080 )	/* less than one week */
			{
				nT1 =  nConTime / 1440;			/* days */
				nT2 = (nConTime % 1440) / 60;	/* hours */
				cSepar = 'd';
			}
			else
			{
				nT1 =  nConTime / 10080;		/* weeks */
				nT2 = (nConTime % 10080) / 1440;/* days */
				cSepar = 'w';
			}
		}

		/* Taille du buffer */
		nBufferSize = BUFFERS_getSize(index, IN) + BUFFERS_getSize(index, OUT);

		/* Nom du port */
		SWITCH_getPortName(index, szPortName);
		TOOLS_maxLength(szPortName, 7);

		BUFFERS_printBuff(StreamNum, OUT, "%4d  %-7s %-10s %-10s%5d  %02d%c%02d\n",	
						  index,
						  szPortName,
						  pszConnectedCall,
						  szLevel,
						  nBufferSize,
						  nT1, cSepar, nT2);
	}

	BUFFERS_printBuff(StreamNum, OUT, "%d port(s) in use\n", nCount);
#ifdef LINUX
	BUFFERS_printBuff(StreamNum, OUT, "%d open socket(s)\n", SWITCH_nSocket);
#endif

	Prompt;
	return TRUE;
}


/*------------------------------------------------------------------------
  ------ SH/USERS --------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_users(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	unsigned char cFlag;
	char		sNodeCall[16];
	char		sUserCall[16];
	char		sCallSign[16];
	int		iNb = 0;
	int		iStreamPos;

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);		/*Nbr d'arguments incorrect*/
		Prompt;
		return TRUE;
	}

	/*Afficher l'entete*/
	MSG_send(StreamNum, MSG40);	/*Cluster configuration*/

	iNb = 0;

	/*Indicatif du cluster*/
	NODE_getNodeCall(0, 0, sNodeCall);
	if( *sNodeCall )	/*S'il y a un indicatif, l'afficher*/
	{
		BUFFERS_printBuff(StreamNum, OUT, "%-14s", sNodeCall);

		/*ICI Afficher la liste des utilisateurs*/
		for(iStreamPos = 1; iStreamPos <= 64; iStreamPos++)
		{
			NODE_getUserCall(0, 0, iStreamPos, sUserCall);
			if( *sUserCall )	/*S'il y a un indicatif, l'afficher*/
			{
				if( iNb++ == 5 )/*Passer  la ligne ?*/
				{
					iNb = 1;
					BUFFERS_addBuff(StreamNum,
						"\n              ", OUT);
				}

				/*Quelle est la configuration de l'utilisateur ?*/
				cFlag =  NODE_getUserFlag(0, 0, iStreamPos);

				if( cFlag & PROTOCOL_USER_HERE )
					strcpy(sCallSign, sUserCall);                 /*SET/HERE*/
				else
					sprintf(sCallSign, "(%s)", sUserCall);        /*SET/NOHERE*/

				if( cFlag & PROTOCOL_USER_CONF )
					strcat(sCallSign, "+");
				else if( cFlag & PROTOCOL_USER_CLUSTERCONF )
					strcat(sCallSign, "*");

				BUFFERS_printBuff(StreamNum, OUT, "%-12s", sCallSign);
			}/*End IF*/
		}/*End FOR*/

		BUFFERS_addBuff(StreamNum, "\n", OUT);

	}/*End IF*/

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/USERCMD ------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_usercmd(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	extern char	STREAMS_callsign[MAX_STREAMS][10];
	char	sFileName[MAXPATH];
	char	sUserCall[16];

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_NOTALLOWED) )
		return FALSE;

	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);		/*Nbr d'arguments incorrect*/
		Prompt;
		return TRUE;
	}

	/*Indicatif de l'utilisateur effectuant la requete*/
	strcpy(sUserCall, STREAMS_callsign[StreamNum]);
	TOOLS_removeSsid(sUserCall);
	strlwr(sUserCall);

	sprintf(sFileName, "%s%s.cmd", USERCMD_PATH, sUserCall);

	if( ! SYSTEM_showFile(StreamNum, sFileName) )
	{
		/*S'il n'y a pas de config pour l'utilisateur, renvoyer la
		  config par defaut*/
		sprintf(sFileName, "%sdefault.cmd", USERCMD_PATH);
		SYSTEM_showFile(StreamNum, sFileName);
	}

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/VERSION ------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_version(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	char	sNodeCall[16];
	int	iStreamOfNode;
	int	iNodePos;

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	if( nParams > 2 )
	{
		MSG_send(StreamNum, MSG58);		/*Nbr d'arguments incorrect*/
		Prompt;
		return TRUE;
	}

	if( nParams == 1 )
	{
		/*Indicatif du cluster non specifie - utiliser celui du serveur*/
		NODE_getNodeCall(0, 0, sNodeCall);
	}
	else
	{
		TOOLS_maxLength (pArg[1], 9);
		strcpy(sNodeCall, strupr (pArg[1]));
	}

	/*Rechercher le cluster en memoire*/
	if( NODE_isNodeConnected(sNodeCall, &iStreamOfNode, &iNodePos) )
	{
		char	sVersion[16];

		strcpy(pArg[2], NODE_getNodeVersion(iStreamOfNode, iNodePos, sVersion));
		MSG_send(StreamNum, MSG44);	/*Transmettre le numero de version*/
	}
	else
	{
		MSG_send(StreamNum, MSG19);	/*Pas trouve*/
	}/*End IF*/

	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/WWFILTER -----------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_wwfilter(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	unsigned int nFlags;
	char         szBuffer[256];
	extern char *MSG_pParams[10];

	/* Cette commande n'existe que si SET/PFILTER est ON */
	if( ! FILTER_isEnabled() )
		return FALSE;

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_NOTALLOWED) )
		return FALSE;

	if( nParams != 1 )
	{
		MSG_send(StreamNum, MSG58);		/*Nbr d'arguments incorrect*/
		Prompt;
		return TRUE;
	}

	/* Quel filtre est applique sur ce stream ? */
	nFlags = USERS_getFlags(StreamNum);
	FILTER_flags2str(nFlags, szBuffer);

	if( szBuffer[0] == SNULL )
	{
		/* Pas de filtre : indique DEVALIDE */
		MSG_get(StreamNum, MSG64, szBuffer); /* devalide */
		if( szBuffer[0] ) /* Si buffer non vide : enlever CR */
			szBuffer[strlen(szBuffer) - 1] = SNULL;
		MSG_pParams[9] = szBuffer;
	}
	else
	{
		/* Filtre actif : afficher la liste */
		MSG_pParams[9] = szBuffer;
	}

	MSG_send(StreamNum, MSG69);
	Prompt;
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ SH/WCY ----------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_wcy(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	char 	szLogger[16];
	WCYSPOT WcySpot;
	int 	nNbr = 5;
	int	nDone = FALSE;
	int	index;

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/* Nombre de WCY a renvoyer */
	if( pArg[0][3] == '/' )
	{
		if( pArg[0][4] != SNULL )	/* Proteger le atoi */
		{
			nNbr = atoi(pArg[0] + 4);

			if( nNbr <= 0 )
				nNbr = 5;
		}
	}

	/* Balayer la table des spots WCY */
	for(index = 1; index <= nNbr; index++)
	{
		if( ! WCY_get(index, &WcySpot) )
			break;	/* Fin de fichier atteinte */

		/* Ajouter un ">" a la fin de l'indicatif
		  (affichage <CALLSIGN>) */
		strcpy(szLogger, WcySpot.szLogger);
		strcat(szLogger, ">");

		/* Si c'est le premier spot, envoyer un entete */
		if( nDone == FALSE )
		{
			nDone = TRUE;
			MSG_send(StreamNum, MSG78);
		}

	/* Date       Hour SFI A   K Exp.K R   SA  GMF Aurora Logger
	   2-Nov-2000 16   204 8   1 1     206 eru qui no     <DK0WCY-3> */

		/* Envoyer le spot WWV a l'utilisateur */
		BUFFERS_printBuff(StreamNum, OUT,
			"%11s  %c%c    %-3d  %-3d  %1d  %1d      %-3d  %-3s  %-3s  %-6s  <%s\n",
			WcySpot.szDate,
			WcySpot.szTime[0], WcySpot.szTime[1],
			WcySpot.nSFI,
			WcySpot.nA,
			WcySpot.nK,
			WcySpot.nExpK,
			WcySpot.nR,
			WcySpot.szSA,
			WcySpot.szGMF,
			WcySpot.szAurora,
			szLogger);
	} /* End FOR */

	Prompt;
	return TRUE;
}


/*------------------------------------------------------------------------
  ------ SH/WWV ----------------------------------------------------------
  ------------------------------------------------------------------------*/
#pragma argsused
int CSHOW_wwv(int StreamNum, int nParams, char **pArg, char **pStrArg)
{
	char 	  szLogger[16];
	WWVSPOT   WwvSpot;
	int 	  iNbr = 5;
	int	  iDone = FALSE;
	int	  index;

	/* Niveau d'utilisateur remote command */
	if( ! RCMD_isAllowed(StreamNum, RCMD_USER) )
		return FALSE;

	/*Nombre de WWV a renvoyer*/
	if( pArg[0][3] == '/' )
	{
		if( pArg[0][4] != SNULL )	/*Proteger le atoi*/
		{
			iNbr = atoi (pArg[0] + 4);

			if( iNbr <= 0 )
				iNbr = 5;
		}
	}/*End IF*/

	/*Balayer la table des spots WWV*/
	for(index = 1; index <= iNbr; index++)
	{
		if( ! WWV_get(index, &WwvSpot) )
			break;	/*Fin de fichier atteinte*/

		/*Ajouter un ">" a la fin de l'indicatif (affichage <CALLSIGN>)*/
		strcpy(szLogger, WwvSpot.szLogger);
		strcat(szLogger, ">");

		/*Si c'est le premier spot, envoyer un entete*/
		if( iDone == FALSE )
		{
			iDone = TRUE;
			MSG_send(StreamNum, MSG42);
		} /*End IF*/

		/*Envoyer le spot WWV a l'utilisateur*/
		BUFFERS_printBuff(StreamNum, OUT,
		      "%11s  %c%c   %3d %3d %3d  %-35s <%-6s\n",
		      WwvSpot.szDate, WwvSpot.szTime[0], WwvSpot.szTime[1],
		      WwvSpot.nSFI, WwvSpot.nA, WwvSpot.nK,
		      WwvSpot.szForecast, szLogger);
	}/*End FOR*/

	Prompt;
	return TRUE;
}
