#include <alloc.h>
#include <dir.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include "buffers.h"
#include "define.h"
#include "medit.h"
#include "tools.h"
#include "commands.h"

extern "C" {
#include "memory.h"
}

#define MEDIT_MAX_LINES 	255
#define MEDIT_LINES_DISPLAYED 	20

static MEDIT_DATA * MEDIT_pData[MAX_STREAMS];
static char 	    MEDIT_cNewFile[MAX_STREAMS];

/*------------------------------------------------------------------------
  ------ Initialisation --------------------------------------------------
  ------------------------------------------------------------------------*/
void MEDIT_init(void)
{
	int i;

	for(i = 0; i < MAX_STREAMS; i++)
		MEDIT_pData[i] = NULL;
}

/*------------------------------------------------------------------------
  ------ Ouvrir un fichier pour l'editer ---------------------------------
  ------------------------------------------------------------------------
  Return TRUE si OK                                                       */
int MEDIT_start(int StreamNum, char * pszFileName)
{
	extern unsigned long COMMANDS_statut[MAX_STREAMS];

	char szBuffer[256];
	int  index;

	/* Allouer de la memoire */
	MEDIT_pData[StreamNum] = (MEDIT_DATA *) malloc(sizeof(MEDIT_DATA));
	if( MEDIT_pData[StreamNum] == NULL )
	{
		BUFFERS_addBuff(StreamNum, "Not enough memory !\n", OUT);
		return FALSE;
	}

	MEDIT_pData[StreamNum]->memHandle = xalloc(MEDIT_MAX_LINES, 256);
	if( MEDIT_pData[StreamNum]->memHandle == 0 )
	{
		free(MEDIT_pData[StreamNum]);
		MEDIT_pData[StreamNum] = NULL;
		BUFFERS_addBuff(StreamNum, "Not enough memory !\n", OUT);
		return FALSE;
	}

	/* Initialiser les zones memoires */
	for(index = 0; index < MEDIT_MAX_LINES; index++)
	{
		memset(szBuffer, 0, sizeof(szBuffer));
		xput((byte*) szBuffer, index, MEDIT_pData[StreamNum]->memHandle);
	}

	/* Charger le fichier en memoire */
	TOOLS_maxLength(pszFileName, MAXPATH);
	strcpy(MEDIT_pData[StreamNum]->pszFileName, pszFileName);
	if( MEDIT_load(StreamNum) == FALSE )
	{
		MEDIT_clean(StreamNum);
		return FALSE;
	}

	MEDIT_pData[StreamNum]->nCurrentLine = 0;

	if( !( COMMANDS_statut[StreamNum] & COMMANDS_meditCreate) )
	{
		MEDIT_display(StreamNum);
		MEDIT_prompt(StreamNum);
	}

	return TRUE;
}

/*------------------------------------------------------------------------
  ------ Charger le fichier ----------------------------------------------
  ------------------------------------------------------------------------
  Return : TRUE  si OK                                                    */
int MEDIT_load(int StreamNum)
{
	extern unsigned long COMMANDS_statut[MAX_STREAMS];

	FILE * fPtr;
	int    nCount = 0;
	char   szBuffer[256];

	MEDIT_pData[StreamNum]->nLineCount = 0;

	/* Ouvrir le fichier */
	fPtr = fopen(MEDIT_pData[StreamNum]->pszFileName, "rt");
	if( fPtr == NULL )
	{
		BUFFERS_addBuff(StreamNum, "File does not exist. Create (Y/N) ?\n", OUT);
		COMMANDS_statut[StreamNum] |= COMMANDS_meditCreate;
		MEDIT_cNewFile[StreamNum] = TRUE;
		return TRUE;
	}
	else
		MEDIT_cNewFile[StreamNum] = FALSE;

	/* Lire le fichier */
	while( fgets(szBuffer, 255, fPtr) )
	{
		MEDIT_pData[StreamNum]->nLineCount++;

		xput((byte*) szBuffer, nCount++, MEDIT_pData[StreamNum]->memHandle);

		if( nCount == MEDIT_MAX_LINES )
		{
			/* Fichier trop long */
			BUFFERS_addBuff(StreamNum, "File too long !\n", OUT);
			fclose(fPtr);
			return FALSE;
		}
	}

	fclose(fPtr);
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ Traiter les commandes -------------------------------------------
  ------------------------------------------------------------------------
  Return TRUE si OK - FALSE is PB ou si fin de MEDIT                      */
int MEDIT_commands(int StreamNum, char * pszInput)
{
	int    nCount;

	TOOLS_maxLength(pszInput, 254);

	/* Aiguiller selon la commande */
	if( isdigit(pszInput[0]) )
	{
		int nLine = atoi(pszInput);

		if( nLine > 0 && nLine <= MEDIT_pData[StreamNum]->nLineCount )
		{
			MEDIT_pData[StreamNum]->nCurrentLine = nLine - 1;
			MEDIT_display(StreamNum);
		}
		else
		{
			BUFFERS_addBuff(StreamNum, "Out of range ...\n", OUT);
		}

		MEDIT_prompt(StreamNum);
		return TRUE;
	}

	switch(tolower(*pszInput))
	{
	case SNULL :
		break;

	case 'd' :	/* Delete */
		if( pszInput[1] == SNULL )
			nCount = 1;
		else if( ! isdigit(pszInput[1]) )
		{
			BUFFERS_addBuff(StreamNum, "Bad argument.\n", OUT);
			break;
		}
		else
			nCount = atoi(pszInput + 1);

		MEDIT_delete(StreamNum, nCount);
		MEDIT_display(StreamNum);
		break;

	case 'e' :	/* Quitter sans sauver */
		MEDIT_clean(StreamNum);
		BUFFERS_addBuff(StreamNum, "Exit without saving.\n", OUT);
		return FALSE;

	case 'i' :	/* Inserer une ligne */
		strcat(pszInput, "\n");
		MEDIT_insert(StreamNum, pszInput + 1);
		MEDIT_display(StreamNum);
		break;

	case '/' :	/* Remplacer une ligne */
		strcat(pszInput, "\n");
		MEDIT_replace(StreamNum, pszInput + 1);
		MEDIT_display(StreamNum);
		break;

	case '*' : 	/* Ajouter une ligne a la fin du fichier */
		strcat(pszInput, "\n");
		MEDIT_insert2end(StreamNum, pszInput + 1);
		MEDIT_display(StreamNum);
		break;

	case '?' :
		MEDIT_help(StreamNum);
		break;

	case 'w' :	/* Sauvegarder et quitter */
		if( tolower(pszInput[1]) == 'q' && pszInput[2] == SNULL )
		{
			if( MEDIT_save(StreamNum) )
			{
				BUFFERS_addBuff(StreamNum,
					"File saved.\n", OUT);
			}
			else
			{
				BUFFERS_addBuff(StreamNum,
					"*** Unexpected error : can't save file.\n",
					OUT);
			}
			MEDIT_clean(StreamNum);
			return FALSE;
		}
		/* Pas de break -> commande non reconnue */

	default :
		BUFFERS_addBuff(StreamNum, "Unrecognized command !\n", OUT);
		break;
	}

	MEDIT_prompt(StreamNum);
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ Afficher les lignes ---------------------------------------------
  ------------------------------------------------------------------------
  Return TRUE si OK                                                       */
int MEDIT_display(int StreamNum)
{
	char *ptr;
	int   index;
	int   nLine;

	/* Premiere ligne a afficher */
	nLine = MEDIT_pData[StreamNum]->nCurrentLine - 5;
	if( (nLine + MEDIT_LINES_DISPLAYED) > MEDIT_pData[StreamNum]->nLineCount)
		nLine = MEDIT_pData[StreamNum]->nLineCount - MEDIT_LINES_DISPLAYED;
	if( nLine < 0 )
		nLine = 0;

	for(index = 0; index < MEDIT_LINES_DISPLAYED; index++)
	{
		if( MEDIT_pData[StreamNum]->nLineCount == nLine )
			break;

		ptr = (char*) xget(nLine, MEDIT_pData[StreamNum]->memHandle);
		BUFFERS_printBuff(StreamNum, OUT, "%3d%c:%s",
			nLine + 1,
			(nLine == MEDIT_pData[StreamNum]->nCurrentLine) ? '*' : ' ',
			ptr);

		nLine++;
	}

	if( MEDIT_pData[StreamNum]->nLineCount == nLine )
		BUFFERS_addBuff(StreamNum, "-EOF-\n", OUT);

	return TRUE;
}

/*------------------------------------------------------------------------
  ------ Afficher le prompt ----------------------------------------------
  ------------------------------------------------------------------------*/
void MEDIT_prompt(int StreamNum)
{
	BUFFERS_addBuff(StreamNum, "(#,D,I,WQ,E,/,*,?) >\n", OUT);
}

/*------------------------------------------------------------------------
  ------ Liberer la memoire ----------------------------------------------
  ------------------------------------------------------------------------*/
void MEDIT_clean(int StreamNum)
{
	if( MEDIT_pData[StreamNum] != NULL )
	{
		xfree(MEDIT_pData[StreamNum]->memHandle);
		free(MEDIT_pData[StreamNum]);
	}
}

/*------------------------------------------------------------------------
  ------ Effacer une ligne -----------------------------------------------
  ------------------------------------------------------------------------*/
void MEDIT_delete(int StreamNum, int nCount)
{
	int   index;
	byte *bPtr;
	byte  szBuffer[256];

	int   nFirst,
	      nLast;

	nFirst = MEDIT_pData[StreamNum]->nCurrentLine;
	nLast  = nFirst + nCount;
	if( nLast > MEDIT_pData[StreamNum]->nLineCount )
	{
		nLast = MEDIT_pData[StreamNum]->nLineCount;
		nCount = nLast;
        }

	memset(szBuffer, 0, sizeof(szBuffer));
	for(index = 0;
	    index < (MEDIT_pData[StreamNum]->nLineCount -
		     MEDIT_pData[StreamNum]->nCurrentLine);
	    index++)
	{
		if( nLast > (MEDIT_MAX_LINES - 1) )
			bPtr = szBuffer;
		else
			bPtr = xget(nLast++, MEDIT_pData[StreamNum]->memHandle);

		xput(bPtr, nFirst++, MEDIT_pData[StreamNum]->memHandle);
	}

	if( MEDIT_pData[StreamNum]->nLineCount > 0 )
		MEDIT_pData[StreamNum]->nLineCount -= nCount;

}

/*------------------------------------------------------------------------
  ------ Remplacer une ligne ---------------------------------------------
  ------------------------------------------------------------------------*/
void MEDIT_replace(int StreamNum, char * pszString)
{
	xput((byte *) pszString,
	     MEDIT_pData[StreamNum]->nCurrentLine,
	     MEDIT_pData[StreamNum]->memHandle);
}

/*------------------------------------------------------------------------
  ------ Inserer une ligne -----------------------------------------------
  ------------------------------------------------------------------------*/
void MEDIT_insert(int StreamNum, char * pszString)
{
	int   nCount = MEDIT_pData[StreamNum]->nLineCount -
		       MEDIT_pData[StreamNum]->nCurrentLine;
	int   nLast  = MEDIT_pData[StreamNum]->nLineCount;
	int   index;
	byte *bPtr;

	if( (MEDIT_pData[StreamNum]->nLineCount + 1) == MEDIT_MAX_LINES )
	{
		BUFFERS_addBuff(StreamNum, "Out of memory !\n", OUT);
		return;
	}

	/* Decaller pour inserer la ligne */
	for(index = 0; index < nCount; index++, nLast)
	{
		bPtr = xget(nLast - 1, MEDIT_pData[StreamNum]->memHandle);
		xput(bPtr, nLast, MEDIT_pData[StreamNum]->memHandle);
		nLast--;
	}

	/* Incrementer le compteur de ligne */
	MEDIT_pData[StreamNum]->nLineCount++;

	/* Copier la nouvelle ligne */
	xput((byte *) pszString,
	     MEDIT_pData[StreamNum]->nCurrentLine,
	     MEDIT_pData[StreamNum]->memHandle);
}
/*------------------------------------------------------------------------
  ------ Inserer une ligne a la fin du fichier ---------------------------
  ------------------------------------------------------------------------*/
void MEDIT_insert2end(int StreamNum, char * pszString)
{
	xput((byte *) pszString,
	     MEDIT_pData[StreamNum]->nLineCount++,
	     MEDIT_pData[StreamNum]->memHandle);
}

/*------------------------------------------------------------------------
  ------ Enregistrer le fichier ------------------------------------------
  ------------------------------------------------------------------------
  Return : TRUE si OK                                                     */
int MEDIT_save(int StreamNum)
{
	FILE * fPtr;
	char * ptr;
	int    index;

	/* Creer un .BAK */
	if( MEDIT_cNewFile[StreamNum] == FALSE )
	{
		if( ! TOOLS_makeBak(MEDIT_pData[StreamNum]->pszFileName) )
			return FALSE;
	}

	/* Ouvrir le fichier et sauver */
	fPtr = fopen(MEDIT_pData[StreamNum]->pszFileName, "wt");
	if( fPtr == NULL )
		return FALSE;

	for(index = 0; index < MEDIT_pData[StreamNum]->nLineCount; index++)
	{
		ptr = (char *) xget(index, MEDIT_pData[StreamNum]->memHandle);
		fputs(ptr, fPtr);
	}

	fclose(fPtr);
	return TRUE;
}

/*------------------------------------------------------------------------
  ------ Affiche l'aide --------------------------------------------------
  ------------------------------------------------------------------------*/
void MEDIT_help(int StreamNum)
{
	BUFFERS_addBuff(StreamNum, "(#,D,I,WQ,E,/,*,?) >\n", OUT);

	BUFFERS_addBuff(StreamNum, "#      : set the line # as current line.\n", OUT);
	BUFFERS_addBuff(StreamNum, "D      : delete the current line.\n", OUT);
	BUFFERS_addBuff(StreamNum, "D#     : delete # lines from the current line.\n", OUT);
	BUFFERS_addBuff(StreamNum, "I<str> : insert a new line <str> before the current line.\n", OUT);
	BUFFERS_addBuff(StreamNum, "WQ     : write (save) file and quit.\n", OUT);
	BUFFERS_addBuff(StreamNum, "E      : exit without saving.\n", OUT);
	BUFFERS_addBuff(StreamNum, "/<str> : put the string <str> in place of the current line.\n", OUT);
	BUFFERS_addBuff(StreamNum, "*<str> : put the string <str> at the end of the file.\n", OUT);
	BUFFERS_addBuff(StreamNum, "?      : show this help file.\n", OUT);
}
